#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "server.hpp"
#include "conf.hpp"
#include "logfile.hpp"
#include "users.hpp"
#include "current.hpp"
#include "hrono.hpp"
#include "firewall.hpp"
#include "sockio.hpp"

extern CConfigurationFile *pConfig;

// 0 - No error
// 1 - Error
// 2 - No such user in the users database
// 3 - Accoun expired
int UserUp(char *pszLogin, char *pszDevice, char *pszHost, char *pszGateway)
{
	CUsersDatabase *pUsersDatabase = NULL;
	CCurrentDatabase *pCurrentDatabase = NULL;
	char *pszMessage1 = NULL;
	char *pszMessage2 = NULL;
	int iResult = 0;

	if ((NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage1, true))) ||
		 (NULL == (pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage2)))) {
		lprintf("Memory allocation error logging as user %s...\n", pszLogin);
		iResult = 1;
	} else if (NULL != pszMessage1 || NULL != pszMessage2) {
		lprintf("%s\n", pszMessage1 != NULL ? pszMessage1 : pszMessage2);
		iResult = 1;
	} else if (false == pUsersDatabase->ExistsRecord(pszLogin)) {
		lprintf("There is no account for user %s...\n", pszLogin);
		iResult = 2;
	} else if (true == pUsersDatabase->ExpiredCheck(pszLogin, pCurrentDatabase->GetTime(pszLogin), pCurrentDatabase->GetTraffic(pszLogin))) {
		lprintf("Account expired for user %s...\n", pszLogin);
		iResult = 3;
	} else if (false == pCurrentDatabase->AddRecord(pszLogin, pszDevice, pszHost, pszGateway)) {
		lprintf("Error adding record to current database when logging as %s, %s, %s...\n", pszLogin, pszDevice, pszHost);
		iResult = 1;
	} else if (false == InsertAccountRule(pszHost)) {
		lprintf("Error adding accounting firewall rule when logging as %s...\n", pszLogin);
		iResult = 1;
	}
	delete pUsersDatabase;
	delete pCurrentDatabase;
	
	if (0 == iResult)
		lprintf("User %s logged on...\n", pszLogin);
	
	return iResult;
}

// 0 - No error
// 1 - Error
// 2 - User is not logged on
int UserDown(char *pszLogin, char *pszDevice, char *pszHost, char *pszGateway)
{
	int iResult;
	
	SCurrentEntry strEntry;
	
	CUsersDatabase *pUsersDatabase = NULL;
	CCurrentDatabase *pCurrentDatabase = NULL;
	CHronoDatabase *pHronoDatabase = NULL;
	char *pszMessage1 = NULL;
	char *pszMessage2 = NULL;
	char *pszMessage3 = NULL;
	int iInputTraffic;
	int iOutputTraffic;
	time_t tLoginTime;
	time_t tLogoutTime;

	if ((NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage1))) ||
		 (NULL == (pHronoDatabase = new CHronoDatabase(pConfig, &pszMessage2))) ||
		 (NULL == (pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage3)))) {
		lprintf("Memory allocation error logging out as user %s...\n", pszLogin);
		iResult = 1;
		goto err_exit;
	}
	if (NULL != pszMessage1 || NULL != pszMessage2 || NULL != pszMessage3) {
		lprintf("%s\n", pszMessage1 ? pszMessage1 : pszMessage2 ? pszMessage2 : pszMessage3);
		iResult = 1;
		goto err_exit;
	}
	
	if (false == pCurrentDatabase->GetRecord(pszHost, &strEntry)) {
		lprintf("User %s tried to logout from host %s, but this host is not logged on...\n", pszLogin, pszHost);
		iResult = 2;
		goto err_exit;
	}
	
	if (0 != strcmp(pszLogin, strEntry.szLogin)) {
		lprintf("User %s tried to logout from host %s, but another user has logged on...\n", pszLogin, pszHost);
		iResult = 2;
		goto err_exit;
	}
		
	iInputTraffic = GetTraffic(pszHost, INPUT_TRAFFIC);
	iOutputTraffic = GetTraffic(pszHost, OUTPUT_TRAFFIC);
	tLoginTime = pCurrentDatabase->GetLoginTime(pszHost);
	tLogoutTime = time(NULL);
// Update entry in the current database
	if ((-1 == iInputTraffic) ||
		 (-1 == iOutputTraffic) ||
		 (false == pCurrentDatabase->UpdateRecord(pszLogin, pszDevice, pszHost, pszGateway, tLogoutTime, iInputTraffic, iOutputTraffic))) {
		lprintf("Error updating current entry for user %s...\n", pszLogin);
		iResult = 1;
		goto err_exit;
	}
// Credit user's account
	if (false == pUsersDatabase->CreditRecord(pszLogin, tLogoutTime - tLoginTime, iInputTraffic + iOutputTraffic)) {
		lprintf("Error updating users' record for %s...\n", pszLogin);
		iResult = 1;
		goto err_exit;
	}
// Add entry to the hrono database		
	if (false == pHronoDatabase->AddRecord(pszLogin, pszDevice, pszHost, tLoginTime, tLogoutTime, iInputTraffic, iOutputTraffic)) {
		lprintf("Error adding hrono entry for user %s...\n", pszLogin);
		iResult = 1;
		goto err_exit;
	}
		
	DeleteAccountRule(pszHost);
	pCurrentDatabase->DeleteRecord(pszHost);
	
	lprintf("User %s logged off...\n", pszLogin);
	iResult = 0;

err_exit:
	delete pUsersDatabase;
	delete pHronoDatabase;
	delete pCurrentDatabase;
	
	return iResult;
}

bool EmergencyUserDown(SCurrentEntry *pCurrentEntry)
{
	CUsersDatabase *pUsersDatabase = NULL;
	CHronoDatabase *pHronoDatabase = NULL;
	char *pszMessage1 = NULL;
	char *pszMessage2 = NULL;

	if ((NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage1))) ||
		 (NULL == (pHronoDatabase = new CHronoDatabase(pConfig, &pszMessage2)))) {
		lprintf("Memory allocation error logging out as user %s...\n", pCurrentEntry->szLogin);
		delete pUsersDatabase;
		delete pHronoDatabase;
		return false;
	}
	if (NULL != pszMessage1 || NULL != pszMessage2) {
		lprintf("%s\n", pszMessage1 ? pszMessage1 : pszMessage2);
		delete pUsersDatabase;
		delete pHronoDatabase;
		return false;
	}
	
// Update entry in the current database
	if (false == pUsersDatabase->CreditRecord(pCurrentEntry->szLogin,
															pCurrentEntry->tFinalTime - pCurrentEntry->tStartTime,
															pCurrentEntry->iFinalInputTraffic + pCurrentEntry->iFinalOutputTraffic)) {
		lprintf("Error updating users entry for user %s...\n", pCurrentEntry->szLogin);
		delete pUsersDatabase;
		delete pHronoDatabase;
		return false;
	}
// Add entry to the hrono database		
	if (false == pHronoDatabase->AddRecord(pCurrentEntry->szLogin,
														pCurrentEntry->szDevice,
														pCurrentEntry->szHost,
														pCurrentEntry->tStartTime,
														pCurrentEntry->tFinalTime,
														pCurrentEntry->iFinalInputTraffic,
														pCurrentEntry->iFinalOutputTraffic)) {
		lprintf("Error adding hrono entry for user %s...\n", pCurrentEntry->szLogin);
		delete pUsersDatabase;
		delete pHronoDatabase;
		return false;
	}
		
	DeleteAccountRule(pCurrentEntry->szHost);
		
	delete pUsersDatabase;
	delete pHronoDatabase;
	lprintf("User %s logged off due to an error condition...\n", pCurrentEntry->szLogin);
	return true;
}

void CheckCurrent(void)
{
	CCurrentDatabase *pCurrentDatabase = NULL;
	char *pszMessage = NULL;

	if (NULL == (pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage))) {
		lprintf("Memory allocation error checking current database...\n");
		return;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		delete pCurrentDatabase;
		return;
	}
	
	SCurrentEntry strCurrentEntry;
	
	int iRecordsToDelete = 0;
	char **pRecordsToDelete = NULL;
	char **pNewRecordsToDelete;
	
	if (pCurrentDatabase->GetFirstRecord(&strCurrentEntry))
		do {
			if (0 == strcmp(strCurrentEntry.szDevice, "-")) {
// Blocked host
				EmergencyUserDown(&strCurrentEntry);
				pNewRecordsToDelete = (char **)realloc(pRecordsToDelete, sizeof(char *) * (iRecordsToDelete + 1));
				char *pszHost = strdup(strCurrentEntry.szHost);
				if (NULL == pNewRecordsToDelete || NULL == pszHost) {
					lprintf("Memory allocation error in initial current database scanning...\n");
				} else {
					pRecordsToDelete = pNewRecordsToDelete;
					pRecordsToDelete[iRecordsToDelete] = pszHost;
					iRecordsToDelete += 1;
				}
			} else if (0 == strcmp(strCurrentEntry.szDevice, "*")) {
// LAN host
				EmergencyUserDown(&strCurrentEntry);
				pNewRecordsToDelete = (char **)realloc(pRecordsToDelete, sizeof(char *) * (iRecordsToDelete + 1));
				char *pszHost = strdup(strCurrentEntry.szHost);
				if (NULL == pNewRecordsToDelete || NULL == pszHost)
					lprintf("Memory allocation error in initial current database scanning...\n");
				else
					pRecordsToDelete = pNewRecordsToDelete;
				pRecordsToDelete[iRecordsToDelete] = pszHost;
				iRecordsToDelete += 1;
			} else {
// Dial up host
			}
		} while (pCurrentDatabase->GetNextRecord(&strCurrentEntry));
	
	for (int i = 0; i < iRecordsToDelete; i++) {
		pCurrentDatabase->DeleteRecord(pRecordsToDelete[i]);
		delete pRecordsToDelete[i];
	}
	
	delete pCurrentDatabase;
	delete pRecordsToDelete;
}

bool KillUser(char *pszLogin, char *pszDevice, char *pszHostToKill)
{
	register int i, j;
	static char szBuf[256];
	char *pszHost;
	
	char *pszAllowedHosts = pConfig->GetString("Trusted", "trusted_hosts", "");
	for (i = j = 0; pszAllowedHosts[i] != '\0'; i++)
	   if (NULL == strchr(" \t\x0a\x0d", pszAllowedHosts[i]))
		   szBuf[j++] = pszAllowedHosts[i];
	szBuf[j] = '\0';

	if (NULL == (pszHost = strtok(szBuf, ",;")))
		return false;
	do {
		int newsockfd = 0;
		
		if ((newsockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
			lprintf("socket error: %s\n", strerror(errno));
			continue;
		}

		struct hostent *pHost = gethostbyname(pszHost);
		if (NULL == pHost) {
			lprintf("resolving error: %s\n", pszHost);
			close(newsockfd);
			continue;
		}
	
		struct sockaddr_in client;
	
		memset(&client, 0, sizeof(client));
		client.sin_family = AF_INET;
		client.sin_addr.s_addr = *((unsigned long int *)(pHost->h_addr));
		client.sin_port = htons(pConfig->GetInteger("Trusted", "port", 1097));
	
		if (connect(newsockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
			lprintf("connect error: %s\n", strerror(errno));
			close(newsockfd);
			continue;
		}
		
		unsigned char c;
		
		read_byte(newsockfd, &c);
		if (WELCOME != c) {
			lprintf("Protocol error with mugger...\n");
			close(newsockfd);
			continue;
		}
		
		write_string(newsockfd, pszLogin);
		write_string(newsockfd, pszDevice);
		write_string(newsockfd, pszHostToKill);
		
		read_byte(newsockfd, &c);		
		if (OK != c)
			lprintf("Error killing user...\n");
		
		close(newsockfd);
	} while (NULL != (pszHost = strtok(NULL, ",;")));
	
	return true;
}