/*
** Copyright (c) 1999, 2000
** Adel I. Mirzazhanov. All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 
**     1.Redistributions of source code must retain the above copyright notice,
**       this list of conditions and the following disclaimer. 
**     2.Redistributions in binary form must reproduce the above copyright
**       notice, this list of conditions and the following disclaimer in the
**       documentation and/or other materials provided with the distribution. 
**     3.The name of the author may not be used to endorse or promote products
**       derived from this software without specific prior written permission. 
** 		  
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR  ``AS IS'' AND ANY EXPRESS
** OR IMPLIED WARRANTIES, INCLUDING,  BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN  NO  EVENT  SHALL THE AUTHOR BE LIABLE FOR ANY
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO,  PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES;  LOSS OF USE,  DATA,  OR  PROFITS;  OR BUSINESS
** INTERRUPTION)  HOWEVER  CAUSED  AND  ON  ANY  THEORY OF LIABILITY,
** WHETHER  IN  CONTRACT,   STRICT   LIABILITY,  OR  TORT  (INCLUDING
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
** Main Module of apg programm
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#ifdef CLISERV
#  include <sys/socket.h>
#  include <netinet/in.h>
#  include <arpa/inet.h>
#  include <syslog.h>
#  define MAXSOCKADDDR 128
#endif

#include "owntypes.h"
#include "pronpass.h"
#include "randpass.h"
#include "restrict.h"
#include "rnd.h"
#include "errs.h"

#define TRUE	1
#define FALSE	0

#ifndef CLISERV
UINT64 get_user_seq (void);
void print_help (void);
#endif

int main (int argc, char *argv[]);
void checkopt(char *opt);

int
main (int argc, char *argv[])
{
 int i = 0;
 int restrict_res = 0;
 
 char *pass_string;
 char *hyph_pass_string;
 time_t tme;
 
 
 int option = 0;                         /* programm option                 */

 int algorithm = 0;                      /* algorithm for generation        */
 int restrictions_present = FALSE;       /* restrictions flag               */
 char *restrictions_file;                /* dictionary file name            */
 unsigned int pass_mode = 0;	         /* password generation mode        */
 unsigned int pass_mode_present = FALSE; /* password generation mode flag   */
 USHORT min_pass_length = 6;             /* min password length             */
 USHORT max_pass_length = 8;             /* max password length             */
 int number_of_pass = 6;                 /* number of passwords to generate */
 UINT64 user_defined_seed = 0;           /* user defined random seed        */
 int user_defined_seed_present = FALSE;  /* user defined random seed flag   */

#ifdef CLISERV
 socklen_t len;
 struct sockaddr_in *cliaddr;
 char delim[2]={0x0d,0x0a};
 char *out_pass;
 char *peer_ip_unknown = "UNKNOWN";
 char *peer_ip;

 openlog(argv[0], LOG_PID, LOG_DAEMON);
 cliaddr = (struct sockaddr_in *)calloc(1,MAXSOCKADDDR);
 len = MAXSOCKADDDR;
 if( getpeername(0, (struct sockaddr *)cliaddr, &len) != 0)
  {
   err_sys("getpeername");
   peer_ip = peer_ip_unknown;
  }
 else
  {
   peer_ip = inet_ntoa(cliaddr->sin_addr);
  }
 syslog (LOG_INFO, "password generation request from %s.%d\n", peer_ip, htons(cliaddr->sin_port));
#endif

 /*
 ** Analize options
 */
#ifndef CLISERV
 while ((option = getopt (argc, argv, "SNCLa:r:sn:m:x:hv")) != -1)
#else
 while ((option = getopt (argc, argv, "SNCLa:r:n:m:x:v")) != -1)
#endif
  {
   switch (option)
    {
     case 'S': /* special symbols required */
      pass_mode = pass_mode | S_SS;
      pass_mode_present = TRUE;
      break;
     case 'N': /* numbers required */
      pass_mode = pass_mode | S_NB;
      pass_mode_present = TRUE;
      break;
     case 'C': /* capital letters required */
      pass_mode = pass_mode | S_CL;
      pass_mode_present = TRUE;
      break;
     case 'L': /* small letters required */
      pass_mode = pass_mode | S_SL;
      pass_mode_present = TRUE;
      break;
     case 'a': /* algorithm specification */
      checkopt(optarg);
      algorithm = atoi (optarg);
      break;
     case 'r': /* restrictions */
      restrictions_present = TRUE;
      restrictions_file = optarg;
      break;
#ifndef CLISERV
     case 's': /* user random seed required */
      user_defined_seed = get_user_seq ();
      user_defined_seed_present = 1;
      break;
#endif
     case 'n': /* number of password specification */
      checkopt(optarg);
      number_of_pass = atoi (optarg);
      break;
     case 'm': /* min password length */
      checkopt(optarg);
      min_pass_length = (USHORT) atoi (optarg);
      break;
     case 'x': /* max password length */
      checkopt(optarg);
      max_pass_length = (USHORT) atoi (optarg);
      break;
#ifndef CLISERV
     case 'h': /* print help */
      print_help ();
      return (0);
#endif
     case 'v': /* print version */
      printf ("APG (Automated Password Generator)");
      printf ("\nversion 1.0");
      printf ("\nCopyright (c) 1999, 2000 Adel I. Mirzazhanov\n");
      return (0);
     default: /* print help end exit */
#ifndef CLISERV
      print_help ();
#endif
      exit (-1);
    }
  }
 if (pass_mode_present != TRUE)
    pass_mode = S_SS | S_NB | S_CL | S_SL;
 if( (tme = time(NULL)) == ( (time_t)-1))
    err_sys("time");
 if (user_defined_seed_present != TRUE)
    x917cast_setseed ( (UINT64)tme);
 else
    x917cast_setseed (user_defined_seed ^ (UINT64)tme);

 /* main code section */
 
 /*
 ** reserv space for password and hyphenated password and report of errors
 */
 if ( (pass_string = (char *)calloc (1, (size_t)(max_pass_length + 1)))==NULL ||
      (hyph_pass_string = (char *)calloc (1, (size_t)(max_pass_length*2)))==NULL)
      err_sys_fatal("calloc");
      
#ifdef CLISERV
 if ( (out_pass = (char *)calloc(1, (size_t)(max_pass_length*3 + 4))) == NULL)
      err_sys_fatal("calloc");
#endif
 /*
 ** generate required amount of passwords using specified algorithm
 ** and check for restrictions if specified with command line parameters
 */
 while (i < number_of_pass)
  {
   if (algorithm == 0)
    {
     if (gen_pron_pass(pass_string, hyph_pass_string,
                       min_pass_length, max_pass_length) == -1)
        err_app_fatal("apg","wrong password length parameter");

     if (restrictions_present == 1)
       {
        restrict_res = check_pass(pass_string, restrictions_file);
        switch (restrict_res)
	  {
	  case 0:
#ifndef CLISERV
	    fprintf (stdout, "%s (%s)\n", pass_string, hyph_pass_string);
	    fflush (stdout);
#else
            snprintf(out_pass, max_pass_length*3 + 4,
	             "%s (%s)", pass_string, hyph_pass_string);	    
	    write (0, (void*) out_pass, strlen(out_pass));
	    write (0, (void*)&delim[0],2);
#endif
	    i++;
	    break;
	  case 1:
	    break;
	  case -1:
	    err_sys_fatal ("check_pass");
	  default:
	    break;
	  } /* switch */
       }
     else /* if (restrictions_present == 0) */
       {
#ifndef CLISERV
        fprintf (stdout, "%s (%s)\n", pass_string, hyph_pass_string);
	fflush (stdout);
#else
        snprintf(out_pass, max_pass_length*3 + 4,
	         "%s (%s)", pass_string, hyph_pass_string);
	write (0, (void*) out_pass, strlen(out_pass));
	write (0, (void*)&delim[0],2);
#endif
	i++;
       }
    } /* end of if (algorithm == 0) */
   else if (algorithm == 1)
    {
     if (gen_rand_pass(pass_string, min_pass_length,
                       max_pass_length, pass_mode) == -1)
        err_app_fatal("apg","wrong password length parameter");
     if (restrictions_present == 1)
       {
        restrict_res = check_pass(pass_string, restrictions_file);
        switch (restrict_res)
	  {
	  case 0:
#ifndef CLISERV
	    fprintf (stdout, "%s\n", pass_string);
	    fflush (stdout);
#else
	    write (0, (void*)pass_string, strlen(pass_string));
	    write (0, (void*)&delim[0],2);
#endif
	    i++;
	    break;
	  case 1:
	    break;
	  case -1:
	    err_sys_fatal ("check_pass");
	  default:
	    break;
	  } /* switch */
       }
     else /* if (restrictions_present == 0) */
       {
#ifndef CLISERV
        fprintf (stdout, "%s\n", pass_string);
	fflush (stdout);
#else
	write (0, (void*)pass_string, strlen(pass_string));
	write (0, (void*)&delim[0],2);
#endif
	i++;
       }
    } /* end of if (algorithm == 1) */
   else
     err_app_fatal ("apg","wrong algorithm type");

   restrict_res = 0;
  } /* end of while (i <= number_of_pass) */
 free((void*)pass_string);
 free ((void*)hyph_pass_string);
#ifdef CLISERV
 free ((void *)out_pass);
 free ((void *)cliaddr);
 close (0);
 closelog();
#endif
 return(0);
} /* end of main */

#ifndef CLISERV
/*
** Routine that gets user random sequense and generates
** sutable random seed according to it
*/
UINT64
get_user_seq (void)
{
 char * seq;
 UINT64 prom = 0L;
 
 printf ("\nPlease enter some random data (only first 8 are significant)\n");
 seq = (char *)getpass("(eg. your old password):>");
 if (strlen(seq) < 8)
  bcopy((void *)seq, (void *)&prom, (int)strlen(seq));
 else
  bcopy((void *)seq, (void *)&prom, 8);
 return (prom);
}
void
print_help (void)
{
 printf ("\napg   Automated Password Generator\n");
 printf ("        Copyright (c) Adel I. Mirzazhanov\n");
 printf ("\napg   [-a algorithm] [-r file] [-S] [-C] [-L]\n");
 printf ("      [-N] [-n num_of_pass] [-m min_pass_len]\n");
 printf ("      [-x max_pass_len] [-s] [-h]\n");
 printf ("\n-S -N -C -L     password modes\n");
 printf ("-r file         apply dictionary check against file\n");
 printf ("-a algorithm    choose algorithm\n");
 printf ("                 1 - random password generation according to\n");
 printf ("                     password modes\n");
 printf ("                 0 - pronounceable password generation\n");
 printf ("-n num_of_pass  generate num_of_pass passwords\n");
 printf ("-m min_pass_len minimum password length\n");
 printf ("-x max_pass_len maximum password length\n");
 printf ("-s              ask user for a random seed for password\n");
 printf ("                generation\n");
 printf ("-h              print this help screen\n");
 printf ("-v              print version information\n");
}
#endif

void
checkopt(char *opt)
{
 int i;
 
 for(i=0; i < strlen(opt);i++)
  if(opt[i] != '0' && opt[i] != '1' && opt[i] != '2' && opt[i] != '3' &&
     opt[i] != '4' && opt[i] != '5' && opt[i] != '6' && opt[i] != '7' &&
     opt[i] != '8' && opt[i] != '9')
      err_app_fatal ("checkopt", "wrong option format");
}
