/*******************************************************************/
/*                                                                 */
/* Programm: xwprint                                               */
/*           Plug-in for the gimp                                  */
/*           printing via xwgui / xw_tools                         */
/*                                                                 */
/* File:     xwprint.c                                             */
/*                                                                 */
/* CREATION: jjsa   (Jims-Jams-Software-Application)               */
/*           17-APR-1999                                           */
/*                                                                 */
/* Content:  all necessary functions for the plug in               */
/*           and a few function due to a bug from Suse 6.0 dist    */
/*                                                                 */
/* CHANGES:                                                        */
/*           24-APR-1999 bug for reading of rc file eliminated     */
/*                       better control of pipe stream and return  */
/*                       code of popen read                        */
/*           25-APR-1999 internationalisation added                */
/*                                                                 */
/*******************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <strings.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#include <X11/Xlib.h>
#include "gtk/gtk.h"
#include "libgimp/gimp.h"

static char defaultCommand[] = "/usr/local/bin/xwgui -gimp%dx%d";

static void query  (void);
static void run    (char    *name,
                    int      nparams,
                    GParam  *param,
                    int     *nreturn_vals,
                    GParam **return_vals);
static void rc_load(char    *command);

static char *msg(char*);

typedef struct messages_s
{
   char *key;
   char *message;
} message_t;
      
       
static message_t mess[] =
{
   { "noconnect",     "xwprint: xwgui not found" },
   { "nomemory",      "xwprint: can't alloc memory" },
   { "printfail",     "xwprint: printing failed (status %d)" },
   { "nointeractive", "xwprint: allows only interactive invocation" },
   { "printing",      "xwprint: printing" },
   { NULL,             NULL }
};

GPlugInInfo PLUG_IN_INFO =
{
   NULL,    /* init_proc */
   NULL,    /* quit_proc */
   query,   /* query_proc */
   run,     /* run_proc */
};

/* on suse 6.0 libgimp..so claim the following fucnctions ! */
#ifdef G_FUNC_ERROR
#undef g_debug()
#undef g_message()
#undef g_warning()
#undef g_error()
g_debug() {}
g_message(){}
g_warning(){}
g_error(){}
#endif

/*******************************************************************/ 
/*                                                                 */ 
/* NAME:      sigpipe                                              */ 
/*                                                                 */ 
/* FUNCTION:  sigfpe may occurs if xwgui fail, tarp it             */
/*                                                                 */
/* INPUT:     -                                                    */
/*                                                                 */
/* OUTPUT:    -                                                    */ 
/*                                                                 */ 
/* RETURN:    -                                                    */ 
/*                                                                 */ 
/* REMARKS:   -                                                    */ 
/*                                                                 */ 
/*******************************************************************/ 

static void sigpipe()
{

}
      
int main(int  argc, char *argv[])
{
   return (gimp_main(argc, argv));
}

/*******************************************************************/ 
/*                                                                 */ 
/* NAME:      query                                                */ 
/*                                                                 */ 
/* FUNCTION:  install the plug-in                                  */
/*                                                                 */
/* INPUT:     -                                                    */
/*                                                                 */
/* OUTPUT:    -                                                    */ 
/*                                                                 */ 
/* RETURN:    -                                                    */ 
/*                                                                 */ 
/* REMARKS:   -                                                    */ 
/*                                                                 */ 
/*******************************************************************/ 

static void query ()
{
   static GParamDef args[] =
   {
      { PARAM_INT32,    "run_mode", "Interactive, [non-interactive]" },
      { PARAM_IMAGE,    "image",    "Input image" },
      { PARAM_DRAWABLE, "drawable", "Input drawable" }
   };

   static int nargs = sizeof (args) / sizeof (args[0]);

   gimp_install_procedure ("xwprint",
                           "print via xwgui / xw_tools",
                           "",
                           "Jean-Jacques Sarton / Stefan Kraus",
                           "Jean-Jacques Sarton / Stefan Kraus",
                           "25th April 1999",
                           "<Image>/File/Print via xwgui, xwtools",
                           "RGB*,GRAY*,INDEXED*",
                           PROC_PLUG_IN,
                           nargs, 0,
                           args, NULL);
}

/*******************************************************************/ 
/*                                                                 */ 
/* NAME:      run                                                  */ 
/*                                                                 */ 
/* FUNCTION:  the main job                                         */
/*                                                                 */
/* INPUT:     as required by the gimp                              */
/*                                                                 */
/* OUTPUT:    -                                                    */ 
/*                                                                 */ 
/* RETURN:    -                                                    */ 
/*                                                                 */ 
/* REMARKS:   -                                                    */ 
/*                                                                 */ 
/*******************************************************************/ 

static void
run (char    *name,
     int      nparams,
     GParam  *param,
     int     *nreturn_vals,
     GParam **return_vals)
{
   static GParam  values[1];
   GDrawable     *drawable;
   GRunModeType   run_mode;
   guchar        *cmap;          /* Colormap (indexed images only) */
   guchar        *in;
   int            ncolors;        /* Number of colors in colormap */
   int            i,j;
   GPixelRgn	  rgn;		/* Image region */
   char          *filename;
   int            imageType;
   FILE          *output;
   char           command[1024];
   char           tmp[1024];
   int            total = 0;  
   int            written;
   guchar        *out;
   guchar        *t;
   extern Display *gdk_display; 
   command[0] = '\0';
   tmp[0]     = '\0';

   run_mode = param[0].data.d_int32;
  
   drawable = gimp_drawable_get(param[2].data.d_drawable);

   filename = gimp_image_get_filename(param[1].data.d_image);
   
   *nreturn_vals           = 1;
   *return_vals            = values;
   values[0].type          = PARAM_STATUS;
   values[0].data.d_status = STATUS_SUCCESS;

   switch (run_mode)
   {
      case RUN_INTERACTIVE: 
      {
         gchar **argv = NULL;
         gint  argc = 0;
                         
         argc = 1;
         argv = g_new (gchar *, 1);
         argv[0] = g_strdup ("xwprint");
         gtk_init (&argc, &argv);
         gtk_rc_parse (gimp_gtkrc ());
         
         rc_load(tmp);
         if ( *tmp == '\0' )
         {
            strcpy(tmp, defaultCommand);
            /*break;*/
         }
         sprintf(command, tmp, drawable->width, drawable->height);

         output = popen(command, "w");
         if ( output == NULL )
         {
            /* noconnect */
            gimp_message (msg("noconnect"));
            values[0].data.d_status = STATUS_EXECUTION_ERROR;
            break;
         }
         signal(SIGPIPE, sigpipe);

         imageType = gimp_image_base_type(param[1].data.d_image);
         
         if ( imageType == INDEXED)
         {
            cmap = gimp_image_get_cmap(param[1].data.d_image, &ncolors);
         }
         else
         {
            cmap    = NULL;
            ncolors = 0;
         }
                  
         in  = g_malloc(drawable->width * drawable->bpp);
         if ( in == NULL )
         {
            gimp_message (msg("nomemory"));
            values[0].data.d_status = STATUS_EXECUTION_ERROR;
            break;
         }
         out = g_malloc(drawable->width * 3);
         if ( out == NULL )
         {
            g_free(in);
            gimp_message (msg("nomemory"));
            values[0].data.d_status = STATUS_EXECUTION_ERROR;
            break;
         }
         
         gimp_pixel_rgn_init(&rgn, drawable, 0, 0,
                             drawable->width,
                             drawable->height,
                             FALSE, FALSE);
         
         /* make ppm file header */

         switch(imageType)
         {
            case INDEXED:
            case RGB_IMAGE:
               sprintf(out,"P6\n# %s\n%d %d\n255\n",
                      filename,drawable->width,drawable->height);
               break;
            case GRAY:
               sprintf(out,"P5\n# %s\n%d %d\n255\n",
                       filename,drawable->width,drawable->height);

               break;
         }

         written = write(fileno(output), out, strlen(out));

         for ( i = 0; i < drawable->height && written > 0 ;i++ )
         {
            if ( (total >= 8192) & (total < 10000) )
            {
               total = 10000;
               gimp_progress_init(msg("printing"));
            }  
            if ( /*(i & 15) == 0 &&*/ total >= 10000)
            {
               gimp_progress_update((double)i / (double)drawable->height);
	       XFlush(gdk_display);
	       XSync(gdk_display, 0);
            }

            gimp_pixel_rgn_get_row(&rgn, in, 0, i, drawable->width);
            t = out;
            switch(imageType)
            {
               guchar *r, *g, *b, *a;
               case INDEXED:
                  g = in;
                  if ( drawable->bpp == 1 )
                  {
                     for ( j = 0; j < drawable->width; j++, g++ )
                     {
                        *t++   = cmap[*g*3];
                        *t++   = cmap[*g*3+1];
                        *t++   = cmap[*g*3+2];
                        total += 3;
                     }
                  }
                  else
                  {
                     for ( j = 0; j < drawable->width; j++, g += drawable->bpp )
                     {
                        *t++   = cmap[*g*3]   * g[1] / 255 + 255 - g[1];
                        *t++   = cmap[*g*3+1] * g[1] / 255 + 255 - g[1];
                        *t++   = cmap[*g*3+2] * g[1] / 255 + 255 - g[1];
                        total += 3;
                     }
                  }
                  break;
               case GRAY:
                  g = in;
                  if ( drawable->bpp == 1 )
                  {
                     for ( j = 0; j < drawable->width; j++, g++ )
                     {
                        *t++ = *g;
                        total++;
                     }
                  }
                  else
                  {
                     for ( j = 0; j < drawable->width; j++, g += drawable->bpp )
                     {
                        *t++ =(g[0] * g[1] / 255) + 255 - g[1];
                        total++;
                     }   
                  }
                  break;
               case RGB:
                  if ( drawable->bpp == 3 )
                  {
                     r = in;
                     g = in+1;
                     b = in+2;
                     for ( j = 0; j < drawable->width; j++)
                     {
                        *t++ = *r;
                        *t++ = *g;
                        *t++ = *b;
                        r += 3;
                        g += 3;
                        b += 3;
                        total += 3;
                     }
                  }
                  else
                  {
                     r = in;
                     g = in+1;
                     b = in+2;
                     a = in+3;
                     for ( j = 0; j < drawable->width; j++)
                     {
                        *t++ = *r**a/255+255-*a;
                        *t++ = *g**a/255+255-*a;
                        *t++ = *b**a/255+255-*a;
                        r   += drawable->bpp;
                        g   += drawable->bpp;
                        b   += drawable->bpp;
                        a   += drawable->bpp;
                        total += 3;
                     }
                 }
                 break;
             }
             {
                int len = t-out;
                int idx = 1;
                written = 0;
                while ( written != len && idx > 0)
                {
                   idx = write(fileno(output),out+written,len-written);
                   if ( idx < 0 )
                   {
                      written = 0;
                      break;
                   }
                   written += idx;
	           XFlush(gdk_display);
	           XSync(gdk_display, 0);
                }
             }
         }
#ifdef DEBUG
         if ( written == 0 )
         {
            fprintf(stderr,"Error %s\n",strerror(errno));
         }
#endif
         g_free(in);
         g_free(out);
         written = pclose(output);
#ifdef DEBUG
         fprintf(stderr,"prog end with %d\n", written>>8);
#endif
         if ( written >> 8 != 0 )
         {
	    char text[256];
	    sprintf(text, msg("printfail"),
	            written >> 8);
            gimp_message (text);
            values[0].data.d_status = STATUS_EXECUTION_ERROR;
         }
      }
      break;
      
      case RUN_WITH_LAST_VALS:
      case RUN_NONINTERACTIVE:
         gimp_message (msg("nointeractive"));
         values[0].data.d_status = STATUS_CALLING_ERROR;
         break;
      
      default:
         break;
   }  
   gimp_drawable_detach(drawable);
	values[0].data.d_status = STATUS_SUCCESS;
}

/*******************************************************************/ 
/*                                                                 */ 
/* NAME:      rc_load                                              */ 
/*                                                                 */ 
/* FUNCTION:  read our own rc file xwprintrc                       */
/*                                                                 */
/* INPUT:     -                                                    */
/*                                                                 */
/* OUTPUT:    char *command                                        */ 
/*                                                                 */ 
/* RETURN:    -                                                    */ 
/*                                                                 */ 
/* REMARKS:   the returned string contain the command to be passed */
/*            to popen()                                           */
/*                                                                 */ 
/*******************************************************************/ 

static void rc_load(char *command)
{
   FILE         *fp;
   static char   key[] = "COMMAND:";
   static char   line[1024];
   char         *s, *t;
   char         *lang = NULL;
   int           len  = 3;  
   message_t    *m;
   
   
   lang = getenv("LANG");

   if ( lang != NULL )
   {
      if ( *lang == 'e' && lang[1] == 'n' )
         lang = strdup("en.");
      else if ( *lang == 'd' && lang[1] == 'e' )
         lang = strdup("de.");
      else if ( *lang == 'f' && lang[1] == 'r' )
         lang = strdup("fr.");
      else
        lang = strdup("en.");
   }
   else if ( lang = getenv("LC_CTYPE") )
   {
      if ( lang != NULL )
      {
         if ( *lang == 'e' && lang[1] == 'n' )
            lang = strdup("en.");
         else if ( *lang == 'd' && lang[1] == 'e' )
            lang = strdup("de.");
         else if ( *lang == 'f' && lang[1] == 'r' )
            lang = strdup("fr.");
         else
            lang = strdup("en.");
      }
   }
   else
   {   
      lang = strdup("en.");
   }

   if (( s = getenv("HOME")) == NULL)
      strcpy(line, "/.gimp/xwprintrc");
   else
      sprintf(line, "%s/.gimp/xwprintrc", s);

   if ((fp = fopen(line, "r")) != NULL)
   {
      while (fgets(line, sizeof(line), fp) != NULL)
      {
         if ( strncmp(line,key, strlen(key)) == 0 )
         {
            s = line + strlen(key);
            while ( *s && isspace(*s) )
               s++;
            t = s;
            while (*t)
               t++;
            while ( t > s && isspace(*t) )
               t--;
            if ( s != t )
            {
               *t = '\0';
               strcpy(command, s);
            }
         }
         else if ( strncmp(line,"LANG:", 5) == 0 )
         {
            s = line + strlen("LANG:");
            while ( *s && isspace(*s) )
               s++;
            t = s;
            while (*t && ! isspace(*t) )
               t++;

            *t++ = '.';
            *t   = '\0';
            
            if ( lang )
            {
               free(lang);
               lang = strdup(s);
               len  = strlen(s);
            }
         }
         else if ( strncmp(lang, line, len ) == 0 )
         {
            m = mess;
            s = line + len;
            while ( m->key != NULL )
            {
               if ( strncmp(m->key, s, strlen(m->key)) == 0 )
               {
                  s += strlen(m->key);
                  while ( *s && isspace(*s) )
                     s++;
                  t = s;
                  while ( *t && *t != '\n' )
                     t++;
                  *t = '\0';
                  m->message = strdup(s);
               }
               m++;
            }
         }
            
      }
      fclose(fp);
   }   
}

/*******************************************************************/ 
/*                                                                 */ 
/* NAME:      mess                                                 */ 
/*                                                                 */ 
/* FUNCTION:  get message text from data base                      */
/*                                                                 */
/* INPUT:     -                                                    */
/*                                                                 */
/* OUTPUT:    char *key                                            */ 
/*                                                                 */ 
/* RETURN:    cgar *message                                        */ 
/*                                                                 */ 
/* REMARKS:                                                        */
/*                                                                 */ 
/*******************************************************************/ 

char *msg(char *key)
{
   message_t *m;
   
   m = mess;
   while ( m->key )
   {
      if ( strcmp(m->key, key) == 0 )
         return m->message;
      m++;
   }
   return "unknown message !";
}


   
