/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "xpm.h"


typedef unsigned char uc;

typedef struct cmap_s
{
   uc r;
   uc g;
   uc b;
} cmap_t;

static char *prgName = "xwGUI2";

static int initialized = 0;

static int levels   = 216;
static int colchars =   2;
static int steps1   =   6;
static int steps2   =  36;

static long cmap[216];
static char str[216][3];

static void mkStr8( char str[216][3] );
static void mkStr27( char str[216][3] );
static void mkStr64( char str[216][3] );
static void mkStr125( char str[216][3] );
static void mkStr216( char str[216][3] );
static void mkCmap8(long  *cmap);
static void mkCmap27(long  *cmap);
static void mkCmap64(long  *cmap);
static void mkCmap125(long  *cmap);
static void mkCmap216(long  *cmap);
static void correctErrorBuf(int *eBuf, int w, int l);
static void dither_line_FSM(unsigned char *src, unsigned char *dest, int *err, int y, int width);
static char pbm_getc (FILE *file);
static int  pbm_getint (FILE *file);
static int  readPpmHead(FILE *fp, int *w, int *h, int *level);

/*
*************************************************************************/
/* Function  setRenderQuality()                                          */
/*           set the internal variables which set the number of collors  */
/*           used for rendering                                          */
/*                                                                       */
/*                                                                       */
/* Input:    int   renderQuality  ( 1 to 5 )                             */
/*                                                                       */
/* Return    -                                                           */
/*                                                                       */
/*************************************************************************/

void setRenderQuality(int renderQuality)
{
   switch (renderQuality)
   {
      case 1:  levels =   8; mkCmap8(cmap);   mkStr8(str);   colchars = 1;
               steps1 = 2; steps2 = 4;
      break;
      case 2:  levels =  27; mkCmap27(cmap);  mkStr27(str);  colchars = 1;
               steps1 = 3; steps2 = 9;
      break;
      case 3:  levels =  64; mkCmap64(cmap);  mkStr64(str);  colchars = 2;
               steps1 = 4; steps2 = 16;
      break;
      case 4:  levels = 125; mkCmap125(cmap); mkStr125(str); colchars = 2;
               steps1 = 5; steps2 = 25;
      break;
      default: levels = 216; mkCmap216(cmap); mkStr216(str); colchars = 2;
               steps1 = 6; steps2 = 36;
      break;
   }
}

/*************************************************************************/
/* Function  correctErrorBuf()                                           */
/*           randomize a little bit the error values so the aspect       */
/*           will be a little bit better.                                */
/*                                                                       */
/*                                                                       */
/* Input:    int  *eBuf error buffer witf RGB tripplet                   */
/*           int   w    the number pixel                                 */
/*           int   l    the actual line                                  */
/*                                                                       */
/* Return    -                                                           */
/*                                                                       */
/*************************************************************************/

static void correctErrorBuf(int *eBuf, int w, int l)
{
   int i = l & 3;
   
#define RVAL 7

   while (w--)
   {
      switch(i)
      {
         case 0:
           *eBuf++ +=  RVAL;
           *eBuf++ +=    0;
           *eBuf++ += -RVAL;
           i++;
           break;
         case 1:
           *eBuf++ +=    0;
           *eBuf++ += -RVAL;
           *eBuf++ +=  RVAL;
           i++;
           break;
         default:
           *eBuf++ += -RVAL;
           *eBuf++ +=  RVAL;
           *eBuf++ +=    0;
           i = 0;
      }
   }
   
   return;
}

/*************************************************************************/
/* Function  dither_lineFSM()                                            */
/*           perform the error diffusion ditering                        */
/*                                                                       */
/* Input:   unsigned char *src    3 bytes / pixel, order RGB             */
/*          int            y      line number                            */
/*          int            witdth width in pixel of the line             */
/*                                                                       */
/* Output:  unsigned char *dest                                          */
/*                                                                       */
/* Return   -                                                            */
/*                                                                       */
/*************************************************************************/

/* a few macros for the rendering */
#define VAL     v = *src + *e + *h;

#define DITHER if ( levels == 216 ) \
               { \
                  if ( v > 234 ) \
                  { \
                     *d  =  5; \
                     v  -= 0xff; \
                  } \
                  else if ( v > 183 ) \
                  { \
                     *d  =  4; \
                     v  -= 0xcc; \
                  } \
                  else if ( v > 128 ) \
                  { \
                     *d  =  3; \
                     v  -= 0x99; \
                  } \
                  else if ( v > 77 ) \
                  { \
                     *d  =  2; \
                     v  -= 0x66; \
                  } \
                  else if ( v > 26 ) \
                  { \
                     *d  =  1; \
                     v  -= 0x33; \
                  } \
                  else \
                  { \
                     *d = 0; \
                  } \
               }\
               else if ( levels == 125 ) \
               { \
                  if ( v > 223 ) \
                  { \
                     *d  =  4; \
                     v  -= 255; \
                  } \
                  else if ( v > 159 ) \
                  { \
                     *d  =  3; \
                     v  -= 192; \
                  } \
                  else if ( v > 92 ) \
                  { \
                     *d  =  2; \
                     v  -= 127; \
                  } \
                  else if ( v > 31 ) \
                  { \
                     *d  =  1; \
                     v  -= 63; \
                  } \
                  else \
                  { \
                     *d = 0; \
                  } \
               } \
               else if ( levels == 64 ) \
               { \
                  if ( v > 213 ) \
                  { \
                     *d  =  3; \
                     v  -= 255; \
                  } \
                  else if ( v > 128 ) \
                  { \
                     *d  =  2; \
                     v  -= 170; \
                  } \
                  else if ( v > 43 ) \
                  { \
                     *d  =  1; \
                     v  -= 85; \
                  } \
                  else \
                  { \
                     *d = 0; \
                  } \
               } \
               else if ( levels == 27 ) \
               { \
                  if ( v > 192 ) \
                  { \
                     *d  =  2; \
                     v  -= 255; \
                  } \
                  else if ( v > 63 ) \
                  { \
                     *d  =  1; \
                     v  -= 127; \
                  } \
                  else \
                  { \
                     *d = 0; \
                  } \
               } \
               else if ( levels == 8 ) \
               { \
                  if ( v > 127 ) \
                  { \
                     *d  =  1; \
                     v  -= 255; \
                  } \
                  else \
                  { \
                     *d = 0; \
                  } \
               }

#define ERROR *e  = v >> 1; \
              *h  = v - (v >> 1);
 
static void dither_line_FSM(unsigned char *src, unsigned char *dest, int *eBuf, int y, int width)
{
   unsigned char  dBuf[3];
   int            hBuf[3] = { 0, 0, 0 };
   int           *e;
   unsigned char *d;
   int           *h;
   int            dir;
   int            j;
   int            v;
   /* add / substract a little values in order to get */
   /* a better rendering                              */

   correctErrorBuf(eBuf,width, y);

   if ( y & 1 )
   {
      dir = 1;
      e   = eBuf;
      h   = hBuf;
   }
   else
   {
      src += (width * 3) - 1;
      e    = eBuf + (width * 3) - 1;
      dest += width - 1;
      dir  = -1;
   }

   for ( j = 0; j < width; j++ )
   {
      d = (dir == 1) ? dBuf : dBuf + 2;
      h = (dir == 1) ? hBuf : hBuf + 2;
      
      VAL
      DITHER
      ERROR

      d    += dir;      
      h    += dir;      
      src  += dir;
      e    += dir;

      VAL
      DITHER
      ERROR
      
      d    += dir;      
      h    += dir;
      src  += dir;
      e    += dir;

      VAL
      DITHER
      ERROR

      src  += dir;
      e    += dir;

      /* calculate index */
      *dest = (dBuf[0]*steps2) + (dBuf[1]*steps1) + (dBuf[2]);
      dest += dir;
   }
   
   return;
}

/*************************************************************************/
/* Function  mkCmap216()                                                 */
/*           Build the colormap with 216 entries                         */
/*                                                                       */
/* Output:  long *cmap   pointer to the array of color values            */
/*                                                                       */
/* Return   -                                                            */
/*                                                                       */
/*************************************************************************/

static void mkCmap216(long  *cmap)
{
   int r, g, b;
   int v[6] = { 0, 51, 102, 153, 204, 255 };

   /****************************************/
   /* allocate memory for our cmap         */
   /****************************************/
   for ( r = 0; r < 6; r++ )
   {
      for ( g = 0; g < 6; g++ )
      {
         for ( b = 0; b < 6; b++ )
         {
            /*fl_mapcolor(((r*36)+(g*6)+b)+24,r,g,b);*/
            cmap[(r*36)+(g*6)+b] = (v[r] <<16) | (v[g] << 8) | v[b];
         }
      }
   }
  
   return;
}

static void mkCmap64(long  *cmap)
{
   int r, g, b;
   int v[6] = { 0, 85, 170, 255 };

   /****************************************/
   /* allocate memory for our cmap         */
   /****************************************/
   for ( r = 0; r < 4; r++ )
   {
      for ( g = 0; g < 4; g++ )
      {
         for ( b = 0; b < 4; b++ )
         {
            /*fl_mapcolor(((r*16)+(g*4)+b)+24,r,g,b);*/
            cmap[(r*16)+(g*4)+b] = (v[r] <<16) | (v[g] << 8) | v[b];
         }
      }
   }
  
   return;
}

static void mkCmap125(long  *cmap)
{
   int r, g, b;
   int v[6] = { 0, 63, 127, 192, 255 };

   /****************************************/
   /* allocate memory for our cmap         */
   /****************************************/
   for ( r = 0; r < 5; r++ )
   {
      for ( g = 0; g < 5; g++ )
      {
         for ( b = 0; b < 5; b++ )
         {
            /*fl_mapcolor(((r*25)+(g*5)+b)+24,r,g,b);*/
            cmap[(r*25)+(g*5)+b] = (v[r] <<16) | (v[g] << 8) | v[b];
         }
      }
   }
  
   return;
}

static void mkCmap27(long  *cmap)
{
   int r, g, b;
   int v[6] = { 0, 127, 255 };

   /****************************************/
   /* allocate memory for our cmap         */
   /****************************************/
   for ( r = 0; r < 3; r++ )
   {
      for ( g = 0; g < 3; g++ )
      {
         for ( b = 0; b < 3; b++ )
         {
            /*fl_mapcolor(((r*9)+(g*3)+b)+24,r,g,b);*/
            cmap[(r*9)+(g*3)+b] = (v[r] <<16) | (v[g] << 8) | v[b];
         }
      }
   }
  
   return;
}

static void mkCmap8(long  *cmap)
{
   int r, g, b;
   int v[6] = { 0, 255 };

   /****************************************/
   /* allocate memory for our cmap         */
   /****************************************/
   for ( r = 0; r < 2; r++ )
   {
      for ( g = 0; g < 2; g++ )
      {
         for ( b = 0; b < 2; b++ )
         {
            /*fl_mapcolor(((r*4)+(g*2)+b)+24,r,g,b);*/
            cmap[(r*4)+(g*2)+b] = (v[r] <<16) | (v[g] << 8) | v[b];
         }
      }
   }
  
   return;
}

static char pbm_getc (FILE *file)
{
   int            ich;
   static char    ch;

   ich = getc( file );
   ch = (char) ich;

   if ( ch == '#' )
   {
      do
      {
         ich = getc( file );
         ch = (char) ich;
      }
      while ( ch != '\n' && ch != '\r' );
   }
   return(ch);
}

static int pbm_getint (FILE *file)
{
   char ch;
   int i;

   do
   {
      ch = pbm_getc( file );
   }
   while ( ch == ' ' || ch == '\t' || ch == '\n' || ch == '\r' );

   if ( ch < '0' || ch > '9' )
      return -1;

   i = 0;
   do
   {
      i = i * 10 + ch - '0';
      ch = pbm_getc( file );
   }
   while ( ch >= '0' && ch <= '9' );

   return(i);
}

/*************************************************************************/
/* Function  readPpmHead()                                               */
/*           Read the ppm file header                                    */
/*                                                                       */
/* Input:   FILE *fp                                                     */
/*                                                                       */
/* Output:  int  *w      wide of image                                   */
/*          int  *h      height of image                                 */
/*          int  *level  number of colors for each component             */
/*                                                                       */
/* Return   FALSE on error, TRUE if all is OK                            */
/*                                                                       */
/*************************************************************************/

static int readPpmHead(FILE *fp, int *w, int *h, int *level)
{
   int i, j;
   i = getc( fp );
   if ( i == EOF || i != 'P' )
   {
      fprintf(stderr, "%s: not a pgm or pnm file\n", prgName);
      return FALSE;
   }
   j = getc( fp );
   if ( j == EOF )
   {
      fprintf(stderr, "%s: not a pgm or pnm file\n", prgName);
      return FALSE;
   }

   switch(j)
   {
      case '6': /* RPGM */
         break;
      default:
         fprintf(stderr, "%s: format not supported for\n", prgName);
         return FALSE;
   }
   
   /* read rows and cols number */

   *w     = pbm_getint(fp);
   *h     = pbm_getint(fp);
   *level = pbm_getint(fp);

   return(TRUE); 
}

/*************************************************************************/
/* Function  mkStr216()                                                  */
/*           Build the strings for the xpm indexes                       */
/*                                                                       */
/* Output: char str[216]83]                                              */
/*                                                                       */
/* Return -                                                              */
/*                                                                       */
/*************************************************************************/

static void mkStr216( char str[216][3] )
{
   int  i;
   char a = 0;
   char b = 0;

   for ( i = 0; i < 216; i++ )
   {
      str[i][0] = b + 'a';
      str[i][1] = a + 'a';
      str[i][2] = '\0';
      a++;
      if ( a == 26 ) { a = 0; b++; }
   }
   
   return;
}

static void mkStr125( char str[216][3] )
{
   int  i;
   char a = 0;
   char b = 0;

   for ( i = 0; i < 125; i++ )
   {
      str[i][0] = b + 'a';
      str[i][1] = a + 'a';
      str[i][2] = '\0';
      a++;
      if ( a == 26 ) { a = 0; b++; }
   }
   
   return;
}

static void mkStr64( char str[216][3] )
{
   int  i;
   char a = 0;
   char b = 0;

   for ( i = 0; i < 64; i++ )
   {
      str[i][0] = b + 'a';
      str[i][1] = a + 'a';
      str[i][2] = '\0';
      a++;
      if ( a == 26 ) { a = 0; b++; }
   }
   
   return;
}

static void mkStr27( char str[216][3] )
{
   int  i;

   for ( i = 0; i < 27; i++ )
   {
      str[i][0] = i + 'a';
      str[i][1] = '\0';
   }
   
   return;
}

static void mkStr8( char str[216][3] )
{
   int  i;

   for ( i = 0; i < 8; i++ )
   {
      str[i][0] = i + 'a';
      str[i][1] = '\0';
   }
   
   return;
}

/*************************************************************************/
/* Function  convert()                                                   */
/*           conver from ppm to xpm                                      */
/*                                                                       */
/* Input:  FILE *in    where to read from                                */
/*         FILE *out   where to write                                    */
/*         int   mode  if error diffusion != 0                           */
/*         unsigned char *ppm    pointer to the RGB data into the memory */
/*         int   x    width of the picture                               */
/*         int   y     height of the picture                             */
/* Output: char ***data converted data as xpmData format                 */
/*                                                                       */
/* Remark: if ppm is NULL input will be taken from in                    */
/*         if data is NULL output ist to the file out else xpmData       */
/*         will be generated and put to *data                            */
/*                                                                       */
/*         Zhe allocated datas must be freed with freeXpmData()          */
/*                                                                       */
/* Return FALSE if an error encountoured                                 */
/*        TRUE if all is OK                                              */
/*                                                                       */
/*************************************************************************/

int convert(FILE *in, FILE *out, unsigned char *ppm, int x, int y, char ***data)
{
#if 0
   static long cmap[216];
   static char str[216][3];
#endif
   int  w,h;
   int  line;
   int  pix;
   int  level;
   uc   *ib=NULL;
   uc   *ob=NULL;
   int  *eb=NULL;
   uc   *xb=NULL;
   uc   *xbt=NULL;
   char **array=NULL;

   if ( ! initialized )
   {
      switch(levels)
      {
         case 216:
            mkCmap216(cmap);
            mkStr216(str);
         break;
         case 125:
            mkCmap125(cmap);
            mkStr125(str);
         break;
         case 64:
            mkCmap64(cmap);
            mkStr64(str);
         break;
         case 27:
            mkCmap27(cmap);
            mkStr27(str);
         case 8:
            mkCmap8(cmap);
            mkStr8(str);
       }
       initialized = 1;
   }

   if ( ppm == NULL )
   {
      if ( readPpmHead( in, &w, &h, &level) == FALSE )
         return FALSE;

      ib = (uc*) calloc(w,3);
      if ( ib == NULL )
      {
         perror(prgName);
         return FALSE;
      }
      
   }
   else
   {
      w = x;
      h = y;
   }

   ob = (uc*) calloc(w,1);
   if ( ob == NULL )
   {
      if ( ib != NULL )
         free(ib);
      perror(prgName);
      return FALSE;
   }
    
   /* alway needed */
   eb = (int *)calloc(w*3,sizeof(int));
   if ( eb == NULL )
   {
      if ( ib != NULL )
         free(ib);
      free(ob);
      perror(prgName);
      return FALSE;
   }

   if ( data == NULL )
   {
      xb = (uc *)calloc(((w<<1)+4)*h,sizeof(char));
      if ( xb == NULL )
      {
         free(eb);
         if ( ib != NULL )
            free(ib);
         free(ob);
         perror(prgName);
         return FALSE;
      }
   

      /* print xpm header */
      fprintf(out,"/* XPM */\n");
      fprintf(out,"static char *image_xpm[] = {\n");
      fprintf(out,"/* width height num_colors chars_per_pixel */\n");
      fprintf(out,"\"%i %i %i %i\",\n",w,h,levels,colchars);

      fprintf(out,"/* colors */\n");

      for(line = 0; line<levels;line++)
      {
        fprintf(out,"\"%s c #%06X\",\n",str[line],(int) cmap[line]);
      }
   
      fprintf(out,"/* pixels */\n");
      xbt = xb;
   }
   else
   {
      /* create an array of pointers (1 + 216 + h) */

      *data = (char **)calloc(h+levels+2,sizeof(char*));

            if ( *data == NULL )
      {
         free(eb);
         if ( ib != NULL )
            free(ib);
         free(ob);
         perror(prgName);
         return FALSE;
      }
      array = (char **)*data;
      if ( ( *array = (char*)calloc(200,1)) == NULL )
      {
         free(eb);
         if ( ib != NULL )
            free(ib);
         free(ob);
         free(*data);
         *data = NULL;
         perror(prgName);
      }  
      snprintf(*array,200,"%i %i %i %i",w,h,levels, colchars);
      
      /* the colormap */
      for(line = 0; line<levels;line++)
      {
         array++;
         *array = (char*)calloc(15,1);
         if ( ( *array = (char*)calloc(200,1)) == NULL )
         {
            free(eb);
            if ( ib != NULL )
               free(ib);
            free(ob);
            array = *data;
            while( *array ) free(*array++);
            free(*data);
            *data = NULL;
            perror(prgName);
            return 0;
         }  
        
         sprintf(*array,"%s c #%06X",str[line],(int) cmap[line]);
      }
   }
   
   for ( line = 0; line < h; line++)
   {
      if ( ib )
      {
         fread(ib, w, 3, in);
         dither_line_FSM(ib, ob, eb, line, w);
      }
      else
      {
         dither_line_FSM(ppm, ob, eb, line, w);
         ppm += (w*3);
      }
      
      
      if ( data == NULL )
         *xbt++ = '"';
      else
      {
         array++;
         if ( (*array = (char*)calloc((w<<1)+1, 1))  == NULL )
         {
            free(eb);
            if ( ib != NULL )
               free(ib);
            free(ob);
            array = *data;
            while( *array ) free(*array++);
            free(*data);
            *data = NULL;
            perror(prgName);
            return 0;
         }  
         
      }
      
      if ( data != NULL )
      {
         xbt = *array;
      }
      for ( pix = 0; pix < w; pix++ )
      {
         *xbt++ = str[ob[pix]][0];
         if ( colchars == 2 )
            *xbt++ = str[ob[pix]][1];
      }
      if ( data == NULL )
      {
         *xbt++ = '"';
         *xbt++ = '\n';
      }
   }
   if ( data == NULL )
   {
      fwrite(xb, h, (w<<1)+3, out);
      fprintf(out,"};\n");
   }

   if ( ib != NULL )
      free(ib);
   free(ob);
   free(eb);
   if ( data == NULL )
   {
      free(xb);
   }
   return(TRUE);
}

/*************************************************************************/
/* Function  freeXpmData()                                               */
/*           free the xpmData we have generated                          */
/*                                                                       */
/* Input:  char ***xpmData                                               */
/*                                                                       */
/* Return --                                                             */
/*                                                                       */
/*                                                                       */
/*************************************************************************/

void freeXpmData(char ***xpmData)
{
   char **data;
   if ( xpmData != NULL )
   {
      data = *xpmData;
      while(*data)
      {
         free(*data);
         data++;
      }
      free(*xpmData);
      *xpmData = NULL;
   }
}
