/* xwGUI -- an X11-GUI for photo prints^
'
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "render.h"

#include "base.h"
#include "error.h"
#include "fileio.h"
#include "font.h"
#include "frame.h"
#include "gfx.h"
#include "itimer.h"
#include "limits.h"
#include "main_gui.h"
#include "paper.h"
#include "picture.h"
#include "progress.h"
#include "resource.h"
#include "wysiwyg.h"
#include "x11.h"

static struct renderdatas  rd;

/* Refresh Command */
void xwRenderRefresh(void)
{
   fl_check_forms();

   return;
}

/* Render Engine */
Pixmap xwRender(float dpi,int mo,int xsize,int ysize)
{
   char           astr[1024],bstr[1024];
   int            az;
   float          aspect;
   Pixmap         pixmap=-1;

   if (mo<10) fl_set_cursor(xwGetMainGui()->window,XC_watch);

   /* Set Refresh Timer */
   if (xwAskRealMode()!=0) setTimer(200,xwRenderRefresh);

   rd.xwDPI=(25.4*(float) xwMM());     /* Inch Measure Factor */
   rd.elements=-1;                     /* Count of Pictures */

   /* PaperSize in Dots */
   rd.paperXSize= (int) ( ((float) xsize/rd.xwDPI) * dpi );
   rd.paperYSize= (int) ( ((float) ysize/rd.xwDPI) * dpi );
   rd.paperLB=(int) ( ((float) xwGetPaperLB()/rd.xwDPI) * dpi);
   rd.paperTB=(int) ( ((float) xwGetPaperTB()/rd.xwDPI) * dpi);

   /* Check Normal Picturemode */
   if (xwCheckFrameMode()!=0)
   {
      rd.elements=xwGetPictureCount();
   }
   else
   {
      rd.elements=xwGetFrameCount();
   }

   /* Exists Elements ? */
   if (rd.elements>-1)
   {
      /* Paper Sheet Datas */
      rd.paperXSize=(int) ( ((float) xwGetPaperXB()/rd.xwDPI) * dpi );
      rd.paperYSize=(int) ( ((float) xwGetPaperYB()/rd.xwDPI) * dpi );

      /* Create Paper */
      if (xwGFXCreatePaper(rd.paperXSize,rd.paperYSize)==xwTrue)
      {

         /* Progressbar */
         if (xwAskRealMode()==0)
         {
            xwOpenProgress(0,1,1,0,1,1);
            xwSetLabelProgress(xwGR(1,"pf.render","Render Print Datas...")," "," ");

            if (xwCheckFrameMode()==xwFalse)
            {
               xwSetLabelProgress("",xwGR(1,"pf.renderpicture","Render Picture..."),"");
            }
            else
            {
               xwSetLabelProgress("",xwGR(1,"pf.renderframe","Render Frame..."),"");
            }

            xwCheckProgressA();

            /* Count of Elements */
            fl_set_slider_bounds(xwGetPW(0),0,rd.elements-1);
            fl_set_slider_step(xwGetPW(0),1);
            fl_set_slider_value(xwGetPW(0),0);
         }

         /* Working Process - Create Pictures */
         for (az=0 ; az<=rd.elements ; az++)
         {
            if (xwAskRealMode()==0)
            {
               fl_set_slider_value(xwGetPW(0),az);
               xwCheckProgressA();
            }

            /*printf("Render Element: %i\n",az);*/

            /* Normal BorderType */
            xwGFXSetDrawBorderMode(0);

/* ################################################################### */
/* Picture Mode */
            if (xwCheckFrameMode()==xwFalse)
            {
               /* FrameSize */
               rd.frmSize=(int) ( ( xwGetFrameF(az)/25.4) * dpi );
               if ( (xwGetFrameF(az)>0) && (rd.frmSize==0) ) rd.frmSize=1;  /* 1 Pixel */
               /* BorderSize */
               rd.frmBorder=(int) ( ( xwGetFrameB(az)/25.4) * dpi );
               if ( (xwGetFrameB(az)>0) && (rd.frmBorder==0) ) rd.frmBorder=1;  /* 1 Pixel */
               rd.frmColor=xwGetFrameC(az);

               /* FrameSize */
               rd.picSize=(int) ( ( xwGetPictureF(az)/25.4) * dpi );
               if ( (xwGetPictureF(az)>0) && (rd.picSize==0) ) rd.picSize=1;  /* 1 Pixel */
               /* BorderSize */
               rd.picBorder=(int) ( ( xwGetPictureB(az)/25.4) * dpi );
               if ( (xwGetPictureB(az)>0) && (rd.picBorder==0) ) rd.picBorder=1;  /* 1 Pixel */
               rd.picColor=xwGetPictureC(az);

               xwGFXSetMaskLColor(0,0,rd.picColor);
               xwGFXSetPicMask(0,0,0,0,0);
               xwGFXUseMask(0);

               /* Picture with no Borders */
               if ( (xwGetPictureB(az)==0) && (xwGetPictureF(az)==0) )
               {
                  /* Normal Picture Mode */

#ifdef xwDebug
                  printf("RenderEngine: A\n");
#endif

                  /* Picture Calculation */
                  rd.rot=xwGetPictureR(az);
                  rd.xpos= (int) ( ((float) (xwGetPictureX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
                  rd.ypos= (int) ( ((float) (xwGetPictureY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );

                  rd.xsize= (int) ( ((float) xwGetPictureW(az)/rd.xwDPI) * dpi )+1;
                  rd.ysize= (int) ( ((float) xwGetPictureH(az)/rd.xwDPI) * dpi )+1;

                  /* Set Transparenz */
                  xwGFXSetTrans(xwGetPictureT(az));

                  /* Set Transparenz Mode */
                  xwGFXSetTransMode(xwGetPictureTM(az));

                  strcpy(astr,xwGetRealPicture(xwGetPictureFile(az)));
                  strcpy(bstr,xwGetFile("","buff.ppm"));
                  if (xwGFXImport(astr,bstr)==xwTrue)
                  {
                     if (xwGetPictureBC(az)==1) xwGFXUseMask(1);
                     if (xwGFXLoadPicture(bstr)==xwTrue)
                     {
                        xwGFXScalePicture(memPicture,memPaper,rd.xpos,rd.ypos,rd.xsize,rd.ysize,rd.rot);

                        xwGFXRemovePicture();
                        unlink(xwGetFile("","buff.ppm"));
                     }
                     xwGFXUseMask(0);
                  }
               }
               else
               {
                  /* Frame or Picture Border */

#ifdef xwDebug
                  printf("RenderEngine: B\n");
#endif

                  /* Picture Calculation */
                  rd.rot=xwGetPictureR(az);
                  rd.xpos= (int) ( ((float) (xwGetPictureX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
                  rd.ypos= (int) ( ((float) (xwGetPictureY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );

                  rd.xsize= (int) ( ((float) xwGetPictureW(az)/rd.xwDPI) * dpi )+1;
                  rd.ysize= (int) ( ((float) xwGetPictureH(az)/rd.xwDPI) * dpi )+1;

                  /* Set Transparenz */
                  xwGFXSetTrans(0);

                  /* Set Transparenz Mode */
                  xwGFXSetTransMode(xwGetPictureTM(az));

                  strcpy(astr,xwGetRealPicture(xwGetPictureFile(az)));
                  strcpy(bstr,xwGetFile("","buff.ppm"));
                  if (xwGFXImport(astr,bstr)==xwTrue)
                  {
                     if (xwGFXLoadPicture(bstr)==xwTrue)
                     {
                        aspect=(float) rd.xsize / (float) rd.ysize;

                        rd.xxsize=rd.xsize;
                        rd.yysize=rd.ysize;

                        if (rd.xsize>rd.ysize)
                        {
                           /* Scalesize */
                           rd.xxsize=rd.xsize - (2* rd.picBorder);
                           rd.yysize=(int) ((float) rd.xxsize/aspect);
                        }
                        else
                        {
                           /* Scalesize */
                           rd.yysize=rd.ysize - (2* rd.picBorder);
                           rd.xxsize=(int) ((float) rd.yysize*aspect);
                        }

                        xwGFXScalePicture(memPicture,createTemp,0,0,rd.xxsize,rd.yysize,rd.rot);

                        if (xwGetPictureBC(az)==1) xwGFXUseMask(1);

                        /* 90 Grad rotation */
                        if ( (rd.rot==-90) || (rd.rot==90) )
                        {
                           xwGFXCreateTemp(rd.ysize,rd.xsize);
                           xwGFXCopyPicture(memPicture,memTemp,(rd.ysize-rd.yysize)/2,(rd.xsize-rd.xxsize)/2);
                           xwGFXSetPicMask(1,rd.picBorder,rd.picBorder,xwGFXGetTempX()-(2*rd.picBorder),xwGFXGetTempY()-(2*rd.picBorder));
                        }
                        else
                        {
                           xwGFXCreateTemp(rd.xsize,rd.ysize);
                           xwGFXCopyPicture(memPicture,memTemp,(rd.xsize-rd.xxsize)/2,(rd.ysize-rd.yysize)/2);
                           xwGFXSetPicMask(1,rd.picBorder,rd.picBorder,xwGFXGetTempX()-(2*rd.picBorder),xwGFXGetTempY()-(2*rd.picBorder));
                        }
                        xwGFXRemovePicture();

                        xwGFXDrawBorderRGB(memTemp,rd.picBorder,xwGFXGetMaskLColor()); /* 0xFFFFFF); */
                        xwGFXDrawBorderRGB(memTemp,rd.picSize,rd.picColor);

                        /*
                        xwGFXTempToPicture();
                        xwGFXSavePicture("ATEST.PPM");
                        */

                        /* Set Transparenz */
                        xwGFXSetTrans(xwGetPictureT(az));

                        xwGFXCopyPicture(memTemp,memPaper,rd.xpos,rd.ypos);
                        if (xwGetPictureBC(az)==1) xwGFXUseMask(0);
                        xwGFXSetPicMask(0,0,0,0,0);

                        xwGFXRemoveTemp();
                        unlink(xwGetFile("","buff.ppm"));
                     }
                  }
               }
            }
            else
            {
/* ################################################################### */
/* Frame Mode */

               /* FrameSize */
               rd.frmSize=(int) ( ( xwGetFrameF(az)/25.4) * dpi );
               if ( (xwGetFrameF(az)>0) && (rd.frmSize==0) ) rd.frmSize=1;  /* 1 Pixel */
               /* BorderSize */
               rd.frmBorder=(int) ( ( xwGetFrameB(az)/25.4) * dpi );
               if (rd.frmBorder<rd.frmSize)
               {
                  rd.frmMax=rd.frmSize;
               }
               else
               {
                  rd.frmMax=rd.frmBorder;
               }
               if ( (xwGetFrameB(az)>0) && (rd.frmBorder==0) ) rd.frmBorder=1;  /* 1 Pixel */
               rd.frmColor=xwGetFrameC(az);

               /* FrameSize */
               rd.picSize=(int) ( ( xwGetFramePF(az)/25.4) * dpi );
               if ( (xwGetPictureF(az)>0) && (rd.picSize==0) ) rd.picSize=1;  /* 1 Pixel */
               /* BorderSize */
               rd.picBorder=(int) ( ( xwGetFramePB(az)/25.4) * dpi );
               if ( (xwGetPictureB(az)>0) && (rd.picBorder==0) ) rd.picBorder=1;  /* 1 Pixel */
               rd.picColor=xwGetFramePC(az);

               xwGFXSetMaskLColor(xwGetFontColor(az),xwGetFrameC(az),rd.picColor);
               xwGFXSetPicMask(0,0,0,0,0);
               xwGFXUseMask(0);

               if (xwGetFramePicture(az)>-1)
               {
                  /* Picture with no Borders */
                  if ( (xwGetFramePB(az)==0) && (xwGetFramePF(az)==0) )
                  {
                     /* Normal Picture Mode */

#ifdef xwDebug
                     printf("RenderEngine: C\n");
#endif

                     /* Picture Calculation */
                     rd.rot=xwGetPictureR(xwGetFramePicture(az));
                     rd.xpos= (int) ( ((float) (xwGetPictureX(xwGetFramePicture(az))-xwGetPaperLB())/rd.xwDPI) * dpi );
                     rd.ypos= (int) ( ((float) (xwGetPictureY(xwGetFramePicture(az))-xwGetPaperTB())/rd.xwDPI) * dpi );

                     rd.xsize= (int) ( ((float) xwGetPictureW(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;
                     rd.ysize= (int) ( ((float) xwGetPictureH(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;

                     /* Set Transparenz */
                     xwGFXSetTrans(xwGetFramePT(az));

                     /* Set Transparenz Mode */
                     xwGFXSetTransMode(xwGetFramePTM(az));

                     /* Frame Calculation */
                     rd.xfpos= (int) ( ((float) (xwGetFrameX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
                     rd.yfpos= (int) ( ((float) (xwGetFrameY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );
                     rd.xfsize=(int) ( ((float) xwGetFrameW(az)/rd.xwDPI) * dpi )+1;
                     rd.yfsize=(int) ( ((float) xwGetFrameH(az)/rd.xwDPI) * dpi )+1;

                     strcpy(astr,xwGetRealPicture(xwGetPictureFile(xwGetFramePicture(az))));
                     strcpy(bstr,xwGetFile("","buff.ppm"));
                     if (xwGFXImport(astr,bstr)==xwTrue)
                     {
                        if (xwGFXLoadPicture(bstr)==xwTrue)
                        {
                           /* Circle Bordertype */
                           if (xwGetFrameBT(az)==gfxCircle)
                           {
                              if (rd.xsize!=rd.ysize)
                              {
                                 if (rd.xsize>rd.ysize)
                                 {
                                    if (xwGFXCreateTemp(xwGFXGetPictureY(),xwGFXGetPictureY())==xwTrue)
                                    {
                                       xwGFXCopyPicture(memPicture,memTemp,((xwGFXGetPictureX()-xwGFXGetPictureY())/2)*-1,0);
                                       xwGFXTempToPicture();
                                    }
                                    rd.xpos=rd.xpos+((rd.xsize-rd.ysize)/2);
                                    rd.xsize=rd.ysize;
                                 }
                                 else if (rd.xsize<rd.ysize)
                                 {
                                    if (xwGFXCreateTemp(xwGFXGetPictureX(),xwGFXGetPictureX())==xwTrue)
                                    {
                                       xwGFXCopyPicture(memPicture,memTemp,0,((xwGFXGetPictureY()-xwGFXGetPictureX())/2)*-1);
                                       xwGFXTempToPicture();
                                    }
                                    rd.ypos=rd.ypos-((rd.ysize-rd.xsize)/2);
                                    rd.ysize=rd.xsize;
                                 }
                              }
                           }

                           aspect=(float) rd.xsize / (float) rd.ysize;

                           rd.xxsize=rd.xsize;
                           rd.yysize=rd.ysize;

                           /* 90 Grad rotation */
                           if ( (rd.rot==-90) || (rd.rot==90) )
                           {
                              /* Scalesize */
                              rd.xxsize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                              rd.yysize=(int) ((float) rd.xxsize/aspect);
                              if (rd.yysize>rd.xfsize - (2* (rd.picBorder+rd.frmMax)))
                              {
                                 /* Scalesize */
                                 rd.yysize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                                 rd.xxsize=(int) ((float) rd.yysize*aspect);
                              }
                              xwGFXScalePicture(memPicture,createTemp,0,0,rd.xxsize,rd.yysize,rd.rot);
                              rd.xsize=rd.yysize;
                              rd.ysize=rd.xxsize;
                           }
                           else
                           {
                              /* Scalesize */
                              rd.xxsize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                              rd.yysize=(int) ((float) rd.xxsize/aspect);
                              if (rd.yysize>rd.yfsize - (2* (rd.picBorder+rd.frmMax)))
                              {
                                 /* Scalesize */
                                 rd.yysize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                                 rd.xxsize=(int) ((float) rd.yysize*aspect);
                              }
                              xwGFXScalePicture(memPicture,createTemp,0,0,rd.xxsize,rd.yysize,rd.rot);
                              rd.xsize=rd.xxsize;
                              rd.ysize=rd.yysize;
                           }

                           /* Create Frame */
                           if (xwGetFrameBC(az)==1) xwGFXUseMask(1);
                           xwGFXCreateTemp(rd.xfsize,rd.yfsize);
                           if (xwGetFrameBC(az)==1) xwGFXUseMask(0);

                           /* Draw Picture into Frame */

                           if (xwGetPicPosition(az)==frmCenter)
                           {
                              /* Center */
                              xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-rd.xsize)/2,(rd.yfsize-rd.ysize)/2);
                              xwGFXSetPicMask(1,(rd.xfsize-rd.xsize)/2,(rd.yfsize-rd.ysize)/2,rd.xsize,rd.ysize);
                           }
                           else if (xwGetPicPosition(az)==frmNWPos)
                           {
                              /* North-West */
                              xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.frmMax);
                              xwGFXSetPicMask(1,rd.frmMax,rd.frmMax,rd.xsize,rd.ysize);
                           }
                           else if (xwGetPicPosition(az)==frmNEPos)
                           {
                              /* North East */
                              xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.frmMax);
                              xwGFXSetPicMask(1,rd.xfsize-(rd.xsize+rd.frmMax),rd.frmMax,rd.xsize,rd.ysize);
                           }
                           else if (xwGetPicPosition(az)==frmSWPos)
                           {
                              /* South West */
                              xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.yfsize-(rd.ysize+rd.frmMax));
                              xwGFXSetPicMask(1,rd.frmMax,rd.yfsize-(rd.ysize+rd.frmMax),rd.xsize,rd.ysize);
                           }
                           else if (xwGetPicPosition(az)==frmSEPos)
                           {
                              /* South East */
                              xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.yfsize-(rd.ysize+rd.frmMax));
                              xwGFXSetPicMask(1,rd.xfsize-(rd.xsize+rd.frmMax),rd.yfsize-(rd.ysize+rd.frmMax),rd.xsize,rd.ysize);
                           }

                           /* Draw Frame Border */
                           if ((rd.frmSize>0) || (xwGetFrameBT(az)==gfxCircle))
                           {
                              xwGFXSetDrawBorderMode(xwGetFrameBT(az));
                              xwGFXDrawBorderRGB(memTemp,rd.frmSize,rd.frmColor);
                           }

                           /* Render Text */
                           xwGFXUseMask(0);
                           xwRNDText(az,dpi);

                           /* Draw Frame into Paper */
                           if (xwGetFrameBC(az)==1) xwGFXUseMask(1);
                           xwGFXCopyPicture(memTemp,memPaper,rd.xfpos,rd.yfpos);
                           xwGFXUseMask(0);

                           xwGFXRemoveTemp();
                           xwGFXRemovePicture();
                           unlink(xwGetFile("","buff.ppm"));
                        }
                     }
                  }
                  else
                  {
                     /* Frame or Picture Border */

#ifdef xwDebug
                     printf("RenderEngine: D\n");
#endif

                     /* Picture Calculation */
                     rd.rot=xwGetPictureR(xwGetFramePicture(az));
                     rd.xpos= (int) ( ((float) (xwGetPictureX(xwGetFramePicture(az))-xwGetPaperLB())/rd.xwDPI) * dpi );
                     rd.ypos= (int) ( ((float) (xwGetPictureY(xwGetFramePicture(az))-xwGetPaperTB())/rd.xwDPI) * dpi );

                     rd.xsize= (int) ( ((float) xwGetPictureW(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;
                     rd.ysize= (int) ( ((float) xwGetPictureH(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;

                     /* Set Transparenz */
                     /*
                     xwGFXSetTrans(xwGetFramePT(az));
                     */
                     /* Set Transparenz Mode */
                     /*
                     xwGFXSetTransMode(xwGetFramePTM(az));
                     */

                     /* Frame Calculation */
                     rd.xfpos= (int) ( ((float) (xwGetFrameX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
                     rd.yfpos= (int) ( ((float) (xwGetFrameY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );
                     rd.xfsize=(int) ( ((float) xwGetFrameW(az)/rd.xwDPI) * dpi )+1;
                     rd.yfsize=(int) ( ((float) xwGetFrameH(az)/rd.xwDPI) * dpi )+1;

                     strcpy(astr,xwGetRealPicture(xwGetPictureFile(xwGetFramePicture(az))));
                     strcpy(bstr,xwGetFile("","buff.ppm"));
                     if (xwGFXImport(astr,bstr)==xwTrue)
                     {
                        /* Load Picture */
                        if (xwGFXLoadPicture(bstr)==xwTrue)
                        {
                           /* Circle Bordertype */
                           if (xwGetFrameBT(az)==gfxCircle)
                           {
                              if (rd.xsize!=rd.ysize)
                              {
                                 if (rd.xsize>rd.ysize)
                                 {
                                    if (xwGFXCreateTemp(xwGFXGetPictureY(),xwGFXGetPictureY())==xwTrue)
                                    {
                                       xwGFXCopyPicture(memPicture,memTemp,((xwGFXGetPictureX()-xwGFXGetPictureY())/2)*-1,0);
                                       xwGFXTempToPicture();
                                    }
                                    rd.xpos=rd.xpos+((rd.xsize-rd.ysize)/2);
                                    rd.xsize=rd.ysize;
                                 }
                                 else if (rd.xsize<rd.ysize)
                                 {
                                    if (xwGFXCreateTemp(xwGFXGetPictureX(),xwGFXGetPictureX())==xwTrue)
                                    {
                                       xwGFXCopyPicture(memPicture,memTemp,0,((xwGFXGetPictureY()-xwGFXGetPictureX())/2)*-1);
                                       xwGFXTempToPicture();
                                    }
                                    rd.ypos=rd.ypos-((rd.ysize-rd.xsize)/2);
                                    rd.ysize=rd.xsize;
                                 }
                              }
                           }
                           
                           aspect=(float) rd.xsize / (float) rd.ysize;

                           /* Calculate Destination Size */
                           rd.xxsize=rd.xsize;
                           rd.yysize=rd.ysize;

/* ################################################################### */
/* Calculate Picture for Frame */

                           /* 90 Grad rotation */
                           if ( (rd.rot==-90) || (rd.rot==90) )
                           {
                              /* Scalesize */
                              rd.xxsize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                              rd.yysize=(int) ((float) rd.xxsize/aspect);
                              if (rd.yysize>rd.xfsize - (2* (rd.picBorder+rd.frmMax)))
                              {
                                 /* Scalesize */
                                 rd.yysize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                                 rd.xxsize=(int) ((float) rd.yysize*aspect);
                              }

                              rd.xsize=rd.xxsize;
                              rd.ysize=rd.yysize;

                              if (xwGetFramePBC(az)==1) xwGFXUseMask(1);
                              xwGFXCreateTemp(rd.ysize,rd.xsize);
                              xwGFXScalePicture(memPicture,memTemp,0,0,rd.xsize,rd.ysize,rd.rot);
                              if (xwGetFramePBC(az)==1) xwGFXUseMask(0);

                              rd.ysize=rd.xxsize;
                              rd.xsize=rd.yysize;
                           }
                           else
                           {
                              /* Scalesize */
                              rd.xxsize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                              rd.yysize=(int) ((float) rd.xxsize/aspect);
                              if (rd.yysize>rd.yfsize - (2* (rd.picBorder+rd.frmMax)))
                              {
                                 /* Scalesize */
                                 rd.yysize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                                 rd.xxsize=(int) ((float) rd.yysize*aspect);
                              }

                              rd.xsize=rd.xxsize;
                              rd.ysize=rd.yysize;

                              if (xwGetFramePBC(az)==1) xwGFXUseMask(1);
                              xwGFXCreateTemp(rd.xsize,rd.ysize);
                              xwGFXScalePicture(memPicture,memTemp,0,0,rd.xsize,rd.ysize,rd.rot);
                              if (xwGetFramePBC(az)==1) xwGFXUseMask(0);

                              rd.xsize=rd.xxsize;
                              rd.ysize=rd.yysize;
                           }

/* ################################################################### */
/* Create Border Picture */
                           rd.xxsize=rd.xsize;
                           rd.yysize=rd.ysize;

                           rd.xsize=rd.xsize+(2*rd.picBorder);
                           rd.ysize=rd.ysize+(2*rd.picBorder);

                           if (xwGetFramePBC(az)==1) xwGFXUseMask(1);
                           xwGFXTempToPicture();
                           xwGFXCreateTemp(rd.xsize,rd.ysize);
                           xwGFXCopyPicture(memPicture,memTemp,rd.picBorder,rd.picBorder);

                           /* Draw Picture Border */
                           if ((rd.picBorder>0) || (rd.picSize>0))
                           {
                              xwGFXDrawBorderRGB(memTemp,rd.picBorder,xwGFXGetMaskLColor()); /*0xFFFFFF);*/
                              xwGFXDrawBorderRGB(memTemp,rd.picSize,rd.picColor);
                           }

                           xwGFXTempToPicture();
                           xwGFXUseMask(0);

                           /*xwGFXSavePicture("ATEST.PPM");*/

                           /* Create Frame */
                           if (xwGetFrameBC(az)==1) xwGFXUseMask(1);
                           xwGFXCreateTemp(rd.xfsize,rd.yfsize);
                           if (xwGetFrameBC(az)==1) xwGFXUseMask(0);

                           /* Draw Picture into Frame */

                           if (xwGetPicPosition(az)==frmCenter)
                           {
                              /* Center */
                              xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-rd.xsize)/2,(rd.yfsize-rd.ysize)/2);
                              xwGFXSetPicMask(1,(rd.xfsize-rd.xxsize)/2,(rd.yfsize-rd.yysize)/2,rd.xxsize,rd.yysize);
                           }
                           else if (xwGetPicPosition(az)==frmNWPos)
                           {
                              /* North-West */
                              xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.frmMax);
                              xwGFXSetPicMask(1,rd.frmMax,rd.frmMax,rd.xxsize,rd.yysize);
                           }
                           else if (xwGetPicPosition(az)==frmNEPos)
                           {
                              /* North East */
                              xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.frmMax);
                              xwGFXSetPicMask(1,rd.xfsize-(rd.xxsize+rd.frmMax),rd.frmMax,rd.xxsize,rd.yysize);

                           }
                           else if (xwGetPicPosition(az)==frmSWPos)
                           {
                              /* South West */
                              xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.yfsize-(rd.ysize+rd.frmMax));
                              xwGFXSetPicMask(1,rd.frmMax,rd.yfsize-(rd.yysize+rd.frmMax),rd.xxsize,rd.yysize);
                           }
                           else if (xwGetPicPosition(az)==frmSEPos)
                           {
                              /* South East */
                              xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.yfsize-(rd.ysize+rd.frmMax));
                              xwGFXSetPicMask(1,rd.xfsize-(rd.xxsize+rd.frmMax),rd.yfsize-(rd.yysize+rd.frmMax),rd.xxsize,rd.yysize);
                           }

                           /* Draw Frame Border */
                           if ((rd.frmSize>0) || (xwGetFrameBT(az)==gfxCircle))
                           {
                              xwGFXSetDrawBorderMode(xwGetFrameBT(az));
                              xwGFXDrawBorderRGB(memTemp,rd.frmSize,rd.frmColor);
                           }

                           /* Set Transparenz */
                           xwGFXSetTrans(0);

                           /* Set Transparenz Mode */
                           xwGFXSetTransMode(0);

                           /* Render Text */
                           xwGFXUseMask(0);
                           xwRNDText(az,dpi);
                           if (xwGetFramePBC(az)==1) xwGFXUseMask(1);

                           /* Set Transparenz */
                           xwGFXSetTrans(xwGetFramePT(az));

                           /* Set Transparenz Mode */
                           xwGFXSetTransMode(xwGetFramePTM(az));

                           /* Draw Frame into Paper */
                           if ((xwGetFrameBC(az)==1) || (xwGetFramePBC(az)==1)) xwGFXUseMask(1);
                           xwGFXCopyPicture(memTemp,memPaper,rd.xfpos,rd.yfpos);
                           xwGFXSetPicMask(0,0,0,0,0);

                           xwGFXUseMask(0);

                           xwGFXRemoveTemp();
                           xwGFXRemovePicture();
                           unlink(xwGetFile("","buff.ppm"));
                        }
                     }
                  }

                  xwGFXUseMask(0);
               }
               else
               {
                  /* Draw Frame Only */


#ifdef xwDebug
                  printf("RenderEngine: E\n");
#endif

                  /* Frame Calculation */
                  rd.xfpos= (int) ( ((float) (xwGetFrameX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
                  rd.yfpos= (int) ( ((float) (xwGetFrameY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );
                  rd.xfsize=(int) ( ((float) xwGetFrameW(az)/rd.xwDPI) * dpi )+1;
                  rd.yfsize=(int) ( ((float) xwGetFrameH(az)/rd.xwDPI) * dpi )+1;

                  /* Create Frame */
                  if (xwGetFrameBC(az)==1) xwGFXUseMask(1);
                  xwGFXCreateTemp(rd.xfsize,rd.yfsize);

                  /* Draw Frame Border */
                  if ((rd.frmSize>0) || (xwGetFrameBT(az)==gfxCircle))
                  {
                     xwGFXSetDrawBorderMode(xwGetFrameBT(az));
                     xwGFXDrawBorderRGB(memTemp,rd.frmSize,rd.frmColor);
                  }

                  /* Set Transparenz */
                  xwGFXSetTrans(0);

                  /* Set Transparenz Mode */
                  xwGFXSetTransMode(0);

                  /* Render Text */
                  xwGFXUseMask(0);
                  xwRNDText(az,dpi);
                  if (xwGetFrameBC(az)==1) xwGFXUseMask(1);

                  /* Set Transparenz */
                  xwGFXSetTrans(xwGetFramePT(az));

                  /* Set Transparenz Mode */
                  xwGFXSetTransMode(xwGetFramePTM(az));

                  /* Draw Frame into Paper */
                  xwGFXCopyPicture(memTemp,memPaper,rd.xfpos,rd.yfpos);
                  xwGFXUseMask(0);
                  xwGFXRemoveTemp();
               }
            }

            if ((xwAskRealMode()==0) && (xwCheckProgress()==0)) az=rd.elements;
         }

         /* Normal BorderType */
         xwGFXSetDrawBorderMode(0);

         if (xwAskRealMode()==0) xwCloseProgress();

         /* Stop Refreshing */
         if (xwAskRealMode()!=0) resetTimer();
         if (mo<10) fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

         if ((mo==0) || (mo==10))
         {
            /* Save Paper - if failed abort */
            if (xwGFXSavePaper(xwGetFile("","prtout.ppm"))==xwFalse)
            {
               xwGFXRemovePaper();
               return(-1);
            }

            if (rd.elements>-1)
            {
               xwGFXRemovePaper();
               return(0);
            }
            else
            {
               xwGFXRemovePaper();
               return(-1);
            }
         }
         else
         {
            pixmap=xwGFXCreatePixmap(memPaper);
         }
         xwGFXRemovePaper();
      }
   }

   /* Stop Refreshing */
   if (xwAskRealMode()!=0) resetTimer();
   if (mo<10) fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

   return(pixmap);
}

/* ################################################################### */
/* Pixmap Routines */

/* Render Engine for Pixmaps for LQ-Mode */
Pixmap xwRenderPixmap(float dpi,int pixNr,int *xsize,int *ysize)
{
   char           astr[1024],bstr[1024];
   int            az;
   float          aspect;
   Pixmap         pixmap=-1;

   fl_set_cursor(xwGetMainGui()->window,XC_watch);

   xwGFXSetMaskLColor(xwGetFontColor(az),xwGetFrameC(az),rd.picColor);
   xwGFXSetPicMask(0,0,0,0,0);
   xwGFXUseMask(0);

   /* Set Refresh Timer */
   setTimer(200,xwRenderRefresh);

   rd.xwDPI=(25.4*(float) xwMM());     /* Inch Measure Factor */
   rd.elements=-1;                     /* Count of Pictures */

   /* PaperSize in Dots */
/*
   rd.paperXSize= (int) ( ((float) xwGetPaperXB()/rd.xwDPI) * dpi );
   rd.paperYSize= (int) ( ((float) xwGetPaperYB()/rd.xwDPI) * dpi );
*/
   rd.paperLB=(int) ( ((float) xwGetPaperLB()/rd.xwDPI) * dpi);
   rd.paperTB=(int) ( ((float) xwGetPaperTB()/rd.xwDPI) * dpi);

   /* Check Normal Picturemode */
   if (xwCheckFrameMode()!=0)
   {
      rd.elements=xwGetPictureCount();
   }
   else
   {
      rd.elements=xwGetFrameCount();
   }

   /* Exists Elements ? */
   if (rd.elements>=pixNr)
   {
      az=pixNr;

/* ################################################################### */
/* Picture Mode */
      if (xwCheckFrameMode()==xwFalse)
      {
         /* FrameSize */
         rd.frmSize=(int) ( ( xwGetFrameF(az)/25.4) * dpi );
         if ( (xwGetFrameF(az)>0) && (rd.frmSize==0) ) rd.frmSize=1;  /* 1 Pixel */
         /* BorderSize */
         rd.frmBorder=(int) ( ( xwGetFrameB(az)/25.4) * dpi );
         if ( (xwGetFrameB(az)>0) && (rd.frmBorder==0) ) rd.frmBorder=1;  /* 1 Pixel */
         rd.frmColor=xwGetFrameC(az);

         /* FrameSize */
         rd.picSize=(int) ( ( xwGetPictureF(az)/25.4) * dpi );
         if ( (xwGetPictureF(az)>0) && (rd.picSize==0) ) rd.picSize=1;  /* 1 Pixel */
         /* BorderSize */
         rd.picBorder=(int) ( ( xwGetPictureB(az)/25.4) * dpi );
         if ( (xwGetPictureB(az)>0) && (rd.picBorder==0) ) rd.picBorder=1;  /* 1 Pixel */
         rd.picColor=xwGetPictureC(az);

         /* Picture with no Borders */
         if ( (xwGetPictureB(az)==0) && (xwGetPictureF(az)==0) )
         {
            /* Normal Picture Mode */

            /* Picture Calculation */
            rd.rot=xwGetPictureR(az);
            rd.xpos= (int) ( ((float) (xwGetPictureX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
            rd.ypos= (int) ( ((float) (xwGetPictureY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );

            rd.xsize= (int) ( ((float) xwGetPictureW(az)/rd.xwDPI) * dpi )+1;
            rd.ysize= (int) ( ((float) xwGetPictureH(az)/rd.xwDPI) * dpi )+1;

            *xsize=rd.xsize;
            *ysize=rd.ysize;

            /* Set Transparenz */
            xwGFXSetTrans(0);

            strcpy(astr,xwGetRealPicture(xwGetPictureFile(az)));
            strcpy(bstr,xwGetFile("","buff.ppm"));
            if (xwGFXImport(astr,bstr)==xwTrue)
            {
               if (xwGFXLoadPicture(bstr)==xwTrue)
               {
                  unlink(xwGetFile("","buff.ppm"));
                  xwGFXScalePicture(memPicture,createTemp,0,0,rd.xsize,rd.ysize,rd.rot);

                  if (xwIsPaperLS()==xwLandscape) xwGFXRotatePicture(90);

                  /* Stop Refreshing */
                  resetTimer();
                  fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

                  pixmap=xwGFXCreatePixmap(memPicture);
                  xwGFXRemovePicture();
                  return(pixmap);
               }
            }
         }
         else
         {
            /* Frame or Picture Border */

            /* Picture Calculation */
            rd.rot=xwGetPictureR(az);
            rd.xpos= (int) ( ((float) (xwGetPictureX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
            rd.ypos= (int) ( ((float) (xwGetPictureY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );

            rd.xsize= (int) ( ((float) xwGetPictureW(az)/rd.xwDPI) * dpi )+1;
            rd.ysize= (int) ( ((float) xwGetPictureH(az)/rd.xwDPI) * dpi )+1;

            *xsize=rd.xsize;
            *ysize=rd.ysize;

            /* Set Transparenz */
            xwGFXSetTrans(0);

            strcpy(astr,xwGetRealPicture(xwGetPictureFile(az)));
            strcpy(bstr,xwGetFile("","buff.ppm"));
            if (xwGFXImport(astr,bstr)==xwTrue)
            {
               if (xwGFXLoadPicture(bstr)==xwTrue)
               {
                  unlink(xwGetFile("","buff.ppm"));

                  aspect=(float) rd.xsize / (float) rd.ysize;

                  rd.xxsize=rd.xsize;
                  rd.yysize=rd.ysize;

                  if (rd.xsize>rd.ysize)
                  {
                     /* Scalesize */
                     rd.xxsize=rd.xsize - (2* rd.picBorder);
                     rd.yysize=(int) ((float) rd.xxsize/aspect);
                  }
                  else
                  {
                     /* Scalesize */
                     rd.yysize=rd.ysize - (2* rd.picBorder);
                     rd.xxsize=(int) ((float) rd.yysize*aspect);
                  }

                  xwGFXScalePicture(memPicture,createTemp,0,0,rd.xxsize,rd.yysize,rd.rot);

                  /* 90 Grad rotation */
                  if ( (rd.rot==-90) || (rd.rot==90) )
                  {
                     xwGFXCreateTemp(rd.ysize,rd.xsize);
                     xwGFXCopyPicture(memPicture,memTemp,(rd.ysize-rd.yysize)/2,(rd.xsize-rd.xxsize)/2);
                  }
                  else
                  {
                     xwGFXCreateTemp(rd.xsize,rd.ysize);
                     xwGFXCopyPicture(memPicture,memTemp,(rd.xsize-rd.xxsize)/2,(rd.ysize-rd.yysize)/2);
                  }
                  xwGFXRemovePicture();

                  xwGFXDrawBorderRGB(memTemp,rd.picBorder,0xFFFFFF);
                  xwGFXDrawBorderRGB(memTemp,rd.picSize,rd.picColor);

                  xwGFXTempToPicture();

                  if (xwIsPaperLS()==xwLandscape) xwGFXRotatePicture(90);

                  /* Stop Refreshing */
                  resetTimer();
                  fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

                  pixmap=xwGFXCreatePixmap(memPicture);
                  xwGFXRemovePicture();
                  return(pixmap);
               }
            }
         }
      }
      else
      {
/* ################################################################### */
/* Frame Mode */

         /* FrameSize */
         rd.frmSize=(int) ( ( xwGetFrameF(az)/25.4) * dpi );
         if ( (xwGetFrameF(az)>0) && (rd.frmSize==0) ) rd.frmSize=1;  /* 1 Pixel */
         /* BorderSize */
         rd.frmBorder=(int) ( ( xwGetFrameB(az)/25.4) * dpi );
         if (rd.frmBorder<rd.frmSize)
         {
            rd.frmMax=rd.frmSize;
         }
         else
         {
            rd.frmMax=rd.frmBorder;
         }
         if ( (xwGetFrameB(az)>0) && (rd.frmBorder==0) ) rd.frmBorder=1;  /* 1 Pixel */
         rd.frmColor=xwGetFrameC(az);

         /* FrameSize */
         rd.picSize=(int) ( ( xwGetFramePF(az)/25.4) * dpi );
         if ( (xwGetPictureF(az)>0) && (rd.picSize==0) ) rd.picSize=1;  /* 1 Pixel */
         /* BorderSize */
         rd.picBorder=(int) ( ( xwGetFramePB(az)/25.4) * dpi );
         if ( (xwGetPictureB(az)>0) && (rd.picBorder==0) ) rd.picBorder=1;  /* 1 Pixel */
         rd.picColor=xwGetFramePC(az);

         if (xwGetFramePicture(az)>-1)
         {
            /* Picture with no Borders */
            if ( (xwGetFramePB(az)==0) && (xwGetFramePF(az)==0) )
            {

               /* Normal Picture Mode */

               /* Picture Calculation */
               rd.rot=xwGetPictureR(xwGetFramePicture(az));
               rd.xpos= (int) ( ((float) (xwGetPictureX(xwGetFramePicture(az))-xwGetPaperLB())/rd.xwDPI) * dpi );
               rd.ypos= (int) ( ((float) (xwGetPictureY(xwGetFramePicture(az))-xwGetPaperTB())/rd.xwDPI) * dpi );

               rd.xsize= (int) ( ((float) xwGetPictureW(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;
               rd.ysize= (int) ( ((float) xwGetPictureH(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;

               *xsize=rd.xsize;
               *ysize=rd.ysize;

               /* Set Transparenz */
               xwGFXSetTrans(0);

               /* Frame Calculation */
               rd.xfpos= (int) ( ((float) (xwGetFrameX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
               rd.yfpos= (int) ( ((float) (xwGetFrameY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );
               rd.xfsize=(int) ( ((float) xwGetFrameW(az)/rd.xwDPI) * dpi )+1;
               rd.yfsize=(int) ( ((float) xwGetFrameH(az)/rd.xwDPI) * dpi )+1;

               strcpy(astr,xwGetRealPicture(xwGetPictureFile(xwGetFramePicture(az))));
               strcpy(bstr,xwGetFile("","buff.ppm"));
               if (xwGFXImport(astr,bstr)==xwTrue)
               {
                  if (xwGFXLoadPicture(bstr)==xwTrue)
                  {
                     unlink(xwGetFile("","buff.ppm"));

                     aspect=(float) rd.xsize / (float) rd.ysize;

                     rd.xxsize=rd.xsize;
                     rd.yysize=rd.ysize;

                     /* 90 Grad rotation */
                     if ( (rd.rot==-90) || (rd.rot==90) )
                     {
                        /* Scalesize */
                        rd.xxsize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                        rd.yysize=(int) ((float) rd.xxsize/aspect);
                        if (rd.yysize>rd.xfsize - (2* (rd.picBorder+rd.frmMax)))
                        {
                           /* Scalesize */
                           rd.yysize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                           rd.xxsize=(int) ((float) rd.yysize*aspect);
                        }
                        xwGFXScalePicture(memPicture,createTemp,0,0,rd.xxsize,rd.yysize,rd.rot);
                        rd.xsize=rd.yysize;
                        rd.ysize=rd.xxsize;
                     }
                     else
                     {
                        /* Scalesize */
                        rd.xxsize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                        rd.yysize=(int) ((float) rd.xxsize/aspect);
                        if (rd.yysize>rd.yfsize - (2* (rd.picBorder+rd.frmMax)))
                        {
                           /* Scalesize */
                           rd.yysize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                           rd.xxsize=(int) ((float) rd.yysize*aspect);
                        }
                        xwGFXScalePicture(memPicture,createTemp,0,0,rd.xxsize,rd.yysize,rd.rot);
                        rd.xsize=rd.xxsize;
                        rd.ysize=rd.yysize;
                     }

                     /* Create Frame */
                     xwGFXCreateTemp(rd.xfsize,rd.yfsize);

                     /* Draw Picture into Frame */

                     if (xwGetPicPosition(az)==frmCenter)
                     {
                        /* Center */
                        xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-rd.xsize)/2,(rd.yfsize-rd.ysize)/2);
                     }
                     else if (xwGetPicPosition(az)==frmNWPos)
                     {
                        /* North-West */
                        xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.frmMax);
                     }
                     else if (xwGetPicPosition(az)==frmNEPos)
                     {
                        /* North East */
                        xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.frmMax);
                     }
                     else if (xwGetPicPosition(az)==frmSWPos)
                     {
                        /* South West */
                        xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.yfsize-(rd.ysize+rd.frmMax));
                     }
                     else if (xwGetPicPosition(az)==frmSEPos)
                     {
                        /* South East */
                        xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.yfsize-(rd.ysize+rd.frmMax));
                     }

                     /* Draw Frame Border */
                     if (rd.frmSize>0) xwGFXDrawBorderRGB(memTemp,rd.frmSize,rd.frmColor);

                     xwGFXTempToPicture();

                     if (xwIsPaperLS()==xwLandscape) xwGFXRotatePicture(90);

                     /* Stop Refreshing */
                     resetTimer();
                     fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

                     pixmap=xwGFXCreatePixmap(memPicture);
                     xwGFXRemovePicture();
                     return(pixmap);
                  }
               }

            }
            else
            {
               /* Frame or Picture Border */

               /* Picture Calculation */
               rd.rot=xwGetPictureR(xwGetFramePicture(az));
               rd.xpos= (int) ( ((float) (xwGetPictureX(xwGetFramePicture(az))-xwGetPaperLB())/rd.xwDPI) * dpi );
               rd.ypos= (int) ( ((float) (xwGetPictureY(xwGetFramePicture(az))-xwGetPaperTB())/rd.xwDPI) * dpi );

               rd.xsize= (int) ( ((float) xwGetPictureW(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;
               rd.ysize= (int) ( ((float) xwGetPictureH(xwGetFramePicture(az))/rd.xwDPI) * dpi )+1;

               *xsize=rd.xsize;
               *ysize=rd.ysize;

               /* Set Transparenz */
               xwGFXSetTrans(0);

               /* Frame Calculation */
               rd.xfpos= (int) ( ((float) (xwGetFrameX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
               rd.yfpos= (int) ( ((float) (xwGetFrameY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );
               rd.xfsize=(int) ( ((float) xwGetFrameW(az)/rd.xwDPI) * dpi )+1;
               rd.yfsize=(int) ( ((float) xwGetFrameH(az)/rd.xwDPI) * dpi )+1;

               strcpy(astr,xwGetRealPicture(xwGetPictureFile(xwGetFramePicture(az))));
               strcpy(bstr,xwGetFile("","buff.ppm"));
               if (xwGFXImport(astr,bstr)==xwTrue)
               {
                  /* Load Picture */
                  if (xwGFXLoadPicture(bstr)==xwTrue)
                  {
                     unlink(xwGetFile("","buff.ppm"));

                     aspect=(float) rd.xsize / (float) rd.ysize;

                     /* Calculate Destination Size */
                     rd.xxsize=rd.xsize;
                     rd.yysize=rd.ysize;

/* ################################################################### */
/* Calculate Picture for Frame */

                     /* 90 Grad rotation */
                     if ( (rd.rot==-90) || (rd.rot==90) )
                     {
                        /* Scalesize */
                        rd.xxsize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                        rd.yysize=(int) ((float) rd.xxsize/aspect);
                        if (rd.yysize>rd.xfsize - (2* (rd.picBorder+rd.frmMax)))
                        {
                           /* Scalesize */
                           rd.yysize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                           rd.xxsize=(int) ((float) rd.yysize*aspect);
                        }

                        rd.xsize=rd.xxsize;
                        rd.ysize=rd.yysize;

                        xwGFXCreateTemp(rd.ysize,rd.xsize);
                        xwGFXScalePicture(memPicture,memTemp,0,0,rd.xsize,rd.ysize,rd.rot);

                        rd.ysize=rd.xxsize;
                        rd.xsize=rd.yysize;
                     }
                     else
                     {
                        /* Scalesize */
                        rd.xxsize=rd.xfsize - (2* (rd.picBorder+rd.frmMax));
                        rd.yysize=(int) ((float) rd.xxsize/aspect);
                        if (rd.yysize>rd.yfsize - (2* (rd.picBorder+rd.frmMax)))
                        {
                           /* Scalesize */
                           rd.yysize=rd.yfsize - (2* (rd.picBorder+rd.frmMax));
                           rd.xxsize=(int) ((float) rd.yysize*aspect);
                        }

                        rd.xsize=rd.xxsize;
                        rd.ysize=rd.yysize;

                        xwGFXCreateTemp(rd.xsize,rd.ysize);
                        xwGFXScalePicture(memPicture,memTemp,0,0,rd.xsize,rd.ysize,rd.rot);

                        rd.xsize=rd.xxsize;
                        rd.ysize=rd.yysize;
                     }

/* ################################################################### */
/* Create Border Picture */
                     rd.xsize=rd.xsize+(2*rd.picBorder);
                     rd.ysize=rd.ysize+(2*rd.picBorder);

                     xwGFXTempToPicture();

                     xwGFXCreateTemp(rd.xsize,rd.ysize);
                     xwGFXCopyPicture(memPicture,memTemp,rd.picBorder,rd.picBorder);

                     /* Draw Picture Border */
                     if ((rd.picBorder>0) || (rd.picSize>0))
                     {
                        xwGFXDrawBorderRGB(memTemp,rd.picBorder,0xFFFFFF);
                        xwGFXDrawBorderRGB(memTemp,rd.picSize,rd.picColor);
                     }

                     xwGFXTempToPicture();

                     /*xwGFXSavePicture("APICTURE.PPM");*/

                     /* Create Frame */
                     xwGFXCreateTemp(rd.xfsize,rd.yfsize);

                     /* Draw Picture into Frame */

                     if (xwGetPicPosition(az)==frmCenter)
                     {
                        /* Center */
                        xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-rd.xsize)/2,(rd.yfsize-rd.ysize)/2);
                     }
                     else if (xwGetPicPosition(az)==frmNWPos)
                     {
                        /* North-West */
                        xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.frmMax);
                     }
                     else if (xwGetPicPosition(az)==frmNEPos)
                     {
                        /* North East */
                        xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.frmMax);
                     }
                     else if (xwGetPicPosition(az)==frmSWPos)
                     {
                        /* South West */
                        xwGFXCopyPicture(memPicture,memTemp,rd.frmMax,rd.yfsize-(rd.ysize+rd.frmMax));
                     }
                     else if (xwGetPicPosition(az)==frmSEPos)
                     {
                        /* South East */
                        xwGFXCopyPicture(memPicture,memTemp,rd.xfsize-(rd.xsize+rd.frmMax),rd.yfsize-(rd.ysize+rd.frmMax));
                     }

                     /* Draw Frame Border */
                     if (rd.frmSize>0) xwGFXDrawBorderRGB(memTemp,rd.frmSize,rd.frmColor);

                     xwGFXTempToPicture();

                     if (xwIsPaperLS()==xwLandscape) xwGFXRotatePicture(90);

                     /* Stop Refreshing */
                     resetTimer();
                     fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

                     pixmap=xwGFXCreatePixmap(memPicture);
                     xwGFXRemovePicture();
                     return(pixmap);
                  }
               }
            }
         }
         else
         {
            /* Draw Frame Only */

            /* Frame Calculation */
            rd.xfpos= (int) ( ((float) (xwGetFrameX(az)-xwGetPaperLB())/rd.xwDPI) * dpi );
            rd.yfpos= (int) ( ((float) (xwGetFrameY(az)-xwGetPaperTB())/rd.xwDPI) * dpi );
            rd.xfsize=(int) ( ((float) xwGetFrameW(az)/rd.xwDPI) * dpi )+1;
            rd.yfsize=(int) ( ((float) xwGetFrameH(az)/rd.xwDPI) * dpi )+1;

            /* Create Frame */
            xwGFXCreateTemp(rd.xfsize,rd.yfsize);

            /* Draw Frame Border */
            if (rd.frmSize>0) xwGFXDrawBorderRGB(memTemp,rd.frmSize,rd.frmColor);

            xwGFXTempToPicture();

            if (xwIsPaperLS()==xwLandscape) xwGFXRotatePicture(90);

            /* Stop Refreshing */
            resetTimer();
            fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

            pixmap=xwGFXCreatePixmap(memPicture);
            xwGFXRemovePicture();
            return(pixmap);
         }
      }
   }

   /* Stop Refreshing */
   resetTimer();
   fl_set_cursor(xwGetMainGui()->window,FL_DEFAULT_CURSOR);

   return(-1);
}


/* ################################################################### */
/* Text Routines */

/* Render Engine for Text */
void xwRNDText(int frm,float dpi)
{
   char              *buffer;
   char              *strptrA,*strptrB;
   int               r,g,b,rgbv;

   int               loXPos,loYPos,roXPos,roYPos,luXPos,luYPos,ruXPos,ruYPos;
   int               border;

   if ( (xwGetFontText(frm)!=NULL) && (strlen(xwGetFontText(frm))>0) )
   {
      xwGFXRemovePicture();

      strptrA=xwGetFontText(frm);
      buffer=malloc(strlen(strptrA)+4);
      if (buffer!=NULL)
      {
         strcpy(buffer,strptrA);
         strptrA=buffer;

         border=0;
         if (rd.frmBorder>rd.frmSize) border=rd.frmBorder;
         else if (rd.frmBorder<rd.frmSize) border=rd.frmSize;

         /* create position datas */
         loXPos=border;
         loYPos=border;
         roXPos=rd.xfsize-border;
         roYPos=border;
         luXPos=border;
         luYPos=rd.yfsize-border;
         ruXPos=rd.xfsize-border;
         ruYPos=rd.yfsize-border;

         /* Calculate RGB-Value */
         rgbv=xwGetFontColor(frm);
         r=rgbv/65536;
         rgbv=rgbv-(r*65536);
         g=rgbv/256;
         rgbv=rgbv-(g*256);
         b=rgbv;
         xwGFXSetTextColor(r,g,b);

         /* split text in lines */
         while ((strptrB=strchr(strptrA,0x0a))!=NULL)
         {
            strptrB[0]=0x00;
            strptrB=strptrB+1;

            if (strlen(strptrA)>0)
            {
               xwRenderText(xwGetFontID(xwGetFontName(frm)),xwGetFontSize(frm),(int) dpi,xwGetFontLine(frm),strptrA);

               /* rotate picture */
               xwGFXSaveTemp();
               if (xwGetFontDirection(frm)==1) xwGFXRotatePicture(90);
               else if (xwGetFontDirection(frm)==2) xwGFXRotatePicture(180);
               else if (xwGetFontDirection(frm)==3) xwGFXRotatePicture(-90);
               xwGFXLoadTemp();

               /* xwGFXCopyPicture(memPicture,memTemp,0,0); */
               xwGFXIgnorePaper(1);
               if (xwGetFontDirection(frm)==0)
               {
                  /* East */

                  /* left */
                  if (xwGetFontMode(frm)==0)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,loXPos,loYPos);
                  }
                  /* middle */
                  else if (xwGetFontMode(frm)==1)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-xwGFXGetPictureX())/2,loYPos);
                  }
                  /* right */
                  else if (xwGetFontMode(frm)==2)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,roXPos-xwGFXGetPictureX(),loYPos);
                  }

                  loYPos=loYPos+xwGFXGetPictureY();
               }
               else if (xwGetFontDirection(frm)==1)
               {
                  /* South */

                  roXPos=roXPos-xwGFXGetPictureX();
                  /* left */
                  if (xwGetFontMode(frm)==0)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,roXPos,roYPos);
                  }
                  /* middle */
                  else if (xwGetFontMode(frm)==1)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,roXPos,(rd.yfsize-xwGFXGetPictureY())/2);
                  }
                  /* right */
                  else if (xwGetFontMode(frm)==2)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,roXPos,ruYPos-xwGFXGetPictureY());
                  }
               }
               else if (xwGetFontDirection(frm)==2)
               {
                  /* West */

                  ruYPos=ruYPos-xwGFXGetPictureY();
                  /* left */
                  if (xwGetFontMode(frm)==0)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,ruXPos-xwGFXGetPictureX(),ruYPos);
                  }
                  /* middle */
                  else if (xwGetFontMode(frm)==1)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-xwGFXGetPictureX())/2,ruYPos);
                  }
                  /* right */
                  else if (xwGetFontMode(frm)==2)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,luXPos,ruYPos);
                  }
               }
               else if (xwGetFontDirection(frm)==3)
               {
                  /* North */

                  /* left */
                  if (xwGetFontMode(frm)==0)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,luXPos,luYPos-xwGFXGetPictureY());
                  }
                  /* middle */
                  else if (xwGetFontMode(frm)==1)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,luXPos,(rd.yfsize-xwGFXGetPictureY())/2);
                  }
                  /* right */
                  else if (xwGetFontMode(frm)==2)
                  {
                     xwGFXCopyPicture(memPicture,memTemp,luXPos,loYPos);
                  }
                  luXPos=luXPos+xwGFXGetPictureX();
               }
               xwGFXIgnorePaper(0);
            }
            else
            {
               if (xwGetFontDirection(frm)==0)
               {
                  /* East */
                  loYPos=loYPos+xwGFXGetPictureY();
               }
               else if (xwGetFontDirection(frm)==1)
               {
                  /* South */
                  roXPos=roXPos-xwGFXGetPictureX();
               }
               else if (xwGetFontDirection(frm)==2)
               {
                  /* West */
                  ruYPos=ruYPos-xwGFXGetPictureY();
               }
               else if (xwGetFontDirection(frm)==3)
               {
                  /* North */
                  luXPos=luXPos+xwGFXGetPictureX();
               }
            }

            strptrA=strptrB;
         }

         if (strlen(strptrA)>0)
         {
            xwRenderText(xwGetFontID(xwGetFontName(frm)),xwGetFontSize(frm),(int) dpi,xwGetFontLine(frm),strptrA);

            /* rotate picture */
            xwGFXSaveTemp();
            if (xwGetFontDirection(frm)==1) xwGFXRotatePicture(90);
            else if (xwGetFontDirection(frm)==2) xwGFXRotatePicture(180);
            else if (xwGetFontDirection(frm)==3) xwGFXRotatePicture(-90);
            xwGFXLoadTemp();


            /* xwGFXCopyPicture(memPicture,memTemp,0,0); */
            xwGFXIgnorePaper(1);
            if (xwGetFontDirection(frm)==0)
            {
               /* East */

               /* left */
               if (xwGetFontMode(frm)==0)
               {
                  xwGFXCopyPicture(memPicture,memTemp,loXPos,loYPos);
               }
               /* middle */
               else if (xwGetFontMode(frm)==1)
               {
                  xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-xwGFXGetPictureX())/2,loYPos);
               }
               /* right */
               else if (xwGetFontMode(frm)==2)
               {
                  xwGFXCopyPicture(memPicture,memTemp,roXPos-xwGFXGetPictureX(),loYPos);
               }

               loYPos=loYPos+xwGFXGetPictureY();
            }
            else if (xwGetFontDirection(frm)==1)
            {
               /* South */

               roXPos=roXPos-xwGFXGetPictureX();
               /* left */
               if (xwGetFontMode(frm)==0)
               {
                  xwGFXCopyPicture(memPicture,memTemp,roXPos,roYPos);
               }
               /* middle */
               else if (xwGetFontMode(frm)==1)
               {
                  xwGFXCopyPicture(memPicture,memTemp,roXPos,(rd.yfsize-xwGFXGetPictureY())/2);
               }
               /* right */
               else if (xwGetFontMode(frm)==2)
               {
                  xwGFXCopyPicture(memPicture,memTemp,roXPos,ruYPos-xwGFXGetPictureY());
               }
            }
            else if (xwGetFontDirection(frm)==2)
            {
               /* West */

               ruYPos=ruYPos-xwGFXGetPictureY();
               /* left */
               if (xwGetFontMode(frm)==0)
               {
                  xwGFXCopyPicture(memPicture,memTemp,ruXPos-xwGFXGetPictureX(),ruYPos);
               }
               /* middle */
               else if (xwGetFontMode(frm)==1)
               {
                  xwGFXCopyPicture(memPicture,memTemp,(rd.xfsize-xwGFXGetPictureX())/2,ruYPos);
               }
               /* right */
               else if (xwGetFontMode(frm)==2)
               {
                  xwGFXCopyPicture(memPicture,memTemp,luXPos,ruYPos);
               }
            }
            else if (xwGetFontDirection(frm)==3)
            {
               /* North */

               /* left */
               if (xwGetFontMode(frm)==0)
               {
                  xwGFXCopyPicture(memPicture,memTemp,luXPos,luYPos-xwGFXGetPictureY());
               }
               /* middle */
               else if (xwGetFontMode(frm)==1)
               {
                  xwGFXCopyPicture(memPicture,memTemp,luXPos,(rd.yfsize-xwGFXGetPictureY())/2);
               }
               /* right */
               else if (xwGetFontMode(frm)==2)
               {
                  xwGFXCopyPicture(memPicture,memTemp,luXPos,loYPos);
               }
               luXPos=luXPos+xwGFXGetPictureX();
            }
            xwGFXIgnorePaper(0);
         }
         else
         {
            if (xwGetFontDirection(frm)==0)
            {
               /* East */
               loYPos=loYPos+xwGFXGetPictureY();
            }
            else if (xwGetFontDirection(frm)==1)
            {
               /* South */
               roXPos=roXPos-xwGFXGetPictureX();
            }
            else if (xwGetFontDirection(frm)==2)
            {
               /* West */
               ruYPos=ruYPos-xwGFXGetPictureY();
            }
            else if (xwGetFontDirection(frm)==3)
            {
               /* North */
               luXPos=luXPos+xwGFXGetPictureX();
            }
         }

         free(buffer);
      }
      xwGFXRemovePicture();
   }

   return;
}