/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "picture.h"

#include "base.h"
#include "fileio.h"
#include "limits.h"
#include "lowlevel.h"
#include "main_gui.h"
#include "paper.h"
#include "pbcache.h"
#include "picviewer.h"
#include "x11.h"

extern struct mainform     mf;

static picturedatas  *pd[picMaxPictures+1];  /* Picture Datas 51 */
static int           pictureAnz=-1;          /* Picture Count */
static int           picActive=-1;           /* Actual Selected Picture */

/* Default Value for Picture Rotation ( -180,-90,0,90,180 ) */
static int           picRotation=0;

/* ################################################################### */
/* Picture Managment */

/* Init Picture Setting */
void xwInitPicture(void)
{
   int         az;

   pictureAnz=-1;
   picActive=-1;
   picRotation=0;

   /* Initialize Picture Array */
   for (az=0 ; az<picMaxPictures ; az++)
   {
      pd[az]=NULL;
   }

   return;
}

/* Free Picture Settings */
void xwFreePicture(int pNr)
{
   int         az;

   /* Free all pictures */
   if (pNr==-1)
   {
      if (pictureAnz>-1)
      {
         for (az=0 ; az<=pictureAnz ; az++)
         {
            if (pd[az]!=NULL)
            {
               /* Free Pixmap */
               if (pd[az]->pFlag!=0)
               {
                  pd[az]->pFlag=0;
                  XFreePixmap(fl_display,pd[az]->pPixmap);
               }

               /* Delete Wysiwyg Chache Entry */
               /*xwDeletePBEntry(pd[az]->filename);*/

               free(pd[az]);
            }
            pd[az]=NULL;
         }
         pictureAnz=-1;
      }
   }
   else
   {
      az=pNr;

      if (pd[az]!=NULL)
      {
         /* Free Pixmap */
         if (pd[az]->pFlag!=0)
         {
            pd[az]->pFlag=0;
            XFreePixmap(fl_display,pd[az]->pPixmap);
         }

         /* Delete Wysiwyg Chache Entry */
         /*xwDeletePBEntry(pd[az]->filename);*/

         free(pd[az]);
         for (az=pNr ; az<pictureAnz ; az++)
         {
            pd[az]=pd[az+1];
         }
         pd[pictureAnz]=NULL;
         pictureAnz--;

         /* Actual picture */
         if (pictureAnz<picActive) picActive=pictureAnz;
      }
   }

   return;
}

/* Swap Picture */
void xwSwapPicture(int pNrA,int pNrB)
{
   pd[picMaxPictures]=pd[pNrA];
   pd[pNrA]=pd[pNrB];
   pd[pNrB]=pd[picMaxPictures];

   return;
}

/* Modify Picture Rotation */
void xwSetPictureRotation(int rotate)
{
   picRotation=rotate;
   return;
}

/* ################################################################### */
/* Pixmap Verwaltung */

/* Set Pixmap */
void xwSetPicturePixmap(int pNr,Pixmap pixmap)
{
   /* exist frames */
   if ( (pictureAnz>-1) && (pNr<=pictureAnz) )
   {
      if (pd[pNr]->pFlag!=0)
      {
         XFreePixmap(fl_display,pd[pNr]->pPixmap);
         pd[pNr]->pFlag=0;
      }
      if ((int) pixmap>-1)
      {
         pd[pNr]->pPixmap=pixmap;
         pd[pNr]->pFlag=1;
      }
   }

   return;
}

/* Clear Pixmap */
void xwClearPicturePixmap(int pNr)
{
   /* exist frames */
   if ( (pictureAnz>-1) && (pNr<=pictureAnz) )
   {
      if (pd[pNr]->pFlag!=0)
      {
         XFreePixmap(fl_display,pd[pNr]->pPixmap);
         pd[pNr]->pFlag=0;
      }
   }

   return;
}

/* Get Pixmap */
Pixmap xwGetPicturePixmap(int pNr)
{
   /* exist frames */
   if ( (pictureAnz>-1) && (pNr<=pictureAnz) )
   {
      if (pd[pNr]->pFlag!=0)
      {
         return(pd[pNr]->pPixmap);
      }
   }

   return(-1);
}

/* Get Pixmap Datas */
void xwGetPicturePMDatas(int pNr,int *w,int *h)
{
   /* exist frames */
   if ( (pictureAnz>-1) && (pNr<=pictureAnz) )
   {
      *w=pd[pNr]->pXSize;
      *h=pd[pNr]->pYSize;
   }

   return;
}

/* Set Pixmap Datas */
void xwSetPicturePMDatas(int pNr,int w,int h)
{
   /* exist frames */
   if ( (pictureAnz>-1) && (pNr<=pictureAnz) )
   {
      pd[pNr]->pXSize=w;
      pd[pNr]->pYSize=h;
   }

   return;
}


/* ################################################################### */
/* Active and Picture Count Datas */

/* Get Active Picture Nr. */
int xwGetActivePicture(void)
{
   return(picActive);
}

/* Set Active Picture Nr. */
void xwSetActivePicture(int pNr)
{
   /* Check Range of Pictures */
   if ((pNr>-2) && (pNr<=pictureAnz))
   {
      picActive=pNr;
   }

   return;
}

/* Get Count of Picture */
int xwGetPictureCount(void)
{
   return(pictureAnz);
}

/* ################################################################### */
/* Picture Geometry in Pixels */

/* Get Picture XPos */
int xwGetPictureX(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->xpos);
   }

   return(-1);
}

/* Get Picture YPos */
int xwGetPictureY(int pNr)
{
  if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->ypos);
   }

   return(-1);
}

/* Get Picture XSize */
int xwGetPictureW(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->xsize);
   }

   return(-1);
}

/* Get Picture YSize */
int xwGetPictureH(int pNr)
{
  if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->ysize);
   }

   return(-1);
}

/* Set Picture Geometry */
void xwSetPictureGeo(int pNr,int xpos,int ypos,int xsize,int ysize)
{
  if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      pd[pNr]->xpos=xpos;
      pd[pNr]->ypos=ypos;
      pd[pNr]->xsize=xsize;
      pd[pNr]->ysize=ysize;
   }

   return;
}

/* ################################################################### */
/* Real Picture Datas */

/* Get Picture Real XSize */
int xwGetPictureRW(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->xpic);
   }

   return(-1);
}

/* Get Picture Real YSize */
int xwGetPictureRH(int pNr)
{
  if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->ypic);
   }

   return(-1);
}

/* Get Picture Aspect */
float xwGetPictureA(int pNr)
{
  if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->paspect);
   }

   return(-1);
}

/* ################################################################### */
/* Picture Rotation */

/* Get Picture Rotation */
int xwGetPictureR(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->rotate);
   }

  return(-1);
}

/* Set Picture Rotation */
void xwSetPictureR(int pNr,int rotate)
{
  if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      pd[pNr]->rotate=rotate;
   }

   return;
}

/* ################################################################### */
/* Border Datas */

/* Get Picture Border Size */
double xwGetPictureB(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->borderSize);
   }

   return(0);
}

/* Get Picture Border Color */
int xwGetPictureBC(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->borderColor);
   }

   return(0);
}

/* Get Picture Frame Size */
double xwGetPictureF(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->frameSize);
   }

   return(0);
}

/* Get Picture Frame Color */
int xwGetPictureC(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->frmrgb);
   }

   return(0);
}

/* Get Picture Transparency */
int xwGetPictureT(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->transparenz);
   }

   return(0);
}

/* Get Picture Transparency mode */
int xwGetPictureTM(int pNr)
{
   if ((pictureAnz>-1) && (pNr<=pictureAnz))
   {
      return(pd[pNr]->transMode);
   }

   return(0);
}

/* Set Picture Border Size */
void xwSetPictureB(int pNr,double size)
{
  if (pictureAnz>-1)
   {
      pd[pNr]->borderSize=size;
   }

   return;
}

/* Set Picture Border Color */
void xwSetPictureBC(int pNr,int mo)
{
   if (pictureAnz>-1)
   {
      pd[pNr]->borderColor=mo;
   }

   return;
}

/* Set Picture Frame Size */
void xwSetPictureF(int pNr,double size)
{
  if (pictureAnz>-1)
   {
      pd[pNr]->frameSize=size;
   }

   return;
}

/* Set Picture Frame Color */
void xwSetPictureC(int pNr,int rgb)
{
   if (pictureAnz>-1)
   {
      pd[pNr]->frmrgb=rgb;
   }

   return;
}

/* Set Picture Transparency */
void xwSetPictureT(int pNr,int trans)
{
   if (pictureAnz>-1)
   {
      pd[pNr]->transparenz=trans;
   }

   return;
}

/* Set Picture Transparency Mode */
void xwSetPictureTM(int pNr,int mo)
{
   if (pictureAnz>-1)
   {
      pd[pNr]->transMode=mo;
   }

   return;
}


/* ################################################################### */
/* Picture Add and Get Filename */

/* Load Picture */
int xwPLoad(const char *filename,int xpos,int ypos,int xsize,int ysize)
{
   /* Reserve and initialize picture entry 49 */
   if (pictureAnz<(picMaxPictures-1))
   {
      pictureAnz++;

      /* reserve memory */
      pd[pictureAnz]= (picturedatas*) malloc(sizeof(picturedatas)+strlen(filename));
      if (pd[pictureAnz]!=NULL)
      {
         /* load and check image */
         if (xwCheckGFXFile(filename)!=-1)
         {
            /* Delete Wysiwyg Cache Entry */
            /*xwDeletePBEntry(filename);*/

            /* Image Supported */
            strcpy(pd[pictureAnz]->filename,filename);

            /* copy image datas to picture data */
            pd[pictureAnz]->xpic=xwGetGFXX();
            pd[pictureAnz]->ypic=xwGetGFXY();
            /* picture aspect */
            pd[pictureAnz]->paspect=(float) pd[pictureAnz]->xpic / (float) pd[pictureAnz]->ypic;
            /* Set Default PicRotation */
            pd[pictureAnz]->rotate=picRotation;

            /* Active Picture */
            picActive=pictureAnz;

            pd[pictureAnz]->pFlag=0;


            pd[pictureAnz]->borderSize=0;
            pd[pictureAnz]->borderColor=0; /* Paper */
            pd[pictureAnz]->frameSize=0;
            pd[pictureAnz]->frmrgb=0;     /* Black */

            pd[pictureAnz]->transparenz=0;

            pd[pictureAnz]->transMode=0;

            /* scale picture into frame */
            xwScalePicture(pictureAnz,xpos,ypos,xsize,ysize);

            return(pictureAnz);
         }
         else
         {
            /* free memory allocation */
            free(pd[pictureAnz]);
            pictureAnz--;

            printf("<%s> incorrect image format !!!\n",filename);
            return(-1);
         }
      }
      else
      {
         /* failed memory allocation */
         pictureAnz--;
         printf("Not enough Memory for Picture Datas !!!\n");
         return(-1);
      }
   }

   return(-1);
}

/* Get Picture Filename */
const char *xwGetPictureFile(int pNr)
{
   if (pictureAnz>-1)
   {
      return(pd[pNr]->filename);
   }

   return("");
}

/* ################################################################### */

/* Scale Picture into Frame */
void xwScalePicture(int pNr,int xpos,int ypos,int xsize,int ysize)
{
   int         ax,ay;

   if ( (pd[pNr]->rotate==-90) || (pd[pNr]->rotate==90) )
   {
      /* rotated picture calculation */

      /* 1st calculation - Y-Size calc */
      ay=IRound( (float) xsize * pd[pNr]->paspect );
      ax=IRound( (float) ysize / pd[pNr]->paspect );

      /* 1st calculation correct */
      if (ay<=ysize)
      {
         /* picture size */
         pd[pNr]->xsize=ay; /* xsize; */
         pd[pNr]->ysize=xsize; /* ay; */

         /* picture position */
         pd[pNr]->xpos=xpos;
         pd[pNr]->ypos=ypos + ((ysize-ay)/2);
      }
      else
      {
         /* picture size */
         pd[pNr]->xsize=ysize; /* ax; */
         pd[pNr]->ysize=ax; /* ysize; */

         /* picture position */
         pd[pNr]->xpos=xpos+ ((xsize-ax)/2);
         pd[pNr]->ypos=ypos;
      }

   }
   else
   {
      /* none rotated picture ( 0,-180,180 Grad rotation ) */

      /* 1st calculation - Y-Size calc */
      ay=IRound( (float) xsize / pd[pNr]->paspect );
      ax=IRound( (float) ysize * pd[pNr]->paspect );

      /* 1st calculation correct */
      if (ay<=ysize)
      {
         /* picture size */
         pd[pNr]->xsize=xsize;
         pd[pNr]->ysize=ay;

         /* picture position */
         pd[pNr]->xpos=xpos;
         pd[pNr]->ypos=ypos + ((ysize-ay)/2);
      }
      else
      {
         /* picture size */
         pd[pNr]->xsize=ax;
         pd[pNr]->ysize=ysize;

         /* picture position */
         pd[pNr]->xpos=xpos+ ((xsize-ax)/2);
         pd[pNr]->ypos=ypos;
      }
   }

   return;
}

/* Draw Picture */
void xwDrawPicture(int pNr)
{
   int         az;

   /* Draw all Pictures */
   if (pNr==1234)
   {
      for (az=0 ; az<=pictureAnz ; az++)
      {
         xwDrawPicture(az);
      }

   }
   else
   {
      /* Draw picture 'pNr' */
      if (pNr<=pictureAnz)
      {
         /* 90 Grad rotation */
         if ( (pd[pNr]->rotate==-90) || (pd[pNr]->rotate==90) )
         {
            if (pNr==picActive)
            {
               xwDraw(drawAPic,pd[pNr]->xpos,pd[pNr]->ypos,pd[pNr]->ysize,pd[pNr]->xsize);
            }
            else
            {
               xwDraw(drawPic,pd[pNr]->xpos,pd[pNr]->ypos,pd[pNr]->ysize,pd[pNr]->xsize);
            }
         }
         else
         {
            if (pNr==picActive)
            {
               xwDraw(drawAPic,pd[pNr]->xpos,pd[pNr]->ypos,pd[pNr]->xsize,pd[pNr]->ysize);
            }
            else
            {
               xwDraw(drawPic,pd[pNr]->xpos,pd[pNr]->ypos,pd[pNr]->xsize,pd[pNr]->ysize);
            }
         }

         /* Draw Titlebar */
         if (pd[pNr]->rotate==-90)  xwDraw(drawTitlebar,pd[pNr]->xpos,pd[pNr]->ypos,picTitle*xwMM(),pd[pNr]->xsize);
         if (pd[pNr]->rotate==0)    xwDraw(drawTitlebar,pd[pNr]->xpos,pd[pNr]->ypos,pd[pNr]->xsize,picTitle*xwMM());
         if (pd[pNr]->rotate==90)   xwDraw(drawTitlebar,(pd[pNr]->xpos+pd[pNr]->ysize)-(picTitle*xwMM()),pd[pNr]->ypos,picTitle*xwMM(),pd[pNr]->xsize);
         if (pd[pNr]->rotate==-180) xwDraw(drawTitlebar,pd[pNr]->xpos,(pd[pNr]->ypos+pd[pNr]->ysize)-(picTitle*xwMM()),pd[pNr]->xsize,picTitle*xwMM());
         if (pd[pNr]->rotate==180)  xwDraw(drawTitlebar,pd[pNr]->xpos,(pd[pNr]->ypos+pd[pNr]->ysize)-(picTitle*xwMM()),pd[pNr]->xsize,picTitle*xwMM());
      }
   }

   return;
}

/* Print any Picture Information */
void xwPrintPictureDatas(int picNr,int mode,int xsize,int ysize)
{
   char           astr[40];

   if ((pictureAnz>-1) && (picNr>-1))
   {
      /* Portrait */
      if (mode==0)
      {
         /* Portrait */

         /* Left */
         sprintf(astr,"%.1f",(float) pd[picNr]->xpos/xwMM());
         xwGeoPrint(geoLeft,astr);

         /* Top */
         sprintf(astr,"%.1f",(float) pd[picNr]->ypos/xwMM());
         xwGeoPrint(geoTop,astr);

         /* Rotated Picture */
         if ((pd[picNr]->rotate==-90) || (pd[picNr]->rotate==90) )
         {
            /* Width */
            sprintf(astr,"%.1f",(float) pd[picNr]->ysize/xwMM());
            xwGeoPrint(geoWidth,astr);
            /* Height */
            sprintf(astr,"%.1f",(float) pd[picNr]->xsize/xwMM());
            xwGeoPrint(geoHeight,astr);

            /* Right */
            sprintf(astr,"%.1f",(float) (xsize-(pd[picNr]->xpos+pd[picNr]->ysize))/xwMM());
            xwGeoPrint(geoRight,astr);
            /* Bottom */
            sprintf(astr,"%.1f",(float) (ysize-(pd[picNr]->ypos+pd[picNr]->xsize))/xwMM());
            xwGeoPrint(geoBottom,astr);

         }
         else
         {
            /* Width */
            sprintf(astr,"%.1f",(float) pd[picNr]->xsize/xwMM());
            xwGeoPrint(geoWidth,astr);
            /* Height */
            sprintf(astr,"%.1f",(float) pd[picNr]->ysize/xwMM());
            xwGeoPrint(geoHeight,astr);

            /* Right */
            sprintf(astr,"%.1f",(float) (xsize-(pd[picNr]->xpos+pd[picNr]->xsize))/xwMM());
            xwGeoPrint(geoRight,astr);
            /* Bottom */
            sprintf(astr,"%.1f",(float) (ysize-(pd[picNr]->ypos+pd[picNr]->ysize))/xwMM());
            xwGeoPrint(geoBottom,astr);
         }

               /* Rotated Picture */
         if ((pd[picNr]->rotate==-90) || (pd[picNr]->rotate==90) )
         {
            /* DPI - Calculation */
            sprintf(astr,"%d",(int) ( (((float) pd[picNr]->ypic/(float) pd[picNr]->ysize)*xwMM())*25.4));
         }
         else
         {
            /* DPI - Calculation */
            sprintf(astr,"%d",(int) ( (((float) pd[picNr]->xpic/(float) pd[picNr]->xsize)*xwMM())*25.4));
         }
         xwGeoPrint(geoDPI,astr);
      }
      else
      {
         /* Landscape */

         /* Rotated Picture */
         if ((pd[picNr]->rotate==-90) || (pd[picNr]->rotate==90) )
         {
             /* Left */
            sprintf(astr,"%.1f",(float) (ysize-(pd[picNr]->xsize+pd[picNr]->ypos))/xwMM());
            xwGeoPrint(geoLeft,astr);

            /* Top */
            sprintf(astr,"%.1f",(float) pd[picNr]->xpos/xwMM());
            xwGeoPrint(geoTop,astr);

            /* Width */
            sprintf(astr,"%.1f",(float) pd[picNr]->xsize/xwMM());
            xwGeoPrint(geoWidth,astr);
            /* Height */
            sprintf(astr,"%.1f",(float) (pd[picNr]->ysize)/xwMM());
            xwGeoPrint(geoHeight,astr);

            /* Right */
            sprintf(astr,"%.1f",(float) pd[picNr]->ypos/xwMM());
            xwGeoPrint(geoRight,astr);
            /* Bottom */
            sprintf(astr,"%.1f",(float) (xsize-(pd[picNr]->xpos+pd[picNr]->ysize))/xwMM());
            xwGeoPrint(geoBottom,astr);
         }
         else
         {
            /* Left */
            sprintf(astr,"%.1f",(float) (ysize-(pd[picNr]->ysize+pd[picNr]->ypos))/xwMM());
            xwGeoPrint(geoLeft,astr);

            /* Top */
            sprintf(astr,"%.1f",(float) pd[picNr]->xpos/xwMM());
            xwGeoPrint(geoTop,astr);

            /* Width */
            sprintf(astr,"%.1f",(float) pd[picNr]->ysize/xwMM());
            xwGeoPrint(geoWidth,astr);
            /* Height */
            sprintf(astr,"%.1f",(float) pd[picNr]->xsize/xwMM());
            xwGeoPrint(geoHeight,astr);

            /* Right */
            sprintf(astr,"%.1f",(float) pd[picNr]->ypos/xwMM());
            xwGeoPrint(geoRight,astr);
            /* Bottom */
            sprintf(astr,"%.1f",(float) (xsize-(pd[picNr]->xpos+pd[picNr]->xsize))/xwMM());
            xwGeoPrint(geoBottom,astr);
         }

         /* Rotated Picture */
         if ((pd[picNr]->rotate==-90) || (pd[picNr]->rotate==90) )
         {
            /* DPI - Calculation */
            sprintf(astr,"%d",(int) ( (((float) pd[picNr]->ypic/(float) pd[picNr]->ysize)*xwMM())*25.4));
         }
         else
         {
            /* DPI - Calculation */
            sprintf(astr,"%d",(int) ( (((float) pd[picNr]->xpic/(float) pd[picNr]->xsize)*xwMM())*25.4));
         }
         xwGeoPrint(geoDPI,astr);

      }
   }
   else
   {
      xwGeoPrint(geoLeft,"");
      xwGeoPrint(geoRight,"");
      xwGeoPrint(geoTop,"");
      xwGeoPrint(geoBottom,"");
      xwGeoPrint(geoWidth,"");
      xwGeoPrint(geoHeight,"");
      xwGeoPrint(geoDPI,"");
   }

   return;
}

/* Create Picture Choice */
void xwCreatePictureChoice(FL_OBJECT *gd)
{
   int            az;
   char           astr[1024];
   char           *strptr;

   if (gd!=NULL)
   {
      fl_clear_browser(gd);

      /* exits pictures ? */
      if (pictureAnz>-1)
      {
         for (az=0 ; az<=pictureAnz ; az++)
         {
            strcpy(astr,pd[az]->filename);
            while ( fl_get_string_width(gd->lstyle,gd->lsize,astr,strlen(astr)) >= gd->w-10)
            {
               strptr=&astr[1];
               strcpy(astr,strptr);
            }
            /*
            fl_add_browser_line(gd,astr);
            */
            fl_insert_browser_line(gd,1,astr);
         }
         fl_select_browser_line(gd,(pictureAnz+1)-picActive);
      }
   }
   else
   {
      /* exist pictures ? */
      if ( (pictureAnz>-1) && (picActive>-1) )
      {
         strcpy(astr,pd[picActive]->filename);
         while ( fl_get_string_width(mf.SelectFP->lstyle,mf.SelectFP->lsize,astr,strlen(astr)) >= mf.SelectFP->w-10)
         {
            strptr=&astr[1];
            strcpy(astr,strptr);
         }
         fl_set_object_label(mf.SelectFP,astr);
      }
      else
      {
         fl_set_object_label(mf.SelectFP,"");
      }
   }

   return;
}


/* MovePictureToBack */
void xwMovePictureToBack(void)
{
   /* More as one Picture ? */
   if (pictureAnz>0)
   {
      /* Is the First Picture the active Picture ? */
      if (picActive>0)
      {
         xwSwapPicture(picActive,picActive-1);
         picActive--;
      }
   }

   return;
}

/* MovePictureToFront */
void xwMovePictureToFront(void)
{
   /* More as one Picture ? */
   if (pictureAnz>0)
   {
      /* Is the active Picture the last picture ? */
      if (picActive<pictureAnz)
      {
         xwSwapPicture(picActive,picActive+1);
         picActive++;
      }
   }

   return;
}

/* Rotate Picture */
void xwRotatePicture(int direction,int xpos,int ypos,int xsize,int ysize)
{
   /* exists pictures ? */
   if (pictureAnz>-1)
   {

      /* Left */
      if (direction == rotateLeft)
      {
         /* Calculate Rotation Grad */
         if (pd[picActive]->rotate == 0) pd[picActive]->rotate=-90;
         else if (pd[picActive]->rotate == -90) pd[picActive]->rotate=-180;
         else if (pd[picActive]->rotate == -180) pd[picActive]->rotate=90;
         else if (pd[picActive]->rotate == 180) pd[picActive]->rotate=90;
         else if (pd[picActive]->rotate == 90) pd[picActive]->rotate=0;

         /* scale picture into frame */
         xwScalePicture(picActive,xpos,ypos,xsize,ysize);
      }

      /* Right */
      if (direction == rotateRight)
      {
         /* Calculate Rotation Grad */
         if (pd[picActive]->rotate == 0) pd[picActive]->rotate=90;
         else if (pd[picActive]->rotate == 90) pd[picActive]->rotate=180;
         else if (pd[picActive]->rotate == 180) pd[picActive]->rotate=-90;
         else if (pd[picActive]->rotate == -180) pd[picActive]->rotate=-90;
         else if (pd[picActive]->rotate == -90) pd[picActive]->rotate=0;

         /* scale picture into frame */
         xwScalePicture(picActive,xpos,ypos,xsize,ysize);
      }
   }

   return;
}

/* Resize Picture */
void xwPictureResize(int mode)
{
   int            aw,ah;
   int            ay;

   /* Portrait Move */
   if (xwIsPaperLS()==xwPortrait)
   {
      /* lower Size */
      if ( (mode<0) && (picActive>-1) )
      {
         aw=pd[picActive]->xsize+mode;
         if (aw<picMinSize) aw=picMinSize;
         ah=(int) ( (float) aw/(float) pd[picActive]->paspect );
         if (ah<picMinSize)
         {
            ah=picMinSize;
            aw=(int) ( (float) ah*(float) pd[picActive]->paspect );
         }
         pd[picActive]->xsize=aw;
         pd[picActive]->ysize=ah;
      }

      /* higher Size */
      if ( (mode>0) && (picActive>-1) )
      {
         /* 90 Grad rotation */
         if ( (pd[picActive]->rotate==-90) || (pd[picActive]->rotate==90) )
         {
            /* Rotated Picture */
            aw=pd[picActive]->ysize+mode;
            ah=(int) ( (float) aw*(float) pd[picActive]->paspect );

            /* Height */
            if (ah>(xwGetPaperYB()+xwGetPaperTB())-pd[picActive]->ypos)
            {
               ah=(xwGetPaperYB()+xwGetPaperTB())-pd[picActive]->ypos;
               aw=(int) ( (float) ah/(float) pd[picActive]->paspect );
            }
            /* Width */
            if (aw>(xwGetPaperXB()+xwGetPaperLB())-pd[picActive]->xpos)
            {
               aw=(xwGetPaperXB()+xwGetPaperLB())-pd[picActive]->xpos;
               ah=(int) ( (float) aw*(float) pd[picActive]->paspect );
            }

            pd[picActive]->xsize=ah;
            pd[picActive]->ysize=aw;
         }
         else
         {
            /* Normal Picture */
            aw=pd[picActive]->xsize+mode;
            ah=(int) ( (float) aw/(float) pd[picActive]->paspect );

            /* Height */
            if (ah>(xwGetPaperYB()+xwGetPaperTB())-pd[picActive]->ypos)
            {
               ah=(xwGetPaperYB()+xwGetPaperTB())-pd[picActive]->ypos;
               aw=(int) ( (float) ah*(float) pd[picActive]->paspect );
            }
            /* Width */
            if (aw>(xwGetPaperXB()+xwGetPaperLB())-pd[picActive]->xpos)
            {
               aw=(xwGetPaperXB()+xwGetPaperLB())-pd[picActive]->xpos;
               ah=(int) ( (float) aw/(float) pd[picActive]->paspect );
            }

            pd[picActive]->xsize=aw;
            pd[picActive]->ysize=ah;
         }
      }
   }
   else
   {
      /* Landscape */

      /* lower Size */
      if ( (mode<0) && (picActive>-1) )
      {
         /* 90 Grad rotation */
         if ( (pd[picActive]->rotate==-90) || (pd[picActive]->rotate==90) )
         {
            /* Normal Picture */
            aw=pd[picActive]->xsize+mode;
            if (aw<picMinSize)
            {
               aw=picMinSize;
            }
            ah=(int) ( (float) aw/(float) pd[picActive]->paspect );
            if (ah<picMinSize)
            {
               ah=picMinSize;
               aw=(int) ( (float) ah*(float) pd[picActive]->paspect );
            }
            pd[picActive]->ypos=pd[picActive]->ypos+(pd[picActive]->xsize-aw);

            pd[picActive]->xsize=aw;
            pd[picActive]->ysize=ah;
         }
         else
         {
            /* Normal Picture */
            ah=pd[picActive]->ysize+mode;
            if (ah<picMinSize)
            {
               ah=picMinSize;
            }
            aw=(int) ( (float) ah*(float) pd[picActive]->paspect );
            if (aw<picMinSize)
            {
               aw=picMinSize;
               ah=(int) ( (float) aw/(float) pd[picActive]->paspect );
            }
            pd[picActive]->ypos=pd[picActive]->ypos+(pd[picActive]->ysize-ah);

            pd[picActive]->xsize=aw;
            pd[picActive]->ysize=ah;
         }
      }

      /* higher Size */
      if ( (mode>0) && (picActive>-1) )
      {
         /* 90 Grad rotation */
         if ( (pd[picActive]->rotate==-90) || (pd[picActive]->rotate==90) )
         {
            /* Rotated Picture */
            ah=pd[picActive]->xsize+mode;
            aw=(int) ( (float) ah/(float) pd[picActive]->paspect );

            ay=pd[picActive]->ypos-mode;
            /* Check Y-Axis */
            if (ay<xwGetPaperTB())
            {
               ay=xwGetPaperTB();
               ah=pd[picActive]->xsize+(pd[picActive]->ypos-xwGetPaperTB());
               aw=(int) ( (float) ah / (float) pd[picActive]->paspect );
            }

            /* Check X-Axis */
            if (pd[picActive]->xpos+aw>xwGetPaperXSize()-xwGetPaperRB())
            {
               aw=(xwGetPaperXSize()-xwGetPaperRB())-pd[picActive]->xpos;
               ah=(int) ( (float) aw*(float) pd[picActive]->paspect );

               ay=pd[picActive]->ypos+(pd[picActive]->xsize-ah);
            }

            pd[picActive]->ypos=ay;

            pd[picActive]->xsize=ah;
            pd[picActive]->ysize=aw;

         }
         else
         {
            /* Normal Picture */
            ah=pd[picActive]->ysize+mode;
            aw=(int) ( (float) ah*(float) pd[picActive]->paspect );

            ay=pd[picActive]->ypos-mode;
            /* Check Y-Axis */
            if (ay<xwGetPaperTB())
            {
               ay=xwGetPaperTB();
               ah=pd[picActive]->ysize+(pd[picActive]->ypos-xwGetPaperTB());
               aw=(int) ( (float) ah * (float) pd[picActive]->paspect );
            }

            /* Check X-Axis */
            if (pd[picActive]->xpos+aw>xwGetPaperXSize()-xwGetPaperRB())
            {
               aw=(xwGetPaperXSize()-xwGetPaperRB())-pd[picActive]->xpos;
               ah=(int) ( (float) aw/(float) pd[picActive]->paspect );

               ay=pd[picActive]->ypos+(pd[picActive]->ysize-ah);
            }

            pd[picActive]->ypos=ay;

            pd[picActive]->xsize=aw;
            pd[picActive]->ysize=ah;
         }
      }
   }

   return;
}

/* Move Picture */
void xwPictureMove(int xpos,int ypos)
{
   int            ax,ay;

   /* Is Picture Selcted */
   if (picActive>-1)
   {
      /* Portrait Move */
      if (xwIsPaperLS()==xwPortrait)
      {
         /* Up */
         if ( (xpos==0) && (ypos<0) )
         {
            ay=pd[picActive]->ypos+ypos;
            if (ay<xwGetPaperTB()) ay=xwGetPaperTB();
            pd[picActive]->ypos=ay;
         }

         /* Left */
         if ( (xpos<0) && (ypos==0) )
         {
            ax=pd[picActive]->xpos+xpos;
            if (ax<xwGetPaperLB()) ax=xwGetPaperLB();
            pd[picActive]->xpos=ax;
         }

         /* Right */
         if ( (xpos>0) && (ypos==0) )
         {
            /* Rotated Picture */
            if ( (pd[picActive]->rotate==-90) || (pd[picActive]->rotate==90) )
            {
               ax=pd[picActive]->xpos+xpos;
               if (ax+pd[picActive]->ysize > xwGetPaperXSize()-xwGetPaperRB()) ax= (xwGetPaperXSize()-xwGetPaperRB())-pd[picActive]->ysize;
               pd[picActive]->xpos=ax;
            }
            else
            {
               /* Normal Picture */
               ax=pd[picActive]->xpos+xpos;
               if (ax+pd[picActive]->xsize > xwGetPaperXSize()-xwGetPaperRB()) ax= (xwGetPaperXSize()-xwGetPaperRB())-pd[picActive]->xsize;
               pd[picActive]->xpos=ax;
            }
         }

         /* Down */
         if ( (xpos==0) && (ypos>0) )
         {
            /* Rotated Picture */
            if ( (pd[picActive]->rotate==-90) || (pd[picActive]->rotate==90) )
            {
               ay=pd[picActive]->ypos+ypos;
               if (ay+pd[picActive]->xsize > xwGetPaperYSize()-xwGetPaperBB()) ay= (xwGetPaperYSize()-xwGetPaperBB())-pd[picActive]->xsize;
               pd[picActive]->ypos=ay;
            }
            else
            {
               /* Normal Picture */
               ay=pd[picActive]->ypos+ypos;
               if (ay+pd[picActive]->ysize > xwGetPaperYSize()-xwGetPaperBB()) ay= (xwGetPaperYSize()-xwGetPaperBB())-pd[picActive]->ysize;
               pd[picActive]->ypos=ay;
            }
         }
      }
      else
      {
         /* Landscape Move */

         /* Up */
         if ( (xpos==0) && (ypos<0) )
         {
            ax=pd[picActive]->xpos+ypos;
            if (ax<xwGetPaperLB()) ax=xwGetPaperLB();
            pd[picActive]->xpos=ax;
         }

         /* Left */
         if ( (xpos<0) && (ypos==0) )
         {
            /* Rotated Picture */
            if ( (pd[picActive]->rotate==-90) || (pd[picActive]->rotate==90) )
            {
               ay=pd[picActive]->ypos+(xpos*(-1));
               if (ay+pd[picActive]->xsize > xwGetPaperYSize()-xwGetPaperBB()) ay= (xwGetPaperYSize()-xwGetPaperBB())-pd[picActive]->xsize;
               pd[picActive]->ypos=ay;
            }
            else
            {
               /* Normal Picture */
               ay=pd[picActive]->ypos+(xpos*(-1));
               if (ay+pd[picActive]->ysize > xwGetPaperYSize()-xwGetPaperBB()) ay= (xwGetPaperYSize()-xwGetPaperBB())-pd[picActive]->ysize;
               pd[picActive]->ypos=ay;
            }
         }

         /* Right */
         if ( (xpos>0) && (ypos==0) )
         {
            ay=pd[picActive]->ypos+(xpos*(-1));
            if (ay<xwGetPaperTB()) ay=xwGetPaperTB();
            pd[picActive]->ypos=ay;
         }

         /* Down */
         if ( (xpos==0) && (ypos>0) )
         {
            /* Rotated Picture */
            if ( (pd[picActive]->rotate==-90) || (pd[picActive]->rotate==90) )
            {
               ax=pd[picActive]->xpos+ypos;
               if (ax+pd[picActive]->ysize > xwGetPaperXSize()-xwGetPaperRB()) ax= (xwGetPaperXSize()-xwGetPaperRB())-pd[picActive]->ysize;
               pd[picActive]->xpos=ax;
            }
            else
            {
               /* Normal Picture */
               ax=pd[picActive]->xpos+ypos;
               if (ax+pd[picActive]->xsize > xwGetPaperXSize()-xwGetPaperRB()) ax= (xwGetPaperXSize()-xwGetPaperRB())-pd[picActive]->xsize;
               pd[picActive]->xpos=ax;
            }
         }
      }
   }

   return;
}

/* Page Information Routines */

/* Get lowest X-Picture-Position */
int xwGetLowXPicture(void)
{
   int            ax,az;

   ax=-1;

   if (pictureAnz>-1)
   {
      for (az=0 ; az<=pictureAnz ; az++)
      {
         if (ax==-1) ax=pd[az]->xpos;

         if (pd[az]->xpos<ax) ax=pd[az]->xpos;
      }
   }

   return(ax);
}

/* Get lowest Y-Picture-Position */
int xwGetLowYPicture(void)
{
   int            ay,az;

   ay=-1;

   if (pictureAnz>-1)
   {
      for (az=0 ; az<=pictureAnz ; az++)
      {
         if (ay==-1) ay=pd[az]->ypos;

         if (pd[az]->ypos<ay) ay=pd[az]->ypos;
      }
   }

   return(ay);
}

/* Get highest X-Picture-Position */
int xwGetHighXPicture(void)
{
   int            ax,az;

   ax=-1;

   if (pictureAnz>-1)
   {
      for (az=0 ; az<=pictureAnz ; az++)
      {
         if (ax==-1) ax=pd[az]->xpos+pd[az]->xsize;

         if (pd[az]->xpos+pd[az]->xsize>ax) ax=pd[az]->xpos+pd[az]->xsize;
      }
   }

   return(ax);
}

/* Get highest Y-Picture-Position */
int xwGetHighYPicture(void)
{
   int            ay,az;

   ay=-1;

   if (pictureAnz>-1)
   {
      for (az=0 ; az<=pictureAnz ; az++)
      {
         if (ay==-1) ay=pd[az]->ypos+pd[az]->ysize;

         if (pd[az]->ypos+pd[az]->ysize>ay) ay=pd[az]->ypos+pd[az]->ysize;
      }
   }

   return(ay);
}

/* ################################################################### */
/* View Picture */
void xwDisplayPicture(void)
{
   if ( (pictureAnz>-1) && (picActive>-1) )
   {
      xwViewPicture(pd[picActive]->filename);
   }

   return;
}