/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "paper.h"

#include "base.h"
#include "main_gui.h"
#include "lowlevel.h"

struct paperdatas   paper;

/* Paper Datas */
static struct paperdef paperdata[]=
{ /* ID  Paper Name                      Width Height in mm */
   {  9,"DIN A6 - 105 x 148 mm"         ,105   ,148   },
   {  0,"DIN A5 - 148 x 210 mm"         ,148   ,210   },
   {  1,"DIN A4 - 210 x 297 mm"         ,210   ,297   },
   {  2,"DIN A3 - 297 x 420 mm"         ,297   ,420   },
   {  3,"Executive - 7.25 x 10.5 Inch"  ,190.5 ,266.7 },
   {  4,"Letter - 8.5 x 11 Inch"        ,215.9 ,279.4 },
   {  5,"Legal - 8.5 x 14 Inch"         ,215.9 ,355.6 },
   {  6,"Ledger - 11 x 17 Inch"         ,279.4 ,431.8 },
   {  7,"Tabloid - 11 x 17 Inch"        ,279.4 ,431.8 },
   {  8,"Panorama - 210 x 594 mm"       ,210   ,594   },

   /* TurboPrint Papers */ 
   { 10,"Half Letter 5,50 x 8,50 Inch"  ,139.7 ,266.7 },
   { 11,"B5"                            ,176   ,250   },
   { 12,"Card 5 x 8 Inch"               ,127   ,203.2 },
   { 13,"Card 8 x 10 Inch"              ,203.2 ,254   },
   { 14,"Envelope #10"                  ,241.3 ,104.8 },
   { 15,"Envelope DL"                   ,219.9 ,110   },
   { 16,"Envelope C6"                   ,162   ,114   },
   { 17,"Evelope 132 x 220 mm"          ,219.9 ,132   },
   { 18,"Photopapier 4 x 6 Inch"        ,101.6 ,152.4 },
   { 19,"Photopapier 10 x 15 cm"        ,100   ,150   },
   { 20,"Photopapier 20 x 30 cm"        ,200   ,300   },
   
   /* Endmark */
   {999,""                              ,-1,-1}
};

static int        paperIndex=0;

/* Init Paper */
void xwInitPaper(void)
{
   /* Paper Orientation */
   paper.landscape=xwPortrait;
   
   /* Paper Size - Default Paper ID=1 */
   paper.page=1; /* <-- ID */
   xwSetPaperSize(paper.page,0,0);
   
   /* Borders */
   paper.pagel=10*xwMM();
   paper.pager=10*xwMM();
   paper.paget=10*xwMM();
   paper.pageb=15*xwMM();

   return;
}

/* ############################################### */

/* GetPaperXSize */
int xwGetPaperXSize(void)
{
   return(paper.pagex);
}

/* GetPaperYSize */
int xwGetPaperYSize(void)
{
   return(paper.pagey);
}

/* Get Left Border */
int xwGetPaperLB(void)
{
   return(paper.pagel);
}

/* Get Right Border */
int xwGetPaperRB(void)
{
   return(paper.pager);
} 

/* Get Top Border */
int xwGetPaperTB(void)
{
   return(paper.paget);
}

/* Get Bottom Border */
int xwGetPaperBB(void)
{
   return(paper.pageb);
}  

/* Get X-Size - L+R-Borders */
int xwGetPaperXB(void)
{
   return(paper.pagex-(paper.pagel+paper.pager));
}

/* Get Y-Size - T+B-Borders */
int xwGetPaperYB(void)
{
   return(paper.pagey-(paper.paget+paper.pageb));
}

/* ### Paper Managment ### */

/* Get First Paper */
char *xwGetPaperName(void)
{
   paperIndex=0;
   return(paperdata[paperIndex].name);
}

/* Get Next Paper */
char *xwGetNextPaperName(void)
{
   if (paperdata[paperIndex].ID!=999)
   {
      paperIndex++;
      return(paperdata[paperIndex].name);
   }

   return("");
}

/* Get Entry Number */
int xwGetEntryNumber(int ID)
{
   int            az=-1;
   
   if ( (ID>-1) && (ID<999) )
   {
      paperIndex=0;
      while (paperdata[paperIndex].ID!=999)
      {
         if (paperdata[paperIndex].ID==ID) az=paperIndex;
         paperIndex++;
      }
   }
   
   return(az);
}

/* Get ID Number */
int xwGetIDNumber(int Entry)
{
   int            az=-1;
   
   /* Count Entrys */
   paperIndex=0;
   while (paperdata[paperIndex].ID!=999)
   {
      paperIndex++;
   }
      
   /* Exist Entry */
   if ( (Entry>-1) && (Entry<paperIndex) )
   {
      az=paperdata[Entry].ID;
   }
   
   return(az);
}

/* Get ID Number */
int xwGetID(void)
{
   float          fxz,fyz;             /* minimal distance */
   int            fxa,fya;             /* paper entry */
   
   int            paperID;   

   fxz=10000;
   fyz=10000;
   fxa=-1;
   fya=-1;

   paperID=paper.page;

   paperIndex=0;
   while (paperdata[paperIndex].ID!=999)
   {
      /* page greater source page */
      if ( (((paperdata[paperIndex].width*xwMM()) -paper.pagex)>=0) && (((paperdata[paperIndex].height*xwMM()) -paper.pagey)>=0) )
      {
         if ( (((paperdata[paperIndex].width*xwMM()) -paper.pagex)<fxz) && (((paperdata[paperIndex].width*xwMM()) -paper.pagex)>=0) )
         {
            fxz=((paperdata[paperIndex].width*xwMM()) -paper.pagex);
            fxa=paperdata[paperIndex].ID;
         }
         
         if ( (((paperdata[paperIndex].height*xwMM()) -paper.pagey)<fyz) && (((paperdata[paperIndex].height*xwMM()) -paper.pagey)>=0)  )
         {
            fyz=((paperdata[paperIndex].height*xwMM()) -paper.pagey);
            fya=paperdata[paperIndex].ID;
         }            
      }      
      paperIndex++;
   }
   
   if (fxz<fyz)
   {
      paperID=fxa;
   }
   else
   {
      paperID=fya;
   } 

   return(paperID);
}

/* Set ID Number */
void xwSetID(int Entry)
{
   paper.page=Entry;
   return;
}

/* Get Paper X Size */
float xwGetPaperWidth(int Entry)
{
   float          az=-1;
   
   /* Count Entrys */
   paperIndex=0;
   while (paperdata[paperIndex].ID!=999)
   {
      paperIndex++;
   }
      
   /* Exist Entry */
   if ( (Entry>-1) && (Entry<paperIndex) )
   {
      az=paperdata[Entry].width*xwMM();
   }
   
   return(az);
}   

/* Get Paper Y Size */
float xwGetPaperHeight(int Entry)
{
   float          az=-1;
   
   /* Count Entrys */
   paperIndex=0;
   while (paperdata[paperIndex].ID!=999)
   {
      paperIndex++;
   }
      
   /* Exist Entry */
   if ( (Entry>-1) && (Entry<paperIndex) )
   {
      az=paperdata[Entry].height*xwMM();
   }
   
   return(az);
}   

/* ############################################### */

/* IsPaperLS */
int xwIsPaperLS(void)
{
   return(paper.landscape);
}

/* Set PaperLS */
void xwSetPaperLS(int mo)
{
   paper.landscape=mo;
   return;
}

/* Set Paper X-Size */
void xwSetPaperX(float az)
{
   paper.pagex=az;
   
   /* Page Aspect Ratio */
   paper.paspect=paper.pagex/paper.pagey;
 
   return;
}

/* Set Paper Y-Size */
void xwSetPaperY(float az)
{
   paper.pagey=az;
   
   /* Page Aspect Ratio */
   paper.paspect=paper.pagex/paper.pagey;
 
   return;
}

/* Set Left Border */
void xwSetPaperLB(int az)
{
   paper.pagel=az;
   return;
} 

/* Set Right Border */
void xwSetPaperRB(int az)
{
   paper.pager=az;
   return;
} 

/* Set Top Border */
void xwSetPaperTB(int az)
{
   paper.paget=az;
   return;
} 

/* Set Bottom Border */
void xwSetPaperBB(int az)
{
   paper.pageb=az;
   return;
} 


/* Set Paper Size */
void xwSetPaperSize(int mo,int xx,int yy)
{
   float          x,y;
   int            a;

   a=0;
   while (paperdata[a].ID!=999)
   {
      if (mo==paperdata[a].ID)
      {
         x=paperdata[a].width*xwMM();
         y=paperdata[a].height*xwMM();
      }
   
      a++;
   }

   /* Custom Paper */
   if (mo == 998)
   {
      x = xx;
      y = yy;
   }

   /* PaperData */
   paper.pagex = x;
   paper.pagey = y;

   /* Page Aspect Ratio */
   paper.paspect=paper.pagex/paper.pagey;

   return;
}

/* Draw Paper */
void xwDrawPaper(void)
{   
   /* Set Pixel Size */
   xwSetPixelSize(paper.pagex,paper.pagey);
   
   /* Draw Paper */
   xwDraw(drawPaper,0,0,paper.pagex,paper.pagey);  

   /* Draw Border Frame */
   xwDraw(drawBorder,paper.pagel,paper.paget,paper.pagex-(paper.pagel+paper.pager),paper.pagey-(paper.paget+paper.pageb));
   
   return;
}

