/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "main_action.h"

/* xwGUI Includes */
#include "about.h"
#include "assistent.h"
#include "base.h"
#include "default.h"
#include "editor.h"
#include "error.h"
#include "fileio.h"
#include "fileselect.h"
#include "frame.h"
#include "frameset.h"
#include "gfx.h"
#include "handler.h"
#include "help.h"
#include "interpreter.h"
#include "io.h"
#include "limits.h"
#include "lowlevel.h"
#include "main_gui.h"
#include "page.h"
#include "paper.h"
#include "picbrowser.h"
#include "picture.h"
#include "pref_paper.h"
#include "pref_printer.h"
#include "pref_xwgui.h"
#include "printer.h"
#include "progress.h"
#include "render.h"
#include "resource.h"
#include "selectfp.h"
#include "variable.h"

extern struct mainform     mf;

static struct printform    prf;

static float               SROld=-1,SRNew=-1;
static float               SBOld=-1,SBNew=-1;

/* GUI Dispatcher */
void xwMGEvent(FL_OBJECT *obj)
{
   /* Select Picture/Frame Editor */
   if (obj == mf.FPSwitch) xwSelectFPEd();

   /* Picture Viewer */
   else if (obj == mf.Eye) xwViewFramePicture();

   /* Paper Zoom */
   else if (obj == mf.PaperZoom) xwZoom(0);

   /* Paper Zoom 100% */
   else if (obj == mf.PaperFull) xwZoom(1);

   /* About */
   else if (obj == mf.Info)
   {
      fl_deactivate_form(mf.form);
      xwAbout();
      fl_activate_form(mf.form);
   }

   /* Load */
   else if (obj == mf.Load) xwLoad();
   /* Save */
   else if (obj == mf.Save) xwSave();
   /* Picture Browser */
   else if (obj == mf.PicBrowser) xwPicBrowser();

   /* Pref Paper */
   else if (obj == mf.PaperPref) xwPaperPref();
   /* Pref xwGUI */
   else if (obj == mf.GuiPref) xwGuiPref();
   /* Pref Printer */
   else if (obj == mf.PrinterPref) xwPrinterPref();

   /* FrameSet */
   else if (obj == mf.FrameSets) xwFrameSets();
   /* Assistent */
   else if (obj == mf.Assistent) xwAssistents();
   /* MTInk */
   else if (obj == mf.MTInk) xwExec(xwGetPrefName(xwgMTInk),0,"");
   
   /* Trash */
   else if (obj == mf.Trash) xwTrash();
   /* HELP */
   else if (obj == mf.DocFile) xwDisplayHelp(HelpAll);
   /* EXIT */
   else if (obj == mf.Exit) xwExit();

   /* Landscape / Portrait switch */
   else if (obj == mf.Landscape) xwAcLandscape();
   /* Add Frame */
   else if (obj == mf.AddFrame) xwAdd();
   /* FEditor */
   else if (obj == mf.Edit) xwEdit();
   /* Frame/Picture Trashcan */
   else if (obj == mf.FPTrash) xwFPTrash();

   /* Move To Back */
   else if (obj == mf.Back) xwBack();
   /* Move To Front */
   else if (obj == mf.Front) xwFront();
   /* Rotate Left */
   else if (obj == mf.RotateLeft) xwRotate(rotateLeft);
   /* Rotate Right */
   else if (obj == mf.RotateRight) xwRotate(rotateRight);

   /* Preview */
   else if (obj == mf.Preview) xwPreview();
   /* Print */
   else if (obj == mf.Print) xwPrint();


   /* SelectFP Event Handling */
   xwSelectFP(obj,mf.SelectFP);

   return;
}

/* Select Frame/Picture Editor */
void xwSelectFPEd(void)
{
   if (xwCheckFrameMode()==0)
   {
      if (xwCheckEditFrame()==0)
      {
         xwDeActivateFrameEd();
         fl_set_button(mf.FPSwitch,0);
      }
      else
      {
         xwActivateFrameEd();
         fl_set_button(mf.FPSwitch,1);
      }
   }
   else
   {
      xwDisplayMessage(mf.form,xwGR(1,"err.FrameEdError","No Frames exists.\nCan't use the Frame Editor !!!"));
      fl_set_button(mf.FPSwitch,0);
   }

   fl_redraw_form(mf.form);
   xwDrawPage(pgRecalc);
   /*fl_redraw_form(mf.form);*/
   
/*
   xwSetPrefValue(xwgLayerXPos,mf.FPSwitch->x);
   xwSetPrefValue(xwgLayerYPos,mf.FPSwitch->y+mf.FPSwitch->h);
*/

   xwActualizeFP();

   return;
}

/* Paper Zoom */
void xwZoom(int az)
{
   float       aaz;
   int         zOld,zNew;

   zOld=xwGetZoomV();
   zNew=fl_get_choice(mf.PaperZoom);

   aaz=0;

   if (az==0)
   {

      if ((zNew>0) && (zNew!=zOld))
      {
         /* Set New Zoom Factor */
         xwSetZoomV(fl_get_choice(mf.PaperZoom));
         xwSetXOffset(0);
         xwSetYOffset(0);
         xwSetXXOffset(0);
         xwSetYYOffset(0);
         xwDrawPage(pgRecalc);
         aaz=1;
      }
   }
   else if (az==1)
   {
      xwSetZoomV(1);
      xwSetXOffset(0);
      xwSetYOffset(0);
      xwSetXXOffset(0);
      xwSetYYOffset(0);
      xwDrawPage(pgRecalc);
      aaz=1;
      fl_set_choice(mf.PaperZoom,1);
   }

   fl_set_scrollbar_value(mf.PaperSR,0);
   fl_set_scrollbar_value(mf.PaperSB,0);

   return;
}

/* Scrollbar Result */
void xwPaperScroller(FL_OBJECT *obj,long data)
{
   if (obj == mf.PaperSR)
   {
      /* Portrait */
      if (xwIsPaperLS()==xwPortrait)
      {
         /* Scroolbar */
         SRNew=fl_get_scrollbar_value(mf.PaperSR);
         xwSetYYOffset(SRNew);
         SRNew=( ( ((float) xwGetPaperYSize()*xwGetPixelSize()) /100)*(SRNew*100));

         /* View Entrys */
         if (SROld != SRNew)
         {
            SROld=SRNew;
            xwSetYOffset((int) SRNew);

            xwDrawPage(pgMove);
         }
      }
      /* Landscape */
      else if (xwIsPaperLS()==xwLandscape)
      {
         /* Scroolbar */
         SRNew=fl_get_scrollbar_value(mf.PaperSR);
         xwSetYYOffset(SRNew);
         SRNew=( ( ((float) xwGetPaperXSize()*xwGetPixelSize()) /100)*(SRNew*100));

         /* View Entrys */
         if (SROld != SRNew)
         {
            SROld=SRNew;
            xwSetYOffset((int) SRNew);

            xwDrawPage(pgMove);
         }
      }
   }
   else if (obj == mf.PaperSB)
   {
      /* Portrait */
      if (xwIsPaperLS()==xwPortrait)
      {
         /* Scroolbar */
         SBNew=fl_get_scrollbar_value(mf.PaperSB);
         xwSetXXOffset(SBNew);
         SBNew=( ( ((float) xwGetPaperXSize()*xwGetPixelSize()) /100)*(SBNew*100));

         /* View Entrys */
         if (SBOld != SBNew)
         {
            SBOld=SBNew;
            xwSetXOffset((int) SBNew);

            xwDrawPage(pgMove);
         }
      }
       /* Landscape */
      else if (xwIsPaperLS()==xwLandscape)
      {
         /* Scroolbar */
         SBNew=fl_get_scrollbar_value(mf.PaperSB);
         xwSetXXOffset(SBNew);
         SBNew=( ( ((float) xwGetPaperYSize()*xwGetPixelSize()) /100)*(SBNew*100));

         /* View Entrys */
         if (SBOld != SBNew)
         {
            SBOld=SBNew;
            xwSetXOffset((int) SBNew);

            xwDrawPage(pgMove);
         }
      }
   }

   return;
}

/* Load Pixmap or Frameset */
void xwLoad(void)
{
   char        astr[80],bstr[80],cstr[80];
   const char  *filename;

   xwDeactivateMain();

   /*if (xwGetPrefValue(xwgUsermode)==0) xwDisplayMessage(NULL,xwGR(1,"inf.xwLoad","You can load an picture, page or frameset with this fuction."));*/

   xwSetFSCustom("xwGUI2",xwGetFile("",""));
   xwSetFSPattern("frameset,page,ppm,pgm,pbm,png,bmp,jpg,jpeg,tif,tiff,xbm,xwd,xpm");
   strcpy(astr,xwGR(1,"fs.SelectPicture","Select an picture, please..."));
   strcpy(bstr,xwGR(1,"fs.Load","Load"));
   strcpy(cstr,xwGR(1,"fs.Abort","Abort"));
   filename=xwFileSelect(fsFile,"",astr,bstr,cstr);
   if (strlen(filename)>0)
   {
      /* Load Picture */
      if (xwFPLoad(filename)==-1)
      {
         /* no pictures loaded - check frameset */
         if (checkfrms(filename)!=-1)
         {

            if ( (xwGetFrameCount()>-1) || (xwGetPictureCount()>-1) )
            {
               if (xwAskMessage(NULL,xwGR(1,"msg.Trash","Delete all frames and pictures ?"))==xwYesMessage)
               {
                  xwFreePicture(-1);
                  xwInitFrames();
                  xwInitPicture();

                  /* ReDraw Page */
                  xwDrawPage(pgRecalc);

                  /* load frameset */
                  xwInterpret(filename,NULL);

                  /* Picture Editor */
                  xwDeActivateFrameEd();
               }
               fl_set_app_mainform(xwGetMainGui());
            }
            else
            {
               /* load frameset */
               xwInterpret(filename,NULL);

               /* Picture Editor */
               xwDeActivateFrameEd();
            }
         }
      }

      /* ReDraw Page */
      xwDrawPage(pgRecalc);
   }

   xwActivateMain();

   xwActualizeFP();

   return;
}

/* Save FrameSet */
void xwSave(void)
{
   char        astr[80],bstr[80],cstr[80];
   char        *filename;
   char        *strptr;

   xwDeactivateMain();

   /*if (xwGetPrefValue(xwgUsermode)==0) xwDisplayMessage(NULL,xwGR(1,"inf.xwSave","You can save an page or frameset with this fuction."));*/

   xwSetFSCustom("xwGUI2",xwGetFile("",""));
   if ( (xwGetPictureCount()==-1) && (xwGetFrameCount()>-1) )
   {
      xwSetFSPattern("frameset");
      strcpy(astr,xwGR(1,"fs.SelectFrameSet","Select an FrameSet..."));
   }
   else
   {
      xwSetFSPattern("page");
      strcpy(astr,xwGR(1,"fs.SelectPage","Select an Page..."));
   }

   strcpy(bstr,xwGR(1,"fs.Save","Save"));
   strcpy(cstr,xwGR(1,"fs.Abort","Abort"));
   filename=xwFileSelect(fsFile,"",astr,bstr,cstr);
   if (strlen(filename)>9)
   {
      /*check filename */
      if ( (xwGetPictureCount()==-1) && (xwGetFrameCount()>=-1) )
      {
         strptr=filename+(strlen(filename)-8);
         strcpy(astr,strptr);
         strUpper(astr);
         if (strcmp(astr,"FRAMESET") != 0)
         {
            strcat(filename,".frameset");
         }
         /* frame exists */
         if (xwGetFrameCount()>-1)
         {
            xwSaveFrameSet(filename);
         }
         else
         {
            xwDisplayMessage(NULL,xwGR(1,"CantSaveFrameSet","Can't save frameset !!!"));
            fl_set_app_mainform(xwGetMainGui());
         }
      }
      else
      {
         strptr=filename+(strlen(filename)-4);
         strcpy(astr,strptr);
         strUpper(astr);
         if (strcmp(astr,"PAGE") != 0)
         {
            strcat(filename,".page");
         }
        /* frame exists */
         if ( (xwGetFrameCount()>-1) || (xwGetPictureCount()>-1) )
         {
            xwSaveFrameSet(filename);
         }
         else
         {
            xwDisplayMessage(NULL,xwGR(1,"CantSavePage","Can't save page !!!"));
            fl_set_app_mainform(xwGetMainGui());
         }
      }
   }

   xwActivateMain();

   return;
}

/* Picture Browser */
void xwPicBrowser(void)
{
   char        *filename;

   xwDeactivateMain();

   /*if (xwGetPrefValue(xwgUsermode)==0) xwDisplayMessage(NULL,xwGR(1,"inf.xwPicBrowser","You can load an picture on this visible fileselection."));*/

   filename=xwPictureBrowser(mf.form);
   if (strlen(filename)>0)
   {
      /* Load Picture */
      xwFPLoad(filename);

      /* ReDraw Page */
      xwDrawPage(pgRecalc);
   }

   xwActivateMain();

   xwActualizeFP();

   return;
}

/* Paper Pref */
void xwPaperPref(void)
{
   xwDeactivateMain();
   if (strcmp(xwGetVarDef("LOCKPAPER"),"ON") != 0)
   {
      xwPrefPaperSetup();
   }
   else
   {
      xwPrefPrinterSetup();
   }
   xwActivateMain();

   return;
}

/* xwGUI Pref */
void xwGuiPref(void)
{
   xwDeactivateMain();
   xwPrefxwGUISetup();
   xwActivateMain();

   return;
}
/* Printer Pref */
void xwPrinterPref(void)
{
   xwDeactivateMain();
   xwPrefPrinterSetup();
   xwActivateMain();

   return;
}

/* FrameSet */
void xwFrameSets()
{
   xwDeactivateMain();

   if ( (xwGetFrameCount()>-1) || (xwGetPictureCount()>-1) )
   {
      if (xwAskMessage(NULL,xwGR(1,"msg.Trash","Delete all frames and pictures ?"))==xwYesMessage)
      {
         fl_set_app_mainform(xwGetMainGui());
         xwFreePicture(-1);
         xwInitFrames();
         xwInitPicture();
         /*xwDrawPage();*/
         xwFrameSetsGUI();
         /* Picture Editor */
         xwDeActivateFrameEd();
         xwDrawPage(pgRecalc);
      }
      else
      {
         fl_set_app_mainform(xwGetMainGui());
      }
   }
   else
   {
      xwFrameSetsGUI();
      /* Picture Editor */
      xwDeActivateFrameEd();
      xwDrawPage(pgRecalc);
   }

   xwActivateMain();

   xwActualizeFP();

   return;
}

/* Assistent */
void xwAssistents()
{
   xwDeactivateMain();

   if ( (xwGetFrameCount()>-1) || (xwGetPictureCount()>-1) )
   {
      if (xwAskMessage(NULL,xwGR(1,"msg.Assistent","Delete all frames an pictures by using the assistent.\nYou want this ?"))==xwYesMessage)
      {
         fl_set_app_mainform(xwGetMainGui());
         xwFreePicture(-1);
         xwInitFrames();
         xwInitPicture();

         xwLockPage();
         xwAssistent(0);
         xwUnLockPage();

         /* Picture Editor */
         xwDeActivateFrameEd();
         xwDrawPage(pgRecalc);
      }
      else
      {
         fl_set_app_mainform(xwGetMainGui());
      }
   }
   else
   {
      xwFreePicture(-1);
      xwInitFrames();
      xwInitPicture();

      xwLockPage();
      xwAssistent(0);
      xwUnLockPage();

      /* Picture Editor */
      xwDeActivateFrameEd();
      xwDrawPage(pgRecalc);
   }

   xwActivateMain();

   xwActualizeFP();

   return;
}

/* Trash */
void xwTrash(void)
{
   if ( (xwGetFrameCount()>-1) || (xwGetPictureCount()>-1) )
   {
      xwDeactivateMain();
      xwLockHandler();
      if (xwAskMessage(NULL,xwGR(1,"msg.Trash","Delete all frames and pictures ?"))==xwYesMessage)
      {
         xwFreePicture(-1);
         xwInitFrames();
         xwInitPicture();
         xwDrawPage(pgRecalc);
      }
      xwUnLockHandler();
      fl_set_app_mainform(xwGetMainGui());

      xwActualizeFP();

      xwActivateMain();
   }
   return;
}

/* HELP */


/* EXIT */
void xwExit(void)
{
   xwDeactivateMain();
   if (xwAskMessage(NULL,xwGR(1,"msg.Exit","Exit xwGUI in progress...\n\nAre you sure ?"))==xwYesMessage)
   {
      xwIsMainGui(1);
   }

   fl_set_app_mainform(xwGetMainGui());

   xwActivateMain();

   return;
}

/* Landscape switch */
void xwAcLandscape(void)
{
   xwDeactivateMain();

   /* Landscape -> Portrait */
   if (xwIsPaperLS()==xwLandscape)
   {
      xwSetPaperLS(xwPortrait);
      xwSetPictureRotation(0);
      xwSetXOffset(0);
      xwSetYOffset(0);
      xwSetXXOffset(0);
      xwSetYYOffset(0);
      xwDrawPage(pgRecalc);
   }
   /* Portrait -> Landscape */
   else if (xwIsPaperLS()==xwPortrait)
   {
      xwSetPaperLS(xwLandscape);
      xwSetPictureRotation(-90);
      xwSetXOffset(0);
      xwSetYOffset(0);
      xwSetXXOffset(0);
      xwSetYYOffset(0);
      xwDrawPage(pgRecalc);
   }

   fl_set_scrollbar_value(mf.PaperSR,0);
   fl_set_scrollbar_value(mf.PaperSB,0);

   xwActivateMain();

   return;
}
/* Add Frame */
void xwAdd(void)
{
   xwAddFrame();

   xwDrawPage(pgRecalc);

   xwActualizeFP();

   return;
}

/* Editor */
void xwEdit(void)
{
   xwDeactivateMain();

   xwEditor();

   xwActivateMain();

   xwDrawPage(pgRecalc);

   return;
}

/* Frame/Picture Trashcan */
void xwFPTrash(void)
{
   xwFPFrameTrash();
   xwDrawPage(pgRecalc);

   xwActualizeFP();

   return;
}

/* Move To Back */
void xwBack(void)
{
   xwMoveFrameToBack();
   xwDrawPage(pgRecalc);

   xwActualizeFP();

   return;
}

/* Move To Front */
void xwFront(void)
{
   xwMoveFrameToFront();
   xwDrawPage(pgRecalc);

   xwActualizeFP();

   return;
}

/* Rotate */
void xwRotate(int direction)
{
   xwRotatePicInFrame(direction);
   xwDrawPage(pgRecalc);

   return;
}

/* Move Frame/Picture */
void xwMove(int xpos,int ypos)
{
   xwFrameMove(xpos,ypos);
   xwDrawPage(pgRecalc);

   return;
}

/* Preview */
void xwPreview(void)
{
   float          dpii;
   float          zoom;

   xwDeactivateMain();

   /* Pictures available for print */
   if ( (xwGetPictureCount()>-1) || (xwGetFrameCount()>-1) )
   {
      zoom=xwGetZoom();
      xwSetZoom(1);
      /* Set Pixel Size */
      xwSetPixelSize(xwGetPaperXSize(),xwGetPaperYSize());

      /* DPI Calculation */
      dpii=( ( (float) xwGetPaperXB() * (float) xwGetPixelSize() ) / ((float) xwGetPaperXB()/25.4) )*xwMM();

      if (xwIsPaperLS()==xwPortrait)
      {
         xwGFXLandscape(0);
      }
      else
      {
         xwGFXLandscape(1);
      }

      if ((int) xwRender(dpii,0,xwGetPaperXB(),xwGetPaperYB())!=-1)
      {
         xwSetZoom(zoom);
         /* Set Pixel Size */
         xwSetPixelSize(xwGetPaperXSize(),xwGetPaperYSize());
         xwGFXPreview(dpii);
      }
      else
      {
         xwSetZoom(zoom);
         /* Set Pixel Size */
         xwSetPixelSize(xwGetPaperXSize(),xwGetPaperYSize());
      }
   }
   else
   {
      xwDisplayMessage(NULL,xwGR(1,"err.CantDisplayError","No Pictures or Frame available for display !!!"));
   }

   xwActivateMain();

   return;
}

/* Print */
void xwPrint(void)
{
   int            dpiPixel;
   int            dpiMM;
   int            dpiCount;
   int            dpiAct;

   int            dpis;
   int            dpi;
   int            sheets;

   int            prtfile;

   char           astr[256];
   char           bstr[256];
   float          xwDPI;

   xwDPI=(25.4*(float) xwMM());

   xwDeactivateMain();

   xwLockMainGui();
   xwLockPage();

   /* Check Print Driver */
   if ( (strlen(xwGetPrefDriver())>0) && (strlen(xwGetPrefPrinter())>0) && (strlen(xwGetPrefSetting())>0) )
   {
      /* Pictures available for print */
      if ( (xwGetPictureCount()>-1) || (xwGetFrameCount()>-1) )
      {
         /* get max. DPI */
         dpi=180;
         dpiCount=xwGetPictureCount();
         for (dpiAct=0 ; dpiAct<=dpiCount; dpiAct++)
         {
            dpiPixel=xwGetPictureRW(dpiAct);
            dpiMM=xwGetPictureW(dpiAct);
            dpis=(int) ( (((float) dpiPixel/(float) dpiMM)*xwMM())*25.4);
            if (dpis>dpi) dpi=dpis;
         }
         if (dpi>360) dpi=360;

         /* Print in File */
         prtfile=0;

         /* count of prints */
         sheets=1;

         prf.form = NULL;
         prf.ready = 0;

         prf.form = fl_bgn_form(FL_UP_BOX, 400,105+30);
         fl_set_border_width(1);

         prf.printer = fl_add_box(FL_DOWN_BOX,5,5,390,50,"");
         strcpy(astr,xwGetPrefDriver());
         strcat(astr,"\n");
         strcat(astr,xwGetPrefPrinter());
         strcat(astr,"\n");
         strcat(astr,xwGetPrefSetting());
         fl_set_object_label(prf.printer,astr);

         if (xwGetPrefValue(xwgExpert)==0)
         {
            prf.sheets = fl_add_input(FL_INT_INPUT,200,5+55,50,20,xwGR(1,"prtf.Sheets","Count of Copys:"));
            fl_set_input_return(prf.sheets,FL_RETURN_CHANGED);
            sprintf(astr,"%i",sheets);
            fl_set_input(prf.sheets,astr);
#ifdef xwWhiteInput
            fl_set_object_color(prf.sheets,FL_WHITE,FL_WHITE);
#endif
         }
         else
         {
            prf.sheets = fl_add_input(FL_INT_INPUT,145,5+55,50,20,xwGR(1,"prtf.Sheets","Count of Copys:"));
            fl_set_input_return(prf.sheets,FL_RETURN_CHANGED);
            sprintf(astr,"%i",sheets);
            fl_set_input(prf.sheets,astr);
#ifdef xwWhiteInput
            fl_set_object_color(prf.sheets,FL_WHITE,FL_WHITE);
#endif

            prf.dpiout = fl_add_input(FL_INT_INPUT,345,5+55,50,20,xwGR(1,"prtf,DPI","DPI:"));
            fl_set_input_return(prf.dpiout,FL_RETURN_CHANGED);
            sprintf(astr,"%i",dpi);
            fl_set_input(prf.dpiout,astr);
#ifdef xwWhiteInput
            fl_set_object_color(prf.dpiout,FL_WHITE,FL_WHITE);
#endif
         }

         prf.tofile = fl_add_lightbutton(FL_PUSH_BUTTON,5,55+30,100,20,xwGR(1,"prtf.ToFile","Print in File"));
         fl_set_button(prf.tofile,0);

         prf.file = fl_add_box(FL_DOWN_BOX,105,55+30,290,20,xwGetFile("","prtout.png"));

         prf.print = fl_add_button(FL_NORMAL_BUTTON,5,80+30,190,20,xwGR(1,"prtf.OK","Print"));
         prf.cancel = fl_add_button(FL_NORMAL_BUTTON,205,80+30,190,20,xwGR(1,"prtf.Cancel","Cancel"));

         fl_end_form();

         if (xwGetPrefValue(xwgWinPos)==0)
         {
            fl_prepare_form_window(prf.form,FL_PLACE_MOUSE,FL_TRANSIENT,xwGR(1,"prtf.Print","Printing..."));
         }
         else
         {
            fl_prepare_form_window(prf.form,FL_PLACE_CENTER,FL_TRANSIENT,xwGR(1,"prtf.Print","Printing..."));
         }

         fl_set_form_dblbuffer(prf.form,0);
         fl_show_form_window(prf.form);

         fl_set_app_mainform(prf.form);

         /* Exit Handler */
         fl_set_form_atclose(prf.form,nclose_cb,(void *) "1");

         prf.ready=0;
         while (prf.ready==0)
         {
            prf.obj= fl_do_forms();


            /* Exit */
            if (prf.obj==prf.cancel) prf.ready=1;

            /* Sheet */
            else if (prf.obj==prf.sheets)
            {
               sheets=atoi(fl_get_input(prf.sheets));
               if (sheets<1)
               {
                  sheets=1;
                  sprintf(astr,"%i",sheets);
                  fl_set_input(prf.sheets,astr);
               }
               if (sheets>50)
               {
                  sheets=50;
                  sprintf(astr,"%i",sheets);
                  fl_set_input(prf.sheets,astr);
               }
            }

            /* DPI */
            else if (prf.obj==prf.dpiout)
            {
               dpi=atoi(fl_get_input(prf.dpiout));
            }

            /* Print In File */
            else if (prf.obj==prf.tofile)
            {
               prtfile=fl_get_button(prf.tofile);
            }

            /* Print */
            else if (prf.obj==prf.print) prf.ready=-1;
         }

         fl_hide_form(prf.form);
         fl_free_form(prf.form);
         prf.form=NULL;

         fl_set_app_mainform(xwGetMainGui());

         xwLockProgress();

         /* Print Now */
         if (prf.ready==-1)
         {
            if ((int) xwRender(dpi,0,xwGetPaperXB(),xwGetPaperYB())>-1)
            {

               /* Convert PPM->PNG */
               if (prtfile==1)
               {
                  /* Print Now - as PNG File */
                  strcpy(astr,xwGetFile("","prtout.ppm"));
                  strcpy(bstr,xwGetFile("","prtout.png"));
                  xwPrintOut(-1,astr,bstr);
               }
               else
               {
                  /* Print now !!! */
                  strcpy(astr,xwGetFile("","prtout.ppm"));
                  xwPrintOut(sheets,astr,"");
               }

               /*unlink(xwGetFile("","prtout.ppm"));*/
            }
            else
            {
               printf("xwGUI II:Render failure !!!\n");
            }
         }
         xwUnLockProgress();
         xwCloseProgress();
      }
      else
      {
         xwDisplayMessage(NULL,xwGR(1,"err.CantPrintError","No Pictures or Frame available for output !!!"));
      }
   }
   else
   {
      /* No Printer selected */
      xwDisplayMessage(NULL,xwGR(1,"err.NoPrinterSelected","Can't printing !!!\nYou must select an printer, please."));
   }

   xwUnLockPage();
   xwUnLockMainGui();

   xwActivateMain();

   return;
}
