/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "interpreter.h"

#include "base.h"
#include "default.h"
#include "driver_turboprint.h"
#include "error.h"
#include "font.h"
#include "frame.h"
#include "help.h"
#include "page.h"
#include "paper.h"
#include "picbrowser.h"
#include "picture.h"
#include "pref_printer.h"
#include "prtmanager.h"
#include "resource.h"
#include "variable.h"
#include "wysiwyg.h"

/*
   Level Function
       0 Normal StartUp Level
    1000 Filter Level

   Token
    1000


   Parameter:
   S=String I=Integer F=Float

*/
static struct ipCommand ipC[]=
{
/*  act.  new   token command              parameters */
   {    0, 1000, 1000,"PREF"              ,""               }, /* Preference Sequence */
   { 1000, 1000, 1001,"SIZE"              ,"II"             }, /* Window Size */
   { 1000, 1000, 1002,"VERSION"           ,"I"              }, /* xwGUI Version Information */
   { 1000, 1000, 1003,"LANGUAGE"          ,"S"              }, /* Language                   <-- */
   { 1000, 1000, 1004,"VIEWER"            ,"S"              }, /* Picture Viewer             <-- */
   { 1000, 1000, 1005,"CENTER"            ,""               }, /* Center Window's            <-- */
   { 1000, 1000, 1006,"GEOINFO"           ,""               }, /* Geometry Information       <-- */
   { 1000, 1000, 1007,"LAYER"             ,"III"            }, /* Layer Window Position */
   { 1000, 1000, 1008,"ICONLAYOUT"        ,""               }, /* GUI Design: Icon Based     <-- */
   { 1000,    0, 9999,"END"               ,""               }, /* End of Preference Sequence */

   { 1000, 1010, 1100,"PAPER"             ,""               }, /* PAPER Sequence */
   { 1010, 1010, 1101,"LANDSCAPE"         ,""               }, /* LANDSCAPE */
   { 1010, 1010, 1102,"BORDER"            ,"IIII"           }, /* BORDER Datas */
   { 1010, 1010, 1103,"PAPERID"           ,"I"              }, /* PAPERID */
   { 1010, 1010, 1104,"1/10MM"            ,""               }, /* 1/10 mm Mode */
   { 1010, 1000, 9999,"END"               ,""               }, /* End of PAPER Sequence */

   { 1000, 1020, 1200,"XWGUI"             ,""               }, /* xwGUI Settings */
   { 1020, 1020, 1201,"ICONLAYOUT"        ,""               }, /* GUI Design: Icon Based */
   { 1020, 1020, 1202,"LANGUAGE"          ,"S"              }, /* Language */
   { 1020, 1020, 1203,"CENTER"            ,""               }, /* Center Window's */
   { 1020, 1020, 1204,"GEOINFO"           ,""               }, /* Geometry Information */
   { 1020, 1020, 1205,"1/10MM"            ,""               }, /* 1/10 mm Mode */
   { 1020, 1020, 1206,"WYSIWYG"           ,""               }, /* WYSIWYG Mode */
   { 1020, 1020, 1207,"VIEWER"            ,"S"              }, /* Picture Viewer */
   { 1020, 1020, 1208,"HELPBROWSER"       ,"S"              }, /* Help Browser */
   { 1020, 1020, 1209,"PSEUDOWYSIWYG"     ,""               }, /* PSEUDO WYSIWYG Mode */
   { 1020, 1020, 1210,"IMAGEQUALITY"      ,"I"              }, /* Image Quality */
   { 1020, 1020, 1211,"INTERPOLATION"     ,""               }, /* Scalling Quality with Interpolation */
   { 1020, 1020, 1212,"NOASSISTENT"       ,""               }, /* Disable Assistent on StartUp */
   { 1020, 1020, 1213,"FRAMESET"          ,"S"              }, /* Load FrameSet on StartUp */
   { 1020, 1020, 1214,"PICBROWSER"        ,"S"              }, /* Picture Browser Path */
   { 1020, 1020, 1215,"FONTAFM"           ,"S"              }, /* AFM Directory */
   { 1020, 1020, 1216,"FONTTYPE1"         ,"S"              }, /* Type1 Directory */
   { 1020, 1020, 1217,"FONTENCDIR"        ,"S"              }, /* Font Enc Directory */
   { 1020, 1020, 1218,"FONTENCFILE"       ,"S"              }, /* Font Enc File */

   { 1020, 1000, 9999,"END"               ,""               }, /* End of xwGUI Settings */

   { 1000, 1030, 1300,"PRINTER"           ,""               }, /* Printer Settings */
   { 1030, 1030, 1301,"DRIVERFILE"        ,"S"              }, /* Driver File */
   { 1030, 1030, 1302,"PRINTER"           ,"S"              }, /* Selected Printer */
   { 1030, 1030, 1303,"SETTINGFILE"       ,"S"              }, /* Setting File */
   { 1030, 1000, 9999,"END"               ,""               }, /* End of Printer Settings */

   {    0, 1040, 1400,"SETTING"           ,""               }, /* Printer Setting */
   { 1040, 1040, 1401,"DEF"               ,"SS"             }, /* Variable Definition */
   { 1040,    0, 9999,"END"               ,""               }, /* End of Printer Setting */

/* PRINTER DEFINITION FILE */

   {    0, 2000, 2000,"PRINTERDEF"        ,""               }, /* Printer Definition File */
   { 2000,    0, 9999,"END"               ,""               }, /* End of Printer Definition File */

   { 2000, 2010, 2001,"WIDGET"            ,"S"              }, /* New Widget + ID */
   { 2010, 2010, 2002,"PRI"               ,"I"              }, /* Widget Priority */
   { 2010, 2010, 2003,"TYPE"              ,"S"              }, /* Widget Type */
   { 2010, 2010, 2004,"NAME"              ,"S"              }, /* Widget Name */
   { 2010, 2010, 2005,"SCRIPT"            ,"S"              }, /* Widget Option */
   { 2010, 2010, 2006,"DEFAULT"           ,"S"              }, /* Widget Default */
   { 2010, 2010, 2007,"MIN"               ,"F"              }, /* Widget MinVal */
   { 2010, 2010, 2008,"MAX"               ,"F"              }, /* Widget MaxVal */
   { 2010, 2010, 2009,"STEP"              ,"F"              }, /* Widget Step */
   { 2010, 2010, 2010,"ENTRYS"            ,"S"              }, /* Widget Entrys */
   { 2010, 2010, 2011,"WENTRYS"           ,"S"              }, /* Widget WEntrys */
   { 2010, 2010, 2012,"VAR"               ,"S"              }, /* Widget Variable */
   { 2010, 2010, 2013,"TAB"               ,""               }, /* Widget Tabulator */
   { 2010, 2000, 9999,"END"               ,""               }, /* End of New Widget */

   { 2000, 2020, 2101,"PRINTER"           ,"S"              }, /* New Printer + Name */
   { 2020, 2020, 2102,"SCRIPT"            ,"S"              }, /* Printer Script */
   { 2020, 2020, 2103,"GUI"               ,"IS"             }, /* Printer GUI */
   { 2020, 2020, 2104,"WIDGET"            ,"IS"             }, /* Printer Widget */
   { 2020, 2020, 2105,"TAB"               ,"I"              }, /* GUI Tab */
   { 2020, 2020, 2106,"DEFAULT"           ,"SS"             }, /* Set New Defaults */
   { 2020, 2020, 2107,"CLONE"             ,"S"              }, /* Clone Printer */
   { 2020, 2020, 2108,"HIDE"              ,""               }, /* Hide Printer */
   { 2020, 2000, 9999,"END"               ,""               }, /* End of New Printer */

/* FRAMESET DEFINITION FILE */
   {    0, 3000, 3000,"FRAMESET"          ,""               }, /* FrameSet Definition File */
   { 3000,    0, 9999,"END"               ,""               }, /* End of FrameSet Definition File */

   { 3000, 3010, 3100,"PAPER"             ,""               }, /* PAPER Sequence */
   { 3010, 3010, 3101,"LANDSCAPE"         ,""               }, /* LANDSCAPE */
   { 3010, 3010, 3102,"BORDER"            ,"IIII"           }, /* BORDER Datas */
   { 3010, 3010, 3103,"PAPERID"           ,"I"              }, /* PAPERID */
   { 3010, 3010, 3104,"1/10MM"            ,""               }, /* 1/10 mm Mode */
   { 3010, 3000, 9999,"END"               ,""               }, /* End of PAPER Sequence */

   { 3000, 3020, 3201,"FRAME"             ,""               }, /* New Frame */
   { 3020, 3020, 3202,"GEO"               ,"IIII"           }, /* Frame Geometry */
   { 3020, 3020, 3203,"BORDER"            ,"F"              }, /* BorderSize */
   { 3020, 3020, 3204,"FRAME"             ,"F"              }, /* FrameSize */
   { 3020, 3020, 3205,"COLOR"             ,"I"              }, /* Frame & Border Color */
   { 3020, 3020, 3206,"POSITION"          ,"I"              }, /* Position into Frame */
   { 3020, 3020, 3207,"FONTNAME"          ,"S"              }, /* Fontname */
   { 3020, 3020, 3208,"FONTSIZE"          ,"I"              }, /* Fontsize */
   { 3020, 3020, 3209,"FONTLINE"          ,"I"              }, /* Fontline ( Underline ) */
   { 3020, 3020, 3210,"FONTDIRECTION"     ,"I"              }, /* Fontdirection */
   { 3020, 3020, 3211,"FONTMODE"          ,"I"              }, /* Fontmode */
   { 3020, 3020, 3212,"FONTCOLOR"         ,"I"              }, /* Fontcolor */
   { 3020, 3020, 3213,"FONTTEXT"          ,"I"              }, /* Fonttext */
   { 3020, 3020, 3214,"TRANSPARENCY"      ,""               }, /* Transparency */
   { 3020, 3020, 3215,"BORDERTYE"         ,"I"              }, /* Frame BorderType */
   { 3020, 3000, 9999,"END"               ,""               }, /* End of New Frame */

/* PAGE DEFINITION FILE */
   {    0, 4000, 4000,"PAGE"              ,""               }, /* Page Definition File */
   { 4000,    0, 4001,"END"               ,""               }, /* End of Page Definition File */

   { 4000, 4010, 4100,"PAPER"             ,""               }, /* PAPER Sequence */
   { 4010, 4010, 4101,"LANDSCAPE"         ,""               }, /*LANDSCAPE */
   { 4010, 4010, 4102,"BORDER"            ,"IIII"           }, /* BORDER Datas */
   { 4010, 4010, 4103,"PAPERID"           ,"I"              }, /* PAPERID */
   { 4010, 4010, 4104,"1/10MM"            ,""               }, /* 1/10 mm Mode */
   { 4010, 4000, 9999,"END"               ,""               }, /* End of PAPER Sequence */

   { 4000, 4020, 4200,"PICTURE"           ,""               }, /* PICTURE Sequence */
   { 4020, 4020, 4201,"FILE"              ,"S"              }, /* Picture File */
   { 4020, 4020, 4202,"GEO"               ,"IIII"           }, /* Picture Geometry */
   { 4020, 4020, 4203,"ROTATION"          ,"I"              }, /* Picture rotation */
   { 4020, 4020, 4204,"NEGATIVETRANS"     ,""               }, /* Negative Transparency */
   { 4020, 4020, 4205,"BORDERS"           ,"FFII"           }, /* Border, Color and Transparency */
   { 4020, 4020, 4206,"TRANSPARENCY"      ,""               }, /* Border Transparency */
   { 4020, 4000, 9999,"END"               ,""               }, /* End of PICTURE Sequence */

   { 4000, 4030, 4300,"FRAME"             ,""               }, /* FRAME Sequence */
   { 4030, 4030, 4301,"GEO"               ,"IIII"           }, /* Frame Geometry */
   { 4030, 4030, 4302,"BORDER"            ,"F"              }, /* BorderSize */
   { 4030, 4030, 4303,"FRAME"             ,"F"              }, /* FrameSize */
   { 4030, 4030, 4304,"COLOR"             ,"I"              }, /* Frame & Border Color */
   { 4030, 4030, 4305,"POSITION"          ,"I"              }, /* Position into Frame */
   { 4030, 4030, 4306,"FONTNAME"          ,"S"              }, /* Fontname */
   { 4030, 4030, 4307,"FONTSIZE"          ,"I"              }, /* Fontsize */
   { 4030, 4030, 4308,"FONTLINE"          ,"I"              }, /* Fontline ( Underline ) */
   { 4030, 4030, 4309,"FONTDIRECTION"     ,"I"              }, /* Fontdirection */
   { 4030, 4030, 4310,"FONTMODE"          ,"I"              }, /* Fontmode */
   { 4030, 4030, 4311,"FONTCOLOR"         ,"I"              }, /* Fontcolor */
   { 4030, 4030, 4312,"FONTTEXT"          ,"I"              }, /* Fonttext */
   { 4030, 4030, 4313,"TRANSPARENCY"      ,""               }, /* Border Transparency */
   { 4030, 4030, 4314,"BORDERTYPE"        ,"I"              }, /* Frame Border Type */
   { 4030, 4000, 9999,"END"               ,""               }, /* End of PAPER Sequence */

   { 4030, 4040, 4400,"PICTURE"           ,""               }, /* PICTURE Sequence */
   { 4040, 4040, 4401,"FILE"              ,"S"              }, /* Picture File */
   { 4040, 4040, 4402,"GEO"               ,"IIII"           }, /* Picture Geometry */
   { 4040, 4040, 4403,"ROTATION"          ,"I"              }, /* Picture rotation */
   { 4040, 4040, 4404,"NEGATIVETRANS"     ,""               }, /* Negative Transparency */
   { 4040, 4040, 4405,"BORDERS"           ,"FFII"           }, /* Border, Color and Transparency */
   { 4040, 4040, 4406,"TRANSPARENCY"      ,""               }, /* Border Transparency */
   { 4040, 4030, 9999,"END"               ,""               }, /* End of PICTURE Sequence */

/* FONTFILE DEFINITION FILE */
   {    0, 5000, 5000,"FONTFILE"          ,""               }, /* FontFile Definition File */
   { 5000,    0, 5001,"END"               ,""               }, /* End of FontFile Definition File */

   { 5000, 5000, 5002,"FONT"              ,"SI"             }, /* FontList Entry */

   {   -1,   -1,   -1,""                  ,""               }
};

static struct ipDataInfo ipDI;


FILE              *iFH;                   /* Interpreter File Handle */
int               iLineNr;                /* acctual line to interpret */
int               iBug;                   /* Error Code 0=None >0 Error */

char              errMessageA[1024];      /* Error Message */
char              errMessageB[1024];
char              errA[1024];
char              errB[1024];

int               frmsMM;                 /* FrameSet Measure / 0 - mm / 1 - 1/10 mm */

int               picFlag;                /* Picture Sequence Flag */

/* Interpreter Offset Table for SpeedUp */
int               iCEntrys=-1;
int               iCOffset[1024];
int               iCIndex[1024];

/* Token Table */
void xwTable(int iCNr)
{

   /* Filter Commands */
   /* SIZE */
   if (ipC[iCNr].token == 1001)
   {
      xwSetPrefValue(xwgWidth,ipDI.iValue[0]);
      xwSetPrefValue(xwgHeight,ipDI.iValue[1]);
   }
   /* VERSION */
   else if (ipC[iCNr].token == 1002) xwSetPrefValue(xwgVersion,ipDI.iValue[0]);
   /* LANGUAGE */
   else if (ipC[iCNr].token == 1003) xwSetLang(ipDI.String[0]);
   /* VIEWER */
   else if (ipC[iCNr].token == 1004) xwSetPrefName(xwgPicViewer,ipDI.String[0]);
   /* CENTER */
   else if (ipC[iCNr].token == 1005) xwSetPrefValue(xwgWinPos,1);
   /* Geometry Information */
   else if (ipC[iCNr].token == 1006) xwSetPrefValue(xwgOneLiner,1);
   /* Layer Window Position */
   else if (ipC[iCNr].token == 1007)
   {
      xwSetPrefValue(xwgLayerXPos,ipDI.iValue[0]);
      xwSetPrefValue(xwgLayerYPos,ipDI.iValue[1]);
      xwSetPrefValue(xwgLayer,ipDI.iValue[2]);
   }
   /* GUI Layout: Icon Based */
   else if (ipC[iCNr].token == 1008) xwSetPrefValue(xwgGUI,1);

   /* ##### PAPER SETTINGS ######################################################################### */

   /* Landscape */
   else if (ipC[iCNr].token == 1101) xwSetPaperLS(xwLandscape);
   /* BORDER */
   else if (ipC[iCNr].token == 1102)
   {
      xwSetPaperLB(ipDI.iValue[0]);
      xwSetPaperRB(ipDI.iValue[1]);
      xwSetPaperTB(ipDI.iValue[2]);
      xwSetPaperBB(ipDI.iValue[3]);
   }
   /* PAPERID */
   else if (ipC[iCNr].token == 1103)
   {
      xwSetID(ipDI.iValue[0]);
      xwSetPaperSize(ipDI.iValue[0],0,0);
      if (strcmp(xwGetVarDef("TURBOPRINT"),"TURBOPRINT") == 0) tpSetCorrectPaperSize(xwGetPaperXSize(),xwGetPaperYSize());
   }
   /* 1/10MM */
   else if (ipC[iCNr].token == 1104) xwSetPrefValue(xwgMM,10);

   /* ##### XWGUI SETTINGS ######################################################################### */

   /* GUI Layout: Icon Based */
   else if ((ipC[iCNr].token == 1201) && (xwGetPrefValue(xwgDepth)>6)) xwSetPrefValue(xwgGUI,1);
   /* LANGUAGE */
   else if (ipC[iCNr].token == 1202) xwSetLang(ipDI.String[0]);
   /* CENTER */
   else if (ipC[iCNr].token == 1203) xwSetPrefValue(xwgWinPos,1);
   /* Geometry Information */
   else if (ipC[iCNr].token == 1204) xwSetPrefValue(xwgOneLiner,1);
   /* 1/10MM */
   else if (ipC[iCNr].token == 1205) xwSetPrefValue(xwgMM,10);
   /* WYSIWYG */
   else if (ipC[iCNr].token == 1206) xwSetPrefValue(xwgWYSIWYG,wmHWYSIWYG);
   /* VIEWER */
   else if (ipC[iCNr].token == 1207) xwSetPrefName(xwgPicViewer,ipDI.String[0]);
   /* Help Browser */
   else if (ipC[iCNr].token == 1208) xwSetHelpBrowser(ipDI.String[0]);
   /* Pseudo WYSIWYG */
   else if (ipC[iCNr].token == 1209) xwSetPrefValue(xwgWYSIWYG,wmLWYSIWYG);
   /* Image Quality */
   else if (ipC[iCNr].token == 1210) xwSetPrefValue(xwgImgQuality,ipDI.iValue[0]);
   /* Scalling Quality */
   else if (ipC[iCNr].token == 1211) xwSetPrefValue(xwgScalling,1);
   /* disable Assistent on StartUp */
   else if (ipC[iCNr].token == 1212) xwSetPrefValue(xwgAssistent,1);
   /* load FrameSet on StartUp */
   else if (ipC[iCNr].token == 1213) xwSetPrefName(xwgFrameSet,ipDI.String[0]);
   /* picture browser path */
   else if (ipC[iCNr].token == 1214)
   {
      xwSetPrefName(xwgBrowserPath,ipDI.String[0]);
      xwSetPicBrowserPath(ipDI.String[0]);
   }
   /* AFM Directory */
   else if (ipC[iCNr].token == 1215)
   {
      xwSetPrefName(xwgAFM,ipDI.String[0]);
   }
   /* Type1 Directory */
   else if (ipC[iCNr].token == 1216)
   {
      xwSetPrefName(xwgType1,ipDI.String[0]);
   }
   /* Encoding Directory */
   else if (ipC[iCNr].token == 1217)
   {
      xwSetPrefName(xwgEnc,ipDI.String[0]);
   }
   /* Encoding File */
   else if (ipC[iCNr].token == 1218)
   {
      xwSetPrefName(xwgEncFile,ipDI.String[0]);
   }

   /* ##### PRINTER SETTINGS ####################################################################### */

   /* DRIVERFILE */
   else if (ipC[iCNr].token == 1301) xwSetPrefDriver(ipDI.String[0]);

   /* PRINTER - selected Printer */
   else if (ipC[iCNr].token == 1302) xwSetPrefPrinter(ipDI.String[0]);

   /* SETTINGFILE */
   else if (ipC[iCNr].token == 1303) xwSetPrefSetting(ipDI.String[0]);

   /* ########## PRINTER SETTINGS ################################################################## */

   /* DEF */
   else if (ipC[iCNr].token == 1401) xwSetVarDef(ipDI.String[0],ipDI.String[1]);

   /* ########## WIDGET DEFINITION ################################################################# */

   /* WIDGET */
   else if (ipC[iCNr].token == 2001) xwAddWidget(xwAddWCount(),ipDI.String[0]);
   /* PRI */
   else if (ipC[iCNr].token == 2002) xwSetWidgetPri(xwGetWidgetCount(),ipDI.iValue[0]);
   /* TYPE */
   else if (ipC[iCNr].token == 2003) xwSetWidgetType(xwGetWidgetCount(),ipDI.String[0]);
   /* NAME */
   else if (ipC[iCNr].token == 2004) xwSetWidgetName(xwGetWidgetCount(),ipDI.String[0]);
   /* OPTION */
   else if (ipC[iCNr].token == 2005) xwSetWidgetOption(xwGetWidgetCount(),ipDI.String[0]);
   /* DEFAULT */
   else if (ipC[iCNr].token == 2006) xwSetWidgetDefault(xwGetWidgetCount(),ipDI.String[0]);
   /* MIN */
   else if (ipC[iCNr].token == 2007) xwSetWidgetMinVal(xwGetWidgetCount(),ipDI.fValue[0]);
   /* MAX */
   else if (ipC[iCNr].token == 2008) xwSetWidgetMaxVal(xwGetWidgetCount(),ipDI.fValue[0]);
   /* STEP */
   else if (ipC[iCNr].token == 2009) xwSetWidgetStep(xwGetWidgetCount(),ipDI.fValue[0]);
   /* ENTRYS */
   else if (ipC[iCNr].token == 2010) xwSetWidgetEntrys(xwGetWidgetCount(),ipDI.String[0]);
   /* WENTRYS */
   else if (ipC[iCNr].token == 2011) xwSetWidgetWEntrys(xwGetWidgetCount(),ipDI.String[0]);
   /* VAR */
   else if (ipC[iCNr].token == 2012) xwSetWidgetVariable(xwGetWidgetCount(),ipDI.String[0]);
   /* TAB */
   else if (ipC[iCNr].token == 2013) xwSetWidgetTab(xwGetWidgetCount(),1);

   /* ########## PRINTER DEFINITION ################################################################ */

   /* PRINTER */
   else if (ipC[iCNr].token == 2101) xwAddPrinter(ipDI.String[0]);
   /* SCRIPT */
   else if (ipC[iCNr].token == 2102) xwSetPrinterScript(ipDI.String[0]);
   /* GUI */
   else if (ipC[iCNr].token == 2103) xwSetPrinterGUIName(ipDI.iValue[0],ipDI.String[1]);
   /* WIDGET */
   else if (ipC[iCNr].token == 2104) xwAddPrinterWidget(ipDI.iValue[0],ipDI.String[1]);
   /* TAB for GUI */
   else if (ipC[iCNr].token == 2105) xwSetPrinterGUITab(ipDI.iValue[0],50);
   /* DEFAULT */
   else if (ipC[iCNr].token == 2106) xwSetPrinterWidgetDefault(ipDI.String[0],ipDI.String[1]);
   /* CLONE */
   else if (ipC[iCNr].token == 2107) xwClonePrinter(ipDI.String[0]);
   /* HIDE */
   else if (ipC[iCNr].token == 2108) xwSetHidePrinter();

   /* ########## FRAMESET DEFINITION ############################################################### */

   /* FrameSets Initialize */
   else if (ipC[iCNr].token == 3100)
   {
      frmsMM=0;   /* 1 mm */
      xwSetPaperLS(xwPortrait);
   }
   /* Landscape */
   else if (ipC[iCNr].token == 3101) xwSetPaperLS(xwLandscape);
   /* BORDER */
   else if (ipC[iCNr].token == 3102)
   {
      xwSetPaperLB(xwMC(ipDI.iValue[0]));
      xwSetPaperRB(xwMC(ipDI.iValue[1]));
      xwSetPaperTB(xwMC(ipDI.iValue[2]));
      xwSetPaperBB(xwMC(ipDI.iValue[3]));
   }
   /* PAPERID */
   else if (ipC[iCNr].token == 3103)
   {
      xwSetID(ipDI.iValue[0]);
      xwSetPaperSize(ipDI.iValue[0],0,0);
      if (strcmp(xwGetVarDef("TURBOPRINT"),"TURBOPRINT") == 0) tpSetCorrectPaperSize(xwGetPaperXSize(),xwGetPaperYSize());
   }
   /* 1/10MM */
   else if (ipC[iCNr].token == 3104)
   {
      frmsMM=1;   /* 1/10 mm */
   }

   /* FRAME */
   else if (ipC[iCNr].token == 3201) xwAddFrame();
   /* GEO */
   else if (ipC[iCNr].token == 3202) xwSetFrameGeo(xwGetFrameCount(),xwMC(ipDI.iValue[0]),xwMC(ipDI.iValue[1]),xwMC(ipDI.iValue[2]),xwMC(ipDI.iValue[3]));
   /* BORDER */
   else if (ipC[iCNr].token == 3203) xwSetFrameB(xwGetFrameCount(),(double) ipDI.fValue[0]);
   /* FRAME */
   else if (ipC[iCNr].token == 3204) xwSetFrameF(xwGetFrameCount(),(double) ipDI.fValue[0]);
   /* COLOR */
   else if (ipC[iCNr].token == 3205) xwSetFrameC(xwGetFrameCount(),ipDI.iValue[0]);
   /* POSITION 3203 */
   else if (ipC[iCNr].token == 3206) xwSetPicPosition(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTNAME */
   else if (ipC[iCNr].token == 3207) xwSetFontName(xwGetFrameCount(),ipDI.String[0]);
   /* FONTSIZE */
   else if (ipC[iCNr].token == 3208) xwSetFontSize(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTLINE */
   else if (ipC[iCNr].token == 3209) xwSetFontLine(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTDIRECTION */
   else if (ipC[iCNr].token == 3210) xwSetFontDirection(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTMODE */
   else if (ipC[iCNr].token == 3211) xwSetFontMode(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTCOLOR */
   else if (ipC[iCNr].token == 3212) xwSetFontColor(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTTEXT */
   else if (ipC[iCNr].token == 3213) xwSetRawFontText(xwGetFrameCount(),ipDI.iValue[0],iFH);
   /* TRANSPARENCY */
   else if (ipC[iCNr].token == 3214) xwSetFrameBC(xwGetFrameCount(),1);
   /* BORDERTYPE */
   else if (ipC[iCNr].token == 3215) xwSetFrameBT(xwGetFrameCount(),ipDI.iValue[0]);
   
   /* ########## PAGE DEFINITION ################################################################### */

   /* Page Initialize */
   else if (ipC[iCNr].token == 4001)
   {
      xwDrawPage(pgRecalc);
   }

   /* Page Initialize */
   else if (ipC[iCNr].token == 4100)
   {
      frmsMM=0;   /* 1 mm */
      xwSetPaperLS(xwPortrait);
   }
   /* Landscape */
   else if (ipC[iCNr].token == 4101) xwSetPaperLS(xwLandscape);
   /* BORDER */
   else if (ipC[iCNr].token == 4102)
   {
      xwSetPaperLB(xwMC(ipDI.iValue[0]));
      xwSetPaperRB(xwMC(ipDI.iValue[1]));
      xwSetPaperTB(xwMC(ipDI.iValue[2]));
      xwSetPaperBB(xwMC(ipDI.iValue[3]));
   }
   /* PAPERID */
   else if (ipC[iCNr].token == 4103)
   {
      xwSetID(ipDI.iValue[0]);
      xwSetPaperSize(ipDI.iValue[0],0,0);
      if (strcmp(xwGetVarDef("TURBOPRINT"),"TURBOPRINT") == 0) tpSetCorrectPaperSize(xwGetPaperXSize(),xwGetPaperYSize());
   }
   /* 1/10MM */
   else if (ipC[iCNr].token == 4104)
   {
      frmsMM=1;   /* 1/10 mm */
   }

   /* FILE */
   else if (ipC[iCNr].token == 4201)
   {
      picFlag=xwPLoad(ipDI.String[0],0,0,100,100);
      if (picFlag==-1)
      {
         xwWriteError("<");
         xwWriteError(ipDI.String[0]);
         xwWriteError(">\nPicture file not found !!!\n\n");
      }
   }
   /* GEO */
   else if ( (ipC[iCNr].token == 4202) && (picFlag>-1) ) xwSetPictureGeo(xwGetPictureCount(),ipDI.iValue[0],ipDI.iValue[1],ipDI.iValue[2],ipDI.iValue[3]);
   /* ROTATION */
   else if ( (ipC[iCNr].token == 4203) && (picFlag>-1) ) xwSetPictureR(xwGetPictureCount(),ipDI.iValue[0]);
      /* NEGATIVETRANS */
   else if ( (ipC[iCNr].token == 4204) && (picFlag>-1) ) xwSetPictureTM(xwGetPictureCount(),1);
   /* BORDERS */
   else if ( (ipC[iCNr].token == 4205) && (picFlag>-1) )
   {
      xwSetPictureB(xwGetPictureCount(),ipDI.fValue[0]);
      xwSetPictureF(xwGetPictureCount(),ipDI.fValue[1]);
      xwSetPictureC(xwGetPictureCount(),ipDI.iValue[2]);
      xwSetPictureT(xwGetPictureCount(),ipDI.iValue[3]);
   }
   /* TRANSPARENCY */
   else if (ipC[iCNr].token == 4206) xwSetPictureBC(xwGetPictureCount(),1);

   /* FRAME */
   else if (ipC[iCNr].token == 4300) xwAddFrame();
   /* GEO */
   else if (ipC[iCNr].token == 4301) xwSetFrameGeo(xwGetFrameCount(),xwMC(ipDI.iValue[0]),xwMC(ipDI.iValue[1]),xwMC(ipDI.iValue[2]),xwMC(ipDI.iValue[3]));
   /* BORDER */
   else if (ipC[iCNr].token == 4302) xwSetFrameB(xwGetFrameCount(),(double) ipDI.fValue[0]);
   /* FRAME */
   else if (ipC[iCNr].token == 4303) xwSetFrameF(xwGetFrameCount(),(double) ipDI.fValue[0]);
   /* COLOR */
   else if (ipC[iCNr].token == 4304) xwSetFrameC(xwGetFrameCount(),ipDI.iValue[0]);
   /* POSITION */
   else if (ipC[iCNr].token == 4305) xwSetPicPosition(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTNAME */
   else if (ipC[iCNr].token == 4306) xwSetFontName(xwGetFrameCount(),ipDI.String[0]);
   /* FONTSIZE */
   else if (ipC[iCNr].token == 4307) xwSetFontSize(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTLINE */
   else if (ipC[iCNr].token == 4308) xwSetFontLine(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTDIRECTION */
   else if (ipC[iCNr].token == 4309) xwSetFontDirection(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTMODE */
   else if (ipC[iCNr].token == 4310) xwSetFontMode(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTCOLOR */
   else if (ipC[iCNr].token == 4311) xwSetFontColor(xwGetFrameCount(),ipDI.iValue[0]);
   /* FONTTEXT */
   else if (ipC[iCNr].token == 4312) xwSetRawFontText(xwGetFrameCount(),ipDI.iValue[0],iFH);
   /* TRANSPARENCY */
   else if (ipC[iCNr].token == 4313) xwSetFrameBC(xwGetFrameCount(),1);
   /* BORDERTYPE */
   else if (ipC[iCNr].token == 4314) xwSetFrameBT(xwGetFrameCount(),ipDI.iValue[0]);
   
   /* FILE */
   else if (ipC[iCNr].token == 4401)
   {
      picFlag=xwFPLoad(ipDI.String[0]);
      if (picFlag==-1)
      {
         xwWriteError("<");
         xwWriteError(ipDI.String[0]);
         xwWriteError(">\nPicture file not found !!!\n\n");
      }
   }
   /* GEO */
   else if ( (ipC[iCNr].token == 4402) && (picFlag>-1) ) xwSetPictureGeo(xwGetPictureCount(),ipDI.iValue[0],ipDI.iValue[1],ipDI.iValue[2],ipDI.iValue[3]);
   /* ROTATION */
   else if ( (ipC[iCNr].token == 4403) && (picFlag>-1) ) xwSetPictureR(xwGetPictureCount(),ipDI.iValue[0]);
   /* NEGATIVETRANS */
   else if ( (ipC[iCNr].token == 4404) && (picFlag>-1) ) xwSetFramePTM(xwGetFrameCount(),1);
   /* BORDERS */
   else if ( (ipC[iCNr].token == 4405) && (picFlag>-1) )
   {
      xwSetFramePB(xwGetFrameCount(),ipDI.fValue[0]);
      xwSetFramePF(xwGetFrameCount(),ipDI.fValue[1]);
      xwSetFramePC(xwGetFrameCount(),ipDI.iValue[2]);
      xwSetFramePT(xwGetFrameCount(),ipDI.iValue[3]);
   }
   /* TRANSPARENCY */
   else if (ipC[iCNr].token == 4406) xwSetFramePBC(xwGetFrameCount(),1);

   /* ########## FONTFILE DEFINITION FILE ########################################################## */
   /* FONT */
   else if (ipC[iCNr].token == 5002) xwAddFontList(ipDI.String[0],ipDI.iValue[1]);


   return;
}

/* Init Interpreter - creates an Offset table for SpeedUp */
void xwInitInterpreter(void)
{
   int                  count;
   int                  az,bz;

   /* get count of entrys */   
   count=0;
   while (ipC[count].oldLevel!=-1)
   {
      count++;
   }
   
   /* Sort Array */
   qsort(ipC,count+1,sizeof(ipC[0]),xwInitSort);

   /* create table */
   az=-1;
   for (bz=0 ; bz<=count ; bz++)
   {
      if (ipC[bz].oldLevel!=az)
      {
         iCEntrys++;
         az=ipC[bz].oldLevel;
         iCIndex[iCEntrys]=az;
         iCOffset[iCEntrys]=bz;
      }
   }

   return;
}

int xwInitSort(const void *ipA,const void *ipB)
{
   struct ipCommand     *ipAA=ipA;
   struct ipCommand     *ipBB=ipB;

   if (ipAA->oldLevel<ipBB->oldLevel) return(-1);
   else if (ipAA->oldLevel>ipBB->oldLevel) return(1);
   else return(0);
}


/* Interpreter */
int xwInterpret(const char *file,FL_FORM *errForm)
{
   char                 iLine[1024];
   char                 iCommand[1024];
   char                 iParm[1024];
   int                  iCNr;
   int                  iCount;
   int                  iReady;

   /* initialize Interpreter */
   iLineNr=0;

   /* Error Code */
   iBug=0;

   /* Parameter Count */
   ipDI.Level=0;

   if ( (iFH=fopen(file,"rb")) != NULL )
   {
      /* Reset error log */
      xwResetError();

      while ( (fgets(iLine,1024,iFH) != NULL) && (iBug==0) )
      {
         /* increment LineNumber */
         iLineNr++;

/*         printf("Line #: %i %s\n",iLineNr,iLine);*/

         /* remove newline character */
         if (strlen(iLine)>0)
         {
            if (iLine[strlen(iLine)-1]==0x0a)
            {
               iLine[strlen(iLine)-1]=0x00;
            }
         }

         /* Get Command */
         strcpy(iCommand,StrGet(iLine));

         if (strlen(iCommand)>0)
         {

            /* upper Command */
            for (iCount=0 ; iCount<strlen(iCommand) ; iCount++)
            {
               iCommand[iCount]=toupper(iCommand[iCount]);
            }

            /* Search Command in Current Level */
            iCNr=0;
            
            /* SpeedUp */
            iCount=0;
            while ((iCIndex[iCount]!=ipDI.Level) || (iCount>iCEntrys))
            {
               iCount++;
            }
            if (iCount<=iCEntrys) iCNr=iCOffset[iCount];
            
            iReady=0;
            while (iReady==0)
            {
               /* found Command */
               if ( (ipC[iCNr].oldLevel==ipDI.Level) && (strcmp(iCommand,ipC[iCNr].command) == 0) )
               {
                  iReady=-1;
               }

               /* command not found */
               else if (ipC[iCNr].oldLevel==-1)
               {
                  iReady=-1;
                  iCNr=-1;
                  iBug=1;

                  strcpy(errMessageA,xwGR(1,"commandnotfound","Command not found !!! \n\nFile      : %s\nLineNumber: %s\nLine      : %s\n"));
                  strcpy(errA,IntStr(iLineNr));
                  sprintf(errMessageB,errMessageA,file,errA,iLine);
                  xwDisplayMessage(errForm,errMessageB);

               }

               /* next check */
               else
               {
                  iCNr++;
               }
            }

            if (iCNr>-1)
            {
               ipDI.Level=-1;

               /* Parse Command Parameters */
               if (strlen(ipC[iCNr].parm)>0)
               {
                  for (iCount=0 ; iCount<strlen(ipC[iCNr].parm) ; iCount++)
                  {
                     /* String */
                     if (ipC[iCNr].parm[iCount]=='S')
                     {
                        strcpy(iParm,StrGet(NULL));
                        ipDI.Level++;
                        strcpy(ipDI.String[ipDI.Level],iParm);
                     }
                     if (ipC[iCNr].parm[iCount]=='I')
                     {
                        strcpy(iParm,StrGet(NULL));
                        if (strlen(iParm)>0)
                        {
                           ipDI.Level++;
                           ipDI.iValue[ipDI.Level]=atoi(iParm);
                        }
                        else
                        {
                           strcpy(errMessageA,xwGR(1,"missingargument","Missing Argument !!! \n\nFile      : %s\nLineNumber: %s\nLine      : %s\n"));
                           strcpy(errA,IntStr(iLineNr));
                           sprintf(errMessageB,errMessageA,file,errA,iLine);
                           xwDisplayMessage(errForm,errMessageB);
                        }
                     }
                     if (ipC[iCNr].parm[iCount]=='F')
                     {
                        strcpy(iParm,StrGet(NULL));
                        if (strlen(iParm)>0)
                        {
                           ipDI.Level++;
                           ipDI.fValue[ipDI.Level]=atof(iParm);
                        }
                        else
                        {
                           strcpy(errMessageA,xwGR(1,"missingargument","Missing Argument !!! \n\nFile      : %s\nLineNumber: %s\nLine      : %s\n"));
                           strcpy(errA,IntStr(iLineNr));
                           sprintf(errMessageB,errMessageA,file,errA,iLine);
                           xwDisplayMessage(errForm,errMessageB);
                        }
                     }

                  }

               }

               /* Spring Table */
               xwTable(iCNr);

               /* New Level */
               ipDI.Level=ipC[iCNr].newLevel;
            }
         }
      }

      fclose(iFH);
   }
   else
   {
      strcpy(errMessageA,xwGR(1,"filenotfound","File not found !!!\n<%s>\n"));
      sprintf(errMessageB,errMessageA,file);
      xwDisplayMessage(errForm,errMessageB);
   }

   xwDisplayError();

   return(iBug);
}

/* Measure Correction */
int xwMC(int source)
{
   /*
   printf("MC: %i/%i - %i\n",frmsMM,xwMM(),source);
   */

   /* destination mm */
   if ((frmsMM==1) && (xwMM()==1))
   {
      return((int) ((source/10)+0.5));
   }
   /* destination 1/10 mm */
   else if ((frmsMM==0) && (xwMM()==10))
   {
      return(source*10);
   }

   return(source);
}