/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gfx.h"

#include "base.h"
#include "default.h"
#include "error.h"
#include "fileio.h"
#include "handler.h"
#include "limits.h"
#include "main_gui.h"
#include "paper.h"
#include "progress.h"
#include "resource.h"
#include "time.h"
#include "wysiwyg.h"
#include "x11.h"
#include "xpm.h"

#include "time.h"

#define TRENNZEICHEN    '/'

/* #define max(a,b) ((a>b)?a:b)
   #define min(a,b) ((a<b)?a:b) */

char LEERZEICHEN[]="[]";
char gfxTZ[]=" ";

char gfxTZZ[]=" []";

/* Paper Datas */
static unsigned char          *gfxPaper;        /* Paper Array */
static int                    gfxXPaper;        /* Width of Paper */
static int                    gfxYPaper;        /* Height of Paper */
static int                    gfxPaperSize;     /* Size of Paper */
static int                    gfxPaperLP;       /* Paper Orientation - Portrait/Landscape */

/* Shared Datas */
static struct sharedMemory    *gfxShared;       /* Shared Array */
static int                    gfxSharedID=-1;   /* Shgared Memory ID */
static int                    gfxXShared;       /* Width of Shared */
static int                    gfxYShared;       /* Height of Shared */
static int                    gfxSharedSize;    /* Size of Shared */

/* Temp Datas */
static unsigned char          *gfxTemp;         /* Temp Array */
static int                    gfxXTemp;         /* Width of Temp */
static int                    gfxYTemp;         /* Height of Temp */
static int                    gfxTempSize;      /* Size of Temp */

/* Picture Datas */
static unsigned char          *gfxPicture;      /* Picture Array */
static int                    gfxXPicture;      /* Width of Picture */
static int                    gfxYPicture;      /* Height of Picture */
static int                    gfxPictureSize;   /* Size of Picture */

/* Second Temp Datas */
static unsigned char          *gfxSTemp;        /* Temp Array */
static int                    gfxXSTemp;        /* Width of Temp */
static int                    gfxYSTemp;        /* Height of Temp */
static int                    gfxSTempSize;     /* Size of Temp */

/* Second Picture Datas */
static unsigned char          *gfxSPicture;     /* Picture Array */
static int                    gfxXSPicture;     /* Width of Picture */
static int                    gfxYSPicture;     /* Height of Picture */
static int                    gfxSPictureSize;  /* Size of Picture */

/* Temporary Saved Temp Datas - NOT REALLY GFX DATAS */
static unsigned char          *gfxTTemp;
static int                    gfxXTTemp;
static int                    gfxYTTemp;
static int                    gfxTTempSize;

/* Ignore Paper */
static int                    gfxIgnore=0;
static int                    gfxRed=0;
static int                    gfxGreen=0;
static int                    gfxBlue=0;

/* Mask Color */
static int                    gfxMRed=255;
static int                    gfxMGreen=255;
static int                    gfxMBlue=255;
static int                    gfxUseMask;

/* Masked Picture Area */
static int                    gfxMX;
static int                    gfxMY;
static int                    gfxMW;
static int                    gfxMH;
static int                    gfxMFlag=0;

/* actal RGB Values */
static int                    RV,GV,BV;

/* Transparenzy Value */
static int                    gfxTrans;
static float                  gfxTF;

/* Transparency Mode */
static int                    gfxTransMode=0;

/* RGB-Rendering Table */
static float                  gfxRGB[100];

/* Draw Border Mode */
static int                    gfxBRDType=0;

/* ################################################################### */

/* Initialize GFX */
void xwGFXInit(void) /*fold00*/
{
   int   az;

   gfxPaper=NULL;
   gfxTemp=NULL;
   gfxPicture=NULL;
   gfxShared=NULL;
   gfxSTemp=NULL;
   gfxSPicture=NULL;

   gfxPaper=0;

   gfxTrans=0;
   gfxTF=0;

   gfxIgnore=0;

   /* RGB Table Initialize */
   for (az=0 ; az<100 ; az++)
   {
      gfxRGB[az]=1-(az*0.01);
   }

  /*
   for (az=0 ; az<25 ; az++)
   {
      gfxRGB[az]=1-(az*0.03);
   }
   for (az=25 ; az<100 ; az++)
   {
      gfxRGB[az]=0.25-(0.003*(az-25));
   }
  */

   /*
   for (az=0 ; az<40 ; az++)
   {
      gfxRGB[az]=1-(az*0.015);
   }
   for (az=40 ; az<100 ; az++)
   {
      gfxRGB[az]=0.40-(0.007*(az-39));
   }
   */


   for (az=0 ; az<15 ; az++)
   {
      gfxRGB[az]=1-(az*0.0066);
   }
   for (az=15 ; az<85 ; az++)
   {
      gfxRGB[az]=0.90-((az-19)*0.0114);
   }
   for (az=85 ; az<100 ; az++)
   {
      gfxRGB[az]=0.10-((az-79)*0.0066);
   }


   return;
}

/* Set GFX Landscape */
void xwGFXLandscape(int mo) /*fold00*/
{
   gfxPaperLP=mo;
   return;
}

/* ################################################################### */
/* Paper Routines */

/* Get Paper Memory */
int xwGFXCreatePaper(int w,int h) /*fold00*/
{
   int         zeiger;

   if (gfxPaper!=NULL) xwGFXRemovePaper();

   gfxXPaper=w;
   gfxYPaper=h;
   gfxPaperSize=(gfxXPaper*gfxYPaper)*3;
   gfxPaper = malloc(gfxPaperSize+4);
   if (gfxPaper!=NULL)
   {
      for (zeiger=0 ; zeiger<=gfxPaperSize ; zeiger++)
      {
         gfxPaper[zeiger]=0xff;
      }

      return(xwTrue);
   }
   else
   {
      return(xwFalse);
   }
}

/* Remove Paper Memory */
void xwGFXRemovePaper(void) /*fold00*/
{
   if (gfxPaper!=NULL)
   {
      free(gfxPaper);
      gfxPaper=NULL;
      gfxXPaper=0;
      gfxYPaper=0;
      gfxPaperSize=0;
   }
   return;
}

/* Save Paper Memory */
int xwGFXSavePaper(char *dest) /*fold00*/
{
   FILE           *fh;
   int            size;

   if (gfxPaper!=NULL)
   {
      if ( (fh=fopen(dest,"w"))!=NULL )
      {
         /* Write Header */
         fputs("P6\n",fh);
         fputs("# written by xwGUI\n",fh);

         if (gfxPaperLP==0)
         {
            fputs(IntStr(gfxXPaper),fh);
            fputs(" ",fh);
            fputs(IntStr(gfxYPaper),fh);
         }
         else
         {
            fputs(IntStr(gfxYPaper),fh);
            fputs(" ",fh);
            fputs(IntStr(gfxXPaper),fh);
         }
         fputs("\n255\n",fh);
         size=fwrite(gfxPaper,1,gfxPaperSize,fh);
         fclose(fh);

         if (size!=gfxPaperSize)
         {
            unlink(dest);
            xwDisplayMessage(NULL,xwGR(1,"WriteError","Not enough space available on disk !!!"));
            return(xwFalse);
         }

         return(xwTrue);
      }
      return(xwFalse);
   }
   return(xwFalse);
}

/* ################################################################### */
/* Shared Routines */

/* Get Shared Memory */
int xwGFXCreateShared(int w,int h) /*fold00*/
{
   int         zeiger;

   if (gfxShared!=NULL) xwGFXRemoveShared();

   gfxXShared=w;
   gfxYShared=h;
   gfxSharedSize=((gfxXShared*gfxYShared)*3)+(sizeof(int)*52);

   gfxSharedID=shmget(IPC_PRIVATE,gfxSharedSize,IPC_CREAT|IPC_EXCL|0777);
   if (gfxSharedID>-1)
   {
      gfxShared = (sharedMemory*) shmat(gfxSharedID, NULL, 0);
      if (gfxShared!=NULL)
      {
         gfxShared->com=0; /* Main Process */
         for (zeiger=0 ; zeiger<=((gfxXShared*gfxYShared)*3) ; zeiger++)
         {
            gfxShared->mem[zeiger]=204;
         }

         for (zeiger=0 ; zeiger<=99 ; zeiger++)
         {
            gfxShared->result[zeiger]=-1;
         }
         return(xwTrue);
      }
      else
      {
         return(xwFalse);
      }
   }
   else
   {
      return(xwFalse);
   }
}

/* Clear Shared Memory */
void xwGFXClearShared(void)
{
   int      zeiger;

   if (gfxShared!=NULL)
   {
      for (zeiger=0 ; zeiger<=((gfxXShared*gfxYShared)*3) ; zeiger++)
      {
         gfxShared->mem[zeiger]=204;
      }

      for (zeiger=0 ; zeiger<=99 ; zeiger++)
      {
         gfxShared->result[zeiger]=-1;
      }
   }

   return;
}

/* Remove Shared Memory */
void xwGFXRemoveShared(void) /*fold00*/
{
   if (gfxShared!=NULL)
   {
      shmdt(gfxShared);
      shmctl(gfxSharedID,IPC_RMID,NULL);

      gfxShared=NULL;
      gfxSharedID=-1;
      gfxXShared=0;
      gfxYShared=0;
      gfxSharedSize=0;
   }
   return;
}

/* Set Communicate Shared Memory */
void xwGFXSharedSetCom(int com)
{
   if (gfxShared!=NULL)
   {
      gfxShared->com=com;
   }
   else
   {
      gfxShared->com=-1;
   }

   return;
}

/* Get Communicate Shared Memory */
int xwGFXSharedGetCom(void)
{
   if (gfxShared!=NULL)
   {
      return(gfxShared->com);
   }
   else
   {
      return(-1);
   }
}

/* Reset Result Data */
void xwGFXSharedResetResult(void)
{
   int         zeiger;

   if (gfxShared!=NULL)
   {
      for (zeiger=0 ; zeiger<=99 ; zeiger++)
      {
         gfxShared->result[zeiger]=-1;
      }
   }

   return;
}
/* Set Result Data */
void xwGFXSharedSetResult(int entry,int result)
{
   if (gfxShared!=NULL)
   {
      if ( (entry>=0) && (entry<=99) )
      {
         gfxShared->result[entry]=result;
      }
   }

   return;
}

/* Get Result Data */
int xwGFXSharedGetResult(int entry)
{
   if (gfxShared!=NULL)
   {
      if ( (entry>=0) && (entry<=99) )
      {
         return(gfxShared->result[entry]);
      }
      else
      {
         return(-1);
      }
   }
   else
   {
      return(-1);
   }
}

/* Save Shared Memory */
int xwGFXSaveShared(char *dest)
{
   FILE           *fh;
   int            size;

   if (gfxShared!=NULL)
   {
      if ( (fh=fopen(dest,"w"))!=NULL )
      {
         /* Write Header */
         fputs("P6\n",fh);
         fputs("# written by xwGUI\n",fh);

         fputs(IntStr(gfxXShared),fh);
         fputs(" ",fh);
         fputs(IntStr(gfxYShared),fh);
         fputs("\n255\n",fh);
         size=fwrite(gfxShared,1,((gfxXShared*gfxYShared)*3),fh);
         fclose(fh);

         if (size!=((gfxXShared*gfxYShared)*3))
         {
            unlink(dest);
            xwDisplayMessage(NULL,xwGR(1,"WriteError","Not enough space available on disk !!!"));
            return(xwFalse);
         }

         return(xwTrue);
      }
      return(xwFalse);
   }
   else
   {
      printf("xwgui2: Can't save shared memory !\n");
   }
   return(xwFalse);
}

/* ################################################################### */
/* Temp Routines */

/* Get Temp Memory */
int xwGFXCreateTemp(int w,int h)
{
   int            zeiger;
   int            rgb;

   if (gfxTemp!=NULL) xwGFXRemoveTemp();

   rgb=0;

   gfxXTemp=w;
   gfxYTemp=h;
   gfxTempSize=(gfxXTemp*gfxYTemp)*3;
   gfxTemp = malloc(gfxTempSize+4);
   if (gfxTemp!=NULL)
   {
      for (zeiger=0 ; zeiger<=gfxTempSize ; zeiger++)
      {
         if (gfxUseMask==1)
         {
            rgb++;
            if (rgb==1)
            {
               gfxTemp[zeiger]=gfxMRed;
            }
            else if (rgb==2)
            {
               gfxTemp[zeiger]=gfxMGreen;
            }
            else if (rgb==3)
            {
               gfxTemp[zeiger]=gfxMBlue;
               rgb=0;
            }
         }
         else
         {
            gfxTemp[zeiger]=0xff;
         }
      }

      return(xwTrue);
   }
   else
   {
      return(xwFalse);
   }
}

/* Remove Temp Memory */
void xwGFXRemoveTemp(void) /*fold00*/
{
   if (gfxTemp!=NULL)
   {
      free(gfxTemp);
      gfxTemp=NULL;
      gfxXTemp=0;
      gfxYTemp=0;
      gfxTempSize=0;
   }
   return;
}

/* Move Temp To Picture */
void xwGFXTempToPicture(void) /*fold00*/
{
   if (gfxPicture!=NULL) xwGFXRemovePicture();

   if (gfxTemp!=NULL)
   {
      gfxPicture=gfxTemp;
      gfxTemp=NULL;

      gfxXPicture=gfxXTemp;
      gfxYPicture=gfxYTemp;
      gfxPictureSize=gfxTempSize;
      gfxXTemp=0;
      gfxYTemp=0;
      gfxTempSize=0;
   }
   return;
}

/* Save Temp Datas */
void xwGFXSaveTemp(void)
{
   gfxTTemp=gfxTemp;
   gfxXTTemp=gfxXTemp;
   gfxYTTemp=gfxYTemp;
   gfxTTempSize=gfxTempSize;

   gfxTemp=NULL;
   gfxXTemp=0;
   gfxYTemp=0;
   gfxTempSize=0;

   return;
}

/* Get Temp Datas */
void xwGFXLoadTemp(void)
{
   if (gfxTemp!=NULL) xwGFXRemoveTemp();

   gfxTemp=gfxTTemp;
   gfxXTemp=gfxXTTemp;
   gfxYTemp=gfxYTTemp;
   gfxTempSize=gfxTTempSize;

   return;
}


/* Get Temp Datas */
int xwGFXGetTempX(void)
{
   if (gfxTemp!=NULL)
   {
      return(gfxXTemp);
   }
   else
   {
      return(-1);
   }
}

/* Get Temp Datas */
int xwGFXGetTempY(void)
{
   if (gfxTemp!=NULL)
   {
      return(gfxYTemp);
   }
   else
   {
      return(-1);
   }
}

/* ################################################################### */
/* Picture Routines */

/* Get Picture Memory */
int xwGFXCreatePicture(int w,int h) /*fold00*/
{
   int         zeiger;
   int         rgb;

   if (gfxPicture!=NULL) xwGFXRemovePicture();

   rgb=0;

   gfxXPicture=w;
   gfxYPicture=h;
   gfxPictureSize=(gfxXPicture*gfxYPicture)*3;
   gfxPicture = malloc(gfxPictureSize+4);
   if (gfxPicture!=NULL)
   {
      for (zeiger=0 ; zeiger<=gfxPictureSize ; zeiger++)
      {
         if (gfxUseMask==1)
         {
            rgb++;
            if (rgb==1)
            {
               gfxPicture[zeiger]=gfxMRed;
            }
            else if (rgb==2)
            {
               gfxPicture[zeiger]=gfxMGreen;
            }
            else if (rgb==3)
            {
               gfxPicture[zeiger]=gfxMBlue;
               rgb=0;
            }
         }
         else
         {
            gfxPicture[zeiger]=0xff;
         }
      }

      return(xwTrue);
   }
   else
   {
      return(xwFalse);
   }
}

/* Remove Picture Memory */
void xwGFXRemovePicture(void) /*fold00*/
{
   if (gfxPicture!=NULL)
   {
      free(gfxPicture);
      gfxPicture=NULL;
      gfxXPicture=0;
      gfxYPicture=0;
      gfxPictureSize=0;
   }
   return;
}

/* Load Picture */
int xwGFXLoadPicture(char *source) /*fold00*/
{
   int            pos,xsize,ysize;
   FILE           *fh;
   int            bug;

   int            count;
   int            x,y;
   int            data;

   /* Get Source Datas */
   if (checkppm(source)==0)
   {
      pos=xwGetGFXP();
      xsize=xwGetGFXX();
      ysize=xwGetGFXY();

      if (xwGFXCreatePicture(xsize,ysize)==xwTrue)
      {
         if ( (fh=fopen(source,"r"))!=NULL)
         {
            bug=fseek(fh,pos,SEEK_SET);

            /* PPM P6 Format */
            if (xwGetGFXID()=='6')
            {
               /*printf("xwGUI2: PPM P6 Format detected\n");*/

               fread(gfxPicture,gfxPictureSize,1,fh);
            }
            else if (xwGetGFXID()=='5')
            {
               /*printf("xwGUI2: PPM P5 Format detected\n");*/

               count=0;
               for (y=0 ; y<ysize ; y++)
               {
                  for (x=0 ; x<xsize ; x++)
                  {
                     data=fgetc(fh);

                     gfxPicture[count]=data;
                     count++;
                     gfxPicture[count]=data;
                     count++;
                     gfxPicture[count]=data;
                     count++;
                  }
               }
            }
            fclose(fh);

            return(xwTrue);
         }
      }
   }
   else
   {
      printf("xwGUI2: Wrong PPM-Format detected !!!\n");
   }
   return(xwFalse);
}

/* Save Picture Memory */
int xwGFXSavePicture(char *dest) /*fold00*/
{
   FILE           *fh;
   int            size;

/*
   char           astr[255];
   int            xp,yp;
   int            count;
   
   unsigned char  *rgb;
*/

   /*
   int            x,y,r;
   float          grad;
   */
   
   if (gfxPicture!=NULL)
   {
   
      /* Save Picture in Hex Datas */
      /*
      if ( (fh=fopen("/home/skraus/image.h","w"))!=NULL)
      {
         fputs("xx=",fh);
         fputs(IntStr(gfxXPicture),fh);
         fputs(";\n",fh);
         fputs("yy=",fh);
         fputs(IntStr(gfxYPicture),fh);
         fputs(";\n",fh);
         
         for (yp=0 ; yp<gfxYPicture ; yp++)
         {
            fputs("else if (yy==",fh);
            sprintf(astr,"%3i",yp);
            fputs(astr,fh);
            fputs(") strcpy(astr,\"",fh);
            
            count=yp*(gfxXPicture*3);
            for (xp=0 ; xp<gfxXPicture ; xp++)
            {
               rgb=gfxPicture+count;
               sprintf(astr,"%03i",(int) rgb[0]);
               fputs(astr,fh);
               count++;
               sprintf(astr,"%03i",(int) rgb[1]);
               fputs(astr,fh);
               count++;
               sprintf(astr,"%03i",(int) rgb[2]);
               fputs(astr,fh);
               count++;
            }
            fputs("\");\n",fh);
         }
         fclose(fh);      
      }
      */
   
   
      /*
      x=gfxXPicture/2;
      y=gfxYPicture/2;
      r=x;
      if (x>y) r=y;;
      
      for (grad==0 ; grad<=360 ; grad=grad+0.001)
      {
         xwGFXSetPixel(memPicture,x+(int) (sin(grad)*(float) r),y+(int) (cos(grad)*(float) r),0,0,0);
      }
      */
   
      if ( (fh=fopen(dest,"w"))!=NULL )
      {
         /* Write Header */
         fputs("P6\n",fh);
         fputs("# written by xwGUI\n",fh);
         fputs(IntStr(gfxXPicture),fh);
         fputs(" ",fh);
         fputs(IntStr(gfxYPicture),fh);
         fputs("\n255\n",fh);
         size=fwrite(gfxPicture,1,gfxPictureSize,fh);
         fclose(fh);

         if (size!=gfxPictureSize)
         {
            unlink(dest);
            xwDisplayMessage(NULL,xwGR(1,"WriteError","Not enough space available on disk !!!"));
            return(xwFalse);
         }
         return(xwTrue);
      }
      return(xwTrue);
   }
   return(xwFalse);
}

/* Get Picture Datas */
int xwGFXGetPictureX(void)
{
   if (gfxPicture!=NULL)
   {
      return(gfxXPicture);
   }
   else
   {
      return(-1);
   }
}

/* Get Picture Datas */
int xwGFXGetPictureY(void)
{
   if (gfxPicture!=NULL)
   {
      return(gfxYPicture);
   }
   else
   {
      return(-1);
   }
}

/* ################################################################### */
/* Second Temp Routines */

/* Get Second Temp Memory */
int xwGFXCreateSTemp(int w,int h)
{
   int            zeiger;
   int            rgb;

   if (gfxSTemp!=NULL) xwGFXRemoveSTemp();

   rgb=0;

   gfxXSTemp=w;
   gfxYSTemp=h;
   gfxSTempSize=(gfxXSTemp*gfxYSTemp)*3;
   gfxSTemp = malloc(gfxSTempSize+4);
   if (gfxSTemp!=NULL)
   {
      for (zeiger=0 ; zeiger<=gfxSTempSize ; zeiger++)
      {
         if (gfxUseMask==1)
         {
            rgb++;
            if (rgb==1)
            {
               gfxSTemp[zeiger]=gfxMRed;
            }
            else if (rgb==2)
            {
               gfxSTemp[zeiger]=gfxMGreen;
            }
            else if (rgb==3)
            {
               gfxSTemp[zeiger]=gfxMBlue;
               rgb=0;
            }
         }
         else
         {
            gfxSTemp[zeiger]=0xff;
         }
      }

      return(xwTrue);
   }
   else
   {
      return(xwFalse);
   }
}

/* Remove Second Temp Memory */
void xwGFXRemoveSTemp(void) /*fold00*/
{
   if (gfxSTemp!=NULL)
   {
      free(gfxSTemp);
      gfxSTemp=NULL;
      gfxXSTemp=0;
      gfxYSTemp=0;
      gfxSTempSize=0;
   }
   return;
}

/* Move Temp To Picture */
void xwGFXSTempToSPicture(void) /*fold00*/
{
   if (gfxSPicture!=NULL) xwGFXRemoveSPicture();

   if (gfxSTemp!=NULL)
   {
      gfxSPicture=gfxSTemp;
      gfxSTemp=NULL;

      gfxXSPicture=gfxXSTemp;
      gfxYSPicture=gfxYSTemp;
      gfxSPictureSize=gfxSTempSize;
      gfxXSTemp=0;
      gfxYSTemp=0;
      gfxSTempSize=0;
   }
   return;
}

/* Get Second Temp Datas */
int xwGFXGetSTempX(void)
{
   if (gfxSTemp!=NULL)
   {
      return(gfxXSTemp);
   }
   else
   {
      return(-1);
   }
}

/* Get Temp Datas */
int xwGFXGetSTempY(void)
{
   if (gfxSTemp!=NULL)
   {
      return(gfxYSTemp);
   }
   else
   {
      return(-1);
   }
}

/* ################################################################### */
/* Second Picture Routines */

/* Get Picture Memory */
int xwGFXCreateSPicture(int w,int h) /*fold00*/
{
   int         zeiger;
   int         rgb;

   if (gfxSPicture!=NULL) xwGFXRemoveSPicture();

   rgb=0;

   gfxXSPicture=w;
   gfxYSPicture=h;
   gfxSPictureSize=(gfxXSPicture*gfxYSPicture)*3;
   gfxSPicture = malloc(gfxSPictureSize+4);
   if (gfxSPicture!=NULL)
   {
      for (zeiger=0 ; zeiger<=gfxSPictureSize ; zeiger++)
      {
         if (gfxUseMask==1)
         {
            rgb++;
            if (rgb==1)
            {
               gfxSPicture[zeiger]=gfxMRed;
            }
            else if (rgb==2)
            {
               gfxSPicture[zeiger]=gfxMGreen;
            }
            else if (rgb==3)
            {
               gfxSPicture[zeiger]=gfxMBlue;
               rgb=0;
            }
         }
         else
         {
            gfxSPicture[zeiger]=0xff;
         }
      }

      return(xwTrue);
   }
   else
   {
      return(xwFalse);
   }
}

/* Remove Picture Memory */
void xwGFXRemoveSPicture(void) /*fold00*/
{
   if (gfxSPicture!=NULL)
   {
      free(gfxSPicture);
      gfxSPicture=NULL;
      gfxXSPicture=0;
      gfxYSPicture=0;
      gfxSPictureSize=0;
   }
   return;
}

/* Load Picture */
int xwGFXLoadSPicture(char *source) /*fold00*/
{
   int            pos,xsize,ysize;
   FILE           *fh;
   int            bug;

   int            count;
   int            x,y;
   int            data;

   /* Get Source Datas */
   if (checkppm(source)==0)
   {
      pos=xwGetGFXP();
      xsize=xwGetGFXX();
      ysize=xwGetGFXY();

      if (xwGFXCreateSPicture(xsize,ysize)==xwTrue)
      {
         if ( (fh=fopen(source,"r"))!=NULL)
         {
            bug=fseek(fh,pos,SEEK_SET);

            /* PPM P6 Format */
            if (xwGetGFXID()=='6')
            {
               /*printf("xwGUI2: PPM P6 Format detected\n");*/

               fread(gfxSPicture,gfxSPictureSize,1,fh);
            }
            else if (xwGetGFXID()=='5')
            {
               /*printf("xwGUI2: PPM P5 Format detected\n");*/

               count=0;
               for (y=0 ; y<ysize ; y++)
               {
                  for (x=0 ; x<xsize ; x++)
                  {
                     data=fgetc(fh);

                     gfxSPicture[count]=data;
                     count++;
                     gfxSPicture[count]=data;
                     count++;
                     gfxSPicture[count]=data;
                     count++;
                  }
               }
            }
            fclose(fh);

            return(xwTrue);
         }
      }
   }
   else
   {
      printf("xwGUI2: Wrong PPM-Format detected !!!\n");
   }
   return(xwFalse);
}

/* Save Picture Memory */
int xwGFXSaveSPicture(char *dest) /*fold00*/
{
   FILE           *fh;
   int            size;
   
   if (gfxSPicture!=NULL)
   {
      if ( (fh=fopen(dest,"w"))!=NULL )
      {
         /* Write Header */
         fputs("P6\n",fh);
         fputs("# written by xwGUI\n",fh);
         fputs(IntStr(gfxXSPicture),fh);
         fputs(" ",fh);
         fputs(IntStr(gfxYSPicture),fh);
         fputs("\n255\n",fh);
         size=fwrite(gfxSPicture,1,gfxSPictureSize,fh);
         fclose(fh);

         if (size!=gfxSPictureSize)
         {
            unlink(dest);
            xwDisplayMessage(NULL,xwGR(1,"WriteError","Not enough space available on disk !!!"));
            return(xwFalse);
         }
         return(xwTrue);
      }
      return(xwTrue);
   }
   return(xwFalse);
}

/* Get Second Picture Datas */
int xwGFXGetSPictureX(void)
{
   if (gfxSPicture!=NULL)
   {
      return(gfxXSPicture);
   }
   else
   {
      return(-1);
   }
}

/* Get Picture Datas */
int xwGFXGetSPictureY(void)
{
   if (gfxSPicture!=NULL)
   {
      return(gfxYSPicture);
   }
   else
   {
      return(-1);
   }
}

/* ################################################################### */
/* Pixel Manipulations */

/* Set Text Color */
void xwGFXSetTextColor(int r,int g,int b)
{
   gfxRed=r;
   gfxGreen=g;
   gfxBlue=b;

   return;
}

/* Ignore Paper Color */
void xwGFXIgnorePaper(int mo)
{
   gfxIgnore=mo;

   return;
}

/* Set Mask Color */
void xwGFXSetMaskColor(int rA,int gA,int bA,int rB,int gB,int bB,int rC,int gC,int bC)
{
   int               r,g,b;

   /* r=255 g=255 b=255 reserved for Paper-Color */
   r=254; g=254; b=254;

   while ((r==rA) || (r==rB) || (r==rC))
   {
      r--;
   }

   while ((g==gA) || (g==gB) || (g==gC))
   {
      g--;
   }

   while ((b==bA) || (b==bB) || (b==bC))
   {
      b--;
   }

   gfxMRed=r;
   gfxMGreen=g;
   gfxMBlue=b;

   return;
}

/* Set Mask Color */
void xwGFXSetMaskLColor(int cA,int cB,int cC)
{
   int               rA,gA,bA;
   int               rB,gB,bB;
   int               rC,gC,bC;

   int               rgb;

   rgb=cA;
   rA=rgb/65536;
   rgb=rgb-(rA*65536);
   gA=rgb/256;
   rgb=rgb-(gA*256);
   bA=rgb;

   rgb=cB;
   rB=rgb/65536;
   rgb=rgb-(rB*65536);
   gB=rgb/256;
   rgb=rgb-(gB*256);
   bB=rgb;

   rgb=cC;
   rC=rgb/65536;
   rgb=rgb-(rC*65536);
   gC=rgb/256;
   rgb=rgb-(gC*256);
   bC=rgb;

   xwGFXSetMaskColor(rA,gA,bA,rB,gB,bB,rC,gC,bC);

   return;
}

/* Get Mask Color */
int xwGFXGetMaskLColor(void)
{
   if (gfxUseMask==0) return(0xffffff);
   
   return((gfxMRed*65536)+(gfxMGreen*256)+gfxMBlue);
}


/* Mask Mode */
void xwGFXUseMask(int mo)
{
   gfxUseMask=mo;

   return;
}

/* Set Picture Mask */
void xwGFXSetPicMask(int mo,int x,int y,int w,int h)
{
   if (mo==1)
   {
      gfxMFlag=1;

      gfxMX=x;
      gfxMY=y;
      gfxMW=w;
      gfxMH=h;

      /*printf("%i %i %i %i\n",x,y,w,h);*/
   }
   else if (mo==0)
   {
      gfxMFlag=0;
   }

   return;
}


/* Set Transparenzy Value */
void xwGFXSetTrans(int trans)
{
   if ((trans>=0) && (trans<=100))
   {
      gfxTrans=trans;
      gfxTF=(0.01 * (float) gfxTrans);
   }
   return;
}

/* Set Transparency Mode */
void xwGFXSetTransMode(int mo)
{
   gfxTransMode=mo;

   return;
}

/* Calc Transparenzy Source/Destination */
void xwGFXCalcTrans(int ir,int ig,int ib,int or,int og,int ob) /*fold00*/
{
   if (gfxTransMode==0)
   {
      /* Positive Transparenz */
      RV=(int)  ( ( (float) ir*(1-gfxTF)) + ( ( ((float) ir*gfxTF) * (float) or) ) /255 );
      GV=(int)  ( ( (float) ig*(1-gfxTF)) + ( ( ((float) ig*gfxTF) * (float) og) ) /255 );
      BV=(int)  ( ( (float) ib*(1-gfxTF)) + ( ( ((float) ib*gfxTF) * (float) ob) ) /255 );

      /* Check Values */
      if (RV<0) RV=0;
      if (GV<0) GV=0;
      if (BV<0) BV=0;
   }
   else if (gfxTransMode==1)
   {
      /* Negative Transparenz */
      RV=(int)  ( ( (float) (255-ir)*(1-gfxTF)) + ( ( ((float) (255-ir)*gfxTF) * (float) (255-or)) ) /255 );
      GV=(int)  ( ( (float) (255-ig)*(1-gfxTF)) + ( ( ((float) (255-ig)*gfxTF) * (float) (255-og)) ) /255 );
      BV=(int)  ( ( (float) (255-ib)*(1-gfxTF)) + ( ( ((float) (255-ib)*gfxTF) * (float) (255-ob)) ) /255 );

      RV=255-RV;
      GV=255-GV;
      BV=255-BV;

      if (RV<0) RV=0;
      if (GV<0) GV=0;
      if (BV<0) BV=0;
   }

   return;
}

/* Copy Pixel */
void xwGFXCopyPixel(int in,int x, int y,int out,int xx,int yy)
{
   static unsigned char    *rgb;
   static int              r,g,b;

   static int              rgbv;
   static int              rgbm;    /* write pixel mode */

   r=255;
   g=255;
   b=255;

   rgbm=0;

   /* Get RGB-Values */
   if (in==memPaper)
   {
      if ((x>=0) && (y>=0) && (x<gfxXPaper) && (y<gfxYPaper))
      {
         if (gfxPaperLP==0)
         {
            rgb=gfxPaper+((x+(y*gfxXPaper))*3);
         }
         else
         {
            rgb=gfxPaper+(((gfxYPaper-y)-1+(x*gfxYPaper))*3);
         }

         /*rgb=gfxPaper+((x+(y*gfxXPaper))*3);*/
         r=rgb[0];
         g=rgb[1];
         b=rgb[2];
      }
   }
   if (in==memShared)
   {
      if ((x>=0) && (y>=0) && (x<gfxXShared) && (y<gfxYShared))
      {
         rgb=gfxShared->mem+((x+(y*gfxXShared))*3);
         r=rgb[0];
         g=rgb[1];
         b=rgb[2];
      }
   }
   else if (in==memTemp)
   {
      if ((x>=0) && (y>=0) && (x<gfxXTemp) && (y<gfxYTemp))
      {
         rgb=gfxTemp+((x+(y*gfxXTemp))*3);
         r=rgb[0];
         g=rgb[1];
         b=rgb[2];
      }
   }
   else if (in==memPicture)
   {
      if ((x>=0) && (y>=0) && (x<gfxXPicture) && (y<gfxYPicture))
      {
         rgb=gfxPicture+((x+(y*gfxXPicture))*3);
         r=rgb[0];
         g=rgb[1];
         b=rgb[2];
      }
   }
   else if (in==memSTemp)
   {
      if ((x>=0) && (y>=0) && (x<gfxXSTemp) && (y<gfxYSTemp))
      {
         rgb=gfxSTemp+((x+(y*gfxXSTemp))*3);
         r=rgb[0];
         g=rgb[1];
         b=rgb[2];
      }
   }
   else if (in==memSPicture)
   {
      if ((x>=0) && (y>=0) && (x<gfxXSPicture) && (y<gfxYSPicture))
      {
         rgb=gfxSPicture+((x+(y*gfxXSPicture))*3);
         r=rgb[0];
         g=rgb[1];
         b=rgb[2];
      }
   }

   /*
   if (in==memTemp)
   {
      printf("%4i %4i - %3i %3i %3i ( %3i %3i %3i )\n",x,y,r,g,b,gfxMRed,gfxMGreen,gfxMBlue);
   }
   */

   if ((gfxUseMask==1) && (gfxIgnore==0))
   {
      /* Picture Area */
      if (gfxMFlag==1)
      {
         if ((x>=gfxMX) && (x<gfxMX+gfxMW) && (y>=gfxMY) && (y<gfxMY+gfxMH))
         {
            rgbm=0;
            /*printf("%i %i %i %i - %i %i\n",gfxMX,gfxMY,gfxMW,gfxMH,x,y);*/
         }
         else
         {
            if ((r==gfxMRed) && (g==gfxMGreen) && (b==gfxMBlue))
            {
               rgbm=1;
            }
         }
      }
      else
      {
         if ((r==gfxMRed) && (g==gfxMGreen) && (b==gfxMBlue))
         {
            rgbm=1;
         }
      }
   }
   else
   {      
      /* RGB-Check */
      rgbv=(r*65536)+(g*256)+b;
      if (gfxIgnore==1)
      {
         if (rgbv==0xffffff)
         {
            rgbm=1;
         }
         else
         {
            r=gfxRed;
            g=gfxGreen;
            b=gfxBlue;
         }
      }
   }

   /* Set Pixel */
   if (rgbm==0)
   {
      if (gfxTrans==0)
      {
         /* Set RGB-Value */
         if (out==memPaper)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXPaper) && (yy<gfxYPaper))
            {
               if (gfxPaperLP==0)
               {
                  rgb=gfxPaper+((xx+(yy*gfxXPaper))*3);
               }
               else
               {
                  rgb=gfxPaper+(((gfxYPaper-yy)-1+(xx*gfxYPaper))*3);
               }
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;

            }
         }
         else if (out==memShared)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXShared) && (yy<gfxYShared))
            {
               rgb=gfxShared->mem+((xx+(yy*gfxXShared))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memTemp)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXTemp) && (yy<gfxYTemp))
            {
               rgb=gfxTemp+((xx+(yy*gfxXTemp))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memPicture)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXPicture) && (yy<gfxYPicture))
            {
               rgb=gfxPicture+((xx+(yy*gfxXPicture))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memSTemp)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXSTemp) && (yy<gfxYSTemp))
            {
               rgb=gfxSTemp+((xx+(yy*gfxXSTemp))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memSPicture)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXSPicture) && (yy<gfxYSPicture))
            {
               rgb=gfxSPicture+((xx+(yy*gfxXSPicture))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
      }
      else
      {
         /* Set RGB-Value */
         if (out==memPaper)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXPaper) && (yy<gfxYPaper))
            {
               if (gfxPaperLP==0)
               {
                  rgb=gfxPaper+((xx+(yy*gfxXPaper))*3);
               }
               else
               {
                  rgb=gfxPaper+(((gfxYPaper-yy)-1+(xx*gfxYPaper))*3);
               }
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;

               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memShared)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXShared) && (yy<gfxYShared))
            {
               rgb=gfxShared->mem+((xx+(yy*gfxXShared))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memTemp)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXTemp) && (yy<gfxYTemp))
            {
               rgb=gfxTemp+((xx+(yy*gfxXTemp))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memPicture)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXPicture) && (yy<gfxYPicture))
            {
               rgb=gfxPicture+((xx+(yy*gfxXPicture))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memSTemp)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXSTemp) && (yy<gfxYSTemp))
            {
               rgb=gfxSTemp+((xx+(yy*gfxXSTemp))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memSPicture)
         {
            if ((xx>=0) && (yy>=0) && (xx<gfxXSPicture) && (yy<gfxYSPicture))
            {
               rgb=gfxSPicture+((xx+(yy*gfxXSPicture))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
      }
   }

   return;
}

/* Get RGB-Value */
void xwGFXGetPixel(int in,int x,int y) /*fold00*/
{
   static unsigned char  *rgb;

   RV=255;
   GV=255;
   BV=255;

   /* Get RGB-Values */
   if (in==memPaper)
   {
      if ((x>=0) && (y>=0) && (x<gfxXPaper) && (y<gfxYPaper))
      {
         if (gfxPaperLP==0)
         {
            rgb=gfxPaper+((x+(y*gfxXPaper))*3);
         }
         else
         {
            rgb=gfxPaper+(((gfxYPaper-y)-1+(x*gfxYPaper))*3);
         }

         /*rgb=gfxPaper+((x+(y*gfxXPaper))*3);*/
         RV=(int) rgb[0];
         GV=(int) rgb[1];
         BV=(int) rgb[2];
      }
   }
   else if (in==memShared)
   {
      if ((x>=0) && (y>=0) && (x<gfxXShared) && (y<gfxYShared))
      {
         rgb=gfxShared->mem+((x+(y*gfxXShared))*3);
         RV=(int) rgb[0];
         GV=(int) rgb[1];
         BV=(int) rgb[2];
      }
   }
   else if (in==memTemp)
   {
      if ((x>=0) && (y>=0) && (x<gfxXTemp) && (y<gfxYTemp))
      {
         rgb=gfxTemp+((x+(y*gfxXTemp))*3);
         RV=(int) rgb[0];
         GV=(int) rgb[1];
         BV=(int) rgb[2];
      }
   }
   else if (in==memPicture)
   {
      if ((x>=0) && (y>=0) && (x<gfxXPicture) && (y<gfxYPicture))
      {
         rgb=gfxPicture+((x+(y*gfxXPicture))*3);
         RV=(int) rgb[0];
         GV=(int) rgb[1];
         BV=(int) rgb[2];
      }
   }
   else if (in==memSTemp)
   {
      if ((x>=0) && (y>=0) && (x<gfxXSTemp) && (y<gfxYSTemp))
      {
         rgb=gfxSTemp+((x+(y*gfxXSTemp))*3);
         RV=(int) rgb[0];
         GV=(int) rgb[1];
         BV=(int) rgb[2];
      }
   }
   else if (in==memSPicture)
   {
      if ((x>=0) && (y>=0) && (x<gfxXSPicture) && (y<gfxYSPicture))
      {
         rgb=gfxSPicture+((x+(y*gfxXSPicture))*3);
         RV=(int) rgb[0];
         GV=(int) rgb[1];
         BV=(int) rgb[2];
      }
   }
   return;
}

/* Set RGB-Value */
void xwGFXSetPixel(int out,int x,int y,int r,int g,int b)
{
   static unsigned char  *rgb;

   static int             rgbv;
   static int             rgbm;  /* write pixel mode */

   rgbm=0;

   /* RGB-Check */
   rgbv=(r*65536)+(g*256)+b;
   if ((gfxIgnore==1) && (rgbv==0xffffff)) rgbm=1;

   /* Set Pixel */
   if (rgbm==0)
   {
      if (gfxTrans==0)
      {
         /* Set RGB-Value */
         if (out==memPaper)
         {
            if ((x>=0) && (y>=0) && (x<gfxXPaper) && (y<gfxYPaper))
            {
               if (gfxPaperLP==0)
               {
                  rgb=gfxPaper+((x+(y*gfxXPaper))*3);
               }
               else
               {
                  rgb=gfxPaper+(((gfxYPaper-y)-1+(x*gfxYPaper))*3);
               }

               /*rgb=gfxPaper+((x+(y*gfxXPaper))*3);*/
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memShared)
         {
            if ((x>=0) && (y>=0) && (x<gfxXShared) && (y<gfxYShared))
            {
               rgb=gfxShared->mem+((x+(y*gfxXShared))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memTemp)
         {
            if ((x>=0) && (y>=0) && (x<gfxXTemp) && (y<gfxYTemp))
            {
               rgb=gfxTemp+((x+(y*gfxXTemp))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memPicture)
         {
            if ((x>=0) && (y>=0) && (x<gfxXPicture) && (y<gfxYPicture))
            {
               rgb=gfxPicture+((x+(y*gfxXPicture))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memSTemp)
         {
            if ((x>=0) && (y>=0) && (x<gfxXSTemp) && (y<gfxYSTemp))
            {
               rgb=gfxSTemp+((x+(y*gfxXSTemp))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
         else if (out==memSPicture)
         {
            if ((x>=0) && (y>=0) && (x<gfxXSPicture) && (y<gfxYSPicture))
            {
               rgb=gfxSPicture+((x+(y*gfxXSPicture))*3);
               rgb[0]=(unsigned char) r;
               rgb[1]=(unsigned char) g;
               rgb[2]=(unsigned char) b;
            }
         }
      }
      else
      {
         /* Set RGB-Value */
         if (out==memPaper)
         {
            if ((x>=0) && (y>=0) && (x<gfxXPaper) && (y<gfxYPaper))
            {
               if (gfxPaperLP==0)
               {
                  rgb=gfxPaper+((x+(y*gfxXPaper))*3);
               }
               else
               {
                  rgb=gfxPaper+(((gfxYPaper-y)-1+(x*gfxYPaper))*3);
               }

               /*rgb=gfxPaper+((x+(y*gfxXPaper))*3);*/
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memShared)
         {
            if ((x>=0) && (y>=0) && (x<gfxXShared) && (y<gfxYShared))
            {
               rgb=gfxShared->mem+((x+(y*gfxXShared))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memTemp)
         {
            if ((x>=0) && (y>=0) && (x<gfxXTemp) && (y<gfxYTemp))
            {
               rgb=gfxTemp+((x+(y*gfxXTemp))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memPicture)
         {
            if ((x>=0) && (y>=0) && (x<gfxXPicture) && (y<gfxYPicture))
            {
               rgb=gfxPicture+((x+(y*gfxXPicture))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memSTemp)
         {
            if ((x>=0) && (y>=0) && (x<gfxXSTemp) && (y<gfxYSTemp))
            {
               rgb=gfxSTemp+((x+(y*gfxXSTemp))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
         else if (out==memSPicture)
         {
            if ((x>=0) && (y>=0) && (x<gfxXSPicture) && (y<gfxYSPicture))
            {
               rgb=gfxSPicture+((x+(y*gfxXSPicture))*3);
               xwGFXCalcTrans(r,g,b,rgb[0],rgb[1],rgb[2]);
               rgb[0]=(unsigned char) RV;
               rgb[1]=(unsigned char) GV;
               rgb[2]=(unsigned char) BV;
               /*
               rgb[0]=(unsigned char) xwGFXCalcTrans(r,rgb[0]);
               rgb[1]=(unsigned char) xwGFXCalcTrans(g,rgb[1]);
               rgb[2]=(unsigned char) xwGFXCalcTrans(b,rgb[2]);
               */
            }
         }
      }
   }

   return;
}

/* ################################################################### */
/* Picture Manipulations */

/* Rotate Picture */
void xwGFXRotatePicture(int rot) /*fold00*/
{

   int         x,y;
   int         rows;
   int         r,g,b;
   int         rr,gg,bb;

   div_t       pp;

   if (gfxPicture!=NULL)
   {
      /* 180 Degrees Rotation */
      if ((rot==180) || (rot==-180))
      {
         pp=div(gfxYPicture,2);
         rows=pp.quot;

         for (y=0 ; y<rows ; y++)
         {
            for (x=0 ; x<gfxXPicture ; x++)
            {
               xwGFXGetPixel(memPicture,x,y);
               r=RV; g=GV; b=BV;
               xwGFXGetPixel(memPicture,gfxXPicture-1-x,gfxYPicture-1-y);
               rr=RV; gg=GV; bb=BV;
               xwGFXSetPixel(memPicture,gfxXPicture-1-x,gfxYPicture-1-y,r,g,b);
               xwGFXSetPixel(memPicture,x,y,rr,gg,bb);
            }
         }
         /* rotate last line */
         if (pp.rem==1)
         {
            for (x=0 ; x<gfxXPicture/2 ;x++)
            {
               xwGFXGetPixel(memPicture,x,rows);
               r=RV; g=GV; b=BV;
               xwGFXGetPixel(memPicture,gfxXPicture-1-x,rows);
               rr=RV; gg=GV; bb=BV;
               xwGFXSetPixel(memPicture,gfxXPicture-1-x,rows,r,g,b);
               xwGFXSetPixel(memPicture,x,rows,rr,gg,bb);
            }
         }
      }

      /* 90 Degrees Rotation */
      else if (rot==90)
      {
         if (xwGFXCreateTemp(gfxYPicture,gfxXPicture)==xwTrue)
         {
            for (y=0 ; y<gfxYPicture ; y++)
            {
               for (x=0 ; x<gfxXPicture ; x++)
               {
                  xwGFXCopyPixel(memPicture,x,y,memTemp,gfxXTemp-1-y,x);
               }
            }
            xwGFXTempToPicture();
         }
      }

      /* -90 Degrees Rotation */
      else if (rot==-90)
      {
         if (xwGFXCreateTemp(gfxYPicture,gfxXPicture)==xwTrue)
         {
            for (y=0 ; y<gfxYPicture ; y++)
            {
               for (x=0 ; x<gfxXPicture ; x++)
               {
                  xwGFXCopyPixel(memPicture,x,y,memTemp,y,gfxYTemp-1-x);
               }
            }
            xwGFXTempToPicture();
         }
      }
   }
   return;
}

/* Copy Picture */
void xwGFXCopyPicture(int in,int out,int x,int y) /*fold00*/
{
   int                  xx,yy,rr;
   float                grad;
   int                  xr,yr;
   struct gfxLines      *xMin=NULL;
   struct gfxLines      *xMax=NULL;         
   
   /* Quad Border */
   if (gfxBRDType==gfxQuad)
   {
      if (in==memPicture)
      {
         /* Copy Into Paper */
         if (out==memPaper)
         {
            if ((gfxPicture!=NULL) && (gfxPaper!=NULL))
            {
               for (yy=0 ; yy<gfxYPicture ; yy++)
               {
                  for (xx=0 ; xx<gfxXPicture ; xx++)
                  {
                     xwGFXCopyPixel(memPicture,xx,yy,memPaper,x+xx,y+yy);
                  }
               }
            }
         }
         if (out==memShared)
         {
            if ((gfxPicture!=NULL) && (gfxShared!=NULL))
            {
               for (yy=0 ; yy<gfxYPicture ; yy++)
               {
                  for (xx=0 ; xx<gfxXPicture ; xx++)
                  {
                     xwGFXCopyPixel(memPicture,xx,yy,memShared,x+xx,y+yy);
                  }
               }
            }
         }
         else if (out==memTemp)
         {
            if ((gfxPicture!=NULL) && (gfxPicture!=NULL))
            {
               for (yy=0 ; yy<gfxYPicture ; yy++)
               {
                  for (xx=0 ; xx<gfxXPicture ; xx++)
                  {
                     xwGFXCopyPixel(memPicture,xx,yy,memTemp,x+xx,y+yy);
                  }
               }
            }
         }
      }
      else if (in==memTemp)
      {
         /* Copy Into Paper */
         if (out==memPaper)
         {
            if ((gfxTemp!=NULL) && (gfxPaper!=NULL))
            {
               for (yy=0 ; yy<gfxYTemp ; yy++)
               {
                  for (xx=0 ; xx<gfxXTemp ; xx++)
                  {
                     xwGFXCopyPixel(memTemp,xx,yy,memPaper,x+xx,y+yy);
                  }
               }
            }
         }
      }
   }
   else if (gfxBRDType==gfxCircle)
   {
      if (in==memPicture)
      {
         /* Circle Border */
         xMin = malloc((gfxYPicture*4)+4);
         if (xMin!=NULL)
         {
            xMax=malloc((gfxYPicture*4)+4);
            if (xMax!=NULL)
            {
               /* Initialize Tab */
               for (yy=0 ; yy<gfxYPicture ; yy++)
               {
                  xMin->pos[yy]=999999;
                  xMax->pos[yy]=-1;
               }
            
               /* Calculate Datas */
               xx=gfxXPicture/2;
               yy=gfxYPicture/2;
               rr=xx;
               if (xx>yy) rr=yy;
        
               /* Draw Circle Border */
               for (grad=0 ; grad<360 ; grad=grad+0.005)
               {
                  xr=xx+(int) (sin(grad)*(float) rr);
                  yr=yy+(int) (cos(grad)*(float) rr);
                  
                  if (xr<xMin->pos[yr]) xMin->pos[yr]=xr;
                  if (xr>xMax->pos[yr]) xMax->pos[yr]=xr;
               }
              
               /* Result */
               for (yy=0 ; yy<gfxYPicture ; yy++)
               {
                  /*
                  printf("Line= %5i Min= %5i Max= %5i\n",yy,xMin->pos[yy],xMax->pos[yy]);
                  */
                  
                  if ((xMin->pos[yy]<999999) && (xMax->pos[yy]>-1))
                  {
                     for (xx=xMin->pos[yy] ; xx<=xMax->pos[yy] ; xx++)
                     {
                        xwGFXCopyPixel(memPicture,xx,yy,out,x+xx,y+yy);
                     }                  
                  }
               }
            
               free(xMax);
               xMax=NULL;
            }
            free(xMin);
            xMin=NULL;
         }
      }
      else if (in==memTemp)
      {
         /* Circle Border */
         xMin = malloc((gfxYTemp*4)+4);
         if (xMin!=NULL)
         {
            xMax=malloc((gfxYTemp*4)+4);
            if (xMax!=NULL)
            {
               /* Initialize Tab */
               for (yy=0 ; yy<gfxYTemp ; yy++)
               {
                  xMin->pos[yy]=999999;
                  xMax->pos[yy]=-1;
               }
            
               /* Calculate Datas */
               xx=gfxXTemp/2;
               yy=gfxYTemp/2;
               rr=xx;
               if (xx>yy) rr=yy;
        
               /* Draw Circle Border */
               for (grad=0 ; grad<360 ; grad=grad+0.005)
               {
                  xr=xx+(int) (sin(grad)*(float) rr);
                  yr=yy+(int) (cos(grad)*(float) rr);
                  
                  if (xr<xMin->pos[yr]) xMin->pos[yr]=xr;
                  if (xr>xMax->pos[yr]) xMax->pos[yr]=xr;
               }
              
               /* Result */
               for (yy=0 ; yy<gfxYTemp ; yy++)
               {
                  /*
                  printf("Line= %5i Min= %5i Max= %5i\n",yy,xMin->pos[yy],xMax->pos[yy]);
                  */
                  
                  if ((xMin->pos[yy]<999999) && (xMax->pos[yy]>-1))
                  {
                     for (xx=xMin->pos[yy] ; xx<=xMax->pos[yy] ; xx++)
                     {
                        xwGFXCopyPixel(memTemp,xx,yy,out,x+xx,y+yy);
                     }                  
                  }
               }           
            
               free(xMax);
               xMax=NULL;
            }
            free(xMin);
            xMin=NULL;
         }      
      }   
   }
   
   return;
}

/* Copy Area */
void xwGFXCopyArea(int in,int x,int y,int w,int h,int out,int xx,int yy)
{
   int                  xp,yp;

   for (yp=0 ; yp<h ; yp++)
   {
      for (xp=0 ; xp<w ; xp++)
      {
         xwGFXCopyPixel(in,x+xp,y+yp,out,xx+xp,yy+yp);      
      }
   }
   
   return;
}

/* Copy Special for PicBrowser ( Pic -> Raw ) */
void xwGFXCopySpecial(int x,int y,int w,int h,int r,int g,int b)
{
   int               xpos,ypos;  /* Image Position */
   int               wpos,hpos;

   int               source;
   int               dest;
   int               rgb;
   int               xx,yy;    

   /* Position Check */
   if ((x+w<=gfxXTemp) && (y+h<=gfxYTemp) && (gfxXPicture<=w) && (gfxYPicture<=h))
   {
      xpos=(w-gfxXPicture)/2;
      ypos=(h-gfxYPicture)/2;
      wpos=xpos+gfxXPicture;
      hpos=ypos+gfxYPicture;
      
      /* Top Area */
      for (yy=0 ; yy<ypos ; yy++)
      {
         rgb=0;
         dest=((x+((yy+y)*gfxXTemp))*3);
         for (xx=0 ; xx<(100*3) ; xx++)
         {
            rgb++;
            if (rgb==1)
            {
               gfxTemp[dest]=r;
            }
            else if (rgb==2)
            {
               gfxTemp[dest]=g;
            }
            else if (rgb==3)
            {
               gfxTemp[dest]=b;
               rgb=0;
            }
            dest++;
         }
      }
      
      /* Left Area */
      for (yy=ypos ; yy<ypos+hpos ; yy++)
      {
         rgb=0;
         dest=((x+((y+yy)*gfxXTemp))*3);
         for (xx=0 ; xx<(xpos*3) ; xx++)
         {
            rgb++;
            if (rgb==1)
            {
               gfxTemp[dest]=r;
            }
            else if (rgb==2)
            {
               gfxTemp[dest]=g;
            }
            else if (rgb==3)
            {
               gfxTemp[dest]=b;
               rgb=0;
            }
            dest++;
         }
      }
      
      /* Copy Image to Paper */
      for (yy=0 ; yy < gfxYPicture ; yy++)
      {
         source=((yy*gfxXPicture)*3);
         dest=(((x+xpos)+((y+ypos+yy)*gfxXTemp))*3);
         for (xx=0 ; xx<(gfxXPicture*3) ; xx++)
         {
            gfxTemp[dest]=gfxPicture[source];
            dest++;
            source++;
         }
      }
   }
   
   return;
}

/* Special Scale Render-Engine for Up-Scale */
void xwGFXRenderRGB(int count,int pos,int ra,int ga,int ba,int rb,int gb, int bb)
{
   float          rd,gd,bd;      /* RGB-Delta */
   int            rgbPos;        /* RGB-Table Position */

   if (count>1)
   {
      /* calculate delta */
      rd=(float) (ra-rb);
      gd=(float) (ga-gb);
      bd=(float) (ba-bb);

      rgbPos=IRound((100/count)*pos);

      /* Red */
      /*
      if (ra<rb) rgbPos=IRound((100/count)*pos);
      else rgbPos=IRound((100/count)*(count-pos));
      */
      RV=rb+IRound(rd*gfxRGB[rgbPos]);

      /* Green */
      /*
      if (ga<gb) rgbPos=IRound((100/count)*pos);
      else rgbPos=IRound((100/count)*(count-pos));
      */
      GV=gb+IRound(gd*gfxRGB[rgbPos]);

      /* Blue */
      /*
      if (ba<bb) rgbPos=IRound((100/count)*pos);
      else rgbPos=IRound((100/count)*(count-pos));
      */
      BV=bb+IRound(bd*gfxRGB[rgbPos]);

      if (RV<0) RV=0;
      else if (RV>255) RV=255;
      if (GV<0) GV=0;
      else if (GV>255) GV=255;
      if (BV<0) BV=0;
      else if (BV>255) BV=255;
   }

   return;
}

/* Scale Picture */
void xwGFXScalePicture(int in,int out,int x,int y,int w,int h,int rot) /*fold00*/
{
   struct rgbIColor     rgbI[20000];               /* RGB Puffer for scale down */

   /* Scale Datas */
   int                  ww,hh;                     /* Original Size */
   float                xstep,ystep;               /* Steps */
   float                xxstep,yystep;             /* Buffered Values */
   float                xcount,ycount;             /* Actual Positions */
   float                xBegin,xEnd,yBegin,yEnd;   /* Range of Datas */
   int                  xdPos,ydPos;               /* Target Position */
   int                  xPos,yPos,xxPos,yyPos;     /* Source Position */
   int                  xDist,yDist;               /* Distance for Interpolaion */
   int                  rv,gv,bv,cv,dv;            /* Color Values */
   int                  rrv,ggv,bbv;
   int                  rrrv,gggv,bbbv;
   int                  rrrrv,ggggv,bbbbv;

   int                  xloop,yloop;
   int                  dxv,dyv;                   /* Index Count for RenderRGB */

   int                  xborder,yborder;           /* Border Size for enhanced scale up */

   int                  bug;
   int                  outStream;
   int                  inStream;

   bug=0;
   if ((in==memPicture) && (gfxPicture!=NULL))
   {
      ww=w;
      hh=h;

      if (w<0) ww=gfxXPicture;
      if (h<0) hh=gfxYPicture;

      /* Calculate Scale Factor */
      xstep=(float) gfxXPicture/(float) ww;
      ystep=(float) gfxYPicture/(float) hh;
   
      bug=1;
   }
   else if ((in==memSPicture) && (gfxSPicture!=NULL))
   {
      ww=w;
      hh=h;

      if (w<0) ww=gfxXSPicture;
      if (h<0) hh=gfxYSPicture;

      /* Calculate Scale Factor */
      xstep=(float) gfxXSPicture/(float) ww;
      ystep=(float) gfxYSPicture/(float) hh;   
   
      bug=1;
   }

   if (bug==1)
   {
      bug=xwTrue;

      outStream=out;
      inStream=in;

      /* Normal Temp */
      if ((out==createTemp) && (gfxTemp==NULL))
      {
         if ((rot==90) || (rot==-90))
         {
            if (xwGFXCreateTemp(hh,ww)==xwTrue) bug=xwFalse;
         }
         else
         {
            if (xwGFXCreateTemp(ww,hh)==xwTrue) bug=xwFalse;
         }
         outStream=memTemp;
      }
      else if ((out==createTemp) && (gfxTemp!=NULL))
      {
         bug=xwFalse;
         outStream=memTemp;
      }

      /* Second Temp */
      if ((out==createSTemp) && (gfxSTemp==NULL))
      {
         if ((rot==90) || (rot==-90))
         {
            if (xwGFXCreateSTemp(hh,ww)==xwTrue) bug=xwFalse;
         }
         else
         {
            if (xwGFXCreateSTemp(ww,hh)==xwTrue) bug=xwFalse;
         }
         outStream=memSTemp;
      }
      else if ((out==createSTemp) && (gfxSTemp!=NULL))
      {
         bug=xwFalse;
         outStream=memSTemp;
      }

      /* Scale Down */
      if (xstep>1)
      {
         /* Y-Axis Scalling */
         ycount=0;
         yBegin=0;

         for (ydPos=0 ; ydPos<hh ; ydPos++)
         {
            yEnd=ycount+ystep;
            yEnd=(float) (int) ((yEnd+0.5)-1);

            dv=(int) ((yEnd-yBegin)+1);
            for (yPos=(int) yBegin ; yPos<=(int) yEnd ; yPos++)
            {

               /* X-Axis Scalling */
               xcount=0;
               xBegin=0;

               for (xdPos=0 ; xdPos<ww ; xdPos++)
               {
                  xEnd=xcount+xstep;
                  xEnd=(float) (int) ((xEnd+0.5)-1);

                  rv=0; gv=0; bv=0;
                  for (xPos=(int) xBegin ; xPos<=(int) xEnd ; xPos++)
                  {
                     xwGFXGetPixel(inStream,xPos,yPos);
                     rv=rv+RV;
                     gv=gv+GV;
                     bv=bv+BV;
                  }
                  cv=(int) ((xEnd-xBegin)+1);

                  /* Y-Scalling Values - sum all rgb-values */
                  if (yPos==yBegin)
                  {
                     /* Init RGB-Values */
                     rgbI[xdPos].r=rv/cv;
                     rgbI[xdPos].g=gv/cv;
                     rgbI[xdPos].b=bv/cv;
                  }
                  else
                  {
                     /* Sum RGB-Values */
                     rgbI[xdPos].r=rgbI[xdPos].r+(rv/cv);
                     rgbI[xdPos].g=rgbI[xdPos].g+(gv/cv);
                     rgbI[xdPos].b=rgbI[xdPos].b+(bv/cv);
                  }

                  xBegin=xEnd+1;
                  xcount=xcount+xstep;
               }
            }

            for (xloop=0 ; xloop<=ww ; xloop++)
            {
               /* Calculate Rasterline */
               if (dv!=0)
               {
                  if (rot==0)
                  {
                     xwGFXSetPixel(outStream,xloop+x,ydPos+y,rgbI[xloop].r/dv,rgbI[xloop].g/dv,rgbI[xloop].b/dv);
                  }
                  else if (rot==90)
                  {
                     xwGFXSetPixel(outStream,(h-ydPos)+x,xloop+y,rgbI[xloop].r/dv,rgbI[xloop].g/dv,rgbI[xloop].b/dv);
                  }
                  else if (rot==-90)
                  {
                     xwGFXSetPixel(outStream,ydPos+x,(w-xloop)+y,rgbI[xloop].r/dv,rgbI[xloop].g/dv,rgbI[xloop].b/dv);
                  }
                  else if ((rot==180) || (rot==-180))
                  {
                     xwGFXSetPixel(outStream,(w-xloop)+x,(h-ydPos)+y,rgbI[xloop].r/dv,rgbI[xloop].g/dv,rgbI[xloop].b/dv);
                  }
               }
            }

            yBegin=yEnd+1;
            ycount=ycount+ystep;
         }

         if (bug==xwFalse)
         {
            if (out==createTemp) xwGFXTempToPicture();
            else if (out==createSTemp) xwGFXSTempToSPicture();
         }
         else
         {
            xwGFXRemovePicture();
         }
         
      }

      /* scale up */
      else if (xstep<1)
      {
         xxstep=(float) (ww)/(float) gfxXPicture;
         yystep=(float) (hh)/(float) gfxYPicture;

         if ( ( (xxstep<2) && (yystep<2) ) || (xwGetPrefValue(xwgScalling)==0) )
         {
            /* simple pixel scalling */
            for (yloop=0 ; yloop<hh ; yloop++)
            {
               for (xloop=0 ; xloop<ww ; xloop++)
               {
                  if (rot==0)
                  {
                     xwGFXCopyPixel(memPicture,(int) (xloop*xstep),(int) (yloop*ystep),outStream,xloop+x,yloop+y);
                  }
                  else if (rot==90)
                  {
                      xwGFXCopyPixel(memPicture,(int) (xloop*xstep),(int) (yloop*ystep),outStream,(hh-yloop)+x,xloop+y);
                  }
                  else if (rot==-90)
                  {
                      xwGFXCopyPixel(memPicture,(int) (xloop*xstep),(int) (yloop*ystep),outStream,yloop+x,(ww-xloop)+y);
                  }
                  else if ((rot==-180) || (rot==180))
                  {
                      xwGFXCopyPixel(memPicture,(int) (xloop*xstep),(int) (yloop*ystep),outStream,(ww-xloop)+x,(hh-yloop)+y);
                  }
               }
            }
         }
         else
         {
            xborder=-1;
            yborder=-1;

            yPos=-1;
            yyPos=-1;
            /* best pixel scalling */
            for (yloop=0 ; yloop<=hh ; yloop++)
            {
               /* Next Source Pixel Y-Stream */
               if (IRound(yloop*ystep)>IRound(yPos*ystep))
               {
                  /* calculate distance */
                  yyPos=yloop;
                  yDist=yyPos-yPos;
                  if ((yyPos>0) && (yborder==-1))
                  {
                     yborder=yyPos;
                  }

                  xPos=-1;
                  xxPos=-1;
                  /* First Line Rendering */
                  for (xloop=0 ; xloop<=ww ; xloop++)
                  {
                     /* Next Source Pixel X-Stream */
                     if (IRound(xloop*xstep)>IRound(xPos*xstep))
                     {
                        /* calculate distance */
                        xxPos=xloop;
                        xDist=xxPos-xPos;
                        if ((xxPos>0) && (xborder==-1))
                        {
                           xborder=xxPos;
                        }

                        /* Horizontal Blending */
                        xwGFXGetPixel(inStream,(int) (IRound(xPos*xstep)-1),(int) (IRound(yyPos*ystep)-1));
                        rv=RV;
                        gv=GV;
                        bv=BV;

                        xwGFXGetPixel(inStream,(int) (IRound(xxPos*xstep)-1),(int) (IRound(yyPos*ystep)-1));
                        rrv=RV;
                        ggv=GV;
                        bbv=BV;

                        dxv=0;
                        for (xcount=xPos+1 ; xcount<=xxPos ; xcount++)
                        {
                           xwGFXRenderRGB(xDist,dxv,rv,gv,bv,rrv,ggv,bbv);
                           dxv++;

                           if (rot==0) xwGFXSetPixel(outStream,(int) (xcount+x),(int) (yyPos+y),RV,GV,BV);
                           else if (rot==90) xwGFXSetPixel(outStream,(int) ((hh-yyPos)+x),(int) (xcount+y),RV,GV,BV);
                           else if (rot==-90) xwGFXSetPixel(outStream,(int) (yyPos+x),(int) ((ww-xcount)+y),RV,GV,BV);
                           else if ((rot==-180) || (rot==180)) xwGFXSetPixel(outStream,(int) ((ww-xcount)+x),(int) ((hh-yyPos)+y),RV,GV,BV);

                           /* Vertical Blending */
                           if (yyPos>0) /*>0)*/
                           {
                              if (rot==0) xwGFXGetPixel(outStream,(int) (xcount+x),(int) (yPos+y));
                              else if (rot==90) xwGFXGetPixel(outStream,(int) ((hh-yPos)+x),(int) (xcount+y));
                              else if (rot==-90) xwGFXGetPixel(outStream,(int) (yPos+x),(int) ((ww-xcount)+y));
                              else if ((rot==-180) || (rot==180)) xwGFXGetPixel(outStream,(int) ((ww-xcount)+x),(hh-yPos)+y);
                              rrrv=RV;
                              gggv=GV;
                              bbbv=BV;
                              if (rot==0) xwGFXGetPixel(outStream,(int) (xcount+x),(int) (yyPos+y));
                              else if (rot==90) xwGFXGetPixel(outStream,(int) ((hh-yyPos)+x),(int) (xcount+y));
                              else if (rot==-90) xwGFXGetPixel(outStream,(int) (yyPos+x),(int) ((ww-xcount)+y));
                              else if ((rot==-180) || (rot==180)) xwGFXGetPixel(outStream,(int) ((ww-xcount)+x),(int) ((hh-yyPos)+y));
                              rrrrv=RV;
                              ggggv=GV;
                              bbbbv=BV;

                              dyv=0;
                              for (ycount=yPos+1 ; ycount<=yyPos ; ycount++)
                              {
                                 xwGFXRenderRGB(yDist,dyv,rrrv,gggv,bbbv,rrrrv,ggggv,bbbbv);
                                 dyv++;

                                 if (rot==0) xwGFXSetPixel(outStream,(int) (xcount+x),(int) (ycount+y),RV,GV,BV);
                                 else if (rot==90) xwGFXSetPixel(outStream,(hh-ycount)+x,xcount+y,RV,GV,BV);
                                 else if (rot==-90) xwGFXSetPixel(outStream,ycount+x,(ww-xcount)+y,RV,GV,BV);
                                 else if ((rot==-180) || (rot==180)) xwGFXSetPixel(outStream,(ww-xcount)+x,(hh-ycount)+y,RV,GV,BV);

                              }
                           }
                        }

                        xPos=xxPos;
                     }
                  }

                  yPos=yyPos;
               }
            }

            /* border correction for repair corrupt borders */

            for (xcount=0 ; xcount<ww ; xcount++)
            {
               if (rot==0) xwGFXGetPixel(outStream,xcount+x,yborder+y);
               else if (rot==90) xwGFXGetPixel(outStream,(hh-yborder)+x,xcount+y);
               else if (rot==-90) xwGFXGetPixel(outStream,yborder+x,(ww-xcount)+y);
               else if ((rot==-180) || (rot==180)) xwGFXGetPixel(outStream,(ww-xcount)+x,(hh-yborder)+y);

               for (ycount=0 ; ycount<yborder ; ycount++)
               {
                  if (rot==0) xwGFXSetPixel(outStream,xcount+x,ycount+y,RV,GV,BV);
                  else if (rot==90) xwGFXSetPixel(outStream,(hh-ycount)+x,xcount+y,RV,GV,BV);
                  else if (rot==-90) xwGFXSetPixel(outStream,ycount+x,(ww-xcount)+y,RV,GV,BV);
                  else if ((rot==-180) || (rot==180)) xwGFXSetPixel(outStream,(ww-xcount)+x,(hh-ycount)+y,RV,GV,BV);
               }

               if (rot==0) xwGFXGetPixel(outStream,xcount+x,(hh-yborder)+y);
               else if (rot==90) xwGFXGetPixel(outStream,yborder+x,xcount+y);
               else if (rot==-90) xwGFXGetPixel(outStream,(hh-yborder)+x,(ww-xcount)+y);
               else if ((rot==-180) || (rot==180)) xwGFXGetPixel(outStream,(ww-xcount)+x,yborder+y);

               for (ycount=0 ; ycount<yborder ; ycount++)
               {
                  if (rot==0) xwGFXSetPixel(outStream,xcount+x,(hh-ycount)+y,RV,GV,BV);
                  else if (rot==90) xwGFXSetPixel(outStream,ycount+x,xcount+y,RV,GV,BV);
                  else if (rot==-90) xwGFXSetPixel(outStream,(hh-ycount)+x,(ww-xcount)+y,RV,GV,BV);
                  else if ((rot==-180) || (rot==180)) xwGFXSetPixel(outStream,(ww-xcount)+x,ycount+y,RV,GV,BV);
               }
            }
            for (ycount=0 ; ycount<hh ; ycount++)
            {
               if (rot==0) xwGFXGetPixel(outStream,xborder+x,ycount+y);
               else if (rot==90) xwGFXGetPixel(outStream,(hh-ycount)+x,xborder+y);
               else if (rot==-90) xwGFXGetPixel(outStream,ycount+x,(ww-xborder)+y);
               else if ((rot==-180) || (rot==180)) xwGFXGetPixel(outStream,(ww-xborder)+x,(hh-ycount)+y);

               for (xcount=0 ; xcount<xborder ; xcount++)
               {
                  if (rot==0) xwGFXSetPixel(outStream,xcount+x,ycount+y,RV,GV,BV);
                  else if (rot==90) xwGFXSetPixel(outStream,(hh-ycount)+x,xcount+y,RV,GV,BV);
                  else if (rot==-90) xwGFXSetPixel(outStream,ycount+x,(ww-xcount)+y,RV,GV,BV);
                  else if ((rot==-180) || (rot==180)) xwGFXSetPixel(outStream,(ww-xcount)+x,(hh-ycount)+y,RV,GV,BV);
               }

               if (rot==0) xwGFXGetPixel(outStream,(ww-xborder)+x,ycount+y);
               else if (rot==90) xwGFXGetPixel(outStream,ycount+x,(ww-xborder)+y);
               else if (rot==-90) xwGFXGetPixel(outStream,ycount+x,xborder+y);
               else if ((rot==-180) || (rot==180)) xwGFXGetPixel(outStream,xborder+x,ycount+y);

               for (xcount=0 ; xcount<xborder ; xcount++)
               {
                  if (rot==0) xwGFXSetPixel(outStream,(ww-xcount)+x,ycount+y,RV,GV,BV);
                  else if (rot==90) xwGFXSetPixel(outStream,ycount+x,(ww-xcount)+y,RV,GV,BV);
                  else if (rot==-90) xwGFXSetPixel(outStream,ycount+x,xcount+y,RV,GV,BV);
                  else if ((rot==-180) || (rot==180)) xwGFXSetPixel(outStream,xcount+x,ycount+y,RV,GV,BV);
               }
            }

         }

         if (bug==xwFalse)
         {
            if (out==createTemp) xwGFXTempToPicture();
            else if (out==createSTemp) xwGFXSTempToSPicture();
         }
         else
         {
            xwGFXRemovePicture();
         }
      }

      /* No Scalling */
      else if (xstep==1)
      {
         for (yloop=0 ; yloop<hh ; yloop++)
         {
            for (xloop=0 ; xloop<ww ; xloop++)
            {
               if (rot==0)
               {
                  xwGFXCopyPixel(inStream,xloop,yloop,outStream,xloop+x,yloop+y);
               }
               else if (rot==90)
               {
                   xwGFXCopyPixel(inStream,xloop,yloop,outStream,(hh-yloop)+x,xloop+y);
               }
               else if (rot==-90)
               {
                   xwGFXCopyPixel(inStream,xloop,yloop,outStream,yloop+y,(ww-xloop)+x);
               }
               else if ((rot==-180) || (rot==180))
               {
                   xwGFXCopyPixel(inStream,xloop,yloop,outStream,(ww-xloop)+x,(hh-yloop)+y);
               }
            }
         }

         if (bug==xwFalse)
         {
            if (out==createTemp) xwGFXTempToPicture();
            else if (out==createSTemp) xwGFXSTempToSPicture();
         }
         else
         {
            xwGFXRemovePicture();
         }
      }
   }
   return;
}

/* Set Draw Border Mode */
void xwGFXSetDrawBorderMode(int mo)
{
   gfxBRDType=mo;

   return;
}

/* Draw Border */
void xwGFXDrawBorder(int out,int bw,int r,int g,int b) /*fold00*/
{
   int            x,y;
   int            w,h;

   int            xx,yy,rr,bww;
   float          grad;
  
   w=-1;
   h=-1;

   /* Get Size of Target */
   if ((out==memPaper) && (gfxPaper!=NULL))
   {
      w=gfxXPaper;
      h=gfxYPaper;
   }
   else if ((out==memShared) && (gfxShared!=NULL))
   {
      w=gfxXShared;
      h=gfxYShared;
   }
   else if ((out==memTemp) && (gfxTemp!=NULL))
   {
      w=gfxXTemp;
      h=gfxYTemp;
   }
   else if ((out==memPicture) && (gfxPicture!=NULL))
   {
      w=gfxXPicture;
      h=gfxYPicture;
   }

   /* Draw Border */
   if ((w>-1) && (h>-1))
   {
      /* Quadratic Border */
      if (gfxBRDType==gfxQuad)
      {
         for (y=0 ; y<h ; y++)
         {
            for (x=0 ; x<w ; x++)
            {
               /* Top Border */
               if (y<bw)
               {
                  xwGFXSetPixel(out,x,y,r,g,b);
               }
               /* Left Border */
               else if ((x<bw) && (y>=bw) && (y<h-bw))
               {
                  xwGFXSetPixel(out,x,y,r,g,b);
               }
               /* Right Border */
               else if ((x>=w-bw) && (y>=bw) && (y<h-bw))
               {
                  xwGFXSetPixel(out,x,y,r,g,b);
               }
               /* Bottom Border */
               else if (y>=h-bw)
               {
                  xwGFXSetPixel(out,x,y,r,g,b);
               }
            }
         }
      }
      else if (gfxBRDType==gfxCircle)
      {
         /* Circle Border */
         xx=w/2;
         yy=h/2;
         rr=xx;
         if (xx>yy) rr=yy;
        
         /* Draw Circle Border */
         for (grad=0 ; grad<=360 ; grad=grad+0.005)
         {
            for (bww=rr ; bww>rr-bw ; bww--)
            {
               /*
               cc=(grad*(2*M_PI))/360;
               xwGFXSetPixel(out,xx+(int) (sin(cc)*(float) bww),yy+(int) (cos(cc)*(float) bww),r,g,b);
               */
               
               xwGFXSetPixel(out,xx+(int) (sin(grad)*(float) bww),yy+(int) (cos(grad)*(float) bww),r,g,b);
            }
         }        
      }
   }

   return;
}

/* Draw Border */
void xwGFXDrawBorderRGB(int out,int bw,int rgb) /*fold00*/
{
   int            r,g,b;
   int            rgbv;

   /* Calculate RGB-Value */
   rgbv=rgb;
   r=rgbv/65536;
   rgbv=rgbv-(r*65536);
   g=rgbv/256;
   rgbv=rgbv-(g*256);
   b=rgbv;

   xwGFXDrawBorder(out,bw,r,g,b);

   return;
}

/* Create Button */
void xwGFXCreateButton(int in)
{
   int               w,h;  /* width height */
   int               x,y;  /* cursor position */
   
   if (in==memPaper)
   {
      w=gfxXPaper;
      h=gfxYPaper;
   }
   else if (in==memTemp)
   {
      w=gfxXTemp;
      h=gfxYTemp;
   }
   else if (in==memPicture)
   {
      w=gfxXPicture;
      h=gfxYPicture;
   }
   
   /* exchange white color to button color */
   for (y=2 ; y<h-1 ; y++)
   {
      for (x=2 ; x<w-1 ; x++)
      {
         xwGFXGetPixel(in,x,y);
         if (RV==255) RV=204;
         if (GV==255) GV=204;
         if (BV==255) BV=204;
         xwGFXSetPixel(in,x,y,RV,GV,BV);
      }
   }
   
   /* Draw Black Border */
   for (x=0 ; x<=w ; x++)
   {
      xwGFXSetPixel(in,x,0,0,0,0);
      xwGFXSetPixel(in,x,h-1,0,0,0);
   }
   for (y=0 ; y<=h ; y++)
   {
      xwGFXSetPixel(in,0,y,0,0,0);
      xwGFXSetPixel(in,w-1,y,0,0,0);
   }
   
   /* Draw Internal Border */
   for (x=2 ; x<w-1 ; x++)
   {
      /*xwGFXSetPixel(in,x,1,89,89,89);*/
      xwGFXSetPixel(in,x,h-2,89,89,89);
   }
   for (y=2 ; y<h ; y++)
   {
      xwGFXSetPixel(in,w-2,y,0,0,0);   
   }
   
   return;
}

/* Draw Image */
void xwGFXDrawImage(int out,int type,int x, int y)
{
   int               xx,yy;
   int               ww,hh;
   
   char              astr[8192];
   char              bstr[8];
   
   int               r,g,b;
   
   bstr[3]='\0';
   
   if (type==gfxDrawer) {  ww=28;   hh=21; }
   else if (type==gfxParent) {  ww=20;  hh=24; }
   else if (type==gfxError) { ww=28; hh=28; }
   
   for (yy=0 ;  yy<hh ; yy++)
   {
      if (type==gfxDrawer)
      {     
         if (yy==  0) strcpy(astr,"204204204204204204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  1) strcpy(astr,"204204204255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  2) strcpy(astr,"204204204255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  3) strcpy(astr,"204204204255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  4) strcpy(astr,"204204204255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204204204255255204");
         else if (yy==  5) strcpy(astr,"255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204255255204204153051255255204");
         else if (yy==  6) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy==  7) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy==  8) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy==  9) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 10) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 11) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 12) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 13) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 14) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 15) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 16) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 17) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 18) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 19) strcpy(astr,"255255204255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051255204051204153051255255204");
         else if (yy== 20) strcpy(astr,"255255204204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051204153051255232058");
         else if (yy== 21) strcpy(astr,"255232058008168151058008113001000000132044173251000160062064000160062064000160062064000160062064191168062064000176062064000160062064000176062064000000000000000000000000000000000000000000000000032240056064004000000000216231062008240014063008000000000008008160060");  
      }
      else if (type==gfxParent)
      {   
         if (yy==  0) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204255255255204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  1) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204255255255000000000204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  2) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204255255255051051255000000000204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  3) strcpy(astr,"204204204204204204204204204204204204204204204204204204255255255051051255051051255000000000204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  4) strcpy(astr,"204204204204204204204204204204204204204204204255255255051051255051051255051051255000000000204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  5) strcpy(astr,"204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  6) strcpy(astr,"204204204204204204204204204255255255051051255051051255051051255051051255051051255000000000204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  7) strcpy(astr,"204204204204204204255255255051051255051051255051051255051051255051051255051051255000000000255255255255255255255255255255255255255255255255255255255255255255255255255255255255255255");
         else if (yy==  8) strcpy(astr,"204204204255255255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255000000000");
         else if (yy==  9) strcpy(astr,"255255255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255000000000");
         else if (yy== 10) strcpy(astr,"255255255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255000000000");
         else if (yy== 11) strcpy(astr,"204204204153153153051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255000000000");
         else if (yy== 12) strcpy(astr,"204204204204204204153153153051051255051051255051051255051051255051051255051051255000000000000000000000000000000000000000000000000000000051051255051051255051051255051051255000000000");
         else if (yy== 13) strcpy(astr,"204204204204204204204204204153153153051051255051051255051051255051051255051051255000000000204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 14) strcpy(astr,"204204204204204204204204204204204204153153153051051255051051255051051255051051255000000000204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 15) strcpy(astr,"204204204204204204204204204204204204204204204153153153051051255051051255051051255000000000204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 16) strcpy(astr,"204204204204204204204204204204204204204204204204204204153153153051051255051051255000000000204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 17) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204153153153051051255000000000204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 18) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204153153153000000000204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 19) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204153153153204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 20) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 21) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 22) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255255255051051255051051255051051255051051255000000000");
         else if (yy== 23) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255255255000000000000000000000000000000000000000000000");
      }
      else if (type==gfxError)
      {
         if (yy==  0) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204204153153255102102255051102255051051255051051255051051255051051255102102204102153204153204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  1) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204102153255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051102204153153204204204204204204204204204204204204204204204204204204204204204");
         else if (yy==  2) strcpy(astr,"204204204204204204204204204204204204204204204204153153255051102255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051102204204204204204204204204204204204204204204204204204204");
         else if (yy==  3) strcpy(astr,"204204204204204204204204204204204204204102153255051051255051051255051051255051051255051051255051051255051051255102102255102102255102102255051102255051051255051051255051051255051051255051051255051051255051102204153153204204204204204204204204204204204204");
         else if (yy==  4) strcpy(astr,"204204204204204204204204204204102153255051051255051051255051051255051051255051051204102153204153204204204204204204204204204204204204204204204204204204204204153204255051102255051051255051051255051051255051051255051051204153153204204204204204204204204204");
         else if (yy==  5) strcpy(astr,"204204204204204204204153153255051051255051051255051051255051051255051102204153204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204153153255051102255051051255051051255051051255051102204204204204204204204204204");
         else if (yy==  6) strcpy(astr,"204204204204204204255051051255051051255051051255051051255051102204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051102255051051255051051255051051255051102204204204204204204");
         else if (yy==  7) strcpy(astr,"204204204255102102255051051255051051255051051255051102204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051255051051255051051255051051204153153204204204");
         else if (yy==  8) strcpy(astr,"204204204255051051255051051255051051255051051204153204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204153153255051051255051051255051051255051102204204204");
         else if (yy==  9) strcpy(astr,"204102153255051051255051051255051051255051102204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204255051051255051051255051051255051051204153204");
         else if (yy== 10) strcpy(astr,"255051102255051051255051051255051051204153153204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204255102102255051051255051051255051051204153153");
         else if (yy== 11) strcpy(astr,"255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204153153255051051255051051255051051255102102");
         else if (yy== 12) strcpy(astr,"255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204153153255051051255051051255051051255102102");
         else if (yy== 13) strcpy(astr,"255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051102");
         else if (yy== 14) strcpy(astr,"255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204153153255051051255051051255051051255102102");
         else if (yy== 15) strcpy(astr,"255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204153153255051051255051051255051051255102102");
         else if (yy== 16) strcpy(astr,"255051051255051051255051051255051051204153204204204204204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204204102153255051051255051051255051051204102153");
         else if (yy== 17) strcpy(astr,"255102102255051051255051051255051051204102153204204204204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051102255051051255051051255051051204153153");
         else if (yy== 18) strcpy(astr,"204153153255051051255051051255051051255051102204204204204204204255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204153153255051051255051051255051051255051051204204204");
         else if (yy== 19) strcpy(astr,"204204204255051102255051051255051051255051051255102102255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051102255051051255051051255051051204102153204204204");
         else if (yy== 20) strcpy(astr,"204204204204153153255051051255051051255051051255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051102255051051255051051255051051255051102204204204204204204");
         else if (yy== 21) strcpy(astr,"204204204204204204255051102255051051255051051255051051255051051255051051255051051204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204255051102255051051255051051255051051255051051204153153204204204204204204");
         else if (yy== 22) strcpy(astr,"204204204204204204204204204255051102255051051255051051255051051255051051255051102204153204204204204204204204204204204204204204204204204204204204204204204204204204204153153255051102255051051255051051255051051255051051204102153204204204204204204204204204");
         else if (yy== 23) strcpy(astr,"204204204204204204204204204204204204255051102255051051255051051255051051255051051255051051255051102255102102204153153204153153204153153204153153255102102255051051255051051255051051255051051255051051255051051255102102204204204204204204204204204204204204");
         else if (yy== 24) strcpy(astr,"204204204204204204204204204204204204204204204255051102255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051204153153204204204204204204204204204204204204204204204");
         else if (yy== 25) strcpy(astr,"204204204204204204204204204204204204204204204204204204204153153255051102255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051051255051102204204204204204204204204204204204204204204204204204204204204204");
         else if (yy== 26) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204153153255102102255051051255051051255051051255051051255051051255051051255051051255051102204102153204153204204204204204204204204204204204204204204204204204204204204204204204204204");
         else if (yy== 27) strcpy(astr,"204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204153153204153153204153153204153153204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204204");
      }
         
      for (xx=0 ; xx<ww ; xx++)
      {
         bstr[0]=astr[(xx*9)];
         bstr[1]=astr[(xx*9)+1];
         bstr[2]=astr[(xx*9)+2];
         r=atoi(bstr);
         bstr[0]=astr[(xx*9)+3];
         bstr[1]=astr[(xx*9)+4];
         bstr[2]=astr[(xx*9)+5];
         g=atoi(bstr);
         bstr[0]=astr[(xx*9)+6];
         bstr[1]=astr[(xx*9)+7];
         bstr[2]=astr[(xx*9)+8];
         b=atoi(bstr);
         
         xwGFXSetPixel(out,x+xx,y+yy,r,g,b);   
      
      }
   }

   return;
}

/* RePaint Paper with Defined Color */
void xwGFXPaintPicture(int out,int r,int g,int b)
{
   int               zeiger;
   int               rgb=0;
   
   if (out==memPaper)
   {   
      if (gfxPaper!=NULL)
      {
         for (zeiger=0 ; zeiger<=gfxPaperSize ; zeiger++)
         {
            rgb++;
            if (rgb==1)
            {
               gfxPaper[zeiger]=r;
            }
            else if (rgb==2)
            {
               gfxPaper[zeiger]=g;
            }
            else if (rgb==3)
            {
               gfxPaper[zeiger]=b;
               rgb=0;
            }
         }   
      }
   }
   else if (out==memTemp)
   {   
      if (gfxTemp!=NULL)
      {
         for (zeiger=0 ; zeiger<=gfxTempSize ; zeiger++)
         {
            rgb++;
            if (rgb==1)
            {
               gfxTemp[zeiger]=r;
            }
            else if (rgb==2)
            {
               gfxTemp[zeiger]=g;
            }
            else if (rgb==3)
            {
               gfxTemp[zeiger]=b;
               rgb=0;
            }
         }   
      }
   }
   else if (out==memPicture)
   {   
      if (gfxPicture!=NULL)
      {
         for (zeiger=0 ; zeiger<=gfxPictureSize ; zeiger++)
         {
            rgb++;
            if (rgb==1)
            {
               gfxPicture[zeiger]=r;
            }
            else if (rgb==2)
            {
               gfxPicture[zeiger]=g;
            }
            else if (rgb==3)
            {
               gfxPicture[zeiger]=b;
               rgb=0;
            }
         }   
      }
   }
   return;
}

/* ################################################################### */
/* Picture to XPM */

/* Generate XPM-Data - Result PIXMAP */
Pixmap xwGFXCreatePixmap(int in) /*fold00*/
{
   Pixmap         pixmap=-1;
   char           **xpmData=NULL;
   int            result=FALSE;

   int            az;

   setRenderQuality(xwGetPrefValue(xwgImgQuality));

   /* Get Size of Target */
   if ((in==memPaper) && (gfxPaper!=NULL))
   {
      if (gfxPaperLP==0)
      {
         result=convert("","",gfxPaper,gfxXPaper,gfxYPaper,&xpmData);
      }
      else
      {
         result=convert("","",gfxPaper,gfxYPaper,gfxXPaper,&xpmData);
      }
   }
   else if ((in==memShared) && (gfxShared!=NULL))
   {
      result=convert("","",gfxShared->mem,gfxXShared,gfxYShared,&xpmData);
   }
   else if ((in==memTemp) && (gfxTemp!=NULL))
   {
      result=convert("","",gfxTemp,gfxXTemp,gfxYTemp,&xpmData);
   }
   else if ((in==memPicture) && (gfxPicture!=NULL))
   {
      result=convert("","",gfxPicture,gfxXPicture,gfxYPicture,&xpmData);
   }

   /* XPM-Datas -> Pixmap */
   if (result==TRUE)
   {
      pixmap=fl_create_from_pixmapdata(fl_default_window(),xpmData,&az,&az,NULL,&az,&az,FL_WHITE);
      if (pixmap!=0)
      {
         freeXpmData(&xpmData);
      }
      else
      {
         freeXpmData(&xpmData);
         return(-1);
      }
   }

   return(pixmap);
}
 /*fold00*/































































































































/* ################################################################### */
/* Preview */
void xwGFXPreview(float dpi) /*fold00*/
{
   char           message[1024];

   FL_FORM        *form;
   FL_OBJECT      *border;
   FL_OBJECT      *xpm;
   FL_OBJECT      *info;
   FL_OBJECT      *ok;

   FILE           *fh;

   int            xres,yres;
   int            xsize,ysize;

   int            xppos,yppos,xpsize,ypsize;

   char           source[1024];
   char           astr[1024];

      /* Landscape */
      if (xwIsPaperLS()==xwLandscape)
      {
         xres=(int) ( (dpi/25.4)* (double) (xwGetPaperXSize()/xwMM()) );
         yres=(int) ( (dpi/25.4)* (double) (xwGetPaperYSize()/xwMM()) );
         xsize=yres;
         ysize=xres;
      }
      else
      {
         xres=(int) ( (dpi/25.4)* (double) (xwGetPaperXSize()/xwMM()) );
         yres=(int) ( (dpi/25.4)* (double) (xwGetPaperYSize()/xwMM()) );
         xsize=xres;
         ysize=yres;
      }

      form = fl_bgn_form(FL_UP_BOX, xsize,ysize+25);
      fl_set_border_width(1);

      border = fl_add_box(FL_SHADOW_BOX,0,0,xsize,ysize,"");

      /* Landscape */
      if (xwIsPaperLS()==xwLandscape)
      {
         xppos=(int) ( (dpi/25.4)* (double) (xwGetPaperBB()/xwMM()) );
         yppos=(int) ( (dpi/25.4)* (double) (xwGetPaperLB()/xwMM()) );
         xpsize=(int) ( (dpi/25.4)* (double) (xwGetPaperYB()/xwMM()) );
         ypsize=(int) ( (dpi/25.4)* (double) (xwGetPaperXB()/xwMM()) );

         xpm = fl_add_pixmap(FL_NORMAL_BITMAP,xppos,yppos,xpsize,ypsize,"");
      }
      else
      {
         xppos=(int) ( (dpi/25.4)* (double) (xwGetPaperLB()/xwMM()) );
         yppos=(int) ( (dpi/25.4)* (double) (xwGetPaperTB()/xwMM()) );
         xpsize=(int) ( (dpi/25.4)* (double) (xwGetPaperXB()/xwMM()) );
         ypsize=(int) ( (dpi/25.4)* (double) (xwGetPaperYB()/xwMM()) );

         xpm = fl_add_pixmap(FL_NORMAL_BITMAP,xppos,yppos,xpsize,ypsize,"");
      }
      fl_hide_object(xpm);

      info = fl_add_box(FL_UP_BOX,xppos+5,yppos+5,xpsize-10,ypsize-10,xwGR(1,"gfx.LoadImage","Image is in progress..."));

      ok = fl_add_button(FL_NORMAL_BUTTON,0,ysize+5,xsize,20,xwGR(1,"gfx.Ok","OK"));

      fl_deactivate_object(ok);

      fl_end_form();

      if (xwGetPrefValue(xwgWinPos)==0)
      {
         fl_prepare_form_window(form,FL_PLACE_MOUSE,FL_TRANSIENT,xwGR(1,"gfx.Preview","Preview"));
      }
      else
      {
         fl_prepare_form_window(form,FL_PLACE_CENTER,FL_TRANSIENT,xwGR(1,"gfx.Preview","Preview"));
      }

      fl_set_form_dblbuffer(form,0);
      fl_show_form_window(form);

      fl_set_app_mainform(form);

      /* Exit Handler */
      fl_set_form_atclose(form,nclose_cb,(void *) "1");

      strcpy(source,xwGetFile("","prtout.ppm"));
      strcpy(astr,xwGetFile("","preview.xpm"));

      xwGFXXPM(source,astr);
      unlink(source);

      if ( (fh=fopen(astr,"rb")) != NULL)
      {
         fclose(fh);

         fl_hide_object(info);
         fl_show_object(xpm);

         fl_set_pixmap_file(xpm,xwGetFile("","preview.xpm"));
      }
      else
      {
         strcpy(message,"xwGUI: preview image not available !!!");
         xwDisplayMessage(NULL,message);
      }

      fl_activate_object(ok);

      while (fl_do_forms()==NULL);

      fl_hide_form(form);

      fl_free_pixmapbutton_pixmap(xpm);

      fl_free_form(form);

      fl_set_app_mainform(xwGetMainGui());


   return;
}
 /*fold00*/
/* ############################################# */
/* Convert PPM to PNG */
void xwGFXPPMTOPNG(const char *source,const char *dest) /*fold00*/
{
   char           message[1024];

   char           ssource[1024];
   char           ddest[1024];

   int            bug;

   bug=0;

   strcpy(ssource,source);
   strcpy(ddest,dest);
   if (xwGFXPNG(ssource,ddest)<0)
   {
      strcpy(message,"xwGUI: converting image failed !!!");
      xwDisplayMessage(NULL,message);
      unlink(ddest);
      bug=1;
   }

   return;
}

/* #################################################################################################
   Special GFX Routines
   ################################################################################################# */

/* Import Picture */
int xwGFXImport(const char *source, const char *dest) /*fold00*/
{
   char           ssource[1024];
   char           ddest[1024];
   char           com[2048];

   unsigned int   size;
   FILE           *Rfh,*Wfh;
   unsigned char  buf[120000];

   if (checkppm(source)==0)
   {
      /* copy cache file */

      if ( (Rfh=fopen(source,"r")) != NULL )
      {
         if ( (Wfh=fopen(dest,"w")) != NULL )
         {
            signal(SIGPIPE,sig_handler);
            while ( (size=fread(buf,1,sizeof(buf),Rfh)) > 0)
            {
               fwrite(buf,1,size,Wfh);
            }
            fclose(Wfh);
         }
         fclose(Rfh);
      }

      return(xwTrue);
   }

   strcpy(ssource,source);
   strcpy(ddest,dest);

   if (xwGetPrefValue(xwgConvert)==1)
   {
      /* nconvert */
      strcpy(com,xwGR(0,"gfxNCImport","nconvert -quiet -out pnm -truecolors -o[OUT] [IN]"));
      return(xwGFXExec(com,0,0,0,0,0,ssource,ddest,""));
   }
   else
   {
      /* ImageMagic */
      strcpy(com,xwGR(0,"gfxImport","convert [IN] [FORMAT][OUT]"));
      return(xwGFXExec(com,0,0,0,0,0,ssource,ddest,xwGR(0,"gfxPPMFORMAT","PNM:")));
   }
}

/* Convert To PNG */
int xwGFXPNG(const char *source,const char *dest) /*fold00*/
{
   char           ssource[1024];
   char           ddest[1024];
   char           com[2048];

   strcpy(ssource,source);
   strcpy(ddest,dest);

   if (xwGetPrefValue(xwgConvert)==1)
   {
      /* nconvert */
      strcpy(com,xwGR(0,"gfxNCPNG","nconvert -quiet -out png -truecolors -o[OUT] [IN]"));
      return(xwGFXExec(com,0,0,0,0,0,ssource,ddest,""));
   }
   else
   {
      /* ImageMagic */
      strcpy(com,xwGR(0,"gfxPNG","convert [IN] [FORMAT][OUT]"));
      return(xwGFXExec(com,0,0,0,0,0,ssource,ddest,xwGR(0,"gfxPNGFORMAT","PNG:")));
   }
}

/* Convert To XPM */
int xwGFXXPM(const char *source,const char *dest) /*fold00*/
{
   char           ssource[1024];
   char           ddest[1024];
   FILE           *rfh;
   FILE           *wfh;
   int            flag;

   strcpy(ssource,source);
   strcpy(ddest,dest);

   /* Check Setting File */
   flag=0;
   rfh=fopen(ssource,"r");
   if (rfh!=NULL)
   {
      wfh=fopen(ddest,"w");
      if (wfh!=NULL)
      {
         setRenderQuality(xwGetPrefValue(xwgImgQuality));
         flag=convert(rfh,wfh,NULL,0,0,NULL);
         fclose(wfh);
      }
      fclose(rfh);
   }

   if ( flag==1)
      return(0);
   else
      return(-1);
}

/* Insert Variable in Convert String */
int xwGFXExec(const char *command,int xpos,int ypos, int xsize,int ysize,int rot,const char *source,const char *dest,const char *format) /*fold00*/
{
   char           cmdS[1024];
   char           cmdD[1024];
   char           buffer[1024];
   char           foStr[1024];

   const char     *strptr;

   char           value[12];

   /* Exec Datas */
   static long    pid;
   static char    args[100][80]; /* Arg Array */
   static char    *argv[100];
   int            prtA,prtB;
   int            status;

   int            bug;

   strcpy(cmdS,command);
   strcpy(cmdD,"");

   strcpy(foStr,"");

   /* create arglist for executation */
   prtA=0;
   prtB=0;
   strptr = strtok(cmdS,gfxTZZ);
   while (strptr != NULL)
   {
      strcpy(buffer,strptr);


      if (strcmp(buffer,"IN") == 0)
      {
         strcpy(buffer,source);
      }
      else if (strcmp(buffer,"FORMAT") == 0)
      {
         strcpy(buffer,"");
         strcpy(foStr,format);
      }
      else if (strcmp(buffer,"OUT") == 0)
      {
         strcpy(buffer,foStr);
         strcat(buffer,dest);
      }
      else if (strcmp(buffer,"XPOS") == 0)
      {
         sprintf(value,"%d",xpos);
         strcpy(buffer,value);
      }
      else if (strcmp(buffer,"YPOS") == 0)
      {
         sprintf(value,"%d",ypos);
         strcpy(buffer,value);
      }
      else if (strcmp(buffer,"WIDTH") == 0)
      {
         sprintf(value,"%d",xsize);
         strcpy(buffer,value);
      }
      else if (strcmp(buffer,"HEIGHT") == 0)
      {
         sprintf(value,"%d",ysize);
         strcpy(buffer,value);
      }
      else if (strcmp(buffer,"ROTATE") == 0)
      {
         sprintf(value,"%d",rot);
         strcpy(buffer,value);
      }

      if (strlen(buffer)>0)
      {
         strcpy(args[prtA],buffer);

         if (prtA>-1)
         {
            argv[prtB] = args[prtA];
            prtB++;
            argv[prtB]=NULL;
         }
         prtA++;
      }

      strptr = strtok(NULL,gfxTZZ);
   }

   pid=0;
   bug=0;
   /* create Process */
   if ( (pid=fork()) == 0)
   {
      bug=execvp(args[0],argv);
      exit(0);
   }

   while (waitpid(pid,&status,WNOHANG)!=-1)
   {
      usleep(100000);

      /* Abort Process */
      if (xwCheckProgressA()==0) kill(pid,SIGTERM);
   }

   if (bug!=0) return(xwFalse);

   return(xwTrue);
}
