/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "frame.h"

#include "base.h"
#include "default.h"
#include "limits.h"
#include "lowlevel.h"
#include "main_gui.h"
#include "paper.h"
#include "picture.h"
#include "x11.h"

#include <stdio.h>

extern struct mainform     mf;

static framedatas          *fd[frameMaxFrames+1];     /* Frame Datas */
static int                 frameAnz=-1;               /* Frame Count */
static int                 frameActive=-1;            /* Actual Selected Frame */
static int                 frameMode=0;               /* Picture (0) -/Frame (1) -Mode */
static int                 frameEdit=0;               /* FrameEditor 0 - Off / 1 - On */
static int                 frameID=1;                 /* Actual Frame Number for FrameName */

/* ################################################################### */
/* Frames Managment */

/* Init Frame Settings */
void xwInitFrames(void) /*fold00*/
{
   int         az;

   /* Deactivate Frames */
   frameMode=0;
   /* Edit Frames */
   frameEdit=0;

   /* Count of Frames */
   frameAnz=-1;
   /* Actual Frame */
   frameActive=-1;

   /* Frame ID */
   frameID=-1;

   for (az=0 ; az<frameMaxFrames ; az++)
   {
      fd[az]=NULL;
   }

   return;
}

/* Free Frame Settings */
void xwFreeFrames(int fNr) /*fold00*/
{
   int         az;

   /* Free all Frames */
   if (fNr==-1)
   {
      if (frameAnz>-1)
      {
         for (az=0 ; az<=frameAnz ; az++)
         {
            if (fd[az]!=NULL)
            {
               /* Free Pixmap */
               if (fd[az]->pFlag!=0)
               {
                  fd[az]->pFlag=0;
                  XFreePixmap(fl_display,fd[az]->pPixmap);
               }

               /* Free Text */
               if (fd[az]->fontText!=NULL)
               {
                  free(fd[az]->fontText);
               }

               free(fd[az]);
               fd[az]=NULL;
            }
         }
         frameAnz=-1;
      }
   }
   else
   {
      az=fNr;

      if (fd[az]!=NULL)
      {
         /* Free Pixmap */
         if (fd[az]->pFlag!=0)
         {
            fd[az]->pFlag=0;
            XFreePixmap(fl_display,fd[az]->pPixmap);
         }

         /* Free Text */
         if (fd[az]->fontText!=NULL)
         {
            free(fd[az]->fontText);
         }

         free(fd[az]);
         for (az=fNr ; az<frameAnz ; az++)
         {
            fd[az]=fd[az+1];
         }
         fd[frameAnz]=NULL;
         frameAnz--;

         if (frameAnz<frameActive) frameActive=frameAnz;
      }
   }

   return;
}

/* Swap Frame */
void xwSwapFrame(int fNrA,int fNrB) /*fold00*/
{
   fd[frameMaxFrames]=fd[fNrA];
   fd[fNrA]=fd[fNrB];
   fd[fNrB]=fd[frameMaxFrames];

   return;
}

/* Check Edit Frame */
int xwCheckEditFrame(void) /*fold00*/
{
   /* Frames Editmode active */
   if (frameEdit==0) return(xwFalse);
   return(xwTrue);
}

/* Check Frame Mode */
int xwCheckFrameMode() /*fold00*/
{
   if (frameMode==0) return(xwFalse);
   return(xwTrue);
}

/* ################################################################### */
/* Pixmap Verwaltung */

/* Set Pixmap */
void xwSetFramePixmap(int fNr,Pixmap pixmap) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      if (fd[fNr]->pFlag!=0)
      {
         XFreePixmap(fl_display,fd[fNr]->pPixmap);
         fd[fNr]->pFlag=0;
      }
      if ((int) pixmap>-1)
      {
         fd[fNr]->pPixmap=pixmap;
         fd[fNr]->pFlag=1;
      }
   }

   return;
}

/* Clear Pixmap */
void xwClearFramePixmap(int fNr) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      if (fd[fNr]->pFlag!=0)
      {
         XFreePixmap(fl_display,fd[fNr]->pPixmap);
         fd[fNr]->pFlag=0;
      }
   }

   return;
}

/* Get Pixmap */
Pixmap xwGetFramePixmap(int fNr) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      if (fd[fNr]->pFlag>0)
      {
         return(fd[fNr]->pPixmap);
      }
   }

   return(-1);
}

/* Get Pixmap Datas */
void xwGetFramePMDatas(int fNr,int *w,int *h) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      *w=fd[fNr]->pXSize;
      *h=fd[fNr]->pYSize;
   }

   return;
}

/* Set Pixmap Datas */
void xwSetFramePMDatas(int fNr,int w,int h) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->pXSize=w;
      fd[fNr]->pYSize=h;
   }

   return;
}

/* ################################################################### */

/* Check and Load Picture */
int xwFPLoad(const char *filename) /*fold00*/
{
   /* Normal Picture Load */
   if (frameMode==0)
   {
      return(xwPLoad(filename,xwGetPaperLB(),xwGetPaperTB(),xwGetPaperXB(),xwGetPaperYB()));
   }
   /* Frames Picture Load */
   else if (frameMode==1)
   {
      /* search first free frame */
      if (fd[frameActive]!=NULL)
      {
         if (fd[frameActive]->picNr>-1)
         {
            xwReorgPic(frameActive);
            xwFreePicture(fd[frameActive]->picNr);
            fd[frameActive]->picNr=-1;
         }

         if (fd[frameActive]->picNr==-1)
         {
            fd[frameActive]->picNr=xwPLoad(filename,fd[frameActive]->xpos,fd[frameActive]->ypos,fd[frameActive]->xsize,fd[frameActive]->ysize);
                        
            return(fd[frameActive]->picNr);
         }
      }
   }

   return(xwFalse);
}

/* Draw Frames */
void xwDrawFrame(void) /*fold00*/
{
   int         az;

   /* Normal Picture Mode */
   if (frameMode==0)
   {
      xwDrawPicture(1234);
   }

   /* Frame Picture Mode */
   else if (frameMode==1)
   {
      for (az=0 ; az<=frameAnz ; az++)
      {
         /* Draw Frame */
         if (frameActive==az)
         {
            xwDraw(drawTFrame,fd[az]->xpos,fd[az]->ypos,fd[az]->xsize,fd[az]->ysize);
         }
         else
         {
            xwDraw(drawFrame,fd[az]->xpos,fd[az]->ypos,fd[az]->xsize,fd[az]->ysize);
         }

         if (fd[az]->picNr>-1)
         {
            /* Frame Edit Mode */
            if (xwCheckEditFrame()==0)
            {
               /* Scale Picture To Frame */
               xwScalePicture(fd[az]->picNr,fd[az]->xpos,fd[az]->ypos,fd[az]->xsize,fd[az]->ysize);
            }

            /* Draw Picture */
            if (xwGetPrefValue(xwgWYSIWYG)==0) xwDrawPicture(fd[az]->picNr);
         }
      }
   }

   return;
}

/* Print any Picture Information */
void xwPrintFrameDatas(int mode,int xsize,int ysize) /*fold00*/
{
   char           astr[40];

   /* Normal Picture Mode */
   if (xwCheckEditFrame()==-1)
   {
      /* Portrait */
      xwPrintPictureDatas(xwGetActivePicture(),mode,xsize,ysize);
   }

   /* Frame Edit Mode */
   if (xwCheckEditFrame()==0)
   {
      /* Landscape Move */
      if (xwIsPaperLS()==xwLandscape)
      {
         /* Left */
         sprintf(astr,"%.1f",(float) (xwGetPaperYSize()-(fd[frameActive]->ypos+fd[frameActive]->ysize))/xwMM());
         xwGeoPrint(geoLeft,astr);

         /* Right */
         sprintf(astr,"%.1f",(float) fd[frameActive]->ypos/xwMM());
         xwGeoPrint(geoRight,astr);

         /* Top */
         sprintf(astr,"%.1f",(float) fd[frameActive]->xpos/xwMM());
         xwGeoPrint(geoTop,astr);

         /* Bottom */
         sprintf(astr,"%.1f",(float) (xwGetPaperXSize()-(fd[frameActive]->xpos+fd[frameActive]->xsize))/xwMM());
         xwGeoPrint(geoBottom,astr);

         /* Width */
         sprintf(astr,"%.1f",(float) fd[frameActive]->ysize/xwMM());
         xwGeoPrint(geoWidth,astr);

         /* Height */
         sprintf(astr,"%.1f",(float) fd[frameActive]->xsize/xwMM());
         xwGeoPrint(geoHeight,astr);

         xwGeoPrint(geoDPI,"");
      }
      else
      {
         /* Left */
         sprintf(astr,"%.1f",(float) fd[frameActive]->xpos/xwMM());
         xwGeoPrint(geoLeft,astr);

         /* Right */
         sprintf(astr,"%.1f",(float) (xwGetPaperXSize()-(fd[frameActive]->xpos+fd[frameActive]->xsize))/xwMM());
         xwGeoPrint(geoRight,astr);

         /* Top */
         sprintf(astr,"%.1f",(float) fd[frameActive]->ypos/xwMM());
         xwGeoPrint(geoTop,astr);

         /* Bottom */
         sprintf(astr,"%.1f",(float) (xwGetPaperYSize()-(fd[frameActive]->ypos+fd[frameActive]->ysize))/xwMM());
         xwGeoPrint(geoBottom,astr);

         /* Width */
         sprintf(astr,"%.1f",(float) fd[frameActive]->xsize/xwMM());
         xwGeoPrint(geoWidth,astr);

         /* Height */
         sprintf(astr,"%.1f",(float) fd[frameActive]->ysize/xwMM());
         xwGeoPrint(geoHeight,astr);

         xwGeoPrint(geoDPI,"");
      }
   }

   return;
}

/* Create Frame Choice */
void xwCreateFrameChoice(FL_OBJECT *gd) /*fold00*/
{
   int            az;
   char           astr[1024],bstr[1024];
   char           *strptr;

   /* Normal Picture Mode */
   if (xwCheckFrameMode()==-1)
   {
      xwCreatePictureChoice(gd);
   }

   /* Frame Edit Mode */
   if (xwCheckFrameMode()==0)
   {
      if (gd!=NULL)
      {
         fl_clear_browser(gd);

         if (frameAnz>-1)
         {
            for (az=0 ; az<=frameAnz ; az++)
            {
               strcpy(astr,fd[az]->name);
               strcat(astr," - ");

               if (fd[az]->picNr>-1)
               {
                  strcpy(bstr,xwGetPictureFile(fd[az]->picNr));
               }
               else
               {
                  strcpy(bstr,"-----");
               }

               while ( fl_get_string_width(FL_NORMAL_STYLE,FL_NORMAL_SIZE,astr,strlen(astr))+fl_get_string_width(FL_NORMAL_STYLE,FL_NORMAL_SIZE,bstr,strlen(bstr)) >= gd->w-10)
               {
                  strptr=&bstr[1];
                  strcpy(bstr,strptr);
               }

               strcat(astr,bstr);
               fl_insert_browser_line(gd,1,astr);
            }
            fl_select_browser_line(gd,(frameAnz+1)-frameActive);
         }
      }
      else
      {
         if (frameAnz>-1)
         {
            strcpy(astr,fd[frameActive]->name);
            strcat(astr," - ");

            if (fd[frameActive]->picNr>-1)
            {
               strcpy(bstr,xwGetPictureFile(fd[frameActive]->picNr));
            }
            else
            {
               strcpy(bstr,"-----");
            }

            while ( fl_get_string_width(FL_NORMAL_STYLE,FL_NORMAL_SIZE,astr,strlen(astr))+fl_get_string_width(FL_NORMAL_STYLE,FL_NORMAL_SIZE,bstr,strlen(bstr)) >= mf.SelectFP->w-10)
            {
               strptr=&bstr[1];
               strcpy(bstr,strptr);
            }

            strcat(astr,bstr);
            fl_set_object_label(mf.SelectFP,astr);
         }
      }
   }

   /* Normal Picture Mode */
   if (xwCheckEditFrame()==-1)
   {
      fl_set_button(mf.FPSwitch,0);
   }

   /* Frame Edit Mode */
   if (xwCheckEditFrame()==0)
   {
      fl_set_button(mf.FPSwitch,1);
   }
   return;
}

/*
   Get Frame Datas
*/

/* Get Active Frame */
int xwGetActiveFrame(void) /*fold00*/
{
   return(frameActive);
}

/* Get Count of Frame */
int xwGetFrameCount(void) /*fold00*/
{
   return(frameAnz);
}

/* Get Active Frame XPos */
int xwGetFrameX(int fNr) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->xpos);
   }
   return(-1);
}

/* Get Frame YPos */
int xwGetFrameY(int fNr) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->ypos);
   }
   return(-1);
}

/* Get Frame XSize */
int xwGetFrameW(int fNr) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->xsize);
   }
   return(-1);
}

/* Get Frame YSize */
int xwGetFrameH(int fNr) /*fold00*/
{
   /* exist frames */
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->ysize);
   }
   return(-1);
}

/* Get Select Frame Number */
int xwGetSelectFrame(int xpos,int ypos) /*fold00*/
{
   int         fNr;
   int         az;

   fNr=-1;

   if (frameAnz>-1)
   {
      for (az=frameAnz ; az>=0 ; az--)
      {
         if (fNr==-1)
         {
            /* xgeomertry */
            if ( (xpos>=fd[az]->xpos) && (xpos<=fd[az]->xpos+fd[az]->xsize) )
            {
               /* ygeomertry */
               if ( (ypos>=fd[az]->ypos) && (ypos<=fd[az]->ypos+fd[az]->ysize) )
               {
                  fNr=az;
               }
            }
         }
      }
   }

   return(fNr);
}

/* Get Frame Picture */
int xwGetFramePicture(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->picNr);
   }

   return(-1);
}

/*
   Set Frame Datas
*/

/* Set Active Frame Nr. */
void xwSetActiveFrame(int fNr) /*fold00*/
{
   /* Check Range of Pictures */
   if ((fNr>-1) && (fNr<=frameAnz))
   {
      frameActive=fNr;
      xwSetActivePicture(fd[frameActive]->picNr);
   }

   return;
}

/* Set Active Frame with Picture Nr */
void xwSetActivePicFrame(int fNr) /*fold00*/
{
   int            az;

   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      /* Search Picture */
      for (az=0 ; az<=frameAnz ; az++)
      {
         if (fd[az]->picNr==fNr)
         {
            frameActive=az;
            xwSetActivePicture(fd[frameActive]->picNr);
         }
      }
   }

   return;
}

/* Set Frame Geometry */
void xwSetFrameGeo(int fNr,int xpos,int ypos,int xsize,int ysize) /*fold00*/
{
  if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->xpos=xpos;
      fd[fNr]->ypos=ypos;
      fd[fNr]->xsize=xsize;
      fd[fNr]->ysize=ysize;
   }

   return;
}

/* Activate Frame Editor */
void xwActivateFrameEd(void) /*fold00*/
{
   frameEdit=1;

   return;
}

/* DeActivate Frame Editor */
void xwDeActivateFrameEd(void) /*fold00*/
{
   frameEdit=0;

   return;
}


/* ################################################################### */
/* Border Datas */

/* Get Border Size */
double xwGetFrameB(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->borderSize);

   return(0);
}

/* Get Border Color */
int xwGetFrameBC(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->borderColor);
   
   return(0);
}

/* Get Frame Size */
double xwGetFrameF(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->frameSize);

   return(0);
}

/* Get Border Color */
int xwGetFrameC(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->color);

   return(0);
}

/* Get Border Type */
int xwGetFrameBT(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->borderType);

   return(0);
}

/* Get Picture Border Size */
double xwGetFramePB(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->pborderSize);

   return(0);
}

/* Get Picture Border Color */
int xwGetFramePBC(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->pborderColor);

   return(0);
}

/* Get Picture Frame Size */
double xwGetFramePF(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->pframeSize);

   return(0);
}

/* Get Picture Frame Color */
int xwGetFramePC(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->pframeColor);

   return(0);
}

/* Get Picture Transparency */
int xwGetFramePT(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->ptransparenz);

   return(0);
}

/* Get Picture Transparency Mode */
int xwGetFramePTM(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->ptransMode);

   return(0);
}


/* Set Border Size */
void xwSetFrameB(int fNr,double size) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->borderSize=size;
   }

   return;
}

/* Set Border Color */
void xwSetFrameBC(int fNr,int mo)
{
   if ((frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->borderColor=mo;
   }
   
   return;
}

/* Set Frame Size */
void xwSetFrameF(int fNr,double size) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->frameSize=size;
   }

   return;
}

/* Set Border Color */
void xwSetFrameC(int fNr,int rgb) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->color=rgb;
   }

   return;
}

/* Set Border Type */
void xwSetFrameBT(int fNr,int mo) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->borderType=mo;
   }

   return;
}

/* Set Picture Border Size */
void xwSetFramePB(int fNr,double size) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->pborderSize=size;
   }

   return;
}

/* Set Picture Border Color */
void xwSetFramePBC(int fNr,int mo)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->pborderColor=mo;
   }
   
   return;
}

/* Set Picture Frame Size */
void xwSetFramePF(int fNr,double size) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->pframeSize=size;
   }

   return;
}

/* Set Picture Border Color */
void xwSetFramePC(int fNr,int rgb) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->pframeColor=rgb;
   }

   return;
}

/* Set Picture Transparency */
void xwSetFramePT(int fNr,int trans) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->ptransparenz=trans;
   }

   return;
}

/* Set Transparency Mode */
void xwSetFramePTM(int fNr,int mo) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->ptransMode=mo;
   }

   return;
}

/* ################################################################### */
/* Horizontal & Vertical Lines */

/* Get Horizontal Line */
double xwGetHLine(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->hline);

   return(0);
}

/* Get Horizontal Line Color */
int xwGetHColor(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->hcolor);

   return(0);
}

/* Get Vertical Line */
double xwGetVLine(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->vline);

   return(0);
}

/* Get Vertical Line Color */
int xwGetVColor(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->vcolor);

   return(0);
}

/* Set Horizontal Line */
void xwSetHLine(int fNr,double size) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->hline=size;
   }

   return;
}

/* Set Horizontal Line Color */
void xwSetHColor(int fNr,int rgb) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->hcolor=rgb;
   }

   return;
}

/* Set Vertical Line */
void xwSetVLine(int fNr,double size) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->vline=size;
   }

   return;
}

/* Set Vertical Line Color */
void xwSetVColor(int fNr,int rgb) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->vcolor=rgb;
   }

   return;
}

/* ################################################################### */
/* Picture Position */

/* Get Picture Position */
int xwGetPicPosition(int fNr) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) ) return(fd[fNr]->pos);

   return(0);
}

/* Set Picture Position */
void xwSetPicPosition(int fNr,int pos) /*fold00*/
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->pos=pos;
   }

   return;
}


/* ################################################################### */
/* Text & Font Routines */

/* Get Fontname */
char *xwGetFontName(int fNr)
{
   static char       astr[1024];

   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      strcpy(astr,fd[fNr]->fontName);
   }
   else
   {
      strcpy(astr,"");
   }

   return(astr);
}

/* Set Fontname */
void xwSetFontName(int fNr,char *fontname)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      strcpy(fd[fNr]->fontName,fontname);
   }

   return;
}

/* Get Fontsize */
int xwGetFontSize(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->fontSize);
   }

   return(0);
}

/* Set Fontsize */
void xwSetFontSize(int fNr,int size)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->fontSize=size;
   }

   return;
}

/* Get Fontline ( 0 - None / 1 - Underline ) */
int xwGetFontLine(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->fontLine);
   }

   return(0);
}

/* Set Fontline ( 0 - None / 1 - Underline ) */
void xwSetFontLine(int fNr,int line)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->fontLine=line;
   }

   return;
}

/* Get Font-Direction */
int xwGetFontDirection(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->fontDirection);
   }

   return(0);
}

/* Set Font-Direction */
void xwSetFontDirection(int fNr,int direction)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->fontDirection=direction;
   }

   return;
}

/* Get Font Mode */
int xwGetFontMode(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->fontMode);
   }

   return(0);
}

/* Set Font Mode */
void xwSetFontMode(int fNr,int mo)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->fontMode=mo;
   }

   return;
}

/* Get Font Color */
int xwGetFontColor(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->fontColor);
   }

   return(0);
}

/* Set Font Color */
void xwSetFontColor(int fNr,int rgb)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      fd[fNr]->fontColor=rgb;
   }

   return;
}

/* Get Font Text */
char *xwGetFontText(int fNr)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      return(fd[fNr]->fontText);
   }

   return("");
}

/* Set Font Text */
void xwSetFontText(int fNr,char *txt)
{
   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      if (fd[fNr]->fontText==NULL)
      {
         fd[fNr]->fontText=malloc(strlen(txt)+4);
         if (fd[fNr]->fontText!=NULL) strcpy(fd[fNr]->fontText,txt);
      }
      else
      {
         fd[fNr]->fontText=realloc(fd[fNr]->fontText,strlen(txt)+4);
         if (fd[fNr]->fontText!=NULL) strcpy(fd[fNr]->fontText,txt);
      }
   }

   return;
}

/* Set Raw Font Text */
void xwSetRawFontText(int fNr,int size,FILE *fh)
{
   char              *strptr;

   if ( (frameAnz>-1) && (fNr<=frameAnz) )
   {
      if (fd[fNr]->fontText==NULL)
      {
         fd[fNr]->fontText=malloc(size+4);
         if (fd[fNr]->fontText!=NULL)
         {               
            fread(fd[fNr]->fontText,1,size,fh);
            strptr=fd[fNr]->fontText;
            strptr=strptr+size;
            strptr[0]=0x00;
         }
      }
      else
      {
         fd[fNr]->fontText=realloc(fd[fNr]->fontText,size+4);
         if (fd[fNr]->fontText!=NULL)
         {
            fread(fd[fNr]->fontText,1,size,fh);
            strptr=fd[fNr]->fontText;
            strptr=strptr+size;
            strptr[0]=0x00;
         }
      }
   }

   return;
}
/* ################################################################### */
/* Page Information Routines */

/* Get lowest X-Frame-Position */
int xwGetLowXFrame(void) /*fold00*/
{
   int            ax,az;

   ax=-1;

   /* Normal Picture Mode */
   if (frameMode==0)
   {
      ax=xwGetLowXPicture();
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      if (frameAnz>-1)
      {
         for (az=0 ; az<=frameAnz ; az++)
         {
            if (ax==-1) ax=fd[az]->xpos;

            if (fd[az]->xpos<ax) ax=fd[az]->xpos;
         }
      }
   }

   return(ax);
}

/* Get lowest Y-Frame-Position */
int xwGetLowYFrame(void) /*fold00*/
{
   int            ay,az;

   ay=-1;

   /* Normal Picture Mode */
   if (frameMode==0)
   {
      ay=xwGetLowYPicture();
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      if (frameAnz>-1)
      {
         for (az=0 ; az<=frameAnz ; az++)
         {
            if (ay==-1) ay=fd[az]->ypos;

            if (fd[az]->ypos<ay) ay=fd[az]->ypos;
         }
      }
   }

   return(ay);
}

/* Get highest X-Frame-Position */
int xwGetHighXFrame(void) /*fold00*/
{
   int            ax,az;

   ax=-1;

   /* Normal Picture Mode */
   if (frameMode==0)
   {
      ax=xwGetHighXPicture();
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      if (frameAnz>-1)
      {
         for (az=0 ; az<=frameAnz ; az++)
         {
            if (ax==-1) ax=fd[az]->xpos+fd[az]->xsize;

            if (fd[az]->xpos+fd[az]->xsize>ax) ax=fd[az]->xpos+fd[az]->xsize;
         }
      }
   }

   return(ax);
}

/* Get highest Y-Frame-Position */
int xwGetHighYFrame(void) /*fold00*/
{
   int            ay,az;

   ay=-1;

   /* Normal Picture Mode */
   if (frameMode==0)
   {
      ay=xwGetHighYPicture();
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      if (frameAnz>-1)
      {
         for (az=0 ; az<=frameAnz ; az++)
         {
            if (ay==-1) ay=fd[az]->ypos+fd[az]->ysize;

            if (fd[az]->ypos+fd[az]->ysize>ay) ay=fd[az]->ypos+fd[az]->ysize;
         }
      }
   }

   return(ay);
}

/*
   Frame Creation and Modification
*/

/* Add Frame */
void xwAddFrame(void) /*fold00*/
{
   int            az,bz;
   char           astr[40];

   /* Picture to Frame convert */
   if ( (frameMode==0) && (xwGetPictureCount()>-1) )
   {
      az=xwGetPictureCount();
      for (bz=0 ; bz<=az ; bz++)
      {
         /* Reserve new Frame */
         frameAnz++;

         fd[frameAnz]= (framedatas*) malloc(sizeof(framedatas)+4);
         if (fd[frameAnz]!=NULL)
         {

            /* Create Frame Name */
            frameID++; /* Frame ID */
            strcpy(fd[frameAnz]->name,"Frame ");
            sprintf(astr,"%d",frameID);
            strcat(fd[frameAnz]->name,astr);

            /* Picture Rotated */
            if ( (xwGetPictureR(bz)==90) || (xwGetPictureR(bz)==-90) )
            {
               fd[frameAnz]->xpos=xwGetPictureX(bz);
               fd[frameAnz]->ypos=xwGetPictureY(bz);
               fd[frameAnz]->xsize=xwGetPictureH(bz);
               fd[frameAnz]->ysize=xwGetPictureW(bz);
            }
            else
            {
               fd[frameAnz]->xpos=xwGetPictureX(bz);
               fd[frameAnz]->ypos=xwGetPictureY(bz);
               fd[frameAnz]->xsize=xwGetPictureW(bz);
               fd[frameAnz]->ysize=xwGetPictureH(bz);
            }

            fd[frameAnz]->pFlag=0;
            fd[frameAnz]->pXSize=-1;
            fd[frameAnz]->pYSize=-1;

            /* Frame Settings for Editor */

            fd[frameAnz]->borderSize=0;
            fd[frameAnz]->borderColor=0; /* Paper */
            fd[frameAnz]->frameSize=0;
            fd[frameAnz]->color=0;

            fd[frameAnz]->hline=0;
            fd[frameAnz]->hcolor=0;

            fd[frameAnz]->vline=0;
            fd[frameAnz]->vcolor=0;

            fd[frameAnz]->borderType=0;

            fd[frameAnz]->pos=frmCenter;

            /* Picture Settings for Editor */
            
            fd[frameAnz]->pborderSize=xwGetPictureB(bz);
            fd[frameAnz]->pborderColor=xwGetPictureBC(bz); /* Paper */
            fd[frameAnz]->pframeSize=xwGetPictureF(bz);
            fd[frameAnz]->pframeColor=xwGetPictureC(bz);
            fd[frameAnz]->ptransparenz=xwGetPictureT(bz);

            fd[frameAnz]->ptransMode=xwGetPictureTM(bz);

            strcpy(fd[frameAnz]->fontName,"");
            fd[frameAnz]->fontSize=20;
            fd[frameAnz]->fontLine=0;
            fd[frameAnz]->fontDirection=0;
            fd[frameAnz]->fontMode=0;
            fd[frameAnz]->fontColor=0;
            fd[frameAnz]->fontText=NULL;

            fd[frameAnz]->picNr=bz;

            /* Deactivate Frames */
            frameMode=1;
            /* Edit Frames */
            frameEdit=1;
            /* Actual Frame */
            frameActive=frameAnz;
         }
         else
         {
            frameAnz--;
         }
      }
   }

   /* Create New frame 49 */
   if (frameAnz<(frameMaxFrames-1))
   {
      frameAnz++;

      fd[frameAnz]= (framedatas*) malloc(sizeof(framedatas)+4);
      if (fd[frameAnz]!=NULL)
      {
         /* Create Frame Name */
         frameID++; /* Frame ID */
         strcpy(fd[frameAnz]->name,"Frame ");
         sprintf(astr,"%d",frameID);
         strcat(fd[frameAnz]->name,astr);

         fd[frameAnz]->xpos=xwGetPaperLB()+((xwGetPaperXB()-(xwGetPaperXB()/2))/2);
         fd[frameAnz]->ypos=xwGetPaperTB()+((xwGetPaperYB()-(xwGetPaperYB()/2))/2);
         fd[frameAnz]->xsize=xwGetPaperXB()/2;
         fd[frameAnz]->ysize=xwGetPaperYB()/2;
         fd[frameAnz]->picNr=-1;

         fd[frameAnz]->pFlag=0;
         fd[frameAnz]->pXSize=-1;
         fd[frameAnz]->pYSize=-1;

         /* Frame Settings for Editor */

         fd[frameAnz]->borderSize=0;
         fd[frameAnz]->borderColor=0; /* Paper */
         fd[frameAnz]->frameSize=0;
         fd[frameAnz]->color=0;

         fd[frameAnz]->hline=0;
         fd[frameAnz]->hcolor=0;

         fd[frameAnz]->vline=0;
         fd[frameAnz]->vcolor=0;

         fd[frameAnz]->borderType=0;

         fd[frameAnz]->pos=frmCenter;

         /* Picture Settings for Editor */

         fd[frameAnz]->pborderSize=0;
         fd[frameAnz]->pborderColor=0; /* Paper */
         fd[frameAnz]->pframeSize=0;
         fd[frameAnz]->pframeColor=0;
         fd[frameAnz]->ptransparenz=0;

         fd[frameAnz]->ptransMode=0;

         strcpy(fd[frameAnz]->fontName,"");
         fd[frameAnz]->fontSize=20;
         fd[frameAnz]->fontLine=0;
         fd[frameAnz]->fontDirection=0;
         fd[frameAnz]->fontMode=0;
         fd[frameAnz]->fontColor=0;
         fd[frameAnz]->fontText=NULL;

         /* Deactivate Frames */
         frameMode=1;
         /* Edit Frames */
         frameEdit=1;
         /* Actual Frame */
         frameActive=frameAnz;
      }
      else
      {
         frameAnz--;
      }

      xwSetActivePicture(fd[frameActive]->picNr);
   }

   return;
}

/* Frame Trashcan */
void xwFPFrameTrash(void) /*fold00*/
{

   /* Normal Mode / None Frame Edit Mode */
   if (xwCheckEditFrame()!=0)
   {
      /* Normal Pictures */
      if (frameMode==0)
      {
         xwFreePicture(xwGetActivePicture());
      }

      /* Frame Mode */
      if (frameMode==1)
      {
         if (fd[frameActive]->picNr>-1)
         {
            xwReorgPic(frameActive);
            xwFreePicture(fd[frameActive]->picNr);
            fd[frameActive]->picNr=-1;
            xwSetActivePicture(fd[frameActive]->picNr);
         }
      }
   }

   /* Frame Edit Mode */
   if (xwCheckEditFrame()==0)
   {
      /* is picture in Frame */
      if (fd[frameActive]->picNr>-1)
      {
         xwReorgPic(frameActive);
         xwFreePicture(fd[frameActive]->picNr);
         /*fd[frameActive]->picNr=-1;*/
         xwSetActivePicture(fd[frameActive]->picNr);
      }

      /* actual frame is not the last frame */
      if (frameAnz>-1)
      {
         xwFreeFrames(frameActive);

         /* No frames exists */
         if (frameAnz==-1)
         {
            frameMode=0;
            frameEdit=0;
            frameID=1;
         }
         else
         {
            xwSetActivePicture(fd[frameActive]->picNr);
         }
      }
   }

   return;
}

/* Reorg Picture Entrys - az -actual frame */
void xwReorgPic(int az) /*fold00*/
{
   int            bz;

   for (bz=0 ; bz<=frameAnz ; bz++)
   {
      if ( (bz<az) || (bz>az) )
      {
         if (fd[bz]->picNr>fd[az]->picNr) fd[bz]->picNr=fd[bz]->picNr-1;
      }
   }

   return;
}

/* MoveFrameToBack */
void xwMoveFrameToBack(void) /*fold00*/
{
   /* Normal Picture Mode */
   if (frameMode==0)
   {
      xwMovePictureToBack();
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      /* is actual frameNr > 0 */
      if (frameActive>0)
      {
         frameActive--;

         xwSwapFrame(frameActive,frameActive+1);
      }
   }

   return;
}

/* MoveFrameToFront */
void xwMoveFrameToFront(void) /*fold00*/
{
   /* Normal Picture Mode */
   if (frameMode==0)
   {
      xwMovePictureToFront();
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      /* is actual frameNr > frameAnz */
      if (frameActive<frameAnz)
      {
         xwSwapFrame(frameActive,frameActive+1);

         frameActive++;
      }
   }

   return;
}

/* Rotate Frame Picture */
void xwRotatePicInFrame(int direction) /*fold00*/
{
   /* Normal Picture Mode */
   if (frameMode==0)
   {
      xwRotatePicture(direction,xwGetPaperLB(),xwGetPaperTB(),xwGetPaperXB(),xwGetPaperYB());
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      if (fd[frameActive]->picNr!=-1)
      {
         xwSetActivePicture(fd[frameActive]->picNr);
         xwRotatePicture(direction,fd[frameActive]->xpos,fd[frameActive]->ypos,fd[frameActive]->xsize,fd[frameActive]->ysize);
      }
   }

   return;
}

/* Exchange Frames */
void xwExchangeFrame(int az,int bz) /*fold00*/
{
   int            picNr;

   if ( (az>-1) && (az<=frameAnz) && (bz>-1) && (bz<=frameAnz) )
   {
      /* Exchange Pictures */
      picNr=fd[az]->picNr;
      fd[az]->picNr=fd[bz]->picNr;
      fd[bz]->picNr=picNr;

      /* Scale Picture into Frame */
      if (fd[az]->picNr>-1) xwScalePicture(fd[az]->picNr,fd[az]->xpos,fd[az]->ypos,fd[az]->xsize,fd[az]->ysize);
      if (fd[bz]->picNr>-1) xwScalePicture(fd[bz]->picNr,fd[bz]->xpos,fd[bz]->ypos,fd[bz]->xsize,fd[bz]->ysize);

      /* Set Active Picture */
      xwSetActivePicture(fd[bz]->picNr);

      /* Set Active Frame */
      xwSetActiveFrame(bz);
   }

   return;
}

/* Resize Frame */
void  xwFrameResize(int mode) /*fold00*/
{
   int            bug;

   /* Normal Mode / None Frame Edit Mode */
   if (xwCheckEditFrame()!=0)
   {
      /* Normal Picture */
      if (frameMode==0)
      {
         xwPictureResize(mode);
      }
   }

   /* Frame Edit Mode */
   if (xwCheckEditFrame()==0)
   {

      /* Portrait Mode */
      if (xwIsPaperLS()==xwPortrait)
      {
         bug=0;

         fd[frameActive]->xsize=fd[frameActive]->xsize+mode;
         fd[frameActive]->ysize=fd[frameActive]->ysize+mode;

         /* to small */
         if (fd[frameActive]->xsize<frameMinSize) bug=1;
         if (fd[frameActive]->ysize<frameMinSize) bug=1;

         /* to big */
         if (fd[frameActive]->xpos+fd[frameActive]->xsize>xwGetPaperXSize()-xwGetPaperRB()) bug=1;
         if (fd[frameActive]->ypos+fd[frameActive]->ysize>xwGetPaperYSize()-xwGetPaperBB()) bug=1;

         /* bug registred */
         if (bug==1)
         {
            /* Size */
            fd[frameActive]->xsize=fd[frameActive]->xsize-mode;
            fd[frameActive]->ysize=fd[frameActive]->ysize-mode;
         }
      }
      else
      /* Landscape Mode */
      {
         bug=0;

         /* Position */
         fd[frameActive]->ypos=fd[frameActive]->ypos+(mode*(-1));

         /* Size */
         fd[frameActive]->xsize=fd[frameActive]->xsize+mode;
         fd[frameActive]->ysize=fd[frameActive]->ysize+mode;

         /* to small */
         if (fd[frameActive]->xsize<frameMinSize) bug=1;
         if (fd[frameActive]->ysize<frameMinSize) bug=1;

         /* to big */
         if (fd[frameActive]->xpos+fd[frameActive]->xsize > xwGetPaperXSize()-xwGetPaperRB()) bug=1;
         if (fd[frameActive]->ypos < xwGetPaperTB()) bug=1;
         if (fd[frameActive]->ypos+fd[frameActive]->ysize > xwGetPaperYSize()-xwGetPaperBB()) bug=1;

         /* bug registred */
         if (bug==1)
         {
            /* Position */
            fd[frameActive]->ypos=fd[frameActive]->ypos-(mode*(-1));

            /* Size */
            fd[frameActive]->xsize=fd[frameActive]->xsize-mode;
            fd[frameActive]->ysize=fd[frameActive]->ysize-mode;
         }
      }

   }

   return;
}

/* Move Frame */
void  xwFrameMove(int xpos,int ypos) /*fold00*/
{
   int            ax,ay;

   /* Normal Mode / None Frame Edit Mode */
   if (xwCheckEditFrame()!=0)
   {
      /* Normal Picture */
      if (frameMode==0)
      {
         xwPictureMove(xpos,ypos);
      }
   }

   /* Frame Edit Mode */
   if (xwCheckEditFrame()==0)
   {
      /* Landscape Move */
      if (xwIsPaperLS()==xwLandscape)
      {
         ax=ypos;
         ay=xpos*(-1);
      }
      else
      {
         ax=xpos;
         ay=ypos;
      }

      fd[frameActive]->xpos=fd[frameActive]->xpos+ax;
      fd[frameActive]->ypos=fd[frameActive]->ypos+ay;

      /* to small */
      if (fd[frameActive]->xpos<xwGetPaperLB()) fd[frameActive]->xpos=xwGetPaperLB();
      if (fd[frameActive]->ypos<xwGetPaperTB()) fd[frameActive]->ypos=xwGetPaperTB();

      /* to big */
      if (fd[frameActive]->xpos+fd[frameActive]->xsize>xwGetPaperXSize()-xwGetPaperRB())
      {
         fd[frameActive]->xpos=xwGetPaperXSize()-(fd[frameActive]->xsize+xwGetPaperRB());
      }
      if (fd[frameActive]->ypos+fd[frameActive]->ysize>xwGetPaperYSize()-xwGetPaperBB())
      {
         fd[frameActive]->ypos=xwGetPaperYSize()-(fd[frameActive]->ysize+xwGetPaperBB());
      }
   }

   return;
}

/* #### SPECIAL FEATURES #### */

/* View Frame Picture */
void xwViewFramePicture(void) /*fold00*/
{
   /* Normal Picture Mode */
   if (frameMode==0)
   {
      xwDisplayPicture();
   }

   /* Frame Picture Mode */
   if (frameMode==1)
   {
      if (fd[frameActive]->picNr!=-1)
      {
         xwDisplayPicture();
      }
   }

   return;
}