/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "fileio.h"

#include "base.h"
#include "limits.h"
#include "x11.h"

static struct picresolution picRes;

static int readMode = MOTOROLA;

static union allType_u           /* zum Einlesen          */
{
    unsigned char  b[4];
    unsigned short s[2];
    unsigned long  l[1];
} allType;

/* Initialize FileIO */
void xwInitFileIO() /*fold00*/
{
   allType.l[0] = 0x01020304;
   if ( allType.b[0] == 4 )
   {
      readMode = WRONG_ORDER;
   }

   return;
}
 /*fold00*/
/* Byte Word oder Long einlesen, Ordnung sicher stellen */
/*
 char *source     Datei die zu lesen ist
 int   fpos       Position wo zu lesen ist
 int   size       1 , 2 oder 4 entsprechene byte, short oder long
 int   fileOrder  entsprechend anordnung der byte bei long / short
                  in der Datei, Wert 0 = Intel, 1 = Motorola
*/
int readValue(const char *source,int fpos, int size, int fileOrder) /*fold00*/
{
    FILE *fh;
    unsigned char byte;
    int           returnValue = 0;

    if ((fh=fopen(source,"r"))!=NULL)
    {
        fseek(fh,fpos,SEEK_SET);
        if (fread(allType.b,size,1,fh) > 0)
        {
            if ( readMode != fileOrder )
            {
                switch(size)
                {
                case 1:
                    returnValue = allType.b[0];
                    break;
                case 2:
                    byte         = allType.b[0];
                    allType.b[0] = allType.b[1];
                    allType.b[1] = byte;
                    returnValue  = allType.s[0];
                    break;
                case 4:
                    byte         = allType.b[0];
                    allType.b[0] = allType.b[3];
                    allType.b[3] = byte;
                    byte         = allType.b[1];
                    allType.b[1] = allType.b[2];
                    allType.b[2] = byte;
                    returnValue  = allType.l[0];
                    break;
                }
            }
            else
            {
                switch(size)
                {
                case 1:
                    returnValue = allType.b[0];
                    break;
                case 2:
                    returnValue = allType.s[0];
                    break;
                case 4:
                    returnValue = allType.l[0];
                    break;
                }
            }
        }
        fclose(fh);
    }
    return(returnValue);
}

/* Text einlesen */
const char *readtext(const char *source,int fpos,int a) /*fold00*/
{
   static char       buf[256];
   FILE              *fh;

   int               az;

   strcpy(buf,"");
   for (az=0 ; az<a ; az++)
   {
      strcat(buf," ");
   }

   /*printf("FILEIO: %i %i\n",a,strlen(buf));*/

   if ((fh=fopen(source,"r"))!=NULL)
   {
      fseek(fh,fpos,SEEK_SET);
      if (fread(&buf,a,1,fh) > 0)
      {
         return(buf);
      }
      else
      {
         return("");
      }
      fclose(fh);
   }
   return("");
}

/* ##################################### */
/* #                                   # */
/* # FILE FORMAT ANALYSE               # */
/* #                                   # */
/* ##################################### */

/* check and get GFX Datas */
int xwCheckGFXFile(const char *source) /*fold00*/
{
   int               gfxFile;
   struct stat       attribut;

   gfxFile=-1;
   if (strlen(source)>0)
   {
      /* Check Size of File */
      if ( stat(source,&attribut) == 0)
      {
         if (attribut.st_size>0)
         {   
            gfxFile = checkgif(source);
            if (gfxFile==-1) gfxFile = checktiff(source);
            if (gfxFile==-1) gfxFile = checkppm(source);
            if (gfxFile==-1) gfxFile = checkbmp(source);
            if (gfxFile==-1) gfxFile = checkjpg(source);
            if (gfxFile==-1) gfxFile = checkpng(source);
            if (gfxFile==-1) gfxFile = checkxwd(source);
         }
      }
   }
   
   return(gfxFile);
}

/* check xwgui files */
int xwCheckXWGFile(const char *source) /*fold00*/
{
   int            xwgFile=-1;

   xwgFile = checkfrms(source);

   return(xwgFile);
}

/* Get ID of Image - Specially for PPM-File Format */
int xwGetGFXID(void)
{
   return(picRes.ID);
}

/* Get X-Size of Image */
int xwGetGFXX(void) /*fold00*/
{
   return(picRes.xsize);
}

/* Get Y-Size of Image */
int xwGetGFXY(void) /*fold00*/
{
   return(picRes.ysize);
}

/* Get Position of Image Datas */
int xwGetGFXP(void) /*fold00*/
{
   return(picRes.image);
}

/* check GIF-Format OK */
int checkgif(const char *source) /*fold00*/
{
   const char     *strptr;
   char           astr[10];
   FILE           *fh;
   char           buf[80];
   unsigned char  *s;

   /* open file and read any bytes */
   if ((fh=fopen(source,"r"))!=NULL)
   {
      fread(&buf,1,sizeof(buf),fh);
      fclose(fh);

      s = (unsigned char*) &buf;

      /* GIF */
      if ( (strncmp((char*)s,"GIF87a",6) == 0) || (strncmp((char*)s,"GIF89a",6) == 0) )
      {
         strptr = readtext(source,0,3);
         strcpy(astr,strptr);

         if (strcmp(astr,"GIF") == 0)
         {
            picRes.xsize = readValue(source,6,2,0);
            picRes.ysize = readValue(source,8,2,0);
         }
      }
      else
      {
         return(-1);
      }
   }
   else
   {
      return(-1);
   }

   return(0);
}

/* check TIF-Format OK */
int checktiff(const char *source) /*fold00*/
{
   const char     *strptr;
   char           astr[10];
   FILE           *fh;
   char           buf[80];
   unsigned char  *s;

   int            filepos=0;
   int            taganz=0;
   int            tag=0;
   int            nr=1;
   int            status=0;
   int            tiffByteOrder = 0;

   /* open file and read any bytes */
   if ((fh=fopen(source,"r"))!=NULL)
   {
      fread(&buf,1,sizeof(buf),fh);
      fclose(fh);

      s = (unsigned char*) &buf;

      /* TIFF */
      if ( (*s == 0115 && s[1] == 0115) || (*s == 0111 && s[1] == 0111) )
      {
         strptr = readtext(source,0,2);
         strcpy(astr,strptr);

         if (strcmp(astr,"MM") == 0)
         {
            tiffByteOrder = 1;
         }
         if (strcmp(astr,"II") == 0 || strcmp(astr,"MM") == 0)
         {
            filepos = readValue(source,4,4,tiffByteOrder);
            taganz = readValue(source,filepos,2,tiffByteOrder);
            filepos = filepos+2;

            for (nr=1 ; nr<=taganz ; nr++)
            {
               tag = readValue(source,filepos,2,tiffByteOrder);

               /* Image Width Tag */
               if (tag == 256)
               {
                  status++;
                  filepos = filepos+8;
                  picRes.xsize = readValue(source,filepos,2,tiffByteOrder);
                  filepos = filepos+4;
                  tag = 0;
               }
               /* Image Length Tag */
               if (tag == 257)
               {
                  status++;
                  filepos = filepos+8;
                  picRes.ysize = readValue(source,filepos,2,tiffByteOrder);
                  filepos = filepos+4;
                  tag = 0;
               }
               /* Next Tag */
               if (tag > 0)
               {
                  filepos = filepos+12;
               }
            }
         }
      }
   }

   if (status==2) return(0);

   return(-1);
}

/* checl PPM-Format OK */
int checkppm(const char *source) /*fold00*/
{
   FILE           *fh;
   int            data;
   
   int            count;
   int            mo;
   
   count=0;
   mo=0;
   
   picRes.xsize=0;
   picRes.ysize=0;
   picRes.depth=0;
   picRes.image=0;
   
   picRes.ID=0;
   
   if ((fh=fopen(source,"r"))!=NULL)
   {
      data=fgetc(fh);
      if (data=='P')
      {
         data=fgetc(fh);
         if ((data=='3') || (data=='5') || (data=='6'))
         {
            picRes.ID=data;
            while (((data=fgetc(fh))!=EOF) && (count<4))
            {
               if (data=='#') while ( ((data=fgetc(fh))!='\n') && (data!=EOF));
               else if ((data>='0') && (data<='9')) 
               {
                  if (mo==0)
                  {
                     mo=1;
                     count++;
                  }
                  
                  if (count==1) picRes.xsize=(picRes.xsize*10)+(data-'0');
                  else if (count==2) picRes.ysize=(picRes.ysize*10)+(data-'0');
                  else if (count==3) picRes.depth=(picRes.depth*10)+(data-'0');
               }
               else if ((data=='\n') || (data==' ') || (data=='\t') || (data=='\r'))
               {
                  mo=0;
                  if (count==3)
                  { 
                     picRes.image=(int) ftell(fh);
                     count++;
                  }
               }               
            }
         }
      }
      fclose(fh);
   }
   
   /*
   printf("Width =%i\n",picRes.xsize);
   printf("Height=%i\n",picRes.ysize);
   printf("Depth =%i\n",picRes.depth);
   printf("Image =%i\n",picRes.image);
   */
   
   /* Sucessfull */
   if ((picRes.xsize>0) && (picRes.ysize>0) && (picRes.depth>0) && (picRes.image>0)) return(0);

   return(-1);
}

/* check BMP-Format OK */
int checkbmp(const char *source) /*fold00*/
{
   FILE           *fh;
   char           buf[12];

   /* open file and read any bytes */
   if ((fh=fopen(source,"r"))!=NULL)
   {
      fread(&buf,1,sizeof(buf),fh);
      fclose(fh);

      /* BMP */
      if (strncmp(buf,"BM",2) == 0)
      {
         picRes.xsize = readValue(source,18,4,0);
         picRes.ysize = readValue(source,22,4,0);
         return(0);
      }
   }

   return(-1);
}

/* check JPG-Format OK */
int checkjpg(const char *source) /*fold00*/
{
   int            filepos=0;
   int            chunkID=0;
   int            chunkSize=0;
   int            ready=0;

   struct stat    statpuff;
   int            filesize=0;

   chunkID = readValue(source,filepos,2,0);
   if (chunkID == 55551)
   {
      filepos = filepos + 2;

      if (stat(source,&statpuff) == 0)
      {
         filesize = statpuff.st_size;
         while ( (ready == 0) && (filepos < filesize) )
         {
            chunkID = readValue(source,filepos,2,1);

            /* gesuchter Chunk */
            if (chunkID == 0xffc0)  /* 65472) */
            {
               picRes.xsize = readValue(source,filepos+7 ,2,1);
               picRes.ysize = readValue(source,filepos+5 ,2,1);
               ready=2;
               chunkID=0;
            }

            /* vermutlich gesuchter Chunk - Progressive Mode */
            if (chunkID == 0xffc2)  /* 65474) */
            {
               picRes.xsize = readValue(source,filepos+7 ,2,1);
               picRes.ysize = readValue(source,filepos+5 ,2,1);
               ready=2;
               chunkID=0;
            }

            /* Compressed Image - Abort */
            if (chunkID == 0xffda) /* 65498) */
            {
               ready=1;
               chunkID=0;
            }

            /* Endkennung */
            if (chunkID == 0xffd9) /* 65497) */
            {
               ready=1;
               chunkID=0;
            }

            /* Chunk berspringen */
            if (chunkID != 0)
            {
               chunkSize = readValue(source,filepos+2,2,1);
               filepos = filepos + chunkSize + 2;
            }

         }
      }
   }

   if (ready==2) return(0);

   return(-1);
}

/* check PNG-Format OK */
int checkpng(const char *source) /*fold00*/
{
    int filepos=0;
    int chunkID=0;
    int chunkSize=0;
    int ready=0;

    struct stat statpuff;
    int filesize=0;

    chunkID = readValue(source,filepos,4,1);
    if (chunkID == 0x89504e47)
    {
        filepos = filepos + 8;

        if (stat(source,&statpuff) == 0)
        {
            filesize = statpuff.st_size;
            while ( (ready == 0) && (filepos < filesize) )
            {
                chunkSize = readValue(source,filepos,4,1); /* Get Chunk Size */
                filepos = filepos + 4;
                chunkID = readValue(source,filepos,4,1); /* Get Chunk ID */
                filepos = filepos + 4;

                /* gesuchter Chunk */
                if (chunkID == 0x49484452)
                {
                    picRes.xsize = readValue(source,filepos,4,1);
                    picRes.ysize = readValue(source,filepos+4 ,4,1);
                    ready=1;
                    chunkID=0;
                }

                /* Chunk berspringen */
                if (chunkID != 0)
                {
                    filepos = filepos + chunkSize + 4;
                }

            }
        }
    }

   if (ready==1) return(0);

   return(-1);
}

/* check XWD-Format */
int checkxwd(const char *source) /*fold00*/
{
   FILE           *fh;
   XWDFileHeader  buf;
   long           *swap;
   int            i;

   /* open file and read any bytes */
   if ((fh=fopen(source,"r"))!=NULL)
   {
      fread(&buf,1,sizeof(buf),fh);
      fclose(fh);

      /* try for xwd format jjsa oct 10 99 */
      if ( readMode == INTEL )
      {
         for ( i = 0; i < 25; i ++)
         {
            swap  = ((long*)&buf) + i;
            *swap = (((*swap >> 24) & 0xff)      ) +
                    (((*swap >> 16) & 0xff) <<  8) +
                    (((*swap >>  8) & 0xff) << 16) +
                    (((*swap      ) & 0xff) << 24);
         }
      }

      if ( (buf.pixmap_format == ZPixmap) &&
           ( buf.bits_per_pixel == 8 || buf.bits_per_pixel == 16 || buf.bits_per_pixel == 32) )
      {
         picRes.xsize = buf.pixmap_width;
         picRes.ysize = buf.pixmap_height;
         return(0);
      }
   }

   return(-1);
}

/* Check FrameSet Files */
int checkfrms(const char *source) /*fold00*/
{
   FILE           *fh;
   char           buff[1024];
   int            nr=0;

   if ((fh=fopen(source,"r"))!=NULL)
   {
      while (nr<50)
      {
         if (fgets(buff,1024,fh) != NULL)
         {
            strUpper(buff);
            if (strcmp(buff,"FRAMESET") == 0)
            {
               fclose(fh);
               return(0);
            }
         }
         nr++;
      }

      fclose(fh);
      return(0);
   }

   return(-1);
}
