/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "editor.h"

#include "base.h"
#include "color.h"
#include "default.h"
#include "font.h"
#include "frame.h"
#include "limits.h"
#include "main_gui.h"
#include "paper.h"
#include "picture.h"
#include "resource.h"
#include "handler.h"
#include "help.h"

#include "pixmaps/ed-east.xpm"
#include "pixmaps/ed-south.xpm"
#include "pixmaps/ed-west.xpm"
#include "pixmaps/ed-north.xpm"

#include "pixmaps/ed-left.xpm"
#include "pixmaps/ed-middle.xpm"
#include "pixmaps/ed-right.xpm"

#include "pixmaps/ed-quadborder.xpm"
#include "pixmaps/ed-circleborder.xpm"
#include "pixmaps/ed-allborder.xpm"

static editorform    ed;

/* Handler Lock */
static int           xwEdHandlerLock=0;

/* New Color Base */
static int           edCB=0;
static int           edR=0,edG=0,edB=0;

/* Frame Datas */
static double        edFrameB;      /* Frame BorderSize */
static int           edFrameBC;     /* Frame Border Background Color - Paper or Transparency */
static double        edFrameF;      /* Frame BorderLineSize */
static int           edFrameC;      /* Frame Color */
static double        edHLine;
static double        edVLine;
static int           edFrameBT;
static int           edPicPosition; /* Picture Position */

static int           edFrameW;
static int           edFrameH;

/* Picture Datas */
static double        edPictureB;
static int           edPictureBC;
static double        edPictureF;
static int           edPictureC;    /* Picture Border Line Color */
static int           edPictureTM;
static int           edPictureT;

static int           edPictureW;
static int           edPictureH;
static int           edPictureR;

static int           edX,edY;

static char          edFontName[1024];
static int           edFontSize;
static int           edFontLine;

/* Editor 690 500 */
void xwEditor(void) /*fold00*/
{
   char              *strptr;

   if ((xwGetFrameCount()>-1) || (xwGetPictureCount()>-1))
   {

      xwDeactivateMain();

      edCB=26; /* FL_FREE_COL1;*/  /* first unused color */

      /* Get Frame & Picture Datas */
      if (xwCheckFrameMode()==xwTrue)
      {
         /* Get Frame & Picture Datas */
         edFrameB=xwGetFrameB(xwGetActiveFrame());
         edFrameBC=xwGetFrameBC(xwGetActiveFrame());
         edFrameF=xwGetFrameF(xwGetActiveFrame());
         edFrameC=xwGetFrameC(xwGetActiveFrame());
         edHLine=xwGetHLine(xwGetActiveFrame());
         edVLine=xwGetVLine(xwGetActiveFrame());
         edFrameBT=xwGetFrameBT(xwGetActiveFrame());
         edPicPosition=xwGetPicPosition(xwGetActiveFrame());

         edFrameW=xwGetFrameW(xwGetActiveFrame())/xwMM();
         edFrameH=xwGetFrameH(xwGetActiveFrame())/xwMM();

         edPictureB=xwGetFramePB(xwGetActiveFrame());
         edPictureBC=xwGetFramePBC(xwGetActiveFrame());
         edPictureF=xwGetFramePF(xwGetActiveFrame());
         edPictureC=xwGetFramePC(xwGetActiveFrame());
         edPictureTM=xwGetFramePTM(xwGetActiveFrame());
         edPictureT=xwGetFramePT(xwGetActiveFrame());

         if (xwGetFramePicture(xwGetActiveFrame())>-1)
         {
            edPictureW=xwGetPictureW(xwGetFramePicture(xwGetActiveFrame()))/xwMM();
            edPictureH=xwGetPictureH(xwGetFramePicture(xwGetActiveFrame()))/xwMM();
            edPictureR=xwGetPictureR(xwGetFramePicture(xwGetActiveFrame()));
         }
         else
         {
            edPictureW=200;
            edPictureH=200;
            edPictureR=0;
         }
      }
      else
      {
         /* Get Picture Datas */
         edFrameB=0;
         edFrameBC=0;
         edFrameF=0;
         edFrameC=0;
         edHLine=0;
         edVLine=0;
         edFrameBT=0;
         edPicPosition=frmCenter;

         edFrameW=0;
         edFrameH=0;

         edPictureB=xwGetPictureB(xwGetActivePicture());
         edPictureBC=xwGetPictureBC(xwGetActivePicture());
         edPictureF=xwGetPictureF(xwGetActivePicture());
         edPictureC=xwGetPictureC(xwGetActivePicture());
         edPictureTM=xwGetPictureTM(xwGetActivePicture());
         edPictureT=xwGetPictureT(xwGetActivePicture());

         edPictureW=xwGetPictureW(xwGetActivePicture())/xwMM();
         edPictureH=xwGetPictureH(xwGetActivePicture())/xwMM();
         edPictureR=xwGetPictureR(xwGetActivePicture());
      }

      /* Frame Border Line */
      /*edFrameC=0xFFFF00;*/
      /*xwSplitRGB(edFrameC);*/

      xwSplitRGBV(edFrameC);
      edR=xwGetRGBSV(getRed);
      edG=xwGetRGBSV(getGreen);
      edB=xwGetRGBSV(getBlue);
      fl_mapcolor(edCB,edR,edG,edB);

      /* printf("Color: %i - %i %i %i - %i\n",edCB,edR,edG,edB,edFrameC); */

      /* Picture Border Line */
      /*edPictureC=0xffff00;*/
      /*xwSplitRGB(edPictureC);*/

      xwSplitRGBV(edPictureC);
      edR=xwGetRGBSV(getRed);
      edG=xwGetRGBSV(getGreen);
      edB=xwGetRGBSV(getBlue);fl_mapcolor(edCB+4,edR,edG,edB);

      /* printf("Color: %i - %i %i %i - %i\n",edCB,edR,edG,edB,edPictureC); */

      /* Frame Mode */
      if (xwCheckFrameMode()==xwTrue)
      {
         ed.form = fl_bgn_form(FL_UP_BOX,690,484+31);
         ed.OK = fl_add_button(FL_NORMAL_BUTTON,295,459+31,100,20,xwGR(1,"ed.OK","OK"));
         ed.HELP = fl_add_button(FL_NORMAL_BUTTON,580,459+31,100,20,xwGR(1,"help.HELP","Help"));
      }
      else
      {
         ed.form = fl_bgn_form(FL_UP_BOX,274,484+31);
         ed.OK = fl_add_button(FL_NORMAL_BUTTON,10,459+31,100,20,xwGR(1,"ed.OK","OK"));
         ed.HELP = fl_add_button(FL_NORMAL_BUTTON,164,459+31,100,20,xwGR(1,"help.HELP","Help"));
      }

      fl_set_border_width(1);

      ed.FrameA = fl_add_frame(FL_DOWN_FRAME,6,6,262,447+31,"");

      /* Display Areal */
      ed.FrameB = fl_add_frame(FL_DOWN_FRAME,12,12,250,250,"");
      fl_set_object_posthandler(ed.FrameB,xwEdHandler);

      /* ################################################################ */
      /* Frame Section */
      ed.FrameSection = fl_add_labelframe(FL_DOWN_FRAME,12,272,250,(114+31)-42,xwGR(1,"ed.FrameSetting","Frame-Setting"));

      ed.FBSize = fl_add_counter(FL_NORMAL_COUNTER,137,280,100,20,xwGR(1,"ed.FBorder","Border (mm):"));
      fl_set_object_lalign(ed.FBSize,FL_ALIGN_LEFT);
      fl_set_counter_bounds(ed.FBSize,0,20);
      fl_set_counter_step(ed.FBSize,0.1,1);
      fl_set_counter_value(ed.FBSize,edFrameB);
      fl_set_counter_return(ed.FBSize,FL_RETURN_CHANGED);

      /* Background Color */
      ed.FBColor = fl_add_button(FL_NORMAL_BUTTON,237,280,20,20,"");
      if (edFrameBC==1)
      {
         fl_set_object_color(ed.FBColor,FL_COL1,FL_COL1);
      }
      else
      {
         fl_set_object_color(ed.FBColor,FL_WHITE,FL_WHITE);
      }

      ed.FBLineSize = fl_add_counter(FL_NORMAL_COUNTER,137,301,100,20,xwGR(1,"ed.FLineBorder","Borderline (mm):"));
      fl_set_object_lalign(ed.FBLineSize,FL_ALIGN_LEFT);
      fl_set_counter_bounds(ed.FBLineSize,0,20);
      fl_set_counter_step(ed.FBLineSize,0.1,1);
      fl_set_counter_value(ed.FBLineSize,edFrameF);
      fl_set_counter_return(ed.FBLineSize,FL_RETURN_CHANGED);

      /* Color Field */
      ed.FBLColor = fl_add_button(FL_NORMAL_BUTTON,237,301,20,20,"");
      fl_set_object_color(ed.FBLColor,edCB,edCB);

      /* Border Type */

      fl_set_border_width(-2);

      ed.FBLQuadBorder = fl_add_pixmapbutton(FL_PUSH_BUTTON,137,322,30,30,xwGR(1,"ed.BorderType","BorderType:"));
      fl_set_object_lalign(ed.FBLQuadBorder,FL_ALIGN_LEFT);
      fl_set_pixmap_data(ed.FBLQuadBorder,ed_quadborder_xpm);
      fl_set_object_color(ed.FBLQuadBorder,FL_BLACK,FL_MCOL);

      ed.FBLCircleBorder = fl_add_pixmapbutton(FL_PUSH_BUTTON,167,322,30,30,"");
      fl_set_pixmap_data(ed.FBLCircleBorder,ed_circleborder_xpm);
      fl_set_object_color(ed.FBLCircleBorder,FL_BLACK,FL_MCOL);

      if (edFrameBT==0)
      {
         fl_set_button(ed.FBLQuadBorder   ,1);
         fl_set_button(ed.FBLCircleBorder ,0);
      }
      else if (edFrameBT==1)
      {
         fl_set_button(ed.FBLQuadBorder   ,0);
         fl_set_button(ed.FBLCircleBorder ,1);
      }

      fl_set_border_width(1);

      ed.FOrientation = fl_add_choice(FL_DROPLIST_CHOICE,137,322+31,120,20,xwGR(1,"ed.Position","Picture Position:"));
      fl_set_object_boxtype(ed.FOrientation,FL_UP_BOX);
      if (xwIsPaperLS()==xwLandscape)
      {
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.Center","Centered"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.NE","Top Right"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.SE","Bottom Right"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.NW","Top Left"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.SW","Bottom Left"));
      }
      else
      {
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.Center","Centered"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.NW","Top Left"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.NE","Top Right"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.SW","Bottom Left"));
         fl_addto_choice(ed.FOrientation,xwGR(1,"ed.SE","Bottom Right"));
      }
      fl_set_choice(ed.FOrientation,edPicPosition+1);

      if (xwCheckFrameMode()==xwFalse)
      {
         fl_set_object_color(ed.FBSize,FL_INACTIVE_COL,FL_INACTIVE_COL);
         fl_set_object_color(ed.FBLineSize,FL_INACTIVE_COL,FL_INACTIVE_COL);
         fl_set_object_color(ed.FOrientation,FL_INACTIVE_COL,FL_INACTIVE_COL);
         fl_deactivate_object(ed.FBSize);
         fl_deactivate_object(ed.FBColor);
         fl_deactivate_object(ed.FBLineSize);
         fl_deactivate_object(ed.FBLColor);
         fl_deactivate_object(ed.FBLQuadBorder);
         fl_deactivate_object(ed.FBLCircleBorder);

         fl_deactivate_object(ed.FOrientation);
      }

      /* ################################################################ */
      /* Picture Section */
      ed.PictureSection = fl_add_labelframe(FL_DOWN_FRAME,12,354+31,250,93,xwGR(1,"ed.PictureSetting","Picture-Setting"));

      ed.PBSize = fl_add_counter(FL_NORMAL_COUNTER,137,362+31,100,20,xwGR(1,"ed.PBorder","Border (mm):"));
      fl_set_object_lalign(ed.PBSize,FL_ALIGN_LEFT);
      fl_set_counter_bounds(ed.PBSize,0,20);
      fl_set_counter_step(ed.PBSize,0.1,1);
      fl_set_counter_value(ed.PBSize,edPictureB);
      fl_set_counter_return(ed.PBSize,FL_RETURN_CHANGED);

      /* Background Color */
      ed.PBColor = fl_add_button(FL_NORMAL_BUTTON,237,362+31,20,20,"");
      if (edPictureBC==1)
      {
         fl_set_object_color(ed.PBColor,FL_COL1,FL_COL1);
      }
      else
      {
         fl_set_object_color(ed.PBColor,FL_WHITE,FL_WHITE);
      }

      ed.PBLineSize = fl_add_counter(FL_NORMAL_COUNTER,137,383+31,100,20,xwGR(1,"ed.PLineBorder","Borderline (mm):"));
      fl_set_object_lalign(ed.PBLineSize,FL_ALIGN_LEFT);
      fl_set_counter_bounds(ed.PBLineSize,0,20);
      fl_set_counter_step(ed.PBLineSize,0.1,1);
      fl_set_counter_value(ed.PBLineSize,edPictureF);
      fl_set_counter_return(ed.PBLineSize,FL_RETURN_CHANGED);

      /* Color Field */
      ed.PBLineColor = fl_add_button(FL_NORMAL_BUTTON,237,383+31,20,20,"");
      fl_set_object_color(ed.PBLineColor,edCB+4,edCB+4);

      /* Transparency Mode */
      ed.PTransMode =  fl_add_choice(FL_DROPLIST_CHOICE,137,404+31,120,20,xwGR(1,"ed.PTransMode","Transparency Mode:"));
      fl_set_object_boxtype(ed.PTransMode,FL_UP_BOX);
      fl_addto_choice(ed.PTransMode,xwGR(1,"ed.PositiveTrans","Positive"));
      fl_addto_choice(ed.PTransMode,xwGR(1,"ed.NegativeTrans","Negative"));
      fl_set_choice(ed.PTransMode,edPictureTM+1);

      ed.PTransparenz = fl_add_counter(FL_NORMAL_COUNTER,137,425+31,100,20,xwGR(1,"ed.PTransparency","Transparency (%):"));
      fl_set_object_lalign(ed.PTransparenz,FL_ALIGN_LEFT);
      fl_set_counter_bounds(ed.PTransparenz,0,100);
      fl_set_counter_step(ed.PTransparenz,1,10);
      fl_set_counter_precision(ed.PTransparenz,0);
      fl_set_counter_value(ed.PTransparenz,edPictureT);
      fl_set_counter_return(ed.PTransparenz,FL_RETURN_CHANGED);

      /* Frame Mode */
      if (xwCheckFrameMode()==xwTrue)
      {
         ed.FrameC = fl_add_frame(FL_DOWN_FRAME,272,6,412,447+31,"");

         ed.FNTName = fl_add_box(FL_DOWN_BOX,277,11,402,20,"");
         fl_set_object_label(ed.FNTName,xwGetFontName(xwGetActiveFrame()));
         ed.FNTSize = fl_add_box(FL_DOWN_BOX,277,31,50,20,"");
         fl_set_object_label(ed.FNTSize,IntStr(xwGetFontSize(xwGetActiveFrame())));
         ed.FNTUnderline = fl_add_box(FL_DOWN_BOX,327,31,150,20,"");
         if (xwGetFontLine(xwGetActiveFrame())==0)
         {
            fl_set_object_label(ed.FNTUnderline,"-----");
         }
         else
         {
            fl_set_object_label(ed.FNTUnderline,xwGR(1,"","Underline"));
         }
         ed.FNTColor =  fl_add_button(FL_NORMAL_BUTTON,477,31,20,20,"");
         /*xwSplitRGB(xwGetFontColor(xwGetActiveFrame()));*/

         xwSplitRGBV(xwGetFontColor(xwGetActiveFrame()));
         edR=xwGetRGBSV(getRed);
         edG=xwGetRGBSV(getGreen);
         edB=xwGetRGBSV(getBlue);
         fl_mapcolor(edCB+8,edR,edG,edB);
         fl_set_object_color(ed.FNTColor,edCB+8,edCB+8);
         ed.FNTSelect = fl_add_button(FL_NORMAL_BUTTON,497,31,182,20,xwGR(1,"ed.FontSelect","Font Select"));

         fl_set_border_width(-2);

         ed.FNTEast = fl_add_pixmapbutton(FL_PUSH_BUTTON,277,53,30,30,"");
         fl_set_pixmap_data(ed.FNTEast,ed_east_xpm);
         fl_set_object_color(ed.FNTEast,FL_BLACK,FL_MCOL);
         ed.FNTSouth = fl_add_pixmapbutton(FL_PUSH_BUTTON,307,53,30,30,"");
         fl_set_pixmap_data(ed.FNTSouth,ed_south_xpm);
         fl_set_object_color(ed.FNTSouth,FL_BLACK,FL_MCOL);
         ed.FNTWest = fl_add_pixmapbutton(FL_PUSH_BUTTON,337,53,30,30,"");
         fl_set_pixmap_data(ed.FNTWest,ed_west_xpm);
         fl_set_object_color(ed.FNTWest,FL_BLACK,FL_MCOL);
         ed.FNTNorth = fl_add_pixmapbutton(FL_PUSH_BUTTON,367,53,30,30,"");
         fl_set_pixmap_data(ed.FNTNorth,ed_north_xpm);
         fl_set_object_color(ed.FNTNorth,FL_BLACK,FL_MCOL);

         if (xwIsPaperLS()==xwLandscape)
         {
            if (xwGetFontDirection(xwGetActiveFrame())==0)
            {
               fl_set_button(ed.FNTEast,0);
               fl_set_button(ed.FNTSouth,1);
               fl_set_button(ed.FNTWest,0);
               fl_set_button(ed.FNTNorth,0);
            }
            else if (xwGetFontDirection(xwGetActiveFrame())==1)
            {
               fl_set_button(ed.FNTEast,0);
               fl_set_button(ed.FNTSouth,0);
               fl_set_button(ed.FNTWest,1);
               fl_set_button(ed.FNTNorth,0);
            }
            else if (xwGetFontDirection(xwGetActiveFrame())==2)
            {
               fl_set_button(ed.FNTEast,0);
               fl_set_button(ed.FNTSouth,0);
               fl_set_button(ed.FNTWest,0);
               fl_set_button(ed.FNTNorth,1);
            }
            else if (xwGetFontDirection(xwGetActiveFrame())==3)
            {
               fl_set_button(ed.FNTEast,1);
               fl_set_button(ed.FNTSouth,0);
               fl_set_button(ed.FNTWest,0);
               fl_set_button(ed.FNTNorth,0);
            }
         }
         else
         {
            if (xwGetFontDirection(xwGetActiveFrame())==0)
            {
               fl_set_button(ed.FNTEast,1);
               fl_set_button(ed.FNTSouth,0);
               fl_set_button(ed.FNTWest,0);
               fl_set_button(ed.FNTNorth,0);
            }
            else if (xwGetFontDirection(xwGetActiveFrame())==1)
            {
               fl_set_button(ed.FNTEast,0);
               fl_set_button(ed.FNTSouth,1);
               fl_set_button(ed.FNTWest,0);
               fl_set_button(ed.FNTNorth,0);
            }
            else if (xwGetFontDirection(xwGetActiveFrame())==2)
            {
               fl_set_button(ed.FNTEast,0);
               fl_set_button(ed.FNTSouth,0);
               fl_set_button(ed.FNTWest,1);
               fl_set_button(ed.FNTNorth,0);
            }
            else if (xwGetFontDirection(xwGetActiveFrame())==3)
            {
               fl_set_button(ed.FNTEast,0);
               fl_set_button(ed.FNTSouth,0);
               fl_set_button(ed.FNTWest,0);
               fl_set_button(ed.FNTNorth,1);
            }
         }

         ed.FNTLeft = fl_add_pixmapbutton(FL_PUSH_BUTTON,402,53,30,30,"");
         fl_set_pixmap_data(ed.FNTLeft,ed_left_xpm);
         fl_set_object_color(ed.FNTLeft,FL_BLACK,FL_MCOL);
         ed.FNTMiddle = fl_add_pixmapbutton(FL_PUSH_BUTTON,432,53,30,30,"");
         fl_set_pixmap_data(ed.FNTMiddle,ed_middle_xpm);
         fl_set_object_color(ed.FNTMiddle,FL_BLACK,FL_MCOL);
         ed.FNTRight = fl_add_pixmapbutton(FL_PUSH_BUTTON,462,53,30,30,"");
         fl_set_pixmap_data(ed.FNTRight,ed_right_xpm);
         fl_set_object_color(ed.FNTRight,FL_BLACK,FL_MCOL);

         if (xwGetFontMode(xwGetActiveFrame())==0)
         {
            fl_set_button(ed.FNTLeft,1);
            fl_set_button(ed.FNTMiddle,0);
            fl_set_button(ed.FNTRight,0);
         }
         else if (xwGetFontMode(xwGetActiveFrame())==1)
         {
            fl_set_button(ed.FNTLeft,0);
            fl_set_button(ed.FNTMiddle,1);
            fl_set_button(ed.FNTRight,0);
         }
         else if (xwGetFontMode(xwGetActiveFrame())==2)
         {
            fl_set_button(ed.FNTLeft,0);
            fl_set_button(ed.FNTMiddle,0);
            fl_set_button(ed.FNTRight,1);
         }

         ed.FNTEditor = fl_add_input(FL_MULTILINE_INPUT,277,85,402,363+31,"");
         fl_set_input_return(ed.FNTEditor,FL_RETURN_CHANGED);
         fl_set_object_lstyle(ed.FNTEditor,FL_FIXED_STYLE);
         fl_set_object_lsize(ed.FNTEditor,FL_NORMAL_SIZE);
         strptr=xwGetFontText(xwGetActiveFrame());
         if (strptr!=NULL)
         {
            fl_set_input(ed.FNTEditor,strptr);
         }
         else
         {
            fl_set_input(ed.FNTEditor,"");
         }
      }

      fl_end_form();

      if (xwGetPrefValue(xwgWinPos)==0)
      {
         fl_prepare_form_window(ed.form,FL_PLACE_MOUSE,FL_TRANSIENT,"Editor");
      }
      else
      {
         fl_prepare_form_window(ed.form,FL_PLACE_CENTER,FL_TRANSIENT,"Editor");
      }

      fl_set_form_dblbuffer(ed.form,0);
      fl_show_form_window(ed.form);

      fl_set_app_mainform(ed.form);

      /* Exit Handler */
      fl_set_form_atclose(ed.form,nclose_cb,(void *) "1");

      xwEdView();

      ed.ready=0;
      while (ed.ready==0)
      {
         ed.obj=fl_do_forms();

         /* OK Widget */
         if (ed.obj==ed.OK) ed.ready=1;

         /* Display Help */
         if (ed.obj==ed.HELP)
         {
            xwDisplayHelp(HelpEditor);
         }

         /* Frame & Picture Settings */
         if (xwCheckFrameMode()==xwTrue)
         {
            /* Frame & Picture Settings */

            /* Frame Border Size */
            if (ed.obj==ed.FBSize) edFrameB=fl_get_counter_value(ed.FBSize);
            /* Frame Border Color */
            else if (ed.obj==ed.FBColor)
            {
               if (edFrameBC==0)
               {
                  edFrameBC=1;
                  fl_set_object_color(ed.FBColor,FL_COL1,FL_COL1);
               }
               else
               {
                  edFrameBC=0;
                  fl_set_object_color(ed.FBColor,FL_WHITE,FL_WHITE);
               }
            }
            /* Frame Line Size */
            else if (ed.obj==ed.FBLineSize) edFrameF=fl_get_counter_value(ed.FBLineSize);
            /* Frame Line Color */
            else if (ed.obj==ed.FBLColor)
            {
               edFrameC=xwColorMixer(ed.form,edCB,edFrameC);
               fl_redraw_form(ed.form);
            }
            /* Horizontal Line */
            else if (ed.obj==ed.FHLine) edHLine=fl_get_counter_value(ed.FHLine);
            /* Vertical Line */
            else if (ed.obj==ed.FVLine) edVLine=fl_get_counter_value(ed.FVLine);
            /* Orientation */
            else if (ed.obj==ed.FOrientation) edPicPosition=fl_get_choice(ed.FOrientation)-1;

            /* BorderType */
            else if (ed.obj==ed.FBLQuadBorder)
            {
               edFrameBT=0;
               fl_set_button(ed.FBLQuadBorder   ,1);
               fl_set_button(ed.FBLCircleBorder ,0);
            }
            else if (ed.obj==ed.FBLCircleBorder)
            {
               edFrameBT=1;
               fl_set_button(ed.FBLQuadBorder   ,0);
               fl_set_button(ed.FBLCircleBorder ,1);
            }
         }

         /* Picture Settings */

         /* Picture Border Size */
         if (ed.obj==ed.PBSize) edPictureB=fl_get_counter_value(ed.PBSize);
         /* Picture Border Color */
         else if (ed.obj==ed.PBColor)
         {
            if (edPictureBC==0)
            {
               edPictureBC=1;
               fl_set_object_color(ed.PBColor,FL_COL1,FL_COL1);
            }
            else
            {
               edPictureBC=0;
               fl_set_object_color(ed.PBColor,FL_WHITE,FL_WHITE);
            }
         }

         /* Picture Line Size */
         else if (ed.obj==ed.PBLineSize) edPictureF=fl_get_counter_value(ed.PBLineSize);
         /* Picture Line Color */
         else if (ed.obj==ed.PBLineColor)
         {
            edPictureC=xwColorMixer(ed.form,edCB+4,edPictureC);
            fl_redraw_form(ed.form);
         }
         /* Transparency Mode */
         else if (ed.obj==ed.PTransMode) edPictureTM=fl_get_choice(ed.PTransMode)-1;
         /* Transparency */
         else if (ed.obj==ed.PTransparenz) edPictureT=fl_get_counter_value(ed.PTransparenz);

         /* Text Settings */

         /* Font Color */
         else if (ed.obj==ed.FNTColor)
         {
            xwSetFontColor(xwGetActiveFrame(),xwColorMixer(ed.form,edCB+8,xwGetFontColor(xwGetActiveFrame())));
            fl_redraw_form(ed.form);
         }
         /* Font Select */
         else if (ed.obj==ed.FNTSelect)
         {
            fl_deactivate_form(ed.form);

            strcpy(edFontName,xwGetFontName(xwGetActiveFrame()));
            edFontSize=xwGetFontSize(xwGetActiveFrame());
            edFontLine=xwGetFontLine(xwGetActiveFrame());

            xwFontSelect(xwfSelect,edFontName,&edFontSize,&edFontLine);

            xwSetFontName(xwGetActiveFrame(),edFontName);
            xwSetFontSize(xwGetActiveFrame(),edFontSize);
            xwSetFontLine(xwGetActiveFrame(),edFontLine);

            fl_set_object_label(ed.FNTName,xwGetFontName(xwGetActiveFrame()));
            fl_set_object_label(ed.FNTSize,IntStr(xwGetFontSize(xwGetActiveFrame())));
            if (xwGetFontLine(xwGetActiveFrame())==0)
            {
               fl_set_object_label(ed.FNTUnderline,"-----");
            }
            else
            {
               fl_set_object_label(ed.FNTUnderline,xwGR(1,"ed.Underline","Underline"));
            }

            fl_activate_form(ed.form);
         }
         /* FNTEast */
         else if (ed.obj==ed.FNTEast)
         {
            if (xwIsPaperLS()==xwLandscape)
            {
               xwSetFontDirection(xwGetActiveFrame(),3); /* 1 */
            }
            else
            {
               xwSetFontDirection(xwGetActiveFrame(),0);
            }
            fl_set_button(ed.FNTEast,1);
            fl_set_button(ed.FNTSouth,0);
            fl_set_button(ed.FNTWest,0);
            fl_set_button(ed.FNTNorth,0);
         }
         /* FNTSouth */
         else if (ed.obj==ed.FNTSouth)
         {
            if (xwIsPaperLS()==xwLandscape)
            {
               xwSetFontDirection(xwGetActiveFrame(),0); /* 2 */
            }
            else
            {
               xwSetFontDirection(xwGetActiveFrame(),1);
            }
            fl_set_button(ed.FNTEast,0);
            fl_set_button(ed.FNTSouth,1);
            fl_set_button(ed.FNTWest,0);
            fl_set_button(ed.FNTNorth,0);
         }
         /* FNTWest */
         else if (ed.obj==ed.FNTWest)
         {
            if (xwIsPaperLS()==xwLandscape)
            {
               xwSetFontDirection(xwGetActiveFrame(),1); /* 3 */
            }
            else
            {
               xwSetFontDirection(xwGetActiveFrame(),2);
            }
            fl_set_button(ed.FNTEast,0);
            fl_set_button(ed.FNTSouth,0);
            fl_set_button(ed.FNTWest,1);
            fl_set_button(ed.FNTNorth,0);
         }
         /* FNTNorth */
         else if (ed.obj==ed.FNTNorth)
         {
            if (xwIsPaperLS()==xwLandscape)
            {
               xwSetFontDirection(xwGetActiveFrame(),2); /* 0 */
            }
            else
            {
               xwSetFontDirection(xwGetActiveFrame(),3);
            }
            fl_set_button(ed.FNTEast,0);
            fl_set_button(ed.FNTSouth,0);
            fl_set_button(ed.FNTWest,0);
            fl_set_button(ed.FNTNorth,1);
         }
         /* FNTLeft */
         else if (ed.obj==ed.FNTLeft)
         {
            xwSetFontMode(xwGetActiveFrame(),0);
            fl_set_button(ed.FNTLeft,1);
            fl_set_button(ed.FNTMiddle,0);
            fl_set_button(ed.FNTRight,0);
         }
         /* FNTMiddle */
         else if (ed.obj==ed.FNTMiddle)
         {
            xwSetFontMode(xwGetActiveFrame(),1);
            fl_set_button(ed.FNTLeft,0);
            fl_set_button(ed.FNTMiddle,1);
            fl_set_button(ed.FNTRight,0);
         }
         /* FNTRight */
         else if (ed.obj==ed.FNTRight)
         {
            xwSetFontMode(xwGetActiveFrame(),2);
            fl_set_button(ed.FNTLeft,0);
            fl_set_button(ed.FNTMiddle,0);
            fl_set_button(ed.FNTRight,1);
         }
         /* TNTEditor */
         else if (ed.obj==ed.FNTEditor)
         {
            strptr=fl_get_input(ed.FNTEditor);
            if (strptr!=NULL)
            {
               xwSetFontText(xwGetActiveFrame(),strptr);
            }
         }

         xwEdView();
      }

      fl_hide_form(ed.form);

      /* Frame & Picture Settings */
      if (xwCheckFrameMode()==xwTrue)
      {
         fl_free_pixmapbutton_pixmap(ed.FNTEast);
         fl_free_pixmapbutton_pixmap(ed.FNTSouth);
         fl_free_pixmapbutton_pixmap(ed.FNTWest);
         fl_free_pixmapbutton_pixmap(ed.FNTNorth);
         fl_free_pixmapbutton_pixmap(ed.FNTLeft);
         fl_free_pixmapbutton_pixmap(ed.FNTMiddle);
         fl_free_pixmapbutton_pixmap(ed.FNTRight);
      }
      fl_free_pixmapbutton_pixmap(ed.FBLQuadBorder);
      fl_free_pixmapbutton_pixmap(ed.FBLCircleBorder);
      /*fl_free_pixmapbutton_pixmap(ed.FBLAllBorder);*/

      fl_free_form(ed.form);

      ed.form=NULL;

      fl_set_app_mainform(xwGetMainGui());

      /* OK Pressed */
      if (ed.ready==1)
      {
         /* Frame & Picture Settings */
         if (xwCheckFrameMode()==xwTrue)
         {
            /* Get Frame & Picture Datas */
            xwSetFrameB(xwGetActiveFrame(),edFrameB);
            xwSetFrameBC(xwGetActiveFrame(),edFrameBC);
            xwSetFrameF(xwGetActiveFrame(),edFrameF);
            xwSetFrameC(xwGetActiveFrame(),edFrameC);
            xwSetHLine(xwGetActiveFrame(),edHLine);
            xwSetVLine(xwGetActiveFrame(),edVLine);
            xwSetFrameBT(xwGetActiveFrame(),edFrameBT);
            xwSetPicPosition(xwGetActiveFrame(),edPicPosition);

            xwSetFramePB(xwGetActiveFrame(),edPictureB);
            xwSetFramePBC(xwGetActiveFrame(),edPictureBC);
            xwSetFramePF(xwGetActiveFrame(),edPictureF);
            xwSetFramePC(xwGetActiveFrame(),edPictureC);
            xwSetFramePTM(xwGetActiveFrame(),edPictureTM);
            xwSetFramePT(xwGetActiveFrame(),edPictureT);
         }
         else
         {
            xwSetPictureB(xwGetActivePicture(),edPictureB);
            xwSetPictureBC(xwGetActivePicture(),edPictureBC);
            xwSetPictureF(xwGetActivePicture(),edPictureF);
            xwSetPictureC(xwGetActivePicture(),edPictureC);
            xwSetPictureTM(xwGetActivePicture(),edPictureTM);
            xwSetPictureT(xwGetActivePicture(),edPictureT);
         }
      }


      xwActivateMain();

   }

   return;
}

/* ################################################################### */
/* Build GFX-Output */

/* Refresh Handler */
int xwEdHandler(FL_OBJECT *obj,int event,FL_Coord mx,FL_Coord my,int key,void *xev) /*fold00*/
{
   if (xwEdHandlerLock==0)
   {
      xwEdHandlerLock=1;

      if (event==FL_DRAW) xwEdView();

      xwEdHandlerLock=0;
   }

   return(0);
}


/* Display Actual Settings */
void xwEdView(void) /*fold00*/
{
   int      xoffset=edDrawOffset;
   int      yoffset=edDrawOffset;
   float    aspect;

   int      xr,yr,wr,hr;/* Picture Area */

   int      x=0,y=0,w=0,h=0;    /* Width and Height in Pixel */
   float    lineFactor;
   int      t;          /* Line Measure */

   if (ed.form!=NULL)
   {
      xr=0;
      yr=0;
      wr=edDrawBox;
      hr=edDrawBox;

       if (xwCheckFrameMode()==xwTrue)
      {
         if (edFrameH>edFrameW)
         {
            lineFactor=(float) edDrawBox/(float) edFrameH;
         }
         else
         {
            lineFactor=(float) edDrawBox/(float) edFrameW;
         }
      }
      else
      {
         if (edPictureH>edPictureW)
         {
            lineFactor=(float) edDrawBox/(float) edPictureH;
         }
         else
         {
            lineFactor=(float) edDrawBox/(float) edPictureW;
         }
      }

      fl_winset(ed.form->window);

      /* Clear Area */
      fl_rectf(0+xoffset,0+yoffset,edDrawBox,edDrawBox,FL_COL1);

   /* ################################################################### */
   /* Display Frame */

      if (xwCheckFrameMode()==xwTrue)
      {

         /* Draw Frame White Frame */
         if ((edFrameB>0) && (edFrameB>edFrameF))
         {
            aspect=(float) edFrameW / (float) edFrameH;

            /* W<H */
            if (aspect<1)
            {
               h=edDrawBox;
               w=(int) ((float) h*aspect);
               y=0;
               x=(edDrawBox-w)/2;
            }
            else if (aspect>=1)
            {
               w=edDrawBox;
               h=(int) ((float) w/aspect);
               x=0;
               y=(edDrawBox-h)/2;
            }

            xwDrawFBox(x,y,w,h,FL_WHITE);
         }

         /* Draw Frame Border */

         /* Draw Frame Area */
         if ((edFrameB>0) || (edFrameF>0))
         {
            aspect=(float) edFrameW / (float) edFrameH;
            /* W<H */
            if (aspect<1)
            {
               h=edDrawBox;
               w=(int) ((float) h*aspect);
               y=0;
               x=(edDrawBox-w)/2;
            }
            else if (aspect>1)
            {
               w=edDrawBox;
               h=(int) ((float) w/aspect);
               x=0;
               y=(edDrawBox-h)/2;
            }
            else if (aspect==1)
            {
               w=edDrawBox;
               h=w;
               x=0;
               y=0;
            }

            if (edFrameB>edFrameF)
            {
               t=(int) (lineFactor*edFrameB);
            }
            else
            {
               t=(int) (lineFactor*edFrameF);
            }

            xr=x+t;
            yr=y+t;
            wr=w-(2*t);
            hr=h-(2*t);

            xwDrawFBox(xr,yr,wr,hr,FL_MCOL);
         }
         else
         {
            aspect=(float) edFrameW / (float) edFrameH;

            /* W<H */
            if (aspect<1)
            {
               h=edDrawBox;
               w=(int) ((float) h*aspect);
               y=0;
               x=(edDrawBox-w)/2;
            }
            else if (aspect>=1)
            {
               w=edDrawBox;
               h=(int) ((float) w/aspect);
               x=0;
               y=(edDrawBox-h)/2;
            }
            xwDrawFBox(x,y,w,h,FL_MCOL);

            /* Picture max. Size */
            xr=x;
            yr=y;
            wr=w;
            hr=h;
         }
      }
      else
      {
         xr=0;
         yr=0;
         wr=edDrawBox;
         hr=edDrawBox;
      }

   /* ################################################################### */
   /* Display Picture */

      if ((edPictureW>0) && (edPictureH>0))
      {

         /* Draw Picture Background */
         if ((edPictureB>0) && (edPictureB>edPictureF))
         {
            if ((edPictureR==-90) || (edPictureR==90))
            {
               aspect=(float) edPictureW / (float) edPictureH;
            }
            else
            {
               aspect=(float) edPictureH / (float) edPictureW;
            }

            /* W<H */
            if (aspect>1)
            {
               w=wr;
               h=(int) ((float) w*aspect);
               if (h>hr)
               {
                  h=hr;
                  w=(int) ((float) hr/aspect);
               }
            }
            else if (aspect<1)
            {
               h=hr;
               w=(int) ((float) hr/aspect);
               if (w>wr)
               {
                  w=wr;
                  h=(int) ((float) wr*aspect);
               }
            }
            else if (aspect==1)
            {
               w=wr;
               h=w;
               if (h>hr)
               {
                  h=hr;
                  w=h;
               }
            }

            xwSetPicInFrame(w,h,wr,hr);

            xwDrawFBox(xr+edX,yr+edY,w,h,FL_WHITE);
         }

         /* Draw Picture Border */
         if (edPictureF>0)
         {
            if ((edPictureR==-90) || (edPictureR==90))
            {
               aspect=(float) edPictureW / (float) edPictureH;
            }
            else
            {
               aspect=(float) edPictureH / (float) edPictureW;
            }

            /* W<H */
            if (aspect>1)
            {
               w=wr;
               h=(int) ((float) w*aspect);
               if (h>hr)
               {
                  h=hr;
                  w=(int) ((float) hr/aspect);
               }
            }
            else if (aspect<1)
            {
               h=hr;
               w=(int) ((float) hr/aspect);
               if (w>wr)
               {
                  w=wr;
                  h=(int) ((float) wr*aspect);
               }
            }
            else if (aspect==1)
            {
               w=wr;
               h=w;
               if (h>hr)
               {
                  h=hr;
                  w=h;
               }
            }
            t=(int) (lineFactor*edPictureF);

            xwSetPicInFrame(w,h,wr,hr);

            xwDrawBox(xr+edX,yr+edY,w,h,edCB+4,t);
         }

         /* Draw Picture */
         if ((edPictureB>0) || (edPictureF>0))
         {
            if ((edPictureR==-90) || (edPictureR==90))
            {
               aspect=(float) edPictureW / (float) edPictureH;
            }
            else
            {
               aspect=(float) edPictureH / (float) edPictureW;
            }

            /* W<H */
            if (aspect>1)
            {
               w=wr;
               h=(int) ((float) w*aspect);
               if (h>hr)
               {
                  h=hr;
                  w=(int) ((float) hr/aspect);
               }
            }
            else if (aspect<1)
            {
               h=hr;
               w=(int) ((float) hr/aspect);
               if (w>wr)
               {
                  w=wr;
                  h=(int) ((float) wr*aspect);
               }
            }
            else if (aspect==1)
            {
               w=wr;
               h=w;
               if (h>hr)
               {
                  h=hr;
                  w=h;
               }
            }

            if (edPictureB>edPictureF)
            {
               t=(int) (lineFactor*edPictureB);
            }
            else
            {
               t=(int) (lineFactor*edPictureF);
            }

            xwSetPicInFrame(w,h,wr,hr);

            x=edX+t;
            y=edY+t;
            w=w-(2*t);
            h=h-(2*t);

            xwDrawFBox(xr+x,yr+y,w,h,FL_WHEAT);
         }
         else
         {
            /* Draw Picture */
            if ((edPictureR==-90) || (edPictureR==90))
            {
               aspect=(float) edPictureW / (float) edPictureH;
            }
            else
            {
               aspect=(float) edPictureH / (float) edPictureW;
            }

            /* W<H */
            if (aspect>1)
            {
               w=wr;
               h=(int) ((float) w*aspect);
               if (h>hr)
               {
                  h=hr;
                  w=(int) ((float) hr/aspect);
               }
            }
            else if (aspect<1)
            {
               h=hr;
               w=(int) ((float) hr/aspect);
               if (w>wr)
               {
                  w=wr;
                  h=(int) ((float) wr*aspect);
               }
            }
            else if (aspect==1)
            {
               w=wr;
               h=w;
               if (h>hr)
               {
                  h=hr;
                  w=h;
               }
            }

            xwSetPicInFrame(w,h,wr,hr);

            xwDrawFBox(xr+edX,yr+edY,w,h,FL_WHEAT);
         }
      }


      if (xwCheckFrameMode()==xwTrue)
      {
         /* Draw Frame Border */
         if (edFrameF>0)
         {
            aspect=(float) edFrameW / (float) edFrameH;

            /* W<H */
            if (aspect<1)
            {
               h=edDrawBox;
               w=(int) ((float) h*aspect);
               y=0;
               x=(edDrawBox-w)/2;
            }
            else if (aspect>=1)
            {
               w=edDrawBox;
               h=(int) ((float) w/aspect);
               x=0;
               y=(edDrawBox-h)/2;
            }

            t=(int) (lineFactor*edFrameF);
            if ((edFrameF>0) && (t==0)) t=1;

            if (edFrameBT==0)
            {
               xwDrawBox(x,y,w,h,edCB,t);
            }
            else
            {
               xwDrawCircle(x,y,w,h,edCB,t);
            }
         }
      }

   }

   return;
}

/* Set Picture Position in Frame */
void xwSetPicInFrame(int w,int h,int wr,int hr) /*fold00*/
{
   if (xwCheckFrameMode()==xwTrue)
   {
      if (edPicPosition==frmCenter)
      {
         edX=(wr-w)/2;
         edY=(hr-h)/2;
      }
      else if (edPicPosition==frmNWPos)
      {
         edX=0;
         edY=0;
      }
      else if (edPicPosition==frmNEPos)
      {
         edX=wr-w;
         edY=0;
      }
      else if (edPicPosition==frmSWPos)
      {
         edX=0;
         edY=hr-h;;
      }
      else if (edPicPosition==frmSEPos)
      {
         edX=wr-w;
         edY=hr-h;
      }
   }
   else
   {
      edX=0;
      edY=0;
   }

   return;
}

/* Draw Filled Box */
void xwDrawFBox(int x,int y,int w,int h,int col) /*fold00*/
{
   int         xoffset=edDrawOffset;
   int         yoffset=edDrawOffset;

   int         xx,yy,ww,hh;

   xx=x;
   yy=y;
   ww=w;
   hh=h;

   if (xwIsPaperLS()==xwLandscape)
   {
      /* Landscape */
      fl_rectf((xoffset+edDrawBox)-(yy+hh),yoffset+xx,hh,ww,col);
   }
   else
   {
      /* Portrait */
      fl_rectf(xx+xoffset,yy+yoffset,ww,hh,col);
   }

   return;
}

/* Draw Box */
void xwDrawBox(int x,int y,int w,int h,int col,int t) /*fold00*/
{
   int         xoffset=edDrawOffset;
   int         yoffset=edDrawOffset;

   int         xx,yy,ww,hh,tt;
   int         swap;

   xx=x;
   yy=y;
   ww=w-1;
   hh=h-1;
   tt=t;

   if (xwIsPaperLS()==xwLandscape)
   {
      /* Landscape */
      for (swap=0 ; swap<=tt ; swap++)
      {
         fl_winset(ed.form->window);
         fl_rect(((xoffset+edDrawBox)-(yy+hh))+swap,(yoffset+xx)+swap,hh-(2*swap),ww-(2*swap),col);
      }
   }
   else
   {
      /* Portrait */
      for (swap=0 ; swap<=tt ; swap++)
      {
         fl_winset(ed.form->window);
         fl_rect((xx+xoffset)+swap,(yy+yoffset)+swap,ww-(2*swap),hh-(2*swap),col);
      }
   }

   return;
}

/* Draw Circle */
void xwDrawCircle(int x,int y,int w,int h,int col,int t)
{
   int         xoffset=edDrawOffset;
   int         yoffset=edDrawOffset;

   int         xx,yy,ww,hh,tt;
   int         swap;


   int                  xp,yp,rr;
   float                grad;
   int                  xr,yr;

   xx=x;
   yy=y;
   ww=w-1;
   hh=h-1;
   tt=t;

   /* Calculate Rotation Datas */
   xp=ww/2;
   yp=hh/2;
   rr=xp;
   if (xp>yp) rr=yp;

   if (xwIsPaperLS()==xwLandscape)
   {
      /* Landscape */
      fl_winset(ed.form->window);

      /* Draw Circle Border */
      for (grad=0 ; grad<360 ; grad=grad+0.075)
      {
         for (swap=rr ; swap>rr-tt ; swap--)
         {
            xr=xoffset+yp+(int) (sin(grad)*(float) swap);
            yr=yoffset+xp+(int) (cos(grad)*(float) swap);
            fl_point(xr,yr,col);
         }
      }
   }
   else
   {
      /* Portrait */
      fl_winset(ed.form->window);

      /* Draw Circle Border */
      for (grad=0 ; grad<360 ; grad=grad+0.075)
      {
         for (swap=rr ; swap>rr-tt ; swap--)
         {
            xr=xoffset+xx+xp+(int) (sin(grad)*(float) swap);
            yr=yoffset+yy+yp+(int) (cos(grad)*(float) swap);
            fl_point(xr,yr,col);
         }
      }
   }

   return;
}
