/* xwGUI -- an X11-GUI for photo prints
 * Copyright (C) 2001 Stefan Kraus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "driver_turboprint.h"

#include "base.h"
#include "default.h"
#include "error.h"
#include "frame.h"
#include "handler.h"
#include "limits.h"
#include "page.h"
#include "paper.h"
#include "picture.h"
#include "resource.h"
#include "variable.h"

/* Printer List */
static int        tpAnzPrinter=-1;           /* count of available printers */
static int        tpDefaultPrinter=-1;       /* default printer */
static char       *tpPrinter[1024];          /* list of printers */

/* Quality List */
static int        tpAnzQuality=-1;           /* count of available qualitys */
static char       *tpQuality[1024];          /* list of quality */

/* Paper List */
static int        tpAnzPaper=-1;             /* count of available paper */
static char       *tpPaper[1024];            /* list of paper */

/* Paper Medium */
static int        tpAnzMedia=-1;             /* count of available media */
static char       *tpMedia[1024];            /* list of media */

/* Paper Datas */
static char       tpID[256];                 /* paper id */
static char       tpName[256];               /* paper name */

/* Paper Size */
static float      tpXSize,tpYSize;           /* paper size */
static float      tpLB,tpRB,tpTB,tpBB;       /* paper borders */

/* Remove All */
void tpRemove(void) /*fold00*/
{
   int            az;

   /* remove old printer list */
   if (tpAnzPrinter>-1)
   {
      for (az=0 ; az<=tpAnzPrinter ; az++)
      {
         free(tpPrinter[az]);
         tpPrinter[az]=NULL;
      }
      tpAnzPrinter=-1;
   }

   /* remove old quality list */
   if (tpAnzQuality>-1)
   {
      for (az=0 ; az<=tpAnzQuality ; az++)
      {
         free(tpQuality[az]);
         tpQuality[az]=NULL;
      }
      tpAnzQuality=-1;
   }

   /* remove old paper list */
   if (tpAnzPaper>-1)
   {
      for (az=0 ; az<=tpAnzPaper ; az++)
      {
         free(tpPaper[az]);
         tpPaper[az]=NULL;
      }
      tpAnzPaper=-1;
   }
   
   /* emove old media list */
   if (tpAnzMedia>-1)
   {
      for (az=0 ; az<=tpAnzMedia ; az++)
      {
         free(tpMedia[az]);
         tpMedia[az]=NULL;
      }
      tpAnzMedia=-1;
   }

   return;
}

/* ################################################################### */
/* # Get TurboPrint Version                                          # */
/* ################################################################### */

int tpGetVersion(void) /*fold00*/
{
   int            result;

   char           cmd[1024];
   FILE           *fh;                       /* File Handle */
   char           line[1024];                /* reading line */

   char           *strptr;

   float          ver;                       /* Version */

   /* ask turboprint for installed printer */
   strcpy(cmd,"/usr/bin/tpprint -h");
   xwExec(cmd,-1,xwGetFile("","tp-output"));
   
   /* read list of printers */
   result=xwFalse;
   if ( (fh=fopen(xwGetFile("","tp-output"),"rb")) != NULL )
   {
      if (fgets(line,1024,fh) != NULL)
      {
         if (strlen(line)>0)
         {
            SetStrGet("");
            strptr=StrGet(line);
            if (strptr!=NULL)
            {
               strptr=StrGet(NULL);
               if (strlen(strptr)>0)
               {
                  strcpy(cmd,strptr);
                  ver=atof(cmd);
                  
                  if (xwGetPrefValue(xwgExpert)!=0)
                  {
                     printf("xwGUI2: TurboPrint Version %s found !\n",strptr);
                     printf("xwGUI2: Internal Version Number: %f\n",ver);
                  }
                  
                  if (ver>=(float) 1.59)
                  {
                     result=xwTrue;
                  }
               }
            }
         }
      }
      fclose(fh);

      if (result==xwFalse)
      {
         xwDisplayMessage(NULL,"You must install or update TurboPrint, please !\nTurboPrint Version >=1.59 is needed !\n");
         xwSetVarDef("TPPRINTER","");
      }
   }

   /* remove printer list */
   unlink(xwGetFile("","tp-output"));

   return(result);
}

/* ################################################################### */
/* # Display any Datas                                               # */
/* ################################################################### */

/* ################################################################### */
/* Read and List Printers */

/* List Printers */
void tpListPrinter(FL_OBJECT *obj) /*fold00*/
{
   char           cmd[1024];
   int            az;

   FILE           *fh;                       /* File Handle */
   char           line[1024];                /* reading line */
   char           lpmark[1024];
   char           *strptr;

   if (tpGetVersion()==xwTrue)
   {

      /* clear browser */
      if (obj!=NULL) fl_clear_browser(obj);

      /* remove old printer list */
      if (tpAnzPrinter>-1)
      {
         for (az=0 ; az<=tpAnzPrinter ; az++)
         {
            free(tpPrinter[az]);
            tpPrinter[az]=NULL;
         }
         tpAnzPrinter=-1;
      }

      /* ask turboprint for installed printer */
      strcpy(cmd,"/usr/bin/tpconfig --listprinter");
      xwExec(cmd,-1,xwGetFile("","tp-output"));

      /* read list of printers */
      if ( (fh=fopen(xwGetFile("","tp-output"),"rb")) != NULL )
      {
         while (fgets(line,1024,fh) != NULL)
         {
            /* remove newline character */
            if (strlen(line)>0)
            {
               if (line[strlen(line)-1]==0x0a)
               {
                  line[strlen(line)-1]=0x00;
               }
            }

            /* remove lp mark */
            if (strlen(line)>4)
            {
               strcpy(lpmark,line);
               lpmark[3]=0x00;

               if (strcmp(lpmark,"lp|") == 0)
               {
                  strcpy(lpmark,line);
                  strcpy(line,&lpmark[3]);
                  tpDefaultPrinter=tpAnzPrinter+1;
               }
            }

            /* save printer to list */
            if (strlen(line)>0)
            {
               tpAnzPrinter++;
               tpPrinter[tpAnzPrinter]=(char*) malloc(strlen(line)+1);
               if (tpPrinter[tpAnzPrinter]!=NULL)
               {
                  strcpy(tpPrinter[tpAnzPrinter],line);
               }
               else
               {
                  tpAnzPrinter--;
               }

               /* add printer to browser */
               if (obj!=NULL)
               {
                  strcpy(lpmark,"");
                  az=0;

                  SetStrGet("|");
                  strptr=StrGet(tpPrinter[tpAnzPrinter]);
                  while (strlen(strptr)>0)
                  {
                     strcpy(line,strptr);
                     strcat(line,baseSpace);

                     az++;

                     /* short */
                     if (az==1) line[5]=0x00;
                     if (az==2) line[37]=0x00;
                     if (az==3) line[37]=0x00;

                     strcat(lpmark,line);

                     strptr=StrGet(NULL);
                  }
                  SetStrGet("");

                  fl_add_browser_line(obj,lpmark);
               }
            }
         }

         fclose(fh);
      }

      /* set selected printer */
      strcpy(lpmark,xwGetVarDef("TPPRINTER"));
      if (strlen(lpmark)==0)
      {
         /* set default printer */
         if (tpDefaultPrinter>-1)
         {
            xwSetVarDef("TPPRINTER",StrGet(tpPrinter[tpDefaultPrinter]));
         }
         else
         {
            xwSetVarDef("TPPRINTER",StrGet(tpPrinter[0]));
         }
         strcpy(lpmark,xwGetVarDef("TPPRINTER"));
      }

      /* search entry and set found entry */
      if ( (tpAnzPrinter>-1) && (obj!=NULL) )
      {
         for (az=0 ; az<=tpAnzPrinter ; az++)
         {
            if (strncmp(lpmark,tpPrinter[az],strlen(lpmark)) == 0)
            {
               fl_select_browser_line(obj,az+1);
            }
         }
      }

      /* remove printer list */
      unlink(xwGetFile("","tp-output"));

      if (tpAnzPrinter>-1)
      {
         /* set quality */
         tpListQuality(NULL);
         /* set paper */
         tpListPaper(NULL);
      }
      else
      {
         xwDisplayMessage(NULL,"WARNING !!!\nYou must install an TurboPrint Printer ( tpsetup or xtpsetup ), please !!!\n");
         xwSetVarDef("TPPRINTER","");
      }
   }

   return;
}

/* ################################################################### */
/* Read and List Quality */

/* List Quality */
void tpListQuality(FL_OBJECT *obj) /*fold00*/
{
   char           cmd[1024];
   int            az;

   FILE           *fh;                       /* File Handle */
   char           line[1024];                /* reading line */
   char           lpmark[1024];
   char           *strptr;

   if (strlen(xwGetVarDef("TPPRINTER"))>0)
   {
      /* clear browser */
      if (obj!=NULL) fl_clear_browser(obj);

      /* remove old quality list */
      if (tpAnzQuality>-1)
      {
         for (az=0 ; az<=tpAnzQuality ; az++)
         {
            free(tpQuality[az]);
            tpQuality[az]=NULL;
         }
         tpAnzQuality=-1;
      }

      /* ask turboprint for installed printer */
      strcpy(cmd,"/usr/bin/tpconfig --printer=");
      strcat(cmd,xwGetVarDef("TPPRINTER"));
      strcat(cmd," --listquality");
      xwExec(cmd,-1,xwGetFile("","tp-output"));

      /* read list of printers */
      if ( (fh=fopen(xwGetFile("","tp-output"),"rb")) != NULL )
      {
         while (fgets(line,1024,fh) != NULL)
         {
            /* remove newline character */
            if (strlen(line)>0)
            {
               if (line[strlen(line)-1]==0x0a)
               {
                  line[strlen(line)-1]=0x00;
               }
            }

            /* save printer to list */
            if (strlen(line)>0)
            {
               tpAnzQuality++;
               tpQuality[tpAnzQuality]=(char*) malloc(strlen(line)+1);
               if (tpQuality[tpAnzQuality]!=NULL)
               {
                  strcpy(tpQuality[tpAnzQuality],line);
               }
               else
               {
                  tpAnzQuality--;
               }

               /* add printer to browser */
               if (obj!=NULL)
               {
                  strcpy(lpmark,"");
                  az=0;

                  SetStrGet(":");
                  strptr=StrGet(tpQuality[tpAnzQuality]);
                  while (strlen(strptr)>0)
                  {
                     strcpy(line,strptr);
                     strcat(line,baseSpace);

                     az++;

                     /* short */
                     if (az==1) line[3]=0x00;
                     if (az==2) line[21]=0x00;
                     if (az==3) line[21]=0x00;
                     if (az==4) line[21]=0x00;

                     strcat(lpmark,line);

                     strptr=StrGet(NULL);
                  }
                  SetStrGet("");

                  fl_add_browser_line(obj,lpmark);
               }
            }
         }

         fclose(fh);
      }

      /* set selected printer */
      strcpy(lpmark,xwGetVarDef("TPQUALITY"));
      if (strlen(lpmark)==0)
      {
         xwSetVarDef("TPQUALITY",StrGet(tpQuality[tpAnzQuality]));
         strcpy(lpmark,xwGetVarDef("TPQUALITY"));
      }

      /* search entry and set found entry */
      if ( (tpAnzQuality>-1) && (obj!=NULL) )
      {
         for (az=0 ; az<=tpAnzQuality ; az++)
         {
            if (strncmp(lpmark,tpQuality[az],strlen(lpmark)) == 0)
            {
               fl_select_browser_line(obj,az+1);
            }
         }
      }

      /* remove printer list */
      unlink(xwGetFile("","tp-output"));
   }
   else
   {
      /* clear browser */
      if (obj!=NULL) fl_clear_browser(obj);
   }

   return;
}

/* ################################################################### */
/* Read and List Paper */

/* List Paper */
void tpListPaper(FL_OBJECT *obj) /*fold00*/
{
   char           cmd[1024];
   int            az;

   FILE           *fh;                       /* File Handle */
   char           line[1024];                /* reading line */
   char           lpmark[1024];

   if (strlen(xwGetVarDef("TPPRINTER"))>0)
   {

      /* clear browser */
      if (obj!=NULL) fl_clear_browser(obj);

      /* remove old paper list */
      if (tpAnzPaper>-1)
      {
         for (az=0 ; az<=tpAnzPaper ; az++)
         {
            free(tpPaper[az]);
            tpPaper[az]=NULL;
         }
         tpAnzPaper=-1;
      }

      /* ask turboprint for installed printer */
      strcpy(cmd,"/usr/bin/tpconfig --printer=");
      strcat(cmd,xwGetVarDef("TPPRINTER"));
      strcat(cmd," --quality=");
      strcat(cmd,xwGetVarDef("TPQUALITY"));
      strcat(cmd," --listpagesize");
      xwExec(cmd,-1,xwGetFile("","tp-output"));

      /* read list of printers */
      if ( (fh=fopen(xwGetFile("","tp-output"),"rb")) != NULL )
      {
         while (fgets(line,1024,fh) != NULL)
         {
            /* remove newline character */
            if (strlen(line)>0)
            {
               if (line[strlen(line)-1]==0x0a)
               {
                  line[strlen(line)-1]=0x00;
               }
            }

            /* save printer to list */
            if (strlen(line)>0)
            {
               tpAnzPaper++;
               tpPaper[tpAnzPaper]=(char*) malloc(strlen(line)+1);
               if (tpPaper[tpAnzPaper]!=NULL)
               {
                  strcpy(tpPaper[tpAnzPaper],line);
               }
               else
               {
                  tpAnzPaper--;
               }

               /* add paper to browser */
               if (obj!=NULL)
               {
                  xwGetPaperDatas(tpPaper[tpAnzPaper]);

                  /* ID */
                  strcpy(line,tpID);
                  strcat(line,baseSpace);
                  line[3]=0x00;
                  strcpy(lpmark,line);

                  /* Name */
                  strcpy(line,tpName);
                  strcat(line,baseSpace);
                  line[21]=0x00;
                  strcat(lpmark,line);

                  /* Paper Size */
                  strcpy(line,strLLength(DoubleStr(tpXSize,2),7));
                  strcat(line," mm x ");
                  strcat(line,strLLength(DoubleStr(tpYSize,2),7));
                  strcat(line," mm");
                  strcat(line,baseSpace);
                  line[24]=0x00;
                  strcat(lpmark,line);

                  /* Border Size */
                  
                  strcpy(line,strLLength(DoubleStr(tpLB,2),5));
                  strcat(lpmark,line);
                  strcat(lpmark," / ");
                  strcpy(line,strLLength(DoubleStr(tpRB,2),5));
                  strcat(lpmark,line);
                  strcat(lpmark," / ");
                  strcpy(line,strLLength(DoubleStr(tpTB,2),5));
                  strcat(lpmark,line);
                  strcat(lpmark," / ");
                  strcpy(line,strLLength(DoubleStr(tpBB,2),5));
                  strcat(lpmark,line);
                  
                  
                  /* Border Size */
                  /*
                  strcpy(line,DoubleStr(tpLB,2));
                  strcat(lpmark,line);
                  strcat(lpmark," / ");
                  strcpy(line,DoubleStr(tpRB,2));
                  strcat(lpmark,line);
                  strcat(lpmark," / ");
                  strcpy(line,DoubleStr(tpTB,2));
                  strcat(lpmark,line);
                  strcat(lpmark," / ");
                  strcpy(line,DoubleStr(tpBB,2));
                  strcat(lpmark,line);
                  */
                  
                  fl_add_browser_line(obj,lpmark);
               }
            }
         }

         fclose(fh);
      }

      /* set selected printer */
      /*strcpy(lpmark,xwGetVarDef("TPPAPER"));*/
      tpSetPaperSize();
      strcpy(lpmark,xwGetVarDef("TPPAPER"));

      /* search entry and set found entry */
      if ( (tpAnzPaper>-1) && (obj!=NULL) )
      {
         for (az=0 ; az<=tpAnzPaper ; az++)
         {
            if (strncmp(lpmark,tpPaper[az],strlen(lpmark)) == 0)
            {
               fl_select_browser_line(obj,az+1);
            }
         }
      }

      /* remove printer list */
      unlink(xwGetFile("","tp-output"));
   }
   else
   {
      /* clear browser */
      if (obj!=NULL) fl_clear_browser(obj);
   }

   return;
}

/* ################################################################### */
/* Read and List Media */

/* List Media */
void tpListMedia(FL_OBJECT *obj) /*fold00*/
{
   char           cmd[1024];
   int            az;

   FILE           *fh;                       /* File Handle */
   char           line[1024];                /* reading line */
   char           lpmark[1024];
   char           *strptr;

   if (strlen(xwGetVarDef("TPPRINTER"))>0)
   {

      /* clear browser */
      if (obj!=NULL) fl_clear_browser(obj);

      /* remove old quality list */
      if (tpAnzMedia>-1)
      {
         for (az=0 ; az<=tpAnzMedia ; az++)
         {
            free(tpMedia[az]);
            tpMedia[az]=NULL;
         }
         tpAnzMedia=-1;
      }

      /* ask turboprint for installed printer */
      strcpy(cmd,"/usr/bin/tpconfig --printer=");
      strcat(cmd,xwGetVarDef("TPPRINTER"));
      strcat(cmd," --quality=");
      strcat(cmd,xwGetVarDef("TPQUALITY"));
      strcat(cmd," --listmedium");
      xwExec(cmd,-1,xwGetFile("","tp-output"));

      /* read list of printers */
      if ( (fh=fopen(xwGetFile("","tp-output"),"rb")) != NULL )
      {
         while (fgets(line,1024,fh) != NULL)
         {
            /* remove newline character */
            if (strlen(line)>0)
            {
               if (line[strlen(line)-1]==0x0a)
               {
                  line[strlen(line)-1]=0x00;
               }
            }

            /* save printer to list */
            if (strlen(line)>0)
            {
               tpAnzMedia++;
               tpMedia[tpAnzMedia]=(char*) malloc(strlen(line)+1);
               if (tpMedia[tpAnzMedia]!=NULL)
               {
                  strcpy(tpMedia[tpAnzMedia],line);
               }
               else
               {
                  tpAnzMedia--;
               }

               /* add media to browser */
               if (obj!=NULL)
               {
                  strcpy(lpmark,"");
                  az=0;

                  SetStrGet(":");
                  strptr=StrGet(tpMedia[tpAnzMedia]);
                  while (strlen(strptr)>0)
                  {
                     strcpy(line,strptr);
                     strcat(line,baseSpace);

                     az++;

                     /* short */
                     if (az==1) line[3]=0x00;
                     if (az==2) line[21]=0x00;

                     strcat(lpmark,line);

                     strptr=StrGet(NULL);
                  }
                  SetStrGet("");

                  fl_add_browser_line(obj,lpmark);
               }
            }
         }

         fclose(fh);
      }

      /* set selected media */
      strcpy(lpmark,xwGetVarDef("TPMEDIA"));
      if (strlen(lpmark)==0)
      {
         xwSetVarDef("TPMEDIA","0");
         strcpy(lpmark,xwGetVarDef("TPMEDIA"));
      }

      /* search entry and set found entry */
      if ( (tpAnzMedia>-1) && (obj!=NULL) )
      {
         for (az=0 ; az<=tpAnzMedia ; az++)
         {
            if (strncmp(lpmark,tpMedia[az],strlen(lpmark)) == 0)
            {
               fl_select_browser_line(obj,az+1);
            }
         }
      }

      /* remove printer list */
      unlink(xwGetFile("","tp-output"));
   }
   else
   {
      /* clear browser */
      if (obj!=NULL) fl_clear_browser(obj);
   }

   return;
}

/* ################################################################### */
/* Paper Management Routines */

/* Paper Settings */
void tpSetPaperSize(void) 
{
   char        astr[1024];
   int         az,bz;

   float       pxs,pys;

   /* search entry and set found entry */
   if (tpAnzPaper>-1)
   {
      bz=-1;

      /* paper is selected */
      strcpy(astr,xwGetVarDef("TPPAPER"));
      if (strlen(astr)>0)
      {
         /* search entry and set datas */
         bz=-1;
         for (az=0 ; az<=tpAnzPaper ; az++)
         {
            if (strncmp(astr,tpPaper[az],strlen(astr)) == 0)
            {
               bz=az;
            }
         }
      }
      else
      {
         /* search entry and set datas */
         bz=-1;
         
         pxs=999999;
         pys=999999;
         for (az=0 ; az<=tpAnzPaper ; az++)
         {
            xwGetPaperDatas(tpPaper[az]);

            if ((tpXSize-210<=pys) && (tpYSize-297<=pxs) && (tpXSize>=210) && (tpYSize>=297))
            {
               
               bz=az;
               pys=tpYSize-297;
               pxs=tpXSize-210;               
            }
         }
      }

      /* set entry */
      if (bz>-1)
      {
         /* parse paper setting and set datas */
         xwGetPaperDatas(tpPaper[bz]);

         xwSetVarDef("TPPAPER",tpID);

         /* set paper datas */
         xwSetPaperX( (tpXSize*(float) xwMM()));
         xwSetPaperY( (tpYSize*(float) xwMM()));

         xwSetPaperLB(IRound( tpLB*(float) xwMM()));
         xwSetPaperRB(IRound( tpRB*(float) xwMM()));
         xwSetPaperTB(IRound( tpTB*(float) xwMM()));
         xwSetPaperBB(IRound( tpBB*(float) xwMM()));

         xwSetPaperSize(998,tpXSize*(float) xwMM(),tpYSize*(float) xwMM());

         /* ReDraw Page */
         xwDrawPage(pgRecalc);
      }
   }

   return;
}

/* Set Correct Paper Size */
void tpSetCorrectPaperSize(int xsize,int ysize) /*fold00*/
{
   int         az,bz;

   float       fxz,fyz;             /* minimal distance */
   int         fxa,fya;             /* paper entry */
   
   tpListPaper(NULL);
   
   /* search entry and set found entry */ 
   if (tpAnzPaper>-1)
   {   
      /* search entry and set datas */
      bz=-1;
      fxz=10000;
      fyz=10000;
      fxa=-1;
      fya=-1;
      
      for (az=0 ; az<=tpAnzPaper ; az++)
      {
         xwGetPaperDatas(tpPaper[az]);

         /* page greater source page */
         if ((tpXSize-((float) xsize/10)>=0) && (tpYSize-((float) ysize/10)>=0))
         {
            if ( (tpXSize-((float) xsize/10)<fxz) && (tpXSize-((float) xsize/10)>=0) )
            {
               fxz=tpXSize-((float) xsize/10);
               fxa=az;
            }
            
            if ( (tpYSize-((float) ysize/10)<fyz) &&  (tpYSize-((float) ysize/10)>=0) )
            {
               fyz=tpYSize-((float) ysize/10);
               fya=az;
            }            
         }
      }      
   
      if (fxz<fyz)
      {
         bz=fxa;
      }
      else
      {
         bz=fya;
      }      
   
      /* correct size found */
      if (bz>-1)
      {
         /* parse paper setting and set datas */
         xwGetPaperDatas(tpPaper[bz]);

         xwSetVarDef("TPPAPER",tpID);

         /* set paper datas */
         xwSetPaperX( (tpXSize*(float) xwMM()));
         xwSetPaperY( (tpYSize*(float) xwMM()));

         xwSetPaperLB(IRound( tpLB*(float) xwMM()));
         xwSetPaperRB(IRound( tpRB*(float) xwMM()));
         xwSetPaperTB(IRound( tpTB*(float) xwMM()));
         xwSetPaperBB(IRound( tpBB*(float) xwMM()));

         xwSetPaperSize(998,tpXSize*(float) xwMM(),tpYSize*(float) xwMM());

         /* ReDraw Page */
         xwDrawPage(pgRecalc);     
      }   
   }
   else
   {
      printf("xwGUI2: Sorry, no TurboPrint Papers found !!!\n");
   }

   return;
}

/* Get Paper Datas */
void xwGetPaperDatas(char *source) /*fold00*/
{
   char        *strptr;
   int         az;
   char        line[1024];                /* reading line */


   char        paperStr[1024];
   char        borderStr[1024];

   if (strlen(source)>0)
   {

      /* cut string in usefull segments */
      az=0;
      SetStrGet(":");
      strptr=StrGet(source);
      while (strlen(strptr)>0)
      {
         strcpy(line,strptr);
         strcat(line,baseSpace);

         az++;

         /* short */
         if (az==1) strcpy(tpID,strptr);
         else if (az==2) strcpy(tpName,strptr);
         else if (az==3) strcpy(paperStr,strptr);
         else if (az==4) strcpy(borderStr,strptr);

         strptr=StrGet(NULL);
      }
      SetStrGet("");

      /* PaperSize */
      if (strlen(paperStr)>0)
      {
         SetStrGet("x");
         strptr=StrGet(paperStr);
         if (strlen(strptr)>0)
         {
            tpXSize=atof(strptr)/1000; /* Inch */
            strptr=StrGet(NULL);
            if (strlen(strptr)>0)
            {
               tpYSize=atof(strptr)/1000; /* Inch */
            }
            else
            {
               tpYSize=11.693;   /* A4 Size */
            }
         }
         else
         {
            tpXSize=8.268;
            tpYSize=11.693;
         }

         tpXSize=tpXSize*25.4;
         tpYSize=tpYSize*25.4;
         
         /* Round Float Value */
         tpXSize=(float) (((int) (tpXSize*100))/100);
         tpYSize=(float) (((int) (tpYSize*100))/100);
         
      }

      /* Borders */
      if (strlen(borderStr)>0)
      {
         az=0;
         SetStrGet(",");
         strptr=StrGet(borderStr);
         while (strlen(strptr)>0)
         {
            az++;

            /* short */
            if (az==1)
            {
               tpLB=(atof(strptr)/1000)*25.4;
               tpLB=(float) (((int) (tpLB*100))/100);
            }
            if (az==2)
            {
               tpRB=(atof(strptr)/1000)*25.4;
               tpRB=(float) (((int) (tpRB*100))/100);
            }
            if (az==3)
            {
               tpTB=(atof(strptr)/1000)*25.4;
               tpTB=(float) (((int) (tpTB*100))/100);
            }
            if (az==4)
            {
               tpBB=(atof(strptr)/1000)*25.4;
               tpBB=(float) (((int) (tpBB*100))/100);
            }
            strptr=StrGet(NULL);
         }
         SetStrGet("");
      }

   }

   return;
}

/* ################################################################### */
/* # Set Actual Selections                                           # */
/* ################################################################### */

/* Get Printer */
void tpGetPrinter(FL_OBJECT *obj) /*fold00*/
{
   int            az;

   /* ask user for delete existing page */
   if ( (xwGetFrameCount()>-1) || (xwGetPictureCount()>-1) )
   {
      if (xwAskMessage(NULL,xwGR(1,"msg.Trash","Delete all frames and pictures ?"))==xwYesMessage)
      {
         xwFreePicture(-1);
         xwInitFrames();
         xwInitPicture();
         /* ReDraw Page */
         xwDrawPage(pgRecalc);

         if (obj!=NULL)
         {
            az=fl_get_browser(obj)-1;
            if (az>-1)
            {
               /* set selected printer */
               xwSetVarDef("TPPRINTER",StrGet(tpPrinter[az]));

               /* set maximum quality */
               tpListQuality(NULL);
               xwSetVarDef("TPQUALITY",StrGet(tpQuality[tpAnzQuality]));

               /* set default paper size */
               xwSetVarDef("TPPAPER","");
               tpListPaper(NULL);
            }
         }
      }
      else
      {
         if (obj!=NULL) tpListPrinter(obj);
      }
   }
   else
   {
      if (obj!=NULL)
      {
         az=fl_get_browser(obj)-1;
         if (az>-1)
         {
            /* set selected printer */
            xwSetVarDef("TPPRINTER",StrGet(tpPrinter[az]));

            /* set maximum quality */
            tpListQuality(NULL);
            xwSetVarDef("TPQUALITY",StrGet(tpQuality[tpAnzQuality]));

            /* set default paper size */
            xwSetVarDef("TPPAPER","");
            tpListPaper(NULL);
            
            /* set default media */
            xwSetVarDef("TPMEDIA","");
            tpListMedia(NULL);

         }
      }
   }

   return;
}

/* Get Quality */
void tpGetQuality(FL_OBJECT *obj) /*fold00*/
{
   int         az;

   if (obj!=NULL)
   {
      az=fl_get_browser(obj)-1;
      if (az>-1)
      {
         /* set selected printer */
         xwSetVarDef("TPQUALITY",StrGet(tpQuality[az]));
      }
   }

   return;
}

/* Get Media */
void tpGetMedia(FL_OBJECT *obj) /*fold00*/
{
   int         az;

   if (obj!=NULL)
   {
      az=fl_get_browser(obj)-1;
      if (az>-1)
      {
         /* set selected printer */
         xwSetVarDef("TPMEDIA",StrGet(tpMedia[az]));
      }
   }

   return;
}

/* Get Paper */
void tpGetPaper(FL_OBJECT *obj) /*fold00*/
{
   int         az;

   /* ask user for delete existing page */
   if ( (xwGetFrameCount()>-1) || (xwGetPictureCount()>-1) )
   {
      if (xwAskMessage(NULL,xwGR(1,"msg.Trash","Delete all frames and pictures ?"))==xwYesMessage)
      {
         xwFreePicture(-1);
         xwInitFrames();
         xwInitPicture();
         /* ReDraw Page */
         xwDrawPage(pgRecalc);

         if (obj!=NULL)
         {
            az=fl_get_browser(obj)-1;
            if (az>-1)
            {
               /* set selected printer */
               xwSetVarDef("TPPAPER",StrGet(tpPaper[az]));
               tpListPaper(NULL);
            }
         }
      }
      else
      {
         if (obj!=NULL) tpListPaper(obj);
      }
   }
   else
   {
      if (obj!=NULL)
      {
         az=fl_get_browser(obj)-1;
         if (az>-1)
         {
            /* set selected printer */
            xwSetVarDef("TPPAPER",StrGet(tpPaper[az]));
            tpListPaper(NULL);
         }
      }
   }

   return;
}

/* Create TurboPrint Script */
void tpPrintScript(void) /*fold00*/
{
   FILE           *fh;                       /* File Handle */
   char           line[1024];                /* reading line */
   char           *strptr;
   char           cmd[1024];                 /* command string */
   char           aas[1024],bbs[1024];

   /* ask turboprint for installed printer */
   strcpy(cmd,"/usr/bin/tpconfig --printer=");
   strcat(cmd,xwGetVarDef("TPPRINTER"));
   strcat(cmd," --getcommandline");
   xwExec(cmd,-1,xwGetFile("","tp-output"));

   /* read list of printers */
   if ( (fh=fopen(xwGetFile("","tp-output"),"rb")) != NULL )
   {
      if (fgets(line,1024,fh) != NULL)
      {
         fclose(fh);

         if (strlen(line)>0)
         {
            line[strlen(line)-1]=0x00;

            strcpy(cmd,"tpprint ");

            SetStrGet(" ");
            strptr=StrGet(line);
            while (strlen(strptr)>0)
            {
               strcpy(aas,strptr);
               strcpy(bbs,aas);
               bbs[2]=0x00;

               if (strcmp(bbs,"-q") == 0)
               {
                  /* Quality */
                  strcat(cmd,"-q");
                  strcat(cmd,xwGetVarDef("TPQUALITY"));
                  strcat(cmd," ");

               }
               else if (strcmp(bbs,"-f") == 0)
               {
                  /* Paperformat */
                  strcat(cmd,"-f");
                  strcat(cmd,xwGetVarDef("TPPAPER"));
                  strcat(cmd," ");
               }
               else if (strcmp(bbs,"-m") == 0)
               {
                  /* Media */
                  strcat(cmd,"-m");
                  strcat(cmd,xwGetVarDef("TPMEDIA"));
                  strcat(cmd," ");
               }
               
               else
               {
                  strcat(cmd,aas);
                  strcat(cmd," ");
               }

               strptr=StrGet(NULL);
            }
            SetStrGet("");
            
            /* Setup B-Flags */
            strptr=strstr(cmd,"-b");
            if (strptr!=NULL)
            {
               strptr[0]=0x00;
               
               strcat(cmd,"-bt");
               strcat(cmd,IntStr(xwGetIVarDef("TPTRUEMATCH")));
               strcat(cmd,"b");
               strcat(cmd,IntStr(xwGetIVarDef("TPBRIGHTNESS")));
               strcat(cmd,"o");
               strcat(cmd,IntStr(xwGetIVarDef("TPCONTRAST")));
               strcat(cmd,"s");
               strcat(cmd,IntStr(xwGetIVarDef("TPSATURATION")));
               strcat(cmd,"i");
               strcat(cmd,IntStr(xwGetIVarDef("TPINKVOLUME")));
               strcat(cmd,"g");
               strcat(cmd,IntStr(xwGetIVarDef("TPSCREENGAMMA")));
               strcat(cmd,"y");
               strcat(cmd,IntStr(xwGetIVarDef("TPYELLOW")));
               strcat(cmd,"m");
               strcat(cmd,IntStr(xwGetIVarDef("TPMAGENTA")));
               strcat(cmd,"c");
               strcat(cmd,IntStr(xwGetIVarDef("TPCYAN")));
               strcat(cmd,"k");
               strcat(cmd,IntStr(xwGetIVarDef("TPBLACK")));
               strcat(cmd," "); 
            }
            
            strcat(cmd,xwGetVarDef("FILE"));
            strcat(cmd," ");
            strcat(cmd,xwGetVarDef("TPDEVICE"));

            /*printf(">%s<\n",cmd);*/

            /* create working script */
            if (strlen(cmd)>8)
            {
               fh=fopen(xwGetFile("","working.script"),"w");
               if (fh!= NULL)
               {
                  fputs(cmd,fh);
                  fputs("\n",fh);
                  fclose(fh);
               }
            }
         }
      }
   }

   /* remove commandline */
   unlink(xwGetFile("","tp-output"));

   return;
}