// pslabels - Place one or different eps files on a page for label printing.
// Copyright (C) 1998, 1999 Per Arne Karlsen <perk@consilio.oslo.no>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// See the file COPYING for the GNU General Public License

// Creation date: 1998/05/09

// Compiler used: gcc egcs-2.90.29 with libstdc++ 2.8
// Some versions of libstdc++ 2.7.x.y may not work because of broken
// stream output of floating point values.

static const char rcsid[] =
"$Id: pslabels.cc,v 1.4 1999/01/03 17:04:25 perk Exp $";

#include <string>
#include <vector>
#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "psprolog.h"

// all sizes are in 1/72" units

float page_w=595.27;  // default page size is A4
float page_h=841.88;

float start_x=0; // lower left start point for labels
float start_y=0;

int count_x=3; // number of labels in each direction
int count_y=8;

float label_w=198.43; // default labels size is 70 mm x 37 mm
float label_h=104.88;
float label_margin=6; // 14.17 points = 5 mm

bool print_marks=false;
bool link_file=false;
bool multiple_files=false;
bool book=false;
bool print_lines=false;

vector<string> filenames;

void usage()
{
  cerr << "Usage: pslabels [options] [ [[-link] <file>] | [-files <file> ..] ]\n";
  cerr << "  options:\n";
  cerr << "  -help\n";
  cerr << "  -marks\n";
  cerr << "  -lines\n";
  cerr << "  -lmargin <points>\n";
  cerr << "  -lw <points>\n";
  cerr << "  -lh <points>\n";
  cerr << "  -lcx <count>\n";
  cerr << "  -lcy <count>\n";
  cerr << "  -startx <points>\n";
  cerr << "  -starty <points>\n";
  cerr << "  -cdbook\n";
  cerr << "  -book\n";
}

void print_strings(char* strarray[])
{
  char** p=strarray;
  while (*p) {
    cout<<*p<<'\n'; p++;
  }
}

// get next word in line starting from position i
string get_word(string& line, int& i)
{
  int len=line.size();
  while (i<len) if (isspace(line[i])) i++; else break;
  if (i==len) return "";
  int i0=i;
  while (i<len) if (!isspace(line[i])) i++; else break;
  return line.substr(i0,i-i0);
}

bool str_to_float(string& s, float& f)
{
  const char* str=s.c_str();
  char* strp;
  f=strtod(str, &strp);
  if (f==0 && strp==str) return false;
  return true;
}

bool get_bbox_file(istream& from, float& x1, float& y1, float& x2, float& y2,
		   bool nocopy)
{
  string keyword="%%BoundingBox:";
  string line;
  int i;
  int keylen=keyword.length();
  bool ret_val=false;
  
  while (getline(from,line)) {
    if (!nocopy) cout<<line<<'\n';
    i=line.find(keyword);
    if (i<line.size()) { // found
      int i2=i+keylen;
      int i3=0;
      bool bbox_ok=true;
      string bbox=line.substr(i2,line.size()-i2);
      string number;
      if ((number=get_word(bbox,i3))=="") bbox_ok=false;
      if (!str_to_float(number,x1)) bbox_ok=false;
      if ((number=get_word(bbox,i3))=="") bbox_ok=false;
      if (!str_to_float(number,y1)) bbox_ok=false;
      if ((number=get_word(bbox,i3))=="") bbox_ok=false;
      if (!str_to_float(number,x2)) bbox_ok=false;
      if ((number=get_word(bbox,i3))=="") bbox_ok=false;
      if (!str_to_float(number,y2)) bbox_ok=false;
      if (bbox_ok) { ret_val=true; break; }
    }
  }
  if (!nocopy)
    while (getline(from,line)) cout<<line<<'\n';
  
  return ret_val;
}

void copy_file_out(istream& from)
{
  string line;
  while (getline(from,line)) cout<<line<<'\n';
}

void make_labels(istream& from, string eps_file_name)
{
  float bx1,by1,bx2,by2;

  cout<<"%!PS-Adobe-3.0\n";
  cout<<"%%BoundingBox: 0 0 595.27 841.88\n";
  cout<<"%%Title: (Multiple labels on a page)\n";
  cout<<"%%Creator: (pslabels)\n";
  cout<<"%%CreationDate: (May 9 1998)\n";
  cout<<"%%Pages: 1\n";
  cout<<"%%EndComments\n";
  cout<<"\n\n";
  cout<<"%%BeginProlog\n\n";
  
  print_strings(ps_prolog);
  
  cout<<"/PA_W "<<page_w<<" def\n";
  cout<<"/PA_H "<<page_h<<" def\n";
  cout<<"/LA_W "<<label_w<<" def\n";
  cout<<"/LA_H "<<label_h<<" def\n";
  cout<<"\n";
  cout<<"%%EndProlog\n";
  cout<<"\n\n";
  
  cout<<"/eps_file {\n";
  cout<<"%%BeginDocument: "<<eps_file_name<<"\n";
  if (!get_bbox_file(from,bx1,by1,bx2,by2,link_file)) {
    // if no bbox found, assume page (A4)
    bx1=0; by1=0; bx2=page_w; by2=page_h;
  }
  if (link_file) cout<<"("<<eps_file_name<<") run\n";
  cout<<"%%EndDocument\n";
  cout<<"} def\n";
  cout<<"\n\n";
  
  float scale;
  float transx=0;
  float transy=0;
  float file_w=fabs(bx2-bx1);
  float file_h=fabs(by2-by1);
  float label_ar=label_w/label_h;
  float file_ar=file_w/file_h;
  if (file_ar<label_ar) { // y axis determines scaling
    scale=(label_h-(2*label_margin))/file_h;
    transx=(label_w-(file_w*scale))/2;
    transy=label_margin;
  } else {                // x axis determines scaling
    scale=(label_w-(2*label_margin))/file_w;
    transy=(label_h-(file_h*scale))/2;
    transx=label_margin;
  }

  cout<<"/draw_label {\n";
  if (print_marks) {
    cout<<"  0.5 setlinewidth\n";
    cout<<"  -10 0 moveto 10 0 lineto stroke 0 -10 moveto 0 10 lineto stroke\n";
  }
  cout<<"  newpath\n";
  cout<<"\n";  
  cout<<"  BeginEPSF\n";
  cout<<"  "<<transx<<" "<<transy<<" translate\n";
  cout<<"  "<<scale<<" "<<scale<<" scale \n";
  cout<<"  "<<-bx1<<" "<<-by1<<" translate\n";
  cout<<"  eps_file\n";
  cout<<"  EndEPSF\n";
  cout<<"} def\n";
  cout<<"\n\n";

  cout<<start_x<<" "<<start_y<<" translate\n";
  cout<<"\n";
  cout<<count_y-1<<" -1 0 {\n";
  cout<<"  /i exch def\n";
  cout<<"  0 1 "<<count_x-1<<" {\n";
  cout<<"    newpath\n";
  cout<<"    gsave\n";
  cout<<"    LA_W mul i LA_H mul translate\n";
  cout<<"    draw_label\n";
  cout<<"    grestore\n";
  cout<<"  } for\n";
  cout<<"} for\n";
  cout<<"\n";
  cout<<"showpage\n";
  cout<<"\n";
  cout<<"%%EOF\n";
}

int print_label_from_file(istream& from, string eps_file_name)
{
  float bx1,by1,bx2,by2;

  if (!get_bbox_file(from,bx1,by1,bx2,by2,true)) {
    // if no bbox found, assume page (A4)
    bx1=0; by1=0; bx2=page_w; by2=page_h;
  }
    
  float scale;
  float transx=0;
  float transy=0;
  float file_w=fabs(bx2-bx1);
  float file_h=fabs(by2-by1);
  float label_ar=label_w/label_h;
  float file_ar=file_w/file_h;
  if (file_ar<label_ar) { // y axis determines scaling
    scale=(label_h-(2*label_margin))/file_h;
    transx=(label_w-(file_w*scale))/2;
    transy=label_margin;
  } else {                // x axis determines scaling
    scale=(label_w-(2*label_margin))/file_w;
    transy=(label_h-(file_h*scale))/2;
    transx=label_margin;
  }
  
  cout<<"newpath\n";
  cout<<"\n";  
  cout<<"BeginEPSF\n";
  cout<<transx<<" "<<transy<<" translate\n";
  cout<<scale<<" "<<scale<<" scale \n";
  cout<<-bx1<<" "<<-by1<<" translate\n\n";

  cout<<"%%BeginDocument: "<<eps_file_name<<"\n";
  from.seekg(0);
  copy_file_out(from);
  cout<<"%%EndDocument\n";
  cout<<"\n\n";

  cout<<"EndEPSF\n";
  cout<<"\n\n";
}

int make_labels_files()
{
  int nfiles=filenames.size();
  
  cout<<"%!PS-Adobe-3.0\n";
  cout<<"%%BoundingBox: 0 0 595.27 841.88\n";
  cout<<"%%Title: (Multiple labels on a page)\n";
  cout<<"%%Creator: (pslabels)\n";
  cout<<"%%CreationDate: (May 9 1998)\n";
  cout<<"%%EndComments\n";
  cout<<"\n\n";
  cout<<"%%BeginProlog\n\n";
  
  print_strings(ps_prolog);
  
  cout<<"/PA_W "<<page_w<<" def\n";
  cout<<"/PA_H "<<page_h<<" def\n";
  cout<<"\n";
  cout<<"%%EndProlog\n";
  cout<<"\n\n";
  
  int npages=nfiles;
  
  // if booklet, adjust number of pages to be a multiple of 4
  if (book) if (nfiles % 4 != 0) npages=nfiles+4-(nfiles % 4);
  
  int phys_page=1;
  int i=0;
  while (i<npages) { // print pages until all eps files are used
    
    cout<<"%%Page: "<<phys_page<<" "<<phys_page<<"\n"; phys_page++;
    cout<<start_x<<" "<<start_y<<" translate\n";
    cout<<"\n";
    
    for (int ix=0; ix<count_x; ix++) {
      for (int iy=0; iy<count_y; iy++) {
	cout<<"newpath\n";
	cout<<"gsave\n";
	if (book) cout<<ix*label_h<<" "<<iy*label_w<<" translate\n"; else
	  cout<<ix*label_w<<" "<<iy*label_h<<" translate\n";

	if (print_marks) {
	  cout<<"0.5 setlinewidth\n";
	  cout<<"-10 0 moveto -2 0 lineto stroke\n";
	  cout<<label_w+2<<" 0 moveto "<<label_w+10<<" 0 lineto stroke\n";
	  if (book)
	    if (((i+1) % 2)==0) {
	      cout<<"-10 "<<label_h<<" moveto -2 "<<label_h<<" lineto stroke\n";
	      cout<<label_w+2<<" "<<label_h<<" moveto "<<label_w+10<<" "
		  <<label_h<<" lineto stroke\n";
	      cout<<"0 "<<label_h+10<<" moveto 0 "<<label_h+2<<" lineto stroke\n";
	      cout<<label_w<<" "<<label_h+10<<" moveto "
		  <<label_w<<" "<<label_h+2<<" lineto stroke\n";
	    } else {
	      cout<<"0 -10 moveto 0 -2 lineto stroke\n";
	      cout<<label_w<<" -10 moveto "<<label_w<<" -2 lineto stroke\n";
	    }
	}
	
	if (book) { // rotate page for booklet placement
	  int ri4=(i % 4);
	  if (ri4==0 || ri4==1) {
	    cout<<label_h<<" 0 translate\n";
	    cout<<"90 rotate\n";
	  } else {
	    cout<<"0 "<<label_w<<" translate\n";
	    cout<<"-90 rotate\n";
	  }
	}

	int iprint=i;
	if (book) { // select page in booklet to print
	  if (((i+1) % 2)==0) iprint=((i+1)/2)-1; else
	    iprint=npages-(((i+2)/2)-1)-1;
	}
		
	if (print_lines) {
	  cout<<"0.5 setlinewidth\n";
	  cout<<"0 0 moveto "<<label_w<<" 0 lineto "<<label_w<<" "<<label_h
	      <<" lineto 0 "<<label_h<<" lineto 0 0 lineto stroke\n";
	}
	
	// print page from eps file
	if (iprint<nfiles) {
	  
	  ifstream from;
	  from.open(filenames[iprint].c_str());
	  if (from) {
	    print_label_from_file(from, filenames[iprint]);
	    from.close();
	  } else {
	    cerr<<"Error opening file!\n";
	  }
	}
	
	cout<<"grestore\n\n";
	i++;
	if (i>=npages) break;
      }
      if (i>=npages) break;
    }
    
    cout<<"\n";
    cout<<"showpage\n";
  }
  
  cout<<"%%Pages: "<<phys_page<<"\n";
  cout<<"\n";
  cout<<"%%EOF\n";
  
  return 0;
}

int main(int argc, char* argv[])
{
  char* file_name=NULL;
  int idummy;
  float fdummy;

  // parse command line arguments
  int argi=1;
  while (argi<argc) {
    if (strcmp("-help",argv[argi])==0) {
      usage(); return 0;
    } else if (strcmp("-marks",argv[argi])==0) {
      print_marks=true;
    } else if (strcmp("-lines",argv[argi])==0) {
      print_lines=true;
    } else if (strcmp("-link",argv[argi])==0) {
      link_file=true;
    } else if (strcmp("-lcx",argv[argi])==0) { 
      argi++;
      if (argi<argc) count_x=atoi(argv[argi]); else {
	usage(); return 1;
      }
    } else if (strcmp("-lcy",argv[argi])==0) { 
      argi++;
      if (argi<argc) count_y=atoi(argv[argi]); else {
	usage(); return 1;
      }
    } else if (strcmp("-lw",argv[argi])==0) {
      argi++;
      if (argi<argc) label_w=atof(argv[argi]); else {
	usage(); return 1;
      }
    } else if (strcmp("-lh",argv[argi])==0) {
      argi++;
      if (argi<argc) label_h=atof(argv[argi]); else {
	usage(); return 1;
      }
    } else if (strcmp("-lmargin",argv[argi])==0) {
      argi++;
      if (argi<argc) label_margin=atof(argv[argi]); else {
	usage(); return 1;
      }
    } else if (strcmp("-startx",argv[argi])==0) {
      argi++;
      if (argi<argc) start_x=atof(argv[argi]); else {
	usage(); return 1;
      }
    } else if (strcmp("-starty",argv[argi])==0) {
      argi++;
      if (argi<argc) start_y=atof(argv[argi]); else {
	usage(); return 1;
      }
    } else if (strcmp("-cdbook",argv[argi])==0) {
      book=true;
      count_x=1; count_y=2; // two booklet pages on a sheet
      label_w=label_h=340.152; // size of CD booklet is 12x12cm
      start_x=(page_w-label_w)/2;
      start_y=(page_h-label_h*2)/2;
      label_margin=0;
    } else if (strcmp("-book",argv[argi])==0) {
      book=true;
      count_x=1; count_y=2; // two booklet pages on a sheet
      label_h=page_w; label_w=page_h/2; // size of A3 booklet
      label_margin=0;
    } else if (strcmp("-files",argv[argi])==0) {
      argi++; multiple_files=true;
      while (argi<argc) {
	filenames.push_back(argv[argi]);
	argi++;
      }
    } else {
      if (argc-argi==1) {
	file_name=argv[argi];
	break;
      }
      cerr<<"Argument "<<argi<<" not recognized!\n";
      usage(); return 1;
    }
    argi++;
  }

  if (!multiple_files) { // print copies of one file
    if (file_name!=NULL) { // read from file
      ifstream from;
      from.open(file_name);
      if (from) {
	make_labels(from, file_name);
	from.close();
      } else {
	cerr<<"Error opening file!\n";
	return 2;
      }
    } else { // read from standard input
      if (link_file) {
	cerr<<"Can not link to file when reading from standard input.\n";
	link_file=false;
      }
      make_labels(cin, "");
    }
  } else { // print multiple files on a page
    return make_labels_files();
  }
  return 0;
}
