/*
** ~ppr/src/rpprd/rpprd_main.c
** Copyright 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This file was last modified 23 September 1996.
*/

#include "global_defines.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/time.h>
#include "rpprd.h"

int do_daemon = FALSE;

/*
** Print an error message and abort.
*/
void fatal(int exitval, const char *string, ... )
    {
    int file;
    va_list va;
    char line[256];

    /* Build the message in the buffer. */
    strcpy(line, "FATAL: ");
    va_start(va, string);
    vsprintf(&line[strlen(line)], string, va);
    va_end(va);
    strcat(line, "\n");

    if( do_daemon )
    	{
	close(3);		/* Free at least one file handle. */

	/* If we can open the log file, write a line to it. */
	if( (file=open(RPPRD_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
            {
	    write(file, line, strlen(line));
            close(file);
            }
	}
    else
	{
	write(1, line, strlen(line));
        }

    exit(exitval);	/* In any case, exit. */
    } /* end of fatal() */

/*
** Print a log line, classifying it as a debug line.
*/
void debug(const char *string, ... )
    {
    int file;
    va_list va;
    char templine[256];
     
    strcpy(templine, "DEBUG: ");
    va_start(va, string);
    vsprintf(&templine[strlen(templine)], string, va);
    va_end(va);
    strcat(templine, "\n");

    if( do_daemon )
	{
	if( (file=open(RPPRD_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
	    {
	    write(file, templine, strlen(templine));
            close(file);
            }
	}
    else
	{
	write(1, templine, strlen(templine));
        }

    } /* end of debug() */

/*
** The array which describes all the file desciptors:
*/
static struct FD fd_list[FD_SETSIZE];
static fd_set fds_read;
static fd_set fds_write;
static fd_set fds_exception;

/*
** Initialize the structure which describes the files descriptors.
*/
void initialize_select(void)
    {
    int x;

    DODEBUG(("initialize_select()"));
    
    for(x=0; x < FD_SETSIZE; x++)
        fd_list[x].used = FALSE;
    } /* end of initialize_select() */
    
/*
** Do a select and then call the functions to handle what we find.
*/
void do_select(void)
    {
    int x;

    DODEBUG(("do_select()"));
        
    FD_ZERO(&fds_read);
    FD_ZERO(&fds_write);
    FD_ZERO(&fds_exception);

    for(x=0; x < FD_SETSIZE; x++)
    	{
	if( ! fd_list[x].used ) continue;
	
	if( fd_list[x].read )
	    FD_SET(x, &fds_read);
	
	if( fd_list[x].write )
	    FD_SET(x, &fds_write);
	
	if( fd_list[x].exception )
	    FD_SET(x, &fds_exception);
    	}

    if( select(FD_SETSIZE, &fds_read, &fds_write, &fds_exception, (struct timeval *)NULL) == -1 )
    	fatal(1, "do_select(): select() failed, errno=%d (%s)", errno, strerror(errno) );

    for(x=0; x < FD_SETSIZE; x++)
    	{
	if( FD_ISSET(x, &fds_read) )
	    {
	    DODEBUG(("ready for read on %d (%s)", x, fd_list[x].name));
	    (*fd_list[x].read)(x, fd_list[x].read_extra);
	    }    	
    	
	if( FD_ISSET(x, &fds_write) )
	    {
	    DODEBUG(("ready for write on %d (%s)", x, fd_list[x].name));
	    (*fd_list[x].write)(x, fd_list[x].write_extra);
	    }    	

	if( FD_ISSET(x, &fds_exception) )
	    {
	    DODEBUG(("exception on %d (%s)", x, fd_list[x].name));
	    (*fd_list[x].exception)(x, fd_list[x].exception_extra);
	    }    	
    	}

    DODEBUG(("do_select(): done"));
    } /* end of select_loop() */

int main(int argc, char *argv[])
    {
    DODEBUG(("rpprd starting"));

    initialize_select();

    open_fifo();

    open_socket();

    while(TRUE)
	do_select();
    } /* end of main() */

/* end of file */
