/*
** ~ppr/src/ppuser/ppuser.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 21 February 1997.
*/

/*
** The operator's user account manipulation program.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <string.h>
#include <stdarg.h>
#include "global_defines.h"
#include "userdb.h"
#include "util_exits.h"

/* for myalloc() and mystrdup() */
const int lib_memory_fatal = EXIT_INTERNAL;

/*
** Handle fatal errors.
** Print a message and exit.
*/
void fatal(int rval, const char *message, ... )
    {
    va_list va;

    va_start(va,message);
    fprintf(stderr,"Fatal:  ");
    vfprintf(stderr,message,va);
    fprintf(stderr,"\n");
    va_end(va);
    exit(rval);
    } /* end of fatal() */

/*
** Handle non-fatal errors.
** This is called from functions in libpprdb.
*/
void error(const char *message, ... )
    {
    va_list va;

    va_start(va,message);
    vfprintf(stderr,message,va);
    fprintf(stderr,"\n");
    va_end(va);
    } /* end of error() */

/*
** Add a record to the database.
*/
int ppuser_add(char *argv[])
    {
    char *username;
    struct userdb data;
    float x;

    if(argv[0]==(char*)NULL || argv[1]==(char*)NULL
            || argv[2]==(char*)NULL || argv[3]==(char*)NULL
            || argv[4]==(char*)NULL || argv[5]==(char*)NULL)
        {
        fputs("Usage:\n"
            "\tppuser add '<user name>' '<real name>' <authcode> <initial balance> <cutoff> <life>\n",stderr);
        return EXIT_SYNTAX;
        }

    username = argv[0];

    strncpy(data.fullname,argv[1],MAX_FULLNAME);	/* the real name */
    data.fullname[MAX_FULLNAME] = (char)NULL;

    strncpy(data.authcode,argv[2],MAX_AUTHCODE);	/* the AuthCode */
    data.authcode[MAX_AUTHCODE] = (char)NULL;
    
    sscanf(argv[3],"%f",&x);			/* the initial balance */
    x *= 100.0;
    data.balance = (int)x;

    sscanf(argv[4],"%f",&x);			/* the credit cutoff point */
    x *= 100.0;
    data.cutoff = (int)x;

    sscanf(argv[5],"%d",&data.lifetime);	/* how long to preserve */

    data.revoked = FALSE;			/* credit not revoked */

    switch(db_add_user(username,&data))
        {
        case USER_ISNT:
            fprintf(stderr,"The record does not exist.\n");
            return EXIT_NOTFOUND;
        case USER_OK:
            return EXIT_OK;
        default:
            fprintf(stderr,"Database error.\n");
            return EXIT_INTERNAL;
        }   
    } /* end of ppuser_add() */

/*                         
** Delete a record from the database.
*/
int ppuser_delete(char *argv[])
    {
    char *username;

    if( (username=argv[0]) == (char*)NULL )
        {
        fprintf(stderr,"Usage:\n\tppuser delete '<user name>'\n");
        return EXIT_SYNTAX;
        }

    switch(db_delete_user(username))
        {
        case USER_ISNT:
            fprintf(stderr,"The record does not exist.\n");
            return EXIT_NOTFOUND;
        case USER_OK:
            return EXIT_OK;
        default:
            fprintf(stderr,"Database error.\n");
            return EXIT_INTERNAL;
        }
    } /* end of ppuser_delete() */

/*
** Show a users recored, all except the authcode field.
*/
int ppuser_show(char *argv[])
    {
    struct userdb entry;
    char *username=argv[0];
    struct tm *t;
    char temp[32];

    if(username==(char*)NULL)
        {
        fprintf(stderr, "Usage:\n\tppuser show '<user name>'\n");
        return EXIT_SYNTAX;
        }

    switch(db_auth(&entry, username))
        {
        case USER_OK:
        case USER_OVERDRAWN:
            printf("Name: %s\n", username);
	    printf("Fullname: %s\n", entry.fullname);
            printf("Balance: %2.2f\n", (double)entry.balance / 100);
            printf("Cutoff point: %2.2f\n", (double)entry.cutoff / 100);
            t = localtime(&entry.last_mod);
            strftime(temp, sizeof(temp), "%d %b %Y %I:%M %p", t);
            printf("Last Modified: %s\n", temp);
            printf("Account lifetime: %d\n", entry.lifetime);
	    printf("Credit revoked: %s\n", entry.revoked ? "TRUE" : "FALSE");
            return EXIT_OK;
        case USER_ISNT:
            printf("Record not found.\n");
            return EXIT_NOTFOUND;
        default:
            fprintf(stderr,"db_auth() returned invalid value\n");
            return EXIT_INTERNAL;
        }
    } /* end of ppuser_show() */

/*
** Change a user's authcode.
*/
int ppuser_authcode(char *argv[])
    {
    char *username;
    char *newauthcode;

    username=argv[0];
    newauthcode=argv[1];

    if( (username==(char*)NULL) || (newauthcode==(char*)NULL) )
        {
        fprintf(stderr,"Usage: ppuser authcode '<UserName>' <NewAuthCode>\n");
        return EXIT_SYNTAX;
        }

    switch(db_new_authcode(username,newauthcode))
        {
        case USER_ISNT:
            fprintf(stderr,"Record not found.\n");
            return EXIT_NOTFOUND;
        case USER_OK:
            return EXIT_OK;
        default:
            fprintf(stderr,"Database error.\n");
            return EXIT_INTERNAL;
        }
    }

/*
** Deposit money in a users account.
*/
int ppuser_transaction(char *argv[], int transaction_type)
    {
    char *username=argv[0];
    float x;
    int amount;
    int ret;

    if(argv[0]==(char*)NULL || argv[1]==(char*)NULL)
        {
        fputs("Usage:\n"
            "\tppuser {deposit,withdraw,charge,correction} '<user name>' <amount>\n",stderr);
        return EXIT_SYNTAX;
        }

    sscanf(argv[1],"%f",&x);
    x *= 100.0;
    amount = (int)x;

    ret = db_transaction(username,amount,transaction_type);

    switch(ret)
        {
        case USER_OK:
            return EXIT_OK;
        case USER_ISNT:
            fprintf(stderr,"User does not exist.\n");
            return EXIT_NOTFOUND;
        default:
            fprintf(stderr,"Fatal error, tranaction not processed.\n");
            return EXIT_INTERNAL;
        }
    } /* end of ppuser_transaction() */

/*
** called from main() if no valid command 
*/
void help(FILE *outfile)
    {
    fputs("Usage:\n"
	"\tppuser add '<user name>' '<real name>' <authcode> <initial balance> <cutoff> <life>\n"
	"\tppuser delete '<user name>'\n"
	"\tppuser show '<user name>'\n"
	"\tppuser authcode '<user name>' <authcode>\n"
	"\tppuser deposit '<user name>' amount\n"
	"\tppuser withdraw '<user name>' amount\n"
	"\tppuser charge '<user name>' amount\n"
	"\tppuser correction '<user name>' amount\n",outfile);
    } /* end of help */

/*
** main() function and command dispatcher 
*/
int main(int argc, char *argv[])
    {
    uid_t uid;

    if( ((uid=getuid()) != 0) && (uid != geteuid()) )
        {
        fprintf(stderr,"You are not allowed to use this utility.\n");
        exit(5);
        }      

    if(argc < 2)        /* if too few to be any valid command */
        {
        help(stderr);
        return EXIT_SYNTAX;
        }

    if( icmp(argv[1],"add") == 0 )
        return ppuser_add(&argv[2]);
    else if( icmp(argv[1],"delete") == 0 )
        return ppuser_delete(&argv[2]);
    else if( icmp(argv[1],"show") == 0 )
        return ppuser_show(&argv[2]);
    else if( icmp(argv[1],"authcode") == 0 )
        return ppuser_authcode(&argv[2]);
    else if( icmp(argv[1],"deposit") == 0 )
        return ppuser_transaction(&argv[2],TRANSACTION_DEPOSIT);
    else if( icmp(argv[1],"withdraw") == 0 )
        return ppuser_transaction(&argv[2],TRANSACTION_WITHDRAWAL);
    else if( icmp(argv[1],"charge") == 0 )
        return ppuser_transaction(&argv[2],TRANSACTION_WITHDRAWAL);
    else if( icmp(argv[1],"correction") == 0 )
        return ppuser_transaction(&argv[2],TRANSACTION_CORRECTION);
    else if( icmp(argv[1],"help") == 0 )
	{
	help(stdout);
    	return EXIT_OK;
    	}
    else
        {
        help(stderr);
        return EXIT_SYNTAX;
        }
    } /* end of main() */

/* end of file */
