/*
** ~ppr/src/pprdrv/pprdrv_res.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 20 January 1997.
*/

/*
** Code for including and striping out resources.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <sys/stat.h>
#include <string.h>
#include "interface.h"
#include "pprdrv.h"

/*
** This routine, which is called by _include_resource() will uncompress
** the PFB font data in the file and write it to the printer.
**
** We need the file name for error messages.
*/
void uncompress_pfb(char *filename, FILE *ifile)
    {
    int c;			/* temporary character storage */
    int larray[4];		/* don't us char! */
    unsigned int length;	/* the computed length */
    int x;			/* loop counter */
    static const char hexchars[] = "0123456789ABCDEF";
    int segment_type;
    int linelength;		/* for breaking hexadecimal lines */
    
    /* We go around this loop once for each segment. */
    while( (c=fgetc(ifile)) != EOF )		/* should never be false */
    	{
    	if( c != 128 )
    	    fatal(EXIT_PRNERR_NORETRY,"PFB file \"%s\" is corrupt (reason 1)",filename);
    	    
	segment_type = fgetc(ifile);		/* should be 1, 2, or 3 */
	
	length = 0;
	if( segment_type == 1 || segment_type == 2 )
	    {
	    for(x=0; x < 4; x++)		/* read 4 byte length */
	    	{
	    	if( (larray[x]=fgetc(ifile)) == EOF )
	    	    fatal(EXIT_PRNERR_NORETRY,"PFB file \"%s\" is corrupt (reason 2)",filename);
	        }
	    for(x=3; x >= 0; x--)
	    	{
		length *= 256;
		length += larray[x];	    	
	    	}
	    }

	switch(segment_type)
	    {
	    case 1:			/* ASCII segment */
		while(length--)
		    {
		    if( (c=fgetc(ifile)) == EOF )
		    	fatal(EXIT_PRNERR_NORETRY,"PFB file \"%s\" is corrupt (reason 3)",filename);

		    printer_putc(c);
		    }
		break;	    
	    case 2:			/* Hexadecimal data segment */
		linelength=0;
	    	while(length--)
	    	    {
	    	    if( (c=fgetc(ifile)) == EOF )
	    	    	fatal(EXIT_PRNERR_NORETRY,"PFB file \"%s\" is currupt (reason 4)",filename);
	    
		    printer_putc( hexchars[c/16] );
		    printer_putc( hexchars[c%16] );
		    
		    if(++linelength >= 40)
		    	{
		    	printer_putc('\n');
		    	linelength=0;
		    	}
		    }
		break;
	    case 3:			/* end of file */
	    	return;
	    default:			/* unknown segment type */
		fatal(EXIT_PRNERR_NORETRY,"PFB file \"%s\" is corrupt (reason 5)",filename);
	    }
    	
    	} 

    fatal(EXIT_PRNERR_NORETRY,"PFB file \"%s\" is corrupt (reason 6)",filename);
    } /* end of uncompress_pfb() */

/*
** Download only those portions of the Mac TrueType font which are necessary.
** What is necessary depends on whether the printer has a TTRasteriser and
** what type it is.
*/
static void _copy(char *fname, char *tline, int tline_len, FILE *cache_file, char *stop)
    {
    int ne;

    do	{
	printer_puts(tline);
	ne = strcmp(tline,stop);
	if( fgets(tline,tline_len,cache_file) == (char*)NULL )
	    fatal(EXIT_PRNERR_NORETRY,"Mac TT font without \"%.*s\" in cache: \"%s\"",(int)strcspn(stop,"\n"),stop,fname);
	} while( ne );

    } /* end of _copy() */

static void _discard(char *fname, char *tline, int tline_len, FILE *cache_file, char *stop)
    {
    int ne;

    do	{
	ne = strcmp(tline,stop);
	if( fgets(tline,tline_len,cache_file) == (char*)NULL )
	    fatal(EXIT_PRNERR_NORETRY,"Mac TT font without \"%.*s\" in cache: \"%s\"",(int)strcspn(stop,"\n"),stop,fname);
	} while( ne );

    } /* end of _discard() */

static void _skipblanklines(char *fname, char *tline, int tline_len, FILE *cache_file)
    {
    while(tline[strspn(tline, " \t\r\n")] == (char)NULL)
    	{
	if( fgets(tline, tline_len, cache_file) == (char*)NULL )
	    return;    	
    	}
    } /* end of _skipblanklines() */

static void insert_macttfont(char *filename)
    {
    FILE *cache_file;
    char tline[2048];		/* seems generous since those samples examined where not longer than 256 */
    int type42_sent = FALSE;

    /* Open the cache file for read. */
    if( (cache_file=fopen(filename,"r")) == (FILE*)NULL)
	fatal(EXIT_JOBERR,"The resource file \"%s\" can't be opened",filename);

    /* Copy up to the first "%begin" line. */
    while( fgets(tline,sizeof(tline),cache_file) != (char*)NULL && strncmp(tline,"%begin",6) )
	printer_puts(tline);
    	
    /* If this is the "%beginsfnt" section, copy or discard it. */
    if(strcmp(tline,"%beginsfnt\n") == 0)
    	{
    	if( printer.type42_ok )
	    {
	    _copy(filename,tline,sizeof(tline),cache_file," %endsfnt\n");
	    type42_sent = TRUE;
	    }
	else
	    {
	    _discard(filename,tline,sizeof(tline),cache_file," %endsfnt\n");
	    }

 	_skipblanklines(filename, tline, sizeof(tline), cache_file);

	/*
	** If the TrueDict compatibility BuildChar procedure comes next
	** and we need it copy it.   If we don't need it (if we will not be
	** printing any TrueType fonts or the printer has a built-in TT
	** rasterizer), discard it.  If we need it and it is absent, 
	** insert one from `memory'.
	*/
	if( strcmp(tline,"%beginsfntBC\n") == 0 )
	    {
	    if( type42_sent && Features.TTRasterizer==TT_ACCEPT68K )
		_copy(filename,tline,sizeof(tline),cache_file," %endsfntBC\n");
	    else
		_discard(filename,tline,sizeof(tline),cache_file," %endsfntBC\n");
	    }
	else if( type42_sent && Features.TTRasterizer==TT_ACCEPT68K )
	    {
	    printer_puts("%beginsfntBC\n"
		"truedictknown type42known not and ( %endsfntBC)exch fcheckload\n"
		"/TrueState 271 string def\n"
		"TrueDict begin sfnts save 72 0 matrix defaultmatrix dtransform dup mul\n"
		"exch dup mul add sqrt cvi 0 72 matrix defaultmatrix dtransform dup mul\n"
		"exch dup mul add sqrt cvi 3 -1 roll restore TrueState initer end\n"
		"/BuildChar{exch begin Encoding 1 index get CharStrings dup 2 index known\n"
		"{exch}{exch pop /.notdef}ifelse get dup xcheck{currentdict systemdict\n"
		"begin begin exec end end}{exch pop TrueDict begin /bander load cvlit\n"
		"exch TrueState render end}ifelse end} bind def\n"
		"\n %endsfntBC\n");
	    }

	_skipblanklines(filename, tline, sizeof(tline), cache_file);

	/* Sanity check, make sure the sfntdef section is next. */
	if( strcmp(tline,"%beginsfntdef\n") )
	    fatal(EXIT_PRNERR_NORETRY,"Cached Mac TT font has no sfntdef section: \"%s\"",filename);

	/* Copy or discard the "sfntdef" section */
	if( type42_sent )
	    _copy(filename,tline,sizeof(tline),cache_file," %endsfntdef\n");
	else
	    _discard(filename,tline,sizeof(tline),cache_file," %endsfntdef\n");
    	}

    _skipblanklines(filename, tline, sizeof(tline), cache_file);

    /* If a type1 font comes next, copy or discard it. */
    if( strcmp(tline,"%beginType1\n") == 0 )
    	{
	if( ! printer.type42_ok || ! type42_sent )
	    {
	    printer_puts(tline);			/* Write out the comment line */
	    fgets(tline,sizeof(tline),cache_file);	/* Load the test line */
	    printer_printf("%%PPR %s",tline);		/* Write it commented out */
	    fgets(tline,sizeof(tline),cache_file);	/* load next line */
	    _copy(filename,tline,sizeof(tline),cache_file," %endType1\n");
	    }    	
    	else
    	    {
	    _discard(filename,tline,sizeof(tline),cache_file," %endType1\n");
    	    }
    	}

    /* Copy the tail end of the file */
    do  {
    	printer_puts(tline);
    	} while( fgets(tline,sizeof(tline),cache_file) != (char*)NULL );

    /* close the cache file */
    fclose(cache_file);
    } /* end of insert_macttfont() */

/*
** include_resource() is called whenever an "%%IncludResource:" comment is 
** encountered.  If the resource in question is in the printer, do
** not include it, just send the comment to the printer; if the 
** resource is in our cache, insert it, bracketed by "%%Begin(End)Resource:"
** comments; if it is not in the printer and not in the cache, then
** just send the comment to the printer in the faint hope that another
** intervening spooler will be able to supply it.
**
** include_resource() is a wrapper function for _include_resource().
*/
static void _include_resource(char *type, char *name, double version, int revision, int forcing_prolog, int forcing_docsetup)
    {
    FILE *ifile;			/* Input file (resource cache file). */
    char tline[MAX_LINE+1];		/* Temporary line for reading resource cache files. */
    int tline_len;    
    struct DRVRES *d;			/* pointer to resource list entry */
    int x;				/* index into resource list */
	
    DODEBUG_RESOURCES(("_include_resource(type=\"%s\", name=\"%s\", version=%s, revision=%d)", type, name, dtostr(version), revision));

    /* 
    ** Search the resource list to find
    ** the resource in question.
    */
    for(x=0; x < drvres_count; x++)
	{
	d = &drvres[x];			/* make a handy pointer */

	DODEBUG_RESOURCES(("trying %s %s %s %d", d->type, d->name, dtostr(d->version), revision));

	if(strcmp(d->type,type))	/* We aren't interested in things */
	    continue;			/* which are of the wrong type. */
	    	
	/*
	** If the current name is what we are looking for or the
	** former name is what we are looking for,
	*/
	if( (strcmp(d->name,name)==0 || (d->former_name!=(char*)NULL && strcmp(d->former_name,name)==0))
		&& d->version==version && d->revision==revision )
	    {
	    #ifdef DEBUG_RESOURCES
	    debug("match!");
	    #endif

	    /*
	    ** If this resource was forced into the docsetup section
	    ** and this is not the docsetup section, ignore this
	    ** request to include it since doing so would
	    ** be redundant. 
	    */
	    if(!forcing_docsetup && d->force_into_docsetup)
	    	return;

	    /* 
	    ** If this resource was forced into the prolog section
	    ** and we are not carrying out that order right now,
	    ** delete it from this position.
	    */
	    if(!forcing_prolog && d->force_into_prolog)
	    	return;

	    if(d->former_name!=(char*)NULL)	/* If resource has a former name, */
		name=d->name;			/* use the new name when loading it. */
		    
	    if(d->filename != (char*)NULL )	/* If should be in cache, */
		{				/* Try to insert if from the cache. */
		if(d->truetype)
		    {
		    printer_printf("%%%%BeginResource: font %s\n",quote(name));
		    insert_ttfont(d->filename);
		    printer_putline("%%EndResource");
		    }
		else if(d->mactt)
		    {
		    printer_printf("%%%%BeginResource: font %s\n",quote(name));
		    insert_macttfont(d->filename);
		    printer_putline("%%EndResource");
		    }
		else
		    {
		    int c;
		    
		    if(strcmp(type,"procset")==0)
			printer_printf("%%%%BeginResource: %s %s %s %d\n",type,quote(name),dtostr(version),revision);
		    else
		    	printer_printf("%%%%BeginResource: %s %s\n",type,quote(name));

		    /* Open the cache file for read */
		    if( (ifile=fopen(d->filename,"r")) == (FILE*)NULL)
		    	fatal(EXIT_JOBERR,"The resource file \"%s\" can't be opened",d->filename);

		    /*
		    ** If the first character is 128, this is a PFB
		    ** font which we must expand.
		    */
		    if( (c=fgetc(ifile)) != EOF )
			{
			ungetc(c,ifile);
			if( c==128 )
			    {
			    uncompress_pfb(d->filename,ifile);
			    }
			else
			    {
			    while( (tline_len=fread(tline,sizeof(char),MAX_LINE+1,ifile)) )
				 printer_write(tline,tline_len);	/* Copy it from the cache file. */
			    }
			}
		    	
		    /* close the cache file */
	    	    fclose(ifile);

	    	    printer_putline("%%EndResource");	/* close commented section */
		    } /* end of if not truetype */
		} /* end of if in cache */
	    else				/* Resource is in printer, */
		{				/* no code needed, just a comment. */
		if(strcmp(type,"procset")==0)
        	    printer_printf("%%%%IncludeResource: %s %s %s %d\n",type,quote(name),dtostr(version),revision);
		else
        	    printer_printf("%%%%IncludeResource: %s %s\n",type,quote(name));
		}

	    if(d->former_name!=(char*)NULL)	/* If substitute (fonts only), */
		{				/* insert extra code. */
		/* Emmit PostScript to define the font under the substituted font's name. */
		printer_printf("/%s /%s findfont %%PPR\n",d->former_name,d->name);	/* oldfont, subfont */
		if(d->subcode)			/* If special substitution code, */
			printer_printf("%s makefont %%PPR user supplied matrix\n",d->subcode);	/* use it. */
		printer_putline("dup maxlength 1 add dict /ppr_subfont exch def %PPR");
		printer_putline("{exch dup /FID ne %PPR");
		printer_putline(" {exch ppr_subfont 3 1 roll put} %PPR");
		printer_putline(" {pop pop} %PPR");
		printer_putline(" ifelse %PPR");
		printer_putline("} forall %PPR");
		printer_putline("ppr_subfont %PPR");
		printer_putline("definefont pop %PPR");
		}
	    return;
	    } /* end of if this is the correct resource */
	} /* end of loop to search resource list */
	
    /*
    ** This should eventualy be commented out because the process of job 
    ** splitting may leave jobs with "%%IncludeResource: comments in their 
    ** prolog or document setup sections for resources which one or more 
    ** fragments do not require.
    */
    fatal(EXIT_JOBERR,"pprdrv_res.c: _include_resource(): resource \"%s %s\" is not in the resource list",type,name);
    } /* end of _include_resource() */

/*
** This wrapper routine is called when an "%%IncludeResource:"
** comment is encountered in the file.
*/
void include_resource(void)
    {
    char *type;             /* resource type, font, procset, etc. */
    char *name;             /* resource name, Courier, Times-Roman, etc. */ 
    double version;
    int revision;

    tokenize();
    type = tokens[1];
    name = tokens[2];

    if(type == (char*)NULL || name == (char*)NULL)
    	return;

    if(strcmp(type, "procset") == 0	/* if procset, read version, */
		&& tokens[3] != (char*)NULL && tokens[4] != (char*)NULL)
        {				/* ppr guarantees they are there */
	version = getdouble(tokens[3]);
        sscanf(tokens[4], "%d", &revision);
        }
    else				/* if not a procset, */    
	{				/* use the dummy values */
	version = 0.0;
	revision = 0;
	}

    _include_resource(type, name, version, revision, FALSE, FALSE);
    } /* end of include_resource() */
    
/*
** This routine is called when a "%%BeginResource:"
** comment is encountered.
**
** This routine is still being planed.
** Currently, it just copies the resource.
** Later it may do late font stripping of fonts that are in
** the printer or of fonts that have been moved into the
** document setup section.
*/
void begin_resource(FILE *infile)
    {
    do  {
	printer_write(line,line_len);
	if(! line_overflow)
	    printer_putc('\n');
        } while( strcmp(line,"%%EndResource") 
            && (dgetline_read(infile)!=(char*)NULL) );
    } /* end of begin_resource() */

/*
** Insert those fonts for which no ``%%IncludeResource:'' comment
** was origionally provided.
**
** Actually, this will work for all resources, not just fonts, however,
** at present only fonts will have fixinclude set to TRUE.
**
** This routine is called during processing of the Document Setup Section
** or at the top of the document in the case of non-conforming documents.
**
** This routine will also include a resource if force_into_setup is TRUE.
** This is used for resource optimization.
*/
void insert_noinclude_fonts(void)
    {
    int x;
    struct DRVRES *d;

    for(x=0; x < drvres_count; x++)
    	{
	if( (d=&drvres[x])->fixinclude || d->force_into_docsetup )
	    {
	    _include_resource(d->type, d->name, d->version, d->revision, FALSE, TRUE); 
    	    }
    	}
    } /* end of insert_noinclude_fonts() */

/*
** Insert additional resources in the prolog.  These additional 
** resources may be required due to N-Up or the inclusion of 
** type 42 fonts.
**
** It is also possible that the resource was already included
** in an individual page description and this routine will move
** it into the prolog.
*/
void insert_extra_prolog_resources(void)
    {
    int x;
    struct DRVRES *d;

    for(x=0; x < drvres_count; x++)
    	{
	d = &drvres[x];
	if(d->force_into_prolog)
	    {
	    _include_resource(d->type, d->name, d->version, d->revision, TRUE, FALSE); 
    	    }
    	}
    } /* end of insert_extra_prolog() */

/* 
** Write the ``%%DocumentNeeded(Supplied)Resources:'' comments.
*/
void write_resource_comments(void)
    {
    int x;
    int started;
    struct DRVRES *d;
    
    /*
    ** Enumerate the resources which we hope somebody else,
    ** such as the printer, will supply for us.  This will
    ** normally be a list of fonts.
    */
    for(started=FALSE,x=0; x < drvres_count; x++)
        {
        d = &drvres[x];
        if(d->needed)			/* use only if ``needed'' */
            {
            if(started)			/* if already started, */
                {			/* use */
                printer_puts("%%+ ");	/* continuation comment */
                }
            else
		{
		started = TRUE;
		printer_puts("%%DocumentNeededResources: ");    
		}
            if(strcmp(d->type, "procset") == 0)			/* procsets */
                printer_printf("%s %s %s %d\n",
                    d->type, quote(d->name),			/* get */
                    dtostr(d->version), d->revision);		/* version info */
            else						/* other */
                printer_printf("%s %s\n", d->type, quote(d->name));  /* resources */
            }							/* don't */
        }

    /*
    ** Enumerate the resources which we are including in the document.
    ** This will normally be a list of procedure sets and downloaded
    ** fonts.
    */
    for(started=FALSE,x=0; x < drvres_count; x++)
	{
	d = &drvres[x];
	if(! d->needed)			/* use only if not ``needed'' */
            {
            if(started)                 /* if already started, */
                {                       /* use */
                printer_puts("%%+ ");	/* continuation comment */
                }
            else
                {
                started = TRUE;
                printer_puts("%%DocumentSuppliedResources: ");    
                }
            if(strcmp(d->type,"procset")==0)                /* procsets */
                printer_printf("%s %s %s %d\n",
                    d->type,quote(d->name),    /* get */
                    dtostr(d->version),d->revision );       /* version info */
            else                                            /* other */
                printer_printf("%s %s\n",d->type,quote(d->name));  /* resources */
            }                                               /* don't */
        }

    } /* end of write_resource_comments() */

/*============================================================================
** Routines for manipulating the DRVRES structure array.
** This array holds a list of the resources used in the document.
============================================================================*/

/*
** Add a DRVRES record and return a pointer to it.
*/
struct DRVRES *add_drvres(int needed, int fixinclude, char *type, char *name,
	double version, int revision)
    {
    struct DRVRES *d;
    
    /* If we need more space in the array, enlarge it. */
    if(drvres_count == drvres_space)
	{
	drvres_space += 100;
	drvres = myrealloc(drvres, drvres_space, sizeof(struct DRVRES));
	}

    /* Take the next drvres[] record position. */
    d = &drvres[drvres_count++];

    d->needed = needed;			/* True if resource not still in PostScript file. */
    d->fixinclude = fixinclude;		/* Must we fix a missing inclusion cue? */
    d->type = type;			/* resource type string: "font", "procset", etc. */
    d->name = name;			/* resource name string: "Times-Roman", "Trincoll-dmm-nup", etc. */
    d->version = version;		/* floating point version number: 3.1, 0.0, etc. */
    d->revision = revision;		/* integer revision number: 15, 0, etc. */
    d->former_name = (char*)NULL;	/* former name (if a substitute font) */
    d->subcode = (char*)NULL;		/* PostScript code to insert during font substitution */
    d->filename = (char*)NULL;		/* Name of file to load resource from (none as yet). */
    d->force_into_docsetup = FALSE;	/* don't `optimize' */
    d->truetype = FALSE;		/* initially, not a TrueType font */
    d->mactt = FALSE;			/* not known to be a Macintosh Type1/Type42 font */
    d->force_into_prolog = FALSE;	/* initially, don't add to prolog */

    return d;				/* return pointer to what we made */
    } /* end of add_drvres() */

/*
** Add a resource to the prolog.  If it is already elsewhere
** in the document, force it into the prolog.
**
** Return 0 if we suceeded, -1 if we couldn't find it.
*/
int add_resource(char *type, char *name, double version, int revision)
    {
    struct DRVRES *d;
    char *fnptr;
    int x;

    /*
    ** See if the resource is already in the document.
    ** If it is, make sure it is loaded in the prolog.
    */
    for(x=0; x < drvres_count; x++)
	{
	d = &drvres[x];
	if(strcmp(d->type, type) == 0 && strcmp(d->name, name) == 0)
	    {
	    d->force_into_prolog = TRUE;
	    return 0;
	    }
	}

    /*
    ** If we have it in the cache, then we will be downloading it.
    ** Notice that it is not a "needed" resource in the sense that
    ** it will not be missing from the data that we send to the
    ** printer.
    */
    if( (fnptr=find_cached_resource(type, name, version, revision, (int*)NULL, (mode_t*)NULL)) != (char*)NULL )
	{
	d = add_drvres(FALSE, FALSE, type, name, version, revision);
	d->filename = fnptr;
	d->force_into_prolog = TRUE;
	return 0;
	}
    else
	{
	return -1;
	}

    } /* end of add_resource() */
    
/*
** Add a font to the document.  If it is already elsewhere
** in the document, force it into the document setup section.
**
** This function is called in order to include any fonts
** need for N-Up titles and "Draft" notices.
**
** !!! What happens if it was not stript out and cached? !!!
*/
void add_resource_font(char *name)
    {
    struct DRVRES *d;
    char *fnptr;
    int x;
    
    /*
    ** See if the font is already in the document.
    ** If it is, make sure it is loaded in the
    ** document setup section.
    */
    for(x=0; x < drvres_count; x++)
	{
	d = &drvres[x];
	if(strcmp(d->type, "font") == 0 && strcmp(d->name, name) == 0)
	    {
	    d->force_into_docsetup = TRUE;
	    return;
	    }
	}

    /*
    ** Not found, so add it.  If it is not in the printer, try
    ** to find it in the cache, but don't worry if we can't
    ** find it, it is not that important.  Of course, if
    ** the document had used it, it would have been found above.
    */
    d = add_drvres(TRUE, TRUE, "font", name, 0.0, 0);

    if(find_font(name))
    	{
    	if( (fnptr=find_cached_resource(d->type, d->name, d->version, d->revision, (int*)NULL, (mode_t*)NULL)) != (char*)NULL )
	    {
	    d->filename = fnptr;	/* save file name for later download */
	    d->needed = FALSE;		/* it will now be a "Supplied" resource */
	    }
	}
    
    } /* end of add_resource_font() */

/* end of file */ 
