/*
** ~ppr/src/pprdrv/pprdrv_reason.c
** Copyright 1995, 1996, Trinity College Computing Center
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This file was last modified 13 December 1996.
*/

/*
** This module appends "Reason: " lines to the queue file in order
** to indicate the reason why a job was arrested.
*/

#include "global_defines.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include "pprdrv.h"
#include "interface.h"

/*
** Append a reason line with the provided content to the
** jobs queue file.  This will tell why the job was arrested.
**
** This routine is called by described_postscript_error(), below,
** but it is also called from pprdrv_capable.c.
**
** Since describe_postscript_error() is called from a signal handler,
** this function must not call any non-reentrant functions such as
** malloc() or stdio.
*/
void give_reason(const char *reason)
    {
    char fname[MAX_PATH];
    int filehandle;
    char temp[80+12];
    
    /* Check for possible buffer overflow. */
    if( strlen(reason) > 80 )
    	fatal(EXIT_PRNERR_NORETRY, "pprdrv_reason.c: give_reason(): reason too long");

    /* Build the name of the queue file. */
    sprintf(fname, QUEUEDIR"/%s", QueueFile);

    if( (filehandle=open(fname, O_WRONLY | O_APPEND)) == -1 )
	fatal(EXIT_PRNERR_NORETRY, "pprdrv_reason.c: give_reason(): can't open queue file, errno=%d (%s)", errno, strerror(errno) );
	
    /* Construct the line in the buffer. */
    sprintf(temp, "Reason: (%s)\n", reason);

    if( write(filehandle, temp, strlen(temp) ) == -1 )
    	fatal(EXIT_PRNERR_NORETRY, "pprdrv_reason.c: give_reason(): can't append to queue file, errno=%d (%s)", errno, strerror(errno) );
    
    close(filehandle);
    } /* end of give_reason() */
    
/*
** Describe a PostScript error in as inteligible a way as we know how.
** This routine is called whenever a PostScript error is detected.
**
** If this routine can assign no reason for the PostScript error, 
** it has the option of doing nothing, however the current implementation
** gives the reason as "PostScript Error".
**
** One or both of the arguments may be a NULL pointer.
**
** This routine is called from a signal handler, so it must not
** call any non-reentrant functions such as malloc() or stdio.
*/
void describe_postscript_error(const char *error, const char *command)
    {
    if( error != (char*)NULL && command != (char*)NULL )
    	{
	/* debug("describe_postscript_error(\"%s\",\"%s\")",error,command); */

	/*
	** Errors related to paramters that are too
	** big.  There will be rather subtle things.
	*/
	if( strcmp(error, "limitcheck") == 0 )
	    {
	    if( strcmp(command, "image") == 0 )
	    	{
	    	give_reason("Oversized graphic");
	    	return;
	    	}
	    }    

	/*
	** Undefined could mean practically anything, 
	** but we will attempt to identify some of
	** the more common ones.
	*/
	else if( strcmp(error, "undefined") == 0 )
	    {
	    if(*command==4)
	    	{
	    	give_reason("Embedded CTRL-D");
	    	return;
	    	}
	    if(*command==27)
	    	{
	    	give_reason("Embedded ESC sequence");
		return;
		}
	    if(*command==1 && command[1]=='M')
	    	{
	    	give_reason("Spurious use of TBCP");
		return;
	    	}
	    if(strcmp(command,"ltr")==0)
	    	{
	    	give_reason("Bad HP driver,use LaserWriter 8");
	    	return;			/* ^ no space after comma */
	    	}
	    if(strncmp(command,"duplex",6)==0)
	    	{		/* any duplex command */
	    	give_reason("Printer lacks duplex commands");
		return;
		}
	    if(strcmp(command,"wpdict")==0)
	    	{
	    	give_reason("Corrupt WordPerfect driver");
	    	return;
	    	}
	    if(strcmp(command,"bn")==0)
	    	{		/* error in certain Macintosh LaserWriter drivers */
	    	give_reason("Error in cached font,remove \"bn\" in last line");
	    	return;
	    	}
	    if(strcmp(command,"t42CSB")==0)
		{		/* Adobe MS-Windows driver for HP4M+ PPD sending to a HP4M */
		give_reason("No TT rasterizer,MS-Windows has|wrong PPD file");
		return;
		}
	    if(strcmp(command, "setuserparams") == 0)
	    	{
		give_reason("Not a level 2 printer,wrong PPD file");
	    	return;
	    	}
	    }

	/*
	** Invalid parameter.
	*/
	else if(strcmp(error,"rangecheck")==0)
	    {
	    if(strcmp(command,"setpapertray")==0)
	    	{
		give_reason("Invalid tray selection");
		return;	    	
	    	}
	    }

	} /* end of if non-NULL elements */

    /* If we get here, we couldn't figure it out. */
    give_reason("PostScript error");    

    } /* end of describe_postscript_error() */

/* end of file */
