/*
** ~ppr/pprdrv/pprdrv_progress.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** Last revised 21 January 1997.
*/

/*
** This module contains functions which write a "Progress:" line to the end of
** the queue file.  The format of this line is:
**
** Progress: xxxxxx yyyyyy zzzzzz
**
** Where xxxxxx is the number of bytes written so far, yyyyyy is the number of
** "%%Page:" comments written so far, and zzzzzz is the number of pages which
** the printer has confirmed printing.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include "pprdrv.h"
#include "interface.h"

/* These are the figures which others tell us to update: */
static int total_pages_started = 0;
static int total_pages_printed = 0;
static long total_bytes = 0;

/* Don't report progress until at least this many bytes sent. */
#define SLACK_BYTES_COUNT 5120

/*
** This routine write the current values to the end of the file.
** The first time it is called it opens the file, seeks to the end,
** records where the end is, and write the line.  On subsequent calls
** it returns to the recorded file position and overwrites the line
** with a new one.  The file handle is never closed.
*/
static void progress__write(void)
    {
    static char qfname[MAX_PATH];	/* The name of the queue file */
    static int handle = -1;		/* The handle of the open queue file */
    static off_t spot;			/* The offset at which we will write the line */

    char buffer[80];
    char *line;
    int len = 31;		/* Length of progress line. */ 
    int retval;

    /*
    ** If we are running in test mode, we have no
    ** business writing to the queue file, so we
    ** will write to stderr.
    */
    if(test_mode)
	{
	sprintf(buffer, "Progress: %010ld %04d %04d\n", total_bytes, total_pages_started, total_pages_printed);
	write(2, buffer, len);
	return;
	}

    if(handle == -1)		/* If file not open yet, */
    	{
	/* Build the file name. */
    	sprintf(qfname, "%s/%s", QUEUEDIR, QueueFile);
    	
	/* Open the file for read and write. */
    	if( (handle = open(qfname,O_RDWR)) == -1 )
    	    fatal(EXIT_PRNERR_NORETRY, "%s: progress__write(): open(\"%s\",O_WRONLY) failed, errno=%d (%s)", __FILE__, qfname, errno, strerror(errno) );

	/* Seek to a spot 31 bytes from the end. */
	if( (spot = lseek(handle, (off_t)(0-len), SEEK_END)) == -1 )
	    fatal(EXIT_PRNERR_NORETRY, "%s: progress__write(): lseek(%d,-len,SEEK_END) failed, errno=%d (%s)", __FILE__, handle, errno, strerror(errno) );

	/* Read what will be "Progress: " if this was done before. */
	if( (retval=read(handle, buffer, 10)) != 10 )
	    fatal(EXIT_PRNERR_NORETRY, "%s: progress__write(): read() returned %d, errno=%d (%s)", __FILE__, retval, errno, strerror(errno));
	
	/* If it isn't, seek to the end. */
	if( memcmp(buffer, "Progress: ", 10) != 0 )
	    {
	    if( (spot = lseek(handle, (off_t)0, SEEK_END)) == -1 )
		fatal(EXIT_PRNERR_NORETRY, "%s: progress__write(): lseek(%d,0,SEEK_END) failed, errno=%d (%s)", __FILE__, handle, errno, strerror(errno));
	    }
    	}

    /* Return to the proper position for the "Progress:" line. */
    if( lseek(handle, spot, SEEK_SET) == -1 )
	fatal(EXIT_PRNERR_NORETRY, "%s: progress__write(): lseek(%d,%ld,SEEK_SET) failed, errno=%d (%s)", __FILE__, handle, (long)spot, errno, strerror(errno) );
    	
    /* Construct the "Progress:" line in the buffer. */
    if(total_bytes > SLACK_BYTES_COUNT)
	{
	sprintf(buffer, "Progress: %010ld %04d %04d\n", total_bytes, total_pages_started, total_pages_printed);
	line = buffer;
	}
    else
	{
    	line = "Progress: 0000000000 0000 0000\n";
	}
    
    /* Write the new "Progress:" line. */
    while( (retval = write(handle, buffer, len)) != len )
	{
	if( retval == -1 && errno == EINTR )
	    continue;

    	fatal(EXIT_PRNERR_NORETRY, "%s: progress__write(): write() returned %d when %d was expected, errno=%d (%s)", __FILE__, retval, len, errno, strerror(errno) );
    	}
    
    } /* end of progress__write() */

/*
** This routine updates the state_update_pprdrv file.
*/
static void state_update_pprdrv__write(int linetype, const char *text)
    {
    static int handle = -1;
    int retry;
    struct stat statbuf;
    char buffer[80];
    int towlen, wlen;

    /*
    ** If pprdrv was invoked with the --test switch or not enough bytes have been
    ** sent yet to fill the communications chanel, then do nothing.
    */
    if(test_mode || total_bytes <= SLACK_BYTES_COUNT)
	return;

    /*
    ** We will break out of this loop as soon as we have
    ** a satisfactory progress file to write to.
    */
    for(retry=0; TRUE; retry++)
	{
	/* If not open, */
	if(handle == -1)
	    {
    	    if((handle = open(STATE_UPDATE_PPRDRV_FILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) == -1)
		fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): open() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
	    }
    
	/* All right, it is open, find out about it. */
	if(fstat(handle, &statbuf) == -1)
	    fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): fstat() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));

	/*
	** If another process has `rewound', we can't use it.
	** If all is well the other process will have unlinked
	** this one just after we opened it.  Of course it is possible
	** that someone left a file with the other execute bit set.  To
	** avoid a never ending loop we will unlink the file ourselves on
	** the second time thru.  We are reluctant to unlink it because
	** we don't want to unlink the new one which the other process
	** just created.
	*/
	if(statbuf.st_mode & S_IXOTH)
	    {
	    if(retry)
	    	if(unlink(STATE_UPDATE_PPRDRV_FILE) == -1 && errno != ENOENT)
		    fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): unlink() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
	    if(close(handle) == -1)
	    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): close() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
	    handle = -1;
	    continue;
	    }

	/* If it is too long, we must `rewind' it. */
	if(statbuf.st_size > STATE_UPDATE_PPRDRV_MAXBYTES)
	    {
	    if(unlink(STATE_UPDATE_PPRDRV_FILE) == -1 && errno != ENOENT)
	    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): unlink() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
	    if( (wlen=write(handle, "REWIND\n", 7)) == -1 )
	    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): write() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
	    else if(wlen != 7)
	    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): write wrote %d bytes of 7", __FILE__, wlen);
	    if(fchmod(handle, UNIX_644 | S_IXOTH) == -1)
	    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): fchmod() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
	    if(close(handle) == -1)
	    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): close() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
	    handle = -1;	    
	    continue;
	    }

	break;
	}

    /*
    ** Emmit a progress line of the requested type.
    */
    switch(linetype)
	{
	case 0:
	    sprintf(buffer, "PGSTA %s %d\n", printer.Name, total_pages_started);
	    break;
	case 1:
	    sprintf(buffer, "PGFIN %s %d\n", printer.Name, total_pages_printed);
	    break;
	case 2:
	    sprintf(buffer, "BYTES %s %ld %ld\n", printer.Name, total_bytes, job.attr.postscript_bytes);
	    break;
	case 3:
	    sprintf(buffer, "MESSAGE %s %s\n", printer.Name, text);
	    break;
	}

    towlen = strlen(buffer);
    if( (wlen = write(handle, buffer, towlen)) == -1 )
    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): write() failed, errno=%d (%s)", __FILE__, errno, strerror(errno));
    else if(wlen != towlen)
    	fatal(EXIT_PRNERR_NORETRY, "%s: state_update_pprdrv__write(): write() wrote %d of %d bytes", __FILE__, wlen, towlen);
    } /* end of state_update_pprdrv__write() */

/*
** This routine is called just after each "%%Page:"
** comment is emmited.
*/
void progress__page_start_comment_sent(void)
    {
    total_pages_started++;
    progress__write();
    state_update_pprdrv__write(0, (char*)NULL);
    }

/*
** This is called each time we get an HP PJL message
** from the printer indicating that a page has hit
** the output tray.
*/
void progress__pages_truly_printed(int n)
    {
    total_pages_printed += n;
    progress__write();
    state_update_pprdrv__write(1, (char*)NULL);
    }

/*
** This is called as each block in flushed into the
** pipe leading to the interface.
*/
void progress__bytes_sent(int n)
    {
    total_bytes += n;    
    progress__write();
    state_update_pprdrv__write(2, (char*)NULL);
    }

/*
** This is called every time the .status file is updated.
*/
void progress__new_message(const char *text)
    {
    state_update_pprdrv__write(3, text);    
    } 


/* end of file */
