%{
/*
** ~ppr/src/pprdrv/pprdrv_ppd_parse.l
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file was last modified 2 December 1996.
*/

/*
** This code is the lexer which pprdrv uses to parse an Adobe PPD file.
**
** It finds certain lines it is interested in, particularly those
** which give feature code and loads them into a hashed data
** structure so that they may be used when printing the document.
*/

#include "global_defines.h"	/* for feature values */
#include <stdlib.h>		/* for exit() and atoi() */
#ifdef FLEX_SCANNER
#include <unistd.h>		/* for read() (flex uses it) */
#include <sys/uio.h>		/* for read() (flex uses it) */
#endif
#include <errno.h>
#include <string.h>
#include "pprdrv.h"		/* for feature structure */
#include "interface.h"		/* for exit codes */

/* This is where we save the original input file while we
   are reading from an include file. */
extern char *ppd_fname[MAX_PPD_NEST];
extern FILE *ppd_f[MAX_PPD_NEST];
extern int ppd_nest_level;

/* Some Lex implementations define yywrap as a macro,
   if that has been done, get rid of it. */
#undef yywrap

/* Here is a yywrap() which switches back to the 
   original file when the include file ends. */
int yywrap(void)
    {
    if(ppd_nest_level > 0)
	{
	DODEBUG_PPD(("end of include file"));
	fclose(yyin);				/* close include file */
	myfree(ppd_fname[ppd_nest_level]);	/* don't need file name anymove */
	ppd_nest_level--;
	yyin = ppd_f[ppd_nest_level];		/* restore current input file */
	return 0;			/* Tell parser that we have opened another file. */
	}				/* (Which is sort of what we have done.) */

    return 1;
    }

%}

%s FONT
%s FONTJUNK
%s JUNKSTR
%s INSTR
%s EXT
%s FAX
%s LANGLEV
%s PAPDIM
%s IMGAREA
%s INCLUDE
%s ORD1
%s ORD2
%s ORD3
%s ORD4
                           
%p 3200

%%

<INITIAL>^\*Extensions:   { BEGIN EXT; }
<EXT>[ \t]+               { /* discard spaces and tabs */ }
<EXT>DPS/[ \t\n]          { Features.Extensions|=EXTENSION_DPS; }
<EXT>CMYK/[ \t\n]         { Features.Extensions|=EXTENSION_CMYK; }
<EXT>Composite/[ \t\n]    { Features.Extensions|=EXTENSION_Composite; }
<EXT>FileSystem/[ \t\n]   { Features.Extensions|=EXTENSION_FileSystem; }
<EXT>[^ \t\n]+/[ \t\n]    { error("Invalid \"*Extensions:\" in PPD file: \"%s\"",
                            yytext); }
<EXT>\n                   { BEGIN INITIAL; }

<INITIAL>^\*FaxSupport: { BEGIN FAX; }
<FAX>[ \t]+             { /* discard spaces and tabs */ }
<FAX>Base/[ \t\n]       { Features.FaxSupport|=FAXSUPPORT_Base; }
<FAX>[^ \t\n]+/[ \t\n]  { error("Invalid \"*FaxSupport:\" in PPD file: \"%s\"",
                            yytext); }
<FAX>\n                 { BEGIN INITIAL; }

<INITIAL>^\*FileSystem:[ \t]+True[ \t]*\n { Features.FileSystem=TRUE; }
<INITIAL>^\*FileSystem:[ \t]+False[ \t]*\n { Features.FileSystem=FALSE; }

<INITIAL>^\*TTRasterizer:[ \t]+None[ \t]*\n { Features.TTRasterizer=TT_NONE; }
<INITIAL>^\*TTRasterizer:[ \t]+Accept68K[ \t]*\n { Features.TTRasterizer=TT_ACCEPT68K; }
<INITIAL>^\*TTRasterizer:[ \t]+Type42[ \t]*\n { Features.TTRasterizer=TT_TYPE42; }

<INITIAL>^\*LanguageLevel:[ \t]*["] { BEGIN LANGLEV; }
<LANGLEV>[0-9]+         { Features.LanguageLevel=atoi(yytext); }
<LANGLEV>["]            { BEGIN INITIAL; }

<INITIAL>^\*Font[ \t]+         { BEGIN FONT; }
<FONT>[^:]+                 { BEGIN FONTJUNK;
                            add_font(yytext); }
<FONTJUNK>.*                { /* discard it */ }

<INITIAL>^\*DefaultOutputOrder:[ \t]+Normal[ \t]*\n { if(printer.OutputOrder==0)
				printer.OutputOrder=1; }
<INITIAL>^\*DefaultOutputOrder:[ \t]+Reverse[ \t]*\n { if(printer.OutputOrder==0)
				printer.OutputOrder=-1; }

<INITIAL>^\*Message:[ \t]+["]		{ BEGIN JUNKSTR; }
<INITIAL>^\*Status:[ \t]+["]		{ BEGIN JUNKSTR; }
<INITIAL>^\*Source:[ \t]+["]		{ BEGIN JUNKSTR; }
<INITIAL>^\*PrinterError:[ \t]+["]	{ BEGIN JUNKSTR; }
<INITIAL>^\*PCFileName:[ \t]+["]	{ BEGIN JUNKSTR; }
<JUNKSTR>[^"]|\n			{ /* discard data */ }
<JUNKSTR>["]				{ BEGIN INITIAL; }

<INITIAL>^\*PaperDimension[ \t]+[^\n:]+[:][ \t]+["] { papersize_moveto(yytext);
					new_string(yytext);
					BEGIN PAPDIM; }
<PAPDIM>[0-9]+[ ]+[0-9]+	{ sscanf(yytext,"%lf %lf",
					&papersize[papersizex].width,
					&papersize[papersizex].height);
					string_line(yytext); }
<PAPDIM>["]			{ BEGIN INITIAL; }

<INITIAL>^\*ImageableArea[ \t]+[^\n:]+[:][ \t]+["] { papersize_moveto(yytext);
					new_string(yytext);
					BEGIN IMGAREA; }
<IMGAREA>[0-9]+[ ]+[0-9]+ { sscanf(yytext,"%lf %lf %lf %lf",
					&papersize[papersizex].lm,
					&papersize[papersizex].tm,
					&papersize[papersizex].rm,
					&papersize[papersizex].bm); 
					string_line(yytext); }
<IMGAREA>["]			{ BEGIN INITIAL; }

<INITIAL>^\*Include:[ \t]["]	{ BEGIN INCLUDE; }
<INCLUDE>[^"\n]*		{ char *ifname;		/* include file name */

				if(++ppd_nest_level == MAX_PPD_NEST)
				    fatal(EXIT_PRNERR_NORETRY, "Too many levels of \"*Include:\" in PPD file");

				if(yytext[0] == '/')
				    {
				    ifname = mystrdup(yytext);
				    }
				else
				    {
				    char *pathend;
				    int len;

				    if( (pathend = strrchr(ppd_fname[ppd_nest_level-1], '/')) == (char*)NULL )
					fatal(EXIT_PRNERR_NORETRY, "pprdrv: internal error: no /");

				    len = pathend - ppd_fname[ppd_nest_level-1];

				    ifname = (char*)myalloc(len + 1 + strlen(yytext) + 1, sizeof(char));

				    sprintf(ifname, "%.*s/%s", len, ppd_fname[ppd_nest_level-1], yytext);
				    }

				DODEBUG_PPD(("opening include file \"%s\"", ifname));

				if( (yyin=fopen(ifname, "r")) == (FILE*)NULL ) 
				    fatal(EXIT_PRNERR_NORETRY, "Can't open PPD include file \"%s\", errno=%d (%s)", ifname, errno, strerror(errno));

				ppd_fname[ppd_nest_level] = ifname;
				ppd_f[ppd_nest_level] = yyin;
				}
<INCLUDE>["]			{ BEGIN INITIAL; }

<INITIAL>^\*[^ \t:\n]+:[ \t]+["]	{ BEGIN INSTR; new_string(yytext); }
<INITIAL>^\*[^? \t:\n]+[ \t]+[^ \t:\n]+(\/[^:\n]+)*:[ ]+["] { BEGIN INSTR;
					new_string(yytext); }
<INSTR>[^"\n]*			{ string_line(yytext); }
<INSTR>[\n]			{ /* discard the newline */ }
<INSTR>["]			{ BEGIN INITIAL; end_string(); }

<INITIAL>^\*OrderDependency:[ \t]+	{ BEGIN ORD1; }
<ORD1>[0-9]+				{ order_dependency_1(atoi(yytext)); 
					BEGIN ORD2; }
<ORD2>[ \t]+				{ }
<ORD2>ExitServer			{ order_dependency_2(ORD_EXITSERVER);
					BEGIN ORD3; }
<ORD2>Prolog				{ order_dependency_2(ORD_PROLOG);
					BEGIN ORD3; }
<ORD2>DocumentSetup			{ order_dependency_2(ORD_DOCUMENTSETUP);
					BEGIN ORD3; }
<ORD2>PageSetup				{ order_dependency_2(ORD_PAGESETUP);
					BEGIN ORD3; }
<ORD2>JCLSetup				{ order_dependency_2(ORD_JCLSETUP);
					BEGIN ORD3; }
<ORD2>AnySetup				{ order_dependency_2(ORD_ANYSETUP);
					BEGIN ORD3; }
<ORD3>[ \t]+				{ }
<ORD3>\*[^ \t\n]+			{ order_dependency_3(yytext);
					BEGIN ORD4; }
<ORD4>[ \t]+				{ }
<ORD4>[^ \t\n]+				{ order_dependency_4(yytext);
					BEGIN INITIAL; }
<ORD4>[\n]				{ order_dependency_4("");
					BEGIN INITIAL; }

.|\n         { BEGIN INITIAL; /* discard unmatched stuff */ }

%%
