/*
** ~ppr/src/pprdrv/pprdrv_ppd.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file last modified 20 January 1997.
*/

/*
** The functions in this file read the PPD file and later
** dispense information from it.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include "pprdrv.h"
#include "interface.h"

extern FILE *yyin;                  /* lex's input file */
int yylex(void);                    /* lex's principle function */

/*
** Globals
*/
struct PPDSTR **ppdstr;		/* PPD strings hash table */
struct PPDFONT **ppdfont;	/* PPD fontlist hash table */
char *ppdname;			/* name of next PPD string */
char *ppdtext;			/* text of next PPD string */

char *ppd_fname[MAX_PPD_NEST];	/* names of all nested PPD files */
FILE *ppd_f[MAX_PPD_NEST];	/* stdio structure pointers for same */
int ppd_nest_level;		/* number of PPD files now open */

int tindex;
struct FEATURES Features;			/* list of features */
struct PAPERSIZE papersize[MAX_PAPERSIZES];
int papersizex = 0;				/* index into papersize[] */
int num_papersizes = 0;

/*
** Hash functions:
**
** Hash the string s, tabsize is the size of the hash
** table.  The return value will be less than tabsize.
*/
int hash(char *s, int tabsize)
    {
    unsigned n=0;           /* intialize the work value */
    
    while(*s)               /* hash the string */
        n=32*n+*s++;
    return n % tabsize;     /* wrap value and return it */
    } /* end of hash() */

/*
** Hash two strings as though they were one.
**
** s2 may be a NULL pointer. 
*/
int hash2(char *s1, char *s2, int tabsize)
    {
    unsigned n=0;           /* initialize the work value */

    while(*s1)              /* hash the 1st part */
        n=32*n+*s1++;

    if(s2!=(char*)NULL)
    	{
    	if(*s2)    	    /* if second part is not empty, */
            n=32*n+' ';     /* hash in a space */
	while(*s2)          /* if 2nd part not empty, hash it too */
            n=32*n+*s2++;
        }

    return n % tabsize;     /* return wraped result */
    } /* end of hash2() */

/*=========================================================
** Functions called by the lexer.
=========================================================*/
/*
** add_font() is called for each *Font: line.
** The hash structure used by this routine and find_font() is
** different from that used by the others in this module.
*/
void add_font(char *fontname)
    {
    int h;
    struct PPDFONT *p;

    #ifdef DEBUG_PPD
    debug("add_font(\"%s\")",fontname!=(char*)NULL ? fontname : "<NULL>");
    #endif

    p = (struct PPDFONT*)myalloc(1,sizeof(struct PPDFONT));  
    p->name = mystrdup(fontname);

    h = hash(fontname,FONT_TABSIZE);
    p->next = ppdfont[h];
    ppdfont[h] = p;
    }

/* 
** This is called by the lexer when it detects the start of 
** a new string.  The argument is cleaned up and stored in
** ppdname[] until we are ready to use it.
*/
void new_string(char *name)
    {
    int c;                              /* next character */
    int x = 0;				/* index into string */    

    #ifdef DEBUG_PPD
    debug("new_string(name=\"%s\")", name != (char*)NULL ? name : "<NULL>");
    #endif

    while(*name && x < MAX_PPDNAME)
        {
        switch(c = *(name++))
            {
            case '/':			/* if slash (translate string intro) */
            case ':':			/* or colon (value seperator) */
                name = "";		/* stop things */
                break;
            case ' ':			/* if space, */
                while(*name == ' ')	/* eat up multiples */
                    name++;		/* and fall thru */
            default:
                ppdname[x++] = c;
            }
        }
        
    ppdname[x] = (char)NULL;	/* terminate it */
    tindex = 0;			/* initialize text buffer index */
    } /* end of new_string() */

/*
** The lexer calls this each time it reads a line of the string value.
** It appends the line to ppdtext for later storage in the hash table. 
**
** The argument is the line of string data that has been read.
*/
void string_line(char *string)
    {
    int string_len;
    
    #ifdef DEBUG_PPD_DETAILED
    debug("string_segment(\"%s\")", string != (char*)NULL ? string : "<NULL>");
    #endif
                              
    if(tindex != 0 && tindex != MAX_PPDTEXT)	/* if second or later line, */
        ppdtext[tindex++] = '\n';		/* seperate with line feed */

    string_len = strlen(string);

    if( (MAX_PPDTEXT - tindex) < (string_len + 1) )
        fatal(EXIT_PRNERR_NORETRY, "pprdrv_ppd.c: string_line(): text too long");  

    while(*string)				/* append this line */
        ppdtext[tindex++] = *string++;

    ppdtext[tindex] = (char)NULL;		/* terminate but don't advance */
    } /* end of string_line() */

/*
** All of the string is found, put it in the hash table.
*/
void end_string(void)
    {
    struct PPDSTR *s;
    struct PPDSTR **p;
    int h;

    #ifdef DEBUG_PPD_DETAILED
    debug("end_string()");
    #endif

    s = (struct PPDSTR*)myalloc(1,sizeof(struct PPDSTR)); 
    s->name = mystrdup(ppdname);		/* duplicate the temp values */
    s->value = mystrdup(ppdtext);		/* and store ptrs to them */

    h = hash(ppdname,PPD_TABSIZE);		/* hash the name */

    p = &ppdstr[h];				/* get pointer to 1st pointer */
    while( *p != (struct PPDSTR *)NULL )	/* search for the null one */
        p = &((*p)->next);                         
    *p = s;					/* set it to point to new entry */
    s->next = (struct PPDSTR *)NULL;		/* and nullify its next pointer */
    } /* end of end_string() */
                       
/*
** Process *OrderDependency information.
*/
void order_dependency_1(int order)
    {
    
    
    } /* end of of order_dependency_1() */

void order_dependency_2(int section)
    {
    
    
    } /* end of of order_dependency_2() */

void order_dependency_3(const char *name1)
    {
    
    
    } /* end of of order_dependency_3() */

void order_dependency_4(const char *name2)
    {
    
    
    } /* end of of order_dependency_4() */

/*==========================================================
** Read the Adobe PostScript Printer Description file.
==========================================================*/
void read_PPD_file(char *ppd_file_name)
    {
    int x;              /* used to initialize structures */              

    /* Set the default values. */
    Features.ColorDevice = FALSE;
    Features.Extensions = 0;
    Features.FaxSupport = FAXSUPPORT_NONE;
    Features.FileSystem = FALSE;
    Features.LanguageLevel = 1;
    Features.TTRasterizer = TT_UNKNOWN;

    /* Create hash tables for code strings and fonts. */
    ppdstr = (struct PPDSTR**)myalloc(PPD_TABSIZE, sizeof(struct PPDSTR*));
    ppdfont = (struct PPDFONT**)myalloc(FONT_TABSIZE, sizeof(struct PPDFONT*));

    for(x=0; x < PPD_TABSIZE; x++)       
        ppdstr[x] = (struct PPDSTR *)NULL;
    for(x=0; x < FONT_TABSIZE; x++)    
        ppdfont[x] = (struct PPDFONT *)NULL;

    /*
    ** A configuration file is not absolutely required to name
    ** a PPD file.  If printer has no PPD file, then stop things
    ** right here before we get to fopen(), which might cause
    ** a core dump.
    */
    if(ppd_file_name == (char*)NULL)
    	return;

    /*
    ** Allocate temporary storage for the current name 
    ** and the current code string.
    */
    ppdname = (char*)myalloc(MAX_PPDNAME+1, sizeof(char));
    ppdtext = (char*)myalloc(MAX_PPDTEXT, sizeof(char));

    /* open the PPD file */
    if( (yyin=fopen(ppd_file_name, "r")) == (FILE*)NULL )
        fatal(EXIT_PRNERR_NORETRY, "pprdrv_ppd.c: read_PPD_file(): can't open \"%s\", errno=%d (%s)", ppd_file_name, errno, strerror(errno));

    ppd_nest_level = 0;
    ppd_fname[ppd_nest_level] = ppd_file_name;
    ppd_f[ppd_nest_level] = yyin;

    /* call the lexer to process the file */
    yylex();

    /* Now, close the outermost file, but DON'T deallocate the name: */
    fclose(yyin);

    /* Free the scratch spaces: */
    myfree(ppdname);
    myfree(ppdtext);
    } /* read_PPD_file() */

/*==========================================================
** Routines for feature inclusion.
==========================================================*/
/*
** Return a pointer to feature code.
** If the requested feature is not available,
** return a NULL pointer.
*/
char *find_feature(char *featuretype, char *option)
    {
    struct PPDSTR *s;               /* pointer to PPD database entry */
    int len;                        /* used in comparison */
    char *string=(char*)NULL;       /* value to return */

    #ifdef DEBUG_PPD
    debug("find_feature(%s,%s)", featuretype, option != (char*)NULL ? option : "<NULL>");
    #endif

    s=ppdstr[hash2(featuretype,option,PPD_TABSIZE)];

    while( s != (struct PPDSTR *)NULL )
        {
        if( (strncmp(s->name,featuretype,len=strlen(featuretype))==0)
            && ( ( option==(char*)NULL && s->name[len]==(char)NULL )
                || ( s->name[len]==' ' && strcmp(&s->name[len+1],option)==0 ) )
            ) /* <-- notice parenthesis */ 
            {                   
            string=s->value;                    /* use the value */
            break;
            }
        else                                    /* if no match */
            {                                   /* if chain continues, */
            s=s->next;                          /* jump to next link */
            }
        }

    return string;
    } /* end of find_feature() */

/*
** Insert feature code, if we have it.
** If not, insert an %%IncludeFeature: comment.
**
** This function is called whenever an ``%%IncludeFeature:'' comment
** is encountered in the input file.  It is also called by
** insert_features() (see below).
*/
void include_feature(char *featuretype, char *option)
    {
    char *string;

    #ifdef DEBUG_PPD
    debug("include_feature()");
    #endif

    if(featuretype==(char*)NULL)	/* Ignore %%IncludeFeature: without arguments */
	return;

    if(strip_binselects)		/* If we are removing binselects, */
    	{				/* then don't copy them and comment out certain features */
	if( (strcmp(featuretype,"*InputSlot")==0) || (strcmp(featuretype,"*TraySwitch")==0) )
	    {
	    #ifdef KEEP_OLD_CODE
	    printer_printf("%% %%%%IncludeFeature: %s %s\n",featuretype,option!=(char*)NULL ? option : "");
	    #endif
	    return;
	    }

	if(strcmp(featuretype,"*PageSize")==0)	/* change *PageSize */
	    {
	    #ifdef KEEP_OLD_CODE
	    printer_printf("%% %%%%IncludeFeature: *PageSize %s\n",option!=(char*)NULL ? option : "");
	    #endif
	    featuretype="*PageRegion";		/* to *PageRegion */
	    }
	}

    /*
    ** This is used to strip out signature and
    ** booklet mode invokation code if PPR is doing the job.
    */
    if( strip_signature
    		&& ( (strcmp(featuretype,"*Signature")==0) || (strcmp(featuretype,"*Booklet")==0) ) )
	{
	#ifdef KEEP_OLD_CODE
	printer_printf("%% %%%%IncludeFeature: %s %s\n",featuretype,option!=(char*)NULL ? option : "");
	#endif
	return;
	}

    /* Look for the feature in the PPD file. */
    string = find_feature(featuretype,option);

    /* The feature code was found. */
    if(string != (char *)NULL)
        {
        if( *option != (char)NULL )     /* anounce the feature start */
            printer_printf("%%%%BeginFeature: %s %s\n",featuretype,option);
        else
            printer_printf("%%%%BeginFeature: %s\n",featuretype);

        if(option==(char*)NULL)		/* No option keyword, convert QuotedValue */
            printer_puts_QuotedValue(string);	/* to binary and write it to the printer. */
        else
            printer_puts(string);              /* include the feature text */

        printer_puts("\n%%EndFeature\n");    /* and flag its end */
        }

    /*
    ** The feature code is not found in the PPD file.
    */
    else
        {
	/*
	** If we are using N-Up and it is a *PageRegion
	** feature, guess at the correct code.
	**
	** We do this because the N-Up machinery may be able 
	** to interpret many commands that the printer can't.
	*/
	if( (job.N_Up.N!=1) && (strcmp(featuretype,"*PageRegion")==0)
			    && (option!=(char*)NULL) )
	    {
	    char *ptr=option;
	    printer_printf("%%%%BeginFeature: *PageRegion %s\n",option); 
	    while(ptr)
	        {
	        *ptr=tolower(*ptr);
	        ptr++;
	        }	    
	    printer_puts(option);	    
	    printer_puts("\n%%EndFeature\n");
	    }

	/*
	** As a last ditch effort, just insert an "%%IncludeFeature:" comment.
	*/
	else
	    {
            if( option != (char*)NULL )
                printer_printf("%%%%IncludeFeature: %s %s\n",featuretype,option);
            else
                printer_printf("%%%%IncludeFeature: %s\n",featuretype);
	    }
        }

    } /* end of include_feature() */

/*
** Call this function when we have "%%BeginFeature: ..." in line[].
**
** The "%%BeginFeature:" line is written out and the feature
** is looked up in the table we built.  If it is found in the PPD
** file table, the code from the PPD file is used to replace the 
** origional code, the origional code is discarded.
** 
** If the feature is not in the PPD file, then the origional code
** is retained, unless the featuretype is "*Duplex", in which case
** it is discarded with the understanding that the printer in 
** question can not do duplex and that duplex code might annoy it.
**
** Option may be a NULL pointer. 
*/
void begin_feature(char *featuretype, char *option, FILE *infile)
    {
    char *string;
    int fallback = job.opts.keep_badfeatures;	/* Should we fall back to the orig code */
						/* if the PPD file does not have it? */    
    #ifdef DEBUG_PPD
    debug("begin_feature()");
    #endif

    if(featuretype == (char*)NULL)		/* Ignore blank feature lines. */
    	return;

    /*
    ** If this is bin select code and we are eating bin select code,
    ** then read until "%%EndFeature" and return.
    ** Also, if we are stripping signature code and this is signature
    ** code then do the same.
    */
    if( ( strip_binselects 
 		&& ( (strcmp(featuretype,"*InputSlot")==0) 
 		|| (strcmp(featuretype,"*TraySwitch")==0) ) )
 	|| ( strip_signature
 		&& ( (strcmp(featuretype,"*Signature")==0)
 		|| (strcmp(featuretype,"*Booklet")==0) ) )       )
        {
	#ifdef KEEP_OLD_CODE
        printer_printf("%% %s\n",line);        /* print 1st line commented out */
        #endif
        while(1)			/* process the rest of the feature */
            {
            if(dgetline(infile) == (char*)NULL)
                fatal(EXIT_JOBERR, "Unterminated feature code.");
	    #ifdef KEEP_OLD_CODE
            printer_printf("%% %s\n",line);    /* Print old line, commented out. */
            #endif
            if(strcmp(line, "%%EndFeature") == 0)
                break;
            }
        return;
        }

    /*
    ** If we are stripping bin selects, then we must change
    ** "*PaperSize" features to "*PageRegion" features.
    */
    if(strip_binselects && (strcmp(featuretype, "*PageSize") == 0))
        {
        printer_printf("%% %s\n",line);
        featuretype = "*PageRegion";
        printer_printf("%%%%BeginFeature: *PageRegion %s\n",option != (char*)NULL ? option : "");
        fallback = FALSE;		/* Don't fall back to old code, it is wrong! */
        }
    else
        printer_putline(line);		/* write the opening comment */

    /*
    ** If duplex code is not in the PPD file then the printer does
    ** not support duplex, so we want the code stript out even
    ** if the -K true switch was used with PPR.
    */
    if(strcmp(featuretype, "*Duplex") == 0)
        fallback = FALSE;

    /*
    ** If we don't have PPD file code for this and we 
    ** should fall back to the code in the file,
    ** copy it.
    */
    if( (string=find_feature(featuretype, option)) == (char*)NULL && fallback )
        {
	while(TRUE)
            {
            if(dgetline(infile) == (char*)NULL)
                fatal(EXIT_JOBERR, "Unterminated feature code.");
            printer_putline(line);                  /* origional code. */
            if(strcmp(line, "%%EndFeature") == 0)
                break;
            }
        }

    /*
    ** If the PPD file has code for this or
    ** fallback is FALSE, eat the code from here `til
    ** the %%EndFeature comment.
    */
    else
        {
        while(TRUE)
            {
            if(dgetline(infile) == (char*)NULL)
                fatal(EXIT_JOBERR, "Unterminated feature code.");
            if(strcmp(line, "%%EndFeature") == 0)
                break;
	    #ifdef KEEP_OLD_CODE
	    /* Print old code line commented. */
            printer_printf("%% %s\n", line);
            #endif
            }

	/* If the PPD file has code for this, */
        if(string != (char*)NULL)
            {					/* then write it */
            if(option == (char*)NULL)		/* as a */
                printer_puts_QuotedValue(string); /* quoted string */
            else				/* or as */
                printer_puts(string);		/* an invokation value. */

	    printer_putc('\n');
            }
	else
	    {
	    printer_putline("% feature not in PPR's PPD file");
	    }
        printer_putline("%%EndFeature");  /* and flag the end */
        }
    } /* end of begin_feature() */
                                                                             
/*
** PostScript stopped context.
**
** These routines emmit PostScript code which surrounds any feature 
** invokation that we insert.  This bracketing code catches errors
** so that if, say, the duplex command failes, the job will still
** print.
*/
void begin_stopped(void)
    {
    printer_putline("[ { %PPR");
    } /* end of begin_stopped() */
    
void end_stopped(char *feature, char *option)
    {
    printer_printf("} stopped {(PPD error: %s %s failed\\012)print} if cleartomark %%PPR\n",feature,option);
    } /* end of end_stopped() */

/*
** Insert one feature for each "Feature:" line read from qstream.
** This is called twice.  When it is called with set==1, is will 
** insert all code which does not set the duplex mode, when it is
** called with set==2, it will insert just the duplex setting code.
**
** This must be called with set==1 before it is called with set==2.
**
** Most of the features inserted by this routine will have been
** requested with the "ppr -F" switch or as a result of "ppr -R duplex"
** automatic duplexing.
*/
char *duplex_code=(char*)NULL;

void insert_features(FILE *qstream, int set)
    {
    char fline[256];
    char featuretype[32];
    char option[32];

    if(set==2)		/* If this is set 2, insert the code for the */
	{		/* remembered duplex feature. */
	if(duplex_code!=(char*)NULL)
	    {
	    printer_putline("[0 0 0 0 0 0] currentmatrix %PPR");
	    begin_stopped();
            include_feature("*Duplex", duplex_code);
	    end_stopped("*Duplex", duplex_code);
            printer_putline("setmatrix %PPR");
            }
	return;
	}
	
    duplex_code=(char*)NULL;	/* just to be on the safe side */

    while( fgets(fline,sizeof(fline),qstream) != (char*)NULL )
        {                   /* work until end of file */
        option[0]=(char)NULL;

        if( strcmp(line,"EndSetups") == 0 )
            break;

        if( ppr_sscanf(fline,"Feature: %#s %#s",
                            sizeof(featuretype),featuretype,
                            sizeof(option),option) == 0 )
            continue;       /* skip non-"Feature:" lines */
  
	/*
	** If the code to be inserted is a duplex feature command,
	** remember it for insertion during pass 2, otherwise,
	** just insert it now.
	*/
        if(strcmp(featuretype, "*Duplex") == 0)
            {
	    duplex_code = mystrdup(option);
            }
        else
            {
	    begin_stopped();
            include_feature(featuretype, option);
            end_stopped(featuretype, option);

	    /*
	    ** New as of 1.30:  Explicitly selecting the input slot causes 
	    ** old bin select code to be stript out and PageSize code
	    ** to be turned into PageRegion code.
	    */
	    if(strcmp(featuretype, "*InputSlot") == 0)
	    	strip_binselects = TRUE;
            }
        } /* end of while loop which itemizes features to insert */

    } /* end of insert_features() */

/*================================================================
** find_font() is called from pprdrv_res.c.
** find_font() returns non-zero if it can't find the font
** in the list from the PPD file. 
================================================================*/
int find_font(char *fontname)
    {
    int h;               
    struct PPDFONT *p;              /* pointer to font structure */

    h=hash(fontname,FONT_TABSIZE);
    p=ppdfont[h];

    while( p != (struct PPDFONT *)NULL )
        {
        if( strcmp(fontname,p->name) == 0 )
            return 0;
        p=p->next;
        }

    return -1;
    } /* end of find_font() */

/*==================================================================
** Move the paper size array index.
==================================================================*/
void papersize_moveto(char *nameline)
    {
    struct PAPERSIZE *p;
    char name[32];
    char *ptr;
    int len;

    ptr=&nameline[strcspn(nameline," \t")];	/* extract the */
    ptr+=strspn(ptr," \t");			/* PageSize name */
    len=strcspn(ptr," \t:/");			/* from the line */
    len=len<=31?len:31;				/* truncate if necessary */
    sprintf(name,"%.*s",len,ptr);

    #ifdef DEBUG_PPD
    debug("papersize_moveto(\"%s\")",name!=(char*)NULL ? name : "<NULL>");
    #endif

    for(papersizex=0;papersizex<num_papersizes;papersizex++)
    	{
    	if(strcmp(papersize[papersizex].name,name)==0)
    	    return;
    	}
        
    if( (papersizex=num_papersizes++) >= MAX_PAPERSIZES )
    	fatal(EXIT_PRNERR_NORETRY,"pprdrv: MAX_PAPERSIZES is not large enough");
    
    p=&papersize[papersizex];
    p->name=mystrdup(name);
    p->width=p->height=p->lm=p->tm=p->rm=p->bm=0;
    } /* end of papersize_moveto() */

/* end of file */

