/*
** ~ppr/src/pprdrv/pprdrv_lw_messages.c
** Copyright 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** Last modified 20 January 1997.
*/

#include "global_defines.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include "pprdrv.h"

/* No attempt is made to translation raw messages longer than this: */
#define MAX_RAW_LENGTH 100
#define MAX_TRANSLATION 50

/*
** The routine regularize_lw_messages() looks up LaserWriter-style
** error and status messages up in a file and returns a standard
** message if the raw message is known.
**
** This routine will be called from a signal handler,
** therefore they will not use the stdio library routines.
*/
char *regularize_lw_message(const char *message_file, const char *raw_message)
    {
    int raw_message_len;		/* length of argument 2 */
    int fd;				/* handle of translation file */
    char buffer[1025];
    int existing_bytes;			/* length of partial line in buffer from last read() */
    int new;				/* return value from last read() */
    char *linestart, *lineend;
    static char last_translation[MAX_TRANSLATION+1];	/* name of last section heading */
    char *translation = (char*)NULL;			/* what we will return */
    
    #ifdef DEBUG_LW_MESSAGES
    debug("regularize_lw_messages(message_file=\"%s\", raw_message=\"%s\")", message_file, raw_message);
    #endif

    /*
    ** Set the last section heading name to "" in case the
    ** translation file is missing its first section header.
    */
    last_translation[0] = (char)NULL;

    /* Knowing how long the raw message is will make comparisons easier.
       We also want to protect later routines from array bounds violations.
       */
    if( (raw_message_len = strlen(raw_message)) > MAX_RAW_LENGTH )
	{
	#ifdef DEBUG_LW_MESSAGES
	debug( "regularize_lw_messages(): raw message length %d exceeds MAX_RAW_LENGTH %d", raw_message_len, MAX_RAW_LENGTH );
	#endif
    	return (char*)NULL;
	}

    /* Open the message translation file: */
    if( (fd=open(message_file, O_RDONLY)) == -1 )
    	{
    	error("Can't open \"%s\", errno=%d (%s)", message_file, errno, strerror(errno) );
	return (char*)NULL;
    	}

    /*
    ** This loop reads all the lines in the file without
    ** the benefit of the stdio library.
    */
    existing_bytes = 0;
    while( translation == (char*)NULL && (new=read(fd, buffer + existing_bytes, 512)) > 0 )
	{
	buffer[existing_bytes+new] = (char)NULL;

	/* Look thru all the complete lines in the buffer: */
	for(linestart = buffer; *linestart; linestart = lineend + 1)
	    {
	    if( (lineend = strchr(linestart, '\n')) == (char*)NULL )
		{
		existing_bytes = strlen(linestart);
		if(existing_bytes)
		    memmove(buffer, linestart, existing_bytes);
		break;
		}

            #ifdef DEBUG_LW_MESSAGES_GRITTY
	    debug("regularize_lw_messages(): \"%.*s\"", lineend-linestart, linestart);
            #endif

	    /* If the line is a comment, */
	    if(*linestart == ';' || *linestart == '#')
	        {
	        continue;
	        }

	    /* If it is a section heading, remember it: */
	    else if(*linestart == '[')
		{
	    	int len = strcspn(linestart + 1, "]");
	    	if(linestart[1 + len] == ']')
		    {
		    if(len > MAX_TRANSLATION) len = MAX_TRANSLATION;
		    strncpy(last_translation, linestart + 1, len);
		    last_translation[len] = (char)NULL;
		    }
		#ifdef DEBUG_LW_MESSAGES_GRITTY
		debug( "pprdrv_lw_messages(): last_translation is now \"%s\"", last_translation );
		#endif
		}

	    /* If it is a raw message string, see if it matches: */
	    else if(*linestart == '\"')
		{
		int len = strcspn(linestart + 1, "\"");
		if(linestart[1 + len] == '\"')
		    {
		    if(len == raw_message_len && strncmp(linestart + 1, raw_message, len) == 0)
		    	{
			translation = last_translation;

                        #ifdef DEBUG_LW_MESSAGES
			debug( "regularize_lw_messages(): raw message string matched to \"%s\"", translation );
                        #endif

			break;
		    	}
		    }
		}	    
	    } /* end of for() */
	} /* end of while() */
    
    if( new == -1 )	/* read error */
	{
    	error("regularize_lw_message(): read() failed, errno=%d (%s)", errno, strerror(errno) );
	}
    else if(translation == (char*)NULL)		/* raw message not found in file */
	{
	int fd2;
	char temp[MAX_RAW_LENGTH+100];
	
        #ifdef DEBUG_LW_MESSAGES
        debug( "regularize_lw_messages(): no match");
        #endif

	if( (fd2=open(message_file, O_WRONLY | O_APPEND, UNIX_644)) == -1 )
	    {
	    error("regularize_lw_message(): can't open \"%s\", errno=%d (%s)", message_file, errno, strerror(errno) );
	    }
	else
	    {
	    temp[0] = (char)NULL;
	    
	    if(last_translation[0] != (char)NULL)
	    	strcat(temp, "\n"
	    		"; Automatically generated section for new messages.\n"
	    		"; Should be left at end of file.\n"
	    		"[]\n");

	    strcat(temp, "\"");
	    strcat(temp, raw_message);
	    strcat(temp, "\"\n");
	    
	    write(fd2, temp, strlen(temp));
	    close(fd2);	    
	    }
	}
	
    close(fd);
    
    #ifdef DEBUG_LW_MESSAGES
    if(translation == (char*)NULL)
	debug("regularize_lw_messages(): returning (char*)NULL" );
    else
	debug("regularize_lw_messages(): returning \"%s\"", translation);
    #endif

    return translation;
    } /* end of regularize_lw_message() */

/* end of file */
