/*
** ~ppr/src/pprdrv/pprdrv_flag.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 27 February 1997.
*/

/*
** Header and trailer page generation code.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include "pprdrv.h"
#include "interface.h"
#include "userdb.h"

/*
** things in other modules 
*/
extern FILE *intstdin;
extern struct QFileEntry job;
extern char *QueueFile;			/* queue file name without dir */
extern struct PPRDRV printer;
extern int sheetcount;			/* in pprdrv.c */

/*
** Global variables for this module.
** For the sake of neatness these are gathered into a structure.
*/
struct 
    {                                                          
    char *medianame;            /* name of selected media */
    double ph;                  /* page height in 72ths of an inch */
    double pw;                  /* page width in 72ths of an inch */
    double weight;
    char *colour;
    char *type;
    int mar;                    /* margin */
    int bor;                    /* border */
    int large_size;             /* point size of large type */
    int small_size;             /* point size of small type */
    int log_size;               /* point size of log file type */
    } flag;

/*
** If job.ForLine consists of two words, reverse them and 
** seperate them with a comma.
**
** In any event, return a pointer to what we should use 
** on the banner page.
**
** This function is called from pprdrv.c to set a global character pointer.
*/
static const char *reversed_for(void)
    {
    char *spc_ptr;
    static const char *result = (char*)NULL;

    /* If ForLine is missing, return "(unspecified)" */
    if(job.ForLine==(char*)NULL)
    	return "(unspecified)";

    /* If we have already figured it out, return the old result. */
    if(result != (char*)NULL)
    	return result;

    /* If search for a space fails, stop now. */
    if( (spc_ptr=strchr(job.ForLine,' ')) == (char*)NULL )
        result = job.ForLine;

    /* If search for another space succeeds, stop now. */
    else if( strchr(spc_ptr+1,' ') != (char*)NULL )
        result = job.ForLine;

    else
    	{
	/* Get a block to hold all the characters, a NULL and an added comma. */
	char *ptr = (char*)myalloc(strlen(job.ForLine)+2,sizeof(char));

	strcpy(ptr, spc_ptr+1);		/* take the part after the space */
	strcat(ptr, ", ");		/* add a comma and a space */
	*spc_ptr = (char)NULL;          /* replace origional space with null */
	strcat(ptr, job.ForLine);	/* add null terminated 1st part */
	*spc_ptr=' ';                   /* put the space back */

	result = ptr;
	}

    return result;
    } /* end of reverse_for() */

/*                                                                          
** Tabulate the votes as to whether we should have a banner or trailer page.
** Return TRUE if we should, FALSE if we should not.
**
** Printer will vote BANNER_REQUIRED, BANNER_ENCOURAGED, BANNER_DISCOURAGED,
** BANNER_FORBIDDEN.  Job will vote BANNER_YESPLEASE, BANNER_DONTCARE,
** or BANNER_NOTHANKYOU.
**
** New feature: if there is a log and this flag is being printed last, and
** the banner or trailer, as applicable, is not prohibited, print the flag
** even if the user or printer configuration have not requested it.
*/
static int flag_page_vote(int printer_vote, int job_vote, int position)
    {
    #ifdef DEBUG_FLAGS
    debug("flag_page_vote(%d,%d)",printer_vote,job_vote);
    #endif

    switch(printer_vote)
        {                      
        case BANNER_FORBIDDEN:			/* if printer doesn't allow, */
            return FALSE;			/* never do it */
        case BANNER_DISCOURAGED:		/* if printer discourages */
            if(job_vote==BANNER_YESPLEASE)	/* then, only if asked for */
                return TRUE;
            else
                return FALSE;
        case BANNER_ENCOURAGED:
            if(job_vote==BANNER_NOTHANKYOU)
                return FALSE;
            else
                return TRUE;
        case BANNER_REQUIRED:
            return TRUE;
        default:                                /* this is an error */
            fatal(EXIT_PRNERR_NORETRY,"pprdrv: flag_page_vote(): printer_vote=%d",printer_vote);
        }
    } /* end of flag_page_vote() */

/*
** Decide which media will be used for the flag.
**
** Fill in the structure "flag" with the statistics about the suitable
** media.  The actual bin selection is done later using select_media().
**
** If no bin has suitable media, return -1, if one is found, return 0,
** if no bins are known to exist, return 1.
**
** We will fill in the flag structure with the dimensions of the selected
** media.  If there are no bins and we return 1, we will fill in the flag
** structure with the dimensions of US letter size media.  This should be
** close enough to A4 that the banner page will not look too bad.  The code 
** that calls this routine will not attempt to select the input tray or
** paper size of we return 1.
*/
static int select_flag_media(void)
    {                                       
    static int select_flag_media_count=0;   /* tells if we were called yet */
    static int rval=-1;			    /* error indicator to return */
    FILE *mediafile;                        /* list of all media */
    struct Media media[MAX_BINS];           /* media in each bin */
    struct Media t;                         /* temporary, for reading */
    static char _medianame[MAX_MEDIANAME+1];
    static char _colour[MAX_COLOURNAME+1];
    static char _type[MAX_TYPENAME+1];
    int x, y;	    	    	    	    /* for looping */
    int bincount;                           /* # of entries in mounted file */
    int found_count=0;                      /* # found in media file */

    #ifdef DEBUG_FLAGS
    debug("select_flag_media()");
    #endif

    /* If this routine was called before, just return the same value as */
    /* we returned last time.  The structure "flag" is still valid. */
    if(select_flag_media_count++ > 0)
        {
        #ifdef DEBUG_FLAGS
        debug("proper media already determined");
        #endif
        return rval;        /* (on 2nd and subsequent calls, do nothing) */
        }
                                      
    /* The defaults.  These come into play if we have no bins. */
    #ifndef A4
    flag.medianame="letter";
    flag.pw=612;                /* 8.5 inches */
    flag.ph=792;                /* 11 inches */
    #else
    flag.medianame="a4";
    flag.pw=595;
    flag.ph=842;
    #endif
    flag.weight=0;
    flag.colour="white";
    flag.type="";               /* blank paper */

    /*
    ** Load the list of mounted media and look each one up in the
    ** media database to get its suitability rating.
    */
    if( (bincount=load_mountedlist()) > 0 )   /* if mounted list found and not zero bins, */
        {
        for(x=0;x<MAX_BINS;x++)
            media[x].flag_suitability=0;

        if((mediafile=fopen(MEDIAFILE, "r")) == (FILE*)NULL)
            fatal(EXIT_PRNERR_NORETRY,"Error %d opening \"%s\"",errno,MEDIAFILE);

	/* find each media */
        while((fread(&t,sizeof(struct Media),1,mediafile) == 1) 
                && (found_count < bincount) )
            {
            for(x=0;x<bincount;x++)
                if(strncmp(t.medianame,mounted[x].media,MAX_MEDIANAME)==0)
                    {
                    memcpy(&media[x],&t,sizeof(struct Media));
                    found_count++;
                    #ifdef DEBUG_FLAGS
                    debug("one match!");
                    #endif
                    }
            }

        fclose(mediafile);

	/* 
	** Examine the media which we have just determined is mounted
	** on each bin in order to find the most suitable.
	** y is initialy set to 1 because media with a suitability rating
	** of one is not usable for banner pages. 
	*/
        for(x=0,y=1;x<bincount;x++)         /* find the most suitable of */
            {                               /* those suitable */
            if( media[x].flag_suitability > y )
                {
                y=media[x].flag_suitability; /* this is best found yet */

                flag.medianame=_medianame;	/* we use this */
                padded_to_ASCIIZ(_medianame,media[x].medianame,MAX_MEDIANAME);

                flag.pw=media[x].width;		/* we use this */
                flag.ph=media[x].height;	/* we use this */
                flag.weight=media[x].weight;	/* we don't use this */

                flag.colour=_colour;	/* we don't use this */
                padded_to_ASCIIZ(_colour,media[x].colour,MAX_COLOURNAME);

                flag.type=_type;	/* we don't use this */
                padded_to_ASCIIZ(_type,media[x].type,MAX_TYPENAME);

		rval=0;		/* we did it! */
                }
            }

        } /* end of if sucessfully opened mounted file */
    else
	{
	#ifdef DEBUG_FLAGS
	if(bincount==-1)
            debug("no mounted media list found, assuming zero bins");
        else
            debug("zero bins found");
	#endif
	rval=1;		/* arrange to return 1 */
	}

    /* compute some things based on the selected form size */
    flag.mar = (int)( ( flag.ph > flag.pw ? flag.pw : flag.ph ) / 15 );
    flag.bor = (int)(flag.mar * 0.70);
    flag.large_size = (int)(flag.ph / 20);   
    flag.small_size = (flag.large_size / 3);
    flag.log_size = (int)((flag.pw - flag.mar) / 80 / 0.6);

    return rval;    
    } /* end of select_flag_media() */

/*
** If proper, print a banner or trailer page.
**
** Flag type of 1 is header, -1 is trailer.
** position is 0 if at start of job, 1 if at end of job.
*/
void print_flag_page(int flag_type, int position)
    {
    FILE *logfile;
    char fname[MAX_PATH];
    char buffer[82];			/* buffer for reading log file */
    time_t timenow;			/* for computing time */
    struct tm *timeptr;			/* at which job was printed */
    char jobid_str[MAX_DESTNAME+15];	/* QueueFile with trailing ".0" stript */
    int len;				/* temporary string length storeage */
    int mediafound;

    #ifdef DEBUG_FLAGS
    debug("print_flag_page(flag_type=%d, position=%d)", flag_type, position);
    #endif

    /* decide if we should print a flag page */
    if(flag_type < 0)           /* trailer page */
        {
        if( !flag_page_vote(printer.do_trailer,job.do_trailer,position) )
            return;
        }
    else                        /* banner page */
        {
        if( !flag_page_vote(printer.do_banner,job.do_banner,position) ) 
            return;
        }

    #ifdef DEBUG_FLAGS
    debug("printing a flag page");
    #endif

    /*
    ** Look for the correct media.  If select_flag_media() indicates
    ** that bins exist but none of them have suitable media mounted,
    ** abort flag printing now. 
    */
    if((mediafound=select_flag_media()) == -1)
        {
	#ifdef DEBUG_FLAGS
        debug("all mounted media is unsuitable for flag pages");
        #endif
        return;
        }

    if(position)			/* Trailing flag needs a job break */
        jobbreak();			/* at its begining. */

    /* format the job name string */
    if( (len=strlen(QueueFile)) > 2)	/* just be very, very safe */
	{
    	if(strcmp(&QueueFile[len-2], ".0") == 0)
    	    {
    	    strncpy(jobid_str, QueueFile, len-2);
    	    jobid_str[len-2] = (char)NULL;
    	    }
	else
	    {
	    strcpy(jobid_str, QueueFile);
	    }
    	}
    
    /* Send the PostScript DSC comments section to the printer */ 
    printer_puts("%!PS-Adobe-3.0\n");
    printer_puts("%%Creator: PPR\n");

    if(flag_type<0)
        printer_puts("%%Title: Trailer Page\n");
    else
        printer_puts("%%Title: Banner Page\n");

    printer_puts("%%DocumentNeededResources: font Courier\n");
    printer_puts("%%Pages: 1\n");
    printer_printf("%%%%DocumentMedia: %s %f %f %f (%s) (%s)\n",
                flag.medianame, flag.pw, flag.ph, flag.weight,
                flag.colour, flag.type);
    printer_puts("%%EndComments\n");

    /* send the prolog to the printer */
    printer_puts("%%BeginProlog\n");

    /* newline routine */
    printer_printf("/nl{/y y ys sub def}bind def\n"); 

    /* printline routine */
    printer_printf("/p{/y y ys 0.7 mul sub def %d y moveto show "
            "/y y ys 0.3 mul sub def}bind def\n",flag.mar);

    printer_puts("%%EndProlog\n");

    /* send the document setup section */
    printer_puts("%%BeginSetup\n");

    if(mediafound==0)                       /* If at least one bin had it, */
	{
	int x;

	/*
	** Select the correct bin.  We will ignore the return
	** value because it should always be TRUE.  In fact, I
	** find it difficult to imagine how it could be FALSE. 
	*/
        select_media(flag.medianame);

	/* 
	** Look thru the paper sizes, and if we find one
	** that matches, use it. 
	*/
	for(x=0;x<num_papersizes;x++)
	    {
	    if( papersize[x].width==flag.pw && papersize[x].height==flag.ph )
	    	{
		begin_stopped();
		include_feature("*PageRegion",papersize[x].name);
		end_stopped("*PageRegion",papersize[x].name);
		break;
	    	}
	    }

        } /* end of if(mediafound==0) */

    /* 
    ** Theoretically, we insert the font here.  The origional code was:
    ** _include_resource("font","Courier",0.0,0);
    ** however, it won't work anymore since _include_resource() can now 
    ** only include resources which are pre-declared.  We really want
    ** the default font if Courier is not available, so we will say:
    */
    printer_puts("%%IncludeResource: font Courier\n");
     
    /*
    ** Scale the font to two sizes.  Large is for the user name,
    ** small is for the other messages.
    */
    printer_printf("/LargeFont /Courier findfont %d scalefont def\n",flag.large_size);
    printer_printf("/SmallFont /Courier findfont %d scalefont def\n",flag.small_size);

    /* Define routines to select the two font sizes. */
    printer_printf("/large_type{LargeFont setfont /ys %d def}bind def\n",(int)(flag.large_size*1.3));
    printer_printf("/small_type{SmallFont setfont /ys %d def}bind def\n",(int)(flag.small_size*1.3));

    printer_puts("%%EndSetup\n");

    /* send the first part of the page */
    printer_puts("%%Page: one 1\n");
    printer_puts("save\n");
    printer_printf("newpath %d dup moveto 0 %d rlineto %d 0 rlineto 0 -%d rlineto ",
        (int) flag.bor,
        (int) (flag.ph-(flag.bor*2)),
        (int) (flag.pw-(flag.bor*2)),
        (int) (flag.ph-(flag.bor*2)) );
    printer_puts("closepath stroke\n");
    printer_printf("/y %d def\n",(int)(flag.ph-flag.mar));

    /* print the text of the page */
    if(flag_type < 0)		/* trailer page */
        {                   
        printer_puts("large_type ("); printer_puts_escaped(reversed_for()); printer_puts(")p\n");
        printer_printf("small_type (End of job %s)p\n",jobid_str);

	if(job.Title!=(char*)NULL)
	    {
	    printer_puts("(Title: "); printer_puts_escaped(job.Title); printer_puts(")p\n");
	    } /* done remove braces */

        }
    else			/* header page */
        {
        printer_puts("large_type ("); printer_puts_escaped(reversed_for()); printer_puts(")p small_type\n");

        printer_printf("(Job ID: %s)p\n", jobid_str);

        if(job.Routing != (char*)NULL)
            {
            printer_puts("(Routing:  "); printer_puts_escaped(job.Routing); printer_puts(")p\n");
            } /* don't remove {} */                      

        if(job.Creator != (char*)NULL)
            {
            printer_puts("(Creator: "); printer_puts_escaped(job.Creator); printer_puts(")p\n");
            } /* don't remove {} */

        if(job.Title != (char*)NULL)
            {
            printer_puts("(Title: "); printer_puts_escaped(job.Title); printer_puts(")p\n");
            } /* don't remove {} */                                    

        /* print the current time */
        time(&timenow);
        timeptr = localtime(&timenow);
        strftime(buffer, sizeof(buffer), "(Print Date: %d-%b-%y, %I:%M%p)p\n",timeptr);
        printer_puts(buffer);

        if(job.attr.pages >= 0)			/* if # of pages known */
            {
            printer_printf("(Pages: %d)p\n", job.attr.pages);   /* virtual pages */  
            printer_printf("(Sheets: %d)p\n", sheetcount);      /* sheets of paper */
            }

        if(job.opts.copies > 1)			/* if multiple copies, */
            printer_printf("(Copies: %d)p\n", job.opts.copies);

	/*
	** If we are charging for printing, say so now.
	** Notice that if the destination was not "protected"
	** at the time the job entered the queue then job.charge_to 
	** will be NULL.
	*/
        if(job.charge_to != (char*)NULL && (printer.charge_per_duplex > 0 || printer.charge_per_simplex > 0))
            {
            struct userdb db;

            if(db_auth(&db, job.charge_to) == USER_ISNT)
                {
                error("User \"%s\" is inexplicably absent from the charge account database.\n", job.charge_to);

		printer_printf("(User \"");
		printer_puts_escaped(job.charge_to);
		printer_printf("\" is inexplicably absent from the charge account database.)p\n");
                }
	    else
	    	{
		struct COMPUTED_CHARGE charge;

                printer_printf("(Balance before printing this job: ");
                printer_puts_escaped(money(db.balance));
                printer_puts(")p\n");

		compute_charge(&charge, printer.charge_per_duplex,
			printer.charge_per_simplex,
			job.attr.pages,
			job.N_Up.N,
			job.attr.pagefactor,
			job.N_Up.sigsheets,
			job.N_Up.sigpart,
			job.opts.copies);

		printer_puts("(Charge for this job: ");

		if(charge.duplex_sheets)
		    {
		    printer_printf("%d duplex sheet%s @ ", charge.duplex_sheets, charge.duplex_sheets > 1 ? "s" : "");
		    printer_puts_escaped(money(charge.per_duplex));
		    }

		if(charge.simplex_sheets)
		    {
		    if(charge.duplex_sheets)
		    	printer_puts(")p\n(      + ");
		    printer_printf("%d simplex sheet%s @ ", charge.simplex_sheets, charge.simplex_sheets > 1 ? "s" : "");
		    printer_puts_escaped(money(charge.per_simplex));
		    }

		printer_puts(" = ");
		printer_puts_escaped(money(charge.total));
                printer_puts(")p\n");
                }
            }
        }

    /* print the log file contents */
    sprintf(fname, "%s/%s-log", DATADIR, QueueFile);
    if((logfile=fopen(fname, "r")) != (FILE*)NULL)
        {
        #ifdef DEBUG_FLAGS
        debug("print_flag_page(): printing log contents");
        #endif

        printer_printf("/Courier findfont %d scalefont setfont\n", flag.log_size);
        printer_printf("/ys %d def\n",(int)(flag.log_size*1.2));
        printer_putline("nl");

        while( fgets(buffer,sizeof(buffer),logfile) != (char*)NULL )
            {
#if 0
            if( buffer[0] == '+' )                   /* skip redirect lines */
                continue;
#endif
            buffer[strcspn(buffer,"\n")]=(char)NULL; /* remove trailing space */
            printer_puts("("); printer_puts_escaped(buffer); printer_puts(")p\n");
            }

        fclose(logfile);
        if(position == 0)   	/* if this page was printed before the job, */
            unlink(fname);  	/* then unlink the log file */
        }

    /* close the page */
    printer_printf("restore showpage\n");

    /* close the postscript */
    printer_puts("%%Trailer\n");
    printer_puts("%%EOF\n");

    /* banner page needs a job break after it */
    if(position==0)            
        jobbreak();
    }

/* end of file */
