/*
** ~ppr/src/pprdrv/pprdrv_feedback.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 23 January 1997.
*/

/*
** Handle data coming back from the interface.  This will generally be
** messages sent by this printer.  These messages include notification of
** PostScript errors, printer status messages, and anything the PostScript
** job writes to stdout.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <ctype.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#ifndef NO_POLL			/* if we have poll() */
#include <stropts.h>
#include <poll.h>
#endif
#include "pprdrv.h"
#include "interface.h"

/*
** Things in other modules:
*/
extern int intstdout;       /* handle of pipe from the interface program */
extern char *QueueFile;     /* name of this job (queue file name w/out path) */

/* 
** For us to use in representing unprintable characters
** received from the printer.
*/
const char hexchars[] = "0123456789ABCDEF";

/*
** Global variables for this module.
*/
int posterror = FALSE;			/* true if a PostScript error is detected */
int ghostscript = FALSE;		/* We think we are talking to Ghostscript */
int status_updates_follow = FALSE;	/* cleared by pprdrv.c:jobbreak() */

/*
** Statics for this module.
*/
static int logfile = -1;		/* possibly open log file */
static int pjl_in_job = FALSE;

/*
** This is called to open the job's log file.  If we are running
** in test mode, stderr is used in stead.
*/
static void open_log(void)
    {
    DODEBUG_FEEDBACK(("open_log()"));

    if(test_mode)
    	{
    	logfile = 2;
    	}
    else
    	{
	char fname[MAX_PATH];
	sprintf(fname, DATADIR"/%s-log", QueueFile);
	if( (logfile=open(fname, O_WRONLY | O_CREAT | O_APPEND, UNIX_644)) == -1 )
	    fatal(EXIT_PRNERR, "pprdrv_feedback.c: open_log(): can't open \"%s\" for append, errno=%d (%s)", fname, errno, strerror(errno) );
	}
    } /* end of open_log() */

/*
** Close the log file if it is open.  This routine is needed because
** the routine sigpoll_handler() tends to open the log file and leave 
** it open.  This routine is called from pprdrv_flag.c.
*/
void close_log(void)
    {
    DODEBUG_FEEDBACK(("close_log()"));

    if(logfile != -1)
	{
	if(!test_mode)
	    close(logfile);
	logfile = -1;
	}
    } /* end of close_log() */

/*
** Updates the printer's .status file.  Of course, in test mode it just write the 
** data to stderr.  This function is called from sigpoll_handler()
** so it should not use stdio.
*/
static void write_printer_status(const char *printer_name, const char *status)
    {
    if(test_mode)
    	{
    	write(2, "status: ", 8);
    	write(2, status, strlen(status));
	write(2, "\n", 1);
    	}
    else
	{
	char fname[MAX_PATH];
	int statfile;

	sprintf(fname, ALERTDIR"/%s.status", printer_name);
	if( (statfile=open(fname, O_WRONLY | O_CREAT | O_TRUNC, UNIX_644)) == -1 )
	    fatal(EXIT_PRNERR, "pprdrv_feedback.c: sigpoll_handler(): failed to open \"%s\" for write, errno=%d (%s)", fname, errno, strerror(errno));

	write(statfile, status, strlen(status));
	close(statfile);

	/* Put the message in the state_update_pprdrv file. */
	progress__new_message(status);
	}
    } /* end of write_printer_status() */

/*
** This signal handler receives control any time there is something to
** read on the pipe coming back from the interface.
**
** This data will be output of the PostScript "print" command, PostScript
** error messages, printer error messages such as "out of paper" and
** possibly other things.
**
** Remember that since this is a signal handler, we must not call
** non-reentrant functions such as malloc().
*/
#define PJL_INITIAL 0
#define PJL_USTATUS_PAGE 1
#define PJL_USTATUS_JOB 2
void sigpoll_handler(int signum)
    {
    static int already_here=0;
    static unsigned char buf[4097];	/* buffer for reading from interface plus a NULL */
    static pjl_state=PJL_INITIAL;
    int buflen,linelen;			/* length of this read(), length of current line */
    int x;				/* used for stepping thru the buffer */
    int c,lastc,y;			/* for line termination conversion */
    unsigned char *ptr;			/* pointer to next line in buffer */
    int saved_errno = errno;		/* save errno of interupted code */

    DODEBUG_FEEDBACK(("sigpoll_handler()"));

    /*
    ** Loop to read the data from the pipe.
    ** The condition at the end of this do...while()
    ** loop is "while(buflen)". 
    */
    do  {
	DODEBUG_FEEDBACK(("sigpoll_handler(): top of loop, already_here=%d", already_here));

        /* Read from interface's stdout/stderr. */
        while( (buflen=read(intstdout, buf+already_here, 4096-already_here)) == -1 )
	    {
	    if(errno == EAGAIN)		/* Handle the result of Posix */
	    	{			/* non-blocking mode (O_NDELAY */
	    	buflen = 0;		/* mode causes read() to return */
	    	break;			/* zero, O_NONBLOCK causes EAGAIN). */
	    	}
	    if(errno != EINTR)
	    	fatal(EXIT_PRNERR, "pprdrv_feedback.c: sigpoll_handler(): error reading from interface, errno=%d (%s)", errno, strerror(errno));
	    }
	    
	DODEBUG_FEEDBACK(("read %d new bytes", buflen));

	buflen += already_here;		/* Add stuff that was already in buffer. */
	already_here = 0;

	DODEBUG_FEEDBACK(("%d bytes now in buffer", buflen));

	/* 
	** Convert line endings.  This includes converting carriage 
	** return termination to line feed termination and converting 
	** CRLF termination to LF termination. 
	*/
	lastc = -1;
	for(x=0, y=0; x < buflen; x++)
	    {
	    c=buf[x];

	    if(c=='\r')			/* Carriage returns get */
	    	{			/* converted to line feeds. */
	    	buf[y++]='\n';
	    	}
	    else if(c=='\n')		/* Line feeds get discarded unless */
	    	{			/* they don't */
	    	if(lastc != '\r')	/* follow CR's. */
	    	    buf[y++]=c;
	    	}
	    else			/* everything else, */
	    	{			/* gets copied. */
	    	buf[y++]=c;
	    	}

	    lastc=c;
	    }

	#ifdef DEBUG_FEEDBACK
	if(buflen != y)
	    debug("line ending termination has changed buffer size to %d bytes",y);
	#endif

	buflen=y;			/* buffer may have been shortened */
	buf[buflen]=(char)NULL;		/* Terminate buffer as though a string. */

	/*
	** If incomplete last line and we can afford 
	** to wait, do so.  This is a hack for Ghostscript
	** which sends the first line of its error message
	** in two writes. 
	**
	** We accept form-feed as a line terminator because
	** HP PJL messages from the printer will contain them.
	*/
	if( buflen > 0 && buflen < 100 && buf[buflen-1]!='\n' && buf[buflen-1]!='\f' )
	    {
	    already_here = buflen;
	    DODEBUG_FEEDBACK(("sigpoll_handler(): defering processing of partial line"));
	    errno = saved_errno;
	    return;
	    }

	/*
	** Handle the whole buffer one line at a time.
	** ptr always equals buf[x]
	**
	** Within this loop, ptr will point to the current line
	** and its length will be stored in linelen.
	*/
	for(x=0,ptr=buf; x<buflen; x+=linelen,ptr+=linelen)
	    {
	    if(*ptr == '\f')			/* Form feed ends PJL response */
	    	{
	    	linelen = 1;			/* it is a one character line */
		pjl_state = PJL_INITIAL;
	    	continue;
	    	}

	    /*
	    ** Determine the length of the next line by
	    ** scanning for the next line feed.
	    */
	    for(linelen=0; (x+linelen) < buflen; linelen++)
	    	{
	    	if(ptr[linelen] == '\n')
	    	    {
	    	    linelen++;
	    	    break;
	    	    }
	    	}

	    #ifdef DEBUG_FEEDBACK_LINES
	    debug("considering line: \"%.*s\"", linelen>0 ? (linelen-1) : 0, ptr);
	    #endif

	    /*
	    ** Handle @PJL page printed messages with their various states.
	    */
	    if(pjl_state == PJL_USTATUS_PAGE)
	    	{
		if(doing_primary_job)
		    progress__pages_truly_printed(job.N_Up.N);	/* one sheet side (two messages for duplex) */
		pjl_state = PJL_INITIAL;			/* page message is not over */
		continue;
	    	}
	    if(pjl_state == PJL_USTATUS_JOB)			/* We expect one or more lines */
	    	{						/* after a "@PJL USTATUS JOB" line, */
		if(strncmp((char*)ptr, "START", 5) == 0)	/* so we do not reset the state to initial, */
		    pjl_in_job = TRUE;				/* we let the FF do that. */
		else if(strncmp((char*)ptr, "END", 3) == 0)
		    pjl_in_job = FALSE;
	    	continue;
	    	}
	    if(strncmp((char*)ptr,"@PJL USTATUS PAGE",17)==0)
	    	{
	    	pjl_state = PJL_USTATUS_PAGE;
	    	continue;
	    	}
	    if(strncmp((char*)ptr, "@PJL USTATUS JOB", 16) == 0)
	    	{
	    	pjl_state = PJL_USTATUS_JOB;
	    	continue;
	    	}

	    /*
	    ** Handle lines in the form "%%[ PrinterError: xxxxxxxxx ]%%"
	    ** Write printer errors to the printer's status file.
	    */
	    if(strncmp((char*)ptr, "%%[ PrinterError: ", 18) == 0)
	        {
	        char *subptr;		/* will be pointer to closing `bracket' */
      	    
		/* If it has the closing bracket stuff too, */
	        if((subptr=strstr((char*)ptr," ]%%")) != (char*)NULL)
		    {
		    /* Cut the string off where the " ]%%" begins */
		    *subptr = (char)NULL;

		    /*
		    ** Write the message into the printer's ".status"
		    ** file so that it can be displayed in ppop status
		    ** listings. 
		    */
		    write_printer_status(printer.Name, &ptr[18]);

		    /*
		    ** Pass the error message to the commentators so that
		    ** they can tell their audiances about it if they feel
		    ** so compelled.
		    */
		    commentary(COM_PRINTER_ERRORS, regularize_lw_message(LW_ERRORS_FILE, (char*)&ptr[18]), (char*)&ptr[18], (char*)NULL);
	    	    } /* end of if closing bracket stuff too */
		continue;	/* keep line out of the job log */
	        }

	    /*
	    ** Handle messages of the form "%%[ status: xxxxxx ]%%"
	    ** or "%%[ job: xxxxxxx; status: yyyyyyy ]%%".
	    **
	    ** The AppleTalk interface will feed us such a message
	    ** just before it exits with EXIT_ENGAGED. 
	    */
	    if(strncmp((char*)ptr, "%%[ status: ", 12) == 0
	    		|| strncmp((char*)ptr, "%%[ job: ", 9) == 0)
	    	{
	        char *subptr, *subptr2;
      	    
		/* If it has the closing bracket stuff too, */
	        if((subptr=strstr((char*)ptr, " ]%%")) != (char*)NULL)
		    {
		    /* Start by truncating string where the " ]%%" begins. */
		    *subptr = (char)NULL;

		    /* Re-ssign subptr to point to the start of the data: */
		    subptr = &ptr[4];

		    /*
		    ** Truncate at semicolon to remove "; source: AppleTalk"
		    ** and similiar messages identifing the interface which
		    ** the current job is being read from.
		    ** That stuff would confuse the commentator too.
		    */
		    if( (subptr2=strstr(subptr, "; source:")) != (char*)NULL )
			*subptr2 = (char)NULL;
			
		    /*
		    ** Write the remaining part of the status message into the 
		    ** printer's .status file so that it can be displayed
		    ** in ppop status listings. 
		    */
		    write_printer_status(printer.Name, subptr);

		    /*
		    ** If it begins with "job: ", skip forward since the job
		    ** name would only confuse the commentator.
		    */
		    if(strncmp(subptr, "job: ", 4) == 0)
		    	{
		    	if( (subptr2=strstr(subptr, "; status: ")) != (char*)NULL )
		    	    subptr = &subptr2[2];
		    	}

		    /* 
		    ** If at the end of that we have a status message, 
		    ** pass the status message to the commentators so that 
		    ** they can tell  their audiances about it if they 
		    ** feel so compelled. 
		    */
		    if(strncmp(subptr, "status: ", 8) == 0)
		    	{
			int comtype = status_updates_follow ? COM_PRINTER_STATUS_UPDATES : COM_PRINTER_STATUS;

			DODEBUG_FEEDBACK(("status_updates_follow=%d", status_updates_follow));

			if(strncmp("PrinterError: ", &subptr[8], 14) == 0)
			    commentary(comtype, regularize_lw_message(LW_ERRORS_FILE, &subptr[22]), &subptr[8], (char*)NULL);
			else
			    commentary(comtype, regularize_lw_message(LW_STATUS_FILE, &subptr[8]), &subptr[8], (char*)NULL);
			}
	    	    }
		continue;	/* keep this line out of the job log */
	    	}

	    /*
	    ** This line will tell us that any "%%[ status: ]%%"
	    ** or "%%[ job:  ; status ]%%" lines which follow are
	    ** the result of the interface program requesting status
	    ** updates while the job is printing.
	    */
	    if(linelen == 34 && strncmp((char*)ptr, "%%[ PPR status updates follow ]%%\n", linelen) == 0)
		{
		DODEBUG_FEEDBACK(("Detected status updates precursor"));
		status_updates_follow = TRUE;
		continue;	/* don't log this line */
		}

	    /*
	    ** If it is a PostScript error, set a flag so that the
	    ** job will be arrested as soon as it is done and call
	    ** a routine which analyzes the error message and attempts
	    ** to write an inteligent description of it into the
	    ** job's log file. 
	    */
            if(strncmp((char*)ptr, "%%[ Error: ", 11) == 0)
                {					
		char *error, *command;
		char *end_error, *end_command;

		/* Set the flag which indicates a PostScript error: */
		posterror = TRUE;

		/* Find the error string. */
		error = (char*)&ptr[11];

		/* Find the offending command in the error message: */
		command = strstr((char*)ptr, "OffendingCommand: ");
		if(command != (char*)NULL) command += 18;

		/* Terminate each one with a NULL */
		if( (end_error=strchr(error, ';')) != (char*)NULL )
		    *end_error = (char)NULL;
		end_command = (char*)NULL;
		if( command != (char*)NULL && (end_command=strchr(command,' ')) != (char*)NULL )
		    *end_command = (char)NULL;

		describe_postscript_error(error, command);

		/* put the origional characters back */
		if(end_error != (char*)NULL )
		    *end_error = ';';
		if(end_command != (char*)NULL )
		    *end_command = ' ';
		} /* drop thru so that line will be entered in job log */

	    /*
	    ** How about a Ghostscript style PostScript error message?
	    ** Normally the messages look like:
	    **
	    ** Error: /undefined in x
	    **
	    ** but sometimes they look like:
	    ** 	    
	    ** Unrecoverable error: undefined in x
	    */
	    {
	    int skiplen;
	    if(strncmp((char*)ptr, "Error: /", skiplen=8) == 0
	    		|| strncmp((char*)ptr, "Unrecoverable error: ", skiplen=21) == 0)
	    	{
		char *error, *command;
		char *end_error, *end_command;
		char end_command_char=0;	/* zero makes GNU-C happy */

		posterror = TRUE;		/* A PostScript error has occured. */
		ghostscript = TRUE;		/* We are talking to Ghostscript. */

		/* Find the error string. */
		error = (char*)&ptr[skiplen];

		/* Find the offending command */
		command = strstr((char*)ptr," in ");
		if(command != (char*)NULL) command += 4;
		while(*command == '-')
		    command++;

		/* Terminate each one with a NULL */
		if( (end_error=strchr(error,' ')) != (char*)NULL )
		    *end_error=(char)NULL;
		end_command=(char*)NULL;
		if( command!=(char*)NULL && (end_command=strpbrk(command,"-\n")) != (char*)NULL )
		    {
		    end_command_char=*end_command;
		    *end_command=(char)NULL;
		    }

		/* Hand the facts to our `inteligent' routine. */
		describe_postscript_error(error,command);

		/* put the origional characters back */
		if(end_error != (char*)NULL )
		    *end_error=' ';
		if(end_command != (char*)NULL )
		    *end_command=end_command_char;
	    	} /* drop thru so that line will be entered in job log */
	    }

	    /*
	    ** If we get this far, allow the *PatchFile query
	    ** to get a look at the line.  If the line is a reply
	    ** to a query it willask us to keep it out of the
	    ** log file.
	    */
	    if( patchfile_query_callback((char*)ptr) )
	    	continue;		

	    /*
	    ** Allow the persistent download query code to review the line.
	    */
	    if( persistent_query_callback((char*)ptr) )
	    	continue;

	    /* 
	    ** If we get this far, we should write the message
	    ** to the job's log file. 
	    */
	    {
	    char outbuf[1024];
	    int cnt;
	    
	    if(logfile == -1)
		open_log();

	    for(y=cnt=0; y < linelen && y < 256; y++)
	    	{
		if( isprint(ptr[y]) || isspace(ptr[y]) )
		    {
		    outbuf[cnt++] = ptr[y];
		    }
		else		/* no printable, represent with hexadecimal */
		    {
		    outbuf[cnt++] = '<';
		    outbuf[cnt++] = hexchars[ptr[y]/16];
		    outbuf[cnt++] = hexchars[ptr[y]%16];
		    outbuf[cnt++] = '>';
 		    }
	    	}

	    if(write(logfile, outbuf, cnt) != cnt)
	    	fatal(EXIT_PRNERR, "pprdrv_feedback.c: sigpoll_handler(): write() to job log failed");
	    }

	    } /* End of loop which considers the lines from the printer. */
        } while(buflen);	/* while last read() returned something */

    DODEBUG_FEEDBACK(("sigpoll_handler() done"));

    errno = saved_errno;		/* restore errno of thread interupted by signal */
    } /* end of sigpoll_handler() */

/*
** If we saw a PJL job start indication, wait for a PJL job
** end indication.  If there was no evidence that PJL job
** start/stop indications are turned on, do nothing.
**
** This is sometimes called from printer_cleanup().
*/
void feedback_pjl_wait(void)
    {
    DODEBUG_FEEDBACK(("feedback_pjl_wait()"));

    if(pjl_in_job)
    	{
	sigset_t nset,oset;

	/* Turn on writemon alarms */
	set_writemon_description("PJL");
	alarm(writemon.interval);
    
	/* Block SIGPOLL. */
	sigemptyset(&nset);
	#ifdef NO_POLL
	sigaddset(&nset, SIGIO);
	#else
	sigaddset(&nset, SIGPOLL);
	#endif
	sigprocmask(SIG_BLOCK, &nset, &oset);

	/*
	** Temporarily unblock it and wait for it to be caught.
	** Notice that we do not continue to wait if a PostScript
	** error has been reported.
	**
	** Under normal circumstances this is not strictly necessary,
	** but under certain odd circumstances an HP4M+ with JetDirect
	** has been known to interpret the Universal Exit Language
	** sequence as a PostScript error.
	*/
	while(pjl_in_job && !posterror)
	    {
	    DODEBUG_FEEDBACK(("feedback_pjl_wait(): waiting"));
	    sigsuspend(&oset);
	    }
 
	/* Unblock SIGPOLL and leave it unblocked. */
	sigprocmask(SIG_SETMASK,&oset,(sigset_t*)NULL);

	/* Turn off writemon alarms */
	alarm(0);
    	}

    DODEBUG_FEEDBACK(("feedback_pjl_wait(): done"));
    } /* end of feedback_pjl_wait() */

/*
** Get the feedback signal handler set up
** and turn on signals.
*/
void feedback_start(void)
    {
    struct sigaction sig;

    DODEBUG_FEEDBACK(("feedback_start()"));

    /* We use the same signal handler for both BSD and SysV. */
    sig.sa_handler = sigpoll_handler;
    sigemptyset(&sig.sa_mask);   
    sigaddset(&sig.sa_mask, SIGALRM);	/* Block alarm signals.  (I don't remember why.) */
    #ifdef SA_RESTART
    sig.sa_flags = SA_RESTART;           
    #else
    sig.sa_flags = 0;
    #endif

    /* BSD way */
    #ifdef NO_POLL
    sigaction(SIGIO, &sig, NULL);

    if(fcntl(intstdout,F_SETOWN,getpid()) < 0)
    	fatal(EXIT_PRNERR, "pprdrv_feedback.c: feedback_start(): fcntl() failed on pipe");

    /* Turn on SIGIO for the pipe file descriptor. */
    if(fcntl(intstdout,F_SETFL,FASYNC | O_NONBLOCK) < 0)
    	fatal(EXIT_PRNERR, "pprdrv_feedback.c: feedback_start(): fcntl() failed on pipe");

    /* System V way */
    #else
    sigaction(SIGPOLL, &sig, NULL); 

    if( ioctl(intstdout,I_SETSIG, S_RDNORM ) )
        fatal(EXIT_PRNERR, "pprdrv_feedback.c: feedback_start(): ioctl() failed on pipe");

    if( fcntl(intstdout, F_SETFL, O_NONBLOCK) == -1 )
    	fatal(EXIT_PRNERR, "pprdrv_feedback.c: feedback_start(): fcntl() failed while setting O_NONBLOCK, errno=%d (%s)",errno,strerror(errno));
    #endif
    } /* end of feedback_start() */

/* end of file */
