/*
** ~ppr/src/pprdrv/pprdrv_commentary.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 17 January 1997.
*/

/*
** This is the function which invokes a commentator.  A commentator
** is a program or a file which receives messages about what is happening
** with a particuliar printer.
**
** This function should be safe to call from within a signal handler.
*/

#include "global_defines.h"
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <sys/wait.h>
#include "pprdrv.h"

/*
** This function is called whenever something happens which might
** be interesting to an operator.  The value of flags indicates
** the catagory of the event.  The string "message" indicates
** precisely what has occured.
**
** The behaviour of this function is defined by the "Commentator:"
** lines in the printer's configuration file.  A commentator line
** has the format:
**
** Commentator: flags name address "options"
**
** The flags field indicates what types of messages should be reported
** using this commenator.  The name indicates which commentator to use.
** If it is "file", the information is written to the file indicated by
** the "address" field.  If the commentator name is not "file", it is the
** name of a program in ~ppr/commentators which should be executed.
**
** We do not wait here for the commentator to exit.  Instead, pprdrv waits
** for all its children to exit before it exits itself.
**
** This function is called from within at least one signal handler,
** so it must not call non-reentrant functions such as malloc() or
** the stdio library.
*/
void commentary(int flags, const char *cooked, const char *raw1, const char *raw2)
    {
    struct COMMENTATOR *com;
    static int last_flags;
    static char last_cooked[15] = {(char)NULL};
    
    DODEBUG_COMMENTARY(("commentary(flags=%d, cooked=\"%s\", raw1=\"%s\", raw2=\"%s\")",
    	flags, cooked ? cooked : "", raw1 ? raw1 : "", raw2 ? raw2 : ""));

    /*
    ** If this is an announcement that the printer is
    ** "otherwise engaged or off line" and the previous message
    ** was that it is "off line" or "otherwise engaged", skip
    ** this message since it is redundant.
    */
    if(flags == COM_EXIT
    		&& last_flags == COM_PRINTER_STATUS
    		&& raw1 != (char*)NULL && strcmp(raw1, "EXIT_ENGAGED") == 0
    		&& (strcmp(last_cooked, "off line") == 0 || strcmp(last_cooked, "busy") == 0))
    	return;

    /*
    ** If this message is a printer status update and the last message
    ** was a printer status update and it said the same thing, skip
    ** this one.
    */
    #if 0
    if(flags == COM_STATUS_UPDATES
    		&& last_flags == COM_STATUS_UPDATES
    		&& cooked != (char*)NULL
    		&& strcmp(cooked, last_cooked) == 0)
    	return;
    #endif

    /*
    ** Preserve information about this message for
    ** future use by the clauses immediately above.
    */
    last_flags = flags;
    strncpy(last_cooked, cooked != (char*)NULL ? cooked : "", sizeof(last_cooked));

    /*
    ** If in test mode, we don't do real commentary since we are not really printing.
    */
    if(test_mode)
    	{
	fprintf(stderr, "commentary: flags=%d, cooked=\"%s\", raw1=\"%s\", raw2=\"%s\"\n",
		flags, cooked ? cooked : "", raw1 ? raw1 : "", raw2 ? raw2 : "");
	return;
    	}

    /*
    ** Send the message to every commentator that is interested.
    */
    for(com=printer.Commentators; com != (struct COMMENTATOR *)NULL; com=com->next)
    	{
	if( ! (flags & com->interests) )	/* If this commentator isn't interested in */
	    continue;				/* events of this type, skip it. */

	/*
	** The first possible type of commentator is not a program
	** at all, it is to append the information to a file.
	**
	** The "file" commentator ignores its options.
	*/
	if( strcmp(com->progname, "file") == 0 )
	    {
	    int fd;
	    time_t seconds_now; 		/* For time stamping the */
	    struct tm *time_detail;		/* print log file. */
	    char time_str[18];

	    seconds_now = time((time_t*)NULL);
	    time_detail = localtime(&seconds_now);
	    strftime(time_str, sizeof(time_str), "%y-%m-%d %H:%M:%S", time_detail);    
	    
	    if( (fd=open(com->address, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
		{	    	 
		char templine[256];
		sprintf(templine, "%s %s %d \"%s\" \"%s\" \"%s\"\n",
			time_str,
			printer.Name, flags,
			cooked != (char*)NULL ? cooked : "",
			raw1 != (char*)NULL ? raw1 : "",
			raw2 != (char*)NULL ? raw2 : "");
		write(fd, templine, strlen(templine) );
		close(fd);
	    	}
	    }

	/*
	** If we have not been instructed to put the commentary in
	** a file, execute the designated program.
	*/
	else
	    {
	    char temp[MAX_PATH];
	    char *progfile;
	    int fd;
	    
	    if(com->progname[0]=='/')
	    	{
	    	progfile = com->progname;
	    	}
	    else
	    	{
	    	sprintf(temp, COMDIR"/%s", com->progname);
	    	progfile = temp;
	    	}
	    
	    if( access(progfile, X_OK) == -1 )
	    	{
	    	error("Can't execute commentator \"%s\", errno=%d (%s)", progfile, errno, strerror(errno) );
	    	}
	    else
	    	{
		char flags_str[4];
		sprintf(flags_str, "%d", flags);

		switch( fork() )
		    {
		    case -1:		/* error */
			error("Fork() failed while executing commentator \"%s\", errno=%d (%s)", progfile, errno, strerror(errno) );
		    	break;
		    case 0:		/* child */		    
			if( (fd=open("/dev/null", O_RDWR)) != -1 )
			    {
			    if(fd != 0) dup2(fd, 0);
			    if(fd > 0) close(fd);
			    }
			if( (fd=open(PPRDRV_LOGFILE, O_WRONLY | O_CREAT | O_APPEND, UNIX_644)) != -1 )
			    {
			    if(fd != 1) dup2(fd, 1);
			    if(fd != 2) dup2(fd, 2);
			    if(fd > 2) close(fd);
			    }
		    	execl(progfile, com->progname,
		    		com->address,
		    		com->options != (char*)NULL ? com->options : "",
		    		printer.Name,
		    		flags_str,
		    		cooked != (char*)NULL ? cooked : "",
		    		raw1 != (char*)NULL ? raw1 : "",
		    		raw2 != (char*)NULL ? raw2 : "",
		    		(char*)NULL);
			_exit(242);
		    default:		/* parent */
		    	break;
		    }
	    	}	    
  	    }

    	} /* end of loop which goes thru all the commentators */
    
    } /* end of commentary() */

/*
** One of the last things main() does is install this SIGALRM
** handler in order to provide a timeout for waiting for
** commentators.
**
** Notice that we don't exit.  We hope that our exiting immediately
** after wait() returns -1 due to SIGALRM being caught will cause
** the hung commentators to receive SIGHUP and die.
*/
static void commentator_timeout_sigalrm(int sig)
    {
    alert(printer.Name, TRUE, "Timeout (60 seconds) waiting for commentator(s) to exit.");
    } /* end of commentator_timeout_sigalrm() */

/*
** Wait for any remaining commentators to exit.  I would expect
** that they would be killed if we exit before they do, but this
** doesn't seem to happen.  Feel free to disable this code
** if you want to.
**
** Notice that calling wait() here robs reapchild() of the
** privledge of receiving the exit codes, but this is ok
** because reapchild() is prepared for that and isn't really 
** interested anyway.
*/
void commentator_wait(void)
    {
    signal(SIGALRM, commentator_timeout_sigalrm);
    alarm(60);
    while(wait((int*)NULL) >= 0) ;
    alarm(0);
    }

/* end of file */
