/*
** ~ppr/src/pprdrv/pprdrv_capable.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This file last revised 16 December 1996.
*/

/*
** Determine if the printer is capable of printing the job.
** Resource substitution descisions are made in this module.
*/

#include "global_defines.h"
#include "global_structs.h"     
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include "interface.h"
#include "pprdrv.h"

/*
** Write a line to the job log file, prepending a "+" to it.
** Lines that begin with "+" concern circumstances which might
** prevent a certain printer from printing the job.
** If the log is not yet open, open it and write the 1st line.
*/
static FILE *log=(FILE*)NULL;

static void logopen(int group_pass)
    {
    char fname[MAX_PATH];

    sprintf(fname,"%s/%s-log",DATADIR,QueueFile);
    if( (log=fopen(fname,"a")) == (FILE*)NULL )
	return;

    if(group_pass)
	{
	if(group_pass==1)
	    {
	    fprintf(log,"+Pass 1\n");
	    }
	else
	    {
	    fprintf(log,"+Pass %d, ProofMode=%s\n",group_pass,
		job.attr.proofmode==PROOFMODE_TRUSTME?"TrustMe":job.attr.proofmode==PROOFMODE_SUBSTITUTE?"Substitute":"NotifyMe");

	    }
	}
    } /* end of logopen() */

/* This is the function we actually call to write the line. */
static void incapable_log(int group_pass, char *s, ...)
    {
    va_list va;
    static int said_incapable=FALSE;

    /* If log file not open, open it now. */
    if( log == (FILE*)NULL )
	logopen(group_pass);
	
    /*
    ** If we have not previously said that this printer 
    ** is incapable, say so now.
    */
    if(! said_incapable)
        {
        fprintf(log,"+Job turned away from printer \"%s\" for "
            "the following reason(s):\n",printer.Name); 

	said_incapable=TRUE;
        }

    fputc('+',log);         /* begin with a plus */        
    va_start(va,s);
    vfprintf(log,s,va);     /* print the error message */
    va_end(va);
    fputc('\n',log);        /* add a newline */
    } /* end of incapable_log() */

/*
** Say that we may be incapable but we will try.  This is a variation
** on incapable_log().
*/
static void incapable_log_willtry(int group_pass, char *s, ...)
    {
    va_list va;

    /*
    ** If the log file is not open (it would be opened by a previous
    ** call to this routine or incapable_log()) open it now and
    ** if this is a group job, say which pass this is and say
    ** what kind of notice or notices follow.
    */
    if( log == (FILE*)NULL )
        {
	logopen(group_pass);

        fprintf(log,"+Printer \"%s\" may not print this job correctly for "
            "the following reason(s):\n",printer.Name); 
        }

    fputc('+',log);         /* begin with a plus */        
    va_start(va,s);
    vfprintf(log,s,va);     /* print the error message */
    va_end(va);
    fputc('\n',log);        /* add a newline */
    } /* end of incapable_log_willtry() */

static void close_incapable_log(void)
    {
    if( log != (FILE*)NULL )
        {
        fclose(log);
        log=(FILE*)NULL;
        }
    } /* end of close_incapable_log() */

/*
** Read the required resources and requirements from
** the open queue file, and return -1 if they can not
** all be met satisfactorily.
**
** Also, check some of the things stored in the "Attr:" line.
*/
int check_if_capable(FILE *qfile, int group_pass)
    {
    int incapable=FALSE;        /* final result of this function */
    struct DRVRES *d;           /* pointer to next resource structure */
    int incapable_fonts=0;	/* number of missing fonts (not substituted) */
    int incapable_resources=0;	/* number of other missing resources */
    int incapable_requirements=0; /* missing requirements */
    int incapable_prohibitions=0; /* GrayOK = FALSE */
    int font;			/* TRUE if current resource is a font. */
    char *fnptr;		/* Temporary pointer to file name. */
    char *ttfnptr;		/* Temporary pointer to TrueType file name. */
    int color_document = FALSE;
    mode_t mode;

    /*-----------------------------------------------------------------
    ** Check for the resources that are required to print this job.
    ** Read the "Res: " lines from the queue file.
    -----------------------------------------------------------------*/
    while( fgets(line, MAX_LINE, qfile) != (char*)NULL )
        {					/* Work until end of file */
        if(strcmp(line, "EndRes\n") == 0)
            break;				/* or better yet, an "EndRes" line. */

        if( (line_len=strlen(line)) )		/* remove last character of line */
            line[--line_len]=(char)NULL;	/* which will always be a \n */

	tokenize();				/* break into ``words'' */

	if(tokens[6]==(char*)NULL)
	    fatal(EXIT_JOBERR, "Queue file has too few arguments on a \"Res:\" line");

	/* Make a new DRVRES record. */
	d = add_drvres(atoi(tokens[1]), atoi(tokens[2]), mystrdup(tokens[3]),
		mystrdup(tokens[4]), getdouble(tokens[5]), atoi(tokens[6]));

	if(! d->needed)				/* ignore those */
	    continue;				/* not still ``needed'' */

	/*
	** If it is a font, set a flag.  If it is a font,
	** try to find it in the printer's PPD file.
	*/
        if(strcmp(d->type, "font") == 0)	/* If it is a font, */
	    {					/* set a flag for */
	    font = TRUE;			/* future reference. */
	    if(find_font(d->name) == 0)		/* If it is in the printer, */
            	continue;			/* then be satisfied. */
            }
        else					/* If not a font, */
            {					/* clear the `it is a font' */
            font = FALSE;			/* flag. */
            }

	/*
	** Don't bother looking for encodings which are built
	** into the printer.  For level 1 printers these are
	** "Standard" and "Symbol", for level 2 printers, "ISOLatin1"
	** is also built in.
	*/
	if(strcmp(d->type, "encoding") == 0)
	    {
	    if( strcmp(d->name,"Standard")==0 || strcmp(d->name,"Symbol")==0
	    	    || ( Features.LanguageLevel > 1 && strcmp(d->name,"ISOLatin1")==0 ) )
	        {
	        continue;
	        }	    
	    }

	/*
	** See if the resource in question is in the cache.
	*/
        if( (fnptr=find_cached_resource(d->type, d->name, d->version, d->revision, (int*)NULL, &mode)) != (char*)NULL )
	    {					/* If found, */
	    if( strcmp(d->type,"font")==0 && mode & FONT_MACTRUETYPE )
		{				/* If it is a Mac truetype font, */
		d->mactt = TRUE;		/* mark it as such and try to get */

		if( mode & FONT_TYPE42 )	/* if it has a type 42 section, */
		    {
	    	    want_ttrasterizer();	/* a rasterizer, even if we have to load one. */

		    /* If rasterizer load failed and there is no Type1 section, */
		    if( ! printer.type42_ok && ! (mode & FONT_TYPE1) )
		    	{
			incapable_log(group_pass,"Font \"%s\" has no Type1 section and printer has no TT rasterizer",d->name);
			incapable = TRUE;
			continue;
		    	}
	    	    }
		}
		
	    d->filename = fnptr;		/* remember the name of the cache file */
	    d->needed = FALSE;			/* resource is now a supplied resource */

	    continue;				/* Go on to next resource. */
	    }

	/*
	** If the resource is a font which we did not find in any of
	** the above steps, and this printer can print TrueType fonts,
	** try to find a TrueType font which can be converted to PostScript. 
	*/
	if( font && (fnptr=find_ttfont(d->name))!=(char*)NULL )
	    {
	    d->filename = mystrdup(fnptr);
	    d->needed = FALSE;
	    d->truetype = TRUE;
	    want_ttrasterizer();	/* <-- See what can be done, but it is ok if this */
	    continue;			/*     fails because we can send a Type3 font. */
	    }

	/*
	** Here is where we do font substitution, but
	** we only do it if the ProofMode is "Substitute".
	*/
	if( (job.attr.proofmode==PROOFMODE_SUBSTITUTE) && font )
	    {
	    FILE *subfile;
	    char temp[80];		/* subfile lines won't be very long */
	    int in_entry=FALSE;		/* TRUE while we process the entry we are looking for. */
	    char *ptr;
	    char *codeptr;
	    int done=FALSE;
	
	    /* If we can find a sub file, try it. */
	    if( (subfile=fopen(FONTSUB,"r")) != (FILE*)NULL )
	        {
	        while( ! done && fgets(temp,sizeof(temp),subfile) != (char*)NULL )
	    	    {
		    if(temp[0]==';' || temp[0]=='#')		/* ignore comments */
		        continue;
		    if(temp[strspn(temp," \t\n")]==(char)NULL)	/* ignore blank lines */
		        continue;

		    if( ! in_entry )				/* if not yet in section for font */
		        {					/* for which we want a substitute, */
		        if(! isspace(temp[0]))			/* if this is a section heading (section headings do not begin with space) */
		    	    {
			    temp[strcspn(temp," \t\n")]=(char)NULL;	/* remove trailing whitespace */	    	
			    if(strcmp(d->name,temp)==0)			/* if match */
			        in_entry=TRUE;				/* then we found our font entry */
			    }    	
		        continue;
		        }	    	
	    	    else				/* in entry */
	    	        {
		        if(! isspace(temp[0]))		/* start of new entry means */
		    	    break;			/* we should give up */
		    
		        ptr=&temp[strspn(temp," \t")];		/* eat leading blank space */
			codeptr=&ptr[strcspn(ptr," \t\n")];	/* point to trailing space */
		        *codeptr=(char)NULL;			/* eat trailing space */
			codeptr++;				/* move past NULL */
			codeptr=&codeptr[strspn(codeptr," \t\n")];	/* eat next leading space */
			codeptr[strcspn(codeptr,"\n")]=(char)NULL;	/* delete trailing line feed */

		        /*
		        ** If the substitute font is found, in the printer
		        ** or in the font cache, set flags to use it instead.
		        */
			fnptr = (char*)NULL;		/* cached PostScript font name pointer */
			ttfnptr = (char*)NULL;		/* cached MS-Windows TT font name pointer */
		        if( (! find_font(ptr)) 
		        	|| (fnptr=find_cached_resource("font",ptr,0,0,(int*)NULL,&mode)) != (char*)NULL 
		        	|| (ttfnptr=find_ttfont(ptr)) != (char*)NULL )
		    	    {
			    if(fnptr != (char*)NULL)		/* If file name, remember it */
				{
				d->filename = fnptr;

				if( mode & FONT_MACTRUETYPE )	/* If Mac TT font, */
				    {
				    d->mactt = TRUE;		/* mark it as such */

				    if( mode & FONT_TYPE42 )	/* if it has a Type42 section, */
					{
					want_ttrasterizer();	/* try to get a rasterizer */
				    
					/* If rasterizer load failed and there is no Type1 section, */
					if( ! printer.type42_ok && ! (mode & FONT_TYPE1) )
					    {
					    incapable_log_willtry(group_pass,"Can't substitute \"%s\" for \"%s\" because it has no Type1 section",ptr,d->name);
					    continue;		/* try next substitute font */
					    }
					}
				    }
				}
			    else if(ttfnptr != (char*)NULL)	/* If MS-Win TrueType file name, */
			    	{				/* remember it. */
			    	d->filename = ttfnptr;
			    	d->truetype = TRUE;
			    	}
			    d->former_name = d->name;		/* save the old name */
			    d->name = mystrdup(ptr);		/* substitute the new */
			    d->needed = FALSE;
			    if(*codeptr != (char)NULL)		/* if substitution code, save it */
			    	d->subcode = mystrdup(codeptr);
			    incapable_log_willtry(group_pass,"Font \"%s\" substituted for \"%s\"",d->name,d->former_name);
			    done = TRUE;
		    	    }
	    	        }
	    	    }
	        fclose(subfile);

	        if(done)		/* If we had a match, don't let loop */
	            continue;		/* pass this point. */
	        }
	    } /* end of font substitution */

	/* Add to count of missing resources. */
	if(font)
	    incapable_fonts++;
	else
	    incapable_resources++;

	/*
	** Here is where we decide if we can go ahead without this
	** resource.  If the proofmode is "NotifyMe", we can't.
	**
	** If the ProofMode is "TrustMe" it ought to be ok to go ahead.
	**
	** If this is a font and the ProofMode is "Substitute" it ought to
	** be ok, it will be replaced with Courier.
	**
	** If this is a group job and we are still on pass 1, don't go ahead
	** without anything, some other printer might have it.
	*/
	if( job.attr.proofmode != PROOFMODE_NOTIFYME
		&& (job.attr.proofmode==PROOFMODE_TRUSTME || font)
		&& group_pass != 1 )
	    {
	    if(strcmp(d->type,"procset")==0)
		incapable_log_willtry(group_pass,"Resource \"%s %s %s %d\" not available.",d->type,d->name,dtostr(d->version),d->revision);
	    else
		incapable_log_willtry(group_pass,"Resource \"%s %s\" not available.",d->type,d->name);
	    continue;
	    }

	/*
	** Put the resource in the incapable log and mark the
	** printer as incapable.
	*/
        if(strcmp(d->type,"procset")==0)          
            incapable_log(group_pass,"Resource \"%s %s %s %d\" not available.",d->type,d->name,dtostr(d->version),d->revision);
        else
            incapable_log(group_pass,"Resource \"%s %s\" not available.",d->type,d->name);

	/* If we get this far, the printer is incapable. */
        incapable = TRUE;		/* printer is incapable */
        } /* end of while loop */

    /*------------------------------------------------------------
    ** Requirements as specified in a "%%Requirements:" comment.
    ------------------------------------------------------------*/
    while( fgets(line,MAX_LINE,qfile) != (char*)NULL )
        {          
        if( strcmp(line,"EndReq\n") == 0 )	/* read requirements */
            break;				/* until end flag */

	if(drvreq_count==MAX_DRVREQ)
	    fatal(EXIT_JOBERR,"internal pprdrv error: DRVREQ overflow");

        if( (line_len=strlen(line)) )    /* remove last character of line */
            line[--line_len]=(char)NULL; /* which will always be a \n */

        tokenize();			/* break line into ``words'' */
        
	if(tokens[1]==(char*)NULL)
	    fatal(EXIT_JOBERR,"queue file has a Req: line with too few arguments");

        if(drvreq_count<MAX_DRVREQ)	/* requirements which exceed the count */
            {				/* won't make it to the output file */
	    drvreq[drvreq_count++]=mystrdup(tokens[1]);
	    }
	else
	    {
	    error("MAX_DRVREQ exceeded");
	    }

	/*
	** In this switch, if the requirement is unrecognized or
	** is satisfied, execute continue.  If we allow execution
	** to drop out of the bottom of this loop, it is understood
	** that the requirement in question has not been met.
	*/
        switch(*(tokens[1]))
            {
	    case 'c':			/* requirements begining with 'c' */
		if(strcmp(tokens[1],"color")==0)
		    {
		    color_document = TRUE;

		    if(!Features.ColorDevice)
		    	break;
		    }
		continue;

            case 'd':			/* requirements begining with 'd' */
		/* This method of checking for duplex support is not    */
		/* the best one possible.  It is possible that duplex   */
		/* is an optional feature and consequently the PPD file */
		/* will contain duplex code with the stipulation that   */
		/* may only be used if the duplex option is installed.  */
                if(strcmp(tokens[1],"duplex")==0)
                    {
                    if(find_feature("*Duplex","None")==(char*)NULL)
			break;
                    }
                else if(strcmp(tokens[1],"duplex(tumble)")==0)
                    {
                    if(find_feature("*Duplex","DuplexTumble")==(char*)NULL)
			break;
                    }
                continue;

            case 'f':			/* requirements begining with 'f' */
                if(strcmp(tokens[1],"fax")==0)
                    {
                    if(Features.FaxSupport==FAXSUPPORT_NONE)
                        break;
                    }
		continue;

	    default:			/* ignore unrecognized */
	    	continue;		/* requirements */
            } /* end of switch */

	/*
	** If we get this far, the requirement can't be met.
	** Increment the count of requirments we can't meet.
	*/
	incapable_requirements++;

	/*
	** If the ProofMode is NotifyMe, then the unsatisified requirement
	** or requirements is a reason we can't print this
	** job.  If this is pass 1 on a group, that is a reason 
	** we can't print it yet.
	*/
	if( job.attr.proofmode==PROOFMODE_NOTIFYME || group_pass==1 )
	    {
	    incapable_log(group_pass,"Requirement \"%s\" can't be met.",tokens[1]);
            incapable = TRUE;
            }
        else
            {
            incapable_log_willtry(group_pass,"Requirement \"%s\" can't be met.",tokens[1]);
            }
        } /* end of while loop */

    /*--------------------------------------------------------
    ** Check to see if the printer language level 
    ** is high enough for the job.
    --------------------------------------------------------*/
    if(job.attr.langlevel > Features.LanguageLevel)
        {
        incapable_log(group_pass,"Required language level exceeds printer's language level.");
        incapable=TRUE;
        incapable_requirements++;
        }

    /*--------------------------------------------------------
    ** If the printer is not a level two printer, 
    ** make sure it has the extensions needed to
    ** print this job.
    --------------------------------------------------------*/
    if(Features.LanguageLevel < 2)
        {
        if( job.attr.extensions & (~Features.Extensions) )
            {
            incapable_log(group_pass,"Printer does not have all required extensions.");
            incapable = TRUE;
            incapable_requirements++; 
            }
        }

    close_incapable_log();

    /*---------------------------------------------------------------
    ** If this is not a colour document and the printer is not
    ** allowed to print grayscale documents, don't print this job.
    ---------------------------------------------------------------*/
    if( !color_document && !printer.GrayOK )
    	{
	incapable_log(group_pass,"Printer is not permitted to print grayscale documents.");
	incapable=TRUE;    	
    	incapable_prohibitions++;
    	}

    /*------------------------------------------------------------
    ** If the printer was judged to be incapable and this is not
    ** a group job on the second pass, add a "Reason: " line
    ** to the queue file so that the user will know why it was
    ** arrested.
    ------------------------------------------------------------*/
    if( incapable && group_pass<2 )
    	{
	char temp[256];		/* string to build reason in */
	int x=0;
	
	temp[0] = (char)NULL;	/* in case nothing matches */

	if(incapable_fonts)
	    {
	    if(incapable_fonts>1)
	    	sprintf(temp,"%d missing fonts",incapable_fonts);
	    else
	    	sprintf(temp,"1 missing font");
	    x+=strlen(temp);		/* more pointer for later append */
	    }
	    
	if(incapable_resources)
	    {
	    char *other;

	    if(x)			/* if text placed above, */
	    	temp[x++]=',';		/* add a comma but no space */
	    	
	    other=incapable_fonts?"other ":"";

	    if(incapable_resources>1)	/* be grammatical */
	    	sprintf(&temp[x],"%d %smissing rsrcs",incapable_resources,other);
	    else
	    	sprintf(&temp[x],"1 %smissing rsrc",other);
	    x+=strlen(&temp[x]);	/* move pointer for later append */
    	    }

    	if(incapable_requirements)
    	    {
	    if(x)
		temp[x++]=',';
		    
	    if( incapable_requirements > 1 )
	    	sprintf(&temp[x],"%d ptr incompatibilities",incapable_requirements);
	    else
	    	sprintf(&temp[x],"1 ptr incompatiblity");

	    x += strlen(&temp[x]);
    	    }
    	
    	if(incapable_prohibitions)
    	    {
	    if(x)
		temp[x++]=',';
		    
	    if( incapable_prohibitions > 1 )
	    	sprintf(&temp[x],"%d ptr prohibitions",incapable_requirements);
	    else
	    	sprintf(&temp[x],"1 ptr prohibition");

	    x += strlen(&temp[x]);
    	    }
    	
	if( temp[0] )
	    give_reason(temp);
    	} /* end of if(incapable) */

    return incapable;
    } /* end of check_if_capable */

/* end of file */
