/*
** ~ppr/src/pprdrv/pprdrv_buf.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** Last modified 5 March 1997.
*/

/*
** These routines buffer output going out over 
** the pipe to the interface.
*/

#include "global_defines.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include "pprdrv.h"
#include "interface.h"

/* We should be the only module outside of pprdrv.c that needs this: */
extern int intstdin;

#define BUFSIZE 4096
char wbuf[BUFSIZE];		/* write buffer */
char *bptr = wbuf;		/* buffer pointer */
int space = BUFSIZE;		/* space left */

/*
** This structure contains information we will use
** to monitor how well write()s are progressing.
**
** The string variable holds a string which describes
** what pprdrv is doing during the current stall.
*/
struct WRITEMON writemon;
static const char *writemon_description = "UNSET";      

/*
** This routine gets called every writmon.interval
** seconds while a write() is in progress.
**
** The purpose of this routine is to detect when the printer
** is stalled.  A printer may become stalled if it is out 
** of paper, if the PostScript code is taking a very long time
** to execute, or if the PostScript code is in an infinite loop.
**
** Interupted system calls will be restarted after this signal
** handler returns.
**
** I believe it is safe to call commentary() from within a
** signal handler.
**
** The first raw field is the number of minutes it has been stalled, the
** second is set by set_writemon_description() and indicates what
** operation is stalled.
*/
static void sigalrm_handler(int sig)
    {
    char num_str[10];
    char temp[40];

    writemon.impatience++;
    sprintf(num_str, "%d", writemon.impatience);
    
    switch(writemon.impatience)
    	{
    	case 1:
	case 2:
    	    commentary(COM_IMPATIENCE, "may be stalled", num_str, writemon_description);
    	    break;
    	case 3:
    	case 4:
    	    commentary(COM_IMPATIENCE, "probably stalled", num_str, writemon_description);
    	    break;
    	default:
	    sprintf(temp, "stalled for %d minutes", writemon.impatience);
    	    commentary(COM_IMPATIENCE, temp, num_str, writemon_description);
    	    break;
    	}
    
    alarm(writemon.interval);
    } /* end of sigalrm_handler() */

/*
** Issue a commentator message indicating that the stall condition
** has cleared.
**
** The first raw field is the number of minute it was stalled, the
** second is set by set_writemon_description() and indicates what
** we were doing when it stalled.
*/
static void cancel_impatience(void)
    {
    char num_str[10];

    sprintf(num_str, "-%d", writemon.impatience);

    /*
    ** Message differs according to how many minutes it
    ** was stalled.
    */
    switch(writemon.impatience)
    	{
	case 0:
	    fatal(EXIT_PRNERR_NORETRY, "pprdrv_buf.c: cancel_impatience(): writemon.impatience == 0");
    	case 1:
	case 2:
    	case 3:
    	case 4:
    	    commentary(COM_IMPATIENCE, "wasn't stalled", num_str, writemon_description);
    	    break;
    	default:
    	    commentary(COM_IMPATIENCE, "no longer stalled", num_str, writemon_description);
    	    break;
    	}
    	
    writemon.impatience = 0;
    } /* end of cancel_impatience() */

/*
** Set the description string which will describe any subsequent stall:
*/
void set_writemon_description(const char *description)
    { writemon_description = description; }

/*
** Initialize the buffer structures.
*/
void printer_bufinit(void)
    {
    struct sigaction sig;

    /*
    ** Install the SIGALRM handler in such a way that
    ** interupted system calls are re-started if at
    ** all possible.  We could just say
    **
    ** signal(SIGALRM,sigalrm_handler);
    **
    ** but this feels better since it causes write()
    ** to be re-started automatically.
    */
    sig.sa_handler = sigalrm_handler;
    sigemptyset(&sig.sa_mask);   
    #ifdef SA_RESTART
    sig.sa_flags = SA_RESTART;
    #else
    sig.sa_flags = 0;
    #endif
    sigaction(SIGALRM, &sig, (struct sigaction *)NULL);

    /*
    ** Initialize the structure which
    ** monitors writing speed.
    */
    writemon.interval = 60;
    writemon.impatience = 0;   
    } /* end of printer_bufinit() */

/* 
** Write all of the buffered data to the pipe now.
**
** We can of course call this in order to flush out query code,
** but it is more often called by the other routines when the
** buffer becomes full.
*/
void printer_flush(void)
    {
    int len;
    char *wptr;		/* pointer to current position in buffer */
    int rval;

    DODEBUG_INTERFACE_GRITTY(("printer_flush()"));

    set_writemon_description("WRITE");

    len = BUFSIZE - space;	/* compute bytes now in buffer */
    wptr = wbuf;		/* start of buffer */

    /* We will continue to call write() until our buffer is empty. */
    while(len > 0)
        {
	/* Set an alarm so that we will know when this is taking too long. */
	alarm(writemon.interval);

	/* Call write(), restarting it if it is interupted. */
        while( (rval = write(intstdin, wptr, len)) == -1 )
	    {
	    if(errno == EPIPE)		/* If pipe error, don't make any attempt */
	    	{			/* to continue, just keep catching signals */
		DODEBUG_INTERFACE(("Pipe write error, Waiting for SIGCHLD"));
		while(TRUE)		/* until the interface dies and reapchild() */
		    {			/* calls exit().  (Or sigpipe_handler() calls exit().) */
		    if(pause() == -1 && errno != EINTR)
		    	fatal(EXIT_PRNERR, "pprdrv_buf.c: printer_flush(): pause() failed, errno=%d (%s)", errno, strerror(errno));
		    }
	    	}

            if(errno != EINTR)
            	fatal(EXIT_PRNERR, "pprdrv_buf.c: printer_flush(): write error, errno=%d (%s)", errno, strerror(errno));
            }

	/* Cancel the alarm. */
	alarm(0);
	if(writemon.impatience) cancel_impatience();

	len -= rval;	/* reduce length left to write */
	wptr += rval;	/* move pointer forward */

	DODEBUG_INTERFACE_GRITTY(("printer_flush(): wrote %d bytes", rval));

	if(doing_primary_job)
	    progress__bytes_sent(rval);
        }

    bptr = wbuf;
    space = BUFSIZE;

    DODEBUG_INTERFACE_GRITTY(("printer_flush(): done"));
    } /* end of printer_flush() */

/*
** Add a single character to the output buffer.
*/
void printer_putc(int c)
    {
    if(space==0)		/* if the buffer is full, */
        printer_flush();	/* write it out */

    *(bptr++)=c;		/* otherwise, store it in the buffer */
    space--;			/* and reduce buffer empty space count */
    } /* end of printer_putc() */

/*
** Turn TBCP (Tagged Binary Communications Protocol) 
** on or off.
*/
void printer_TBCP_on_off(int state)
    {
    } /* end of printer_TBCP_on_off() */

/*
** Write a QuotedValue to the interface pipe.
** This is used to send strings from the PPD file.
**
** For a definition of a QuotedValue, see "PostScript Printer 
** Description File Format Specification" version 4.0, page 20.
*/
void printer_puts_QuotedValue(const char *s)
    {
    int x;                  /* partial value */
    #ifdef GNUC_HAPPY
    int val=0;              /* partial value (zero is for GNUC warning) */
    #else
    int val;
    #endif
    int place;              /* set to 1 after 1st character of hex byte */

    while(*s)               /* loop until end of string */
        {
        if( *s != '<' )     /* if doesn't introduce a hex substring, */
            {               /* just */
            printer_putc(*(s++));  /* send it */
            }
        else                /* hex substring: */
            {
            s++;            /* we don't need "<" anymore */
            place=0;
            while(*s && (*s != '>'))    /* loop until end of substring */
                {   
                if( *s >= '0' && *s <= '9' )    /* convert hex to int */
                    x = (*(s++) - '0');
                else if( *s >= 'a' && *s <= 'z' )
                    x = (*(s++) - 'a' + 10);
                else if( *s >= 'A' && *s <= 'Z' )
                    x = (*(s++) - 'A' + 10);
                else                            /* ignore other chars */
                    {                           /* (such as spaces, */
                    s++;                        /* tabs and newlines) */
                    continue;
                    }

                if(place)                       /* if 2nd character */
                    {
                    val+=x;                     /* add to last */
                    printer_putc(val);                 /* and send to output */
                    place=0;                    /* and set back to no chars */
                    }
                else                            /* if 1st character */
                    {
                    val=x<<4;                   /* store it */
                    place=1;                    /* and note fact */
                    }

                } /* end of inner while */
            if(*s)          /* skip closeing ">" */
                s++;
            } /* end of else */  
        } /* end of outer while */
    } /* end of write_quoted() */

/*
** Print a string with PostScript character
** quoting (\) before "(", ")", and "\".
*/
void printer_puts_escaped(const char *str)
    {
    int c;

    while( (c=*(str++)) )
        {
        switch(c)
            {
            case '(':
            case ')':
            case 0x5C:              /* backslash */
                printer_putc(0x5C); 
            default:
                if(isprint(c))
                    printer_putc(c);
                else
                    printer_putc('?');
            }
        }
    } /* end of printer_puts_escaped() */

/*
** Print a string to the interface.
*/
void printer_puts(const char *string)
    {
    while(*string)
        printer_putc(*(string++));
    } /* end of printer_puts() */

/*
** put a line to the interface
*/
void printer_putline(const char *string)
    {
    while(*string)
        printer_putc(*(string++));
    printer_putc('\n');
    } /* end of printer_putline() */

/*
** Print a formated string to the interface.
*/
void printer_printf(const char *string, ... )
    {
    va_list va;
    char *sptr;
    int n;
    double dn;
    char nstr[25];

    va_start(va,string);

    while(*string)
        {
        if(*string=='%')
            {
            string++;               /* discard the "%%" */
            switch(*(string++))
                {   
                case '%':			/* literal '%' */
                    printer_putc('%');
                    break;
                case 's':			/* a string */
                    sptr=va_arg(va,char *);
                    while(*sptr)
                        printer_putc(*(sptr++));
                    break;
                case 'd':			/* a decimal value */
                    n=va_arg(va,int);
                    sprintf(nstr,"%d",n);
                    sptr=nstr;
                    while(*sptr)
                        printer_putc(*(sptr++));
                    break;
                case 'f':			/* a double */
		case 'g':
                    dn=va_arg(va,double);
                    sptr=dtostr(dn);
                    while(*sptr)
                        printer_putc(*(sptr++));
                    break;      
                default:
                    fatal(EXIT_PRNERR_NORETRY,
                        "printer_printf(): illegal format spec: %s",string-2);
                    break;
                }
            }
        else
            {
            printer_putc(*(string++));
            }
        }

    va_end(va);
    } /* end of printer_printf() */

/*
** Write the indicated number of bytes to the interface.
** This is used to write lines when the lines may
** contain NULLs.
*/
void printer_write(const char *buf, int len)
    {
    while(len--)
    	printer_putc(*(buf++));
    } /* end of pwrite() */

/* end of file */
