/* 
** ~ppr/src/pprdrv/pprdrv.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 14 March 1997.
*/

/*
** This program is called by pprd to do the actual printing.  It launches the 
** interface program and resembles the job and sends it to the interface
** program.
*/

#include "global_defines.h"
#include "global_structs.h"
#ifdef USE_SOCKETPAIR_SIGIO
#include <sys/socket.h>
#endif
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>		/* for gettimeofday() */
#include "pprdrv.h"
#include "interface.h"
#include "interfaces.h"
#include "userdb.h"

/*====================================================
** Global variables.
====================================================*/

/* Are we running with the --test switch? */
int test_mode = FALSE;

/* Interface variables. */
int intstdin;			/* pipe to interface program */
int intstdout;			/* pipe from interface program */

/* Queue entry variables. */
char *QueueFile;				/* name of queue file of job to print */
FILE *qstream;					/* queue file "handle" */
struct QFileEntry job;				/* queue file entry */
struct DRVRES *drvres = (struct DRVRES *)NULL;	/* driver structures to hold resource names */
int drvres_count = 0;				/* count of structure slots used */
int drvres_space = 0;
char *drvreq[MAX_DRVREQ];			/* array of strings holding requirement names */
int drvreq_count = 0;				/* number of array slots used */

/* Flag page variables */
int doing_primary_job = FALSE;		/* main part of job, not flag pages */
static int new_doing_primary_job = FALSE;

/* Line buffers */
char line[MAX_LINE+1];		/* input buffer (MAX_LINE plus room for NULL) */
int line_len;			/* length of current line in bytes */
int line_overflow;		/* true or false */
static char pline[256];		/* buffer for "-pages" file lines */

int level = 0;			/* Document nesting level */

FILE *comments;			/* header comments file */
FILE *page_comments;		/* page level comments file */
FILE *text;			/* other parts of file */

struct PPRDRV printer;		/* defines things about the printer */

/* Things we want to strip out because we are doing it some other way: 
 * These are used to control begin_feature(). */
int strip_binselects = FALSE;	/* TRUE when begin_feature() should strip bin selection code */
int strip_signature = FALSE;	/* TRUE when begin_feature() should strip signature features */

/* 
** Variables needed to re-order pages.
**
** current_job_direction is the same as job.attr.pageorder except
** that when the latter has the value 0, the former has the value 1.
**
** print_direction is 1 if we will send the pages in ascending order,
** -1 if we will send them in descending order.  If the PageOrder
** is "Special", print_direction will be the same as current_job_direction.
*/
int current_job_direction;	/* 1 if input is forward, -1 if input is backwards */
int print_direction;		/* final page emmission order, 1 if ascend, -1 if descend */

/*
** The number of sheets in each copy of the job.
** This variable is set in page_computations(). 
** It is referred to when re-ordering pages and
** at other times.
*/
int sheetcount;

/*====================================================
** Some constants for library routines.
====================================================*/
const int lib_memory_fatal = EXIT_PRNERR;	/* value to return out out of memory */
const int lib_tokenize_fatal = EXIT_JOBERR;	/* another job might not have so long a line */
const int lib_misc_fatal = EXIT_PRNERR;		/* fatal exit value if ppr_sscanf() has fatal error */

/*===================================================
** Static variables (Those having a scope limited
** to this source module.
===================================================*/

/* The time at which we started running: */
static struct timeval start_time;

/*
** These three variables define how we will print copies.
** The first is actually global because pprdrv_req.c must
** read it.
*/
int copies_auto;			/* TRUE if should define #copies */
static int copies_doc_countdown;	/* number of times to send document */
static int copies_pages_countdown;	/* number of times to send pages */

/* The "%%PageMedia:" from the document defaults section. */
static char default_pagemedia[MAX_MEDIANAME+1]={(char)NULL};

/* Arrays used to change page order. */
static long int *pages_offsets;		/* ptr to array offsets into the "-pages" file */
static long int *text_offsets;		/* ptr to array offsets into the "-text" file */

/* Misc */
static int sigcaught;				/* set when SIGUSR1 is caught */
static int killing_interface = FALSE;		/* TRUE if we have deliberatly killed the interface */
static int interface_termination_expected;	/* tells reapchild() whether or not to consider it an error */
static int interface_exit_code;			/* for communication between reapchild() and close_interface() */
static pid_t intpid = 0;			/* interface process id */

/*========================================================
** Global variables in other modules:
========================================================*/
extern int posterror;			/* pprdrv_feedback.c */
extern int ghostscript;			/* pprdrv_feedback.c */
extern int status_updates_follow;	/* pprdrv_feedback.c */

/*=============================================================
** Exiting, fatal error, and debug message code.
=============================================================*/

/*
** Get the current time with microsecond resolution
** and compute the difference between the current time
** and the time that was recorded when we started.
*/
static void get_elapsed_time(struct timeval *time_elapsed)
    {
    struct timeval time_now;

    gettimeofday(&time_now, (struct timezone *)NULL);
    time_elapsed->tv_sec = time_now.tv_sec - start_time.tv_sec;
    time_elapsed->tv_usec = time_now.tv_usec - start_time.tv_usec;
    if(time_elapsed->tv_usec < 0) {time_elapsed->tv_usec += 1000000; time_elapsed->tv_sec--;}
    }

/*
** We must call this function whenever we are going to exit
** for any reason.  If the caller supplies a non-NULL string it
** will be used as the `cooked' argument to commentary().  Feel
** free to call it with a NULL string argument, a default 
** cooked message will be provided.
**
** The purpose of this function is to call the commentary()
** function so as to generate a commentary message which will
** indicate why we are exiting.
**
** As its last act, fatal() calls exit_with_commentary() which
** calls this function.
*/
static void exit_commentary(int rval, const char *explain)
    {
    int com_code = COM_EXIT;
    const char *cooked;
    const char *raw;
    
    /*
    ** Allocate space for the composite string,
    ** adding space for string if it turned out to
    ** be non-NULL.
    */
    switch(rval) 
    	{
	case EXIT_PRINTED:
	    raw = "EXIT_PRINTED";
	    cooked = "has printed a job";
	    com_code = COM_MUNDANE_EXIT;
	    break;
    	case EXIT_PRNERR:
    	    raw = "EXIT_PRNERR";
    	    cooked = "printer error";
    	    break;
    	case EXIT_PRNERR_NORETRY:
    	    raw = "EXIT_PRNERR_NORETRY";
	    cooked = "printer error, no retry";
    	    break;
    	case EXIT_JOBERR:
    	    raw = "EXIT_JOBERR";
    	    cooked = "job error";
    	    break;
	case EXIT_SIGNAL:
	    raw = "EXIT_SIGNAL";
	    cooked = "?";
	    break;
	case EXIT_ENGAGED:
	    raw = "EXIT_ENGAGED";
	    cooked = "otherwise engaged or off-line";
	    break;
	case EXIT_STARVED:
	    raw = "EXIT_STARVED";
	    cooked = "starved for system resources";
	    break;
	case EXIT_INCAPABLE:
	    raw = "EXIT_INCAPABLE";
	    cooked = "incapable of printing this job";
	    com_code = COM_MUNDANE_EXIT;
	    break;
	default:
	    raw = "EXIT_?";
	    cooked = "invalid exit code";
	    break;
    	}

    /*
    ** If the user supplied an explaination, use it as the cooked
    ** argument for commentary() in stead of the default one.
    */
    if(explain != (char*)NULL)
	cooked = explain;

    /*
    ** Feed the whole thing to the commentary function.
    ** Then, wait for the commentators to exit.
    */
    commentary(com_code, cooked, raw, "");
    } /* end of exit_commentary() */
    
/*
** Call exit_commentary, wait for any commentators launched
** to exit, then exit.
*/
#ifdef __GNUC__
static void exit_with_commentary(int rval, const char *explain) __attribute__ ((noreturn));
#endif
static void exit_with_commentary(int rval, const char *explain)
    {
    exit_commentary(rval, explain);
    commentator_wait();
    exit(rval);
    } /* end of exit_with_commentary() */

/*
** This function adds an error message to pprdrv's log file, and if not in 
** test mode, to either the printer's or the job's log file.  It then calls 
** exit_with_commentary().
**
** Not all fatal errors pass through this function.  In some cases, alert() 
** is called and then exit_with_commentary().  Doing it that way can produce
** prettier messages.  Theirfor this function is used for reporting the most
** unlikely faults.
**
** This function ought to be ok to call from within
** signal handlers.
*/
void fatal(int rval, const char *string, ...)
    {
    va_list va;
    int file;			/* the pprdrv log file */
    char templine[256];
    struct timeval time_elapsed;

    get_elapsed_time(&time_elapsed);
    sprintf(templine, "FATAL: (%ld %s %ld.%02d) ", (long)getpid(), datestamp(), (long)time_elapsed.tv_sec, (int)(time_elapsed.tv_usec / 10000) );
    va_start(va, string);
    vsprintf(&templine[strlen(templine)], string, va);
    va_end(va);
    strcat(templine, "\n");

    if(test_mode)
    	{
    	write(2, templine, strlen(templine));
    	}
    else
	{
	/* Append the message to the pprdrv log file. */
	if( (file=open(PPRDRV_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
	    {                                   
	    write(file, templine, strlen(templine));
	    close(file);
	    }

	/*
	** If it is a job error and this isn't just test mode,
	** log the error in the job's log file.
	*/
	if(rval == EXIT_JOBERR)
	    {
	    char fname[MAX_PATH];
	    sprintf(fname, DATADIR"/%s-log", QueueFile);
	    if( (file=open(fname, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
		{
		write(file, templine, strlen(templine));
		close(file);
		}
	    }

	/*
	** If it is a printer or spooler error or this is test mode,
	** is the alert() function to append a message to the printer's
	** alert log, or if test mode, to stderr.
	*/
	else
	    {
	    valert(printer.Name, TRUE, string, va);
	    }
	}

    exit_with_commentary(rval, (char*)NULL);
    } /* end of fatal() */

/*
** Print an error line in the pprdrv log file
** and append it the printer's alerts file.
**
** This routine is generally used only for unusual error
** conditions, internal errors.  It is not used to report
** routine conditions such as printer faults.  It is used
** for debug messages which we still want to see when 
** debugging is not compiled in.
**
** This function ought to be ok to call from within signal
** handlers.
*/
void error(const char *string, ... )
    {
    va_list va;
    int file;

    va_start(va,string);

    /*
    ** This is the part which sends it to the pprdrv log file.
    */
    if( (file=open(PPRDRV_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
        {
	char templine[256];
	struct timeval time_elapsed;

	get_elapsed_time(&time_elapsed);
	sprintf(templine, "ERROR: (%ld %s %ld.%02d) ", (long)getpid(), datestamp(), (long)time_elapsed.tv_sec, (int)(time_elapsed.tv_usec / 10000) );
        vsprintf(&templine[strlen(templine)], string, va);
        strcat(templine, "\n");
	write(file, templine, strlen(templine));
        close(file);
        }

    /*
    ** This is the command which puts the message in the
    ** printer's alerts file.
    */
    valert(printer.Name, TRUE, string,va);

    va_end(va);
    } /* end of error() */

/*
** Print a debug line in the pprdrv log file.
**
** This code is not included in production versions.
**
** This function ought to be ok to call from within signal handlers.
*/
#ifdef DEBUG
void debug(const char *string, ... )
    {
    int file;

    if( (file=open(PPRDRV_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
        {
	va_list va;
	char templine[256];
	struct timeval time_elapsed;
	
	get_elapsed_time(&time_elapsed);
	sprintf(templine, "DEBUG: (%ld %s %ld.%02d) ", (long)getpid(), datestamp(), (long)time_elapsed.tv_sec, (int)(time_elapsed.tv_usec / 10000) );
	va_start(va, string);
        vsprintf(&templine[strlen(templine)], string, va);
	va_end(va);
	strcat(templine, "\n");
	write(file, templine, strlen(templine));
        close(file);
        }
    } /* end of debug() */
#endif

/*=============================================================
** Routines for running the interface.
=============================================================*/

/*
** Catch reapchild signals.
**
** We ignore the termination of things other than the interface,
** such as commentators.
**
** Just before we expect the interface to terminate we set the 
** variable interface_termination_expected to TRUE.  When it does
** exit, this routine will set the variable interface_exit_code.
**
** If the interface dies on a signal when killing_interface 
** is not TRUE then we will post an alert and exit.
*/
static void reapchild(int sig)
    {
    pid_t waitid;
    int wstat;

    /*
    ** In this loop we retrieve process exit status until there
    ** are no more children whose exit status is available.  When
    ** there are no more, we return.  If the child is the interface,
    ** we drop out of the loop and let the code below handle it.
    **
    ** Notice that it is important to test for a return value of
    ** 0 before we test for a return value that is equal to intpid
    ** since if there is no interface intpid will be 0.
    */
    while(TRUE)
	{
	waitid = waitpid((pid_t)-1, &wstat, WNOHANG);

	/* If no more children are dead and unreaped, */
	if(waitid == 0)
	    return;

	/* If there are no more children at all. */
	if(waitid == -1 && errno == ECHILD)
	    return;

	/* If the child that died was the interface, handle it below. */
	if(waitid == intpid)
	    break;

	/* If a genuin error, */
	if(waitid == -1 && errno != EINTR)
	    fatal(EXIT_PRNERR, "pprdrv: reapchild(): waitpid() failed, errno=%d (%s)", errno, strerror(errno) );

	/* If we get this far, a commentator terminated. */
	#ifdef DEBUG
	debug("process %ld terminated", (long int)waitid);
	#endif
    	}

    /*
    ** Make a note of the fact that the interface is dead.
    */
    DODEBUG_INTERFACE(("reapchild(): interface terminated"));
    intpid = 0;

    /*
    ** See if interface just died when it got a signal.
    */
    if(WIFSIGNALED(wstat))
	{
	if(killing_interface)	/* If printing is being canceled, */
	    return;		/* we need do nothing here. */

	/*
	** Since we did not send the signal, it is a fatal error,
	** however we will not call fatal() because we must
	** issue a rather complicated message.  We will make calls
	** to alert(), and exit_with_commentary().
	*/
	alert(printer.Name, TRUE, "Interface terminated abruptly after receiving signal %d (%s)", WTERMSIG(wstat), strsignal(WTERMSIG(wstat)));
	alert(printer.Name, FALSE, "from a non-spooler process (possibly the OS kernel).");

	/* If the signal caused the interface to dump core, say so. */
	if( WCOREDUMP(wstat) )
            {
            alert(printer.Name, FALSE, "Interface dumped core.");
	    exit_with_commentary(EXIT_PRNERR_NORETRY, "interface core dump");
	    }

	exit_with_commentary(EXIT_SIGNAL, "interface program killed");
        } /* end of if interface died due to the receipt of a signal */

    /*
    ** This is similiar to the section just above.
    ** 
    ** See if the interface exited with a status code which indicates that
    ** it caught a signal and then exited volunarily.
    **
    ** If we set killing_interface in sigterm_handler() then any comment
    ** upon the fact of the interface's untimely death will be suppressed.
    */
    if( WIFEXITED(wstat) && (WEXITSTATUS(wstat) == EXIT_SIGNAL) )
    	{
    	if(killing_interface)		/* if we did it deliberatly, */
    	    {				/* then we need to nothing further */
    	    return;
    	    }
    	else
    	    {
    	    alert(printer.Name, TRUE, "The interface performed a controlled shutdown because a non-spooler");
    	    alert(printer.Name, FALSE, "process (possibly the OS kernel) sent it a signal.");
	    exit_with_commentary(EXIT_SIGNAL, "interface program killed");
    	    }
    	} /* end of if interface caught a signal */
    
    /*
    ** It is possible that the interface did not exit.  I don't know 
    ** why that might be, but it is possible.
    */
    if(WIFSTOPPED(wstat))
	fatal(EXIT_PRNERR_NORETRY, "Interface stopped by signal \"%s\".", strsignal(WSTOPSIG(wstat)));
    if(! WIFEXITED(wstat))
	fatal(EXIT_PRNERR_NORETRY, "Bizaar interface malfunction: SIGCHLD w/out signal or exit.");

    /*
    ** If the exit code is a legal one for an interface, use it;
    ** if not, assume something really bad has happened and
    ** ask to have the printer put in no-retry-fault-mode.
    **
    ** Oops!  There is one more exception!  Ghostscript will exit
    ** immediatly if a Postscript error occurs.  We must be ready
    ** for that.
    */
    if(WEXITSTATUS(wstat) > EXIT_INTMAX)
        fatal(EXIT_PRNERR_NORETRY, "Interface malfunction: returned invalid exit code %d", WEXITSTATUS(wstat));

    /*
    ** If close_interface() is waiting for termination,
    ** it will know that the interface has terminated when
    ** interface_exit_code is set to something other than -1.
    */
    if(interface_termination_expected)
	{
	interface_exit_code = WEXITSTATUS(wstat);

	/* Paranoid check: */
	if(interface_exit_code == -1)
	    fatal(EXIT_PRNERR_NORETRY, "%s: reapchild(): interface_exit_code = -1", __FILE__);

	return;
	}

    /*
    ** Very strange termination.  Notice that normal termination
    ** would have been caught by the clause above.
    */
    if(WEXITSTATUS(wstat) == EXIT_PRINTED)
	{
	fatal(EXIT_PRNERR_NORETRY, "Interface malfunction: it claims to have printed the\n"
		"\tjob but it exited before accepting all of the data.");
	}

    /*
    ** Interface terminated with error code before it 
    ** had accepted all of the data, pass the code back to pprd.
    */
    exit_with_commentary(WEXITSTATUS(wstat), (char*)NULL);
    } /* end of reapchild() */

/*
** This routine launches the interface program.
**
** This routine sets the global variables intstdout and intstdin.
*/
static void start_interface(void)
    {
    int _stdin[2];           		/* for opening pipe to interface */
    int _stdout[2];			/* for pipe from interface */
    int new_stdin, new_stdout;
    char fname[MAX_PATH];		/* possibly holds path of interface program */
    char *fname_ptr;
    char jobbreak_str[2];		/* jobbreak setting converted to ASCII */
    char feedback_str[2];		/* feedback setting converted to ASCII */

    DODEBUG_INTERFACE(("start_interface()"));

    /* If running in test mode, don't really start an interface. */
    if(test_mode)
    	{
	intstdin = 1;
	intstdout = 0;    	
    	return;
    	}

    /* get ready for a SIGUSR1 */
    sigcaught = FALSE;

    if(*printer.Interface == '/')	/* if absolute path, */
	{				/* use it */
	fname_ptr = printer.Interface;
	}
    else				/* Otherwise, prepend the interfaces */
	{				/* directory name */
	sprintf(fname,INTDIR"/%s", printer.Interface);
	fname_ptr = fname;
	}

    if( access(fname_ptr, X_OK) )	/* make sure we can execute */
	fatal(EXIT_PRNERR_NORETRY,	/* the interface program */
            "Interface \"%s\" does not exist or is not executable.", printer.Interface);

    /* Catch premature interface termination. */
    interface_termination_expected = FALSE;

    if(pipe(_stdin))         		/* open a pipe to interface */
        fatal(EXIT_PRNERR,"start_inteface(): pipe() failed, errno=%d",errno);

    /*
    ** Open a pipe from the interface.  We may use a socketpair instead
    ** of a pipe if SIGIO doesn't work on pipes in our operating system.
    ** (This is true of Linux at least thru version 1.1.85.)
    */
    #ifdef USE_SOCKETPAIR_SIGIO
    if( socketpair(AF_UNIX,SOCK_STREAM,0,_stdout) )
	fatal(EXIT_PRNERR,"start_interface(): socketpair() failed, errno=%d",errno);
    #else
    if(pipe(_stdout))        		/* open a pipe from interface */
        fatal(EXIT_PRNERR,"start_interface(): pipe() failed, errno=%d",errno);
    #endif

    switch(intpid=fork())       	/* duplicate this process */
        {            
        case -1:                        /* case: fork failed */
	    /* We don't want this information building up
	       in the pprdrv log file, so we will not
	       use fatal(). */
	    DODEBUG_INTERFACE(("start_interface(): fork() failed, errno=%d (%s)", errno, strerror(errno) ));
	    exit_with_commentary(EXIT_STARVED, (char*)NULL);

        case 0:                         /* case: we are child */
            setsid();                   /* make interface proc group leader */
            new_stdin=dup(_stdin[0]);   /* read end of stdin is for us */
            new_stdout=dup(_stdout[1]); /* write end of stdout is for us */
            close(_stdin[0]);           /* we don't need these any more */
            close(_stdin[1]);           /* we must clear them away now */
            close(_stdout[0]);          /* as some of them are shurly */
            close(_stdout[1]);          /* file descritors 0, 1, and 2 */
            dup2(new_stdin,0);          /* copy the temporaries to the real */
            dup2(new_stdout,1);         /* things */
            dup2(new_stdout,2);
            close(new_stdin);           /* close the temporaries */
            close(new_stdout);
           
	    /* Convert two of the parameters from integers to strings. */
	    sprintf(jobbreak_str,"%1.1d", printer.jobbreak);
	    sprintf(feedback_str,"%1.1d", printer.feedback);

	    /*
	    ** Our child should not start out with
	    ** signals blocked, even if we have some blocked.
	    */
            {
            sigset_t sigset;        
            sigemptyset(&sigset);   
            sigprocmask(SIG_SETMASK, &sigset, (sigset_t*)NULL);
            }

	    /* We will now overlay this child process with the interface. */
            execl(fname_ptr, printer.Interface,	/* replace child pprdrv with */
                printer.Name, printer.Address,	/* the interface */
                printer.Options, QueueFile,
                job.Routing != (char*)NULL ? job.Routing : "",
		jobbreak_str, feedback_str,
		job.ForLine != (char*)NULL ? job.ForLine : "",
                (char*)NULL);

	    sleep(1);			/* Give parent time to record PID in intpid. */

            _exit(242);			/* in case of execl() failure */

        default:                        /* case: we are parent */
            close(_stdin[0]);           /* close our copy of read end */   
            close(_stdout[1]);          /* close unneeded write end */         
            intstdin = _stdin[1];
            intstdout = _stdout[0];

            if( fcntl(intstdout,F_SETFL,O_NDELAY) )
                fatal(EXIT_PRNERR,"fcntl() failed on pipe");

            /*
            ** If SIGUSR1 protocol is being used, wait for the
	    ** child to inform us that it has installed _its_
	    ** SIGUSR1 handler by sending _us_ SIGUSR1.
	    **
	    ** We use the stall warning mechanism in this code
	    ** even though it is not quite right for the job.
	    ** Our intent is to detect an interface which does
	    ** not start up properly or does not support 
	    ** JOBBREAK_SIGNAL correctly.
	    */
            if(printer.jobbreak==JOBBREAK_SIGNAL || printer.jobbreak==JOBBREAK_SIGNAL_PJL)
                {
		sigset_t nset, oset;		/* newset, oldset */

		DODEBUG_INTERFACE(("start_interface(): waiting for SIGUSR1"));

		sigemptyset(&nset);		/* Block SIGUSR1. */
		sigaddset(&nset, SIGUSR1);		
		sigprocmask(SIG_BLOCK, &nset, &oset);

		set_writemon_description("SIGUSR1");
		alarm(writemon.interval);

		while(! sigcaught)		/* If it has not already occured, */
		    sigsuspend(&oset);		/* temporarily unblock it and wait for it. */

		alarm(0);

		sigprocmask(SIG_SETMASK, &oset, (sigset_t*)NULL);
		}				/* Unblock SIGUSR1. */

        } /* end of switch */

    DODEBUG_INTERFACE(("start_interface(): done"));
    } /* end of start_interface() */

/*
** This routine sends an end of file to the interface process.
**
** If the interface called exit(), we will return the code, 
** otherwise, we return the code it should have returned.
*/
static int close_interface(void)
    {
    sigset_t nset, oset;

    DODEBUG_INTERFACE(("close_interface()"));

    /*
    ** Flush out the last of the printer data.
    */
    printer_flush();

    /*
    ** If we are running in test mode there is nothing more to do.
    */
    if(test_mode)
    	return EXIT_PRINTED;

    /*
    ** Close the pipe to the interface.
    */
    interface_exit_code = -1;
    interface_termination_expected = TRUE;
    close(intstdin);

    /*
    ** Wait for the interface to terminate.
    ** (We must keep calling sigsuspend() because
    ** commentators may be exiting.)
    */
    sigemptyset(&nset);					/* Create signal set */
    sigaddset(&nset, SIGCHLD);				/* to block SIGCHLD. */
    sigprocmask(SIG_BLOCK, &nset, &oset);

    set_writemon_description("CLOSE");
    alarm(writemon.interval);				/* Turn on writemon alarms */
    
    while(interface_exit_code == -1)			/* Wait for exit code to */
	sigsuspend(&oset);				/* be reported. */

    alarm(0);						/* Turn off writemon alarms */

    sigprocmask(SIG_SETMASK, &oset, (sigset_t*)NULL);	/* Unblock SIGCHLD */

    /*
    ** Close the pipe from the interface.
    */
    close(intstdout);

    DODEBUG_INTERFACE(("close_interface(): returning %d", interface_exit_code));
    return interface_exit_code;
    } /* end of close_interface() */

/*
** Handler for signal from interface (user signal 1).
** If we are using the JobBreak method "signal" or "signal/pjl", 
** the interface will send us this signal as soon as it has 
** set-up its SIGUSR1 handler.  It will also send us that signal
** to acknowledge any USR1 signals we send it.
*/
static void user_sighandler(int sig)
    {
    DODEBUG_INTERFACE(("signal \"%s\" caught", strsignal(sig)));
    sigcaught = TRUE;
    } /* end of user_sighandler() */

/*
** Routine to send SIGUSR1 job breaks.  This is only used with 
** interfaces which use the "signal" or "signal/pjl" jobbreak
** methods.
*/
static void signal_jobbreak(void)
    {
    sigset_t nset, oset;

    DODEBUG_INTERFACE(("signal_jobbreak()"));

    /* Force buffered data into pipe. */
    printer_flush();		
     
    /* Block SIGUSR1. */
    sigemptyset(&nset);
    sigaddset(&nset, SIGUSR1);
    sigprocmask(SIG_BLOCK,&nset, &oset);

    sigcaught = FALSE;		/* get ready to catch signal */

    kill(intpid, SIGUSR1);	/* Send the signal to the interface */

    set_writemon_description("JOBBREAK");
    alarm(writemon.interval);	/* Turn on writemon alarms */
    
    while( ! sigcaught )	/* and wait for the response */
	sigsuspend(&oset);

    alarm(0);

    /* Unblock SIGUSR1 and leave it unblocked. */
    sigprocmask(SIG_SETMASK,&oset,(sigset_t*)NULL);

    DODEBUG_INTERFACE(("signal_jobbreak(): child has responded"));
    } /* end of signal_jobbreak() */

/*
** Send setup strings to the printer.  We use the information 
** from the "JobBreak: " line to decide what to send.
*/
static void printer_setup(void)
    {
    #define MAX_MLEN 16
    char message[MAX_MLEN+1];

    /*
    ** Construct a message for the little readout on HP printers.
    */
    if(job.ForLine == (char*)NULL)
	strncpy(message, "<unknown>", MAX_MLEN);
    else
	strncpy(message, job.ForLine, MAX_MLEN);

    message[MAX_MLEN] = (char)NULL;

    /*
    ** Now, act according to jobbreak method.
    */
    switch(printer.jobbreak)
    	{
    	case JOBBREAK_CONTROL_D:		/* <-- `Serial' control-d protocol */
    	    printer_putc(4);
    	    break;

    	case JOBBREAK_PJL:			/* <-- PJL without signals */
	case JOBBREAK_SIGNAL_PJL:		/* <-- PJL with signals */
	case JOBBREAK_TBCP:			/* <-- TBCP and PJL */
	    printer_puts("\33%-12345X");	/* universal exit language */
	    printer_printf("@PJL RDYMSG DISPLAY = \"%s\"\n",message);
						/* message for display */
	    printer_puts("@PJL USTATUS PAGE = ON\n");
						/* send a message at the end of each page */
	    printer_puts("@PJL USTATUS JOB = ON\n");
						/* send a message at end of each job */
	    printer_puts("@PJL JOB\n");		/* start of job (banners and job together) */
	    printer_puts("@PJL ENTER LANGUAGE = POSTSCRIPT\n");
	    
	    if(printer.jobbreak == JOBBREAK_TBCP)
	    	{
	    	printer_putc(1);
	    	printer_putc('M');
	    	printer_TBCP_on_off(TRUE);
	    	}
	    break;

	case JOBBREAK_SAVE_RESTORE:		/* <-- silly, optimistic method */
	    printer_puts("save\n");
	    break;

	default:				/* <-- all others, do nothing */
	    break;
	}
    } /* end of printer_setup() */
    
/*
** Printer cleanup.  Undo what we did or stuff like that.
** We use the information from the "JobBreak: " line.
*/
static void printer_cleanup(void)
    {
    switch(printer.jobbreak)
    	{
    	case JOBBREAK_CONTROL_D:		/* <-- simple ``serial'' control-D protocol */
    	    printer_putc(4);
    	    break;

	case JOBBREAK_TBCP:			/* <-- TBCP, control-D and PJL */
	    printer_TBCP_on_off(FALSE);
    	case JOBBREAK_PJL:			/* <-- PJL with control-D */
    	case JOBBREAK_SIGNAL_PJL:		/* <-- PJL with signals */
	    if(printer.jobbreak == JOBBREAK_SIGNAL_PJL)
	    	signal_jobbreak();
	    else
		printer_putc(4);

	    printer_puts("\33%-12345X");	/* "Universal Exit Language" command */
	    printer_puts("@PJL EOJ\n");
	    printer_flush();
	    /*
	    ** Would you believe, we mustn't transmit
	    ** the command to turn this stuff off until
	    ** we have received all the notifications? 
	    */
	    feedback_pjl_wait();
	    printer_puts("@PJL USTATUS PAGE = OFF\n");
	    printer_puts("@PJL USTATUS JOB = OFF\n");
	    printer_puts("@PJL RDYMSG DISPLAY = \"00 READY\"\n");
	    printer_puts("\33%-12345X");
	    break;

	case JOBBREAK_SAVE_RESTORE:		/* <-- that silly optimistic method */
	    printer_puts("\nrestore\n");
	    break;

	default:				/* <-- all other methods */
	    break;
	}
    } /* end of printer_cleanup() */

/*
** A PostScript job ends here and another begins.  Make the interface tell
** the printer so.  We use the code number from the "JobBreak: " line in
** the printer's configuration file to decide how to accomplish that.
**
** We use calls to this function to separate the print job
** from its banner and trailer pages and to separate
** multiple copies of the jobs when we find it necessary
** to print multiple copies the hard way (by sending the
** whole job multiple times).
*/
void jobbreak(void)
    {
    DODEBUG_INTERFACE(("sending end of job"));

    switch(printer.jobbreak)
	{
    	case JOBBREAK_CONTROL_D: 	/* <-- Simple control-D protocol */
	    printer_putc(4);
	    break;

	case JOBBREAK_SIGNAL:		/* <-- signals only */
	    signal_jobbreak();
	    break;

	case JOBBREAK_SIGNAL_PJL:	/* <-- signals with PJL */
	case JOBBREAK_TBCP:		/* <-- TBCP and PJL */
	case JOBBREAK_PJL:		/* <-- PJL alone */
	    if(printer.jobbreak == JOBBREAK_TBCP)
	    	printer_TBCP_on_off(FALSE);

	    if(printer.jobbreak == JOBBREAK_SIGNAL_PJL)
		signal_jobbreak();
	    else
		printer_putc(4);
            
	    printer_puts("\33%-12345X@PJL ENTER LANGUAGE = POSTSCRIPT\n");

	    if(printer.jobbreak == JOBBREAK_TBCP)
	    	{
		printer_putc(1);
		printer_putc('M');
		printer_TBCP_on_off(TRUE);	    	
	    	}
	    break;			/* UEL, select PostScript again */

	case JOBBREAK_SAVE_RESTORE:	/* <-- silly optimistic method again */
	    printer_puts("\nrestore\nsave\n");
	    break;

	case JOBBREAK_NEWINTERFACE:	/* <-- Stop the interface and start it again */
	    close_interface();		/* Error code ignored. */
	    start_interface();
	    status_updates_follow = FALSE;
	    break;

	default:			/* <-- all other methods */
	    break;
        }

    /*
    ** Now that we have sent the job break, 
    ** the change in state of doing_primary_job is 
    ** truly complete.
    */
    doing_primary_job = new_doing_primary_job;
    } /* end of jobbreak() */

/*=============================================================
** Routines for re-assembling the job.
=============================================================*/

/*
** Copy data between "%%BeginData:" and "%%EndData" comments.
*/
static void copy_data(FILE *infile)
    {
    long int len;

    printer_putline(line);                         /* copy thru to output now */

    tokenize();

    if(tokens[1]==(char*)NULL || tokens[2]==(char*)NULL || tokens[3]==(char*)NULL)
    	return;

    sscanf(tokens[1],"%ld",&len);           /* read the number of items */

    if( strcmp(tokens[3],"Lines")==0 )      /* find if lines or bytes */
        {                                   /* lines */
        while(len--)
            {
            if(fgets(line,sizeof(line),infile)==(char*)NULL)
                break;
            printer_puts(line);		    /* this doesn't change line termination */
            }
        }
    else                                    /* bytes */
        {
        while(len--)
            {
            printer_putc(fgetc(infile));
            }        
        }
    } /* end of copy_data() */

/*-------------------------------------------------------------------------
** Sanitized getline
** This routine reads from a specified file.
** It is intended that the file be either "-text" or "-comments".
** This routine removes the line termination.
**
** Also, in the case of the comments "%%IncludeFeature:" and "%%BeginFeature:",
** this routine passes them thru to the destination file immediatly, the 
** higher level routines never see them.
-------------------------------------------------------------------------*/

/*
** Read a line from the specified file into the line buffer.
** Return its length in line_len.
**
** The length of line[] is defined above as MAX_LINE+1.
*/
char *dgetline_read(FILE *infile)
    {
    int c;				/* the character read */
    
    line_overflow=FALSE;		/* we don't yet know that it will overflow */

    for(line_len=0; line_len<MAX_LINE; line_len++)
    	{
	if( (c=fgetc(infile)) == EOF )  /* if end of file */
	    {
	    if(line_len)		/* If we have read a line */
	    	break;			/* already, EOF is termination. */
	    else			/* otherwise, */
	        return (char*)NULL;	/* return a null pointer as end of file. */
	    }

	if(c=='\n')			/* If end of line, */
	    break;			/* break off. */

	line[line_len]=c;		/* a character, store it */
    	}

    line[line_len] = (char)NULL;	/* terminate the line */

    if(line_len == MAX_LINE)		/* If for loop terminated because the */
    	line_overflow = TRUE;		/* line was too long, then say so. */

    return line;			/* return pointer to the line */
    } /* end of dgetline_read() */

/*
** If the line should not be returned to the caller, this 
** routine return -1, otherwise it return 0. 
**
** This routine quitely copies feature code, documents, and resources.
** End feature or resource marker lines which occur without begin
** lines are unlikely to have any effect.
*/
int dgetline_parse(FILE *infile)
    {
    if( *line=='%' )                        /* If it is a comment, */
        {                                   /* look for specific comments */
        if(strncmp(line, "%%BeginFeature:", 15) == 0)
            {
            tokenize();
            begin_feature(tokens[1], tokens[2], infile);
            return -1;
            }
        if(strncmp(line, "%%IncludeFeature:", 17) == 0)
            {
            tokenize();
            include_feature(tokens[1], tokens[2]);
            return -1;
            }
        if(strncmp(line,"%%IncludeResource:", 18) == 0)
            {
            include_resource();
            return -1;
            }
        if(strncmp(line, "%%BeginResource:", 16) == 0)
            {
            begin_resource(infile);     /* copy the resource */
            return -1;                  /* including %%EndResource */
            }
        if(strncmp(line,"%%BeginData:", 12) == 0)
            {
            copy_data(infile);
            return -1;
            }
        
        if(strncmp(line,"%%BeginDocument:",16) == 0)
            level++;
        else if( (strcmp(line,"%%EndDocument") == 0) && level )
            level--;

        } /* end of if comment */
    return 0;
    } /* end of dgetline_parse() */

char *dgetline(FILE *infile)
    {
    char *lptr;

    do  {
        lptr=dgetline_read(infile);
        } while( (lptr != (char*)NULL) && *lptr && dgetline_parse(infile) );
			/* this skips parsing for blank lines... interesting */
    return lptr;
    } /* end of dgetline() */

/*----------------------------------------------------------------------
** Routines to emmit the early part of the job, the part before the 
** script section.
**--------------------------------------------------------------------*/

/*
** Copy the header comments from
** the -comments file to the output file.
*/
static void copy_header(void)
    {
    if( dgetline(comments) != (char*) NULL )    /* copy %!... line */
        {					/* we don't have to worry about */
        printer_putline(line);			/* line_overflow (I hope). */                            
        }

    /*
    ** Now the stuff we must restore to the file
    */
    if(job.ForLine == (char*)NULL)
	printer_printf("%%%%For: (<unknown>)\n");
    else
	printer_printf("%%%%For: (%s)\n", job.ForLine);

    if(job.Creator!=(char*)NULL)
	printer_printf("%%%%Creator: (%s)\n", job.Creator);

    if(job.Routing!=(char*)NULL)
	printer_printf("%%%%Routing: %s\n", job.Routing);

    if(job.Title!=(char*)NULL)
	printer_printf("%%%%Title: (%s)\n", job.Title);

    /*
    ** If the number of pages is not unspecified, then print it.
    ** If the number of copies is 1 or copies_pages_countdown starts
    ** at 1, then the complicated formula below is reduced
    ** to ``job.attr.pages''.
    */
    if(job.attr.pages != -1)
	{			
        printer_printf("%%%%Pages: %d\n",
       	    ((sheetcount*job.attr.pagefactor*(copies_pages_countdown-1))+job.attr.pages));
        }

    /*
    ** If we are not re-ordering the job or we are re-ordering
    ** the job but we are not doing duplex or N-Up (which make
    ** for a very confusing page order), print the pageorder.
    ** For now, we will overlook the strange aspects of a document
    ** produced by a copies_pages>1.
    **
    ** The expression which is fed to the switch is rather odd.
    ** The first part, (job.attr.pageorder * current_job_direction)
    ** will always evaluate to 1 or 0.
    */
    if(print_direction==1 || job.attr.pagefactor==1)    
        {
	switch( job.attr.pageorder * current_job_direction * print_direction )
            {
            case PAGEORDER_ASCEND:                  /* 1 */
                printer_putline("%%PageOrder: Ascend");
                break;
            case PAGEORDER_DESCEND:                 /* -1 */
                printer_putline("%%PageOrder: Descend");
                break;
            case PAGEORDER_SPECIAL:                 /* 0 */
                printer_putline("%%PageOrder: Special");
                break;
            }
        }

    /* Orientation */
    switch( job.attr.orientation )
    	{
	case ORIENTATION_PORTRAIT:
	    printer_putline("%%Orientation: Portrait");
	    break;
	case ORIENTATION_LANDSCAPE:
	    printer_putline("%%Orienation: Landscape");
	    break;
    	}

    /* Write the ``%%DocumentNeeded(Supplied)Resources:'' comments. */
    write_resource_comments();

    /* Write the ``%%Requirements:'' comments. */
    write_requirement_comments();

    /* Write the ProofMode */
    switch(job.attr.proofmode)
    	{
    	case PROOFMODE_TRUSTME:
	    printer_printf("%%%%ProofMode: TrustMe\n");
	    break;
	case PROOFMODE_SUBSTITUTE:
	    printer_printf("%%%%ProofMode: Substitute\n");
	    break;
	case PROOFMODE_NOTIFYME:
	    printer_printf("%%%%ProofMode: NotifyMe\n");
	    break;
	}

    /* now the stuff in the "-comments" file */
    while( dgetline(comments) != (char*) NULL ) /* until end of file */
        {
        printer_printf("%s\n",line);
        }

    /* now, explicitly close the comments section */
    printer_printf("%%%%EndComments\n");

    } /* end of copy_header() */

/*
** Copy the document defaults section from the start of the
** "-pages" file.  This will be the first instance of reading
** from the "-pages" file.  If there are no defaults, we will
** rewind the file.
*/
static void copy_defaults(void)
    {
    int defaults=0;

    while( dgetline(page_comments) != (char*)NULL ) /* read a line */
        { 
        if( ! defaults )                            /* if not found yet */
	    {
	    if( strcmp(line,"%%BeginDefaults")==0 ) /* If it is present, */
		{                                   /* then ok. */
		printer_putc('\n');		    /* put a line before "%%BeginDefaults" */
		defaults=TRUE;
		}
	    else                                    /* Otherwise, */
                {                                   /* rewind the */
                rewind(page_comments);              /* "-pages" file, */
                break;				    /* and get out of here. */
                }
	    }

	printer_printf("%s\n",line);				/* sent the line to printer */

	ppr_sscanf(line,"%%%%PageMedia: %#s",sizeof(default_pagemedia),default_pagemedia);

	if(strncmp(line,"%%EndDefaults",13)==0)		/* If it was the last line, */
	    break;					/* then stop. */
	}
    } /* end of copy_defaults() */

/*
** Copy the prolog to the output file.
** Return non-zero if the ensuing "%%BeginSetup" is definitly missing.
** When we exit, the start of the setup section or a page or the    
** trailer should be in the line buffer.
**
** Return 0 if "%%BeginSetup" is in the buffer, return -1 
** if anything else is in the buffer. 
*/     
static int copy_prolog(void)
    {
    /*
    ** Copy blank lines before inserting extra lines.
    ** We must use a special version of dgetline() which
    ** does not automatically copy commented resources.
    ** This is important because otherwise, it might
    ** copy a proceedure set before we get to add our's.
    */
    while( dgetline_read(text)!=(char*)NULL && *line==(char)NULL )
        printer_putline(line);

    /*
    ** If the 1st non-blank line was "%%BeginProlog",
    ** write it to the output file before inserting extra code.
    */
    if(strcmp(line,"%%BeginProlog")==0)
        {                       /* If it is "%%BeginProlog", */
        printer_putline(line);         /* write to output */
        dgetline_read(text);         /* and read the next line. */
        }
    
    /*
    ** Give the code in pprdrv_patch.c a chance to load
    ** any *JobPatchFile sections from the PPD file.
    */
    jobpatchfile();

    /* Insert things in the prolog if we haven't already. */
    if(job.attr.prolog)
        {
        /* If we will need the N-Up library or some other
           extra resource, insert it here. */
	insert_extra_prolog_resources();
        }
    
    /* The line in the buffer didn't get parsed above,
       do it now. */
    while(dgetline_parse(text))
        {                   /* equivelent of dgetline(). */
         if(dgetline_read(text)==(char*)NULL)
            break;
        }

    /* Now, start the main copy loop. */
    do  {
        if( level==0 && (strncmp(line,"%%Page:",7)==0) )
            return -1;
        if( level==0 && (strcmp(line,"%%Trailer")==0) )
            return -1;
        if( level==0 && (strcmp(line,"%%BeginSetup")==0) )
            return 0;

        printer_write(line, line_len);
        if(!line_overflow)
            printer_putc('\n');

        if( level==0 && (strncmp(line,"%%EndProlog",11)==0) )
            {
            dgetline(text);
            return 0;
            }
        } while( dgetline(text) != (char*)NULL );

    return -1;
    } /* end of copy_prolog() */

/*
** Copy the Document Setup section.
**
** Return -1 if we hit the %%Trailer comment or end of file.
*/
static int copy_setup(void)
    {
    /* Copy blank lines before inserting bin select code. */
    while(*line == (char)NULL)				/* while line is zero length */
        {
        printer_putline(line);				/* copy it to the output */
        if( dgetline_read(text) == (char*)NULL)		/* and get the next one */
            break;					/* defering parsing */
        }

    /* Copy "%%BeginSetup" line before inserting bin select code. */
    if(strcmp(line, "%%BeginSetup") == 0)
        {
        printer_putline(line);
        dgetline_read(text);		/* and get the next one, */
        }				/* defering parsing */
    
    /*
    ** Set the job name so that others who want to use the printer will be
    ** able to see who is using it and come looking for our blood.
    */
    set_jobname();

    /*
    ** If this document appears to require only one media and
    ** ppr was invoked with the "-B true" switch (or no -B switch),
    ** then insert bin select code at the top of the document
    ** setup section.  If we insert such code, we will strip
    ** out any pre-existing code.
    */
    if( media_count==1 && job.opts.binselect )
        {
        strip_binselects = select_media(media_xlate[0].pprname);
        }

    /*
    ** If we are printing signatures ourselves, then strip out any code
    ** in the document which might invoke such features.
    ** (job.N_Up.sigsheets is zero if we are not printing signatures.) 
    */
    strip_signature = job.N_Up.sigsheets;

    /*
    ** Now insert those -F switch things which go at the beginning
    ** of the document setup section (set 1).
    ** Most should go here because those which select a page region
    ** do an implicit initmatrix which should occur before the
    ** document setup code scales the coordinate system.
    **
    ** This may set strip_binselects to TRUE.
    */
    insert_features(qstream, 1);

    /* Do this now as the document setup code may select fonts. */
    insert_noinclude_fonts();

    /* If we are using N-Up, insert the N-Up invokation.  This has been
    ** placed after the auto bin select because the N-Up package makes 
    ** itself at home on the currently selected media.  This is after 
    ** insert_noinclude_fonts() so that the `Draft' notice font is
    ** downloaded before N-Up is invoked.
    */
    invoke_N_Up();

    /*
    ** We must now do the line parsing we defered above.  This may
    ** require that we read more lines.
    */
    while(dgetline_parse(text))
	{
        if(dgetline_read(text)==(char*)NULL)
            break;
        }
        
    /*
    ** Now, copy the origional document setup section.
    **
    ** "%%EOF" should not occur since ppr should make sure all 
    ** files have a "%%Trailer" comment.  We do not have to look
    ** for "%%Page:" comments because any file which has any
    ** pages will already have had "%%EndSetup" added.
    */
    do  {
        if(*line=='%' && level==0)
            {
            if(strcmp(line,"%%Trailer")==0)         /* If hit trailer, */
                {
                strip_binselects = FALSE;	    /* stop stripping */
		strip_signature = FALSE;
                return -1;			    /* then no pages. */
                }				    /* (that is why we return -1, */
            }					    /* if we had hit %%EndSetup we would have returned 0.) */

        if( *line=='%' && level==0 && strncmp(line, "%%EndSetup", 10)==0 )
            {                           
            insert_features(qstream,2); /* insert ppr -F *Duplex switch things */

            if(job.opts.copies != -1)   /* specify number of copies */
                {
                if(copies_auto)         /* if auto, insert auto code */
                    set_copies(job.opts.copies);
                else                    /* if not auto copies, */
                    set_copies(1);      /* disable any old auto copies code */
                }                 

            printer_putline("%%EndSetup");     /* print the "%%EndSetup" */

            dgetline(text);             /* get 1st line for page routine */
            strip_binselects = FALSE;	/* stop stripping binselects */
	    strip_signature = FALSE;
            return 0;			/* Return 0 because we hit %%EndSetup, */
            }     			/* there will be pages. */

        printer_write(line, line_len);
        if(!line_overflow)
            printer_putc('\n');
        } while(dgetline(text) != (char*)NULL );

    fatal(EXIT_JOBERR,"queue file lacks \"%%%%Trailer\" comment");
    } /* end of copy_setup() */

/*
** Copy text between the end of the document setup section and
** the start of the 1st page.  This could be important in an EPS 
** file which contains no pages.
** Return -1 if we hit "%%Trailer" or end of file
*/
static int copy_nonpages(void)
    {
    do  {
        if( line[0]=='%' && level==0 )
            {
            if(strcmp(line,"%%Trailer")==0)
                return -1;
            if(strncmp(line,"%%Page:",7)==0)
                return 0;
            }

        printer_write(line, line_len);
        if(!line_overflow)
            printer_putc('\n');
        } while(dgetline(text) != (char*)NULL);

    return -1;
    } /* end of copy_nonpages() */

/*--------------------------------------------------------------------
** Copy the pages to the output file. 
** The principal function, copy_pages() is called with a %%Page 
** line already in the buffer.
--------------------------------------------------------------------*/

/* Get "-pages" file line. */
static char *getpline(void)
    {
    int len;

    if( fgets(pline,256,page_comments) == (char*)NULL )
        return (char*)NULL;
    len = strlen(pline);
    if(len) pline[len-1] = (char)NULL;

    return pline;
    } /* end of getpline() */

/* 
** Make a table of the offset of each page record in the "-pages" file.
** I think this routine leaves the pointer of the pages file at the 
** begining of the line which gives the offset of the start of 
** the trailer.
*/
static int make_pagetable(void)
    {
    int pages;
    int index;

    if(job.attr.pages != -1)	/* If number of pages is declared, */
	pages = job.attr.pages;	/* then use that number. */
    else			/* Otherwise, */
	pages = 500;		/* set arbitrary limit. */

    pages_offsets = (long int *)myalloc(pages, sizeof(long int));
    text_offsets = (long int *)myalloc(pages, sizeof(long int));

    getpline();				/* discard first -pages %%Page: line */
    if(strncmp(pline, "%%Page:", 7))	/* if not a %%Page: line, */
        return 0;			/* there are no pages */

    for(index=0;index<pages;index++)
        {
        if(getpline()==(char*)NULL)     /* read "Offset:" line */
            fatal(EXIT_JOBERR,"make_pagetable(): unexpected end of -pages file");
            
        if( sscanf(pline,"Offset: %ld",&text_offsets[index]) != 1 )
            fatal(EXIT_JOBERR,"make_pagetable(): error reading page offset from -pages");

        pages_offsets[index]=ftell(page_comments);

        while( TRUE )			/* eat up this page */
            {				/* out of the pages file */
            if(getpline()==(char*)NULL)
                return index+1;
            if(*pline=='%')
                {
                if(strncmp(pline,"%%Page:",7)==0)
                    break;
                if(strcmp(pline,"%%Trailer")==0)
                    return index+1;
                }
            }
        }
    return index;                       /* land here only if extra pages */
    } /* end of make_pagetable() */

/*                  
** This is called from copy_pages().
**
** Copy from the "-text" and "-pages" files to the interface pipe until the 
** start of the next page or the start or the trailer is encountered.
** The "-text" and "-pages" files must be at the correct possitions
** before this routine is called.  The "-text" file must be at the start of
** the "%%Page:" comment, the "-pages" file must be just after the 
** "Offset:" comment.
**
** The parameters "newnumber" is used to replace the ordinal number
** in the "%%Page:" comment.
*/
static void copy_page(int newnumber)
    {
    char pagemedia[MAX_MEDIANAME+1];
    char *ptr;

    pagemedia[0]=(char)NULL;

    /*
    ** Copy the line that says "%%Page:".
    ** (Actually, we don't precisely copy it, we read it, parse it, 
    ** and print a new "%%Page:" line with a new ordinal page number.)
    */
    dgetline(text);
    tokenize();
    printer_printf("%%%%Page: %s %d\n",tokens[1]!=(char*)NULL ? tokens[1] : "?",newnumber);
    progress__page_start_comment_sent();	/* tell routines in pprdrv_progress.c */

    /* copy "-pages" file until end of page */
    while( (getpline()!=(char*)NULL) && ( *pline!='%' ||
            (strncmp(pline,"%%Page:",7) && strcmp(pline,"%%Trailer")) ) )
        {
        printer_printf("%s\n",pline);
        
        ppr_sscanf(pline,"%%%%PageMedia: %#s",sizeof(pagemedia),pagemedia);
        }

    /* Copy blank lines before "%%BeginPageSetup". */
    while( (ptr=dgetline(text)) != (char*)NULL && line[0]==(char)NULL )
    	{
    	printer_putc('\n');
    	}

    /* Copy the "%%BeginPageSetup" line if it is there. */
    if( strncmp(line,"%%BeginPageSetup",15)==0)
    	{
    	printer_putline(line);
    	ptr=dgetline(text);
    	}

    /* We may want to insert bin selection code here. */
    #ifdef DEBUG_BINSELECT_INLINE
    printer_printf("%% media_count=%d, job.opts.binselect=%d, pagemedia=\"%s\", default_pagemedia=\"%s\"\n",media_count,job.opts.binselect,pagemedia,default_pagemedia);
    #endif
    if( media_count>1 && job.opts.binselect && (pagemedia[0]!=(char)NULL || default_pagemedia[0]!=(char)NULL) )
        {
	/* 
	** If we manage to select the media, strip_binselects will
	** be set to TRUE to that the old bin select code will be
	** removed.  If this printer does not have bins defined, this
	** function call will return FALSE.
	*/
	if(pagemedia[0] != (char)NULL)
	    strip_binselects = select_media_by_dsc_name(pagemedia);
	else
	    strip_binselects = select_media_by_dsc_name(default_pagemedia);
        }

    /* Copy rest of page text from "-text" file. */
    while( ptr!=(char*)NULL && ( level || line[0]!='%' ||
                (strncmp(line,"%%Page:",7) && strcmp(line,"%%Trailer")) ) )
        {
        printer_write(line, line_len);		/* Copy the line to the printer. */

        if(!line_overflow)			/* If it was a whole line, */
            printer_putc('\n');			/* terminate it. */

	ptr = dgetline(text);
        }

    strip_binselects = FALSE;
    } /* end of copy_page() */ 
            
/*
** Called from main, this ties together a number of routines such as
** make_pagetable() and copy_page().
**
** What does this mean?:
** In this routine, we number the pages from 0 to npages-1.
*/
static void copy_pages(void)
    {
    long int text_trailer_offset;       /* offset of document trailer */
    long int page_comments_trailer_offset;
    int sheetnumber;                     /* cardinal current sheet number */ 
    int sheetlimit;                      /* page number at which we stop */
    int sheetmember;			/* index into current sheet (cardinal) */
    int pages;                          /* Number of pages in document (actual, logical) */
    int pagenumber;			/* cardinal number of current page */
    int pageindex;			/* array index of printing page */
            
    pages = make_pagetable();		/* If no pages, */
    if(pages==0) return;		/* do nothing. */
    
    /*
    ** Remember the offset of the start of the trailer in the "-pages" file.
    ** We do this so that we can move the file pointer back when this
    ** function is done.
    */
    page_comments_trailer_offset = ftell(page_comments);

    /*
    ** Read the "Offset:" line for the trailer from the pages file.
    */
    if(getpline()==(char*)NULL)
        fatal(EXIT_JOBERR, "copy_pages(): unexpected end of -pages file");
    if( sscanf(pline,"Offset: %ld",&text_trailer_offset) != 1 )
        fatal(EXIT_JOBERR, "copy_pages(): error reading page offset from -pages");

    /*
    ** This is the start of a loop which is used to emmit multiple copies
    ** of the script section of a DSC commented PostScript file.  This might
    ** be done to produce collated copies.
    */
    while(copies_pages_countdown--)
	{
    	if(print_direction==-1)		/* If printing backwards, */
            {
            sheetnumber = sheetcount-1;	/* start with last sheet */
            sheetlimit = -1;		/* and end when we overshoot the 1st */
            }
    	else				/* If printing forward, */
            {                               
            sheetnumber = 0;		/* start with 1st sheet */
            sheetlimit = sheetcount;	/* and end when we overshoot last */
            }

    	/*
    	** We go thru this loop once for each sheet.
    	** We will either start at the first sheet
    	** and work toward the end of the document
    	** or start at the end and work toward the 
    	** begining.
    	*/
    	while(sheetnumber != sheetlimit)
      	    {                  
	    #ifdef DEBUG_SIGNITURE_INLINE
	    printer_printf("%% sheetnumber = %d\n", sheetnumber);
	    #endif

	    /*
	    ** If the -p switch was used it could be that we
	    ** are supposed to skip this sheet.
	    ** The evenness and oddness tests look backwards
	    ** but they are not.  They look backwards because
	    ** sheetnumber is a cardinal number.
	    ** Both job.portion.start and job.portion.stop will
	    ** be equal to zero if there is no start or stop limit
	    ** respectively.
	    */
	    if(    ( (sheetnumber%2 == 0) && !job.portion.odd )
	    	|| ( (sheetnumber%2 == 1) && !job.portion.even )
	    	|| ( job.portion.start && sheetnumber < (job.portion.start-1) )
		|| ( job.portion.stop && sheetnumber > (job.portion.stop-1) )
			)
		{
		#ifdef DEBUG_SIGNITURE_INLINE
		printer_printf("%% skiping this sheet\n");
		#endif
		sheetnumber += print_direction;
		continue;
		}

	    /*
	    ** We go thru this loop once for each virtual page on
	    ** the sheet.
	    */
	    for(sheetmember=0; sheetmember<job.attr.pagefactor; sheetmember++)
        	{
		#ifdef DEBUG_SIGNITURE_INLINE
		printer_printf("%%  sheetmember = %d", sheetmember);
		#endif
		/*
		** Compute the page number of the virtual page to print
		** next.  If we are not doing signiture printing this
		** is a straightforward operation.
		**
		** The number we compute is the logical page number
		** with pages number from 0.  If the pages appear 
		**  backwards in the file 0 will be the last page 
		** in the file.
		*/
		if(job.N_Up.sigsheets==0)
		    {
		    pagenumber = sheetnumber * job.attr.pagefactor + sheetmember;
		    }			
		else			/* If not printing this side, */
		    {			/* (-s fonts or -s backs) */
		    if( (pagenumber=signature(sheetnumber, sheetmember)) == -1 )
			{
			#ifdef DEBUG_SIGNITURE_INLINE
			printer_printf(", skipping\n");
			#endif
		    	continue;	/* skip this page. */
		    	}
		    }

		/*
		** Compute the page's index into the
		** file, counting the first page in
		** the file as page 0.  This is where
		** we deal with documents with the
		** pages in reverse order.
		*/
        	if(current_job_direction==1)		/* If we are printing forward, */
            	    pageindex = pagenumber;		/* use number as is. */
        	else					/* If printing in reverse, */
                    pageindex = pages-1-pagenumber;	/* index from other end of document. */

		#ifdef DEBUG_SIGNITURE_INLINE
		printer_printf(", pagenumber = %d, pageindex = %d\n", pagenumber, pageindex);
		#endif

        	if(pageindex < pages)			/* If we have such a page, */
            	    {
            	    if(fseek(page_comments,pages_offsets[pageindex],SEEK_SET))
                	fatal(EXIT_JOBERR, "copy_pages(): fseek() error (-pages)");

		    if( fseek(text,text_offsets[pageindex],SEEK_SET) )
			fatal(EXIT_JOBERR, "copy_pages(): fseek() error (-text)");

		    copy_page(pagenumber+1);      /* copy this one page */
		    }                       
		else                              /* No such page, print a dummy page */
            	    {                             /* if not last sheet to be emmited */
            	    if( copies_pages_countdown	  /* and not last copy */
                    	|| job.N_Up.sigsheets
                    	    || ((sheetnumber+print_direction)!=sheetlimit) )
                	{                                   
                	printer_printf("%%%%Page: dummy %d\n",pagenumber+1);
                	printer_putline("showpage");
                    	}
	    	    }
	    	} /* end of page loop */               
	    sheetnumber += print_direction;       /* compute next sheet number */
      	    } /* end of sheet loop */
    	} /* end of the pages multiple copies loop */

    /*
    ** Take up position at the start of the document trailer which
    ** is where one would expect the -text and -pages file pointers to be
    ** after copying all the pages.
    */
    if( fseek(text, text_trailer_offset, SEEK_SET)
            || fseek(page_comments, page_comments_trailer_offset, SEEK_SET) )
        fatal(EXIT_JOBERR,"copy_pages(): can't seek to trailer");
    dgetline(text);                     /* get the "%%Trailer" line */
    } /* end of copy_pages() */

/*
** Copy the trailer to the output file.
** Unless we hit end of file or "%%EOF", the line "%%Trailer"  
** should be in the buffer. 
*/
static void copy_trailer(void)
    {
    printer_putline("%%Trailer");

    while(dgetline(text)!=(char*)NULL)
        {    
        printer_write(line, line_len);	    /* write the line, specifying length in case of embeded nulls */
        if(!line_overflow)	    /* If line is not a partial one, */
            printer_putc('\n');	    /* terminate it. */

        if( level==0 && (strcmp(line,"%%EOF")==0) )
            break;		    /* If we see our "%%EOF", then stop. */
        }

    close_N_Up();                   /* If N-Up was invoked, clean up. */

    printer_putline("%%EOF");
    } /* end of copy_trailer() */

/*================================================================
** End of job reasembly code,
** Start of helper routines for main().
================================================================*/

/*
** Read the parts of the printer configuration that we need.
** We must get the name of the interface, the address, and the options.
*/
static void pprdrv_read_printer_conf(void)
    {
    FILE *prncf;		/* printer conf file */
    char fname[MAX_PATH];	/* name of printer conf file */
    int line = 0;		/* current line number */
    char tempstr[256];		/* for reading lines */
    char tempstr2[250];		/* for extractions */
    float x1, x2;		/* temporary variable for Charge: */
    int count;			/* sscanf() return value */
    int len;

    sprintf(fname, "%s/%s", PRCONF, printer.Name);
    if( (prncf=fopen(fname,"r")) == (FILE*)NULL )
        fatal(EXIT_PRNERR,"can't open printer config file \"%s\"",fname);

    printer.Address = printer.Name;		/* default address */
    printer.Options = "";			/* default options */
    printer.Commentators = (struct COMMENTATOR *)NULL;
    printer.do_banner = BANNER_DISCOURAGED;	/* default flag */
    printer.do_trailer = BANNER_DISCOURAGED;	/* page settings */
    printer.OutputOrder = 0;			/* unknown */
    printer.charge_per_duplex = 0;		/* default: no charge (not same as zero charge) */
    printer.charge_per_simplex = 0;
    printer.type42_ok = FALSE;			/* unnecessary default, if important will be set by want_ttrasterizer() */
    printer.GrayOK = TRUE;			/* most printers allow non-colour jobs */

    while(fgets(tempstr, sizeof(tempstr), prncf) != (char*)NULL)
        {
        line++;                             /* increment current line */

        if(*tempstr==';' || *tempstr=='#')  /* ignore comments */
            continue;

        len=strlen(tempstr);                    /* remove trailing spaces */
        while(len && isspace(tempstr[--len]))   /* and line feeds */
            tempstr[len]=(char)NULL;

        /* Read name of interface program. */
        if(strncmp(tempstr,"Interface: ",11)==0)
            {
	    int index;

            ppr_sscanf(tempstr,"Interface: %#s",sizeof(tempstr2),tempstr2);
            printer.Interface=mystrdup(tempstr2);
	    
	    /* Set the defaults in case the lookup fails. */
	    printer.feedback = ANSWER_FALSE;		/* default, false */
	    printer.jobbreak = JOBBREAK_CONTROL_D;	/* default */

	    /*
	    ** Look up the default "jobbreak" and "feedback" for
	    ** this interface in the interfaces[] database.
	    */
	    for(index=0;interfaces[index].name!=(char*)NULL;index++)
	    	{
	    	if(strcmp(printer.Interface,interfaces[index].name)==0)
	    	    {
		    printer.feedback = interfaces[index].feedback;
		    printer.jobbreak = interfaces[index].jobbreak;		
		    break;
	    	    }
		}

            } /* end of if "Interface: " */

        /* read the feedback setting */
        else if(strncmp(tempstr,"Feedback: ",10)==0)
            {
            if((printer.feedback=torf(&tempstr[10])) == ANSWER_UNKNOWN)
                fatal(EXIT_PRNERR_NORETRY,"Invalid \"Feedback:\" (%s, line %d)",fname,line);
            }

        /* Read control-d use setting. */
        else if(sscanf(tempstr,"JobBreak: %d",&printer.jobbreak)==1)
            {
	    /* no code */
            }

        /*
        ** Read address to use with interface.  Be careful to 
        ** allow addresses with embedded spaces.
        */
        else if(strncmp(tempstr,"Address: ",9)==0)
            {                               
	    if(tempstr[9]=='"')				/* if quoted */
	    	{
		char *ptr = &tempstr[10];
		
		ptr[strcspn(ptr,"\"")] = (char)NULL;	/* use until quote */
		printer.Address = mystrdup(ptr);
	    	}
	    else
            	{
            	printer.Address = mystrdup(&tempstr[9]);
            	}
            }

        /* Read options to use with interface. */
        else if(strncmp(tempstr, "Options: ", 9) == 0)
            {                                                       
            int len=strlen(&tempstr[9]);			/* remove */
	    while( (--len>0) && isspace(tempstr[9+len]) )	/* trailing */
		tempstr[9+len]=(char)NULL;			/* spaces */
            printer.Options = mystrdup(&tempstr[9]);		/* duplicate */
            }

        /*
        ** Read amount to charge the poor user.
	** We read two amounts in dollars or pounds, etc. 
	** and then convert them to cents, new pence, etc.
        */
        else if((count=sscanf(tempstr, "Charge: %f %f", &x1, &x2)) >= 1)
            {   /* ^ notice use of sscanf() rather than ppr_sscanf() ^ */
            printer.charge_per_duplex = (int)(x1 * 100.0);
            if(count == 2)
            	printer.charge_per_simplex = (int)(x2 * 100.0);
	    else
	    	printer.charge_per_simplex = printer.charge_per_duplex;
            }

        /* Read the printer's flag page vote. */
        else if(ppr_sscanf(tempstr,"FlagPages: %d %d",
                    &printer.do_banner,&printer.do_trailer)==2)
            { 
            if( printer.do_banner > BANNER_REQUIRED
                    || printer.do_banner < BANNER_FORBIDDEN
                    || printer.do_trailer > BANNER_REQUIRED
                    || printer.do_trailer < BANNER_FORBIDDEN )
                fatal(EXIT_PRNERR_NORETRY,
                    "Invalid \"FlagPages:\" (%s, line %d)",fname,line);
            }

	/*
	** Read the printers OutputOrder specification.
	** Remember that if there is no such line, the
	** information is taken from the PPD file.
	*/
        else if(strncmp(tempstr,"OutputOrder: ",13)==0)
            {
            char *ptr=&tempstr[13+strspn(&tempstr[13]," \t")];

            if(strcmp(ptr,"Normal")==0)
                printer.OutputOrder=1;
            else if(strcmp(ptr,"Reverse")==0)
                printer.OutputOrder=-1;
            else
                fatal(EXIT_PRNERR_NORETRY, "Invalid \"OutputOrder:\" (%s, line %d)", fname, line);
            }

        else if(strncmp(tempstr, "PPDFile: ", 9) == 0)
            {
	    int x=9+strspn(&tempstr[9]," \t");
            int len=strlen(tempstr);                        /* remove */
            while( (--len>0) && isspace(tempstr[len]) )     /* trailing */
                tempstr[len]=(char)NULL;                    /* spaces */

	    if(tempstr[x]=='/')				/* use absolute path */
		{
		printer.PPDFile=mystrdup(&tempstr[x]);  /* directly */
		}
	    else
	        {
		sprintf(tempstr2, PPDDIR"/%s", &tempstr[x]);
		printer.PPDFile = mystrdup(tempstr2);
	        }
            }

	else if(strncmp(tempstr, "Commentator: ", 13) == 0)
	    {
	    struct COMMENTATOR *newcom;
	    
	    newcom = (struct COMMENTATOR*)myalloc(1,sizeof(struct COMMENTATOR));
	    newcom->options = (char*)NULL;

	    if(ppr_sscanf(tempstr, "Commentator: %d %S %Q %Q",
	    		&newcom->interests,		/* interesting commentary() flags values */
	    		&newcom->progname,		/* commentator program */
	    		&newcom->address,		/* address */
			&newcom->options) >= 3 )
	    	{
		newcom->next = printer.Commentators;
		printer.Commentators = newcom;
	    	}
	    else
	    	{
		fatal(EXIT_PRNERR_NORETRY,"Invalid \"Commentator:\" line");	    	
	    	}
	    }
	else if(strncmp(tempstr, "GrayOK: ", 8) == 0)
	    {
	    if( (printer.GrayOK=torf(&tempstr[8])) == ANSWER_UNKNOWN )
	    	fatal(EXIT_PRNERR_NORETRY,"Invalid \"GrayOK:\" line");
	    }

        } /* end of while(), unknown lines are ignored because we don't use all possible lines */
    } /* end of pprdrv_read_printer_conf() */

/*
** Choose the methode by which we will print multiple copies.
** This routine will always return.  It is called once by main().
*/
static void select_copies_method(void)
    {
    if(job.opts.copies < 2)
        {
        copies_auto = FALSE;
        copies_doc_countdown = 1;
        copies_pages_countdown = 1;
        return;
        }

    if(!job.opts.collate)
        {
        copies_auto = TRUE;
        copies_doc_countdown = 1;
        copies_pages_countdown = 1;
        }
    else
        {
        if( !job.attr.script || (job.attr.pageorder==PAGEORDER_SPECIAL) )
            {
            copies_auto = FALSE;
            copies_doc_countdown = job.opts.copies;
            copies_pages_countdown = 1;
            }
        else
            {
            copies_auto = FALSE;
            copies_doc_countdown = 1;
            copies_pages_countdown = job.opts.copies;
            }
        }
    } /* end of select_copies_method() */

/*
** Handler for terminate signal.
** If we have launched the interface, kill it.
*/
static void sigterm_handler(int sig)
    {
    #ifdef DEBUG
    debug("sigterm_handler(): caught signal %d (%s)", sig, strsignal(sig));
    #endif

    /* if interface has been launched */
    if(intpid)
        {
        DODEBUG_INTERFACE(("sigterm_handler(): killing interface, intpid=%d", intpid));

	killing_interface = TRUE;	/* suppress error response */

	kill((intpid*(-1)), SIGHUP);	/* kill the interface */
        }                               /* and its children */

    /* If there is no interface living: */
    #ifdef DEBUG_INTERFACE
    else
        debug("sigterm_handler(): no interface to kill");
    #endif

    /*
    ** If we are testing the code which is invoked only if certain
    ** things happen while halting the printer or canceling the active
    ** job then delay here so that the cancel operation will
    ** take a noticable period of time.
    **
    ** We have to restart sleep() because SIGIO and SIGCHLD will
    ** interupt it.
    */
    #ifdef DEBUG_DIE_DELAY
    {
    int unslept = DEBUG_DIE_DELAY;
    debug("sigterm_handler(): delaying for %d seconds", DEBUG_DIE_DELAY);
    while( (unslept=sleep(unslept)) > 0 );
    debug("sigterm_handler(): delay ended");
    }
    #endif

    /*
    ** Exit, letting the commentators and pprd know that though
    ** we are exiting gracefully after catching a signal.
    */
    exit_with_commentary(EXIT_SIGNAL, "printing halted");
    } /* end of sigterm_handler() */

/*
** If the interface dies while we are writing to it we may
** receive SIGPIPE before we receive SIGCHLD.  If we are 
** running in test mode it is easily possible that we will
** receive SIGPIPE.  In test mode we exit.  If not in test mode
** we will wait for SIGCHLD.
*/
static void sigpipe_handler(int sig)
    {
    if(test_mode)
	fatal(EXIT_PRNERR, "SIGPIPE caught");
    #ifdef DEBUG
    else
	debug("SIGPIPE received, notification of child death anticipated");
    #endif
    } /* end of sigpipe_handler() */

/*
** Install the signal handlers.  This is called once from main().
*/
static void install_signal_handlers(void)
    {
    struct sigaction sig;

    /*
    ** Set a signal handler for hangups and such.  Remember, signal()
    ** is BSDish even on non-BSD systems.
    */
    signal(SIGCHLD, reapchild);		/* child will eventually terminate */
    signal(SIGHUP, sigterm_handler);	/* who knowns, these might happen */
    signal(SIGINT, sigterm_handler);
    signal(SIGTERM, sigterm_handler);	/* pprd may terminate me */
    signal(SIGPIPE, sigpipe_handler);	/* interface may die as infant */

    /* 
    ** Install the SIGUSR1 handler in such a way that
    ** interupted system calls are re-started if at
    ** all possible. 
    **
    ** SIGUSR1 handler may be used as part of the communications
    ** protocol between pprdrv and the interface.
    */
    sig.sa_handler = user_sighandler;
    sigemptyset(&sig.sa_mask);   
    #ifdef SA_RESTART
    sig.sa_flags = SA_RESTART;
    #else
    sig.sa_flags = 0;
    #endif
    sigaction(SIGUSR1, &sig, (struct sigaction *)NULL); 
    } /* end of install_signal_handlers() */
    
/*
** Compute things about pages.  This is called once from main().
** This routine will only exit if there is an internal error.
*/
static void page_computations(void)
    {
    /*
    ** Compute the number of physical pages in one copy of this job.
    **
    ** job.N_Up.sigsheets is the number of pieces of paper in each
    ** signiture.
    **
    ** job.attr.pages is the number of "%%Page:" comments
    ** in the file.
    **
    ** job.attr.pagefactor is the number of "%%Page:" comments per 
    ** piece of paper.  It is the N-Up factor times 1 for simplex 
    ** or 2 for duplex.
    */
    if(job.N_Up.sigsheets == 0)		/* if not signature printing, */
	{
    	sheetcount = (job.attr.pages + job.attr.pagefactor - 1) / job.attr.pagefactor;
	}
    else				/* if signature printing, */
    	{
	/* Compute number of signatures required, counting any partial sig as whole */
	int signature_count=
		(job.attr.pages+(job.N_Up.N * 2 * job.N_Up.sigsheets)-1)
			/
		(job.N_Up.N * 2 * job.N_Up.sigsheets);

    	/* Now that we know the number of signitures, compute the number of
    	   sheets.   The clause (job.N_Up.N * 2)/job.attr.pagefactor) doubles
    	   the number of sheets if duplex mode is off. */
	sheetcount = signature_count * job.N_Up.sigsheets * ((job.N_Up.N * 2)/job.attr.pagefactor);
    	}

    /*
    ** Determine which way the pages are ordered now.
    */
    if(job.attr.pageorder == PAGEORDER_SPECIAL)
        current_job_direction = 1;	/* assume it is not backwards */
    else if(job.attr.pageorder == PAGEORDER_ASCEND)                     
        current_job_direction = 1;	/* it is forward */
    else if(job.attr.pageorder == PAGEORDER_DESCEND)
        current_job_direction = -1;	/* it is backwards */
    else
        fatal(EXIT_JOBERR, "PageOrder is invalid");

    /*
    ** If changing the page order is permissible, then reverse 
    ** the printing direction if we believe this printer has
    ** a face up output tray.
    */
    if(job.attr.pageorder == PAGEORDER_SPECIAL)
    	print_direction = 1;
    else
	print_direction = printer.OutputOrder;

    DODEBUG_MAIN(("main(): page_computations(): pageorder=%d, OutputOrder=%d, print_direction=%d, current_job_direction=%d",
	job.attr.pageorder, printer.OutputOrder, print_direction, current_job_direction));
    } /* end of page_computations() */

/* 
** If we suceeded in printing the document and a printlog
** file exists, put an entry in it.   This routine
** is called once by main().
*/
static void printer_use_log(struct timeval *start_time)
    {
    int printlogfd;

    if( (printlogfd=open(PRINTLOG_PATH, O_WRONLY | O_APPEND)) != -1 )
	{
	FILE *printlog;			/* For print log file. */

	if( (printlog=fdopen(printlogfd,"a")) == (FILE*)NULL )
	    {
	    error("printer_use_log(): fdopen() failed on printlog file descriptor");
	    close(printlogfd);
	    }
	else
	    {
	    int sidecount;
	    int total_printed_sheets;
	    int total_printed_sides;

	    struct timeval time_now, time_elapsed;
	    time_t seconds_now; 		/* For time stamping the */
	    struct tm *time_detail;		/* print log file. */
	    char time_str[13];

	    /*
	    ** Compute the number of printed sides.  Normally this is
	    ** straight-forward, but in signiture mode we must account
	    ** for "pages" that are left blank.
	    */
	    if(job.N_Up.sigsheets == 0)
	    	{
	    	sidecount = (job.attr.pages + job.N_Up.N - 1) / job.N_Up.N;
	    	}
	    else
	    	{
		if((job.N_Up.sigpart & SIG_BOTH) == SIG_BOTH)
	    	    sidecount = sheetcount * 2;
		else
	    	    sidecount = sheetcount;
		}

	    /*
	    ** Compute the total number of sheets and the total number of printed sides
	    ** in all copies of this job.  Remember that if the number of copies desired
	    ** was not specified then job.opts.copies will be -1.  In this case
	    ** we will assume that the PostScript code of the job only prints one copy.
	    */
	    if(job.opts.copies > 1)		/* if multiple copies */
		{
		total_printed_sheets = sheetcount * job.opts.copies;
        	total_printed_sides = sidecount * job.opts.copies;
        	}
	    else				/* 1 or unknown # of copies */
        	{
        	total_printed_sheets = sheetcount;
        	total_printed_sides = sidecount;
        	}

	    /*
	    ** Get the current time with microsecond resolution
	    ** and compute the difference between the current time
	    ** and the time that was recorded when we started.
	    */
	    gettimeofday(&time_now, (struct timezone *)NULL);
	    time_elapsed.tv_sec = time_now.tv_sec - start_time->tv_sec;
	    time_elapsed.tv_usec = time_now.tv_usec - start_time->tv_usec;
	    if(time_elapsed.tv_usec < 0) {time_elapsed.tv_usec += 1000000; time_elapsed.tv_sec--;}

	    /* Convert the current time to a string of ASCII digits. */
	    seconds_now = /* time((time_t*)NULL) */ time_now.tv_sec;
	    time_detail = localtime(&seconds_now);
	    strftime(time_str, sizeof(time_str), "%y%m%d%H%M%S", time_detail);    

	    fprintf(printlog, "%s,%s,%s,\"%s\",%s,\"%s\",%d,%d,%d,%ld,%ld.%02d\n",
	    	time_str,
	    	QueueFile,
	    	printer.Name,
		job.ForLine == (char*)NULL ? "<missing>" : job.ForLine,
		job.username,
		job.proxy_for == (char*)NULL ? "" : job.proxy_for,
		job.attr.pages,
	    	total_printed_sheets,
	    	total_printed_sides,
	    	(long)(seconds_now - job.time),
	    	(long)time_elapsed.tv_sec, (int)(time_elapsed.tv_usec / 10000) );

	    if(fclose(printlog) == EOF)
	    	error("printer_use_log: fclose() failed");
	    }
	}
    } /* end of print_use_log() */
    
/*
** Copy function for the transparent mode hack.
*/
static void transparent_hack_copy(char *qf)
    {
    char fname[MAX_PATH];
    int handle;
    unsigned char buffer[4096];
    int len;

    sprintf(fname, DATADIR"/%s-infile", qf);

    if( (handle = open(fname, O_RDONLY)) == -1 )
	fatal(EXIT_JOBERR, "transparent_hack_copy(): can't open \"%s\", errno=%d (%s)", fname, errno, strerror(errno));

    while( (len=read(handle, buffer, sizeof(buffer))) > 0 )
	printer_write(buffer, len);
	
    if( len == -1 )
    	fatal(EXIT_PRNERR_NORETRY, "transparent_hack_copy(): read() failed, errno=%d (%s)", errno, strerror(errno) );

    close(handle);    
    } /* end of transparent_hack_copy() */

/*
** main procedure
*/
int main(int argc, char *argv[])
    {
    int result;		/* exit code of interface */
    int group_pass;	/* Number of the pass thru the group (a pprdrv invokation parameter). */
    int argi = 1;
    
    /* Note the start time for the logs: */
    gettimeofday(&start_time, (struct timezone *)NULL);

    /* Should we run in test mode? */
    if(argc >= 2 && strcmp(argv[argi], "--test") == 0)
    	{
    	argi++;
    	test_mode = TRUE;
	chdir(HOMEDIR);
    	}

    /* We mustn't call fatal() here because printer.Name is not set. */
    if((argc - argi) < 3)	/* if fewer than 3 remaining arguments, */
	{
	fputs("Usage: pprdrv [--test] <printer> <queuefile> <pass>\n", stderr);
	exit(EXIT_PRNERR_NORETRY);
	}

    /* Assign each of the three parameters to a variable with a meaningful name: */
    printer.Name = argv[argi];
    QueueFile = argv[argi+1];
    group_pass = atoi(argv[argi+2]);

    DODEBUG_MAIN(("main(): printer.Name=\"%s\", QueueFile=\"%s\", group_pass=%d", printer.Name, QueueFile, group_pass));
    if(test_mode) fprintf(stderr, "Test mode, formatting job %s for printer %s.\n", QueueFile, printer.Name);

    /* Install handlers for SIGTERM, SIGCHLD, SIGUSR1, etc. */
    install_signal_handlers();

    /* Read the printer configuration. */
    DODEBUG_MAIN(("main(): pprdrv_read_printer_conf()"));
    pprdrv_read_printer_conf();
    DODEBUG_MAIN(("main(): read_PPD_file()"));
    read_PPD_file(printer.PPDFile); 
    DODEBUG_MAIN(("main(): interface=\"%s\", address=\"%s\", options=\"%s\"", printer.Interface, printer.Address, printer.Options));

    /* If the outputorder is still unknown, make it forward. */
    if(printer.OutputOrder == 0)
        printer.OutputOrder = 1;

    /*
    ** If we are running in test mode, change some things.
    */
    if(test_mode)
    	{
    	printer.Name = "-";		/* for alert() */
    	printer.charge_per_duplex = 0;
    	printer.charge_per_simplex = 0;
	printer.feedback = FALSE;
	printer.jobbreak = JOBBREAK_NONE;
	printer.OutputOrder = 1;
	printer.do_banner = BANNER_FORBIDDEN;
	printer.do_trailer = BANNER_FORBIDDEN;
    	}

    /*
    ** Read the queue file into the structure "job".
    **/
    DODEBUG_MAIN(("main(): reading queue file"));
    {
    char tempfname[MAX_PATH];

    sprintf(tempfname, QUEUEDIR"/%s", QueueFile);

    if( (qstream=fopen(tempfname,"r")) == (FILE*)NULL )
    	fatal(EXIT_JOBERR, "can't open queue file \"%s\", errno=%d (%s)", tempfname, errno, strerror(errno));

    if( read_struct_QFileEntry(qstream, &job) )
        fatal(EXIT_JOBERR, "defective queue file data");

    read_media_lines(qstream);
    }

    /*
    ** Read the requirement lines from the queue file and
    ** and determine if we can meet the requirements.
    ** This routine takes the ProofMode into account.
    ** (See RBII p. 664.)
    */
    DODEBUG_MAIN(("main(): check_if_capable()"));
    if( check_if_capable(qstream, group_pass) )
	{
	DODEBUG_MAIN(("main(): not capable, exiting"));
        return EXIT_INCAPABLE;
        }

    /* Give the N-Up machinery time to ask for its resources. */
    prestart_N_Up_hook();

    /*
    ** Compute the number of physical pages and sheets in one copy
    ** of this job.
    */
    page_computations();

    /* 
    ** Give the output buffering routines a chance to set
    ** up their SIGALRM handler. 
    */
    DODEBUG_MAIN(("main(): printer_bufinit()"));
    printer_bufinit();
    
    /*
    ** Launch the interface program.
    */
    DODEBUG_MAIN(("main(): start_interface()"));
    start_interface();

    /*
    ** Set for signals when data received from interface.
    */
    DODEBUG_MAIN(("main(): feedback_start()"));
    feedback_start();

    /*
    ** If feedthru mode is turned on, just copy the origional job file
    ** and then skip to the end.
    */
    if( job.opts.hacks & HACK_TRANSPARENT )
	{
	transparent_hack_copy(QueueFile);
	goto feedthru_skip_point;    	
	}

    /*
    ** open the job files
    */
    DODEBUG_MAIN(("main(): opening job files"));
    {
    char tempfname[MAX_PATH];

    sprintf(tempfname, DATADIR"/%s-comments", QueueFile);
    if( (comments=fopen(tempfname, "r")) == (FILE*) NULL )
        fatal(EXIT_JOBERR, "can't open \"%s\"", tempfname);

    sprintf(tempfname, DATADIR"/%s-pages", QueueFile);
    if( (page_comments=fopen(tempfname, "r")) == (FILE*) NULL )
        fatal(EXIT_JOBERR, "can't open \"%s\"", tempfname);

    sprintf(tempfname, DATADIR"/%s-text", QueueFile);
    if( (text=fopen(tempfname, "r")) == (FILE*) NULL )
        fatal(EXIT_JOBERR, "can't open \"%s\"", tempfname);
    }
    
    /* Possibly send setup strings to the printer. */
    DODEBUG_MAIN(("main(): printer_setup()"));
    printer_setup();

    /* Download any patchfile if it is not already downloaded. */
    DODEBUG_MAIN(("main(): patchfile()"));
    patchfile();

    /* Download any persistent fonts or other resources. */
    DODEBUG_MAIN(("main(): persistent_download_now()"));
    persistent_download_now();

    /* Print banner or trailer page. */          
    DODEBUG_MAIN(("main(): print_flag_page()"));
    print_flag_page(printer.OutputOrder,0);        

    /* 
    ** Now that we are done with patches and banner page, 
    ** we will now begin sending the primary job.
    */
    doing_primary_job = new_doing_primary_job = TRUE;

    /*
    ** Select a multiple copies method
    */
    select_copies_method();

    /*
    ** Start of the loop we used the print multiple copies by sending
    ** the whole job multiple times in stead of repeating the script section.
    */
    while(copies_doc_countdown--)
        {
	/* write the document over the pipe */

	/* copy header comments */
	DODEBUG_MAIN(("main(): copy_header()"));
	copy_header();

	/* copy document defaults */
	DODEBUG_MAIN(("main(): copy_defaults()"));
	copy_defaults();

	/*
	** If the job contains not marked prolog section,
	** include a fake one at the top.
	*/
	if(!job.attr.prolog)
            {
            printer_putline("% no prolog, PPR will add code here:");

	    /*
	    ** Give the code in pprdrv_patch.c a change to include
	    ** any *JobPatchFile sections from the PPD file.
	    */
	    jobpatchfile();

	    /*
	    ** Include any extra resources such as the N-Up
	    ** dictionary or the TrueType dictionary.
	    */
	    insert_extra_prolog_resources();

            printer_putline("% end of non-conforming prolog");
            }

	/*
	** If this document does not have a marked docsetup section,
	** include a fake one at the top.
	*/
	if(!job.attr.docsetup)			/* if no conforming doc setup section, */
            {					/* put non-conforming one at job start */
            printer_putline("% no document setup section, PPR will add code here:");
	    set_jobname();
            if(job.opts.copies != -1)		/* don't worry about copies_auto==FALSE */
                printer_printf("/showpage { /#copies %d def showpage } bind def %%PPR\n", job.opts.copies);
            if( media_count==1 && job.opts.binselect )
                select_media(media_xlate[0].pprname);
            invoke_N_Up();
            insert_features(qstream, 1);	/* insert most ppr -F switch things */
            insert_features(qstream, 2);	/* insert ppr -F *Duplex switch thing */
            insert_noinclude_fonts();
            printer_putline("% end of non-conforming document setup"); 
            }
                    
	/*
	** Copy the main prolog, document setup, and script sections:
	*/
	DODEBUG_MAIN(("main(): copying various code sections"));
	if(! copy_prolog() )            /* Copy prolog, and if it did not end */
	    if(! copy_setup() )         /* with "%%Trailer", copy Document Setup */
		if(!copy_nonpages())    /* if it didn't end, copy non-page text */
		    copy_pages();       /* and then if still no end, copy pages. */

	/* copy the trailer */	
	DODEBUG_MAIN(("main(): copy_trailer()"));
	copy_trailer();

	/*
	** If we are printing multiple copies by sending the
	** whole document again and again, then send end of
	** file to the printer and rewind those files and
	** go back and do it again.  This clause is not 
	** ``true'' on the last copy.
	*/
	if(copies_doc_countdown)
	    {
	    DODEBUG_MAIN(("main(): jobbreak()"));
	    jobbreak();
	    rewind(comments);
	    rewind(page_comments);
	    rewind(text);
	    }
	} /* this bracket goes with the start of the copies loop */

    /*
    ** We are done doing the main job.  Anything after this is 
    ** a trailer page.  (We don't change doing_primary_job directly
    ** because it mustn't change until and unless the trailer page
    ** code calls jobbreak().  If it changed too soon then PJL
    ** USTATUS PAGE messages from the printer would be credited to
    ** the trailer page instead of the main job.)
    */
    new_doing_primary_job = FALSE;

    /* close the files */
    fclose(qstream);            /* close the queue file */
    fclose(comments);           /* close the comments file */
    fclose(page_comments);      /* close the page comments and index file */
    fclose(text);               /* close the body text file */

    /*
    ** Print a header or trailer page if proper to do so.
    ** We must close the job log so we can print it.
    */
    close_log();
    DODEBUG_MAIN(("main(): print_flag_page()"));
    print_flag_page(printer.OutputOrder*-1, 1);

    /* Send final code make the printer happy for the next job. */
    DODEBUG_MAIN(("main(): printer_cleanup()"));
    printer_cleanup();

    /* Point to skip to after doing a raw copy: */
    feedthru_skip_point:

    /*
    ** Close the pipes to the interface and wait
    ** for it to exit. 
    */
    DODEBUG_MAIN(("main(): close_interface()"));
    result = close_interface();
    DODEBUG_MAIN(("main(): interface terminated with code %d", result));

    /*
    ** If job was transmitted ok but a PostScript error was detected, then
    ** we have job error.  Since Ghostscript has very erratic error
    ** return codes, we rely on our own judgement more if we think
    ** we are talking to Ghostscript.  The variable "ghostscript" 
    ** will be true if we have received at least one error message
    ** in Ghostscript's format.
    **
    ** HACK:  The Kodak ColorEase PS is so broken that it drops the 
    ** AppleTalk connexion after announcing that it is going to
    ** flush.  That is why I have removed most of the condition
    ** for now.
    */
    #if 0
    if( (result==EXIT_PRINTED || ghostscript) && posterror )
    #else
    if(posterror)
    #endif
        result = EXIT_JOBERR;
                                            
    /*
    ** Tell the commentators why we are exiting.  We don't just wait to
    ** call exit_with_commentary() because we want the commentator
    ** to be able to run in parallel with our final operations.
    */
    if(result==EXIT_JOBERR && posterror)
	exit_commentary(result, "postscript error");
    else
	exit_commentary(result, (char*)NULL);

    /*
    ** If we charge for the use of this printer and the
    ** job printed normally, then charge now.
    ** (Don't bother posting charges of zero.)
    */
    if(result == EXIT_PRINTED && (printer.charge_per_duplex > 0 || printer.charge_per_simplex > 0))
        {
	/*
	** Of course, we can't charge nobody.  Not that we would 
	** expect such a job to get this far, though it could be that
	** the printer was protected after this job entered the queue.
	*/
	if(job.charge_to == (char*)NULL)
	    {
	    error("main(): can't charge because \"%%%%For:\" line is blank!");
	    }
        else 
            {
	    struct COMPUTED_CHARGE charge;

	    compute_charge(&charge, 
	    	printer.charge_per_duplex, printer.charge_per_simplex,
	    	job.attr.pages, job.N_Up.N, job.attr.pagefactor,
	    	job.N_Up.sigsheets, job.N_Up.sigpart, job.opts.copies);
	
	    if(db_transaction(job.charge_to, charge.total, TRANSACTION_CHARGE) != USER_OK)
		error("main(): failed to charge user using db_transaction()");
            }
        }
                        
    /* Possibly add an entry to the log of jobs printed. */
    if(result == EXIT_PRINTED)
	printer_use_log(&start_time);

    /*
    ** If job transmission was sucessful, delete the printer status file.
    */
    if( (result==EXIT_PRINTED || result==EXIT_JOBERR) && !test_mode )
	{
	char fname[MAX_PATH];
	sprintf(fname, ALERTDIR"/%s.status", printer.Name);
	unlink(fname);
	progress__new_message("");
	}

    /*
    ** Wait for all the commentators to exit and
    ** then exit, returning a result code to pprd.
    */
    commentator_wait();
    DODEBUG_MAIN(("main(): exiting with code %d", result));
    return result;
    } /* end of main() */

/* end of file */
