/*
** ~ppr/src/pprd/pprd_respond.c 
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 27 February 1997.
*/

/*
**
** Routines to respond to users.  Users are informed of job completion, job deletion,
** and other interesting events by running programs in the responders directory.
**
** These routines may appear to be similiar to those in ppr_respond.c and you
** may well wonder why the same source module us not used for both.  The 
** reason is that none of the actual response message occur in both.
**
** The function respond() is most often called from within a signal handler.
** This means that none of the code in this module should call routines
** which are not reentrant.  That condition is not currently met.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include "pprd.h"
#include "respond.h"
#include "pprd.auto_h"

#define MAX_RFOR 32	/* max length of "For: " line argument to use in response */
#define MAX_RTITLE 32	/* max length of "Title: " line argument to use in response */
#define MAX_RREASON 64

struct RINFO
    {
    const char *destname;
    int id;
    int subid;
    const char *homenode;

    int response_code;
    int prnid;
    const char *prnname;

    /* Responder program, address and options. */
    char response_method[MAX_RESPONSE_METHOD+1];
    char response_address[MAX_RESPONSE_ADDRESS+1];
    char responder_options[MAX_RESPONDER_OPTIONS+1];

    char ForLine[MAX_RFOR+1];		/* person for whom we are doing this */
    char Title[MAX_RTITLE+1];

    long Time;
    char Reason[MAX_RREASON+1];

    int pages_printed;
    struct COMPUTED_CHARGE charge;	/* the amount of money to charge */
    } ;

/*
** Read enough information from the queue file to send the response to the
** right place with the right parameters.
**
** This routine is passed a partially filled in RINFO structure, it must fill
** in the rest.
**
** Since this routine is called from the SIGCHLD handler, it must not call
** malloc() or stdio.
**
** Since this routine operates on files generated by ppr and not by hand, 
** it will assume that one space follows the colon that separates the keyword
** from its value.  We also assume that there is no trailing space to be 
** removed.
*/
static void respond_get_info(struct RINFO *rinfo)
    {
    char fname[MAX_PATH];		/* room to build name of queue file */
    int fhandle;			/* file handle of queue file */
    #define MAX_QF 8192			/* max number of bytes to scan for information */
    char data[MAX_QF+1];		/* buffer for queue file contents */
    int data_len;			/* length of data in buffer */
    int index;				/* index of line we are working on */
    int line_len;			/* length of line we are working on */
    int pages, copies, N, pagefactor, sigsheets, sigpart;
    int charge_to = FALSE;
    
    pages = copies = -1;
    pagefactor = N = 1;

    DODEBUG_RESPOND(("respond_get_info(\"%s\", %d, %d, \"%s\", rinfo)", destname, id, subid, homenode));

    /* Be paranoid, fill in default values. */
    strcpy(rinfo->response_method, "none");
    rinfo->response_address[0] = (char)NULL;
    rinfo->responder_options[0] = (char)NULL;
    rinfo->ForLine[0] = (char)NULL;
    rinfo->Title[0] = (char)NULL;
    rinfo->Time = 0;
    rinfo->Reason[0] = (char)NULL;

    sprintf(fname, QUEUEDIR"/%s:%s-%d.%d(%s)", ppr_get_nodename(), rinfo->destname, rinfo->id, rinfo->subid, rinfo->homenode);

    if( (fhandle=open(fname, O_RDONLY)) == -1 )
        {
        error("respond_get_info(): can't open \"%s\", errno=%d (%s)", fname, errno, strerror(errno));
        return;
        }

    if( (data_len=read(fhandle, data, MAX_QF)) == -1 )
    	{
    	error("respond_get_info(): read() failed, errno=%d (%s)", errno, strerror(errno));
    	return;
    	}

    /* If we will miss the end of the queue file, make a note of it. */
    if(data_len == MAX_QF)
    	error("respond_get_info(): insufficient room for entire queue file");

    /* Be paranoid. */
    if(data_len > MAX_QF || data_len < 0)
    	fatal(0, "pprd_respond.c: respond_get_info(): read returned %d", data_len);

    /* NULL terminate the buffer. */
    data[data_len] = (char)NULL;

    /* Consider each line in the queue file. */
    for(index=0; index < data_len; index += (line_len+1))
        {
	char *line = &data[index];
	line_len = strcspn(line, "\n");
	line[line_len] = (char)NULL;

	if(ppr_sscanf(line, "Response: %#s %#s %#z",
		MAX_RESPONSE_METHOD, rinfo->response_method,
		MAX_RESPONSE_ADDRESS, rinfo->response_address,
		MAX_RESPONDER_OPTIONS, rinfo->responder_options) >= 2)
            {
	    /* nothing to do */
	    }

        else if(strncmp(line, "For: ", 5) == 0)
            {
	    int len = strlen(&line[5]);
	    if(len > MAX_RFOR) len = MAX_RFOR;
            strncpy(rinfo->ForLine, &line[5], len);
            rinfo->ForLine[len] = (char)NULL;
            }

	else if(strncmp(line, "Title: ", 7) == 0)
	    {
	    int len = strlen(&line[7]);
	    if(len > MAX_RTITLE) len = MAX_RTITLE;
	    strncpy(rinfo->Title, &line[7], len);
	    rinfo->Title[len] = (char)NULL;
	    }

	else if(strncmp(line, "Time: ", 6) == 0)
	    {
	    ppr_sscanf(&line[6], "%ld", &rinfo->Time);	    
	    }
	    
	else if(strncmp(line, "Reason: ", 8) == 0)
	    {
	    if(rinfo->response_code == RESP_ARRESTED
	    	    || rinfo->response_code == RESP_ARRESTED_PRINTER_INCAPABLE
	    	    || rinfo->response_code == RESP_ARRESTED_GROUP_INCAPABLE)
		{
		int len = strlen(&line[8]);
		if(len > MAX_RREASON) len = MAX_RREASON;
		strncpy(rinfo->Reason, &line[8], len);
		rinfo->Reason[len] = (char)NULL;	    
		}
    	    }
	else if(sscanf(line, "Attr: %*s %*s %d %*s %*s %*s %*s %*s %d", &pages, &pagefactor) == 9) {}
	else if(sscanf(line, "Opts: %*s %d", &copies) == 2) {}
	else if(sscanf(line, "N_Up: %d %*s %d %d", &N, &sigsheets, &sigpart) == 4) {}
    	else if(strncmp(line, "Charge-To:", 10) == 0)
    	    charge_to = TRUE;
        }

    close(fhandle);
    
    /*
    ** Compute the number of pages which will have been printed.  This
    ** is sort of an after the fact prediction.
    */
    rinfo->pages_printed = pages / N;
    if(copies >= 0) rinfo->pages_printed *= copies;

    /* 
    ** Compute how much the user has been charged if this job
    ** was printed.  If the responder message is to the effect
    ** that the job was submitted to an unknown destination 
    ** then prnid will be -1 and we can't compute the charge.
    */
    if(rinfo->response_code == RESP_FINISHED && rinfo->prnid >= 0)
	{
	compute_charge(&rinfo->charge, 
	    	printers[rinfo->prnid].charge_per_duplex,
		printers[rinfo->prnid].charge_per_simplex, pages, N, pagefactor,
		sigsheets, sigpart, copies);
	}
    else
    	{
    	rinfo->charge.total = 0;
    	}
    } /* end of respond_get_info() */

/*
** Build a message in final_str, basing it upon the code "response" and inserting
** particuliar information from the other arguments. 
*/
static void respond_build_message(char *response_str, struct RINFO *rinfo)
    {
    switch(rinfo->response_code)
        {
        case RESP_FINISHED:
            sprintf(response_str,
            "Your print job \"%s\" has been printed on \"%s\".",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode), rinfo->prnname);
            break;
        case RESP_ARRESTED:
            sprintf(response_str,
            "Your print job \"%s\" was arrested after an attempt\n"
            "to print it on \"%s\" resulted in a job error.",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode), rinfo->prnname);
            break;
        case RESP_CANCELED:
            sprintf(response_str,
            "Your print job \"%s\" has been canceled.",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode));
            break;
        case RESP_CANCELED_PRINTING:
            sprintf(response_str,
            "Your print job \"%s\" was "
            "canceled while printing on \"%s\".",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode), rinfo->prnname);
            break;
        case RESP_CANCELED_BADDEST:
            sprintf(response_str,
            "Your print job \"%s\" was canceled because\n"
            "\"%s\" is not a known destination.",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode), rinfo->destname);
            break;
        case RESP_CANCELED_REJECTING:
            sprintf(response_str,
            "Your print job \"%s\" was canceled because\n"
            "the destination \"%s\" is not acceping requests.",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode), rinfo->destname);
            break;
        case RESP_ARRESTED_PRINTER_INCAPABLE:
            sprintf(response_str,
            "Your print job \"%s\" was arrested because\n"
            "the printer \"%s\" is incapable of printing it.",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode), rinfo->prnname);
            break;
        case RESP_ARRESTED_GROUP_INCAPABLE:
            sprintf(response_str,
            "Your print job \"%s\" was arrested because no\n"
            "member of the group \"%s\" is capable of printing it.",
            local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode), rinfo->destname);
            break;
        default:
            sprintf(response_str,
            	"Undefined response code %d for your job \"%s\".",
            	rinfo->response_code, local_jobid(rinfo->destname,rinfo->id,rinfo->subid,rinfo->homenode));
        }
    } /* end of respond_build_message() */

/*
** Build a second message string which the responder can use
** if there is enough room.
*/
static void respond_build_message2(char *s, struct RINFO *rinfo)
    {
    s[0] = (char)NULL;
    } /* end of build_response_message2() */

/*
** Fork() and exec() a responder.
*/
static void respond_launch(struct RINFO *rinfo, const char *response_message, const char *response_message2)
    {
    pid_t pid;					/* Process id of responder */

    if((pid=fork()) == -1)			/* if error, */
        {
        error("can't fork() in respond_launch()");
        }
    else if(pid == 0)				/* if child, */
        {
	int fd;					/* fd opening job log and pprd log */
	char fname[MAX_PATH];	    	    	/* scratch space for file names */
	
	char code_in_ascii[3];
	char time_in_ascii[16];
	char numpages_in_ascii[7];
        char queue_id[100];	    	    	/* Actually too big (59 characters used). */
    
    	/* Build several parameter strings. */
	sprintf(code_in_ascii, "%d", rinfo->response_code);
	sprintf(time_in_ascii, "%ld", rinfo->Time); 
	if(rinfo->response_code == RESP_FINISHED && rinfo->pages_printed < 1000000)	/* 1,000,000 */
	    sprintf(numpages_in_ascii, "%d", rinfo->pages_printed);
	else
	    strcpy(numpages_in_ascii, "?");
        sprintf(queue_id, "%s %s %d %d %s", ppr_get_nodename(), rinfo->destname, rinfo->id, rinfo->subid, rinfo->homenode);

        /* 
	** Open the log file, or, if there is none, open /dev/null.
	** The idea is that the responder should find the log file 
	** on stdin. 
	*/
	sprintf(fname, DATADIR"/%s:%s-%d.%d(%s)-log", ppr_get_nodename(), rinfo->destname, rinfo->id, rinfo->subid, rinfo->homenode);
	if( (fd=open(fname, O_RDONLY)) == -1 )
	    {
	    if( (fd=open("/dev/null", O_RDONLY)) == -1 )
	    	fatal(1, "respond_launch(): child: can't open /dev/null, errno=%d (%s)", errno, strerror(errno) );
	    }

	/* If the handle we got was not stdin, make it stdin. */
        if(fd != 0)
            {
            dup2(fd, 0);
	    close(fd);
	    }

	/* Connect stdout and stderr to the pprd log file. */
	if( (fd = open(PPRD_LOGFILE, O_WRONLY | O_CREAT | O_APPEND, UNIX_644)) == -1)
	    fatal(1, "respond_launch(): child: can't open pprd log file, errno=%d (%s)", errno, strerror(errno) );
	if(fd != 1) dup2(fd, 1);
	if(fd != 2) dup2(fd, 2);
	if(fd > 2) close(fd);

	/*
	** Construct the path of the responder program.  This path
	** is probably relative to the ppr home directory.
	*/
        sprintf(fname, RESPONDERDIR"/%s", rinfo->response_method);

	/* Replace this child program with the responder. */
        execl(fname, rinfo->response_method,
        	rinfo->ForLine,			/* name of person job printed for */
        	rinfo->response_address,	/* address to send message to */
        	response_message,		/* suggested message text */
		response_message2,
		rinfo->responder_options,   	/* user specified responder options */
        	code_in_ascii,			/* code number of message type */
		queue_id,		    	/* full queue id of job */
		rinfo->prnname,			/* name of printer or text of error message */
        	rinfo->Title,			/* title of job */
        	time_in_ascii,			/* time job was submitted */
        	rinfo->Reason,			/* reason last arrested */
		numpages_in_ascii,		/* number of pages printed */
		rinfo->charge.total != 0 ? money(rinfo->charge.total) : "",
		(char*)NULL);

	/* Catch execl() failure: */
	DODEBUG_RESPOND(("respond_launch(): execl() failed, errno=%d (%s)", errno, strerror(errno)));
        _exit(242);
        }
    } /* end of respond_launch() */

/*
** Send a response to the user by means of the designated response method.
**
** This routine is sometimes called directly, at other times it is called
** by the wrapper routine respond().
**
** If there is no prnid, then use -1.
*/
void respond2(const char *destname, int id, int subid, const char *homenode, const char *prnname, int prnid, int response_code)
    {
    char response_message[256];			/* too big (none of the messages top 170 characters) */
    char response_message2[1024];
    struct RINFO rinfo;

    DODEBUG_RESPOND(("respond2(destname=\"%s\", id=%d, subid=%d, homenode=\"%s\", prnname=\"%s\", prnid=%d, response_code=%d)", destname, id, subid, prnname, prnid, response_code));

    /* Fill in some of rinfo, respond_get_info() will fill in the rest. */
    rinfo.destname = destname;
    rinfo.id = id;
    rinfo.subid = subid;
    rinfo.homenode = homenode;
    rinfo.response_code = response_code;
    rinfo.prnname = prnname;
    rinfo.prnid = prnid;

    /* Read the responding information from the queue file. */
    respond_get_info(&rinfo);
    
    /* If no response is wanted, then we are done. */
    if(strcmp(rinfo.response_method, "none") == 0)
        return;

    DODEBUG_RESPOND(("respond2(): method=\"%s\", address=\"%s\" options=\"%s\"", rinfo.response_method, rinfo.response_address, rinfo.responder_options));

    /* Construct the first message string and possibly re-wrap it. */
    respond_build_message(response_message, &rinfo);
    wrap_string(response_message, get_responder_width(rinfo.response_method));

    /* Do the same for the second message string: */
    respond_build_message2(response_message2, &rinfo);
    wrap_string(response_message2, get_responder_width(rinfo.response_method));

    /* Launch the responder. */
    respond_launch(&rinfo, response_message, response_message2);
    } /* end of respond2() */

/*
** This is the outer routine.  It takes a destination and a node id number and
** converts them to names before calling respond2() which does the real work.
**
** Sometimes the prnid will be -1, but get_dest_name() can deal with that.
*/
void respond(int destid, int id, int subid, int homenode_id, int prnid, int response)
    {
    DODEBUG_RESPOND(("respond(destid=%d, id=%d, subid=%d, homenode_id=%d, prnid=%d, response=\"%s\")", destid, id, subid, homenode_id, prnid, response));
    respond2(get_dest_name(destid), id, subid, nodeid_to_nodename(homenode_id), get_dest_name(prnid), prnid, response);
    }

/* end of file */
