/*
** ~ppr/src/pprd/pprd_ppopint.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 27 February 1997.
*/

/*
** This module handles requests from "ppop" and "ppad".
*/

#include "global_defines.h"
#include "global_structs.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include "pprd.h"
#include "pprd.auto_h"
#include "respond.h"
#include "util_exits.h"

long reply_pid;		/* <-- notice that this is not pid_t! */
FILE *reply_file;

/*
** Open the reply file.
*/
static int open_reply(void)
    {
    char reply_fname[MAX_PATH];

    sprintf(reply_fname, "%s/ppop%ld", TEMPDIR, reply_pid);
    if( (reply_file=fopen(reply_fname, "w")) == (FILE*)NULL )
        {
        error("open_reply(): can't open %s, errno=%d (%s)", reply_fname, errno, strerror(errno));
        return -1;
	}
	
    return 0;
    } /* end of open_reply() */
    
/*
** Set the user execute bit on the reply file.  This is done
** if the reply file is filled with data such as a queue 
** listing.
*/
static void set_reply_data(void)
    {
    fchmod(fileno(reply_file), S_IRUSR | S_IRUSR | S_IXOTH);
    } /* end of set_reply_data() */
    
/*
** Close the reply file and send SIGUSR1 to ppop
** so that it will know that the reply file is ready.
*/
static void close_reply(void)
    {
    fclose(reply_file);

    if( kill((pid_t)reply_pid, SIGUSR1) == -1 )
	{
	char reply_fname[MAX_PATH];
	debug("close_reply(): kill() failed, errno=%d (%s), deleting reply file", errno, strerror(errno));
	sprintf(reply_fname, "%s/ppop%ld", TEMPDIR, reply_pid);
	unlink(reply_fname);
	}
    } /* end of close_reply() */

/*=======================================================================
** List print jobs by writing them into a file.
** This if for the commands "ppop list", "ppop lpq", "ppop qquery", etc.
=======================================================================*/
void ppop_list(char *command)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    char *homenode = (char*)NULL;
    int destid;				/* Destination id to match */
    int id;				/* Queue id to match */
    int subid;				/* Queue sub id to match */
    int homenode_id;
    int x;
    char fname[MAX_PATH];
    int qfile;
    char buffer[1024];
    int len;
    struct stat statbuf;

    #ifdef DEBUG_PPOPINT
    debug("ppop_list(\"%s\")", command);
    #endif

    /*
    ** Pull the relevent information from the command we received.
    */
    if( ppr_sscanf(command,"l %ld %S %S %d %d %S",
		&reply_pid,
		&destnode,
		&destname,
		&id,
		&subid,
		&homenode) != 6 )
        {
        error("ppop_list(): invalid list command");
	if(destnode != (char*)NULL) myfree(destnode);
	if(destname != (char*)NULL) myfree(destname);
	if(homenode != (char*)NULL) myfree(homenode);
        return;
        }

    #ifdef DEBUG_PPOPINT
    debug("ppop_list(): destnode=\"%s\", destname=\"%s\", id=%d, subid=%d, homenode=\"%s\"", destnode, destname, id, subid, homenode);
    #endif

    /*
    ** If the destination node name is not the name of this
    ** node, then the command should have gone to RPPRD.
    */
    if( strcmp(destnode, ppr_get_nodename()) )
    	{
	error("ppop_list(): wrong destnode");
	myfree(destnode);
	myfree(destname);
	myfree(homenode);
        return;
    	}

    myfree(destnode);		/* that's all we wanted it for! */

    /*
    ** Open the temporary file into which
    ** we will write the queue listing.
    */
    if(open_reply())
	{
	myfree(destname);
	myfree(homenode);
        return;
        }

    /*
    ** Convert the destination (printer or group) name into an id 
    ** number.  The destination "all" is converted into the wildcard
    ** id number -1.
    */
    if( strcmp(destname, "all") == 0 || strcmp(destname, "any") == 0 )
	{				/* If wildcard name, use special destination id. */
	destid = -1;
	}
    else                                /* Otherwise, */
	{				/* get the destination id number. */
	if( (destid=get_dest_id(destname)) == -1 )
	    {
	    fprintf(reply_file, REPLY_BADDEST"Destination \"%s\" does not exist.\n", destname);
	    close_reply();
	    myfree(destname);
	    myfree(homenode);
            return;
            }
        }

    /* Convert the homenode name to an id number. */
    homenode_id = assign_nodeid(homenode);

    /* Free the memory blocks which ppr_sscanf() allocated
       since we have converted the strings to id numbers. */
    myfree(destname); myfree(homenode);

    DODEBUG_PPOPINT(("ppop_list(): destid=%d, id=%d, subid=%d, homenode_id=%d", destid, id, subid, homenode_id));

    lock();

    for(x=0; x<queue_entries; x++)
        {
	/*
	** If we are printing all queue entries, then we print this one,
	** otherwise, it must match the destid and it must match the id
	** and the subid if they are non-zero.
	*/
	if( (destid==-1 || queue[x].destid==destid)
        	   && (id==-1 || queue[x].id==id)
        	   && (subid==-1 || queue[x].subid==subid)
        	   && (homenode_id==-1 || queue[x].homenode_id==homenode_id) )
            {        
	    /* Open the queue file: */
	    sprintf(fname, QUEUEDIR"/%s:%s-%d.%d(%s)",
		ppr_get_nodename(),
		get_dest_name(queue[x].destid),
		queue[x].id,
		queue[x].subid,
		nodeid_to_nodename(queue[x].homenode_id));

	    if( (qfile = open(fname, O_RDONLY)) == -1 )
		fatal(0, __FILE__": ppop_list(): can't open \"%s\", errno=%d (%s)", fname, errno, strerror(errno) );
		
	    /* 
	    ** If the job was arrested, figure out when.  The date we
	    ** use is the date of the last inode change.
	    */
	    if(queue[x].status == STATUS_ARRESTED)
	    	{
		fstat(qfile, &statbuf);	    	
	    	}
	    else
	    	{
	    	statbuf.st_ctime = 0;
	    	}

	    /*
	    ** Print a line with the information from our job array and
	    ** when the job was arrested if it was.
	    */
	    fprintf(reply_file, "%s %s %d %d %s %d %d %s %d %d %d %ld\n",
	    	ppr_get_nodename(),
	    	get_dest_name(queue[x].destid),
	    	queue[x].id,
	    	queue[x].subid,
	    	nodeid_to_nodename(queue[x].homenode_id),
	    	queue[x].priority,
	    	queue[x].status,
	    	queue[x].status >= 0 ? get_dest_name(queue[x].status) : "?",
	    	queue[x].never,
	    	queue[x].notnow,
	    	queue[x].pass,
	    	(long)statbuf.st_ctime);

	    /*
	    ** Copy the queue file to the reply file and 
	    ** append a line with a single period to 
	    ** indicate the end of the reply file.
	    */
	    while( (len = read(qfile, buffer, sizeof(buffer))) > 0 )
	    	{
		if( fwrite(buffer, sizeof(char), len, reply_file) != len )
		    fatal(0, __FILE__": ppop_list(): fwrite() failed");
	    	}
	    	
	    if( len == -1 )
	    	fatal(0, __FILE__": ppop_list(): read() failed, errno=%d (%s)", errno, strerror(errno) );

	    close(qfile);
	    
	    fputs(QF_ENDTAG1, reply_file);
	    fputs(QF_ENDTAG2, reply_file);
	    }
	}

    unlock();			/* we are done with the queue */

    set_reply_data();		/* say there is a queue listing in the reply file */
    close_reply();		/* close reply file and signal ppop */

    free_nodeid(homenode_id);	/* very important to do */
    } /* end of ppop_list() */

/*============================================================================
** Return the status of one or more printers.
** This is acomplished with these two functions.
============================================================================*/
static void _ppop_status(FILE *outfile, int prnid)
    {
    int status = printers[prnid].status;

    /*
    ** The first line is a dump of the printers[] entry.
    */
    if(status == PRNSTATUS_PRINTING || status == PRNSTATUS_CANCELING
    		|| status == PRNSTATUS_STOPPING || status == PRNSTATUS_HALTING
    		|| status == PRNSTATUS_SEIZING)
	{
	fprintf(outfile,"%s %s %d %d %d %s %d %d %s\n",
		ppr_get_nodename(),				/* node printer is on (this node) */
		get_dest_name(prnid),				/* printer name */
		printers[prnid].status,				/* printer status */
		printers[prnid].next_error_retry + printers[prnid].next_engaged_retry,  /* retry number of next retry */
		printers[prnid].countdown,			/* seconds until next retry */
		get_dest_name(printers[prnid].jobdestid),	/* destination of job being printed */
		printers[prnid].id,				
		printers[prnid].subid,
		nodeid_to_nodename(printers[prnid].homenode_id));
	}
    else	/* not printing */
	{
	fprintf(outfile,"%s %s %d %d %d ? 0 0 ?\n",
		ppr_get_nodename(),
		get_dest_name(prnid),
		printers[prnid].status,
		printers[prnid].status == PRNSTATUS_ENGAGED ? printers[prnid].next_engaged_retry : printers[prnid].next_error_retry,
		printers[prnid].countdown);
	}

    /*
    ** The second line is the last message received from the
    ** printer.  This message is deposited in a file by pprdrv.
    */
    {
    char fname[MAX_PATH];
    int statusfile;
    char message[MAX_STATUS_MESSAGE+1];
    
    message[0] = (char)NULL;

    sprintf(fname, ALERTDIR"/%s.status", get_dest_name(prnid) );

    if( (statusfile=open(fname, O_RDONLY)) != -1 )
	{
	int readlen;
	readlen = read(statusfile, message, MAX_STATUS_MESSAGE);
	message[readlen >= 0 ? readlen : 0] = (char)NULL;
	close(statusfile);	    
	}

    fputs(message, outfile);
    fputc('\n', outfile);
    }
    } /* end of _ppop_status() */

void ppop_status(char *command)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    int x;
    int destid;

    #ifdef DEBUG_PPOPINT
    debug("ppop_status(\"%s\")", command);
    #endif

    if( ppr_sscanf(command,"s %ld %S %S", &reply_pid, &destnode, &destname) != 3 )
        {
        error("ppop_status(): invalid \"s\" command");
	if(destnode != (char*)NULL) myfree(destnode);
	if(destname != (char*)NULL) myfree(destname);
        return;
        }

    /*
    ** Make sure that the request is for the status of
    ** a local printer.  Status requests for remote
    ** printers should go to rpprd in stead.
    */
    if( strcmp(destnode, ppr_get_nodename()) )
    	{
	error("ppop_status(): incorrect destnode");
	myfree(destnode);
	myfree(destname);
	return;
    	}
    myfree(destnode);			/* no longer needed */

    if(open_reply())			/* Create reply file. */
	{				/* If it fails, */
	myfree(destname);
        return;
        }

    if( strcmp(destname, "all") == 0)		/* if "all", then step */
        {
        lock();
        for(x=0; x < printer_count; x++)	/* thru all the printers */
	    {
            if(printers[x].status != PRNSTATUS_DELETED)
            	_ppop_status(reply_file, x);
            set_reply_data();
            }
        unlock();
        }
    else if( (destid=get_dest_id(destname)) == -1 )
	{
	fprintf(reply_file, REPLY_BADDEST"The destination \"%s\" does not exist.\n", destname);
	}
    else				/* do just that destination */
	{				/* whose id was determined in the previous clause */
	if(isgroup(destid))		/* be it a group */
	    {
	    int x, y;
	    x = gindex(destid);		/* x is group index */
	    lock();
	    for(y=0; y < groups[x].members; y++)
		_ppop_status(reply_file, groups[x].printers[y]);
	    unlock();
	    }
	else				/* or a printer */
	    {
	    lock();
	    _ppop_status(reply_file, destid);
	    unlock();
	    }
	set_reply_data();		/* say there is data */
        }

    close_reply();

    myfree(destname);
    } /* end of ppop_printer_status() */

/* 
** Start a printer at the request of a user (action 0). 
** Stop a printer after the current job finishes (action 1).
** Stop a printer now (action 2).
** Stop the printer but don't signal ppop until it is done (action 129). 
*/
void ppop_startstop_printer(char *command, int action)
    {
    char *prnnode;
    char *prnname;
    int prnid;
    int mustwait = FALSE;

    DODEBUG_PPOPINT(("ppop_startstop_printer(\"%s\")", command));

    if( ppr_sscanf(command+1," %ld %S %S", &reply_pid, &prnnode, &prnname) != 3 )
        {
        error("ppop_startstop_printer(): invalid \"t\", \"p\", ,\"P\", or \"b\" command");
	if(prnnode!=(char*)NULL) myfree(prnnode);
	if(prnname!=(char*)NULL) myfree(prnname);
        return;
        }

    if( strcmp(prnnode,ppr_get_nodename()) )
    	{
    	error("ppop_startstop_printer(): bad prnnode");
    	myfree(prnnode);
    	myfree(prnname);
    	return;
    	}

    myfree(prnnode);		/* no longer needed */

    if(open_reply())		/* open the file for replying to ppop */
	{
	myfree(prnname);
        return;
        }
 
    /*
    ** Get the id number of the printer.  If there is a group
    ** with the same name, we want the id of the printer.
    */
    if( (prnid=reversed_get_dest_id(prnname)) == -1 )
        {
        fprintf(reply_file,
        	REPLY_BADDEST"The printer \"%s\" does not exist.\n",prnname);
        close_reply();
	myfree(prnname);
        return;
        }

    if( (prnid>=MAX_PRINTERS) ) 	/* If it is a group name, */
        {				/* complain. */
        fprintf(reply_file,
            REPLY_PRNONLY"\"%s\" is a group of printers, not a printer.\n",prnname);
        close_reply();
	myfree(prnname);
        return;
        }

    lock();				/* lock the printers database */

    if( (action&3) == 0 )		/* If should start the printer */
	{				/* (action 0), */
	switch(printers[prnid].status)	/* act according to current */ 
            {				/* printer state. */
	    case PRNSTATUS_IDLE:
            case PRNSTATUS_CANCELING:
            case PRNSTATUS_ENGAGED:
            case PRNSTATUS_STARVED:
		fprintf(reply_file, REPLY_ALREADY"\"%s\" is not stopt.\n", prnname);
		break;

	    case PRNSTATUS_PRINTING:
		fprintf(reply_file, REPLY_ALREADY"\"%s\" is already printing.\n", prnname);
		break;

	    case PRNSTATUS_FAULT:
	    case PRNSTATUS_STOPT:
		new_prn_status(&printers[prnid], PRNSTATUS_IDLE);
		startstop_update_waitreason(prnid);   
		look_for_work(prnid);
		break;

            case PRNSTATUS_HALTING:
		fprintf(reply_file, REPLY_INTERNAL"There is an outstanding halt command for \"%s\".\n", prnname);
		break;

            case PRNSTATUS_STOPPING:
		if(printers[prnid].ppop_pid)    /* If another ppop waiting for a stop command */
		    {                           /* to finish, let it wait no longer. */
		    kill(printers[prnid].ppop_pid, SIGUSR1);
                    printers[prnid].ppop_pid = (uid_t)0; /* (Unfortunately, it won't know it failed.) */
                    }

		new_prn_status(&printers[prnid], PRNSTATUS_PRINTING);
                startstop_update_waitreason(prnid);   
                break;

            default:
		fprintf(reply_file, REPLY_INTERNAL"spooler error, \"%s\" in an undefined state.\n", prnname);
		break;
            } /* end of switch() */
	}
    else			/* action = 1 or 2, stop or halt */
	{			/* stop may be with or without wait (128) */
	switch(printers[prnid].status)
	    {
	    case PRNSTATUS_FAULT:   /* If not printing now, */
	    case PRNSTATUS_IDLE:    /* we may go directly to stopt. */
	    case PRNSTATUS_ENGAGED:
	    case PRNSTATUS_STARVED:
		new_prn_status(&printers[prnid], PRNSTATUS_STOPT);
		startstop_update_waitreason(prnid);   
		break;

	    case PRNSTATUS_HALTING:  
		fprintf(reply_file, REPLY_ALREADY"\"%s\" is already halting.\n", prnname);
		break;

	    case PRNSTATUS_STOPT:    
		fprintf(reply_file, REPLY_ALREADY"\"%s\" is already stopt.\n", prnname);
		break;

	    case PRNSTATUS_CANCELING:		/* if pprdrv already sent kill signal */
		new_prn_status(&printers[prnid], PRNSTATUS_HALTING);
		startstop_update_waitreason(prnid);   
		mustwait = TRUE;		/* say we must wait */
		break;

	    case PRNSTATUS_STOPPING: 
		if( (action&3) == 1 )	/* if "ppop stop" */
		    {			/* we can't do better than "stopping" */
		    fprintf(reply_file, REPLY_ALREADY"\"%s\" is already stopping.\n", prnname);
		    break;
		    }			/* if halt, drop thru */

	    case PRNSTATUS_PRINTING:
		if( (action&3) == 2 )		/* if halt, not stop */
		    {				/* change state to halting */
		    new_prn_status(&printers[prnid], PRNSTATUS_HALTING);

		    DODEBUG_PPOPINT(("killing pprdrv (printer=%s, pid=%ld)", get_dest_name(prnid), (long)printers[prnid].pid));
		    if(kill(printers[prnid].pid, SIGTERM) == -1)
			{
			error("pprd_ppopint.c: ppop_startstop_printer(): kill(%ld, SIGTERM) failed, errno=%d (%s)",
			    	(long)printers[prnid].pid, errno, strerror(errno) );
			}
		    }
		else
		    {				/* if stop requested, arrange to stop */
		    new_prn_status(&printers[prnid], PRNSTATUS_STOPPING);
		    }				/* at the end of this job */
		startstop_update_waitreason(prnid);   
		mustwait = TRUE;		/* either way, we must wait */
		break;

	    default:                        
		fprintf(reply_file, REPLY_INTERNAL"spooler error, \"%s\" is in an undefined state.\n", prnname);
		break;
            }
	}

    if( (action&128) && mustwait )		/* If we are instructed to wait if */
	{					/* necessary and it is necessary, */
	if(printers[prnid].ppop_pid)
	    {					/* if someone else is */
	    fprintf(reply_file,			/* using notify, we can't */
		REPLY_CANTWAIT"Notify feature already in use for \"%s\".\n",prnname);
	    }
	else
	    {					/* rather, use notify */
	    fclose(reply_file);
	    printers[prnid].ppop_pid=(pid_t)reply_pid;	/* feature */
	    }
	}
    else					/* if no wait, signal now */
        {
	close_reply();
        }

    unlock();					/* allow other things to hapen again */

    myfree(prnname);
    } /* end of ppop_startstop_printer() */

/* 
** hold or release a print job
** action=0 means hold
** action=1 means release
*/
void ppop_holdrelease_job(char *command,int action)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    int id, subid;
    char *homenode = (char*)NULL;
    int destid;
    int homenode_id;

    #ifdef DEBUG_PPOPINT
    debug("ppop_holdrelease_job(\"%s\")", command);
    #endif

    if(ppr_sscanf(command+1," %ld %S %S %d %d %S",&reply_pid,&destnode,&destname,&id,&subid,&homenode) != 6 )
        {
        error("ppop_holdrelease_job(): invalid \"h\", or \"r\" command");
	if(destnode!=(char*)NULL) myfree(destnode);
	if(destname!=(char*)NULL) myfree(destname);
	if(homenode!=(char*)NULL) myfree(homenode);
        return;
        }

    if(strcmp(destnode,ppr_get_nodename()))
    	{
    	error("ppop_holdrelease_job(): bad destnode");
	myfree(destnode);
	myfree(destname);
	myfree(homenode);
	return;
	}

    myfree(destnode);			/* no longer needed */

    if(open_reply())
	{
	myfree(destname);
	myfree(homenode);
        return;
        }

    homenode_id = assign_nodeid(homenode);
    myfree(homenode);

    if( (destid=get_dest_id(destname)) == -1 )	/* if lookup fails, */
        {					/* say destination bad */
        fprintf(reply_file, REPLY_BADDEST"The destination \"%s\" does not exist.\n", destname);
        }
    else					/* otherwise, */
        {
        int x;

        lock();					/* lock the queue */

        for(x=0; x<queue_entries; x++)		/* and search it */
            {
            if( queue[x].destid==destid && queue[x].id==id
                && queue[x].subid==subid && queue[x].homenode_id==homenode_id )
                {
                if(action==0)           	/* if we should hold the job */
                    {
                    switch(queue[x].status)
                        {
                        case STATUS_WAITING:        /* if not printing, */
                        case STATUS_WAITING4MEDIA:  /* just quitely go to `hold' */
			    p_job_new_status(&queue[x], STATUS_HELD);
                            break;
                        case STATUS_HELD:           /* if already held, say so */
                            fprintf(reply_file,
                                REPLY_ALREADY"\"%s\" is already held.\n",
                                local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
                            break;
                        case STATUS_ARRESTED:       /* can't hold an */
                            fprintf(reply_file,     /* arrested job */
                                REPLY_ALREADY"\"%s\" is arrested.\n",
                                local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
                            break;
			case STATUS_SEIZING:		/* already going to held */
			    fprintf(reply_file,
			    	REPLY_ALREADY"\"%s\" is already in transition to the held state.\n",
                                local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
                            break;
			case STATUS_CANCEL:		/* if being canceled, hijack the operation */
			    fprintf(reply_file,
			    	REPLY_OK"Converting cancel to hold for job \"%s\".\n",
                                local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
			    new_prn_status(&printers[queue[x].status], PRNSTATUS_SEIZING);
			    printers[queue[x].status].cancel_job = FALSE;
			    p_job_new_status(&queue[x], STATUS_SEIZING);
			    break;
                        default:			/* printing? */
			    if(queue[x].status >= 0)	
				{
				int prnid = queue[x].status;

				fprintf(reply_file,
					REPLY_OK"Seizing job \"%s\" which is printing on \"%s\".\n",
                                	local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)),
                                	get_dest_name(prnid));

				new_prn_status(&printers[prnid], PRNSTATUS_SEIZING);
				p_job_new_status(&queue[x], STATUS_SEIZING);

				DODEBUG_PPOPINT(("killing pprdrv (printer=%s, pid=%ld)", get_dest_name(prnid), (long)printers[prnid].pid));
				if(kill(printers[prnid].pid, SIGTERM) == -1)
				    {
				    error("pprd_ppopint.c: ppop_holdrelease_job(): kill(%ld, SIGTERM) failed, errno=%d (%s)", 
					(long)printers[prnid].pid, errno, strerror(errno) );
				    }
			    	}	
			    else
				{
				fprintf(reply_file,
					REPLY_INTERNAL"\"%s\" has unknown status %d.\n",
                                	local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)),
                                	queue[x].status);
				}
                            break;
                        }
                    }
                else                    	/* action: release a job */
                    {
                    switch(queue[x].status)
                        {
                        case STATUS_HELD:       /* "held" or "arrested" jobs */
                        case STATUS_ARRESTED:   /* may be made "waiting" */
                            set_notnow_for_job(p_job_new_status(&queue[x], STATUS_WAITING), TRUE);
                            start_suitable_printer(&queue[x]);
                            break;
			case STATUS_SEIZING:
			    fprintf(reply_file,
			    	REPLY_ALREADY"\"%s\" can't be released until the hold operation finishes.\n",
                                local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
			    break;
			case STATUS_WAITING:
			case STATUS_WAITING4MEDIA:
			default:			/* printing */
                            fprintf(reply_file,
                                REPLY_ALREADY"\"%s\" is not held.\n",
                                local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
                            break;
                        }
                    }
                break;
                }
            }
        unlock();
        if(x==queue_entries)
            fprintf(reply_file, REPLY_BADJOB"Job \"%s\" does not exist.\n",local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
        }

    close_reply();
    
    myfree(destname);
    free_nodeid(homenode_id);
    } /* end of ppop_holdrelease_job() */

/* 
** Cancel a job or jobs 
*/
void ppop_cancel_or_purge(char *command)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;	/* name of destination */
    int id, subid;			/* queue id number */
    char *homenode = (char*)NULL;
    int inform;				/* should the user be notified? */
    int destid;				/* id number of destination */            
    int homenode_id;
    int prnid;				/* temporary printer id */
    int canceled_count = 0;		/* number of jobs canceled */

    DODEBUG_PPOPINT(("ppop_cancel_or_purge(\"%s\")", command));

    if(ppr_sscanf(command, "c %ld %S %S %d %d %S %d", &reply_pid, &destnode, &destname, &id, &subid, &homenode, &inform) != 7 )
        {
        error("ppop_cancel(): invalid command: %s", command);
        if(destnode != (char*)NULL) myfree(destnode);
        if(destname != (char*)NULL) myfree(destname);
	if(homenode != (char*)NULL) myfree(homenode);
        return;
        }

    if( strcmp(destnode, ppr_get_nodename()) )
    	{
    	error("ppop_cancel(): bad destnode");
    	myfree(destnode);
    	myfree(destname);
    	myfree(homenode);
    	}

    myfree(destnode);		/* no longer needed */

    if(open_reply())		/* open file to ppop */
	{			/* if fails, */
	myfree(destname);
	myfree(homenode);
        return;
        }

    homenode_id = assign_nodeid(homenode);
    myfree(homenode);

    if( (destid=get_dest_id(destname)) == -1 && strcmp(destname, "all") && strcmp(destname, "any") )
        {
        fprintf(reply_file, REPLY_BADDEST"The destination \"%s\" does not exist.\n", destname);
        }
    else                        /* printer or group exists */
        {
        int x;			/* index into queue array */

	DODEBUG_PPOPINT(("ppop_cancel(): canceling jobs matching destid=%d, id=%d, subid=%d, homenode_id=%d", destid, id, subid, homenode_id));

        lock();
        for(x=0; x < queue_entries; x++)	/* search the whole */
            {					/* queue */
            if( (destid == -1 || queue[x].destid == destid)
			&& (id == -1 || queue[x].id == id)
			&& (subid == -1 || queue[x].subid == subid)
			&& (homenode_id == -1 || queue[x].homenode_id == homenode_id) )
                {
                canceled_count++;

		/* If the job is being printed, */
                if( (prnid=queue[x].status) >= 0 )
		    {
		    /* If it is printing we can say it is now canceling, but
		       if it is halting or stopping we don't want to mess with
		       that at new_prn_status() will force the state to stopt.
		       */
		    if(printers[prnid].status == PRNSTATUS_PRINTING)
			new_prn_status(&printers[prnid], PRNSTATUS_CANCELING);

                    printers[prnid].cancel_job = TRUE;	/* and say job should die when it dies. */

		    p_job_new_status(&queue[x], STATUS_CANCEL);

		    DODEBUG_PPOPINT(("Killing pprdrv process %ld", (long)printers[prnid].pid));
		    if(kill(printers[prnid].pid, SIGTERM) == -1)
			{
			error("pprd_ppopint.c: ppop_cancel_or_purge(): kill(%ld, SIGTERM) failed, errno=%d (%s)", 
				(long)printers[prnid].pid, errno, strerror(errno) );
			}
                    }

		/* Not being printed */
                else				
                    {
		    /* 
		    ** If the user should be informed and this job is not arrested,
		    ** use the responder to inform the user that we are canceling it.
		    */
		    if(inform && queue[x].status != STATUS_ARRESTED)
			{
	                respond(queue[x].destid, queue[x].id, queue[x].subid,
				queue[x].homenode_id,
				-1,   /* impossible printer */
				RESP_CANCELED);
			}
		    /* 
		    ** Remove the job files (we must do this after responding)
		    */
		    unlink_job(queue[x].destid, queue[x].id, queue[x].subid, queue[x].homenode_id);

		    /*
		    ** We may not remove it from the queue array
		    */
		    dequeue_job(queue[x].destid, queue[x].id, queue[x].subid, queue[x].homenode_id);

		    x--;        /* compensate for deletion */
		    }
		}
	    }
	unlock();

	if( canceled_count==0 && id!=-1 && subid!=-1 ) /* if no match, no wildcard, */
            {
            fprintf(reply_file, REPLY_BADJOB"The print job \"%s\" does not exist.\n", local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)) );
            }
	else
	    {
	    fprintf(reply_file, REPLY_OK"%d", canceled_count);
	    set_reply_data();
	    }
        } /* end if else printer or group exists */

    close_reply();
    
    myfree(destname);

    free_nodeid(homenode_id);
    } /* end of ppop_cancel() */

/*
** List media currently mounted on a printer or printers.
*/
void ppop_media(char *command)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    struct fcommand1 f1;
    struct fcommand2 f2;
    int x, y;
    int all = FALSE;
    int group = FALSE;
    #ifdef GNUC_HAPPY
    int destid=0;	/* zero pleases GNU-C (otherwise destid is almost used undefined) */
    #else
    int destid;
    #endif

    #ifdef DEBUG_PPOPINT
    debug("ppop_media(\"%s\")", command);
    #endif

    if( ppr_sscanf(command, "f %ld %S %S", &reply_pid, &destnode, &destname) != 3 )
        {
        error("ppop_media(): invalid \"f\" command");
	if(destnode!=(char*)NULL) myfree(destnode);
	if(destname!=(char*)NULL) myfree(destname);
        return;
        }

    /* The node name must be our node name. */
    if( strcmp(destnode, ppr_get_nodename()) )
    	{
	error("ppop_media(): bad destnode");
	myfree(destnode);
	myfree(destname);
	return;    	
    	}

    myfree(destnode);			/* no longer needed */

    if(open_reply())			/* if we can't open the reply file, */
	{
	myfree(destname);
        return;
        }

    /* set all, group, and destid */
    if( strcmp(destname, "all") == 0 )
	{
        all = TRUE;
        }
    else
	{
        if( (destid=get_dest_id(destname)) == -1 )
            {
            fprintf(reply_file,REPLY_BADDEST"The destination \"%s\" does not exist.\n",destname);
            close_reply();
	    myfree(destname);
            return;
            }
        if(isgroup(destid))
            group = TRUE;
	}

    myfree(destname);
    
    for(x=0; x < printer_count; x++)
        {
        if( all || (x==destid) || (group && get_member_offset(destid,x)!=-1) )
            {
            f1.nbins = printers[x].nbins;
            strcpy(f1.prnname, printers[x].name);
            fwrite(&f1, sizeof(struct fcommand1), 1, reply_file);
            for(y=0; y < printers[x].nbins; y++)
                {
                #ifdef DEBUG_MEDIA
                debug("x=%d, y=%d, media=%d",x,y,printers[x].media[y]);
                #endif
                strcpy(f2.bin, get_bin_name(printers[x].bins[y]) );
                strcpy(f2.media, get_media_name(printers[x].media[y]));
                fwrite(&f2, sizeof(struct fcommand2), 1, reply_file);
                }
            }
        } /* end of for(x....) */

    set_reply_data();
    close_reply();
    } /* end of ppop_media() */

/*
** Mount new media.
*/
void ppop_mount(char *command)
    {
    char *prnnode = (char*)NULL;
    char *printer = (char*)NULL;
    char binname[MAX_BINNAME+1];
    char medianame[MAX_MEDIANAME+1];
    int binid;                      /* bin id number of specified bin */
    int prnid;                      /* id number of printer to change */
    int x;

    #ifdef DEBUG_PPOPINT
    debug("ppop_mount(\"%s\")", command);
    #endif

    /* Parse the command we received over the pipe. */
    if(ppr_sscanf(command,"M %ld %S %S %#s %#s",
			&reply_pid,
			&prnnode,
			&printer,
			sizeof(binname),binname,
			sizeof(medianame),medianame) != 5 )
	{
	error("ppop_mount(): invalid \"M\" command");
	if(prnnode!=(char*)NULL) myfree(prnnode);
	if(printer!=(char*)NULL) myfree(printer);
	return;  
	}

    if( strcmp(prnnode,ppr_get_nodename()) )
    	{
	error("ppop_mount(): bad prnnode");
    	myfree(prnnode);
    	myfree(printer);
    	return;
    	}

    myfree(prnnode);

    /* Open the communications file to which we will write our output. */
    if(open_reply())
	{
	myfree(printer);
        return;
        }

    /*
    ** Get the id of the printer.  If there is a group with the
    ** same name, we want the id of the printer, not of the group.
    */
    if( (prnid=reversed_get_dest_id(printer))==-1 )
        {
        fprintf(reply_file,REPLY_BADDEST"Printer \"%s\" does not exist.\n",printer);
	close_reply();
	myfree(printer);
        return;
        }

    if( isgroup(prnid) )
        {
        fprintf(reply_file,REPLY_PRNONLY"\"%s\" is not a printer.\n",printer);
	close_reply();
	myfree(printer);
        return;
        }

    binid = get_bin_id(binname);

    x=0;                                    /* find the bin in question */
    while( printers[prnid].bins[x] != binid )
        {
        if(x==printers[prnid].nbins)        /* if bin not found */
            {
            fprintf(reply_file,
                REPLY_BADBIN"\"%s\" does not have a bin called \"%s\".\n",printer,binname);
	    close_reply();
	    myfree(printer);
            return;        
            }    
        x++;
        } 

    printers[prnid].media[x] = get_media_id(medianame);

    close_reply();
    
    update_notnow(prnid);           /* update list of printable jobs */
    if(printers[prnid].status==PRNSTATUS_IDLE) /* if printer idle */
        look_for_work(prnid);       /* see if anything to do (needn't lock) */

    save_medialist(prnid);          /* save for restart */

    /* Let the queue display programs know about the change. */
    state_update("MOUNT %s %s %s",printer,binname,medianame);

    myfree(printer);
    } /* end of ppop_mount() */

/*
** Set a printer or group to accept or reject print jobs.
** action=0 means reject
** action=1 means accept
*/
void ppop_accept_reject(char *command, int action)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    int destid;
    char conffile[MAX_PATH];
    struct stat st;

    #ifdef DEBUG_PPOPINT
    debug("ppop_mount(command=\"%s\", action=%d)", command, action);
    #endif

    if(ppr_sscanf(&command[1]," %ld %S %S",&reply_pid,&destnode,&destname) != 3)
        {
        error("ppop_accept_reject(): invalid \"A\" or \"R\" command");
	if(destnode!=(char*)NULL) myfree(destnode);
	if(destname!=(char*)NULL) myfree(destname);
        return;
        } 

    if( strcmp(destnode,ppr_get_nodename()) )
    	{
    	error("ppop_accept_reject(): bad destnode");
    	myfree(destnode);
    	myfree(destname);
    	return;
    	}

    myfree(destnode);

    if(open_reply())
	{
	myfree(destname);
        return;
        }
   
    /*
    ** If there is both a printer and a group with the specified
    ** name, we will operate on the group.  This is probably the
    ** best thing to do because ppr(1) will attempt to submit to
    ** the group, not the printer.
    */
    if( (destid=get_dest_id(destname)) == -1 )
        {
        fprintf(reply_file,
	    REPLY_BADDEST"\"%s\" is not a valid destination.\n",destname);
        close_reply();
	myfree(destname);
        return;
        }

    if(!isgroup(destid))
        {                               /* printer: */
        printers[destid].accepting=action;
        sprintf(conffile,"%s/%s",PRCONF,destname);
        stat(conffile,&st);             /* modify group execute bit */
        }
    else                                /* group: */
        {
        groups[gindex(destid)].accepting=action;
        sprintf(conffile,"%s/%s",GRCONF,destname);
        stat(conffile,&st);             /* modify group execute bit */
        }

    if(action==0)                       /* reject */
        chmod(conffile, st.st_mode | S_IXGRP );
    else                                /* accept */
        chmod(conffile, st.st_mode & (0777 ^ S_IXGRP) );

    close_reply();			/* close communications file (empty) */

    myfree(destname);
    } /* end of ppop_accept_reject() */

/*
** Show destinations, their type, and whether or not they are accepting.
*/
void _show_printer(FILE *outfile, int prnid)
    {
    fprintf(outfile,"%s 0 %d %d\n",printers[prnid].name,printers[prnid].accepting,printers[prnid].protect);
    }

void _show_group(FILE *outfile, int groupnum)
    {
    fprintf(outfile,"%s 1 %d %d\n",groups[groupnum].name,groups[groupnum].accepting,groups[groupnum].protect);
    }

void ppop_dest(char *command)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    int destid;
    int x;

    #ifdef DEBUG_PPOPINT
    debug("ppop_dest(\"%s\")", command);
    #endif

    if(ppr_sscanf(&command[1]," %ld %S %S",&reply_pid,&destnode,&destname) != 3)
        {
        error("ppop_show_destinations(): invalid \"D\" command");
        if(destnode!=(char*)NULL) myfree(destnode);
        if(destname!=(char*)NULL) myfree(destname);
        return;
        } 

    if( strcmp(destnode,ppr_get_nodename()) )
    	{
    	error("ppop_show_destinations(): bad destnode");
    	myfree(destnode);
    	myfree(destname);
	return;
    	}

    myfree(destnode);

    if(open_reply())			/* open the response file */
	{
	myfree(destname);
        return;
        }

    if(strcmp(destname, "all")==0)
        {
        for(x=0;x<printer_count;x++)
	    {
	    if(printers[x].status!=PRNSTATUS_DELETED)
                _show_printer(reply_file,x);
            }
        for(x=0;x<group_count;x++)
            {
	    if( ! groups[x].deleted )
		_show_group(reply_file,x);
            }
        }
    else	/* a specific destination, */
        {
        if( (destid=get_dest_id(destname)) == -1 )
            {
            fprintf(reply_file, REPLY_BADDEST"\"%s\" is not a valid destination.\n", destname);
	    close_reply();
	    myfree(destname);
            return;
            }

	if(isgroup(destid))
	    _show_group(reply_file, gindex(destid));
	else
	    _show_printer(reply_file, destid);
        }

    set_reply_data();
    close_reply();
    
    myfree(destname);
    } /* end of ppop_show_destinations() */

/*
** Move jobs from one queue to another.
*/
void ppop_move(char *command)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    int id,subid;
    char *homenode = (char*)NULL;
    int homenode_id;
    char *newdestnode = (char*)NULL;
    char *newdestname = (char*)NULL;
    char oldname[MAX_PATH];
    char newname[MAX_PATH];
    int x;
    int moved=0;                        /* count of files moved */
    int printing=0;                     /* not moved because printing */
    FILE *logfile;                      /* used to write to log */
    int rank2;				/* rank amoung jobs for new destination */
    int real_move;

    int destid;
    int newdestid;

    #ifdef DEBUG_PPOPINT
    debug("ppop_move(\"%s\")", command);
    #endif

    if(ppr_sscanf(command,"m %ld %S %S %d %d %S %S %S",&reply_pid,&destnode,&destname,&id,&subid,&homenode,&newdestnode,&newdestname) != 8)
        {
        error("ppop_move(): invalid \"m\" command");
	if(destnode!=(char*)NULL) myfree(destnode);
	if(destname!=(char*)NULL) myfree(destname);
	if(homenode!=(char*)NULL) myfree(homenode);
	if(newdestnode!=(char*)NULL) myfree(newdestnode);
	if(newdestname!=(char*)NULL) myfree(newdestname);
        return;
        } 

    /* Make sure we are not being asked to do something rpprd ought to do. */
    if( strcmp(destnode, ppr_get_nodename()) || strcmp(newdestnode, ppr_get_nodename()) )
	{
	error("ppop_move(): destnode or newdestnode not this node");
	myfree(destnode);
	myfree(destname);
	myfree(homenode);
	myfree(newdestnode);
	myfree(newdestname);
	return;
	}

    /* Being satisfied, we don't need these anymore. */
    myfree(destnode);
    myfree(newdestnode);

    /* Try to open the reply file.
       If it doesn't work, free those
       memory blocks. */
    if(open_reply())
	{
	myfree(destname);
	myfree(newdestname);
	myfree(homenode);
        return;
        }

    if( (destid = get_dest_id(destname)) == -1 )
        {                               /* get id of source */
        fprintf(reply_file,
	    REPLY_BADDEST"No printer or group is called \"%s\".\n",destname);
	close_reply();
	myfree(destname);
	myfree(newdestname);
	myfree(homenode);
        return;
        }

    if( (newdestid = get_dest_id(newdestname)) == -1 )
        {                               /* get id of destination */         
        fprintf(reply_file,
            REPLY_BADDEST"No printer or group is called \"%s\".\n",newdestname);
	close_reply();
	myfree(destname);
	myfree(newdestname);
	myfree(homenode);
        return;
        }

    homenode_id = assign_nodeid(homenode);
    myfree(homenode);

    /*
    ** Set real_move to non-zero if the source and destination
    ** queues are different.  (It is legal to move a job to
    ** its own queue.  It resets the never mask.)
    */
    real_move = strcmp(destname,newdestname);

    lock();				/* lock the queue array */

    for( rank2=x=0; x < queue_entries; x++ )
        {
        if( (queue[x].destid==destid)	/* if match, */
                && ( id==-1 || queue[x].id==id )
                && ( subid==-1 || queue[x].subid==subid ) 
                && ( homenode_id==-1 || queue[x].homenode_id==homenode_id ) )
            {
            if(queue[x].status>=0)          /* If it is printing, we */
                {                           /* can't move it. */
                printing++;
                continue;
                }

	    /* Inform queue monitoring programs of the move. */
	    if(real_move)
		state_update("MOV %s %s %d",local_jobid(destname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id)), newdestname, rank2++);

            /* change the destination id in the queue array */
            queue[x].destid = newdestid;
                                       
            /* Rename all the queue files */
	    sprintf(oldname,"%s/%s:%s-%d.%d(%s)",
                QUEUEDIR,ppr_get_nodename(),destname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            sprintf(newname,"%s/%s:%s-%d.%d(%s)",
                QUEUEDIR,ppr_get_nodename(),newdestname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            rename(oldname,newname);

            sprintf(oldname,"%s/%s:%s-%d.%d(%s)-comments",
                DATADIR,ppr_get_nodename(),destname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            sprintf(newname,"%s/%s:%s-%d.%d(%s)-comments",
                DATADIR,ppr_get_nodename(),newdestname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            rename(oldname,newname);

            sprintf(oldname,"%s/%s:%s-%d.%d(%s)-pages",
                DATADIR,ppr_get_nodename(),destname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            sprintf(newname,"%s/%s:%s-%d.%d(%s)-pages",
                DATADIR,ppr_get_nodename(),newdestname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            rename(oldname,newname);

            sprintf(oldname,"%s/%s:%s-%d.%d(%s)-text",
                DATADIR,ppr_get_nodename(),destname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            sprintf(newname,"%s/%s:%s-%d.%d(%s)-text",
                DATADIR,ppr_get_nodename(),newdestname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
	    rename(oldname,newname);

	    /* this one may not be present */
            sprintf(oldname,"%s/%s:%s-%d.%d(%s)-log",
                DATADIR,ppr_get_nodename(),destname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            sprintf(newname,"%s/%s:%s-%d.%d(%s)-log",
                DATADIR,ppr_get_nodename(),newdestname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id));
            rename(oldname,newname);

            /*
            ** In the job's log file, make a note of the fact
            ** that it was moved from one destination to another.
            */
            if( (logfile=fopen(newname,"a")) != (FILE*)NULL )
                {
                fprintf(logfile,
                    "Job moved from destination \"%s\" to \"%s\".\n",
                    destname,newdestname);           
                fclose(logfile);
                }

            queue[x].never = 0;				/* clear any never flags */
            set_notnow_for_job(&queue[x],TRUE);		/* not truly new, but ... */
            if(queue[x].status==STATUS_WAITING)		/* if not held, etc. */
                start_suitable_printer(&queue[x]); 	/* try to start it */ 

            moved++;				/* increment count */
            }

	/*
	** Not a job to be moved, but do we have to note it as a job
	** that is already in the destination queue?
	*/
	else if( queue[x].destid == newdestid )
	    {
	    rank2++;
	    }
        }

    unlock(); 			/* we are done modifying the queue array */

    switch(moved)               /* tell how many jobs where moved */
        {
        case 0:
            if(id == -1)	/* if moving all files on destination, */
		{			     
                fprintf(reply_file,
                    REPLY_BADJOB"No jobs are queued for \"%s\".\n",destname);
                }
            else
                {
                if(printing==0) /* suppress if we already said it was */
                    {           /* printing */
		    fprintf(reply_file,REPLY_BADJOB"Job \"%s\" does not exist.\n",local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));
                    }
                }
            break;
        case 1:
            fprintf(reply_file,REPLY_OK"1 file was moved.\n");
            break;
        default:  
            fprintf(reply_file,REPLY_OK"%d files where moved.\n",moved);
        }

    if(printing)
        fprintf(reply_file,"1 file was not moved because it was printing.\n");

    close_reply();

    myfree(destname);
    myfree(newdestname);
    free_nodeid(homenode_id);
    } /* end of ppop_move() */

/*
** Rush a job.
*/
void ppop_rush(char *command)
    {
    char *destnode = (char*)NULL;
    char *destname = (char*)NULL;
    int id,subid;
    char *homenode = (char*)NULL;
    int homenode_id;
    int destid;
    struct QEntry t;                    /* temp storage for queue entry */
    int x;                              

    #ifdef DEBUG_PPOPINT
    debug("ppop_rush(\"%s\")", command);
    #endif

    /* Parse the command we received over the pipe. */
    if(ppr_sscanf(command,"U %ld %S %S %d %d %S",&reply_pid,&destnode,&destname,&id,&subid,&homenode) != 6)
        {
        error("ppop_rush(): invalid \"U\" command");
	if(destnode!=(char*)NULL) myfree(destnode);
	if(destname!=(char*)NULL) myfree(destname);
	if(homenode!=(char*)NULL) myfree(homenode);
        return;
        } 

    if( strcmp(destnode,ppr_get_nodename()) )
    	{
    	error("ppop_rush(): bad destnode");
    	myfree(destnode);
    	myfree(destname);
    	myfree(homenode);
    	return;
    	}

    myfree(destnode);
    
    /* Using the name parsed from the command, open a response file. */
    if(open_reply())
	{
	myfree(destname);
        return;
        }

    /*
    ** Look up the destination id for the destination name.
    ** If the lookup fails it return -1 indicating that the
    ** destination in question does not exist.
    */
    if( (destid=get_dest_id(destname)) == -1 )
        {
        fprintf(reply_file,
            REPLY_BADDEST"No printer or group is called \"%s\".\n",destname);
	close_reply();
	myfree(destname);
	myfree(homenode);
        return;
        }

    /* Get a node id for the homenode name and free the name. */
    homenode_id = assign_nodeid(homenode);
    myfree(homenode);

    lock();                             /* exclusive right to modify queue */

    for(x=0; x<queue_entries; x++)	/* Examine the whole */
        {                               /* queue if we must. */
        if( queue[x].destid==destid     /* If we have a match, */ 
                && queue[x].id==id 
                && queue[x].subid==subid && queue[x].homenode_id==homenode_id )
            {                           /* save the matching entry, */
	    /* Inform queue display programs. */
	    state_update("RSH %s", local_jobid(destname,queue[x].id,queue[x].subid,nodeid_to_nodename(queue[x].homenode_id)));

            memcpy(&t,&queue[x],sizeof(struct QEntry));
            while(x)                    /* and slide the others up. */
                {
                memcpy(&queue[x],&queue[x-1],sizeof(struct QEntry));
                x--;
                }
            memcpy(&queue[0],&t,sizeof(struct QEntry));
            queue[0].priority = 0;	/* highest priority */
            break;                           
            }
        }

    unlock();                           /* done with queue */

    if(x==queue_entries)                /* if ran to end with no match */
        fprintf(reply_file,REPLY_BADJOB"Queue entry \"%s\" does not exist.\n",local_jobid(destname,id,subid,nodeid_to_nodename(homenode_id)));

    close_reply();

    myfree(destname);
    free_nodeid(homenode_id);
    } /* end of ppop_rush() */

/* end of file */
