/*
** ~ppr/src/pprd/pprd.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 27 February 1997.
*/

/*
** A PostScript Print Spooler Daemon written by David Chappell
** at Trinity College.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <dirent.h>
#include <sys/wait.h>
#include <errno.h>
#include <ctype.h>
#include "pprd.h"
#include "respond.h"
#include "pprd.auto_h"
#include "interface.h"

/* Exit codes for errors in library routines. */
const int lib_memory_fatal = ERROR_DUMPCORE;
const int lib_misc_fatal = ERROR_DUMPCORE;

/*
** Misc global variables
*/
struct QEntry *queue;		/* array holding terse queue */
int queue_size;			/* number of entries for which there is room */
int queue_entries = 0;		/* entries currently used */
struct Printer *printers;	/* array of printer description structures */
int printer_count = 0;		/* how many printers do we have? */
struct Group *groups;		/* array of group structures */
int group_count = 0;		/* how many groups? */
int upgrade_countdown = UPGRADE_INTERVAL;
int active_printers = 0;	/* number of printers currently active */
int starving_printers = 0;	/* printers currently waiting for rations */

/*
** questionable global variables 
*/
#define MAX_MEDIAS 20		/* media name to id translate table */
char *media[MAX_MEDIAS];	/* up to twenty media */
int media_count = 0;		/* how many do we have now? */

#define MAX_BINNAMES 10		/* bin name to id translate table */
char *binname[MAX_BINNAMES];
int binnames_count = 0;

/*==========================================================================
** General Utility Routines
**
** These functions ought to be safe to call from within a signal handler
** because they do not use the standard C I/O library which might
** call malloc() or not be reentrant for some other reason.
**
** Notice that writing them this way required the use of a fixed length
** buffer in which to build the string.  That means that restraint must
** be exercised as to the length of the arguments passed to these functions.
**
** Also notice that if an error prevents us from writing to the log file
** then we simply ignore it.  This is because there is little we could
** do even if we detected it.
==========================================================================*/

/*
** Print an error message and abort.
*/
void fatal(int exitval, const char *string, ... )
    {
    int file;

    /*
    ** Free at least one file handle.
    ** If we don't do this then we will not be able
    ** to print an error message is some operation 
    ** fails because we have run out of file
    ** handles.
    */
    close(3);

    /*
    ** If we can or open or create the log file, append a line to it.
    */
    if( (file=open(PPRD_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
        {
	va_list va;
	char line[256];
        sprintf(line, "FATAL: (%s) ", datestamp() );
	va_start(va, string);
        vsprintf(&line[strlen(line)], string, va);
	va_end(va);
	strcat(line, "\n");
	write(file, line, strlen(line));
        close(file);
        }

    /*
    ** If we have been passed the magic exit value, then try to 
    ** dump core.  We do two kill()s because for some mysterious
    ** reason SunOS 5.5.1 cron launches processes which can't 
    ** receive SIGQUIT.  Possibly it is because they don't 
    ** have a terminal.
    */
    if(exitval == ERROR_DUMPCORE)
	{
    	kill(getpid(), SIGQUIT);
    	kill(getpid(), SIGABRT);

	if( (file=open(PPRD_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
	    {
	    write(file, "Suicide failed, no core file created.\n", 38);
	    close(file);
	    }
    	}

    /* Remove the lock file which also has our PID in it. */
    unlink(PPRD_LOCKFILE);
    
    exit(exitval);
    } /* end of fatal() */

/*
** Print a log line classifying it as an error.
*/
void error(const char *string, ... )
    {
    int file;

    if( (file=open(PPRD_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
        {
	va_list va;
	char line[256];
        sprintf(line, "ERROR: (%s) ", datestamp() );
	va_start(va,string);
        vsprintf(&line[strlen(line)], string, va);
	va_end(va);
	strcat(line, "\n");
	write(file, line, strlen(line));
        close(file);
        }
    } /* end of error() */

/*
** Print a log line, classifying it as a debug line.
*/
void debug(const char *string, ... )
    {
    int file;

    if( (file=open(PPRD_LOGFILE, O_WRONLY | O_APPEND | O_CREAT, UNIX_644)) != -1 )
        {
	va_list va;
	char templine[256];
        sprintf(templine, "DEBUG: (%s) ", datestamp() );
	va_start(va, string);
        vsprintf(&templine[strlen(templine)], string, va);
	va_end(va);
	strcat(templine, "\n");
	write(file, templine, strlen(templine));
        close(file);
        }
    } /* end of debug() */

/*=========================================================================
** Send a line to the file which is read by programs which display
** some aspect of the spooler state.
**
** This routine is sometimes called from within a signal handler, so
** it should not call non-reentrant routines such as malloc().
=========================================================================*/
void state_update(const char *string, ... )
    {
    static int countdown = 0;	/* countdown til start of next file */
    static int handle = -1;	/* handle of open file */
    static int serial = 1;	/* file serial number */
    va_list va;
    char line[128];

    while( handle == -1 || countdown <= 0 )
    	{
    	if(handle == -1)
    	    {
    	    if( (handle = open(STATE_UPDATE_FILE, O_WRONLY | O_CREAT | O_APPEND, UNIX_755)) == -1 )
    	    	fatal(0,"Failed to open %s for append, errno=%d (%s)",STATE_UPDATE_FILE,errno,strerror(errno));
    	    }
    	
	if(countdown <= 0)		/* If this file is full, */
	    {
	    write(handle,"REWIND\n",7);
	    close(handle);
	    handle = -1;
	    unlink(STATE_UPDATE_FILE);
	    countdown = STATE_UPDATE_MAXLINES;
	    continue;
	    }

	sprintf(line,"SERIAL %d\n",serial++);
	write(handle,line,strlen(line));
    	}

    va_start(va,string);
    vsprintf(line,string,va);
    strcat(line,"\n");
    va_end(va);

    write(handle, line, strlen(line) );

    countdown--;
    } /* end of state_update() */

/*=========================================================================
** Lock and unlock those data structures which must not be simultainiously 
** modified.  The one I can think of at the moment is the queue.  This 
** locking is necessary because otherwise SIGCHLD or SIGALRM may interupt 
** a queue modification operation.
=========================================================================*/
int lock_level = 0;
sigset_t lock_set;

void lock(void)
    {
    lock_level++;
    if(lock_level==1)
        {
        sigprocmask(SIG_BLOCK, &lock_set, (sigset_t*)NULL);
        }
    } /* end of lock */
   
void unlock(void)
    {
    lock_level--;
    if(lock_level==0)
        {
	sigprocmask(SIG_UNBLOCK, &lock_set, (sigset_t*)NULL);
        }
    } /* end of unlock() */

/*======================================================================
** Load the printer and group configurations
======================================================================*/

/*
** Save the mounted media list of a certain printer.
*/
void save_medialist(int prnid)
    {
    FILE *sf;
    char fname[MAX_PATH];
    char pbin[MAX_BINNAME];
    char pmedia[MAX_MEDIANAME];
    int x;
    struct Printer *p;
     
    sprintf(fname, "%s/%s", MOUNTEDDIR, get_dest_name(prnid));
    if( (sf=fopen(fname, "w"))==(FILE*)NULL )
        fatal(ERROR_DIE, "save_medialist(): can't open \"%s\"", fname);

    p=&printers[prnid];
    for(x=0;x<p->nbins;x++)
        {
        ASCIIZ_to_padded(pbin,get_bin_name(p->bins[x]),sizeof(pbin));
        ASCIIZ_to_padded(pmedia,get_media_name(p->media[x]),sizeof(pmedia));
        fwrite(pbin,sizeof(pbin),1,sf);
        fwrite(pmedia,sizeof(pmedia),1,sf);
        }

    fclose(sf);
    } /* end of save_medialist() */

/*
** Recover the mounted media list of a certain printer.
*/
void recover_medialist(int prnid)
    {
    FILE *rf;
    char fname[MAX_PATH];
    char pbin[MAX_BINNAME];             /* padded bin name */
    char pmedia[MAX_MEDIANAME];         /* padded media name */
    char abin[MAX_BINNAME+1];           /* ASCIIZ bin name */
    char amedia[MAX_MEDIANAME+1];       /* ASCIIZ media name */
    int x;
    struct Printer *p;

    #ifdef DEBUG_RECOVER
    debug("recovering media for \"%s\"",get_dest_name(prnid));
    #endif

    sprintf(fname,"%s/%s",MOUNTEDDIR,get_dest_name(prnid));
    if( (rf=fopen(fname,"r"))==(FILE*)NULL )
        return;             /* missing file means nothing mounted */

    p=&printers[prnid];
    while( fread(pbin,sizeof(pbin),1,rf) && fread(pmedia,sizeof(pmedia),1,rf) )
        {
        padded_to_ASCIIZ(abin,pbin,sizeof(pbin));
        padded_to_ASCIIZ(amedia,pmedia,sizeof(pmedia));
        #ifdef DEBUG_RECOVER
        debug("recovering: abin=\"%s\", amedia=\"%s\"",abin,amedia);
        #endif
        for(x=0;x<p->nbins;x++)
            {
            if( strcmp(get_bin_name(p->bins[x]),abin)==0 )
                {
                p->media[x]=get_media_id(amedia);
                break;
                }
            }
        }

    fclose(rf);
    } /* end of recover_medialist() */

/*
** Load the data on a single printer into the array.
** This routine is called with a pointer to a printer array entry
** and the name of the printer.
*/
void load_printer(struct Printer *printer,char *filename)
    {
    FILE *prncf;
    char tempstr[256];              /* for reading lines */
    char tempstr2[MAX_BINNAME+1];   /* for extracting bin names */
    struct stat pstat;      
    mode_t newmod;                  /* new file mode */
    int count; float x1, x2;

    sprintf(tempstr,"%s/%s",PRCONF,filename);
    if( (prncf=fopen(tempstr,"r")) == (FILE*)NULL )
	fatal(0,"load_printer(): can't open printer config file \"%s\", errno=%d.",    
        	tempstr,errno);

    strcpy(printer->name,filename);		/* store the printer name */

    printer->alert_interval = 0;		/* no alerts */
    printer->alert_method = (char*)NULL;	/* (At least not until we */
    printer->alert_address = (char*)NULL;	/* read an "Alert:" line.) */

    printer->next_error_retry = 0;		/* Not in fault state */
    printer->next_engaged_retry = 0;

    printer->protect = FALSE;			/* not protected that we know of yet */
    printer->charge_per_duplex = 0;
    printer->charge_per_simplex = 0;

    printer->nbins = 0;				/* start with zero bins */
    printer->AutoSelect_exists = FALSE;		/* start with no "AutoSelect" bin */
    printer->ppop_pid = (pid_t)0;		/* nobody waiting for stop */
    printer->previous_status = PRNSTATUS_IDLE;
    printer->status = PRNSTATUS_IDLE;		/* it is idle now */
    printer->accepting = TRUE;			/* is accepting */

    printer->cancel_job = FALSE;		/* don't cancel a job on next pprdrv exit */

    fstat(fileno(prncf),&pstat);
    if(pstat.st_mode & S_IXUSR)			/* if user execute set, */
        printer->status = PRNSTATUS_STOPT;	/* printer is stop */ 
    if(pstat.st_mode & S_IXGRP)			/* if group execute is set, */
        printer->accepting = FALSE;		/* printer not accepting */
    
    while(fgets(tempstr, sizeof(tempstr), prncf)!=(char*)NULL)
        {
        if(*tempstr==';' || *tempstr=='#')
            continue;

	/* For "Alert:" lines, read the interval, method, and address. */
	else if(strncmp(tempstr,"Alert: ",7)==0)
            {
            int x=7;                                    /* len of "Alert: " */
            int len;

	    x+=strspn(&tempstr[x]," \t");               /* skip spaces */
	    sscanf(&tempstr[x],"%d",&printer->alert_interval);
	    x+=strspn(&tempstr[x]," \t-0123456789");	/* skip spaces and */
							/* digits */
            len=strcspn(&tempstr[x]," \t");             /* get word length */
            printer->alert_method = (char*)myalloc(len+1,sizeof(char));
            strncpy(printer->alert_method,&tempstr[x],len);  /* copy */
            printer->alert_method[len]=(char)NULL;           /* terminate */
            x+=len;                                     /* move past word */
            x+=strspn(&tempstr[x]," \t");               /* skip spaces */

            len=strcspn(&tempstr[x]," \t\n");           /* get length */
            printer->alert_address = (char*)myalloc(len+1,sizeof(char));
            strncpy(printer->alert_address,&tempstr[x],len); /* copy */
            printer->alert_address[len] = (char)NULL;	/* terminate */
            }

        /* For each "Bin:" line, add the bin to the list */
        else if(strncmp(tempstr, "Bin: ", 5) == 0)
            {
            if( printer->nbins < MAX_BINS )
                {
                int t=ppr_sscanf(tempstr,"Bin: %#s",sizeof(tempstr2),tempstr2);
                
                if(t)                   /* if it has 1 or more arguments */
                    {                   /* use the bin name */
                    printer->bins[printer->nbins] = get_bin_id(tempstr2);
		    if(strcmp(tempstr2,"AutoSelect")==0)	/* If this is an "AutoSelect" bin, */
		    	printer->AutoSelect_exists = TRUE;	/* then set the flag. */
		    printer->media[printer->nbins] = -1; 	/* nothing mounted yet */  
                    printer->nbins++;
                    }
                }
            else
                {
                error("Printer \"%s\" has too many bins.", printer->name);
                }
            }

        /*
        ** Set per sheet/page charge.  Actually, here we just see if
        ** we will charge and if so, make the printer a 
        ** protected printer
        */
        else if((count=sscanf(tempstr, "Charge: %f %f", &x1, &x2)) >= 1)
            {
	    printer->charge_per_duplex = (int)(x1 * 100.0);

	    /* In order to be backwards compatible, we will set the per-page
	       charge the same as the per sheet charge if it is missing. */
	    if(count == 2)
	    	printer->charge_per_simplex = (int)(x2 * 100.0);
	    else
	    	printer->charge_per_simplex = printer->charge_per_duplex;

            printer->protect = TRUE;
            }
        } /* end of while(), unknown lines are ignored */

    /*
    ** If printer is protected, turn on ``other'' execute bit,
    ** otherwise, turn it off.
    */
    if(printer->protect)
        newmod = pstat.st_mode | S_IXOTH; 
    else
        newmod = pstat.st_mode & (~ S_IXOTH);

    if(newmod != pstat.st_mode)
        fchmod(fileno(prncf), newmod);

    /* Close that configuration file! */
    fclose(prncf);
    } /* end of load_printer() */

/* 
** Search the printer configuration directory and
** call load_printer() once for each printer.
*/
void load_printers(void)
    {
    DIR *dir;           /* directory to search */
    struct dirent *direntp;
    int x;
    int len;

    printers = (struct Printer*)myalloc(MAX_PRINTERS, sizeof(struct Printer));

    dir=opendir(PRCONF);
    if(dir==(DIR*)NULL)
        fatal(0,"load_printers(): opendir() failed");

    x=0; printer_count=0;
    while( (direntp=readdir(dir)) != (struct dirent*)NULL )
        {
	/* Skip . and .. and hidden files. */
        if( direntp->d_name[0] == '.' )
            continue;

	/* Skip Emacs style backup files. */
	len=strlen(direntp->d_name);
	if( len > 0 && direntp->d_name[len-1]=='~' )
	    continue;

        if(x==MAX_PRINTERS)
	    {    
            error("load_printers(): too many printers");
            break;		/* break out of loop */
            }

        load_printer(&printers[x], direntp->d_name);
        printer_count++;        /* do now so get_dest_name() ok */
        recover_medialist(x);   /* get those forms back */
        save_medialist(x);	/* this list must be up to date for pprdrv */
        x++;
        }

    closedir(dir);
    } /* end of load_printers() */

/*
** Load a new printer configuration file.
** This is called when ppad sends a command over the pipe.
**
** We must be careful to free any allocated memory blocks
** in an old printer configuration.
*/
void new_printer_config(char *printer)
    {
    int x;
    int first_deleted=-1;	/* index of first deleted printer entry */
    int is_new=FALSE;
    int saved_status;
    pid_t saved_ppop_pid;
    char fname[MAX_PATH];
    FILE *testopen;
    
    lock();		/* don't let printer state change while we do this */

    /*
    ** Find the printer in the printer array
    ** and free its memory blocks.
    */
    for(x=0;x<printer_count;x++)
        {
	if(printers[x].status==PRNSTATUS_DELETED)	/* take note */
	    {
	    if(first_deleted==-1)			/* if there are any */
		first_deleted=x;			/* deleted slots we can use */
	    continue;
	    }
        if(strcmp(printers[x].name,printer)==0)		/* If name */
	    {						/* matches existing printer, */
	    if(printers[x].alert_method!=(char*)NULL)	/* free its memory blocks. */
	        myfree(printers[x].alert_method);
	    if(printers[x].alert_address!=(char*)NULL)
	        myfree(printers[x].alert_address);
            break;
            }
        }
    
    if(x==printer_count)	/* if x points to just after end of list, */
    	{
	is_new = TRUE;		/* this is a new printer */
    	if(first_deleted != -1)	/* if we have an empty */
    	    {			/* slot, */
    	    x = first_deleted;	/* re-use it */
    	    }
    	}

    if(x==MAX_PRINTERS)		/* if new printer and no more room, */
	{			/* just say there is an error */
        error("new_printer_config(): too many printers");
        unlock();		/* and ignore the request */
        return;
    	}

    if(x==printer_count)	/* If we are appending to printer list, */
	printer_count++;	/* add to count of printers. */

    /*
    ** Try to open the printer configuration file
    ** in order to determine if it is being deleted.
    ** (When a printer is deleted, ppad asks us to
    ** update its status anyway.  This is where we
    ** find out it was deleted.)
    */
    sprintf(fname,"%s/%s",PRCONF,printer);
    if( (testopen=fopen(fname,"r")) == (FILE*)NULL )
    	{
	if(is_new)				/* if new printer */
	    {
	    error("attempt to delete printer \"%s\" which never existed",printer);
	    }
	else
	    {
	    state_update("PRNDELETE %s",printer);	/* inform queue display programs */
	    printers[x].status = PRNSTATUS_DELETED;	/* mark as deleted */
	    }    	
	unlock();
	return;
    	}
    fclose(testopen);		/* We don't want this, so close it. */

    state_update("PRNRELOAD %s",printer);	/* Inform queue display programs. */

    saved_status = printers[x].status;		/* We will use these in a moment */
    saved_ppop_pid = printers[x].ppop_pid;	/* if the printer is not new. */

    load_printer(&printers[x],printer);		/* load printer configuration */
    recover_medialist(x);			/* load the list of mounted media */
    save_medialist(x);				/* save updated (very important for pprdrv) */
    
    if( ! is_new)		/* if old printer, */
        {			/* restore its status */
        printers[x].status = saved_status;
        printers[x].ppop_pid = saved_ppop_pid;

	update_notnow(x);           /* update list of printable jobs */
        if(printers[x].status==PRNSTATUS_IDLE) /* if printer idle */
            look_for_work(x);       /* see if anything to do (needn't lock) */
        }

    unlock();		/* ok, let things move again */
    } /* end of new_printer_config() */

/*
** Load the configuration of a single group of printers into the array.
** This routine is called with a pointer to a group array entry and
** the name of the group to put in it.
*/
void load_group(struct Group *cl,char *filename)
    {
    FILE *clcf;             /* class (group) config file */
    char fname[MAX_PATH];
    char tempstr[256];      /* for reading lines */
    char tempstr2[32];      /* for extractions */
    int y;
    struct stat cstat;
    mode_t newmod;      
    int line = 0;

    sprintf(fname,"%s/%s",GRCONF,filename);
    if( (clcf=fopen(fname,"r")) == (FILE*)NULL )
        fatal(0,"load_group(): can't open \"%s\", errno=%d",fname,errno);

    strcpy(cl->name,filename);		/* store the group name */
    cl->last = -1;			/* initialize last value */
    
    fstat(fileno(clcf),&cstat);

    if(cstat.st_mode & S_IXGRP)		/* if group execute is set, */
        cl->accepting = FALSE;		/* group not accepting */
    else
        cl->accepting = TRUE;		/* is accepting */

    if(cstat.st_mode & S_IXUSR)		/* if user execute bit is set, */
        cl->held = TRUE;		/* group is held */
    else
        cl->held = FALSE;

    cl->protect = FALSE;		/* don't protect by default */
    cl->deleted = FALSE;		/* it is not a deleted group! */
    cl->rotate = TRUE;			/* rotate is default */

    y=0;
    while(fgets(tempstr,256,clcf)!=(char*)NULL)
        {
        line++;              

        /* Read the name of a group member */
        if( ppr_sscanf(tempstr,"Printer: %#s",sizeof(tempstr2),tempstr2)==1 )
            {
            if(y>=MAX_GROUPSIZE)            /* if group has overflowed, */
                {                           /* note error and ignore member */
                error("group \"%s\" exceeds %d member limit",
                        cl->name,MAX_GROUPSIZE);
                continue;
                }
            if( (cl->printers[y]=get_printer_id(tempstr2)) == -1 )
                {         
                error("group \"%s\":  member \"%s\" does not exist",
                       cl->name,tempstr2);
                }
            else                                
                {
                if(printers[cl->printers[y]].protect)
                                        /* if this printer is protected, */
                    cl->protect=TRUE;   /* protect the group */
                y++;            /* increment members index */
                }
            continue;
            }

        /* read a rotate flag value */
        if(strncmp(tempstr,"Rotate:",7)==0)
            {
            if( (cl->rotate=torf(&tempstr[7])) == ANSWER_UNKNOWN )
                fatal(0,"Invalid Rotate option (%s, line %d)",
                    fname,line);     
            continue;
            } 
        }

    /*
    ** If group is protected, turn on user execute bit,
    ** otherwise, turn it off.
    */
    if(cl->protect)
        newmod=cstat.st_mode | S_IXOTH; 
    else
        newmod=cstat.st_mode & (~ S_IXOTH);

    if(newmod != cstat.st_mode)		/* (only act if new mode is different) */
        fchmod(fileno(clcf),newmod);

    fclose(clcf);           /* close the group configuration file */
    cl->members=y;          /* set the members count */
    } /* end of load_group() */

/*
** Call load_group() once for each group in the groups directory.
*/
void load_groups(void)
    {
    DIR *dir;           /* directory to search */
    struct dirent *direntp;
    int x;
    int len;

    groups=(struct Group*) calloc(MAX_GROUPS,sizeof(struct Group));    
    if(groups==(struct Group*)NULL)
        fatal(0,"out of memory in load_groups");

    if( (dir=opendir(GRCONF)) == (DIR*)NULL )
        fatal(0,"opendir() failed in load_groups()");

    x=0;
    while( (direntp=readdir(dir)) != (struct dirent*)NULL )
        {

	/* Skip "." and ".." and hidden files. */
        if( direntp->d_name[0] == '.' )
            continue;

	/* Skip Emacs style backup files. */
	len=strlen(direntp->d_name);
	if( len > 0 && direntp->d_name[len-1] == '~' )
	    continue;

        if(x==MAX_GROUPS)
            {
            error("load_groups(): too many groups");
            break;
            }

        load_group(&groups[x],direntp->d_name);
        x++;
        }

    group_count=x;                    /* remember how many groups we have */

    closedir(dir);
    } /* end of load_groups() */

/*
** Load a new group configuration file.
** This is called when a command from ppad is 
** received over the pipe.
*/
void new_group_config(char *group)
    {
    int x;				/* group array index */
    int y;				/* members array index */
    int destid;
    int first_deleted=-1;
    int is_new=FALSE;
    char fname[MAX_PATH];
    FILE *testopen;
    
    lock();				/* prevent clashing queue changes */

    for(x=0; x<group_count; x++)	/* find the group in the group array */
    	{
	if( groups[x].deleted )		/* if we see any deleted group slots, */
	    {				/* remember where the 1st one is */
	    if(first_deleted==-1)
	    	first_deleted=x;
	    continue;
	    }
    	if( strcmp(groups[x].name,group) == 0 )
    	    break;
    	}
    	
    if(x==group_count)
    	{
    	is_new=TRUE;
    	if(first_deleted!=-1)
    	    x=first_deleted;
    	else
    	    group_count++;
    	}

    if(x==MAX_GROUPS)			/* if adding, make sure there is room */
        {
	unlock();
        error("new_group_config(): too many groups");
        return;
        }
        
    /* see if we are deleting this group */
    sprintf(fname,"%s/%s",GRCONF,group);
    if( (testopen=fopen(fname,"r")) == (FILE*)NULL )
    	{
    	if(is_new)
	    {
    	    error("attempt to delete group \"%s\" which never existed",group);
    	    }
	else
	    {
	    state_update("GRPDELETE %s",group);		/* inform queue display programs */
	    groups[x].deleted = TRUE;			/* mark as deleted */
	    }
	unlock();
	return;
	}
    fclose(testopen);

    state_update("GRPRELOAD %s",group);	/* inform queue display programs */

    load_group(&groups[x],group);	/* read the group file */
    
    /* fix all the jobs for this group */
    destid = get_id_from_gindex(x);
    for(x=0; x<queue_entries; x++)
    	{
    	if(queue[x].destid==destid)	/* if job is for this group, */
     	    {				/* reset the media ready lists */
	    set_notnow_for_job(&queue[x], TRUE);
	    queue[x].never = 0;		/* never bits may be invalid */
    	    }				/* so be reckless and clear them */
    	}				/* (they will be set again if necessary) */

    unlock();	

    /* look for work for any group members which are idle */
    for(y=0; y<groups[x].members; y++)
    	{
        if(printers[groups[x].printers[y]].status==PRNSTATUS_IDLE)
            look_for_work(groups[x].printers[y]);
	}
    } /* end of new_group_config() */

/*=====================================================================
** Routines for operating on printer and group names and numbers.
=====================================================================*/

/*
** Convert a destination (group or printer) id to a destination name.
*/
char *get_dest_name(int destid)
    {
    if( destid < MAX_PRINTERS )
        {
        if( destid < printer_count )
            return printers[destid].name;
        else
            return "<invalid>";
        }
    else
        {
        destid-=MAX_PRINTERS;
        if(destid<group_count)
            return groups[destid].name;
        else
            return "<invalid>";
        }
    } /* end of get_dest_name() */

/*
** Return the destination id which matches a destination name.
** If the destination name is unknown, return -1. 
*/
int get_printer_id(char *name)
    {
    int x;

    for(x=0;x<printer_count;x++)                /* try all printers */
        {
        if(strcmp(printers[x].name,name)==0)    /* if matches, */
            {
            if(printers[x].status==PRNSTATUS_DELETED) /* if deleted, */
            	return -1;			/* pretend it doesn't exist. */
	    else
            	return x;                       /* return id number */
            }
        }

    return -1;
    } /* end of get_printer_id() */

int get_group_id(char *name)
    {
    int x;

    for(x=0;x<group_count;x++)                /* try all groups */
        {
        if(strcmp(groups[x].name,name)==0)
	    {
            if( ! groups[x].deleted)		/* if not a deleted group, */
            	return x+MAX_PRINTERS;		/* return the destid */
            else				/* otherwise, */
            	return -1;			/* return not found */
            }
        }

    return -1;
    } /* end of get_group_id() */

/* This is the one we normally use, it finds a group name
   in preference to a printer name. */
int get_dest_id(char *name)
    {
    int ret;

    if( (ret=get_group_id(name)) != -1 )
    	return ret;
    	
    if( (ret=get_printer_id(name)) != -1 )
    	return ret;

    return -1;              /* we didn't find it */
    } /* end of get_dest_id() */
      
/* This one is rarely used.  It prefers a printer name. */
int reversed_get_dest_id(char *name)
    {
    int ret;

    if( (ret=get_printer_id(name)) != -1 )
    	return ret;

    if( (ret=get_group_id(name)) != -1 )
    	return ret;
    	
    return -1;              /* we didn't find it */
    } /* end of reversed_get_dest_id() */
      
/*
** Return TRUE if the destination id in question is a group id.
*/
int isgroup(int id)
    {
    if(id>printer_count)
        return -1;
    else
        return 0;
    } /* end of isgroup() */

/*
** Get the offset of a certain printer into a certain group array.
** Return -1 if the printer is not a member. 
*/
int get_member_offset(int destid,int prnid)
    {
    struct Group *cl;
    int x;

    cl=&groups[destid-MAX_PRINTERS];
    for(x=0;x<cl->members;x++)
        {
        if(cl->printers[x]==prnid)
            return x;
        }
    return -1;
    } /* end of get_member_offset() */

/*
** Return true if the indicated destination is accepting requests.
*/
int accepting(int destid)
    {
    if(isgroup(destid))  
        return groups[gindex(destid)].accepting;
    else
        return printers[destid].accepting;
    } /* end of accepting() */

/*
** Convert a destination id to a group array index.
*/
int gindex(int destid)
    {
    return destid-MAX_PRINTERS;
    } /* end of gindex() */
    
/*
** Convert a group array index to a destination id.
*/
int get_id_from_gindex(int gindex)
    {
    return gindex+MAX_PRINTERS;
    } /* end of get_id_from_gindex() */

/*======================================================================
** Media handling routines
======================================================================*/

/*
** Get the bitmask which identifies a particular printer in
** the "never" and "notnow" bit fields of jobs with a certain
** destination id.  If the printer is not included in the 
** specified destination (i.e. the destination id is not the 
** printer id and is not that of a group containing the printer)
** then, return 0.
** If the destination is the printer, return 1.
*/
int get_prn_bitmask(int prnid, int destid)
    {
    int off;

    if(prnid==destid)                               /* if job is directly */
        return 1;                                   /* to this printer */

    off=get_member_offset(destid,prnid);
    if(off==-1)                                     /* if prn not in group, */
        return 0;                                   /* an error, return 0 */
    else                                            /* if in group, */
        return 1<<off;                              /* shift bit to prop pos */
    } /* end of get_prn_bitmask() */

/*
** Convert a media id to a name. 
*/
char *get_media_name(int mediaid)
    {
    if( mediaid<media_count && mediaid>=0 ) /* if we have such a media name */ 
         return media[mediaid];       /* give its name */
    else
        return "<invalid>";
    } /* end of get_media_name() */

/*
** Convert a media name to a media id.
*/       
int get_media_id(char *medianame)
    {
    int x;

    for(x=0;x<media_count;x++)                  /* search for it */
        {
        if(strcmp(media[x],medianame)==0)       /* if found, */
            return x;                           /* pass id back to caller */
        }

    if(x==MAX_MEDIAS)
        fatal(0,"get_media_id(): media array overflow");

    media[x] = mystrdup(medianame);		/* add new entry */
    media_count++;
    return x;
    } /* end of get_media_id() */

/*
** Convert a bin id to a name. 
*/
char *get_bin_name(int binid)
    {
    if( binid<binnames_count && binid>=0 ) /* if we have such a bin name */
          return binname[binid];       /* give its name */
    else
        return "<invalid>";
    } /* end of get_bin_name() */
 
/*
** Convert a bin name to a media id.
*/
int get_bin_id(char *nbinname)
    {
    int x;

    for(x=0;x<binnames_count;x++)               /* search for it */
        {
        if(strcmp(binname[x],nbinname)==0)      /* if found, */
            return x;                           /* pass id back to caller */
        }

    if(x==MAX_BINNAMES)
        fatal(0,"get_bin_id(): binname[] overflow");
                                                
    binname[x]=mystrdup(nbinname);             /* add new entry */
    binnames_count++;
    return x;
    } /* end of get_bin_id() */

/*
** Get the stopt mask for a group.
*/
int stoptmask(int destid)
    {
    struct Group *g;
    int mask=0;
    int x;

    if(!isgroup(destid))
        {
        if(printers[destid].status < PRNSTATUS_DELIBERATELY_DOWN)
            return 0;
        else
            return 1;
        }

    g=&groups[gindex(destid)];

    for(x=0;x<g->members;x++)
        {
        if( printers[g->printers[x]].status >= PRNSTATUS_DELIBERATELY_DOWN )
            mask|=1<<x;
        }    

    return mask;
    } /* end of stoptmask() */

/*========================================================================
** The routines below are related to media.
=========================================================================*/

/*
** Check to see if a particular printer has the proper media 
** mounted to print a certain job.  Return TRUE if it has.
*/
int hasmedia(int prnid, struct QEntry *job)
    {
    int x,y;

    #ifdef DEBUG_MEDIA
    debug("hasmedia(prnid=%d, struct QEntry job->destid=%d):",
        prnid,job->destid);
    #endif

    /* If printer has no bins, assume it can print anything! */
    if(printers[prnid].nbins==0)
    	return TRUE;
 
    /* If printer has a bin called "AutoSelect", */
    /* assume it can print anything. */
    if(printers[prnid].AutoSelect_exists)
    	return TRUE;

    /* test each requred media type */
    for(x=0;x<MAX_DOCMEDIA && job->media[x]!=-1; x++)
        {
        for(y=0; ;y++)                              /* look in every bin */
            {
            if(y==printers[prnid].nbins)            /* if not in any bin */
                {
                #ifdef DEBUG_MEDIA
                debug("media absent");
                #endif
                return FALSE;                       /* then prn hasn't form */
                }
            if(job->media[x]==printers[prnid].media[y])
                break;                              /* if found in one bin */
            }                                       /* that is good enough */
        }
 
    #ifdef DEBUG_MEDIA
    debug("all media present");
    #endif
    return TRUE;                  /* return true if all passed test */
    } /* end of hasmedia() */

/*
** Compute the `notnow' bitmask for a job.  This operates on a structure which
** is not in the queue yet, so it need not be done with the queue locked,
** however, the caller may want to lock the queue first to be sure the
** information will not become outdated before it can be placed in the queue.
**
** This routine is also used when responding to the "ppop mount" command
** and when responding to a change of a groups member list.
*/
void set_notnow_for_job(struct QEntry *nj, int inqueue)
    {
    #ifdef DEBUG_NOTNOW
    debug("set_notnow_for_job()");
    #endif

    if(isgroup(nj->destid))             /* check for each printer: */
        {
        struct Group *gptr;             /* ptr to group array entry */
        int x;
        gptr = &groups[gindex(nj->destid)]; 

        nj->notnow=0;                   /* start with clear mask */

        for(x=0;x<gptr->members;x++)    /* do for each printer */
            if( ! hasmedia(gptr->printers[x],nj) )
                {
                nj->notnow|=1<<x;       /* if hasn't media, set bit */  
                }
        }

    else                                /* just one printer: */
        {
        if( hasmedia(nj->destid,nj) )   /* if printer has all forms */
            {
            nj->notnow=0;               /* bit zero clear */
            }
        else                            /* else */
            {
            nj->notnow=1;               /* bit zero set */
            }
        }

    set_waitreason(nj,stoptmask(nj->destid),inqueue);
    } /* end of set_notnow_for_newjob() */

/*
** Update the notnow bitmask for every job which might print
** on the specified printer.
**
** update_notnow2() is called only from update_notnow().
** update_notnow2() scans the queue for jobs for a specific
** destination and sets the specified printer's notnow bit
** in each matching queue entry. 
**
** update_notnow() is called whenever new media is mounted
** on a printer.
** update_notnow() calls update_notnow2() once for the printer
** as the destination, and once for each group which has the
** printer as a member.
*/
void update_notnow2(int destid, int prnbit, int prnid)
    {
    int x;
    int stopt=stoptmask(destid);        /* mask of stop members */

    for(x=0;x<queue_entries;x++)        /* scan the entire queue */
	{
        if(queue[x].destid==destid)     /* if job is for this destination */
            {                           /* then */
            if( hasmedia(prnid,&queue[x]) )
                queue[x].notnow&=(0xff)^prnbit;  /* !!! */
            else
                queue[x].notnow|=prnbit;

            /* set waiting to prn or media */
            set_waitreason(&queue[x],stopt,TRUE); 
            }
	}
    } /* end of update_notnow2() */

void update_notnow(int prnid)
    {
    int g;
    int prnbit;

    #if DEBUG_NOTNOW
    debug("update_notnow()");
    #endif

    lock();

    /* Update for jobs with this printer as their dest. */
    update_notnow2(prnid,1,prnid);         

    /* For each group if this prn is a member update it. */
    for(g=0;g<group_count;g++)
        if( (prnbit=get_prn_bitmask(prnid,g+MAX_PRINTERS)) )
            update_notnow2(g+MAX_PRINTERS,prnbit,prnid); 

    unlock();    
    } /* end of update_notnow() */

/*
** If the job status is `waiting for printer' or `waiting for media',
** examine the notnow bits and update the status.
**
** The caller should call this from a section of code bracketed by
** lock() and unlock().  For this reason, the routine does not call them.
**
** The inqueue parameter is TRUE if the job is already in the queue.  If it
** is, we must call p_job_new_status().
*/
void set_waitreason(struct QEntry *job, int stopt_members_mask, int inqueue)
    {
    if( (job->status==STATUS_WAITING) || (job->status==STATUS_WAITING4MEDIA) )
        {
	int new_status;

        if( isgroup(job->destid) )      /* set for a group */
            {
            int allmask = (1<<groups[gindex(job->destid)].members)-1;

	    if( ((job->notnow | stopt_members_mask)==allmask) && (stopt_members_mask!=allmask) )
		new_status = STATUS_WAITING4MEDIA;
	    else                                 
		new_status = STATUS_WAITING;
            }

        else                            /* set for a printer */
            {
	    if( job->notnow && (stopt_members_mask==0) )
		new_status = STATUS_WAITING4MEDIA;
	    else
		new_status = STATUS_WAITING;
            }

	/* If we have selected a new status, write it now. */
	if(new_status != job->status)
	    {
	    if(inqueue)
		p_job_new_status(job, new_status);
	    else
		job->status = new_status;
	    }
	}
    } /* end of set_waitreason() */

/*
** This routine is called whenever a printer is "started"
** or "stopt" with ppop.  It updates the status of any job which 
** could possibly print on that printer.
**
** startstop_update_waitreason2() is a supporting routine and
** is called only by startstop_update_waitreason().
*/
void startstop_update_waitreason2(int destid)
    {
    int x;
    int stopt = stoptmask(destid);	/* mask of stop members */

    for(x=0;x<queue_entries;x++)	/* scan the entire queue */
        {
        if(queue[x].destid==destid)	/* if job is for this destination */
            {				/* then */
            /* set waiting to prn or media */
            set_waitreason(&queue[x],stopt,TRUE);
            }
	}
    } /* end of startstop_update_waitreason2() */

void startstop_update_waitreason(int prnid)
    {
    int g;

    lock();

    /* Update for jobs with this printer as their dest. */
    startstop_update_waitreason2(prnid);         

    /* For each group if this prn is a member update it. */
    for(g=0;g<group_count;g++)
	{
        if( (get_member_offset(g+MAX_PRINTERS,prnid)!=-1) )
            startstop_update_waitreason2(g+MAX_PRINTERS); 
	}

    unlock();    
    } /* end of startstop_update_waitreason() */

/*=======================================================================
** Routines for keeping printers busy.
=======================================================================*/

/*
** Start a specific printer to print a specific job.
** If this routine does start the job it returns 0,
** if the printer is busy it, returns -1,
** if printer can't print this job but is idle, return -2.
**
** This routine will also return -1 if fork() fails.
**
** Please call this routine with tables locked.
**
** You should not call this routine for jobs which are not
** ready to print.
**
** This function is often called from within a signal handler,
** so it should not call routines which are not reentrant.
*/
int start_printer(int prnid, struct QEntry *job)
    {
    pid_t pid;                  /* process id of pprdrv */
    int bitmask;                /* this printers bit offset */

    #ifdef DEBUG_PRNSTART
    debug("start_printer(prnid=%d)",prnid);
    #endif

    /*
    ** Don't start if the printer is already printing.
    */
    if( printers[prnid].status != PRNSTATUS_IDLE )
        {
        #ifdef DEBUG_PRNSTART
        debug("printer \"%s\" is not idle",get_dest_name(prnid));
        #endif
        return -1;      
        }

    /*
    ** Don't start it if the destination is a group and it is held.
    */
    if( isgroup(job->destid) && groups[gindex(job->destid)].held )
	{
	DODEBUG_PRNSTART(("destination \"%s\" is held",get_dest_name(job->destid)));
        return -2;
        }

    /*
    ** Don't start it the printer doesn't have the forms or
    ** is incapable of printing this document.
    */
    if( (job->notnow & (bitmask=get_prn_bitmask(prnid,job->destid)))
            || (job->never & bitmask) )
        {
	DODEBUG_PRNSTART(("printer has notnow or never bit set"));
        return -2;
        }

    /*
    ** Make sure we have not hit MAX_ACTIVE printers.
    ** If we have, this printer must starve for now.
    */
    if(active_printers == MAX_ACTIVE)
    	{
	DODEBUG_PRNSTART(("Starting printer \"%s\" would exceed MAX_ACTIVE", get_dest_name(prnid)));
	new_prn_status(&printers[prnid], PRNSTATUS_STARVED);
	return -1;    	
    	}

    /*
    ** Make sure we are not required to yield to a printer which
    ** has been waiting for rations.
    ** If we yield, we become a starving printer.
    */
    if(printers[prnid].previous_status != PRNSTATUS_STARVED && (active_printers+starving_printers) >= MAX_ACTIVE)
    	{
    	DODEBUG_PRNSTART(("\"%s\" yielding to a starving printer", get_dest_name(prnid)));
	new_prn_status(&printers[prnid], PRNSTATUS_STARVED);
	return -1;    	
    	}

    /* start pprdrv */
    if( (pid=fork()) == -1 )            /* if error */
        {				
        error("Couldn't fork, printer \"%s\" not started", get_dest_name(prnid));
	new_prn_status(&printers[prnid], PRNSTATUS_STARVED);
        return -1;
        }

    if(pid)				/* parent */
        {
        DODEBUG_PRNSTART(("Starting printer \"%s\", pid=%d", get_dest_name(prnid), pid));
	active_printers++;		    /* add to count of printers printing */
	printers[prnid].pid = pid;          /* remember which process is printing it */
	printers[prnid].jobdestid = job->destid; /* remember what job is being printed */
	printers[prnid].id = job->id;
	printers[prnid].subid = job->subid; 
	printers[prnid].homenode_id = job->homenode_id;
	new_prn_status(&printers[prnid], PRNSTATUS_PRINTING);

	job_new_status(job->id, job->subid, job->homenode_id, prnid);

        if(isgroup(job->destid))            /* mark last used in */
            {				    /* its group */
            groups[gindex(job->destid)].last = get_member_offset(job->destid,prnid);
            }
        }
    else				/* child */
        {
        char jobname[MAX_PATH];
        char pass_str[10];
        
	/*
	** Set the session id so lost commentators will get SIGHUP.
	** (That is, all of pprdrv's children will die with it.)
	** For some reason this does not seem to have the desired effect.
	*/
	setsid();

	/*
	** Reconstruct the queue file name.
	** We can not use the library routine "local_jobid()" here
	** because it tends to ommit parts which conform 
	** to default values.
	*/
        sprintf(jobname,"%s:%s-%d.%d(%s)",
		ppr_get_nodename(),
                get_dest_name(job->destid),
                job->id,job->subid,
                nodeid_to_nodename(job->homenode_id));

	/*
	** Convert the pass number to a string so that
	** we may use it as a argument to execl().
	*/
	sprintf(pass_str,"%d",job->pass);

	/*
	** Our child should not start out with
	** signals blocked, even if we have some blocked.
	*/
        {
        sigset_t sigset;        
        sigemptyset(&sigset);   
        sigprocmask(SIG_SETMASK,&sigset,(sigset_t*)NULL);
        }

	/* Overlay this child process with pprdrv. */
        execl(PPRDRV_PATH,"pprdrv",	/* execute the driver program */
            get_dest_name(prnid),	/* printer name */
            jobname,			/* full job id string */
	    pass_str,			/* pass number as a string */
            (char*)NULL);

	sleep(1);			/* Give parent time to record PID. */

        _exit(255);
        }

    return 0;
    } /* end of start_printer() */

/*
** If there are any jobs this printer may print, start it now.
** Nothing really bad will happen if this routine is called while the
** printer is not idle, but doing so will waste time.
**
** This routine checks `never' and `notnow' for itself.  Doing so is
** not necessary but the code was accidentally written and probably
** makes things minutely faster, so it was left in.
**
** This routine may be called from within a signal handler so it
** should not call no-reentrant routines.
*/
void look_for_work(int prnid)
    {
    int x;
    int moff;                   /* member offset */
    int destid;                 /* temporary storage, destination id */
    int mask;                   /* mask for never and notnow */

    #ifdef DEBUG_PRNSTART
    debug("Looking for work for printer \"%s\"",get_dest_name(prnid));
    #endif

    lock();                     /* lock out others while we modify */

    for(x=0;x<queue_entries;x++)
        {
        #ifdef DEBUG_PRNSTART_GRITTY
        debug("trying job: destid=%d, id=%d, subid=%d, homenode_id=%d, status=%d",
            queue[x].destid,queue[x].id,queue[x].subid,queue[x].homenode_id,queue[x].status);
        #endif
        if( (queue[x].status==STATUS_WAITING)         /* if job ok to print */ 
            && ( ((destid=queue[x].destid)==prnid) || /* if for this printer */
                ( isgroup(destid)                     /* or for group */ 
                    && ((moff=get_member_offset(destid,prnid))!=-1) /* &we in */                        && !(queue[x].never&(mask=(1<<moff)))   /* & capable */
                            && !(queue[x].notnow&mask) ) ) )    /* & no media */
            {                                                   /* problem */
            if(!start_printer(prnid,&queue[x])!=-2) /* if printer busy */
                break;                              /* then stop looking */
            }                                       /* (-2 means can't print */
        }                                           /* this job) */

    #ifdef DEBUG_PRNSTART 
    if(x==queue_entries)
        debug("no work for \"%s\"",get_dest_name(prnid));
    #endif

    unlock();                   /* allow others to use tables now */

    } /* end of look_for_work() */

/*
** Figure out what printers might be able to start this job
** and try them in turn.  We let start_printer() determine if
** `notnow' or `never' bits are set.
**
** You should not call this routine for jobs which are not
** waiting to be printed.
**
** This routine is often called from within a signal handler,
** so it should not call non-reentrant routines.
*/
void start_suitable_printer(struct QEntry *job)
    {
    #ifdef DEBUG_PRNSTART
    debug("start_suitable_printer()");
    #endif

    lock();

    if(isgroup(job->destid))        /* if group, we have many to try */
        {
        struct Group *cl;
        int x,y;

        cl = &groups[gindex(job->destid)];

        if(cl->rotate)              /* if we should rotate */
            y=cl->last;             /* set just before next one */
        else                        /* otherwise, set just before */
            y=-1;                   /* first printer */

        for(x=0; x < cl->members; x++)
            {
            #ifdef DEBUG_PRNSTART_GRITTY
            debug("last printer in group was %d",y);
            #endif
            y=(y+1)%cl->members;    /* rotate to next printer */
            #ifdef DEBUG_PRNSTART_GRITTY
            debug("trying member %d",y);
            #endif
            if(start_printer(cl->printers[y],job)==0)
                break;
            }
        }
    else                            /* if a single printer, */
        {                           /* we can try only one */
        start_printer(job->destid,job);
        }

    unlock();
    } /* end of start_suitable_printer() */

/*=======================================================================
** Queue array routines
=======================================================================*/

/* 
** Initialize the queue, loading existing jobs into it.
*/
void initialize_queue(void)
    {
    DIR *dir;			/* the open queue directory */
    struct dirent *direntp;	/* one directory entry */
    FILE *qfile;		/* a queue file */
    struct QEntry newent;	/* the new terse queue entry we are building */
    struct QEntry *newentp;	/* pointer to the new queue entry */
    struct stat qstat;		/* for reading queue file flags */
    const char *thisnode;
    size_t thisnode_len;
    int len;
    char *ptr;
    char home_node[MAX_NODENAME+1];

    #ifdef DEBUG_RECOVER
    debug("initialize_queue()");
    #endif

    thisnode = ppr_get_nodename();
    thisnode_len = strlen(thisnode);

    /* Allocate memory to hold the queue. */    
    queue_size = QUEUE_SIZE_INITIAL;
    queue = (struct QEntry *)myalloc(queue_size, sizeof(struct QEntry));

    /* Open the queue directory. */
    if( (dir=opendir(QUEUEDIR)) == (DIR*) NULL )
        fatal(0,"initialize_queue(): can't open directory \"%s\"",QUEUEDIR);

    /* Protect the queue array during modification. */
    lock();

    /* loop to read file names */
    while( (direntp=readdir(dir)) != (struct dirent*)NULL )
        {
        if( direntp->d_name[0] == '.' )		/* ignore "." and ".." */
            continue;

	/* Ignore jobs destined for remote printers. */
	if( strcspn(direntp->d_name, ":") != thisnode_len 
		|| strncmp(direntp->d_name,thisnode,thisnode_len) )
	    continue;

	/* Ignore files whose names do not contain a colon.
	   (They shouldn't exist but we can't count of not
	   finding spurious files in the queue directory.) */
	if( direntp->d_name[thisnode_len] != ':' )
	    {
	    error("initialize_queue(): ignoring file \"%s\"", direntp->d_name);
	    continue;
	    }

        #ifdef DEBUG_RECOVER
        debug("initialize_queue(): inheriting queue file \"%s\"", direntp->d_name);
        #endif

	/*
	** Recover the destination id and queue id numbers from the name.
	*/
	ptr = &(direntp->d_name[thisnode_len+1]);	/* start after node name and colon */
	
	/* Get length of part before dash followed by digit: */
	for(len = 0; TRUE; len++)
	    {
	    len += strcspn(&ptr[len],"-");

	    if( ptr[len] != '-' || isdigit(ptr[len+1]) )
	    	break;
	    }
        if( ptr[len] != '-' )
            {
            error("initialize_queue(): invalid queue file name \"%s\"",direntp->d_name);
            continue;
            }

        ptr[len] = (char)NULL;				/* terminate destination name */

        if( (newent.destid=get_dest_id(ptr)) == -1 )	/* convert to destination id number */
            {
            error("initialize_queue(): destination \"%s\" no longer exists",ptr);
            continue;
            }

	ptr[len] = '-';					/* undo damage to queue file name */

	/* Scan for id number, subid number, and home node name. */
        if( (sscanf(&ptr[len+1],"%hd.%hd(%[^)])",&newent.id,&newent.subid,home_node) != 3)
                || (newent.id<0) || (newent.subid<0) )
            {
            error("initialize_queue(): invalid queue file name \"%s\"",direntp->d_name);
            continue;
            }

	/* Convert the home node name to a node id number.
	   These id numbers are used to save space in the 
	   queue array. */
	newent.homenode_id = assign_nodeid(home_node);

	/* Open the queue file. */
	{
	char qfname[MAX_PATH];
	
        sprintf(qfname,"%s/%s",QUEUEDIR,direntp->d_name);

        if( (qfile=fopen(qfname,"r")) == (FILE*)NULL )
            {
            error("initialize_queue(): couldn't open \"%s\"",direntp->d_name); 
	    free_nodeid(newent.homenode_id);
            continue;
            }
	}

        /* stat the file, we will use the information below */
        fstat(fileno(qfile), &qstat);

        /* recover the job status from the permission bits */
        newent.status = STATUS_WAITING;		/* Start with not held or arrested. */
        if(qstat.st_mode & BIT_JOB_HELD)	/* Is it held? */
            newent.status = STATUS_HELD;
        if(qstat.st_mode & BIT_JOB_ARRESTED)	/* Is it arrested? */
            newent.status = STATUS_ARRESTED;

        /*
        ** If the file is of zero length, then we can't use it,
        ** delete the job files.
        */
        if(qstat.st_size==0)
            {
            error("initialize_queue(): queue file \"%s\" was of zero length",
                direntp->d_name);                
            fclose(qfile);
            unlink_job(newent.destid, newent.id, newent.subid, newent.homenode_id);
	    free_nodeid(newent.homenode_id);
            continue;
            }

	/*
	** Read the priority from the queue file.
	** There is no good reason for a default. 
	*/
        newent.priority = 20;
	{
	char tempstr[MAX_QFLINE+2];
        while( fgets(tempstr,sizeof(tempstr),qfile) != (char*)NULL )
            {
            if( sscanf(tempstr,"Priority: %hd",&newent.priority) == 1 )
                { /* no code */ }
            }
	}

        /*
        ** If the job fits in the queue and it is judged suitable for
        ** immediate printing, then try to start a printer for it.
        */
	{
	int rank1, rank2;
	
        if( ((newentp=enqueue_job(&newent,&rank1,&rank2)) != (struct QEntry*)NULL)
        	&& (newentp->status==STATUS_WAITING) )
	    {
            start_suitable_printer(newentp);
            }
	}

        fclose(qfile);
        } /* end directory search loop */

    unlock();
    closedir(dir);

    #ifdef DEBUG_RECOVER
    debug("initialize_queue() done");
    #endif
    } /* end of initialize_queue() */

/*
** Enqueue a job.  If for some reason we can't, we are allowed to return a NULL pointer.
*/
struct QEntry *enqueue_job(struct QEntry *newentry, int *rank1, int *rank2)
    {
    int x;
    FILE *qfile;
    char qfname[MAX_PATH];
    char tline[256];
    char tmedia[MAX_MEDIANAME+1];
    int destmates_passed = 0;

    DODEBUG_NEWJOB(("enqueue_job(): destid=%d id=%d subid=%d status=%d",
            newentry->destid, newentry->id, newentry->subid, newentry->status));

    /* Read in the list of required media. */
    sprintf(qfname, "%s/%s:%s-%d.%d(%s)",	/* queue file name */
	QUEUEDIR, ppr_get_nodename(), get_dest_name(newentry->destid),
        newentry->id, newentry->subid, nodeid_to_nodename(newentry->homenode_id) );
    if( (qfile=fopen(qfname,"r")) == (FILE*)NULL )
        {                                       /* open it */
        error("enqueue_job(): can't open \"%s\"",qfname);
        return (struct QEntry*)NULL;
        }                                       

    x=0;
    while( (x<MAX_DOCMEDIA) && (fgets(tline,sizeof(tline), qfile) != (char*)NULL) )
	{
        if(ppr_sscanf(tline,"Media: %#s", sizeof(tmedia), tmedia) == 1)
            newentry->media[x++] = get_media_id(tmedia);
        }
    fclose(qfile);

    DODEBUG_MEDIA(("%d media requirement(s) read", x));

    while(x < MAX_DOCMEDIA)		/* fill in entra spaces with -1 */
        newentry->media[x++] = -1;

    /*
    ** If this is a group job, set pass number
    ** to one, otherwise, set pass number to zero
    ** which will indicate the pprdrv that it is not
    ** a group job.
    */
    if( isgroup(newentry->destid) )
	newentry->pass = 1;
    else
    	newentry->pass = 0;

    lock();                             /* must lock before set_notnow... */

    /* Set the bitmask which shows which printers have the form. */
    set_notnow_for_job(newentry, FALSE);

    /* Initialize the bitmask of printers which can't possibly print it. */
    newentry->never = 0;

    /* Sanity check: */
    if(queue_entries > queue_size || queue_entries < 0)
    	fatal(1, "enqueue_job(): assertion failed: queue_entries=%d, queue_size=%d", queue_entries, queue_size);

    /*
    ** The the queue is already full, try to expand the queue array,
    ** otherwise just don't put it in the queue.  If that is done, 
    ** the job will not be printed until pprd is restarted.
    */
    if(queue_entries == queue_size)
        {
	if( (queue_size + QUEUE_SIZE_GROWBY) <= QUEUE_SIZE_MAX )
	    {
	    DODEBUG_NEWJOB(("enqueue_job(): expanding %d entry queue to %d entries", queue_size, queue_size+QUEUE_SIZE_GROWBY));
	    queue_size += QUEUE_SIZE_GROWBY;
	    queue = myrealloc(queue, queue_size, sizeof(struct QEntry));
	    }
	else
	    {
	    error("enqueue_job(): queue array overflow");
	    unlock();
	    return (struct QEntry*)NULL;
	    }
	}

    destmates_passed = 0;
    for(x=0; x < queue_entries; x++)	/* Find or make a space in the queue array. */
        {
	/*
	** Lower priority number mean more urgent jobs.  If we have found 
	** a job with a higher priority number than the job we are inserting,
	** move all the jobs from here on one slot furthur toward the end
	** of the queue and break out of the loop.
	*/
        if( newentry->priority < queue[x].priority )
            {
            int y;

	    /* We must do this in segments due to problems with overlapping copies. */
            for( y=queue_entries; y >= x; y-- )
                memmove(&queue[y+1],&queue[y],sizeof(struct QEntry));

            break;                                     
            }

        /*
        ** If we are passing a job which is for the same destination as
        ** the jbo we are inserting, add one to the count which will
        ** be stored in rank2.
        */
	if( queue[x].destid == newentry->destid )
	    destmates_passed++;
        }

    /* Copy the new queue entry into the place prepared for it. */
    memcpy(&queue[x], newentry, sizeof(struct QEntry));

    queue_entries++;            /* increment our count of queue entries */

    unlock();			/* release our lock on the queue array */

    *rank1 = x;			/* fill in queue rank of new entry */
    *rank2 = destmates_passed;

    return &queue[x];           /* return pointer to what we put it in */
    } /* end of enqueue_job() */

/*
** Remove an entry from the queue.
*/
void dequeue_job(int destid, int id, int subid, int homenode_id)
    {
    int x;

    DODEBUG_DEQUEUE(("dequeue_job(destid=%d, id=%d, subid=%d, homenode_id=%d)", destid, id, subid, homenode_id));

    /* Inform queue display programs of job deletion. */
    state_update("DEL %s", local_jobid(get_dest_name(destid),id,subid,nodeid_to_nodename(homenode_id)) );

    lock();		/* lock the queue array while we modify it */

    for(x=0; x < queue_entries; x++)
	{
        if( queue[x].id==id && queue[x].subid==subid )
            {
            DODEBUG_DEQUEUE(("job %s-%d.%d removed from queue",
                get_dest_name(queue[x].destid), queue[x].id, queue[x].subid));

            if( (queue_entries-x) > 1 )		/* do a move if not last entry */
                {				/* (BUG: was queue_entries was queue_size prior to version 1.30) */
                memmove(&queue[x], &queue[x+1],
                    sizeof(struct QEntry) * (queue_entries-x-1) );
                }

            queue_entries--;			/* one less in queue */
            break;				/* and we needn't look farther */
            }
	}
	
    unlock();		/* unlock queue array */
    } /* end of dequeue_job() */

/*
** unlink job files
*/
void unlink_job(int destid, int id, int subid, int homenode_id)
    {
    char fname[MAX_PATH];
    const char *nodename;
    const char *destname;
    const char *homenode_name;

    DODEBUG_DEQUEUE(("unlink_job(destid=%d, id=%d, subid=%d, homenode_id=%d)", destid, id, subid, homenode_id));

    nodename = ppr_get_nodename();
    destname = get_dest_name(destid);
    homenode_name = nodeid_to_nodename(homenode_id);

    sprintf(fname, "%s/%s:%s-%d.%d(%s)",
	QUEUEDIR, nodename, destname, id, subid, homenode_name);
    unlink(fname);

    sprintf(fname, "%s/%s:%s-%d.%d(%s)-comments",
	DATADIR, nodename, destname, id, subid, homenode_name);
    unlink(fname);

    sprintf(fname, "%s/%s:%s-%d.%d(%s)-pages",
	DATADIR, nodename, destname, id, subid, homenode_name);
    unlink(fname);

    sprintf(fname, "%s/%s:%s-%d.%d(%s)-text",
	DATADIR, nodename, destname, id, subid, homenode_name);
    unlink(fname);

    sprintf(fname, "%s/%s:%s-%d.%d(%s)-log",
	DATADIR, nodename, destname, id, subid, homenode_name);
    unlink(fname);

    sprintf(fname, "%s/%s:%s-%d.%d(%s)-infile",
	DATADIR, nodename, destname, id, subid, homenode_name);
    unlink(fname);
    } /* end of unlink_job() */

/*
** Unlink a job which hasn't been put in the queue
** because we decided to cancel it as soon as it arrived.
*/
void unlink_new_job(const char *destnode, const char *destname, int id, int subid, const char *homenode_name)
    {
    char filename[MAX_PATH];

    sprintf(filename, "%s/%s:%s-%d.%d(%s)",
	QUEUEDIR, destnode, destname, id, subid, homenode_name);
    unlink(filename);

    sprintf(filename, "%s/%s:%s-%d.%d(%s)-comments",
	DATADIR, destnode, destname, id, subid, homenode_name);
    unlink(filename);

    sprintf(filename, "%s/%s:%s-%d.%d(%s)-pages",
	DATADIR, destnode, destname, id, subid, homenode_name);
    unlink(filename);

    sprintf(filename, "%s/%s:%s-%d.%d(%s)-text",
	DATADIR, destnode, destname, id, subid, homenode_name);
    unlink(filename);

    sprintf(filename, "%s/%s:%s-%d.%d(%s)-log",
	DATADIR, destnode, destname, id, subid, homenode_name);
    unlink(filename);

    sprintf(filename, "%s/%s:%s-%d.%d(%s)-infile",
	DATADIR, destnode, destname, id, subid, homenode_name);
    unlink(filename);
    } /* end of unlink_new_job() */

/*
** Change the status of a job. 
** We will update the execute bits on the queue file if required.
**
** The version p_job_new_status() takes a pointer to the job structure.
** The version job_new_status() takes the id and subid and finds the 
** job in the queue.
**
** This routine is sometimes called from within a signal handler, so
** it must not call routines which are not reentrant.
*/
struct QEntry *p_job_new_status(struct QEntry *job, int newstat)
    {
    char filename[MAX_PATH];
    char *destname;

    /* lock(); */

    destname = get_dest_name(job->destid);

    /*
    ** Inform queue display programs.
    */
    {
    char *status_string = (char*)NULL;

    switch(newstat)
    	{
	case STATUS_WAITING:
	    status_string = "waiting for printer";
	    break;
	case STATUS_HELD:
	    status_string = "held";
	    break;
	case STATUS_SEIZING:
	    status_string = "being seized";
	    break;
	case STATUS_WAITING4MEDIA:			/* never used */
	    status_string = "waiting for media";	/* see set_waitreason() */
	    break;
    	case STATUS_ARRESTED:
    	    status_string = "arrested";
    	    break;
    	case STATUS_CANCEL:
    	    status_string = "being canceled";
    	    break;
    	default:
	    state_update("JST %s printing on %s",
	    	local_jobid(destname,job->id,job->subid,nodeid_to_nodename(job->homenode_id)),
	    	get_dest_name(newstat));
    	    break;
    	}

    if( status_string != (char*)NULL )
	{
	state_update("JST %s %s",
		local_jobid(destname,job->id,job->subid,nodeid_to_nodename(job->homenode_id)),
		status_string);
	}
    }

    /* Re-construct the name of the queue file. */
    sprintf(filename, QUEUEDIR"/%s:%s-%d.%d(%s)", ppr_get_nodename(), destname, job->id, job->subid, nodeid_to_nodename(job->homenode_id));

    /*
    ** If necessary, do a chmod() on the queue file to reflect the new status.
    */
    if(newstat==STATUS_HELD)					/* If held, */
	chmod(filename, BIT_JOB_BASE | BIT_JOB_HELD);		/* set user exec bit. */
    else if(newstat==STATUS_ARRESTED)				/* If arrested, */
	chmod(filename, BIT_JOB_BASE | BIT_JOB_ARRESTED);	/* set group exec bit. */
    else if( (job->status==STATUS_HELD) ||			/* If was held or */
	    (job->status==STATUS_ARRESTED) )			/* arrested but is no longer, clear */
	chmod(filename, BIT_JOB_BASE);				/* the bits. */

    job->status = newstat;		/* save new status */

    /* unlock(); */

    return job;				/* return a pointer to the queue entry */
    } /* end of p_job_new_status() */

struct QEntry *job_new_status(int id, int subid, int homenode_id, int newstat)
    {
    int x;

    lock();				/* lock queue array while we work on it */

    for(x=0; x < queue_entries; x++)	/* Find the job in the */
        {				/* queue array. */
        if( (queue[x].id==id) && (queue[x].subid==subid) && (queue[x].homenode_id==homenode_id) )
            {
	    p_job_new_status(&queue[x],newstat);
            break;			/* break for loop */
            }
        }
    if(x==queue_entries)
        fatal(0, "job_new_status(): %d %d not found in queue",id,subid);

    unlock();			/* unlock queue array */

    return &queue[x];		/* return a pointer to the queue entry */
    } /* end of job_new_status() */

/*=====================================================================
** Support routines to operate on printers.
=====================================================================*/

/*
** Change the status of a printer.
**
** This function might be called from within a signal 
** handler so it should not call non-reentrant routines.
**
** This function is called from signal handlers, so it must not
** call non-reentrant routines such as malloc().
*/
void new_prn_status(struct Printer *printer, int newstatus)
    {
    lock();					/* lock the queue and printer list */

    if(printer->status==PRNSTATUS_DELETED)	/* deleted printers */
	{					/* can't change state */
	error("new_prn_status(): attempt to change state of deleted printer");
	unlock();
        return;
        }

    /* Stopping printers can only become stopt. */
    if(printer->status == PRNSTATUS_STOPPING)
        newstatus = PRNSTATUS_STOPT;

    /* Halting printers can only become stopt. */
    else if(printer->status == PRNSTATUS_HALTING)
	newstatus = PRNSTATUS_STOPT;
    
    if(newstatus == PRNSTATUS_FAULT)		/* if entering fault state */
	{
	if( ++printer->next_error_retry )		/* inc number of next retry */
	    {
	    printer->countdown = printer->next_error_retry * RETRY_MULTIPLIER;
	    if(printer->countdown > MIN_RETRY)	/* no less frequently */
		printer->countdown = MIN_RETRY;	/* than MIN_RETRY seconds */ 
	    }
	else					/* if next_error_retry was -1 */
	    {					/* that means no auto-retry mode */
	    printer->countdown = 0;
	    }
        }

    else if(newstatus == PRNSTATUS_ENGAGED)	/* if entering otherwise engaged state, */
	{
	printer->next_engaged_retry++;		/* increment retry count */
	printer->countdown = ENGAGED_RETRY;	/* and set time for retry */
	}

    else if(newstatus==PRNSTATUS_STARVED)	/* If this printer is being set */
    	{					/* to starved, increment */
    	starving_printers++;			/* count of starving printers */
    	}					/* which will need to be started up later. */

    /* Modify the user execute bit as necessary. */
    if(newstatus==PRNSTATUS_STOPT)
        {
	char filename[MAX_PATH];
	struct stat prncf_stat;
	sprintf(filename, "%s/%s", PRCONF, printer->name);
        stat(filename, &prncf_stat);             /* turn on user execute */
        chmod(filename, prncf_stat.st_mode | S_IXUSR);
        } 
    else if(printer->status==PRNSTATUS_STOPT)
        {
	char filename[MAX_PATH];
	struct stat prncf_stat;
        sprintf(filename,"%s/%s", PRCONF, printer->name);
	stat(filename,&prncf_stat);             /* turn off user execute */
        chmod(filename,prncf_stat.st_mode & 0677 );
        }

    /* If ppop is waiting (ppop wstop), inform it that printer has stopt. */
    if( (newstatus==PRNSTATUS_STOPT) && printer->ppop_pid )
        {
	kill(printer->ppop_pid,SIGUSR1);
        printer->ppop_pid=(pid_t)0;
        }

    printer->previous_status = printer->status;	/* save for reference in start_printer() */
    printer->status = newstatus;		/* write the new status */    

    unlock();

    /*
    ** Inform queue display programs of the new printer status.
    ** Act differenly according to the new status.
    **
    ** Notice that we must do this after the new status has gone into
    ** effect, otherwise, the retry count and retry interval will
    ** not yet be correct.
    */
    {
    switch(newstatus)
    	{
	case PRNSTATUS_PRINTING:
	    state_update("PST %s printing %s %d",
		printer,
	    	local_jobid(get_dest_name(printer->jobdestid),printer->id,printer->subid,nodeid_to_nodename(printer->homenode_id)),
	    	printer->next_error_retry);
	    break;
	case PRNSTATUS_IDLE:
	    state_update("PST %s idle", printer);
	    break;
	case PRNSTATUS_CANCELING:
	    state_update("PST %s canceling %s",
	    	printer,
	    	local_jobid(get_dest_name(printer->jobdestid),printer->id,printer->subid,nodeid_to_nodename(printer->homenode_id)));
	    break;
	case PRNSTATUS_SEIZING:
	    state_update("PST %s seizing %s",
	    	printer,
	    	local_jobid(get_dest_name(printer->jobdestid),printer->id,printer->subid,nodeid_to_nodename(printer->homenode_id)));
	    break;
	case PRNSTATUS_FAULT:
	    state_update("PST %s fault %d %d",
	    	printer,
		printer->next_error_retry,
		printer->countdown);
	    break;
	case PRNSTATUS_ENGAGED:
	    state_update("PST %s engaged %d %d",
		printer,
		printer->next_engaged_retry,
		printer->countdown);
	    break;
	case PRNSTATUS_STARVED:
	    state_update("PST %s starved", printer);
	    break;
	case PRNSTATUS_STOPT:
	    state_update("PST %s stopt", printer);
	    break;	
	case PRNSTATUS_STOPPING:
	    state_update("PST %s stopping (printing %s)",
		printer,
	    	local_jobid(get_dest_name(printer->jobdestid),printer->id,printer->subid,nodeid_to_nodename(printer->homenode_id)));
	    break;
	case PRNSTATUS_HALTING:
	    state_update("PST %s halting (printing %s)",
		printer,
	    	local_jobid(get_dest_name(printer->jobdestid),printer->id,printer->subid,nodeid_to_nodename(printer->homenode_id)));
	    break;
    	}
    }
    } /* end of new_prn_status() */

/*===================================================================
** This routine takes whatever action is necessary when a child 
** process terminates.  In the case of a responder process, no 
** action is necessary.  In the case of a pprdrv process, it will
** be necessary to update the status of the printer and possibly 
** start another job.
===================================================================*/
static void reapchild(int signum)
  {
  pid_t pid;
  int wstat;		/* result of waitpid() */
  int x;		/* index into array of printers */
  int estat;		/* exit status */
  int saved_errno = errno;
  int myalloc_saved = myalloc_checkpoint_get();

  lock_level++;		/* set our flag */
     
  while( (pid=waitpid((pid_t)-1, &wstat, WNOHANG)) > (pid_t)0 )
    {
    #ifdef DEBUG_PRNSTOP
    debug("Child terminated, pid=%ld, exit=%i", (long)pid, WEXITSTATUS(wstat));
    #endif

    for(x=0; x < printer_count; x++)		/* examine each printer */
        {
        if( printers[x].pid == pid )		/* if pprdrv was for this one, */
            {
	    active_printers--;			/* a printer is no longer active */

	    printers[x].pid = 0;		/* prevent future false match */

            if(WIFEXITED(wstat))		/* if exited normally, */
                {
                estat = WEXITSTATUS(wstat);	/* use that exit value */
                }
            else if(WIFSIGNALED(wstat))		/* if killed, */
                {
		alert(printers[x].name, TRUE, "pprdrv killed by signal %d (%s).", WTERMSIG(wstat), strsignal(WTERMSIG(wstat)) );
		estat = EXIT_PRNERR;

                if(WCOREDUMP(wstat))		/* if signal caused a core dump, */
                    {
                    alert(printers[x].name, FALSE, "pprdrv dumped core.");
                    estat = EXIT_PRNERR_NORETRY;
                    }
                }
	    else if(WIFSTOPPED(wstat))		/* stopt by signal */
	    	{
		alert(printers[x].name, TRUE, "pprdrv stopt by signal %d (%s).", WSTOPSIG(wstat), strsignal(WSTOPSIG(wstat)));
		estat = EXIT_PRNERR_NORETRY;
	    	}
	    else
	    	{
		alert(printers[x].name, TRUE, "pprdrv didn't exit wasn't killed, wasn't stopt, what happened?");
	    	estat = EXIT_PRNERR_NORETRY;
	    	}

            if(estat > EXIT_MAX)		/* Unknown exit codes */
                {				/* are printer errors. */
                error("pprdrv exited with unknown value %d", estat);
                estat = EXIT_PRNERR;
                }

            switch(estat)			/* act on pprdrv exist code */
                {
                case EXIT_PRINTED:		/* no error */
                    #ifdef DEBUG_PRNSTOP
                    debug("(printed normally)");
                    #endif
		    respond(printers[x].jobdestid, printers[x].id, printers[x].subid, printers[x].homenode_id, x, RESP_FINISHED);
			/* ^ tell the user his job has been printed */
		    dequeue_job(printers[x].jobdestid,printers[x].id,printers[x].subid,printers[x].homenode_id);
			/* ^ remove from queue array */
                    unlink_job(printers[x].jobdestid, printers[x].id, printers[x].subid, printers[x].homenode_id);
			/* ^ delete job files */
           	    maybe_cancel_alert(printers[x].name,	/* Maybe tell */
                            printers[x].alert_interval,		/* operator printer */
                            printers[x].alert_method,		/* is ok now. */
                            printers[x].alert_address,
                            printers[x].next_error_retry);
                    printers[x].next_error_retry = 0;
                    printers[x].next_engaged_retry = 0;
                    new_prn_status(&printers[x], PRNSTATUS_IDLE);
		    free_nodeid(printers[x].homenode_id);
                    break;		/* ^ make printer idle again */

                case EXIT_JOBERR:		/* error in job */
                    #ifdef DEBUG_PRNSTOP
                    debug("(job error)");
                    #endif
		    /* Tell the user the job was arrested. */
                    respond(printers[x].jobdestid, printers[x].id, printers[x].subid, printers[x].homenode_id, x, RESP_ARRESTED);
		    /* place in special held state */
                    job_new_status(printers[x].id, printers[x].subid, printers[x].homenode_id,STATUS_ARRESTED);
           	    maybe_cancel_alert(printers[x].name,	/* Maybe tell */
                            printers[x].alert_interval,		/* operator */
                            printers[x].alert_method,		/* printer is ok now. */
                            printers[x].alert_address,
                            printers[x].next_error_retry);
                    printers[x].next_error_retry = 0;
		    printers[x].next_engaged_retry = 0;
                    new_prn_status(&printers[x], PRNSTATUS_IDLE);
                    break;                      /* let printer go on working */
    
                case EXIT_INCAPABLE:            /* capabilities exceeded */
                    {
                    int y;
                    int id,subid;
                    int arrest = FALSE;

                    id = printers[x].id;	/* x is the printer id */
                    subid = printers[x].subid;

                    lock();
                    for(y=0; ; y++)             /* find the queue entry */
                        {    
                        if( y == queue_entries )
                            fatal(0, "reapchild(): job missing from array");
                        if( queue[y].id==id && queue[y].subid==subid )
                            break;
                        } 

                    /*
                    ** If a single printer, set the never mask to 1,
		    ** arrest the job, and say the printer is incapable.
		    */
                    if( ! isgroup(printers[x].jobdestid) ) 
                        {
                        queue[y].never |= 1;
                        arrest = TRUE;
                        respond(printers[x].jobdestid,
                        	printers[x].id, printers[x].subid,
                        	printers[x].homenode_id,
				x, RESP_ARRESTED_PRINTER_INCAPABLE);
                        }

                    /*
                    ** If a group of printers, set the appropriate
                    ** bit in the never mask.  If the bit has been
                    ** set for every printer in the group, arrest
                    ** the job and inform the user.
                    */
                    else
                        {
			/*
			** Get a number with the bit corresponding to this
			** printer's possition in this destination group
			** and add that bit to the bits set in the never
			** mask.  If the printer has been removed from
			** the group since the job was started, then
			** get_prn_bitmask() will return 0 which will
			** be ok.
			*/
                        queue[y].never |= get_prn_bitmask(x,printers[x].jobdestid); 

			/*
			** If every never bit has been set, then arrest the
			** job and inform the user.  But, if that was the
			** 1st pass, we give the job a second chance.
			*/
                        if( queue[y].never==((1<<groups[gindex(printers[x].jobdestid)].members)-1) )
                            {
			    if( ++(queue[y].pass) > 2 )	/* if beyond the second pass */
				{
                            	arrest = TRUE;
                            	respond(printers[x].jobdestid,
                            		printers[x].id,printers[x].subid,
					printers[x].homenode_id,
                            		x, RESP_ARRESTED_GROUP_INCAPABLE);
                                }
                            else
                            	{
                            	queue[y].never = 0;
                            	}
                            }
                        } /* end of else (group of printers) */

                    unlock();

                    /* 
                    ** If we are to arrest the job, do it.
                    ** Otherwise, start it on another printer. 
                    */
                    if(arrest)
                        job_new_status(printers[x].id,printers[x].subid,printers[x].homenode_id,STATUS_ARRESTED);
                    else                                  
                        start_suitable_printer(job_new_status(printers[x].id,printers[x].subid,printers[x].homenode_id,STATUS_WAITING) );

		    /*
		    ** In any event, this printer can start printing 
		    ** its next job.  Notice that we don't zero the 
		    ** retry count because failing to print an unprintable 
		    ** job does not prove that the printer could have 
		    ** printed a good job.
		    */
                    new_prn_status(&printers[x], PRNSTATUS_IDLE);
                    }
                    break;

                case EXIT_SIGNAL:           /* signal for printer halt */
                    #ifdef DEBUG_PRNSTOP
                    debug("(aborted due to signal)");
                    #endif
		    if(printers[x].status == PRNSTATUS_HALTING)
			{
			if(!printers[x].cancel_job)
			    start_suitable_printer(job_new_status(printers[x].id, printers[x].subid, printers[x].homenode_id, STATUS_WAITING));
			new_prn_status(&printers[x], PRNSTATUS_STOPT);
			break;
			}
		    if(printers[x].status == PRNSTATUS_SEIZING)
			{
			job_new_status(printers[x].id, printers[x].subid, printers[x].homenode_id, STATUS_HELD);
			new_prn_status(&printers[x], PRNSTATUS_IDLE);
			break;
			}
                    if(printers[x].status == PRNSTATUS_CANCELING)
			{
			new_prn_status(&printers[x], PRNSTATUS_IDLE);
			break;      
			}   /* otherwise, fall through */

                case EXIT_PRNERR:			/* printer error */
                case EXIT_PRNERR_NORETRY:
		case EXIT_ENGAGED:
		case EXIT_STARVED:
                    #ifdef DEBUG_PRNSTOP
                    debug("(pprdrv indicates printer error)");
                    #endif

                    if(estat==EXIT_PRNERR_NORETRY)	/* If no retry should occur, */
                        {				/* say so in alert log. */
                        alert(printers[x].name, FALSE, "Printer placed in fault mode.");
                        }
                    else if(estat==EXIT_PRNERR)     /* Otherwise, if fault retry */
                        {                           /* coming, say so in alert. */
                        alert(printers[x].name, FALSE, "Printer placed in auto-retry mode.");
                        }

		    start_suitable_printer(         /* try to start the job on another printer */
			job_new_status(printers[x].id, printers[x].subid, printers[x].homenode_id, STATUS_WAITING) );

		    if(estat==EXIT_ENGAGED)	    /* if not a true fault, but was otherwise engaged, */
		        {			    /* then put in special retry state. */
			new_prn_status(&printers[x], PRNSTATUS_ENGAGED);
		        }	/* we won't post any alerts on busy printers */
		    else if(estat==EXIT_STARVED)    /* status for printers */
		    	{			    /* whose pprdrv can't fork */
		    	new_prn_status(&printers[x], PRNSTATUS_STARVED);
		    	}
		    else
			{
                        if(estat==EXIT_PRNERR_NORETRY)		/* If no retry coming, then set  */
                            printers[x].next_error_retry = -1;	/* set retry number to -1 which new_prn_status() will make 0 */

			new_prn_status(&printers[x], PRNSTATUS_FAULT);

                        post_alert(printers[x].name,   /* maybe tell operator */
                            printers[x].alert_interval,
                            printers[x].alert_method,
                            printers[x].alert_address,
                            printers[x].next_error_retry);
                        }
                    break;			/* end of PRNERR case */                        
                }				/* end of pprdrv exit code switch */

	    /* If the job was canceled, do it now no matter what the printer did. */
	    if(printers[x].cancel_job)
		{
                respond(printers[x].jobdestid,
                        	printers[x].id, printers[x].subid,
                                printers[x].homenode_id,
                                x, RESP_CANCELED_PRINTING);
                dequeue_job(printers[x].jobdestid,printers[x].id,printers[x].subid,printers[x].homenode_id);
                unlink_job(printers[x].jobdestid,
                                printers[x].id, printers[x].subid, printers[x].homenode_id);
		free_nodeid(printers[x].homenode_id);

		printers[x].cancel_job = FALSE;
		}

            break;      /* we found our printer */
            }           /* end of if(this is the right printer) */
        } /* end of for which searches printers array */

    if( x == printer_count )		/* if we didn't find it */
        {
        #ifdef DEBUG_RESPOND
        debug("(respond process?)");	/* then protest! */
        #endif
	
	if( WIFSIGNALED(wstat) )
	    error("Process %ld was killed by signal %d (%s)", (long)pid, WTERMSIG(wstat), strsignal(WTERMSIG(wstat)) );
	else if( WIFEXITED(wstat) && WEXITSTATUS(wstat) != 0 )
	    error("Process %ld exited with error code %d", (long)pid, WEXITSTATUS(wstat) );

	if( WCOREDUMP(wstat) )
	    error("Process %ld dumped core", (long)pid);

        }
    else				/* otherwise, look for new job */
        {
        if(printers[x].status==PRNSTATUS_IDLE)
            look_for_work(x);		/* to start on this printer */
        }
    }					/* end of while(watipid()) loop */

  lock_level--;				/* clear our flag */

  #ifdef DEBUG_PRNSTOP
  debug("end of reapchild()");
  #endif

  myalloc_checkpoint_put(myalloc_saved);
  errno = saved_errno;
  } /* end of reapchild() */

/*
** Receive a new job.
**
** This function calls state_update() in order to inform
** queue display programs that there is a new job in the queue.
** This function is `instrumented' rather than enqueue_job()
** because enqueue_job() is used to re-load jobs that are
** already in the queue when pprd starts.
*/
static void ppr_new_job(char *command)
    {
    struct QEntry newent;		/* space for building new queue entry */
    char destname[MAX_DESTNAME+1];	/* name of destination */
    char destnode[MAX_NODENAME+1];
    char homenode[MAX_NODENAME+1];
    int hold;
    struct QEntry *newentp;		/* new queue entry */
    int rank1, rank2;

    /* Parse command: */
    sscanf(command,"j %s %s %hd %hd %s %hd %d",
		destnode,
		destname,			/* group or printer name */
		&newent.id,			/* read major queue id */
		&newent.subid,			/* minor queue id */
		homenode,
		&newent.priority,		/* and queue priority */
		&hold);
		
    #ifdef DEBUG_NEWJOBS
    debug("ppr_new_job(): destnode=%s destname=%s id=%d subid=%d homenode=%s priority=%d hold=%d",
	destnode, destname, newent.id, newent.subid, homenode, newent.priority, hold);
    #endif

    /*
    ** If the job does not have a destination node of this node,
    ** it is an error.  The job should have been sent to rpprd
    ** in stead.
    */
    if( strcmp(destnode, ppr_get_nodename()) )
    	{
	error("Job received for node \"%s\", discarding it", destnode);
        unlink_new_job(destnode, destname, newent.id, newent.subid, homenode);
	return;
    	}

    /* Assign the initial job status: */
    newent.status = STATUS_WAITING;
    if(hold)
	newent.status = STATUS_HELD;

    /*
    ** Convert the home node name to an id number.
    */
    newent.homenode_id = assign_nodeid(homenode);

    /*
    ** If the destination (printer or group) is a known one, 
    */
    if( (newent.destid=get_dest_id(destname)) != -1 )
        {
        /* 
        ** If destination is accepting jobs,
	** then try to put the job in the queue.
	**
	** Once it is in the queue, we will try to start a printer
	** for it unless the job is held.
	*/
        if(accepting(newent.destid))
	    {
            if( (newentp=enqueue_job(&newent, &rank1, &rank2)) != (struct QEntry*)NULL )
		{			/* If it fit in the array, */
		state_update("JOB %s %d %d",
			local_jobid(destname, newent.id, newent.subid, homenode),
			rank1, rank2);
		if(!hold)
		    start_suitable_printer(newentp);
                }
	    else
	    	{
		/*
		** If we end up here it is because there was no room
		** in the job.  enqueue_job() will have added an error
		** message to the log, so there is nothing we must do.
		*/
	    	}
            }

	/*
	** If the destination is not accepting
	** jobs, tell the user.
	*/
        else
            {
            respond(newent.destid, newent.id, newent.subid, newent.homenode_id, -1, RESP_CANCELED_REJECTING);
            unlink_job(newent.destid, newent.id, newent.subid, newent.homenode_id);
            free_nodeid(newent.homenode_id);
            }
        }

    /*
    ** The destination is unknown, inform the user and cancel the job.
    ** We can't use unlink_job() here because we don't have a valid 
    ** destination to pass to it.
    */
    else
        {
        respond2(destname, newent.id, newent.subid, homenode, destname, -1, RESP_CANCELED_BADDEST);
        unlink_new_job(destnode, destname, newent.id, newent.subid, homenode);
	free_nodeid(newent.homenode_id);
        }
    } /* end of ppr_new_job() */

/*=======================================================================
** Timer tick routine (SIGALRM handler).
**
** This is called about every TICK_INTERVAL seconds.  It restarts 
** printers which are in a fault state if the proper time has arrived.
**
** It is important that this routine not call any non-reentrant 
** library routines such as malloc().
=======================================================================*/
static void tick(int sig)
    {
    int x;
    static int hungry_countdown = 0;	/* seconds till next mercy for starving printers */
    static int hungry_x = 0;		/* where we left off in search for starving printers */
    int saved_errno = errno;		/* save errno of interupted code */
    int myalloc_saved = myalloc_checkpoint_get();

    alarm(TICK_INTERVAL);		/* set timer for next tick */

    #ifdef DEBUG_TICK
    debug("tick()");
    #endif

    #ifdef DEBUG_TICK
    debug("active_printers=%d, starving_printers=%d", active_printers, starving_printers);
    #endif

    /*
    ** If a printer is in fault retry mode or was printing for another
    ** computer and the retry time has expired then set its state
    ** to idle and look for a job to start on it.  (Perhaps it would
    ** help to explain that printers[x].next_error_retry will be zero if the
    ** printer is in fault-no-retry mode.)
    */
    for(x=0; x < printer_count; x++)
        {
        if( (printers[x].status==PRNSTATUS_FAULT && printers[x].next_error_retry )
		|| printers[x].status==PRNSTATUS_ENGAGED )
            {                       /* if faulted and retry allowed, */
            if( (printers[x].countdown -= TICK_INTERVAL) <= 0 )
                {
                new_prn_status(&printers[x], PRNSTATUS_IDLE);
                look_for_work(x);
                }
            }
        }

    /*
    ** Every UPGRADE_INTERVAL ticks, raise the priority of each job
    ** by one point until a job has achieved a priority of zero.
    */
    if( --upgrade_countdown == 0 )
        {
        /* lock(); */				/* lock() not appropriate */
        for(x=0; x < queue_entries; x++)	/* in this signal handler */
            {
            if(queue[x].priority)
                queue[x].priority-=1;
            }
        /* unlock(); */				/* unlock() not appropriate */
        upgrade_countdown = UPGRADE_INTERVAL;
        }

    /*
    ** If we have starving printers and the STARVING_RETRY_INTERVAL has
    ** expired, then try to restart some starving printers.
    */
    if( (hungry_countdown -= TICK_INTERVAL) <= 0 )
	{
	hungry_countdown = STARVING_RETRY_INTERVAL;

	x = 0;
	while( starving_printers && (active_printers < MAX_ACTIVE) && (x++ < printer_count) )
	    {
	    if(hungry_x >= printer_count)	/* wrap around if necessary */
	    	hungry_x = 0;
	    	
	    if(printers[hungry_x].status == PRNSTATUS_STARVED)
	    	{
	    	new_prn_status(&printers[hungry_x], PRNSTATUS_IDLE);
		starving_printers--;		/* now considered fed */
	    	look_for_work(hungry_x);
	    	}	
	    
	    hungry_x++;				/* move on to next printer */
	    }
	}

    myalloc_checkpoint_put(myalloc_saved);
    errno = saved_errno;
    } /* end of tick() */

/*========================================================================
** Signal handlers for stuff we might get but don't want.
========================================================================*/
/*
** I think that with this one, we rely on the fact
** that the signal will interupt the write() call.
*/
static void sigpipe_handler(int sig)
    {
    error("Received SIGPIPE");
    }
    
static void sighup_handler(int sig)
    {
    error("Received SIGHUP, ignoring it");
    }

static void sigterm_handler(int sig)
    {
    state_update("SHUTDOWN");
    fatal(0, "Received SIGTERM, exiting");
    }

/*========================================================================
** Open the FIFO
========================================================================*/
static FILE *open_fifo(void)
    {
    int rfd, wfd;
    int flags;
    FILE *FIFO;
    
    #ifdef DEBUG_PROGINIT
    debug("open_fifo()");
    #endif

    unlink(FIFO_NAME);

    if( mkfifo(FIFO_NAME,S_IRUSR | S_IWUSR) == -1 )
	fatal(0, "can't make FIFO, errno=%d (%s)", errno, strerror(errno) );

    /* Open the read end. */
    while( (rfd=open(FIFO_NAME, O_RDONLY | O_NONBLOCK)) == -1 )
	fatal(0, "can't open FIFO for read, errno=%d (%s)", errno, strerror(errno) );

    /* Open the write end which will prevent EOF on the read end. */
    if( (wfd=open(FIFO_NAME, O_WRONLY)) == -1 )
	fatal(0, "can't open FIFO for write, errno=%d (%s)", errno, strerror(errno) );

    /* Clear the non-block flag for rfd. */
    flags = fcntl(rfd, F_GETFL);
    flags &= ~O_NONBLOCK;
    fcntl(rfd, F_SETFL, flags);

    /* Set the two FIFO descriptors to close on exec(). */
    flags = fcntl(rfd, F_GETFD);
    flags |= FD_CLOEXEC;
    fcntl(rfd, F_SETFD, flags);

    flags = fcntl(wfd, F_GETFD);
    flags |= FD_CLOEXEC;
    fcntl(wfd, F_SETFD, flags);

    if( (FIFO=fdopen(rfd,"r")) == (FILE*) NULL )
        fatal(0, "fdopen() failed, errno=%d (%s)", errno, strerror(errno) );

    #ifdef DEBUG_PROGINIT
    debug("open_fifo(): done");
    #endif
        
    return FIFO;    
    } /* end of open_fifo() */

/*========================================================================
** The Main Procedure
** Initialization and FIFO command dispatch routine. 
========================================================================*/
int main(int argc, char *argv[])
    {
    FILE *FIFO;			/* First-in-first-out which feeds requests */
    struct sigaction sig;

    /* 
    ** This rather complicated bit of code eventually makes
    ** sure all the user IDs are ``ppr''.  This program should
    ** be setuid "ppr" and getgid "ppop".
    **
    ** This code must not call fatal(), fatal() should not be
    ** used until we are sure the permissions are right.
    */
    {
    uid_t uid, euid;
    gid_t gid, egid;

    uid = getuid();
    euid = geteuid();
    gid = getgid();
    egid = getegid();

    if(euid == 0)
        {
        fputs("pprd: security problem: euid = 0\n"
        	"(pprd should be setuid \"ppr\".)\n", stderr);
        exit(1);
        }
        
    if(egid == 0)
        {
        fputs("pprd: security problem: egid = 0\n"
        	"(pprd should be setgid \"ppop\".)\n", stderr);
        exit(1);
        }

    /*
    ** If the real user id is initially root, it takes the
    ** opportunity to create /var/spool/ppr if it does not exist.
    */
    if(uid == 0)
        {
	/*
	** Set uid and euid to "root" for the next few operations.
	*/
	setuid(0);
	setgid(0);

	/* Make our spool directory and set its permissions correctly. */
	mkdir(VAR_SPOOL_PPR, S_IRWXU);
	chown(VAR_SPOOL_PPR, euid, egid);
	chmod(VAR_SPOOL_PPR, UNIX_755);

	/*
	** If /var/spool/ppr/queue or /var/spool/ppr/data exist, 
	** will fix their permissions.
	*/
	chown(QUEUEDIR, euid, egid);
	chown(DATADIR, euid, egid);

	/*
	** Give up root privledges forever.  This sets all three user id's and
	** all three group id's.  This is only possible because we executed
	** setuid(0) and setgid(0) above.  (The man pages for various systems
	** seem to contradict one another on whether setgid(0) is necessary.)
	*/
	/* This debug line will end up in the old log file: */
	/* debug("Resigning root privledges by setting uid=%ld and gid=%ld", (long)euid, (long)egid); */
	setgid(egid);
        setuid(euid);
        }

    /*
    ** This daemon was not started by root.  Make sure it was
    ** started by ppr.  (Actually, we make sure it was started
    ** by the same user as it suid's to.)
    */
    else
        {
        if( uid != euid )
            {
            fputs("Only \"ppr\" or \"root\" may start pprd.\n", stderr);
            exit(1);
            }
        }
    }

    /* Fork so as to drop into the background: */
    daemon();

    /* For safety, set some environment variables. */
    #ifndef NO_PUTENV
    putenv("PATH="SHORT_PATH);
    putenv("IFS= \t");
    putenv("SHELL=/bin/sh");
    putenv("HOME="HOMEDIR);
    #endif

    /* Signal handlers for silly stuff. */
    signal(SIGPIPE,sigpipe_handler);
    signal(SIGHUP,sighup_handler);
    signal(SIGTERM,sigterm_handler);

    /* Create the lock file which ensures only one pprd at a time. */
    {
    int lockfilefd;
    char temp[10];
    if( (lockfilefd=open(PPRD_LOCKFILE, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR)) == -1 )
        fatal(1, "can't open \"%s\", errno=%d (%s)", PPRD_LOCKFILE, errno, strerror(errno));
    if(lock_exclusive(lockfilefd, FALSE))
        fatal(1, "pprd already running");
    sprintf(temp, "%ld\n", (long)getpid());
    write(lockfilefd, temp, strlen(temp));  
    }

    /* If there is an old log file, rename it. */
    {
    char newname[MAX_PATH];
    int fh;
    
    if( (fh=open(PPRD_LOGFILE, O_RDONLY)) != -1 )
        {
	close(fh);
	sprintf(newname, "%s.old", PPRD_LOGFILE);
	unlink(newname);
	rename(PPRD_LOGFILE,newname);
	}
    }

    /*
    ** Do this only after identity is fixed (ppr, not root) so
    ** that the owner of the log file is right.  We also wait
    ** until the old log files have been deleted.
    **
    ** Further, we must not call state_update before we call daemon()
    ** because state_update() leaves a file open and daemon() closes
    ** all open file descriptors.
    */
    debug("PPRD startup, pid=%ld", (long)getpid());	/* message for pprd log file */
    state_update("STARTUP");				/* message for queue display programs */

    /* 
    ** Create our directories if they are not present.
    */
    mkdir(QUEUEDIR,S_IRWXU);
    chmod(QUEUEDIR,S_IRWXU);
    mkdir(DATADIR,S_IRWXU);	/* This directory especially must */
    chmod(DATADIR,S_IRWXU);	/* be private. */
    mkdir(MOUNTEDDIR,S_IRWXU);
    chmod(MOUNTEDDIR,S_IRWXU);
    mkdir(ALERTDIR,S_IRWXU);
    chmod(ALERTDIR,S_IRWXU);
    mkdir(LOGDIR,UNIX_755);	/* Others should be able to read */
    chmod(LOGDIR,UNIX_755);	/* the log directory. */

    /*
    ** Create a signal mask which will be needed by the
    ** lock() and unlock() functions.
    */
    sigemptyset(&lock_set);
    sigaddset(&lock_set, SIGALRM);
    sigaddset(&lock_set, SIGCHLD);

    /* Arrange for child termination to be noted .*/
    sig.sa_handler=reapchild;		/* call reapchild() on SIGCHLD */
    sigemptyset(&sig.sa_mask);		/* block no additional sigs */
    sigaddset(&sig.sa_mask,SIGALRM);	/* block alarm signals */
    #ifdef SA_RESTART
    sig.sa_flags=SA_RESTART;		/* restart interupted sys calls */
    #else
    sig.sa_flags=0;
    #endif
    sigaction(SIGCHLD, &sig, (struct sigaction *)NULL);

    /* Load the printers database. */
    DODEBUG_PROGINIT(("loading printers database"));
    load_printers();

    /* Load the groups database. */
    DODEBUG_PROGINIT(("loading groups database"));
    load_groups();

    /* Set up the FIFO. */
    FIFO = open_fifo();

    /* Initialize the queue.  This is likely to start printers. */
    DODEBUG_PROGINIT(("initializing the queue"));
    initialize_queue();    

    /* Arrange for alarm clock to tick every TICK_INTERVAL seconds. */
    sig.sa_handler = tick;		/* call tick() on SIGALRM */ 
    sigemptyset(&sig.sa_mask);		/* clear list of blocked signals */
    sigaddset(&sig.sa_mask, SIGCHLD);	/* block SIGCHLD */
    #ifdef SA_RESTART
    sig.sa_flags = SA_RESTART;		/* restart interupted system calls */
    #else
    sig.sa_flags = 0;
    #endif
    sigaction(SIGALRM, &sig, NULL);
    alarm(TICK_INTERVAL);

    /*
    ** Main Loop
    **
    ** This daemon terminates only if it is killed,
    ** so the condition on this loop is always TRUE.
    */
    while(TRUE)
        {
        char command[256];		/* buffer for received command */
        
        #if DEBUG_PPRDMAIN
        debug("top of main loop");
        #endif

	/* Get a line from the FIFO. */
        if(fgets(command,256,FIFO)==(char*)NULL)
            {
            fatal(0,"fgets() failed in main(), errno=%d (%s)", errno, strerror(errno) );
            }

        if( strlen(command)==0 || command[strlen(command)-1]!='\n' )
            {
            error("ignoring misformed command from pipe");
            continue;
            }

        /* Remove the line feed which terminates the command. */
        command[strlen(command)-1]=(char)NULL;

        #if DEBUG_PPRDMAIN
        debug("command: %s",command);
        #endif

        switch(command[0])          /* examine the first character */
            {                           
            case 'j':                   /* a print job */
                ppr_new_job(command);
                break;

            case 'l':                   /* list print jobs */
		myalloc_checkpoint();
                ppop_list(command);
                myalloc_assert(0);
                break;

            case 's':                   /* show printer status */
		myalloc_checkpoint();
                ppop_status(command);
                myalloc_assert(0);
                break;

            case 't':                   /* starT a printer */
		myalloc_checkpoint();
                ppop_startstop_printer(command,0);
                myalloc_assert(0);
                break;

            case 'p':                   /* stoP a printer */
		myalloc_checkpoint();
                ppop_startstop_printer(command,1);
                myalloc_assert(0);
                break;

            case 'P':                   /* stoP a printer, wait */
		myalloc_checkpoint();
                ppop_startstop_printer(command,129);
                myalloc_assert(0);
                break;

            case 'b':                   /* stop printer with a Bang */
		myalloc_checkpoint();
                ppop_startstop_printer(command,2);
                myalloc_assert(0);
                break;

            case 'h':                   /* place job on hold */
		myalloc_checkpoint();
                ppop_holdrelease_job(command,0);
                myalloc_assert(0);
                break;

            case 'r':                   /* release a job */
		myalloc_checkpoint();
                ppop_holdrelease_job(command,1);
                myalloc_assert(0);
                break;

            case 'c':                   /* cancel */
                ppop_cancel_or_purge(command);
                break;

            case 'f':                   /* media */
		myalloc_checkpoint();
                ppop_media(command);
                myalloc_assert(0);
                break;

            case 'M':                   /* mount media */
                ppop_mount(command);
                break;

            case 'A':                   /* accept for printer or group */
		myalloc_checkpoint();
                ppop_accept_reject(command,1);
                myalloc_assert(0);
                break;

            case 'R':                   /* reject for printer or group */
		myalloc_checkpoint();
                ppop_accept_reject(command,0);
                myalloc_assert(0);
                break;
                
            case 'D':                   /* show destinations */
		myalloc_checkpoint();
                ppop_dest(command);
                myalloc_assert(0);
                break;

            case 'm':                   /* move job(s) */
		myalloc_checkpoint();
                ppop_move(command);
                myalloc_assert(0);
                break;

            case 'U':                   /* rush job */
		myalloc_checkpoint();
                ppop_rush(command);
                myalloc_assert(0);
                break;

	    case 'N':			/* new printer or group config */
		if(command[1]=='P')
		    new_printer_config(&command[3]);
		else
		    new_group_config(&command[3]);
		break;
		
	    case 'n':			/* Nag operator by email */
		myalloc_checkpoint();
	    	nag();
	    	myalloc_assert(0);
	    	break;

            default:                    /* throw away anything else */
                error("unknown command: %s",command);
                break;
            }
        } /* end of endless while() loop */
    } /* end of main() */

/* end of file */
