/*
** ~ppr/src/ppr/ppr_rcache.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Resource Cache routines.
**
** These routines are called when "%%BeginResource:" and "%%EndResource"
** comments are seen.  The routines may decide to put the resource
** in the cache.  If the strip resources switch (-S) was used, the
** resource will be removed from the incoming file and will later be
** re-inserted from the cache.
**
** This file was last modified 6 March 1996.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "ppr.h"
#include "ppr_exits.h"

extern int strip_resources;		/* in ppr_main.c */

static char tcache_name[MAX_PATH];	/* temporary file name */
static char cache_name[MAX_PATH];	/* name of final cache file */
static mode_t cache_mode;		/* Unix permissions of cache file */

static int is_font;			/* fonts receive special treatment in end_resource() */
static int merging;			/* merging type 1 and type 42 */

/*
** This routine is called by _clean_getline() in ppr_old2new.c in order
** to allow the resource cache to examine the start of a new resource. 
** It is very important that nest_level()==1 when this routine is called;
** it must not be called at the start of a nested resource.
*/
void start_resource(void)
    {
    char *type = tokens[1];	/* DSC comment type of resource */
    char *name = tokens[2];	/* DSC comment name of resource */
    double version;		/* DSC comment version number of resource */
    int revision;		/* DSC comment revision number of resource */
    int c1='%';			/* Used in approving fonts for caching */
    int c2='!';
    struct stat statbuf;
    int is_procset;
    char *ptr;

    if( nest_level() != 1 )
	fatal(PPREXIT_OTHERERR,"ppr_rcache.c: start_resource(): nest_level() != 1\n");

    /* If the type or resource or the resource name is missing, */
    if( type==(char*)NULL || name==(char*)NULL )
        {
        warning(WARNING_SEVERE,"invalid \"%%%%BeginResource:\" line:\n%s",line);
        return;
        }

    /* Set flags indicating which type of resource it is for later use. */
    if(strcmp(type,"procset")==0)
    	{
    	is_procset = TRUE;
    	is_font = FALSE;
    	}
    else if(strcmp(type,"font")==0)
    	{
    	is_procset = FALSE;
    	is_font = TRUE;
    	}
    else
    	{
    	is_procset = FALSE;
    	is_font = FALSE;
    	}

    /*
    ** If this is a procedure set, read the
    ** version and revision numbers.
    */
    if(is_procset)
        {
        /* Read version and revision numbers if present. */
        if( (tokens[3]!=(char*)NULL) && tokens[4]!=(char*)NULL 
                && sscanf(tokens[4],"%d",&revision) )
            {
            version=getdouble(tokens[3]);
            }
        else		/* Proceedure set with missing version & revision! */
            {		/* (Warning is issued elsewhere.) */
            version = 0.0;
            revision = 0;    
            }
        }
    else		/* Non-proceedure set resources don't */
        {		/* need version and revision numbers, we will */
        version = 0.0;	/* just use zeros. */
        revision = 0;
        }

    #ifdef DEBUG_RESOURCES
    printf("start_resource(): %s %s %s %d\n", type, quote(name), dtostr(version), revision);
    #endif

    /* 
    ** If this is a font, eat up line feeds and carriage returns 
    ** and then look to make sure it begins with "%!".
    */
    if(is_font)
    	{
	while( (c1=in_getc())=='\n' || c1=='\r' );  /* don't remove semicolon! */

	c2=in_getc();		/* get second character */
	in_ungetc(c2);		/* put it back */
	
	in_ungetc(c1);		/* put back the character which was not blank */

	if(c1 != '%' || c2 != '!')
	    warning(WARNING_PEEVE,"dubious font downloaded in print job");
    	}

    /*
    ** If this is not a no version proceedure set.
    ** (We don't touch no version procedure sets since they
    ** are errors and hence rather dangerous to mess with.)
    ** Also, we will examine c1 and c2 which contain the first
    ** and second characters of a font or '%','!' if the resource
    ** is not a font.
    */
    if( ( !is_procset || version || revision ) && c1=='%' && c2=='!' )
      	{
      	sprintf(cache_name,CACHEDIR"/%s",type);		/* cache dir name */

      	if( (stat(cache_name,&statbuf) == 0) && (statbuf.st_mode & S_IFDIR) )
            {                                   /* if cache dir present, */
	    #ifdef DEBUG_RESOURCES
	    printf("start_resource(): resources of this type are cachable\n");
	    #endif

	    /*
	    ** If not in cache or TrueType font with only part in cache,
	    */
	    if( (ptr=noalloc_find_cached_resource(type,name,version,revision,(int*)NULL,&cache_mode)) == (char*)NULL 
            		|| (is_font && (merging=truetype_more_needed(cache_mode))) )
            	{
		#ifdef DEBUG_RESOURCES
		printf("start_resource(): resource wanted for cache\n");
		#endif

		/*
		** Build the file name we will eventually give the resource.
		*/
		if(is_procset)
		    sprintf(cache_name, CACHEDIR"/procset/%s-%s-%d", name, dtostr(version), revision);
		else
		    sprintf(cache_name, CACHEDIR"/%s/%s", type, name);

		/* Make sure we are not trying to add a segment to a font in primary cache. */
		if( ptr != (char*)NULL && strcmp(cache_name, ptr) )
		    fatal(PPREXIT_OTHERERR, "Partial Mac TrueType font \"%s\" should not be in primary cache.", ptr);
		    
		/* Build the name of a file to put it in temporarily. */
            	sprintf(tcache_name, CACHEDIR"/%s/%ld", type, (long)getpid());

		/* Open the temporary file to hold the resource. */
            	if( (cache_file=fopen(tcache_name,"w")) != (FILE*)NULL )
                    rgrab = 1;		/* "1", not "TRUE"! */
            	else
                    warning(WARNING_SEVERE,"start_resource(): Error opening \"%s\"",tcache_name);

            	} /* if this if() fails, it is already in the cache */

	    /*
	    ** We are already in an if() which is true if we cache resources
	    ** of this type.  Now if we are stripping cached and cachable
	    ** resources:
	    */
            if(strip_resources)         /* If should strip resources, */
            	{ 
		#ifdef DEBUG_RESOURCES
		printf("start_resource(): striping this resource out\n");
		#endif

            	resource(REREF_REMOVED,tokens[1],2);	/* Mark as removed. */

            	if(is_procset)   			/* procsets get version/rev */
                    fprintf(text,"%%%%IncludeResource: %s %s %s %d\n",type,quote(name),dtostr(version),revision);
            	else                            	/* other types don't */
                    fprintf(text,"%%%%IncludeResource: %s %s\n",type,quote(name));

            	if(rgrab)		/* Since we are stripping, if we are grabbing */
                    rgrab = 2;		/* we will not return with the 1st line */

            	while( nest_level() && ! logical_eof)   /* until end of resource, */
                    clean_getline();    /* recursively call clean_getline() */

            	clean_getline();        /* and leave a fresh line in line[] */
            	}

	    } /* If this if() fails, we don't cache this kind. */

	} /* If this fails if this is a no version procedure set. */

    #ifdef DEBUG_RESOURCES
    printf("start_resource() done, rgrab = %d\n", rgrab);
    #endif
    } /* end of start_resource() */

/*
** This routine is called by _clean_getline() in ppr_old2new.c
** in order to tell the cache of the end of a resource.
** It is very important that nest_level()==1 when this routine is called; 
** it must not be called at the end of a nested resource.
*/
void end_resource(void)
    {
    #ifdef DEBUG_RESOURCES
    printf("end_resource(), rgrab=%d, is_font=%d, merging=%d\n", rgrab, is_font, merging);
    #endif

    /*
    ** If we are not copying any part of this file into the cache,
    ** we don't have anything to do here.
    */
    if(rgrab==0)
	return;

    /* Sanity check. */
    if( nest_level() != 1 )
	fatal(PPREXIT_OTHERERR,"ppr_rcache.c: end_resource(): nest_level() != 1\n");

    /* Close the file into which we were collecting the resource. */
    fclose(cache_file);

    /* Set the flag so no attempt will be made to add more lines. */
    rgrab = 0;				/* 0, not "TRUE"! */

    /* If this is a font, */
    if( is_font )
    	{
	char *fontname;
	
	/* set fontname to point to the file name part of old */
	if( (fontname = strrchr(cache_name,'/')) == (char*)NULL )
	    fatal(PPREXIT_OTHERERR,"ppr_rcache.c: end_resource(): invalid cache_name");

	/* Have we decided to try to merge Type1 and Type42 versions of the font? */
	if(merging)
	    {
	    truetype_merge_fonts(fontname, cache_name, tcache_name);
    	    }

	/* If not, just set the flags to indicate whether this is a Mac TT font. */
    	else
	    {
	    if( truetype_set_fontmode(tcache_name) != -1 )	/* If valid font, */
		{
		rename(tcache_name, cache_name);		/* move into place */
		}
	    else						/* otherwise, */
		{
	    	unlink(tcache_name);				/* discard it. */
		warning(WARNING_SEVERE,"Invalid Macintosh TrueType PostScript font discarded");
		}
	    }
    	}

    /*
    ** Not a font, must be some other kind or resource, just
    ** move it into place.
    */
    else
    	{
	rename(tcache_name, cache_name);	/* move to final name */
	}
	
    } /* end of end_resource() */

/* 
** This routine is called if nest_level() is non-zero at the end of the job.
** It gets rid of any resource cache file we were trying to create.
** This is done because a resource which does not end before the end
** of the file is corrupt.
**
** This is also called by file_cleanup().
*/
void abort_resource(void)
    {
    if(rgrab)
        {
        fclose(cache_file);
        unlink(tcache_name);
        }
    } /* end of abort_resource() */

/* end of file */
