/*
** ~ppr/src/ppr/ppr_nest.c
** Copyright 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** The functions in this module have a role in interpreting nested structures
** in DSC complient PostScript documents.  Nested structures include
** resources and documents.
**
** This also contains functions which track which part of the outermost 
** document is being parsed.
** 
** This file was last modified 23 October 1996.
*/

#include "global_defines.h"
#include "ppr.h"
#include "ppr_exits.h"
#include "ppr_gab.h"

/*=========================================================================
** Code to keep track of what sort of nested structure we are inside 
** of now.
=========================================================================*/

static int level = 0;
static int available_levels = 0;
static int *nest_stack = (int*)NULL;

/*
** This routine returns a string representing the type of
** a nesting level.
*/
static const char *str_nest_types(int nesttype)
    {
    switch(nesttype)
    	{
    	case NEST_RES: return "resource";
    	case NEST_DOC: return "document";
    	case NEST_BADEPS: return "bad EPS";
	default: return "<INVALID>";
    	}
    } /* end of str_nest_types() */

/*
** This function is called whenever a comment which indicates a new level
** of nesting is encountered.
*/
void nest_push(int leveltype, const char *name)
    {
    level++;

    if(gab_mask & GAB_STRUCTURE_NESTING)
	printf("nesting: new level: type=\"%s\", new level=%d (%s)\n", str_nest_types(leveltype), level, name != (char*)NULL ? name : "<NULL>");

    /* If we need more space, then allocate it. */
    if( level > available_levels )
	{
	available_levels += 10;
    	nest_stack = (int*)myrealloc(nest_stack, available_levels, sizeof(int));
    	}

    nest_stack[level-1] = leveltype;
    } /* end of nest_push() */

/*
** This is called whenever a comment which indicates the end of a nested
** structure is encountered.
*/
void nest_pop(int leveltype)
    {
    if(--level < 0)
    	fatal(PPREXIT_OTHERERR, "ppr_nest.c: nest_pop(): stack underflow");

    if(gab_mask & GAB_STRUCTURE_NESTING)
    	printf("nesting: end level: type=\"%s\", popped by \"%s\", new level=%d\n", str_nest_types(nest_stack[level]), str_nest_types(leveltype), level);

    if(leveltype != nest_stack[level])
    	{
	warning(WARNING_SEVERE, "Bad nesting, %s terminated as though it were a %s", str_nest_types(nest_stack[level]), str_nest_types(leveltype));
    	level++;		/* try to ignore the problem by ignoring the offending line */
    	}
    } /* end of nest_pop() */
    
/*
** This function returns an integer which indicates how many nested 
** structures the current line is enclosed within.
*/
int nest_level(void)
    {
    return level;
    } /* end of nest_level() */

/*
** This returns a code which indicates the type of the innermost
** enclosing nested structure.
*/
int nest_inermost_type(void)
    {
    if(level == 0)
        return NEST_NONE;
    else
	return nest_stack[level-1];
    } /* end of nest_inermost_type() */
    
/*=======================================================================
** Functions to keep track of which section of the outermost document
** we are currently in.
=======================================================================*/

static int outermost = OUTERMOST_UNDEFINED;

/*
** This routine returns a string representing the type of
** a nesting level.
*/
static const char *str_outermost_types(int nesttype)
    {
    switch(nesttype)
    	{
    	case OUTERMOST_UNDEFINED: return "UNDEFINED";
    	case OUTERMOST_HEADER_COMMENTS: return "header comments";
    	case OUTERMOST_PROLOG: return "prolog";
    	case OUTERMOST_DOCDEFAULTS: return "document defaults";
	case OUTERMOST_DOCSETUP: return "document setup";
	case OUTERMOST_SCRIPT: return "script";
	case OUTERMOST_TRAILER: return "trailer";
	default: return "<INVALID>";
    	}
    } /* end of str_outermost_types() */

/*
** This function is called to mark the start of a section of 
** the outermost document.
*/
void outermost_start(int sectioncode)
    {
    if(gab_mask & GAB_STRUCTURE_SECTIONS)
    	printf("section start: %s\n", str_outermost_types(sectioncode));

    outermost = sectioncode;
    } /* end of outermost_start() */

/*
** This function is called to mark the end of a section of the outermost
** document.  It will get upset if we end a section that we did not begin.
** If this happens it is a bug in PPR.  We make one exception:  the
** prolog may ommit "%%BeginProlog".
*/
void outermost_end(int sectioncode)
    {
    if(gab_mask & GAB_STRUCTURE_SECTIONS)
    	printf("section end:   %s\n", str_outermost_types(sectioncode));

    if( sectioncode != outermost && !(sectioncode == OUTERMOST_PROLOG && outermost == OUTERMOST_UNDEFINED) )
    	warning(WARNING_SEVERE, "PPR bug: outermost_end(): end %s in %s", str_outermost_types(sectioncode), str_outermost_types(outermost) );
    	
    outermost = OUTERMOST_UNDEFINED;
    } /* end of outermost_end() */
    
/*
** This function returns a code which indicates which part of the
** outermost document is being parsed.
*/
int outermost_current(void)
    {
    return outermost;
    } /* end outermost_current() */
    
/* end file file */
