/*
** ~ppr/src/ppr/ppr_media.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 20 January 1997.
*/

/*
** Media comment routines.
**
** One of these is called every time a media comment is encountered
** in the input file, others are called when it is necessary to dump
** lists of the required media into the queue file.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>		/* NT with Hippix needs this */
#include <string.h>
#include "ppr.h"
#include "ppr_exits.h"
#include "respond.h"

extern char *default_media;		/* in ppr_main.c */
extern struct Media guess_media;	/* in ppr_main.c */

/*
** Record a reference to a media type requirment.  This will be 
** as a result of a "%%DocumentMedia:" or "%%PageMedia:" comment.
**
** The parameter "reftype" is either MREF_DOC or MREF_PAGE.
*/
void media(int reftype, int first)
    {
    if(reftype==MREF_DOC)	/* if document wide media reference */
        {
        struct Media *newmedia;	/* pointer to new media structure */

	/*
	** Make sure the comment has all the proper
	** parameters.
	*/
        if(tokens[first] == (char*)NULL
                || tokens[first+1] == (char*)NULL
                || tokens[first+2] == (char*)NULL
                || tokens[first+3] == (char*)NULL
                || tokens[first+4] == (char*)NULL
                || tokens[first+5] == (char*)NULL )
            {
            warning(WARNING_SEVERE, "invalid \"%%%%DocumentMedia:\" line, insufficient parameters");
            return;
            }

	/* Make sure the things array is not full. */
	things_space_check();

	/*
	** Allocate space for and fill in a new
	** media record.
	*/
        newmedia = (struct Media*)myalloc(1,sizeof(struct Media));
        ASCIIZ_to_padded(newmedia->medianame,tokens[first],sizeof(newmedia->medianame));
        newmedia->width = getdouble(tokens[first+1]);
        newmedia->height = getdouble(tokens[first+2]);
        newmedia->weight = getdouble(tokens[first+3]);
        ASCIIZ_to_padded(newmedia->colour,tokens[first+4],sizeof(newmedia->colour));
        ASCIIZ_to_padded(newmedia->type,tokens[first+5],sizeof(newmedia->type));

	/*
	** Set the next available things[] array entry
	** to point to our newly created media record.
	*/
        things[thing_count].th_ptr = (void*)newmedia;
        things[thing_count].th_type = TH_MEDIA;
        things[thing_count].R_Flags = MREF_DOC;
        thing_count++;
        }

    /* If page level media reference, */
    else if(reftype == MREF_PAGE)
        {
        char medianame[MAX_MEDIANAME];
        int x;
	int found=FALSE;

	ASCIIZ_to_padded(medianame,tokens[first],sizeof(medianame));

	for(x=0; x < thing_count; x++)		/* Look thru the whole thing list */
	    {
	    if(things[x].th_type != TH_MEDIA)	/* Ignore those that are not media */
		continue;

	    if(padded_cmp(medianame,((struct Media*)things[x].th_ptr)->medianame,MAX_MEDIANAME))
	    	{
		found = TRUE;
		things[x].R_Flags |= MREF_PAGE;
		set_thing_bit(x);	    	
	    	}
	    }

	if( ! found )
	    {
	    warning(WARNING_SEVERE,"Page %d requires undefined media \"%s\"",pagenumber,tokens[first]);
	    }

        }

    /* if unrecognized media reference type code */
    else
        {
        fatal(PPREXIT_OTHERERR,"media(): invalid reference type: %d",reftype);
        }
    } /* end of media() */

/*
** Dump media catalog for the whole document.
** That is, produce a "%%DocumentMedia:" comment.
**
** This write to ofile rather than directly to FILE *comments because
** if we split the job, ppr_split.c:split_job() will copy the comments
** file, leaving out all "%%DocumentMedia:" line and then call this
** function for each new -comments file.
*/
void dump_document_media(FILE *ofile, int frag)
    {
    int x;
    int number=0;               
    struct Media *m;
    char medianame[sizeof(m->medianame)+1];
    char colour[sizeof(m->colour)+1];
    char type[sizeof(m->type)+1];

    for(x=0;x<thing_count;x++)
        {
        if(things[x].th_type==TH_MEDIA)
            {
	    if( ! is_thing_in_current_fragment(x, frag) ) /* Ignore media if not in */
		continue;				/* this job fragment. */

            if(number++)
                fprintf(ofile,"%%%%+ ");
            else
                fprintf(ofile,"%%%%DocumentMedia: ");

            m=(struct Media *) things[x].th_ptr;
            padded_to_ASCIIZ(medianame,m->medianame,sizeof(m->medianame));
            padded_to_ASCIIZ(colour,m->colour,sizeof(m->colour));
            padded_to_ASCIIZ(type,m->type,sizeof(m->type));
            fprintf(ofile,"%s %s ",medianame, dtostr(m->width) );
            fprintf(ofile,"%s ", dtostr(m->height) );
            fprintf(ofile,"%s %s ",dtostr(m->weight),quote(colour));
            fprintf(ofile,"%s\n",quote(type) );
            }
        }
    } /* end of dump_document_media() */

/*
** Write a "%%PageMedia:" comment if necessary.
** This is called once for each page.
*/
void dump_page_media(void)
    {
    int x;
    struct Media *m;
    char medianame[sizeof(m->medianame)+1];

    for(x=0;x<thing_count;x++)
        {
        if( (things[x].th_type==TH_MEDIA) && (things[x].R_Flags & MREF_PAGE) )
            {
            m=(struct Media *) things[x].th_ptr;
            padded_to_ASCIIZ(medianame,m->medianame,sizeof(m->medianame));

            fprintf(page_comments,"%%%%PageMedia: %s\n",medianame);

	    things[x].R_Flags &= (~MREF_PAGE);
            break;
            }
        }
    } /* end of dump_page_media() */

/*
** Look one media up in the media database and write an appropriate 
** "Media:" line into the queue file.
**
** This function is called by write_media_lines() below.
**
** If this function failes, it returns -1.  If it suceeds, it increments the 
** number pointed to by match_count and returns 0.
**
** This function must always leave the media file at its start.
*/
int _write_media_line(FILE *qfile, FILE *mfile, struct Media *m, int *match_count)
    {
    struct Media km;				/* a known media record */
    char medianame[MAX_MEDIANAME+1];		/* ASCIIZ version of matched media */
    char psmedianame[MAX_MEDIANAME+1];
    int success=FALSE;
            
    /* Make an ASCIIZ version if the media name from the PostScript comment. */
    padded_to_ASCIIZ(psmedianame,m->medianame,MAX_MEDIANAME);

    /* 
    ** Search the media library file.
    **
    ** On this try we will require that the width and height
    ** match exactly, that the weight match exactly or that
    ** the document not specify the weight, that the colour
    ** match exactly or the document not specify the colour
    ** and the actual media colour be white, and that
    ** the media type be an exact match. 
    */
    while(fread(&km,sizeof(struct Media),1,mfile))			/* 1st try */
	{                              
	#ifdef DEBUG_MEDIA_MATCHING
	padded_to_ASCIIZ(medianame,km.medianame,sizeof(km.medianame));
	printf("Trying media %s\n",medianame);
	#endif

	if( m->height==km.height && m->width==km.width			/* if height and with are correct */
		&& ( m->weight==0                         		/* and weight unspecified */
		    || m->weight==km.weight )				/* or weight correct */
		&& ( (m->colour[0]==' '					/* and colour unspecified */
		    && strncmp("white ",km.colour,6)==0 )		/* matching white */
		    || padded_cmp(m->colour,km.colour,MAX_COLOURNAME) ) /* or correct */
		&& padded_cmp(m->type,km.type,MAX_TYPENAME) ) 		/* and type correct */
	    {
	    padded_to_ASCIIZ(medianame,km.medianame,			/* add this media */
                          sizeof(km.medianame));           		/* type */
	    fprintf(qfile,"Media: %s %s\n",medianame,psmedianame);  	/* to queue file */
	    #ifdef DEBUG_MEDIA_MATCHING
	    printf("Exact media match: \"%s\" matches \"%s %.1f %.1f %.1f (%.6s) (%.6s)\"\n",
	    	medianame,psmedianame,km.width,km.height,km.weight,km.colour,km.type);
	    #endif
	    (*match_count)++;                           		/* and set flag */
	    success=TRUE;
	    break;                                   			/* and stop search */
            }
	}
	
    rewind(mfile);		/* return to begining of media file */
    if(success) return 0;	/* if we matched above, go to next media */

    /*
    ** We try again to find a match, this time with more liberal
    ** requirements.  This time the height and width must be close,
    ** the weight and colour are not considered, but the type must 
    ** still be an exact match. 
    */
    while( fread(&km,sizeof(struct Media),1,mfile) )		/* 2nd try */
	{							/* anything close in size, and */
	if( abs( (int)(m->height-km.height) ) < 20		/* of the right type */
  			&& abs( (int)(m->width-km.width) ) < 20
  			&& padded_cmp(m->type,km.type,MAX_TYPENAME) )
	    {
	    padded_to_ASCIIZ(medianame,km.medianame,sizeof(km.medianame));
	    fprintf(qfile,"Media: %s %s\n",medianame,psmedianame);
	    #ifdef DEBUG_MEDIA_MATCHING
	    printf("Imperfect media match: \"%s\" matches \"%s\"\n",medianame,psmedianame);
	    #endif
	    (*match_count)++;
	    success=TRUE;		
	    warning(WARNING_SEVERE,"media used is an imperfect match");
	    break;
	    }
	}

    rewind(mfile);		/* return to begining of media file */
    if(success) return 0;	/* if matched above, continue with next media in document */

    /* Abort if the ProofMode is "NotifyMe". */
    if( qentry.attr.proofmode==PROOFMODE_NOTIFYME )
	{
    	respond(RESP_BADMEDIA,(char*)NULL);
    	exit(PPREXIT_NOMATCH);
    	}

    /* 
    ** This is the final try.  We will be very careless.
    ** We will accept any paper without an explicit type (i. e., is blank)
    ** which has a size within half an inch of the desired size. 
    */
    while( fread(&km,sizeof(struct Media),1,mfile) )
	{
	if( abs( (int)(m->height-km.height) ) < 36 && abs( (int)(m->width-km.width) ) < 36 
  		&& km.type[0]==' ' )
	    {
	    padded_to_ASCIIZ(medianame,km.medianame,sizeof(km.medianame));
	    fprintf(qfile,"Media: %s %s\n",medianame,psmedianame);
	    #ifdef DEBUG_MEDIA_MATCHING
	    printf("Very poor media match: \"%s\" matches \"%s\"\n",medianame,psmedianame);
	    #endif
	    (*match_count)++;
	    success=TRUE;		
	    warning(WARNING_SEVERE,"media used is a very poor match");
	    break;
	    }
	}

    rewind(mfile);		/* return to begining of media file */
    if(success) return 0;	/* if matched above, skip to next loop iteration */
    return -1;
    } /* end of _write_media_line() */

/*
** Write the "Media:" lines to the queue file.
** This proceedure is called from write_queue_file().
*/
void write_media_lines(FILE *qfile, int frag)
    {
    FILE *mfile;
    int x;                  /* used for moving through things[] */
    int match_count=0;	    /* "Media:" lines written due to matches */

    #ifdef DEBUG_MEDIA_MATCHING
    printf("write_media_lines()\n");
    #endif

    /* Open the media database file. */
    if( (mfile=fopen(MEDIAFILE,"r")) == (FILE*)NULL )
        fatal(PPREXIT_OTHERERR,"can't open media database \"%s\"",MEDIAFILE);

    for(x=0; x < thing_count; x++)		/* try each thing */
	{
	if(things[x].th_type != TH_MEDIA)	/* Ignore things which are */
	    continue;				/* are not media. */

	if( ! is_thing_in_current_fragment(x,frag) )	/* Ignore media not in */
	    continue;				/* this job fragment. */

	if(_write_media_line(qfile,mfile,(struct Media *)things[x].th_ptr,&match_count))
	    warning(WARNING_SEVERE,"requested media is not in the database");

	#ifdef DEBUG_MEDIA_MATCHING
	printf("match_count=%d\n",match_count);
	#endif
	} /* end of for each `thing' */

    /*
    ** Default media.
    **
    ** If no %%DocumentMedia: comments resulted in matches, use the default
    ** media as modified by information gathered from Feature comments
    ** in the document setup section.
    */
    if(match_count==0)          /* if no valid media found */
	{			/* use the default media adjusted to PageSize etc. */
	char padded_default[MAX_MEDIANAME];
	struct Media def;	/* for reading media */
	int modified=FALSE;

	#ifdef DEBUG_MEDIA_MATCHING
	printf("Resorting to default media\n");
	#endif

	/* Convert the default media name to a padded string. */
	ASCIIZ_to_padded(padded_default,default_media,MAX_MEDIANAME);

	/*
	** Find the entry for the default media.
	**
	** We look for it by name, but what we load is only used
	** as a guide and as a last resort.  If the document selects
	** a radically different page size, we could end up with
	** something quite different.  For example, if the default
	** media is "letter" and there is a comment in the document
	** which indicates that "a4" paper is selected, we will end
	** up selecting the media "a4" if it exists in the media
	** database.
	*/
	while(TRUE)
	    {
	    if( fread(&def,sizeof(struct Media),1,mfile) == 0 )	/* if end of file, */
	    	fatal(PPREXIT_NOMATCH,"default media \"%s\" does not exist",default_media);

	    /* 
	    ** If we have found the default media, then change the
	    ** `DSC comment name' to "NULL" and break out of this loop. 
	    **
	    ** There are a number of reasons we change it to "NULL", not
	    ** the least of these being that once we are thru adjusting
	    ** the page size and such we may not select that
	    ** media at all.
	    */
	    if(memcmp(def.medianame,padded_default,MAX_MEDIANAME)==0)
		{
		ASCIIZ_to_padded(def.medianame,"NULL",MAX_MEDIANAME);
		break;	    
		}
	    } /* end of loop to find the default media */

	/* If *PageSize or *PageRegion is given us a width, change the default */
	if( (guess_media.width!=0.0) && (guess_media.height!=0.0)
		&& ((guess_media.width!=def.width) || (guess_media.height!=def.height)) )
	    {
	    def.width=guess_media.width;
	    def.height=guess_media.height;
	    modified=TRUE;
	    #ifdef DEBUG_MEDIA_MATCHING
	    printf("Default media size modified\n");
	    #endif
	    }

	/* if "*MediaType" or "*PageSize Comm10", etc. */
	if( (guess_media.type[0]!=(char)NULL) && memcmp(guess_media.type,def.type,MAX_TYPENAME) )
	    {
   	    memcpy(def.type,guess_media.type,MAX_TYPENAME);
	    modified=TRUE;
	    #ifdef DEBUG_MEDIA_MATCHING
	    printf("Default media type modified\n");
	    #endif
	    }

	/* *MediaWeight */
	if( (guess_media.weight!=0.0) && (guess_media.weight!=def.weight) )
	    {
	    def.weight=guess_media.weight;
	    modified=TRUE;
	    #ifdef DEBUG_MEDIA_MATCHING
	    printf("Default media weight modified\n");
	    #endif
	    }

	/* *MediaColor */
	if( (guess_media.colour[0]!=(char)NULL) && memcmp(guess_media.colour,def.colour,MAX_COLOURNAME) )
	    {
	    memcpy(def.colour,guess_media.colour,MAX_COLOURNAME);
	    modified=TRUE;
	    #ifdef DEBUG_MEDIA_MATCHING
	    printf("Default media colour modified\n");
	    #endif
	    }

	/* 
	** If the default media was not modified, then just use it.
	*/
	if(!modified)
	    {
	    fprintf(qfile,"Media: %s NULL\n",default_media);
	    }
	/*
	** If the guess_media in any way modified the default media,
	** then look for the modified media. 
	*/
	else
	    {
	    rewind(mfile);
	    if(_write_media_line(qfile,mfile,&def,&match_count))
	    	{
	    	warning(WARNING_SEVERE,"default media as modified does not exist");

		/* Use the unmodified default media. */
	    	fprintf(qfile,"Media: %s NULL\n",default_media);
	    	}
	    }
	} /* end of if(no match) */

    fclose(mfile);              /* close the media library */

    #ifdef DEBUG_MEDIA_MATCHING
    printf("write_media_lines() done\n");
    #endif

    } /* end of write_media_lines() */

/* end of file */

