/*
** ~ppr/src/ppr/ppr_main.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last revised 21 March 1997.
*/

/*
** Main module of the program used to submit jobs to the
** PPR print spooler.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <pwd.h>		/* for /etc/passwd routines */
#include <ctype.h>
#include <string.h>
#include <sys/wait.h>
#include <errno.h>
#include <grp.h>		/* for "ppr" group membership testing */
#include "ppr.h"
#include "userdb.h"
#include "ppr_exits.h"
#include "respond.h"
#include "version.h"		/* for "-v" switch */
#include "ppr_gab.h"

/*
** global variables
*/

/* input line */
char line[MAX_LINE+2];	    	    /* input line plus one, plus NULL */
int line_len;			    /* length of input line in bytes */
int line_overflow;		    /* is lines truncated? true or false */

/* output files */
FILE *comments = (FILE*)NULL;		/* file for header & trailer comments */
FILE *page_comments = (FILE*)NULL;	/* file for page level comments */
FILE *text = (FILE*)NULL;		/* file for remainder of text */
FILE *cache_file;			/* file to copy resource into */
int files_created = FALSE;		/* TRUE once we have created some files, used for cleanup on abort */

/* Information used for determining privledges and such. */
uid_t uid;				/* Unix user id of person submitting the job */
uid_t setuid_uid;			/* uid of spooler owner (ppr) */
gid_t gid;
gid_t setgid_gid;

/* Command line option settings. */
int warning_level = WARNING_SEVERE;	/* these and more serious allowed */
int warning_log = FALSE;		/* set true if warnings should go to log */
int use_authcode = FALSE;		/* true if using authcode line to id */
int preauthorized = FALSE;		/* set true by -A switch */
const char *features[MAX_FEATURES];	/* max features to add */
int features_count = 0;			/* number asked for (number of -F switches?) */
int strip_resources = TRUE;		/* TRUE if should strip those in cache */ 
int ignore_truncated = FALSE;		/* TRUE if should discard without %%EOF */
int ppr_respond_by = PPR_RESPOND_BY_STDERR;
int nofilter_hexdump = FALSE;		/* don't encourage use of hexdump when no filter */
char *filter_options = (char*)NULL;	/* contents of -o switch */
int unlink_jobfile = FALSE;		/* Was the -U switch used? */
int use_username = FALSE;		/* User username instead of comment as default For: */
int TrueTypeQuery = TT_UNKNOWN;		/* for ppr_mactt.c */
unsigned int gab_mask = 0;		/* Mask to tell what to gab about. */

/* -R switch command line option settings. */
int read_copies = FALSE;		/* TRUE if should read copies from file */
int read_duplex = TRUE;			/* TRUE if we should guess duplex */
int read_duplex_enforce = FALSE;	/* TRUE if we should insert duplex code after determination */
int current_duplex = DUPLEX_SIMPLEX;	/* set when we read duplex */
int read_signature = TRUE;		/* TRUE if we should implement signature features */
int read_for = FALSE;			/* Pay attention to "%%For:" lines? */

/* Code numbers for library routines to pass to exit()
   in case of fatal errors. */
const int lib_memory_fatal = PPREXIT_OTHERERR;
const int lib_tokenize_fatal = PPREXIT_OTHERERR;
const int lib_misc_fatal = PPREXIT_OTHERERR;

/* odds and ends */
FILE *FIFO = (FILE*)NULL;		/* streams library thing for pipe to pprd */
int rgrab = 0;				/* 0, 1 or 2 for grabing a resource (see ppr_rcache.c) */
struct QFileEntry qentry;		/* structure in which we build our queue entry */
int option_hold = FALSE;		/* should it be held when it is submitted? */
int option_show_id = FALSE;
int pagenumber = 0;			/* count of %%Page: comments */
char *AuthCode = (char*)NULL;		/* clear text of authcode */
int auth_needed;			/* TRUE if use_authcode or dest protect */
int eof_comment_present = FALSE;	/* TRUE if file ended with ``%%EOF'' */
char starting_directory[1024];		/* Not MAX_PATH!  (MAX_PATH might be just large enough for PPR file names) */
static const char *charge_to_switch_value = (char*)NULL;
static const char *default_ForLine = (char*)NULL;

/* default media */
#ifndef A4
const char *default_media = "letter";	/* media to start with if no media comment */
#else
const char *default_media = "a4";	/* media to start with if no media comment */
#endif
struct Media guess_media;		/* used to gather information to guess proper media */

/* Page factors */
int duplex_pagefactor;			/* pages per sheet multiple due to duplex */

/* Array of things (resources, requirements, etc.). */
struct Thing *things = (struct Thing *)NULL;	/* fonts, media and such */
int thing_count = 0;				/* number of entries used */

/* external globals */
extern int found_ProofMode;		/* in ppr_dscdoc.c, set by doopt() */
extern int found_Title;			/* in ppr_dscdoc.c, set by doopt() */

/* Table of hacks which may be enabled or disabled with the -H switch. */
struct {char *name; unsigned int bit;} hack_table[]=
{
{"keepinfile", HACK_KEEPINFILE},
{"transparent", HACK_TRANSPARENT},
{"badeps", HACK_BADEPS},
{(char*)NULL, 0}
};

/* Table of things to gab about. */
struct {char *name; unsigned int bit;} gab_table[]=
{
{"infile:autotype", GAB_INFILE_AUTOTYPE},
{"infile:filter", GAB_INFILE_FILTER},
{"structure:nesting", GAB_STRUCTURE_NESTING},
{"structure:sections", GAB_STRUCTURE_SECTIONS},
{(char*)NULL, 0}
};

/*=========================================================================
** Error Routines
=========================================================================*/

/*
** Handle fatal errors.
** Print a message and exit.
*/
void fatal(int exitval, const char *message, ... )
    {
    va_list va;
    char errbuf[256];

    fflush(stdout);			/* In case we will be printing on stderr */

    va_start(va, message);		/* Format the error message */
    vsprintf(errbuf, message, va);	/* as a string. */
    va_end(va);

    if(exitval == PPREXIT_SYNTAX)
	respond(RESP_FATAL_SYNTAX, errbuf);
    else
	respond(RESP_FATAL, errbuf);

    exit(exitval);			/* And give our parent a hint. */
    } /* end of fatal() */

/*
** The libpprdb library routines require this
** function to be present.  We have implemented
** it so it does basicaly the same thing as
** fprintf(stderr,...).
*/
void error(const char *message, ... )
    {
    va_list va;

    va_start(va,message);		/* Format the error message */
    vfprintf(stderr,message,va);	/* as a string. */
    va_end(va);
    } /* end of error() */

/*
** Call this function to issue a warning.  It is printed on
** stderr or sent to the log file, depending on whether
** or not the ``-w log'' switch was used.
*/
void warning(int level, const char *message, ... )
    {
    va_list va;
    char wfname[MAX_PATH];
    FILE *wfile;

    if(qentry.destnode == (char*)NULL)
        fatal(PPREXIT_OTHERERR, "ppr_main.c: warning(): qentry.destnode == (char*)NULL");

    if(level < warning_level)	/* if warning level too low */
        return;			/* do not print it */

    if(warning_log)		/* if warnings are to go to log file, */
	{			/* open this job's log file */
	sprintf(wfname, DATADIR"/%s:%s-%d.0(%s)-log",
        	qentry.destnode,
        	qentry.destname, qentry.id, qentry.homenode);
	if( (wfile=fopen(wfname,"a")) == (FILE*)NULL )
	    {
	    fprintf(stderr,"Failed to open log file, using stderr.\n");
            wfile = stderr;	/* if fail, use stderr anyway */
            }
	}
    else			/* if not logging warnings, */
	{			/* send them to stderr */
	wfile = stderr;
	}

    va_start(va,message);
    fprintf(wfile,"WARNING: ");	/* now, print the warning message */
    vfprintf(wfile,message,va);
    fprintf(wfile,"\n");
    va_end(va);

    if(wfile != stderr)		/* if we didn't use stderr, */
	fclose(wfile);		/* close what we did use */
    else			/* if was stderr, */
	fflush(stderr);		/* empty output buffer */
				/* (important for papsrv) */
    } /* end of warning() */

/*==================================================================== 
** Input file handling routines.  Most of these have been moved
** to "ppr_infile.c".
====================================================================*/
/*
** copy data between %%Begin(End)Data: or %%Begin(End)Binary: comments
*/                                                 
void copy_data(FILE *outfile)
    {
    int c;
    long int len;

    fprintf(outfile,"%s\n",line);           /* Copy thru to output. */
    if(rgrab)                               /* If caching a resource now */
        fprintf(cache_file,"%s\n",line);    /* then put in the cache file. */

    sscanf(tokens[1],"%ld",&len);           /* read the number of items */

    if( strcmp(tokens[3],"Lines")==0 )      /* find if lines or bytes */
        {                                   /* Lines: */
        while(len--)
            {
            getline();				/* get a line */
            fprintf(outfile,"%s\n",line);	/* send it */
            if(rgrab)				/* posibly cache it */
                fprintf(cache_file,"%s\n",line);
            }
        }
    else                                    /* Bytes: */
        {
        if(rgrab)
            while(len--)
                {
                c=in_getc();
                fputc(c,outfile);
                if(rgrab)
                    fputc(c,cache_file);
                }
        else
            while(len--)
                {
                fputc(in_getc(),outfile);
                }
        }
    } /* end of copy_data() */

/*=========================================================================
** Miscelainious support routines for main().
=========================================================================*/
/*
** Write the Feature: lines to the queue file.
*/
void write_feature_lines(FILE *qfile)
    {
    int x;

    for(x=0;x<features_count;x++)
        fprintf(qfile,"Feature: %.*s\n",MAX_QFLINE-9,features[x]);

    } /* end of write_features() */

/*
** Assert that a variable contains a valid value.
** Invalid values are those that contains "\n"
** or those that consist entirely of whitespace
** if blankok is false.  Notice that a NULL value
** is always ok.
**
** We must filter these out now because if they
** get into the queue file the queue file reading
** routines will choke on them.
*/
static void assert_ok_value(const char *value, int blankok, const char *name)
    {
    if(value != (char*)NULL)
    	{
    	if(strchr(value, '\n') != (char*)NULL)
	    fatal(PPREXIT_OTHERERR, "LF in %s", name);
    	if(! blankok && strspn(value, "\t ") == strlen(value))
    	    fatal(PPREXIT_OTHERERR, "%s is blank", name);
	}
    }

/*
** Create and fill the queue file.
** Return -1 if we run out of disk space. 
*/
int write_queue_file(struct QFileEntry *qentry, int fragment, int pages, int hold)
    {
    char tempstr[MAX_PATH];
    int fd;
    FILE *Qfile;
    int len;

    /* This code looks for things that could make a mess of the queue file. */
    assert_ok_value(qentry->ForLine, FALSE, "qentry->ForLine");
    assert_ok_value(qentry->charge_to, FALSE, "qentry->charge_to");
    assert_ok_value(qentry->Title, FALSE, "qentry->Title");
    assert_ok_value(qentry->Draft, FALSE, "qentry->Draft");
    assert_ok_value(qentry->Creator, FALSE, "qentry->Creator");
    assert_ok_value(qentry->Routing, FALSE, "qentry->Routing");
    assert_ok_value(qentry->lpqFileName, FALSE, "qentry->lpqFileName");
    assert_ok_value(qentry->response_method, FALSE, "qentry->response_method");
    assert_ok_value(qentry->response_address, FALSE, "qentry->response_address");
    assert_ok_value(qentry->responder_options, TRUE, "qentry->responder_options");

    /* Construct the queue file name. */
    sprintf(tempstr,"%s/%s:%s-%d.%d(%s)", QUEUEDIR,
    	qentry->destnode, qentry->destname, qentry->id, fragment, qentry->homenode);

    if( (fd=open(tempstr, O_WRONLY | O_CREAT | O_EXCL, hold ? (BIT_JOB_BASE | BIT_JOB_HELD) : BIT_JOB_BASE )) == -1 )
        fatal(PPREXIT_OTHERERR, "can't open queue file \"%s\", errno=%d (%s)", tempstr, errno, strerror(errno) );
    if( (Qfile=fdopen(fd, "w")) == (FILE*)NULL )
    	fatal(PPREXIT_OTHERERR, "write_queue_file(): fdopen() failed");

    /* This will be useful once distributed printing is implemented. */
    fprintf(Qfile, "PPRVersion: %s\n", SHORT_VERSION);

    /* This is so we will know when the job was submitted. */
    fprintf(Qfile, "Time: %ld\n", qentry->time);

    fprintf(Qfile, "User: %ld %s %s\n",
    	qentry->user,			/* Unix user id */
    	qentry->username,		/* Unix user name */
	qentry->proxy_for != (char*)NULL ? qentry->proxy_for : "");
	
    fprintf(Qfile, "Priority: %d\n", qentry->priority);

    fprintf(Qfile, "For: %.*s\n", MAX_QFLINE-5, qentry->ForLine);

    if(qentry->charge_to != (char*)NULL)
	fprintf(Qfile, "Charge-To: %.*s\n", MAX_QFLINE-11, qentry->charge_to);

    if(qentry->Title != (char*) NULL )
        fprintf(Qfile, "Title: %.*s\n", MAX_QFLINE-7, qentry->Title);

    if(qentry->Draft != (char*)NULL)
	fprintf(Qfile, "Draft-Notice: %.*s\n", MAX_QFLINE-14, qentry->Draft);

    if(qentry->Creator != (char*) NULL )
        fprintf(Qfile, "Creator: %.*s\n", MAX_QFLINE-9, qentry->Creator);

    if(qentry->Routing != (char*)NULL )
        fprintf(Qfile, "Routing: %.*s\n", MAX_QFLINE-9, qentry->Routing);

    if(qentry->lpqFileName != (char*)NULL )
    	fprintf(Qfile, "lpqFileName: %.*s\n", MAX_QFLINE-10, qentry->lpqFileName);
    	
    fprintf(Qfile, "Banners: %d %d\n", qentry->do_banner, qentry->do_trailer);

    fprintf(Qfile, "Response: %.*s %.*s %.*s\n", MAX_RESPONSE_METHOD, qentry->response_method,
	MAX_RESPONSE_ADDRESS, qentry->response_address,
	MAX_RESPONDER_OPTIONS, qentry->responder_options != (char*)NULL ? qentry->responder_options : "");

    fprintf(Qfile, "Attr: %d %.1f %d %d %d %d %d %d %d %d %d %ld %ld %d\n",
                qentry->attr.langlevel,
                qentry->attr.DSClevel,
                pages,
                qentry->attr.pageorder,
                qentry->attr.prolog,
                qentry->attr.docsetup,
                qentry->attr.script,
                qentry->attr.extensions,
                qentry->attr.pagefactor,
                qentry->attr.orientation,
                qentry->attr.proofmode,
                qentry->attr.input_bytes,
                qentry->attr.postscript_bytes,
                qentry->attr.parts
                );

    fprintf(Qfile,"Opts: %d %d %d %d %u\n",
                qentry->opts.binselect,
                qentry->opts.copies,
                qentry->opts.collate,
                qentry->opts.keep_badfeatures,
                qentry->opts.hacks);

    fprintf(Qfile,"N_Up: %d %d %d %d\n",
		qentry->N_Up.N,			/* virtual pages on each side of sheet */
    		qentry->N_Up.borders,		/* should we print borders? */
    		qentry->N_Up.sigsheets,		/* how many sheets per signature? (0=no signature printing) */
		qentry->N_Up.sigpart);		/* Fronts, backs, both */
					
    fprintf(Qfile,"Portion: %d %d %d %d\n",
    		qentry->portion.odd,
    		qentry->portion.even,
    		qentry->portion.start,
    		qentry->portion.stop);

    write_media_lines(Qfile,fragment);		/* add "Media:" lines */
    fprintf(Qfile, "EndMedia\n");

    write_resource_lines(Qfile,fragment);	/* add Res: lines */
    fprintf(Qfile, "EndRes\n");

    write_requirement_lines(Qfile,fragment);	/* add Req: lines */
    fprintf(Qfile, "EndReq\n");

    write_feature_lines(Qfile);			/* add "Feature:" lines */

    /* 
    ** We write this line in a special way so as to
    ** detect disk full conditions.  Rather crude. 
    */
    fflush(Qfile);
    len = write(fileno(Qfile), "EndSetups\n",10);
                                                   
    fclose(Qfile);
    
    /* Now, see if we detected disk full a minute ago. */
    if(len < 10)
    	return -1;
    else
    	return 0;
    } /* end of write_queue_file() */

/*
** Open the FIFO to pprd or rpprd.  This FIFO will be used to tell
** the daemon that the job has been placed in the queue.
*/
FILE *open_fifo(char *name)
    {
    int fifo;		/* file handle of pipe to pprd */
    int newfifo;

    /* Try to open it. */
    if( (fifo=open(name,O_WRONLY | O_NDELAY))==-1 )
	return (FILE*)NULL;
	
    /* 
    ** This loop takes care of things if this program was
    ** invoked without file descriptors 0 thru 2 already open.
    ** It is popularly supposed that the operating system
    ** connects these to stdin, stdout, and stderr, but actually
    ** the shell does this.  Some daemons may invoke ppr without
    ** opening something on these file descriptors.  If this
    ** happens, we must move the fifo up and open /dev/null
    ** on descriptors 0 thru 2. 
    */
    while(fifo <= 2)
    	{
	newfifo = dup(fifo);		/* move fifo up to next handle */
	close(fifo);			/* close old one */
	if( open("/dev/null", (fifo==0 ? O_RDONLY : O_WRONLY) ) != fifo )
	    exit(PPREXIT_OTHERERR);	/* Mustn't call fatal! */
	fifo=newfifo;			/* adopt the new handle */
    	}

    /* Create an I/O stream which represents the FIFO. */
    return fdopen(fifo,"w");
    } /* end of open_fifo() */

/*
** Code to check if the user needs to be specially authorized to
** submit jobs to this destination, and if so, check if he is
** authorized.  If authorization is required but has not been
** obtained, it is a fatal error and this routine never returns.
**
** When this routine is called, qentry.ForLine should already be set.
*/
void authorization(void)
    {
    if( (auth_needed=(use_authcode || destination_protected(qentry.destnode, qentry.destname))) )
        {
        struct userdb user; 
        int ret;

    	/*
    	** Figure out which account we must charge this to.
    	** If the --charge-to switch was used that is the 
    	** answer.  If not and the ForLine was set by the -f switch
    	** or a "%%For:" line with -R for then use that.  If that 
    	** doesn't work, use the Unix username.
    	*/
        if(charge_to_switch_value != (char*)NULL)
            qentry.charge_to = charge_to_switch_value;
        else if(qentry.ForLine != default_ForLine)
	    qentry.charge_to = qentry.ForLine;
	else
	    qentry.charge_to = qentry.username;
        
        if(!preauthorized)			/* do user lookup only */
            {					/* if not preauthorized */
	    ret = db_auth(&user, qentry.charge_to);	/* user lookup */

            if(ret == USER_ISNT)		/* If user not found, */ 
                {				/* then, turn away. */
                respond(RESP_CANCELED_NOTAUTH, qentry.charge_to);
                exit(PPREXIT_NOTAUTH);
                }
        
            else if(ret == USER_OVERDRAWN)	/* If account overdrawn, */ 
                {				/* then, turn away. */
		respond(RESP_CANCELED_OVERDRAWN, qentry.charge_to);
                exit(PPREXIT_OVERDRAWN);
                }

            /* We check for database error. */
            else if(ret == USER_ERROR)
            	{
            	fatal(PPREXIT_OTHERERR, "can't open user database");
            	}

	    /*
	    ** If in authcode mode and "-u yes" has not been set
	    ** then use the comment field from the user database
	    ** as the ForLine.
	    */
	    if(use_authcode && !use_username)
		qentry.ForLine = mystrdup(user.fullname);

            /*
            ** If -a switch and an authcode is required for this user and
            ** no or wrong authcode, send a message and cancel the job.
            */
            if(use_authcode && user.authcode[0] != (char)NULL &&   
			(AuthCode == (char*)NULL || strcmp(AuthCode, user.authcode)) )  
                {                              
		respond(RESP_CANCELED_BADAUTH, qentry.charge_to);
                exit(PPREXIT_BADAUTH);
                }
            } /* end of if not preauthorized */
        } /* end of if use_authcode or protected printer */

    /* Do not allow jobs w/out page counts on cost per page printers. */
    if(auth_needed)
        {
        if(qentry.attr.pages < 0)
	    {
	    respond(RESP_CANCELED_NONCONFORMING, (char*)NULL);
	    exit(PPREXIT_NONCONFORMING);
	    }

        if(qentry.opts.copies == -1)	/* If secure printer, force */
            qentry.opts.copies = 1;	/* unspecified copies to 1. */

        }

    } /* end of authorization() */

/*------------------------------------------------------------------
** `Mop routines' for accident cleanup
------------------------------------------------------------------*/

/*
** Call this function before aborting if there is any chance
** of files existing.  It closes the open files and removes them.
**
** Since respond() calls this function, this function must not call
** respond() since doing so would almost certainly create infinite
** recursion.
*/
void file_cleanup(void)
    {
    if(comments != (FILE*)NULL)
    	fclose(comments);
    if(page_comments != (FILE*)NULL)
    	fclose(page_comments);
    if(text != (FILE*)NULL)
    	fclose(text);

    /* Remove any temporary cache file. */
    abort_resource();

    /* Remove the partially completed job files. */
    if(files_created)
    	{
	char fname[MAX_PATH];
	
	sprintf(fname,"%s/%s:%s-%d.0(%s)",
		QUEUEDIR,
		qentry.destnode, qentry.destname, qentry.id, qentry.homenode);
	unlink(fname);

	sprintf(fname,"%s/%s:%s-%d.0(%s)-comments",
		DATADIR,
		qentry.destnode, qentry.destname, qentry.id, qentry.homenode);
	unlink(fname);

	sprintf(fname,"%s/%s:%s-%d.0(%s)-text",
		DATADIR,
		qentry.destnode, qentry.destname, qentry.id, qentry.homenode);
	unlink(fname);

	sprintf(fname,"%s/%s:%s-%d.0(%s)-pages",
		DATADIR,
		qentry.destnode, qentry.destname, qentry.id, qentry.homenode);
	unlink(fname);

	sprintf(fname,"%s/%s:%s-%d.0(%s)-log",
		DATADIR,
		qentry.destnode, qentry.destname, qentry.id, qentry.homenode);
	unlink(fname);

	sprintf(fname,"%s/%s:%s-%d.0(%s)-infile",
		DATADIR,
		qentry.destnode, qentry.destname, qentry.id, qentry.homenode);
	unlink(fname);
    	}

    /* Let's not do this twice: */
    comments = page_comments = text = (FILE*)NULL;
    files_created = FALSE;    
    } /* end of file_cleanup() */

/*
** Signal handler which deletes the partially 
** formed queue entry if we are killed.
*/
void gallows_speach(int signum)
    {
    fprintf(stderr, "PPR: mortally wounded by signal %d (%s).\n", signum, strsignal(signum));
    file_cleanup();
    exit(PPREXIT_KILLED);
    } /* end of gallows_speach() */

/*
** Child exit handler.
** The purpose of this routine is to detect core dumps and 
** other failures in filters.
*/
void reapchild(int signum)
    {
    int wstat;		/* storage for child's exit status */

    /* Get the child's exit code. */
    if(wait(&wstat)==-1)
    	fatal(PPREXIT_OTHERERR,"reapchild(): wait() failed, errno=%d (%s)\n",errno,strerror(errno));

    if(WIFEXITED(wstat))
    	{
    	switch(WEXITSTATUS(wstat))
    	    {
    	    case 0:			/* Normal exit, */
    	        return;			/* this signal handler need do nothing. */
    	    case 242:
    	        fatal(PPREXIT_OTHERERR,"filter failed, exit code = 242 (exec() failed?)");
    	        break;
    	    default:
    	        fatal(PPREXIT_OTHERERR,"filter failed, exit code = %d",WEXITSTATUS(wstat));
    	        return;
    	    }
	}
    else if(WIFSIGNALED(wstat))
	{
	fatal(PPREXIT_OTHERERR,"filter died on receipt of signal %d%s",
	    	WTERMSIG(wstat), WCOREDUMP(wstat) ? ", core dumped" : "");
	}
    else
    	{
    	fatal(PPREXIT_OTHERERR,"ppr_main.c: reapchild(): Bizzar child termination");
    	}
    } /* end of reapchild() */

/*
** Function to return true if the current user is a privledged
** user.  A privledged user is defined as "root", "ppr", or a
** member of the group called "pprprox".  The first
** time this routine is called, it will cache the answer.
**
** The result of this routine determines whether the user is allowed
** to override his user name on headers with a new name in the
** "-f" switch or "%%For:" header line.  It also determines whether
** a user can use the "-A" switch.
*/
int privledged(void)
    {
    static int answer = -1;	/* -1 means undetermined */
    
    if(answer == -1)		/* if undetermined */
    	{
	answer = FALSE;		/* Start with "false". */

	/*
	** Of course, "ppr" is privledged, as is "root" (uid 0).
	*/
	if( uid==0 || uid == setuid_uid )
	    {
	    answer=1;		/* Change to "true". */
	    }

	/* As are all users who are members of the group "ppr". */
	else		
	    {
    	    struct group *gr;
    	    int x;
	    char *ptr;

	    if( (gr=getgrnam(GROUP_PPRPROX)) != (struct group *)NULL )
	        {
	        x=0;
    	        while( (ptr=gr->gr_mem[x++]) != (char*)NULL )
    	    	    {
    	    	    if(strcmp(ptr, qentry.username) == 0)
    	    	        {
    	    	        answer=1;	/* Change to "true". */
    	    	        break;
    	    	        }
    	            }
    	        }
    	    }
    	} /* end of if answer not determined yet */

    return answer;
    } /* end of privledged() */

/*--------------------------------------------------------------------
** Routines for parsing the parameters.
--------------------------------------------------------------------*/
/* aAbBCdDefFGHIKmNnoOpPqQrRsStTUvwYXZ */ 
static const char *option_description_string = "ad:e:f:i:m:r:b:t:w:D:F:T:S:q:B:N:n:AC:H:R:Z:O:p:K:s:P:Io:UY:X:u:G:Q:";

static const struct ppr_getopt_opt option_words[] =
	{
	/*
	** These are preliminary and not yet documented.  As they are
	** finalized they will be moved to the next section and added to
	** the man page.
	*/
	{"dest", 'd', TRUE},
	{"destination", 'd', TRUE},
	{"for", 'f', TRUE},
	{"use-username", 'u', TRUE},
	{"title", 'C', TRUE},
	{"routing", 'i', TRUE},
	{"banner", 'b', TRUE},
	{"trailer", 't', TRUE},
	{"feature", 'F', TRUE},
	{"errors", 'e', TRUE},
	{"warnings", 'w', TRUE},
	{"proofmode", 'P', TRUE},
	{"strip-resources", 'S', TRUE},
	{"unlink", 'U', FALSE},
	{"portion", 'p', TRUE},
	{"signature", 's', TRUE},
	{"copies", 'n', TRUE},
	{"n-up", 'N', TRUE},
	{"draft-notice", 'O', TRUE},
	{"split", 'Y', TRUE},
	{"proxy-for", 'X', TRUE},
	{"default-media", 'D', TRUE},
	{"input-type", 'T', TRUE},
	{"priority", 'q', TRUE},
	{"auto-bin-select", 'B', TRUE},
	{"read", 'R', TRUE},
	{"require-eof", 'Z', TRUE},
	{"authcode-mode", 'a', FALSE},
	{"preauthorized", 'A', FALSE},
	{"truetype", 'Q', TRUE},
	{"keep-bad-features", 'K', TRUE},
	{"switchset", 'I', FALSE},
	{"filter-options", 'o', TRUE},
	{"gab", 'G', TRUE},
	{"hack", 'H', TRUE},
    	{"show-id", 1006, FALSE},

	/* These are final: */
	{"help", 1000, FALSE},
	{"version", 1001, FALSE},
	{"ignore-dsc-title", 1002, FALSE},
	{"lpq-filename", 1003, TRUE},
	{"hold", 1004, FALSE},
	{"responder", 'm', TRUE},
	{"responder-address", 'r', TRUE},
	{"responder-options", 1005, TRUE},
	{"charge-to", 1007, TRUE},
	{(char*)NULL, 0, FALSE}
	} ;

/*
** Print the help screen to the indicated file (stdout or stderr).
*/
void help(FILE *outfile)
    {
fprintf(outfile,
"usage: ppr [switches] [filename]\n"
"\n"
"\t-d <destname>              specifies printer or group\n"
"\t-f <string>                text for \"%%%%For:\" line\n"
"\t--charge-to <string>       user account to charge printing to\n"
"\t-u yes                     use username to identify jobs in queue\n"
"\t-u no                      use /etc/passwd comment instead (default)\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-m <method>                response methode\n"
"\t-m none                    no response\n"
"\t-r <address>               response address\n"
"\t--responder-options <list> list of name=value responder options\n"
"\t-b {yes,no,dontcare}       vote on banner page\n"
"\t-t {yes,no,dontcare}       vote on trailer page\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-w log                     sends warnings to log file\n"
"\t-w stderr                  sends warnings to stderr (default)\n"
"\t-w {severe,peeve,none}     sets warning level\n"
"\t-a                         turns on authcode mode\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-A                         root or ppr has preauthorized\n"
"\t-D <medianame>             sets default media\n"
"\t-F '<feature name>'        inserts setup code for features\n"
"\t-S true                    strip out resources already in cache (default)\n"
"\t-S false                   dont't strip resources\n");

minus_tee_help(outfile);	/* drag in -T stuff */

fprintf(outfile,
"\t-q <integer>               sets priority of print job\n"
"\t-B false                   disable automatic bin selection\n"
"\t-B true                    enable automatic bin selection (default)\n"
"\t-N <positive integer>      print pages N-Up\n"
"\t-N noborders               turn off N-Up borders\n"
"\t-n <positive integer>      print n copies\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-n collate                 print collated copies\n"
"\t-C <string>                default title\n"
"\t--ignore-dsc-title         ignore \"%%Title:\" comments\n"
"\t--lpq-filename <string>    filename for lpq listings\n"
"\t-i <string>                routing instructions\n"
"\t-R copies                  read copy count from document\n"
"\t-R duplex:softsimplex      read duplex mode, assume simplex (default)\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-R duplex:simplex          read duplex mode, assume simplex\n"
"\t-R duplex:duplex           read duplex mode, assume normal duplex\n"
"\t-R duplex:duplextumble     read duplex mode, assume tumble duplex\n"
"\t-R ignoreduplex            don't read duplex mode\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-Z true                    ignore jobs without %%%%EOF\n"
"\t-Z false                   don't ignore (default)\n"
"\t-O <string>                overlay `Draft' notice\n"
"\t-P notifyme                set ProofMode to NotifyMe\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-P substitute              set ProofMode to Substitute\n"
"\t-P trustme                 set ProofMode to TrustMe\n"
"\t-K true                    keep feature code though not in PPD file\n"
"\t-K false                   don't keep bad feature code (default)\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-e none                    don't user stderr or responder for errors\n"
"\t-e stderr                  report errors on stderr (default)\n"
"\t-e responder               report errors by responder\n"
"\t-e both                    report errors with both\n"
"\t-e hexdump                 always use hexdump for no filter\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-e dishexdump              discourage hexdump for no filter\n"
"\t-s <positive integer>      signature sheet count\n"
"\t-s booklet                 automatic signature sheet count\n"
"\t-s {both,fronts,backs}     indicate part of each signature to print\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-I                         Insert destination's switchset macro here\n"
"\t-o <string>                specify filter options\n"
"\t-U                         unlink job file after queuing it\n"
/* "\t-Y                        !!! NEW !!!\n" */
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-X <principal string>      identify principal for whom we act\n"
"\t-G <string>                gab about subject indicated by <string>\n"
"\t-Q <string>                TrueType query answer given (for papsrv)\n"
"\t--hold                     job should be held immediately\n"
#ifdef SHORT_STRINGS
); fprintf(outfile,
#endif
"\t-H <hack name>             turn on a hack\n"
"\t-H no-<hack name>          turn off a hack\n"
"\t-p <portion spec>          print only part of the document\n"
"\t--version                  print PPR version information\n"
"\t--help                     print this help\n"
);
    } /* end of help() */

/*
** Convert a flag page option string to a code number.
*/
int flag_option(const char *optstr)
    {
    if(strcmp(optstr, "yes") == 0)
        return BANNER_YESPLEASE;
    else if(strcmp(optstr, "no") == 0)
        return BANNER_NOTHANKYOU;
    else if(strcmp(optstr, "dontcare") == 0)
        return BANNER_DONTCARE;
    else                                        /* no match */
        return -1;                              /* is an error */
    } /* end of flag_option() */

/*
** Mark a feature for insertion.
** We pay special attention to Duplex because it can affect
** the page factor.
**
** This function is called when an -F switch is read or
** after all of the file is read if read_duplex_enforce is TRUE.
** In this latter case, setting current_duplex is unnecessary.
*/
void mark_feature_for_insertion(const char *name)
    {
    if(features_count >= MAX_FEATURES)
        fatal(PPREXIT_SYNTAX, "Too many -F switches");

    features[features_count++] = name;

    if(strncmp(name,"*Duplex ",8)==0)         	/* if any kind */
        {                                       /* of duplex, */
        if(strcmp(name,"*Duplex None")==0)    	/* if "none" */
            {
	    current_duplex=DUPLEX_SIMPLEX;
            delete_requirement("duplex");     	/* duplex not required */
	    delete_requirement("duplex(tumble)");
            }					
        else                                  	/* otherwise, */
            {					/* it is duplex! */
            if(strcmp(name,"*Duplex DuplexTumble")==0) /* if tumble */
		{
		current_duplex=DUPLEX_TUMBLE;
		delete_requirement("duplex");
  		requirement(REQ_DOC,"duplex(tumble)");
		}
            else                                /* otherwise, */
		{
		current_duplex=DUPLEX_DUPLEX;
                delete_requirement("duplex(tumble)");
                requirement(REQ_DOC,"duplex");  /* ordinary duplex */
		}
            }
        } /* end of if feature is "*Duplex" */

    } /* end of mark_feature_for_inclusion() */
    
/*
** This routine is called by the routine I_switch() below.
** This routine calls doopt once for each switch in the line.
*/
static void I_switch_line_parse(char *line)
    {
    int x;			/* line index */
    #ifdef GNUC_HAPPY
    int optchar=0;		/* switch character we are processing (0 is for GNU-C) */
    #else
    int optchar;
    #endif
    char *argument;		/* character's argument */
    char *ptr;			/* pointer into option_description_string[] */
    int stop;
    void doopt(int optchar, const char *optarg, const char *option);
    
    stop = strlen(line);

    for(x=0; x < stop; x++)	/* move thru the line */
    	{
	optchar = line[x++];

	argument = &line[x];				/* The stuff after the switch */
	argument[strcspn(argument, "|")] = (char)NULL;	/* NULL terminate */

	if(optchar == '-')				/* if it is a long option */
	    {
	    int y, len;
	    char *value = strchr(argument, '=');
	    
	    if(value != (char*)NULL)
	    	{
	    	len = value - argument;
	    	value++;
	    	}
	    else
	    	{
	    	len = strlen(argument);
	    	}

	    for(y = 0; option_words[y].name != (char*)NULL; y++)
	    	{
		if(strlen(option_words[y].name) == len && strncmp(option_words[y].name, argument, len) == 0)
		    {
		    if(value == (char*)NULL && option_words[y].needsarg)
			fatal(PPREXIT_SYNTAX, "Missing argument to --%s option in -I switch insertion", argument);
		    
		    if(value != (char*)NULL && ! option_words[y].needsarg)
			fatal(PPREXIT_SYNTAX, "Unneeded argument to --%.*s option in -I switch insertion", len, argument);

		    doopt(option_words[y].code, value, option_words[y].name);		    
		    break;		    
		    }	    	
	    	}

	    if(option_words[y].name == (char*)NULL)
	    	fatal(PPREXIT_SYNTAX, "Unknown option --%.*s inserted by -I switch", len, argument);
	    }
	else
	    {
	    if((ptr=strchr(option_description_string, optchar)) == (char*)NULL)
		fatal(PPREXIT_SYNTAX, "Unknown option -%c inserted by -I switch", optchar);
	    
	    if(argument[0] == (char)NULL && ptr[1] == ':')
		fatal(PPREXIT_SYNTAX, "Missing argument to -%c switch in -I switch insertion", optchar);

	    if(argument[0] != (char)NULL && ptr[1] != ':')
		fatal(PPREXIT_SYNTAX, "Unneeded argument to -%c switch in -I switch insertion", optchar);

	    doopt(optchar, argument, "");	/* actually process it */
	    }

    	x += strlen(argument);		/* move past this one */
    	}				/* (x++ in for() will move past the NULL) */

    } /* I_switch_line_parse() */

/*
** This is the -I switch routine.  It gets the line from the
** destination configuration file and feeds it to 
** I_switch_line_parse().
*/
void I_switch(void)
    {
    static int I_level = 0;
    char *ptr;

    /*
    ** We get get the switchset for a destination if 
    ** one hasn't been specified!
    */
    if(qentry.destname == (char*)NULL)
    	fatal(PPREXIT_SYNTAX, "-d must be used before -I");

    /*
    ** An -I switch within a switchset has the 
    ** potential to create an infinite loop.
    ** This test breaks such a loop.
    */
    if(I_level++ > 1)
    	fatal(PPREXIT_SYNTAX, "Illegal nested -I switch");
    	
    /*
    ** If we can find a switchset in the printer's or group's
    ** configuration file, parse it.
    */
    if( (ptr=extract_switchset()) != (char*)NULL )
	I_switch_line_parse(ptr);

    I_level--;			/* back up one nesting level */
    } /* end of I_switch() */

/*
** Parse a destination specification.
*/
void parse_d(const char *arg)
    {
    int len;

    if(qentry.destnode != (char*)NULL) myfree((char*)qentry.destnode);

    len = strcspn(arg, ":");		/* length before first ':' */

    if( arg[len] == ':' )		/* if there is a node name specified, */
	{
	qentry.destname = &arg[len+1];
	qentry.destnode = mystrndup(arg,len);
	}
    else				/* If no node specified, */
	{				/* the whole argument is */
	qentry.destname = arg;		/* the destination and there */
	qentry.destnode = (char*)NULL;	/* is not destination node. */
	}
    } /* end of parse_d() */

/*
** Do what is necessary for an option.
**
** We feed this routine the option character, such as 'd', and the
** option string such as "chipmunk".
*/
void doopt(int optchar, const char *optarg, const char *true_option)
    {
    switch(optchar)
        {
        case 'd':				/* destination */
	    parse_d(optarg);
            break;

        case 'f':				/* for whom */
            qentry.ForLine = optarg;
            break;

	case 'u':				/* Use username in stead of comment */
	    if( (use_username=torf(optarg)) == ANSWER_UNKNOWN )
	    	fatal(PPREXIT_SYNTAX, "The -u option must be followed by \"yes\" or \"no\"");
	    break;

        case 'm':				/* response method */
            qentry.response_method = optarg;
            break;

        case 'r':				/* response address */
            qentry.response_address = optarg;
            break;

        case 'b':				/* banner */
            if( (qentry.do_banner=flag_option(optarg)) == -1 )   
                fatal(PPREXIT_SYNTAX, "Invalid -b option");
            break;

        case 't':				/* trailer */
            if( (qentry.do_trailer=flag_option(optarg)) == -1 )
                fatal(PPREXIT_SYNTAX, "Invalid -b option");
            break;

        case 'w':				/* set warning option */
            if( strcmp(optarg, "log") == 0)
                warning_log = TRUE;
	    else if( icmp(optarg, "stderr") == 0)
	    	warning_log = FALSE;
            else if(strcmp(optarg, "severe") == 0)
                warning_level = WARNING_SEVERE;
            else if(strcmp(optarg, "peeve") == 0)
                warning_level = WARNING_PEEVE;
            else if(strcmp(optarg, "none") == 0)
                warning_level = WARNING_NONE;
            else
                fatal(PPREXIT_SYNTAX, "invalid -w option");
            break;

        case 'a':				/* authcode mode */
            use_authcode = TRUE;
	    read_for = TRUE;
            break;

        case 'D':				/* set default media */
            default_media = optarg;
            break;

        case 'F':				/* add a feature */
	    mark_feature_for_insertion(optarg);
            break;

        case 'T':				/* force input type */
            if(force_in_type(optarg))
                fatal(PPREXIT_SYNTAX, "-T %s is unrecognized", optarg);
            break;

        case 'S':				/* strip resources */
            if( (strip_resources=torf(optarg)) == ANSWER_UNKNOWN )
            	fatal(PPREXIT_SYNTAX, "-S must be followed by \"true\" or \"false\"");
            break;

        case 'q':				/* queue priority */
            if( ((qentry.priority=atoi(optarg)) < 0) || (qentry.priority > 39) )
                fatal(PPREXIT_SYNTAX, "Priority must be between 0 and 39");
            break;

        case 'B':				/* disable or enable automatic bin selects */    
            if( (qentry.opts.binselect=torf(optarg)) == ANSWER_UNKNOWN )
            	fatal(PPREXIT_SYNTAX, "-B must be followed by \"true\" or \"false\"");
            break;

        case 'N':				/* N-Up */
	    if(icmp(optarg,"noborders")==0)	/* noborders option, */
	    	{				/* turn */
	    	qentry.N_Up.borders = FALSE;	/* borders off */
	    	}
	    else
		{
            	qentry.N_Up.N = atoi(optarg);
            	if( (qentry.N_Up.N < 1) /* || (qentry.N_Up.N > 16) */ ) 
                    fatal(PPREXIT_SYNTAX, "N must be between 1 and 16");
                }
            break;

        case 'n':				/* number of copies */
	    if(strcmp(optarg,"collate")==0) /* (or ``collate'') */
	    	{
		qentry.opts.collate = TRUE;
		}
	    else
	    	{	
                if( (qentry.opts.copies=atoi(optarg)) < 1)
                    fatal(PPREXIT_SYNTAX,"Number of copies must be positive");
		}
            break;

        case 'A':				/* a preauthorized job */
            if( ! privledged() )
                fatal(PPREXIT_SYNTAX, "Only privledged users may use the -A switch");
	    else
                preauthorized = TRUE;
            break;

        case 'C':				/* default title */
            qentry.Title = optarg;
            break;

	case 'i':				/* routing instructions */
	    qentry.Routing = optarg;
	    break;

        case 'R':
            if(strcmp(optarg, "copies") == 0)
		{
                read_copies = TRUE;
                }
	    else if(strcmp(optarg, "ignorecopies") == 0)
	    	{
	    	read_copies = FALSE;
	    	}
	    else if(strcmp(optarg, "duplex:duplex") == 0)
		{
                read_duplex=TRUE;
                read_duplex_enforce = TRUE;
		current_duplex = DUPLEX_DUPLEX;
		}
	    else if(strcmp(optarg, "duplex:simplex") == 0)
	    	{
	    	read_duplex = TRUE;
	    	read_duplex_enforce = TRUE;
	    	current_duplex = DUPLEX_SIMPLEX;
	    	}
	    else if(strcmp(optarg, "duplex:duplextumble") == 0)
	        {
	        read_duplex = TRUE;
	        read_duplex_enforce = TRUE;
	        current_duplex = DUPLEX_TUMBLE;
	        }
	    else if(strcmp(optarg, "duplex:softsimplex") == 0)
	    	{
	    	read_duplex = TRUE;
	    	read_duplex_enforce = FALSE;
	    	current_duplex = DUPLEX_SIMPLEX;
	    	}
	    else if(strcmp(optarg, "ignoreduplex") == 0)
	    	{
		read_duplex = FALSE;
		read_duplex_enforce = FALSE;
		/* current_duplex = DUPLEX_SIMPLEX; */ /* don't do this! */
		}
	    else if(strcmp(optarg, "for") == 0)
	    	{
		if(!privledged())
		    fatal(PPREXIT_SYNTAX, "Only privledged users may use \"-R for\"");
	    	read_for = TRUE;
	    	}
	    else if(strcmp(optarg, "ignorefor") == 0)
	    	{
	    	read_for = FALSE;
	    	}
            else
                {
                fatal(PPREXIT_SYNTAX, "Unrecognized -R option: %s", optarg);
                }
            break;

	case 'Z':
	    if( (ignore_truncated=torf(optarg)) == ANSWER_UNKNOWN )
	    	fatal(PPREXIT_SYNTAX, "-Z must be followed by \"true\" or \"false\"");
	    break;

	case 'O':			/* overlay `Draft' notice */
	    qentry.Draft = optarg;	
	    break;

	case 'P':
	    if(icmp(optarg,"NotifyMe") == 0)
	    	qentry.attr.proofmode = PROOFMODE_NOTIFYME;
	    else if(icmp(optarg,"Substitute") == 0)
	    	qentry.attr.proofmode = PROOFMODE_SUBSTITUTE;
	    else if(icmp(optarg,"TrustMe") == 0)
	    	qentry.attr.proofmode = PROOFMODE_TRUSTME;
	    else
	    	fatal(PPREXIT_SYNTAX, "-P must be followed by \"notifyme\", \"substitute\", or \"trustme\"");
	    found_ProofMode = TRUE;
	    break;	    	

	case 'K':
	    if( (qentry.opts.keep_badfeatures=torf(optarg)) == ANSWER_UNKNOWN )
	    	fatal(PPREXIT_SYNTAX,"-K must be set to \"true\" or \"false\"");
	    break;

	case 'e':
	    if(icmp(optarg,"none")==0)
	    	ppr_respond_by=PPR_RESPOND_BY_NONE;
	    else if(icmp(optarg,"stderr")==0)
	    	ppr_respond_by=PPR_RESPOND_BY_STDERR;
	    else if(icmp(optarg,"responder")==0)
	    	ppr_respond_by=PPR_RESPOND_BY_RESPONDER;
	    else if(icmp(optarg,"both")==0)
	    	ppr_respond_by=PPR_RESPOND_BY_BOTH;
	    else if(icmp(optarg,"hexdump")==0)
	    	nofilter_hexdump=TRUE;
	    else if(icmp(optarg,"dishexdump")==0)
	    	nofilter_hexdump=FALSE;
	    else
		fatal(PPREXIT_SYNTAX,"-e must be followed by \"none\", \"stderr\", \"responder\", or \"both\"");
	    break;

	case 's':					/* signature option */
	    if(icmp(optarg,"fronts")==0)
	    	qentry.N_Up.sigpart = SIG_FRONTS;
	    else if(icmp(optarg,"backs")==0)
	    	qentry.N_Up.sigpart = SIG_BACKS;
	    else if(icmp(optarg,"both")==0)
	    	qentry.N_Up.sigpart = SIG_BOTH;
	    else if(icmp(optarg,"booklet")==0)
		{
		qentry.N_Up.sigsheets = (-1);
		}
	    else if( (qentry.N_Up.sigsheets=atoi(optarg)) >= 0 )
		{ /* no action */ }
	    else
	    	fatal(PPREXIT_SYNTAX, "-s option must be \"fronts\", \"backs\", \"both\", \"booklet\", or a positive integer");
	    break;

	case 'I':					/* insert switch set */
	    I_switch();
	    break;

	case 'o':					/* Filter options */
	    {
	    const char *ptr;				/* eat leading space */
	    ptr = &optarg[strspn(optarg," \t")];	/* (It may cause trouble for exec_filter()) */

	    if( filter_options == (char*)NULL )		/* If this is the first -o switch, */
	    	{					/* Just make a copy of it. */
	    	filter_options = mystrdup(ptr);		/* (Yes, it is not necessary to */
	    	}					/* make a copy, but it makes the */
	    else					/* else clause easier.) */
	    	{
		char *ptr2 = (char*)myalloc( strlen(filter_options) + 1 + strlen(ptr) + 1, sizeof(char) );
		sprintf(ptr2,"%s %s",filter_options,ptr);	/* concatenate */
		myfree(filter_options);				/* free old one */
		filter_options = ptr2;				/* make new one current */
	    	}
	    }
	    break;

	case 'U':				/* unlink the input file */
	    unlink_jobfile = TRUE;
	    break;

	case 'Y':				/* Split the job */
	    Y_switch(optarg);
	    break;

	case 'X':				/* Identify principal */
	    qentry.proxy_for = optarg;
	    break;

	case 'G':				/* What to gab about */
	    {
	    const char *ptr = optarg;
	    int no = FALSE;
	    int x;

	    if( icmpn(ptr, "no-", 3) == 0 )
	    	{
	    	no = TRUE;
	    	ptr += 3;
	    	}
	    	
	    for(x=0; gab_table[x].name != (char*)NULL; x++)
	    	{
		if( icmp(ptr, gab_table[x].name) == 0 )
		    {
		    if( ! no )
			gab_mask |= gab_table[x].bit;
		    else
		    	gab_mask &= ~gab_table[x].bit;

		    break;
		    }
	    	}
	    
	    if( gab_table[x].name == (char*)NULL )
		fatal(PPREXIT_SYNTAX, "Unrecognized -G option: %s", optarg);
	    }
	    break;

	case 'Q':				/* TrueType query answer (for papsrv) */
	    if(strcmp(optarg,"None")==0)
	    	TrueTypeQuery = TT_NONE;
	    else if(strcmp(optarg,"Accept68K")==0)
	    	TrueTypeQuery = TT_ACCEPT68K;
	    else if(strcmp(optarg,"Type42")==0)
	    	TrueTypeQuery = TT_TYPE42;
	    else
	    	fatal(PPREXIT_SYNTAX,"Unrecognized -Q option: \"%s\"",optarg);
	    break;

        case 'H':				/* turn on a hack */
	    {
	    const char *ptr = optarg;
	    int no = FALSE;
	    int x;

	    if( icmpn(ptr, "no-", 3) == 0 )
	    	{
	    	no = TRUE;
	    	ptr += 3;
	    	}
	    	
	    for(x=0; hack_table[x].name != (char*)NULL; x++)
	    	{
		if( icmp(ptr, hack_table[x].name) == 0 )
		    {
		    if( ! no )
			qentry.opts.hacks |= hack_table[x].bit;
		    else
		    	qentry.opts.hacks &= ~hack_table[x].bit;

		    break;
		    }
	    	}
	    
	    if( hack_table[x].name == (char*)NULL )
		fatal(PPREXIT_SYNTAX, "Unrecognized -H option: %s", optarg);
	    }
	    break;

	case 'p':				/* print only a portion */
	    {
	    char name[8], value[8];
	    int result;

	    options_start(optarg);
	    
	    while( (result=options_get_one(name, sizeof(name), value, sizeof(value) )) == 1 )
		{
		if( icmp(name,"odd") == 0 )
		    {
		    if( (qentry.portion.odd=torf(value)) == ANSWER_UNKNOWN )
		    	fatal(PPREXIT_SYNTAX, "-p odd must be set to true or false");
		    }
		else if( icmp(name,"even") == 0 )
		    {
		    if( (qentry.portion.even=torf(value)) == ANSWER_UNKNOWN )
		    	fatal(PPREXIT_SYNTAX, "-p even must be set to true or false");
		    }
		else if( icmp(name,"start") == 0 )
		    {
		    if( (qentry.portion.start=atoi(value)) <= 0 )
		    	fatal(PPREXIT_SYNTAX, "-p start must be set to a sheet number");
		    }
		else if( icmp(name,"stop") == 0 )
		    {
		    if( (qentry.portion.stop=atoi(value)) <= 0 )
		    	fatal(PPREXIT_SYNTAX, "-p stop must be set to a sheet number");
		    }
		else
		    {
		    fatal(PPREXIT_SYNTAX,"Unrecognized -p option");
		    }
		}

	    if( result == -1 )
		{
		fatal(PPREXIT_SYNTAX, "Problem with -p option: \"%s\", %s", &options_string[options_error_context_index], options_error);
		}
	    }
	    break;

	case 1000:				/* --help */
	    help(stdout);
	    exit(PPREXIT_OK);
	    
	case 1001:				/* --version */
	    puts(VERSION);
	    puts(COPYRIGHT);
	    puts(AUTHOR);
	    exit(PPREXIT_OK);

	case 1002:				/* --ignore-dsc-title */
	    found_Title = TRUE;
	    break;
	    
	case 1003:				/* --lpq-filename */
	    qentry.lpqFileName = optarg;
	    break;

	case 1004:				/* --hold */
	    option_hold = TRUE;
	    break;

	case 1005:				/* --responder-options */
	    qentry.responder_options = optarg;
	    break;

    	case 1006:  	    	    	    	/* --show-id */
    	    option_show_id = TRUE;
    	    break;

	case 1007:				/* --charge-to */
	    if(! privledged() && ! use_authcode)
	    	fatal(PPREXIT_SYNTAX, "Non-privledged users may not use --charge-to w/out -a");
	    /* This is not copied into the queue structure unless it is needed. */
	    charge_to_switch_value = optarg;
	    break;

        case '?':				/* Unrecognized switch */
	    fatal(PPREXIT_SYNTAX, "Unrecognized switch %s.  Try --help", true_option);

	case ':':				/* argument required */
	    fatal(PPREXIT_SYNTAX, "The %s option requires an argument", true_option);

	case '!':				/* bad aggreation */
	    fatal(PPREXIT_SYNTAX, "Switches, such as %s, which take an argument must stand alone", true_option);
		
	case '-':				/* spurious argument */
	    fatal(PPREXIT_SYNTAX, "The %s switch does not take an argument", true_option);

        default:				/* Missing case */
	    fatal(PPREXIT_OTHERERR, "Missing case %d in switch dispatch switch()", optchar);
        } /* end of switch */
    } /* end of doopt() */
    
/*
** Main procedure.
**
** Among other things, we set default values in the queue file structure,
** parse the command line parameters, order the input file opened, order
** verious routines to read it until they get to the ends of various
** sections, order our authority to print to be verified, order a slew
** of checks for various inconsistencies, and order the file submitted
** to the spooler.
*/
int main(int argc, char *argv[])
    {
    char *real_filename = (char*)NULL;
    int optchar;			/* Used with ppr_getopt() */
    struct ppr_getopt_state getopt_state;
    struct passwd *pw;			/* to get information on invoking user */
    int x;				/* various very short term uses */
    struct sigaction sig;		/* used for setting signal handlers */
    char *ptr;				/* general use */

    /* First thing, we take these security precautions. */
    #ifndef NO_PUTENV
    putenv("PATH="SHORT_PATH);
    putenv("IFS= \t");		/* no tricks now! */
    putenv("SHELL=/bin/sh");	/* a nice, safe shell? */
    putenv("HOME="HOMEDIR);	/* for X-Windows programs */
    #endif

    /* Deduce the non-setuid user id and user name. */
    uid = getuid();
    gid = getgid();
    if( (pw=getpwuid(uid)) == (struct passwd *)NULL)
    	fatal(PPREXIT_OTHERERR, "getpwuid(uid) failed, can't find you in /etc/passwd");
    if(pw->pw_name == (char*)NULL)
    	fatal(PPREXIT_OTHERERR, "strange getpwuid() error, pw_name is NULL");

    /*
    ** Save the uid that is effective now.
    ** (This will be ``ppr''.)
    */
    setuid_uid = geteuid();
    setgid_gid = getegid();
                                 
    /* Remember what directory we started in. */
    starting_directory[0] = (char)NULL;
    getcwd(starting_directory, sizeof(starting_directory) );

    /* Set the umask to prevent snooping. */
    umask(0077);

    /*
    ** Clear parts of the queue entry, fill in default values
    ** elsewhere.  It is important that we do this right away
    ** since one of the things we do is get our queue id.
    */
    qentry.destnode = (char*)NULL;			/* name of node to send job to */
    qentry.destname = (char*)NULL;			/* name of printer or group */
    qentry.id = 0;					/* not assigned yet */
    qentry.subid = 0;					/* job fragment number */
    qentry.homenode = ppr_get_nodename();		/* this are the node this job came from */
    qentry.time = time((time_t*)NULL);			/* job submission time */
    qentry.priority = 20;				/* default priority */
    qentry.user = uid;					/* record real user id of submitter */
    qentry.username = pw->pw_name;			/* fill in name associated with id */
    qentry.ForLine = (char*)NULL;			/* start with no %%For: line */
    qentry.charge_to = (char*)NULL;			/* ppuser account to charge to */
    qentry.Title = (char*)NULL;				/* start with no %%Title: line */
    qentry.Draft = (char*)NULL;				/* message to print diagonally */
    qentry.Creator = (char*)NULL;			/* "%%Creator:" */
    qentry.Routing = (char*)NULL;			/* "%%Routing:" */
    qentry.lpqFileName = (char*)NULL;			/* filename for lpq */
    qentry.nmedia = 0;					/* no forms */
    qentry.do_banner = BANNER_DONTCARE;			/* don't care */
    qentry.do_trailer = BANNER_DONTCARE;		/* don't care */
    qentry.attr.langlevel = 1;				/* default to PostScript level 1 */
    qentry.attr.pages = -1;				/* number of pages undetermined */
    qentry.attr.pageorder = PAGEORDER_ASCEND;		/* assume ascending order */
    qentry.attr.extensions = 0;				/* no Level 2 extensions */
    qentry.attr.orientation = ORIENTATION_UNKNOWN;	/* probably not landscape */
    qentry.attr.proofmode = PROOFMODE_SUBSTITUTE; 	/* don't change this default (RBII p. 664-665) */
    qentry.attr.input_bytes = 0;			/* size before filtering if filtered */
    qentry.attr.postscript_bytes = 0;			/* size of PostScript */
    qentry.attr.parts = 1;				/* for now, assume one part */
    qentry.opts.binselect = TRUE;			/* do auto bin select */
    qentry.opts.copies = -1;				/* unspecified number of copies */
    qentry.opts.collate = FALSE;			/* by default, don't collate */
    qentry.opts.keep_badfeatures = FALSE;		/* delete feature code not in PPD file */
    qentry.opts.hacks = 0;				/* no hacks */
    qentry.N_Up.N = 1;					/* start with 1 Up */
    qentry.N_Up.borders = TRUE;				/* print borders when doing N-Up */
    qentry.N_Up.sigsheets = 0;				/* don't print signatures */
    qentry.N_Up.sigpart = SIG_BOTH;			/* print both sides of signature */
    qentry.portion.odd = TRUE;				/* print odd sheets */
    qentry.portion.even = TRUE;				/* print even sheets */
    qentry.portion.start = 0;				/* no start sheet */
    qentry.portion.stop = 0;				/* no stop sheet */
    qentry.Draft = (char*)NULL;				/* assume no `Draft' notice */

    /*
    ** We would like to find a default response method in the veriable
    ** PPR_RESPONDER.  If we don't we will use "write".
    */
    if( (qentry.response_method=getenv("PPR_RESPONDER")) == (char*)NULL )
    	qentry.response_method = "write";

    /*
    ** Since a responder address may not be specified with the -r switch,
    ** we will look for a default value in the environment variable
    ** PPR_RESPONDER_ADDRESS.  If no such variable exists,
    ** we must use the user name which the user logged in with.
    ** The login user name is the proper address for the default
    ** responder "write".  Notice that if the user used su to become
    ** a different user after loging in, the current user id will differ
    ** from the current user id.  In order for the responder to use the
    ** "write" command sucessfully, we must determine the login name.
    ** That is why we try the environment variables "USER" and "LOGNAME"
    ** before resorting to the current user id.
    */
    if( (qentry.response_address=getenv("PPR_RESPONDER_ADDRESS")) == (char*)NULL )
	{
	if( (qentry.response_address=getenv("USER")) == (char*)NULL )
	    {
	    if( (qentry.response_address=getenv("LOGNAME")) == (char*)NULL )
		{
		qentry.response_address = qentry.username;
		}
	    }
	}

    /*
    ** The default responder options come from the environment.
    ** If the variable is undefined, getenv() will return NULL
    ** which is just what we want.
    */
    qentry.responder_options = getenv("PPR_RESPONDER_OPTIONS");

    /* clear default media things */
    guess_media.medianame[0] = (char)NULL;
    guess_media.width = 0.0;
    guess_media.height = 0.0;
    guess_media.weight = 0.0;
    guess_media.colour[0] = (char)NULL;
    guess_media.type[0] = (char)NULL;

    /* Establish the fatal signal handler. */
    signal(SIGTERM, gallows_speach);	/* software termination */
    signal(SIGHUP, gallows_speach);	/* parent terminates */
    signal(SIGINT, gallows_speach);	/* control-C pressed */

    /* 
    ** Arrange for child termination to be noted.  This is
    ** complicated because we want to be sure that is causes
    ** as little disruption as possible. 
    */
    sig.sa_handler = reapchild;	/* call reapchild() on SIGCHLD */
    sigemptyset(&sig.sa_mask);	/* block no additional sigs */
    #ifdef SA_RESTART
    sig.sa_flags = SA_RESTART;	/* restart interupted sys calls */
    #else
    sig.sa_flags = 0;
    #endif
    sigaction(SIGCHLD, &sig, NULL);

    /*==============================================================
    ** Start of option parsing code
    ==============================================================*/

    /* Process the options. */
    ppr_getopt_init(&getopt_state, argc, argv, option_description_string, option_words);
    while( (optchar=ppr_getopt(&getopt_state)) != -1 )
	doopt(optchar, getopt_state.optarg, getopt_state.name);
	
    /*
    ** Settle on a destination now.
    **
    ** (We want to do this as soon as possible so that
    ** message which refer to the job id will look right.)
    **                                         
    ** If no -d switch set qentry.destname, try to set it
    ** from the environment variable "PPRDEST", if that
    ** doesn't work, use a destination name of "default".
    */
    if(qentry.destname == (char*)NULL)
	{
	char *ptr;
	if( (ptr=getenv("PPRDEST")) == (char*)NULL )
	    ptr = "default";
	parse_d(ptr);
	}

    /* 
    ** A empty or syntactically invalid destination name
    ** can cause real problems later, therefor, try to
    ** detect it now.
    */
    if(qentry.destname[0] == (char)NULL)
    	fatal(PPREXIT_SYNTAX, "Destination (printer or group) name is empty");
    if(strlen(qentry.destname) > MAX_DESTNAME)
	fatal(PPREXIT_SYNTAX, "Destination (printer or group) name is too long");
    if(strpbrk(qentry.destname, DEST_DISALLOWED) != (char*)NULL)
	fatal(PPREXIT_SYNTAX, "Destination (printer or group) name contains a disallowed character");
    if(strchr(DEST_DISALLOWED_LEADING, (int)qentry.destname[0]) != (char*)NULL)
	fatal(PPREXIT_SYNTAX, "Destination (printer or group) name starts with a disallowed character");
    
    /*
    ** Check for illegal combinations of options.
    */
    if(use_authcode && preauthorized)
        fatal(PPREXIT_SYNTAX, "-a and -A are mutually exclusive");

    if( qentry.portion.stop < qentry.portion.start )
    	fatal(PPREXIT_SYNTAX, "-p switch specifies a non ascending range");

    /*
    ** Do a crude test of the syntax of the -o option.
    ** (The -o option passes options to an input file filter.) 
    */
    if( filter_options != (char*)NULL )
      {
      ptr = filter_options;
      ptr += strspn(ptr," \t");
      while(*ptr)
	{
	/* eat up keyword */
	ptr += strcspn(ptr," \t=");
		
	/* Look for space before "=". */
	if(isspace(*ptr))
	    fatal(PPREXIT_SYNTAX,"spaces may not preceed \"=\" in -o option");
		    
	if(*ptr != '=')
	    fatal(PPREXIT_SYNTAX,"-o syntax is keyword=value");
		    
	ptr++;			/* move beyond equal sign */

	if(*ptr == (char)NULL)
	    fatal(PPREXIT_SYNTAX,"last -o keyword has no value");

	if(isspace(*ptr))
	    fatal(PPREXIT_SYNTAX,"spaces may not follow \"=\" in -o option");

	ptr += strcspn(ptr," \t");
	ptr += strspn(ptr," \t");
	}
      } /* end of if there were -o switches */

    /* 
    ** Any `non option' arguments should be processed here.
    ** (This means one and only one file name.) 
    */
    if(argc > getopt_state.optind)
        real_filename = argv[getopt_state.optind];

    if( (argc - getopt_state.optind) > 1 )
        fatal(PPREXIT_SYNTAX, "only one file name allowed");

    /*
    ** If no -C switch was used but a file name was used,
    ** make the file name the default title.  The default
    ** title may be overridden by a "%%Title:" line.
    */
    if(qentry.Title == (char*)NULL)
    	if( (qentry.Title = qentry.lpqFileName) == (char*)NULL )
    	    qentry.Title = real_filename;

    /*
    ** If the destination node hasn't been specified, it
    ** is this node.
    */
    if(qentry.destnode == (char*)NULL)
    	qentry.destnode = qentry.homenode;

    /*
    ** If we don't have at least a provisional ForLine, 
    ** then use either the Unix user name or the comment
    ** field from /etc/passwd.
    **
    ** The variable default_ForLine is used by authorization()
    ** to determine if the current value of qentry.ForLine
    ** was set here.
    */
    if(qentry.ForLine == (char*)NULL || (! privledged() && ! use_authcode))
	{
	if(use_username)	/* <-- set with -u switch */
	    default_ForLine = qentry.ForLine = qentry.username;
	else
	    default_ForLine = qentry.ForLine = pw->pw_gecos;
        }

    /*===========================================================
    ** End of option parsing code
    ===========================================================*/

    /*
    ** Open the input file before we change to our home directory.
    ** When this function returns, the PPR home directory will be
    ** the current directory.
    **
    ** If no file was specified, real_filename will still be NULL.
    */
    if(get_input_file(real_filename))	/* If input file is of zero length, */
	goto zero_length_file;		/* we have nothing to do. */

    /* 
    ** Open the FIFO now so we will know the daemon is running.
    ** In former versions, we did this sooner so that the responder
    ** in pprd could be reached, but now that we have our own
    ** responder we can wait until the destination name is set so
    ** that respond() will use the correct job name. 
    **
    ** We open a different FIFO depending on whether the job is being sent
    ** to the local spooler or to a remote one.
    **
    ** Note that at this point qentry.destnode will be defined already.
    */
    if( strcmp(qentry.destnode, qentry.homenode) == 0 )
    	{		/* if sending to a local printer */
	if( (FIFO = open_fifo(FIFO_NAME)) == (FILE*)NULL )
	    {
	    respond(RESP_NOSPOOLER, (char*)NULL);
	    exit(PPREXIT_NOSPOOLER);
	    }
	}
    else
	{
	if( (FIFO = open_fifo(FIFO_NAME_RPPRD)) == (FILE*)NULL )
	    {
	    respond(RESP_NORSPOOLER, (char*)NULL);
	    exit(PPREXIT_NOSPOOLER);
	    }
	}

    /*
    ** We are about to start creating queue files.  We must
    ** assign a queue id now.
    **
    ** It is possible that get_input_file() will have been
    ** oblidged to call get_next_id() because it created
    ** a "-infile" file.  That is why we test to see
    ** if it has been assigned yet.
    */
    if(qentry.id == 0)
    	qentry.id = get_next_id(NEXTIDFILE);

    /* 
    ** Set a flag so that if we abort suddenly or are killed and
    ** are able to exit gracefully, we will know to remove the
    ** output files we are about to create. 
    */
    files_created = TRUE;

    /*
    ** Open the queue files in which we will store the output.
    */
    open_output();
    
    /*
    ** Chew thru the whole input file.
    */
    read_comments();		/* read header comments */

    if( ! read_prolog() )	/* If prolog (prolog, docdefaults, and docsetup) doesn't end with */
	{			/* doesn't stretch to EOF or trailer, */
	read_pages();		/* read the pages. */
	}
    read_trailer();		/* read trailer & trailer comments */

    /*
    ** Eat up any characters between logical and physical end of file.
    ** (Logical end of file is indicated by a "%%EOF" comment or 
    ** some other indication such as a control-D.
    **
    ** If there are fewer than 50 characters it is probably something
    ** like PJL code, so we only express anoyance.  If there are
    ** more than that, it looks serious so it merits a severe warning.
    */
    for(x = 0; in_getc() != EOF; x++);

    if(x==1)
        warning(WARNING_PEEVE, "1 character follows EOF mark");
    else if(x < 50)
        warning(WARNING_PEEVE, "%d characters follow EOF mark", x);
    else if(x)
        warning(WARNING_SEVERE, "%d characters follow EOF mark", x);

    /*
    ** We are now done with the input file.
    */
    close_infile();

    /* 
    ** Check if this is a truncated job that we should ignore.
    ** (That is, a job without "%%EOF" when "-S true".)
    ** (We do this check first since other checks are pretty
    ** meaninless if the input file is truncated.) 
    **
    ** (It may be important to remember that if no "%%EOF" comment
    ** is present, getline() may insert an "%%EOF" comment to
    ** replace a protocol specific end of file indication.)
    */
    if( ignore_truncated && !eof_comment_present )
    	fatal(PPREXIT_TRUNCATED,"input file was truncated");

    /* 
    ** Check for unclosed resources.  (This is an example of a check
    ** for which an abnormal result would not be suprising if the
    ** input file where truncated.)
    **
    ** The call to abort_resource() does not cause an exit, it
    ** mearly removes any temporary resource cache file.
    */
    if( nest_level() )
        {
        warning(WARNING_SEVERE, "Unclosed resource or resources");
        abort_resource();
        }

    /*
    ** Turn on duplexing if we are printing booklets or signatures
    ** and we are printing both sides at once,
    */
    if(qentry.N_Up.sigsheets)		/* if not zero (not disabled) */
    	{
	read_duplex_enforce = TRUE;	/* try to turn duplex on or off */
	qentry.N_Up.borders = FALSE;

	if(qentry.N_Up.N == 1)		/* if N-Up has not been set yet, */
	    qentry.N_Up.N = 2;		/* set it now. */
	   
	if(qentry.N_Up.sigpart == SIG_BOTH)	/* if printing both sides */
	    {					/* use duplex */
	    if(qentry.N_Up.N == 2)	    	/* if 2-Up, */
		current_duplex = DUPLEX_TUMBLE;	/* use tumble duplex */
	    else				/* for 4-Up, */
	    	current_duplex = DUPLEX_DUPLEX;	/* use no-tumble duplex */
	    }
	else					/* if printing only one side, */
	    {					/* use simplex */
	    current_duplex = DUPLEX_SIMPLEX;
	    }    	
    	}

    /* 
    ** If we looked for clews as to the desired duplex mode,
    ** we may want to dump our results here so that even if
    ** we are wrong, our prediction will come true. 
    */
    if(read_duplex_enforce)		/* if we are to enforce our duplex finding, */
    	{				/* then */
	switch(current_duplex)		/* insert appropriate code */
	    {
	    case DUPLEX_SIMPLEX:
	    	mark_feature_for_insertion("*Duplex None");
	    	break;
	    case DUPLEX_DUPLEX:
	    	mark_feature_for_insertion("*Duplex DuplexNoTumble");
	    	break;
	    case DUPLEX_TUMBLE:
	    	mark_feature_for_insertion("*Duplex DuplexTumble");
	    	break;
	    default:
	    	fatal(PPREXIT_OTHERERR,"invalid value for current_duplex");
	    }
    	}

    /* 
    ** Set the duplex_pagefactor variable according to the
    ** duplex setting. 
    */
    switch(current_duplex)
	{
	case DUPLEX_SIMPLEX:
	    duplex_pagefactor=1;
	    break;
	case DUPLEX_DUPLEX:
	case DUPLEX_TUMBLE:
	    duplex_pagefactor=2;
	    break;
	default:
	    fatal(PPREXIT_OTHERERR,"invalid value for current_duplex");
	}

    /* Dump the gathered information into the -comments file. */
    dump_document_media(comments,0);

    /* 
    ** Close those queue files which we are done with.
    ** Only the one in the "queue" directory remains open. 
    **
    ** We must be sure to set the pointers to NULL, otherwise
    ** file_cleanup() could try to close them again and cause
    ** a core dump.
    */
    fclose(comments);
    fclose(page_comments);
    fclose(text);
    comments = page_comments = text = (FILE*)NULL;

    /* 
    ** Compare "qentry.attr.pages" to "pagenumber".
    ** If we have "%%Page: " comments but no "%%Pages: "
    ** comment, use the number of "%%Page: " comments. 
    */
    if( (pagenumber>0) && (qentry.attr.pages==-1) )
        {
        warning(WARNING_SEVERE,"\"%%%%Pages:\" comment missing.");
        qentry.attr.pages = pagenumber;
        }

    /* 
    ** The variable "pagenumber" now contains the number of "%%Page:"
    ** comments encountered.
    **
    ** If we have "%%Page: " comments and there are more "%%Page: "
    ** comments than the number of pages stated in the "%%Pages: "
    ** comment, then use the count of "%%Page: " comments 
    ** since it would seem harder to get that wrong. 
    **
    ** What to do when the "%%Pages:" comment indicates more pages than
    ** there are "%%Page:" comments is more troublesome.  The program 
    ** pslpr will make this mistake when extracting selected pages from
    ** a document.
    */
    if(pagenumber > 0)
	{
	if(pagenumber > qentry.attr.pages)
	    {
	    warning(WARNING_SEVERE,"\"%%%%Pages:\" comment is wrong, changing it from %d to %d",
		qentry.attr.pages, pagenumber);
	    qentry.attr.pages = pagenumber;
	    }
	else if(pagenumber < qentry.attr.pages)
	    { 
	    warning(WARNING_SEVERE, "Missing \"%%%%Page:\" comments or incorrect \"%%%%Pages:\" comment.");
	    qentry.attr.pages = pagenumber;
	    }
	}

    /* 
    ** If we found no "%%Page:" comments, then the pageorder is special.
    ** (Re-ordering pages which we can not locate is not possible.) 
    */
    if(pagenumber==0)
        qentry.attr.pageorder = PAGEORDER_SPECIAL;

    /* 
    ** If we found "%%Page:" comments, then qentry.attr.script
    ** should be set to TRUE.  This will indicate that this file
    ** has what the DSC specification calls a "script" section.
    */
    if(pagenumber != 0)
        qentry.attr.script = TRUE;
    else
        qentry.attr.script = FALSE;

    /*
    ** Compute the final pagefactor.
    ** The pagefactor is the number of virtual pages on each sheet
    ** of paper.  It is the N-Up N times two if we are printing
    ** in duplex.
    */
    qentry.attr.pagefactor = duplex_pagefactor * qentry.N_Up.N;

    /* 
    ** If in booklet mode (automatically sized single signature),
    ** then compute qentry.N_Up.sigsheets. 
    **
    ** We will know we are in booklet mode if qentry.N_Up.sigsheets
    ** is -1.  If we are not doing signitures, it will be 0.
    */
    if( (qentry.N_Up.sigsheets==-1) && (qentry.attr.pages>0) )
    	{
    	int sigfactor = (qentry.N_Up.N * 2);
    	qentry.N_Up.sigsheets = (qentry.attr.pages+sigfactor-1)/sigfactor;
    	}

    #ifdef ANNOYING_ROUNDING_WHICH_I_HAVE_THOUGHT_BETTER_OF
    /*
    ** Round the number of pages up to the next multiple of the
    ** smallest printable unit.  Normally, this is 1 page, but 
    ** when printing 4-Up, it is 4 pages, when printing 4-Up duplex
    ** it is 8 pages, when printing 8 sheet 2-Up duplex signitures,
    ** 32 pages.
    */
    if( qentry.attr.pagefactor != 1 )
	{
	int unit;
	
	unit = qentry.attr.pagefactor;
	
	if(qentry.N_Up.sigsheets > 1)
	    unit *= qentry.N_Up.sigsheets;
	    
	qentry.attr.pages = (qentry.attr.pages + unit - 1) / unit * unit;
	}
    #endif

    /*
    ** Attempt to resolve resource descrepancies before we
    ** write the queue file. 
    */
    rationalize_resources();

    /* 
    ** Authorization, consequences of.
    ** (This was once just after write_queue_file(), but I think
    ** that since respond() was changed this is where it belongs.) 
    */
    authorization();

    /*
    ** Give the job splitting machinery a chance to work.
    */
    if( split_job() )		/* If it does do a split, */
	{			/* remove what would have been */
	file_cleanup();		/* the files for a monolithic job. */
	}

    /* 
    ** If this else clause is executed, the job was not submitted 
    ** as a bunch of little jobs by split_job(), we must submit it
    ** ourselves as one large job. 
    */
    else
    	{
    	/* 
    	** Create and fill the queue file.  It is sad that this
    	** is the only place we check for disk full.
    	*/
    	if(write_queue_file(&qentry, 0, qentry.attr.pages, option_hold) == -1)
    	    fatal(PPREXIT_DISKFULL, "Disk full");

	/*
	** FIFO was opened above, write a command to it now.  This command
	** tells pprd that the file is ready to print.
	**
	** In this case qentry.subid will always be 0.
	*/
	fprintf(FIFO, "j %s %s %d %d %s %d %d\n",
		qentry.destnode,
		qentry.destname, qentry.id, qentry.subid, qentry.homenode,
		qentry.priority, option_hold);

    	if(option_show_id)
    	    printf("request id is %s\n", remote_jobid(qentry.destnode, qentry.destname, qentry.id, qentry.subid, qentry.homenode));
	}

    /*
    ** This goto target is used when the length of the input file is zero.
    */
    zero_length_file:

    /* 
    ** We are done with the communications channel to
    ** the spooler, we can close it now.
    */
    fclose(FIFO);

    /* 
    ** If -U switch was used, unlink the input file.  This feature
    ** is really handy if Samba is submitting the jobs! 
    */
    if( real_filename != (char*)NULL && unlink_jobfile )
	{
	if(seteuid(uid))		/* only unlink files the user may unlink */
	    fprintf(stderr, "ppr: -U: can't setuid(%ld)\n", (long)uid);

	if( chdir(starting_directory) == -1 )
	    fprintf(stderr, "ppr: -U: can't chdir(\"%s\")\n", starting_directory);

    	if( unlink(real_filename) == -1 )
    	    fprintf(stderr, "ppr: -U: can't unlink(\"%s\"), errno=%d (%s)\n", real_filename, errno, strerror(errno));

        if(seteuid(setuid_uid))          /* become "ppr" again */
            fprintf(stderr, "ppr: -U: can't setuid(%ld)\n", (long)setuid_uid);
    	}

    /* 
    ** We think we may safely state that we
    ** have fulfilled our job correctly. 
    */
    return PPREXIT_OK;
    } /* end of main */

/* end of file */
