/*
** ~ppr/src/ppr/ppr_infile.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appears in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 3 March 1997.
*/

/*
** This module is for input file routines.  It examines the input file,
** converts it to PostScript if necessary, and sets up buffering routines.
**
** If you wish to add a new input file type with filter, this
** is where you do it.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <math.h>
#include "ppr.h"
#include "ppr_exits.h"
#include "respond.h"
#include "ppr_gab.h"

/* external variables */
extern char *AuthCode;
extern int found_For;
extern int found_Routing;
extern int files_created;

/* 
** structure used to analyze the input
*/
struct ANALYZE {      
    int cr;			/* count of carriage returns */           
    int lf;			/* count of line feeds */
    int ff;			/* count of form feeds */
    int esc;			/* count of escape characters */
    int hp;			/* count of HP PCL escapes */
    int epson;			/* count of epson escapes */
    int simple_epson;		/* count of single character epson controls */
    int left_brace;		/* count of "{"'s */
    int right_brace;		/* count of "}"'s */
    int pdef;			/* count of PostScript definitions */
    int left_parenthesis;	/* count of PostScript left quotes */
    int right_parenthesis;	/* count of PostScript right quotes */
    int non_ascii;		/* count of characters 128-255 */
    int troff_dot_commands;	/* count of dot commands */
    int tex_commands;		/* count of TeX/LaTeX commands */
    int p_quoted;		/* approximate number of characters in PostScript strings */
    } ;

/*
** Input file types.  Add to this list if you want to add a new input file
** type with filter.  You must also add it to the table below.  If you want
** auto-detect to work for your type, you must add code to analyze_input().
*/

/* PostScript */
#define IN_TYPE_POSTSCRIPT 0		/* ordinary PostScript */
#define IN_TYPE_D_POSTSCRIPT 1		/* ^D, then PostScript */
#define IN_TYPE_PJL_POSTSCRIPT 2	/* PS with PJL wrapper */
#define IN_TYPE_BAD_POSTSCRIPT 3	/* tail of severed PostScript */
#define IN_TYPE_TBCP_POSTSCRIPT 4	/* tagged binary communications protocol PostScript */

/* Basically plain text */
#define IN_TYPE_LP 100			/* line printer format */
#define IN_TYPE_LP_AUTOLF 101		/* line printer, CR line endings */
#define IN_TYPE_PR 102			/* plain file, pass thru pr */
#define IN_TYPE_FORTRAN 103		/* Fortran carriage control */
#define IN_TYPE_EMAIL 104		/* RFC-822 Email messages */

/* Metafile formats */
#define IN_TYPE_CAT4 200		/* CAT/4 phototypesetter */
#define IN_TYPE_DITROFF 201		/* ditroff output */
#define IN_TYPE_DVI 202			/* TeX dvi format */
#define IN_TYPE_WMF 203			/* MS-Windows Metafile */
#define IN_TYPE_EMF 204			/* MS-Windows Enhanced Metafile */

/* Printer languages */
#define IN_TYPE_PCL 300			/* HP PCL format */
#define IN_TYPE_DOTMATRIX 301		/* Generic dot matrix printer */

/* Document languages */
#define IN_TYPE_TROFF 500		/* TRoff source */
#define IN_TYPE_TEX 501			/* TeX/LaTeX source */
#define IN_TYPE_WP 502			/* WordPerfect document */
#define IN_TYPE_RTF 503			/* Rich Text Format */
#define IN_TYPE_TEXINFO 504		/* GNU TeXInfo format */
#define IN_TYPE_HTML 505		/* Hyper Text Markup Language */
#define IN_TYPE_SGML 506		/* Other SGML */
#define IN_TYPE_PDF 507			/* Adobe Portable Document Format */

/* Graphics formats */
#define IN_TYPE_JPEG 600		/* JPEG compressed picture */
#define IN_TYPE_GIF 601			/* GIF compressed picture */
#define IN_TYPE_SUNRAS 602		/* Sun Raster format */
#define IN_TYPE_PLOT 603		/* Berkeley plot library output */
#define IN_TYPE_CIF 604			/* CIF format */
#define IN_TYPE_TIFF 605		/* TIFF format */
#define IN_TYPE_BMP 606			/* MS-Windows & OS/2 format */
#define IN_TYPE_XBM 607			/* X-Windows bitmap */
#define IN_TYPE_XPM 608			/* X-Windows pixel map */
#define IN_TYPE_XWD 609			/* X-Windows window dump */
#define IN_TYPE_PNM 610			/* Portable aNy Map */
#define IN_TYPE_PNG 611			/* Portable Network Graphics format */
#define IN_TYPE_FIG 612			/* FIG format */

#define IN_TYPE_UNKNOWN 1000		/* unidentified file type */
#define IN_TYPE_BINARY 1001		/* unidentified binary file */

/*
** Structure of an entry in the table of filters.
*/
struct FILTER
    {
    int in_type;		/* IN_TYPE_* */
    char *name;			/* keyword for -T switch */
    char *help;			/* help description */
    char *excuse;		/* what we can't print */
    } ; 

/*
** The table of filters.  The first entry, "postscript", is not really
** a filter at all, there is special case code in get_input_file()
** that takes care of it.
**
** New input types must be added here.  It is also best to add auto-detect
** code to analyze_input().
*/	
struct FILTER filters[]=
{
/* IN_TYPE_*		-T name		-T description					can't print message */
{IN_TYPE_POSTSCRIPT,	"postscript",	"input file is PostScript",			""},
{IN_TYPE_LP,		"lp",		"pass file thru line printer emulator",		"line printer files"},
{IN_TYPE_LP_AUTOLF,	"lp_autolf",	"file type is Macintosh text",			"Macintosh text"},
{IN_TYPE_PR,		"pr",		"pass file thru pr",				"using pr"},
{IN_TYPE_FORTRAN,	"fortran",	"file employs Fortran carriage control",	"Fortran files"},
{IN_TYPE_EMAIL,		"email",	"file is an RFC-??? email message",		"email messages"},
{IN_TYPE_CAT4,		"cat4",		"file is old troff output",			"CAT/4 (old Troff) files"},
{IN_TYPE_DITROFF,	"ditroff",	"file type is Ditroff (modern Troff) output",	"Ditroff output"},
{IN_TYPE_DVI,		"dvi",		"file type is TeX DVI",				"TeX DVI files"},
{IN_TYPE_WMF,		"wmf",		"file is an MS-Windows Metafile",		"MS-Windows Metafiles"},
{IN_TYPE_EMF,		"emf",		"file is an MS-Windows Enhanced Metafile",	"MS-Windows Enhanced Metafiles"},
{IN_TYPE_PCL,		"pcl",		"file type is HP PCL",				"HP PCL files"},
{IN_TYPE_DOTMATRIX,	"dotmatrix",	"file type is generic dot matrix",		"dot matrix printer files"},
{IN_TYPE_TROFF,		"troff",	"file is Troff source",				"Troff source files"},
{IN_TYPE_TEX,		"tex",		"file is TeX or LaTeX source",			"TeX and LaTeX source"},
{IN_TYPE_WP,		"wp",		"file is a WordPerfect document",		"WordPerfect documents"},
{IN_TYPE_RTF,		"rtf",		"file is in Reich Text Format",			"Reich Text Format documents"},
{IN_TYPE_TEXINFO,	"texinfo",	"file is in TeXInfo format",			"TeXInfo documents"},
{IN_TYPE_HTML,		"html",		"file is in HTML format",			"HTML documents"},
{IN_TYPE_SGML,		"sgml",		"file is in SGML format (other than HTML)",	"SGML documents"},
{IN_TYPE_PDF,		"pdf",		"file is in Adobe Portable Document Format",	"Adobe Portable Document Format files"},
{IN_TYPE_JPEG,		"jpeg",		"file is in JPEG File Interchange Format",	"JPEG (JFIF) pictures"},
{IN_TYPE_GIF,		"gif",		"file is a GIF compressed picture",		"GIF pictures"},
{IN_TYPE_SUNRAS,	"sunras",	"file is in Sun raster format",			"Sun raster files"},
{IN_TYPE_PLOT,		"plot",		"file is in Berkeley plot library format",	"Berkeley Plot files"},
{IN_TYPE_CIF,		"cif",		"file is in CIF graphics language",		"CIF files"},
{IN_TYPE_TIFF,		"tiff",		"file is in TIFF graphics format",		"TIFF pictures"},
{IN_TYPE_BMP,		"bmp",		"file is an MS-Windows or OS/2 BMP",		"MS-Windows Bitmaps"},
{IN_TYPE_XBM,		"xbm",		"file is an X-Windows bit map",			"X-Windows bitmaps"},
{IN_TYPE_XPM,		"xpm",		"file is an X-Windows pixel map",		"X-Windows pixel maps"},
{IN_TYPE_XWD,		"xwd",		"file is an X-Windows window dump",		"X-Windows window dumps"},
{IN_TYPE_PNM,		"pnm",		"file is a Portable aNy Map",			"Portable aNy Maps"},
{IN_TYPE_PNG,		"png",		"file is in Portable Network Graphics format",	"Portable Network Graphics files"},
{IN_TYPE_FIG,		"fig",		"file is in FIG format",			"FIG files"}
} ;

static const int filters_count = ( sizeof(filters) / sizeof(filters[0]) );

/* This static will be used until I can find a better way. */
static int input_is_file;

/* Type of input file, -1 means `unknown as yet'. */
static int in_type = -1;

/* input file handle */
static int in_handle;

/* The input file buffer */
static unsigned char *in_buffer_rock_bottom;	/* used for in_ungetc() */
static unsigned char *in_buffer;		/* start of input buffer */
static int in_bsize;				/* size in bytes of input buffer */
static unsigned char *in_ptr;			/* pointer to next character in buffer */
static int in_left;				/* characters remaining in in buffer */
static int in_eof;				/* TRUE if physical end of file yet */
static int buffer_count = 0;			/* number of buffer loads done */

/*
** Used to avoid messing with cr or lf in binary tokens.
** Used in getline() and exec_filter().
*/
static int bintoken_grace = 0;

/*
** Load the next block into the input file buffer.
**
** The bytes are read into in_buffer.  Up to in_bsize bytes are
** read.  The in_ptr is reset to the start of the buffer.
**
** The number of bytes read is added to the count of PostScript
** input bytes.  If it later turns out that the stuff we were
** reading wasn't PostScript we will reset this variable to zero.
*/
static void load_in_buffer(void)
    {
    buffer_count++;			/* add to count of input buffers */

    /* Under DEC OSF/1 3.2 we get mysteriously interupted
       system calls.  That is why we have to be fancy here. */
    while( (in_left = read(in_handle, in_buffer, in_bsize)) == -1 )
	{
	if(errno != EINTR)
	    fatal(PPREXIT_OTHERERR, "read() failed on input file, errno=%d (%s)", errno, strerror(errno));
	}

    if(in_left == 0)			/* if didn't get all we asked for, */
        in_eof = TRUE;			/* it was end of file */

    in_ptr = in_buffer;

    qentry.attr.postscript_bytes += in_left;
    } /* end of load_in_buffer() */

/*
** Get a single character from the input file.
** Return EOF if nothing is left. 
*/
int in_getc(void)
    {
    while( in_left == 0 )	/* we use while and not if because */
        {                       /* the file size may be an exact */
        if(in_eof)              /* multiple of the buffer size */
            return EOF;
        load_in_buffer();
        }

    in_left--;                  /* there will be one less */
    return *(in_ptr++);         /* return the character, INCing pointer */    
    } /* end of in_getc() */

/*
** "unget" an input character.
**
** If we think that the caller is attempting to 
** put back an end of file indicator, just ignore it.
**
** This routine guaratees that we can put back at
** least 11 bytes.
*/
void in_ungetc(int c)
    {
    if( c != EOF )
	{
	if( in_ptr > in_buffer_rock_bottom )
	    {                  
	    in_ptr--;
	    in_left++;
	    *in_ptr = c;
	    }
	else
	    {
            fatal(PPREXIT_OTHERERR, "in_ungetc(): too far!");
	    }
	}
    } /* end of in_ungetc() */

/*
** Read a line from the input file, into line[].
**
** In doing so, the line ending will be removed.
** The length of the line (exclusive of the line
** ending) is placed in line_len.
**
** Called only by _clean_getline(), copy_data(), read_header(), 
** and check_for_PJL().
**
** This function shows special consideration for binary 
** token encoding (see RBII pp. 106-119).
**
** This function may get logical_eof.  Once it is set TRUE,
** this function will return zero length lines.  clean_getline()
** will set logical_eof to TRUE if it sees "%%EOF", thus our setting
** it just before returning a "%%EOF" line which we have substituted
** for a protocol specific EOF indication is redundant, but we
** are going to do it anyway.
*/
void getline(void)
    {
    int c, c2, c3, c4;
    unsigned int x = 0;
    int in_comment = FALSE;

    line_overflow = FALSE;			/* we don't yet know it will overflow */

    while(!logical_eof)				/* Don't try to read past */
        {					/* logical end of file. */
        if( (c=in_getc()) == EOF )		/* Physical end of file */
            {					/* makes a good */
            logical_eof = TRUE;			/* logical end of file :)*/
            break;
            }
        if( c == '\n' && !bintoken_grace )	/* if line feed, */
            {
            if( (c=in_getc()) != '\r' )		/* Eat a carriage return */
                in_ungetc(c);			/* if it is there */
            break;				/* accept as end of line */
            }
        if( c == '\r' && !bintoken_grace )	/* if CR, */
            {
            if( (c=in_getc()) != '\n' )		/* eat a line feed */
                in_ungetc(c);			/* if it is there */
            break;				/* accept as end of line */
            }
	if( c == '%' )				/* If start of comment, set flag */
	    in_comment = TRUE;			/* for ignoring binary tokens. */
        
	/* It passed those tests!  Put the character into the buffer. */
	line[x++] = c;

	/*
	** This block maintains the value of bintoken_grace which
	*/
	if(bintoken_grace)
	    bintoken_grace--;
	else if(! in_comment)
	    {
	    switch(c)
	    	{
		case 128:		/* binary object sequences */
		case 129:		/* (not implemented) */
		case 130:
		case 131:
		    break;

	    	case 132:		/* 32-bit integer, high-order byte first */
	    	case 133:		/* 32-bit integer, low-order byte first */
	    	case 138:		/* 32-bit IEEE standard real, high-order byte first */
	    	case 139:		/* 32-bit IEEE standard real, low-order byte first */
	    	case 140:		/* 32-bit native real */
	    	    bintoken_grace = 4;
	    	    break;
	    	case 134:		/* 16-bit integer, high-order byte first */
	    	case 135:		/* 16-bit integer, low-order byte first */
	    	    bintoken_grace = 2;
	    	    break;
	    	case 136:		/* 8-bit integer, signed */
	    	case 141:		/* Boolean */
	    	case 145:		/* Literal name from the system name table */
	    	case 146:		/* Executable name from the system name table */
	    	case 147:		/* Literal name from the user name table */
		case 148:		/* Executable name from the user name table */
	    	    bintoken_grace = 1;
	    	    break;
	    	case 137:		/* 16 or 32 bit fixed point !!! */
		    if( (c2=in_getc()) <= 31 || (c2 >= 128 && c2 <= 159) )
		    	bintoken_grace = 2;
		    else
		    	bintoken_grace = 5;
		    in_ungetc(c2);
	    	    break;
		case 142:		/* String of length c2 */
		    c2 = in_getc();
		    bintoken_grace = c2 + 1;
		    in_ungetc(c2);
		    break;
		case 143:		/* String of length c2,c3, high-order byte first */
		    c2 = in_getc();
		    c3 = in_getc();
		    in_ungetc(c3);
		    in_ungetc(c2);
		    bintoken_grace = (c2 * 256 + c3) + 2;
		    break;		    
		case 144:		/* String of length c2,c3, low-order byte first */
		    c2 = in_getc();
		    c3 = in_getc();
		    in_ungetc(c3);
		    in_ungetc(c2);
		    bintoken_grace = c3 * 256 + c2;
		    break;
		case 149:		/* Homogenous number array */
		    c2 = in_getc();	/* <-- type or numbers */
		    c3 = in_getc();	/* <-- length byte 1 */
		    c4 = in_getc();	/* <-- length byte 2 */
		    in_ungetc(c4);
		    in_ungetc(c3);
		    in_ungetc(c2);
		    if( c2 <= 31 || c2 == 48 || c2 == 49 )	/* 32-bit fixed point or real, */
		    	bintoken_grace = (c3 * 256 + c4) * 4;	/* high-order byte first */
		    else if( c2 <= 47 )				/* 16-bit fixed point, */ 
		    	bintoken_grace = (c3 * 256 + c4) * 2;	/* high-order byte first */
		    else if( c2 < 128 )				/* Undefined */
		    	bintoken_grace = 0;
		    else if( (c2-=128) <= 31 || c2 == 48 || c2 == 49 )	/* same, low-order byte first */
		    	bintoken_grace = (c4 * 256 + c3) * 4;
		    else if( c2 <= 47 )
		    	bintoken_grace = (c4 * 256 + c3) * 2;
		    else
		    	bintoken_grace = 0;
	    	    break;
	    	}
	    }
	    
	/*
	** Handle line overflows.  Normally, we just note the
	** fact, if we set line_overflow to TRUE, the line will
	** be re-assembled properly.  If it is a hexadecimal 
	** data line we just split it up.
	*/
        if( x >= MAX_LINE )
	    {
	    line[x] = (char)NULL;	/* terminate what we have, */

#ifdef OBSOLETE_HEX_HANDLING
	    static int hex_line_overflow = FALSE;

	    if(strspn(line,"01234567890abcdefABCDEF") == x)
	        {
		if(!hex_line_overflow)	/* issue warning only once */
		    {
		    hex_line_overflow = TRUE;
		    warning(WARNING_PEEVE, "overly long hexadecimal line wrapped");
		    }
		line_len = x;
	        return;
		}
	    else			/* not a hexadecimal line */
	    	{
#endif
		line_overflow = TRUE;
		line_len = x;
		return;
#ifdef OBSOLETE_HEX_HANDLING
	    	}
#endif
            } /* if(x >= MAX_LINE) */

        } /* while( !logical_eof ) */

    line[x] = (char)NULL;		/* Terminate line, */
    line_len = x;			/* and pass length back in global variable. */

    /*
    ** Accept a lone control-D as EOF, but don't be nice about it.
    **
    ** Of course, this might cause problems if we are receiving
    ** binary data.  As a quick hack, we won't accept ^D as EOF
    ** if TBCP is being used.
    */
    if( line[0] == 4 && line[1] == (char)NULL && in_type != IN_TYPE_TBCP_POSTSCRIPT )
        {
        warning(WARNING_PEEVE, "^D as EOF is spurious in a spooled environment");
        strcpy(line, "%%EOF");
	line_len = 5;
        logical_eof = TRUE;
        }

    /* 
    ** If this file had HP Printer Job Language commands at
    ** the begining, then accept "Universal Exit Language" as EOF.
    **
    ** (Actually, no control-D will proceed UEL when in_type is
    ** IN_TYPE_TBCP_POSTSCRIPT because tbcp2bin will have deleted
    ** it if it was present.)
    */
    if( (in_type == IN_TYPE_PJL_POSTSCRIPT || in_type == IN_TYPE_TBCP_POSTSCRIPT)
            && ( ((line[0]==27)&&(strncmp(line,"\x1b%-12345X",9)==0))
            	|| ((line[0]==4)&&(strncmp(line,"\x04\x1b%-12345X",10)==0)) ) )
        {
        strcpy(line,"%%EOF");
        line_len = 5;
        logical_eof = TRUE;
        }

    } /* end of getline() */

/*
** This function is so that in_handle does not 
** have to be externaly visible.
*/
void close_infile(void)
    {
    close(in_handle);
    } /* end of close_infile() */

/*
** Read an identifying header if one is present.
**
** These headers may be used on non-PostScript jobs
** to give information which would normally be placed in
** the PostScript header comments.
**
** The program PRNAUTH for MS-DOS is an example of a program 
** which may generate such headers.
**
** PRNAUTH was developed for use with PPR.  These headers were
** invented by David Chappell.
*/
static void read_header(void)
    {
    if(in_buffer[0] != '.' || in_buffer[1] != '!')	/* If header not present, */
        return;						/* do nothing. */

    getline();				/* Read the ".!" line. */

    while(!logical_eof)			/* Stop if end of file */
        {                           
        getline();			/* or */

        if(strcmp(line,".end")==0)	/* ".end" line */
            return;

        /* 
        ** If we find a ".For:" line, act as we would if we later found
        ** a "%%For:" line in the PostScript.
        */
        if(read_for && strncmp(line, ".For: ", 6) == 0)
            {
            qentry.ForLine = mystrdup(&line[6]);
            found_For = TRUE;
            }
        
        /*
        ** If we find an ".AuthCode:" line, we will act just as we later
	** would if we found a "%TCHCTAuthCode:" line.
	*/
        else if( strncmp(line, ".AuthCode: ", 11) == 0 ) 
            {
            AuthCode = mystrdup(&line[11]);         
            }

        /*
        ** If we find a ".Routing:" line, we will act just as we
	** later would if we found a "%%Routing:" line.
	*/
        else if( strncmp(tokens[0], ".Routing: ", 10) == 0 ) 
            {
            qentry.Routing = mystrdup(&line[10]);
            found_Routing = TRUE;
            }
        }

    /*
    ** Check if a second buffer was loaded while we
    ** where reading the header.
    */
    if(buffer_count!=1)
        fatal(PPREXIT_BADHEADER,"dot header is unterminated or exceeds buffer size");

    } /* end of read_header() */

/*
** If a PJL header is found, strip it off after setting the language
** type to that which corresponds to the "ENTER LANGUAGE" command.
**
** If the "ENTER LANGUAGE" command specifies either an unknown
** language or one for which a filter is required, but none is
** available, the job will rejected.  If the langauge is unknown,
** it will sort of happen in this function, if the filter is missing,
** the problem will not be discoved until we try to execute it.  That
** happens well after this function is done.
*/
static void check_for_PJL(void)
    {
    char *ptr;
    int len;
    int c;

    /*
    ** If "universal exit language" command
    ** is not found at the start of the job,
    ** there is nothing for this routine to do.
    */
    if( in_left < 9 || strncmp((char*)in_ptr, "\x1b%-12345X", 9) )
        return;

    /* Consume UEL. */
    in_ptr += 9;
    in_left -= 9;

    while(!logical_eof)
        {
	getline();				/* Read a PJL line */

	if(strncmp(line, "@PJL", 4))		/* if not PJL line */
	    return;				/* then go back to auto-detect */

	ptr = &line[4+strspn(&line[4], " \t")];	/* eat whitespace */

	if(icmpn(ptr, "ENTER", 5))		/* look for "ENTER" */
	    continue;				/* ignore line if "ENTER" not found */

	ptr = &ptr[5+strspn(&ptr[5], " \t")];	/* eat "ENTER" and whitespace */

	if(icmpn(ptr, "LANGUAGE", 8))		/* ignore this line if */
	    continue;				/* "LANGUAGE" not found */

	ptr = &ptr[8+strspn(&ptr[8], " \t")];	/* eat "LANGUAGE" and white */

	if(*ptr != '=')				/* if "=" doesn't follow, */
	    continue;				/* ignore this line */

        ptr = &ptr[1+strspn(&ptr[1], " \t")];	/* eat "=" and white space */

        len = strlen(ptr);			/* remove trailing blanks */
        while(--len && isspace(ptr[len]) )
            ptr[len] = (char)NULL;

	if(gab_mask & GAB_INFILE_AUTOTYPE)
	    printf("@PJL ENTER LANGUAGE = %s\n", ptr);

	/* Accept the specified input type, ignore any error return. */
        force_in_type(ptr);

	/* If PJL code selected PostScript, */
        if( in_type == IN_TYPE_POSTSCRIPT )
            {   
	    /* Set to special kind of PS */
            in_type = IN_TYPE_PJL_POSTSCRIPT;

	    /*
	    ** Eat up control-d which some drivers insert 
	    ** and blank line inserted by Adobe MS-Windows driver 4.
	    */
            while( (c=in_getc()) == 4 || c == '\r' || c == '\n' ) ;
	    in_ungetc(c);
            }

        return;
        } /* this while loop only ends if we hit EOF */
    
    if( buffer_count != 1 )
        fatal(PPREXIT_BADHEADER, "PJL header exceeds buffer size");
    } /* end of check_for_PJL() */

/*
** This routine counts things in the first block 
** of the input file.  It is a state machine.
**
** This routine is only called after an attempt to identify the
** file by magic number has failed.  The statistics gathered by
** this routine are used to make educated guesses as to the
** type of the file's contents.
*/
#define INITIAL 0       /* after each carriage return */
#define PSYMBOL 1       /* after "/" in column 1 */
#define ESCAPE	2       /* escape seen */
#define ESCAPE_AND 3	/* second character of HP escape found */
#define MIDLINE 4       /* after 1st character */
#define DOT	5	/* line began with dot */
#define DOT1	6	/* first character of dot command seen */
#define DOT2	7	/* second character of dot command seen */
#define BSLASH	8	/* line begin with backslash */
                                                
static void count_things(struct ANALYZE *ccount)
    {
    int x;
    int state=INITIAL;
    int p_quote_level=0;		/* 1 if we are in PostScript string */

    ccount->cr=                         /* zero all the counts */
        ccount->lf=
        ccount->ff=         
        ccount->esc=
        ccount->hp=
	ccount->epson=
	ccount->simple_epson=
        ccount->left_brace=
        ccount->right_brace=
        ccount->pdef=
        ccount->left_parenthesis=
        ccount->right_parenthesis=
        ccount->non_ascii=
        ccount->troff_dot_commands=
        ccount->tex_commands=
        ccount->p_quoted=0;
                            
    for(x=0; x<in_left; x++)	    /* scan the whole buffer */
	{
        switch(in_ptr[x])
            {
            case 13:                /* carriage returns */
                ccount->cr++;       /* are counted */
                state=INITIAL;      /* and reset to initial state */
                p_quote_level=0;    /* PostScript strings do not cross line bounderies */
                break;              
            case 10:                /* line feeds */
                ccount->lf++;       /* are counted */
                state=INITIAL;      /* and reset the state to the initial */
                p_quote_level=0;    /* PostScript strings do not cross line bounderies */
                break;              
            case 12:                /* form feeds */
                ccount->ff++;       /* are counted */
                state=INITIAL;      /* and reset the state to the initial */
                break;
            case 27:                /* escapes */
                ccount->esc++;      /* are counted */
                state=ESCAPE;       /* and set the state to a new level */
                break;
            case '{':                   /* left braces */
                ccount->left_brace++;   /* are counted */
                if(state==PSYMBOL)      /* and if PostScript symbol state, */
                    ccount->pdef++;     /* are counted as a definition */
                state=MIDLINE; /* <-- formerly was else... */
                break;
            case '}':                   /* right braces */
                ccount->right_brace++;  /* are counted */
                state=MIDLINE;
                break; 
	    case '*':			/* 2nd HP escape character */
		if(state==ESCAPE)	/* or maybe epson graphics, */	
		    ccount->epson++;	/* fall thru */
            case '&':                   /* 2nd char of HP escape */
                if(state==ESCAPE)       /* if the state is escape */
                    {                   
                    state=ESCAPE_AND;   /* go to next state */
                    break;
                    }
                goto _default;
	    case '@':			/* epson reset */
	    case 'K':			/* graphics */
	    case 'L':			/* graphics */
	    case 'Y':			/* graphics */
	    case 'Z':			/* graphics */
	    case '^':			/* graphics */
	    	if(state==ESCAPE)	/* (Epson graphics could otherwise */
	    	    {			/* cause false binary detection.) */
		    ccount->epson++;
	    	    state=INITIAL;
	    	    break;
	    	    }
	    	goto _default;
	    case 14:			/* simple Epson codes */
	    case 15:
	    case 18:
	    case 20:
	    	ccount->simple_epson++;
	    	state=MIDLINE;
	    	break;
            case 'l':			/* third character of HP escape */
            case 'p':
                if(state==ESCAPE_AND)
                    {
                    ccount->hp++;
                    state=MIDLINE;
                    break;
                    }
                goto _default;
            case '/':                   /* slash in initial state */
                if(state==INITIAL)      /* may be a PostScript symbol */
                    state=PSYMBOL;
                else
                    state=MIDLINE;
                break;
	    case '(':			/* left parenthesis */
                if(state==ESCAPE)       /* If the state is ESCAPE, it is */
                    {                   /* second character of HP escape, */
                    state=ESCAPE_AND;   /* go to next state. */
                    }
	    	else			/* Otherwise, may be start of */
	    	    {			/* a PostScript string. */
	    	    ccount->left_parenthesis++;
	    	    state=MIDLINE;
		    p_quote_level++;
		    }
		break;
	    case ')':			/* right parenthesis */
	    	ccount->right_parenthesis++;
		state=MIDLINE;
		if(p_quote_level)	/* PostScript closing quote */
		    p_quote_level--;
		break;
	    case '.':
	    	if(state==INITIAL)
	    	    state=DOT;
	    	break;
	    case 0x5C:
	    	if(state==INITIAL)
	    	    state=BSLASH;
		break;
            _default:
            default:                    /* other characters change state */
                if(in_ptr[x] > 127)	/* count out of ASCII range byte codes */
                    ccount->non_ascii++;    
                    
		if(state==DOT && isalpha(in_ptr[x]))
		    {
		    state=DOT1;
		    break;		/* break prevents change to MIDLINE state */
		    }

		if(state==DOT1 && isalpha(in_ptr[x]))
		    {
		    state=DOT2;
		    break;		/* break prevents change to MIDLINE state */
		    }

		if( (state==DOT1 || state==DOT2) && isspace(in_ptr[x]) )
		    ccount->troff_dot_commands++;

		if(state==BSLASH && isalpha(in_ptr[x]))	/* backslash command */
		    ccount->tex_commands++;

                if(state!=PSYMBOL)	/* except in PostScript symbol */ 
                    state=MIDLINE;

                if(p_quote_level==1)	/* if in PostScript quotes */
                    ccount->p_quoted++;

		break;                    
            } /* end of switch */
	} /* end of for loop */
    } /* end of count_things() */

/*
** Determine what type of input we must deal with.
**
** First, we try to identify the file by magic number.  If that
** doesn't work, we call count_things() and try to make an 
** educated guess based uppon the quantities of certain characters
** and constructs present.
*/
static int analyze_input(void)
    {
    struct ANALYZE ccount;

    /*
    ** Anything begining with "%!" is PostScript.
    */
    if(in_ptr[0] == '%'  && in_ptr[1] == '!')
        return IN_TYPE_POSTSCRIPT;

    /*
    ** Anything begining with control-D is probably PostScript
    ** generated by a stupid code generator which does not know
    ** it is talking to a spooler.
    */
    if(*in_ptr == 4)
        return IN_TYPE_D_POSTSCRIPT;            

    /*
    ** If it begins with "#!" it is code for some sort of script language
    ** such as sh, perl, or tcl.
    */
    if(in_ptr[0] == '#' && in_ptr[1] == '!')
    	return IN_TYPE_LP;

    /*
    ** If the 1st line is the start of a ditroff style output device
    ** type line, assume it is ditroff output.
    */
    if(strncmp((char*)in_ptr, "x T ", 4) == 0)
        return IN_TYPE_DITROFF;

    /*
    ** If the 1st two bytes what I think is the signiture for TeX
    ** DVI files, assume this is one.  (This relies on the fact
    ** that the input buffer is unsigned chars.)
    */
    if(in_ptr[0] == 0xF7 && in_ptr[1] == 0x02)
        return IN_TYPE_DVI;

    /*
    ** MS-Windows placable metafiles have the signiture 0x9AC6CDD7.
    */
    if(in_ptr[0] == 0xD7 && in_ptr[1] == 0xCD && in_ptr[2] == 0xC6 && in_ptr[3] == 0x9A)
    	return IN_TYPE_WMF;

    /*
    ** Test for GIF files.  The signature is 
    ** "GIF87a" or "GIF89a".
    */
    if( in_left > 6 && (strncmp((char*)in_ptr,"GIF87a",6)==0 || strncmp((char*)in_ptr,"GIF89a",6)==0) )
    	return IN_TYPE_GIF;
    	
    /*
    ** Test for JFIF (JPEG) files.
    */
    if(in_left > 10 && in_ptr[0] == 0xFF && in_ptr[1] == 0xD8
    		&& strncmp((char*)&in_ptr[6], "JFIF", 4) == 0)
    	return IN_TYPE_JPEG;
    	
    /*
    ** Test for TIFF files, both big and little endian.
    */
    if(in_left > 2 && ( (in_ptr[0]=='\115' && in_ptr[1]=='\115') 
    		      || (in_ptr[0]=='\111' && in_ptr[1]=='\111') ) )
	return IN_TYPE_TIFF;
	
    /*
    ** Test for MS-Windows and OS/2 BMP files.
    */
    if( in_left > 2 && strncmp((char*)in_ptr,"BM",2)==0 )
	return IN_TYPE_BMP;

    /*
    ** Test for portable bit/gray/pixel maps.
    */
    if( in_left > 2 && in_ptr[0] == 'P' && in_ptr[1] >= '0' && in_ptr[1] <= '6' )
    	return IN_TYPE_PNM;

    /*
    ** Test for X-Windows bit maps by looking for the XBM signiture 
    ** Since the signiture is a C comment we must use special quoting to
    ** get it thru cpp.
    */
    if(in_left > 10 && strncmp((char*)in_ptr, "/""* XBM *""/", 9) == 0)
	return IN_TYPE_XBM;

    /*
    ** Most X-Windows bit maps do not have a signiture.  Since they are 
    ** valid C code we must be careful not to misidentify ordinary C code
    ** as an X bitmap.
    */
    if(in_left > 60 && strncmp((char*)in_ptr, "#define", 7) == 0 && isspace(((char*)in_ptr)[7]))
	{
	char copy[200];		/* Space for ASCIIZ copy of header */
	char *p, *p2, *name;
	int namelen;

	/* Make a NULL terminated copy of the supposed XBM header. */
	strncpy(copy, in_ptr + 8, 200);
	copy[199] = (char)NULL;

	/* Locate the bitmap name. */
	name = copy;
	name += strspn(name, " \t");
	if( (p = strstr(name, "_width")) != (char*)NULL && (p2 = strchr(name, '\n')) != (char*)NULL && p < p2 )
	    {
	    namelen = p - name;
	
	    if(strncmp(name + namelen, "_width", 6) == 0)
		{
		p += strcspn(p, "\n");		/* move to next line */
		p += strspn(p, "\n");

		if(strncmp(p, "#define ", 8) == 0)
		    {
		    p = p + 8;
		    p += strspn(p, " \t");
		    if(strncmp(p, name, namelen) == 0 && strncmp(p + namelen, "_height", 7) == 0)
			{
			return IN_TYPE_XBM;
			}
		    }
		}
	    }
	}

    /*
    ** Test for X-Windows pixel maps.  Again, we must use quoting
    ** tricks so that the signiture is not recognized as a comment when
    ** we compile this.
    */
    if( in_left > 10 && strncmp((char*)in_ptr,"/""* XPM *""/",9)==0 )
    	return IN_TYPE_XPM;
 
    /*
    ** Test for X-Windows window dump.
    */
    if( in_left > 8 && in_ptr[4]==0 && in_ptr[5]==0
    		&& in_ptr[6]==0 && in_ptr[7]==7 )
    	return IN_TYPE_XWD;

    /*
    ** Portable Network Graphics format.
    ** This format has an 8 byte magic number.
    */
    if( in_left > 8 && memcmp(in_ptr,"\x89\x50\x4E\x47\x0D\x0A\x1A\x0a",8)==0 )
    	return IN_TYPE_PNG;

    /*
    ** Test for WordPerfect files.
    ** At present, we can't distinguish between
    ** the different types of WordPerfect files.
    */
    if( in_left > 4 && in_ptr[0]==0xFF && strncmp((char*)&in_ptr[1],"WPC",3)==0 )
	return IN_TYPE_WP;

    /*
    ** Test for TeXInfo files.
    ** We identify them by the fact that they begin
    ** with an instruction to TeX to load the texinfo macros.
    */
    if( in_left > 14 && strncmp((char*)in_ptr,"\\input texinfo",14)==0 )
	return IN_TYPE_TEXINFO;

    /*
    ** Test for HTML files which are tagged as such,
    ** then test for other types of SGML. 
    */
    {
    char *p = (char*)in_ptr;
    int left = in_left;
    
    while(left > 0 && isspace(*p))
    	{
    	p++;
    	left--;
    	}
    
    if(left > 6 && icmpn(p, "<HTML>", 6) == 0)
    	return IN_TYPE_HTML;

    if(left > 30 && icmpn(p, "<!DOCTYPE html PUBLIC", 21) == 0)
    	return IN_TYPE_HTML;

    if(left > 30 && icmpn(p, "<!DOCTYPE ", 10) == 0)
    	return IN_TYPE_SGML;
    }
    
    /*
    ** Test for Adobe Portable Document Format.
    */
    if( in_left > 5 && strncmp((char*)in_ptr, "%PDF-", 5) == 0 )
	return IN_TYPE_PDF;

    /*
    ** Test for plot format files.  This test will only work if
    ** the plot size is square.  I don't know, this may always
    ** be the case.
    */
    if( in_left > 9 && in_ptr[0]=='s' 
	&& in_ptr[1]==(unsigned char)0 && in_ptr[2]==(unsigned char)0
	&& in_ptr[3]==(unsigned char)0 && in_ptr[4]==(unsigned char)0
	&& in_ptr[5]==in_ptr[7] && in_ptr[6]==in_ptr[8] )
    	return IN_TYPE_PLOT;

    /*
    ** The FIG figure drawing format.  It begins with a line like
    ** #FIG 2.1
    */
    if(in_left > 10 && strncmp((char*)in_ptr, "#FIG ", 5) == 0
    		&& strspn((char*)&in_ptr[5], "0123456789.") == strcspn((char*)&in_ptr[5], "\n"))
    	return IN_TYPE_FIG;

    /*
    ** Things are getting difficult, time to roll out the big guns.
    ** Get statistics for the buffer.
    */
    count_things(&ccount);

    /* Print out debugging information */
    if( gab_mask & GAB_INFILE_AUTOTYPE )
	{
	printf("Statistics for %d bytes:\n\t{\n",(int)in_left);
	printf("\tcr = %d\n",ccount.cr);
	printf("\tlf = %d\n",ccount.lf);
	printf("\tff = %d\n",ccount.ff);
	printf("\tesc = %d\n",ccount.esc);
	printf("\tleft_brace = %d\n",ccount.left_brace);
	printf("\tright_brace = %d\n",ccount.right_brace);
	printf("\tpdef = %d\n",ccount.pdef);
	printf("\thp = %d\n",ccount.hp);
	printf("\tepson = %d\n",ccount.epson);
	printf("\tsimple_epson = %d\n",ccount.simple_epson);
	printf("\tleft_parenthesis = %d\n",ccount.left_parenthesis);
	printf("\tright_parenthesis = %d\n",ccount.right_parenthesis);
	printf("\tnon_ascii = %d\n",ccount.non_ascii);
	printf("\tp_quoted = %d\n",ccount.p_quoted);
	printf("\t}\n");
	}

    /* 
    ** If there is at least one escape code that is certainly not
    ** an Epson escape code, conclude that the file is in PCL.
    */
    if( (ccount.hp - ccount.epson) >= 1 )
        return IN_TYPE_PCL;

    /*
    ** First try at identifying Epson stuff.  This one will work only
    ** if the input file contains at least one Epson escape code.
    ** Not all files which could benefit from being passed thru
    ** the dotmatrix filter will meet this condition since Epson
    ** printers support some single character control codes.
    */
    if( ccount.epson > 0 )
    	return IN_TYPE_DOTMATRIX;

    /* 
    ** If we have gotten this far and there is an awful lot
    ** of non-ASCII stuff, assume it is binary.  Had the file 
    ** contained a dot matrix printer graphic, it would have 
    ** been caught by the clause above.
    */
    if( ccount.non_ascii > (in_left/5+1) )	/* 20% */
    	return IN_TYPE_BINARY;

    /* 
    ** If there is at least one escape code, identify file as
    ** dot matrix printer format.  Or, if at least two
    ** (presumably a pair of) single byte dot matrix
    ** printer codes.
    */
    if( ccount.esc > 0 || ccount.simple_epson > 1 )
    	return IN_TYPE_DOTMATRIX;

    /*
    ** Last ditch effort to identify PostScript.
    ** If the file is more than 0.1% left braces and more than
    ** 0.1% right braces and the braces are less than 10%
    ** mismatched and there is at least one object which looks
    ** like a PostScript definition, then decide it is PostScript.
    */
    if( (ccount.left_brace > (in_left/1000)) 
        && (ccount.right_brace > (in_left/1000))
        && ( fabs( (double)(ccount.left_brace-ccount.right_brace)
                /
            (double)(ccount.left_brace+ccount.right_brace) ) < 0.05 )
        && (ccount.pdef > 0) )				/* ^ yes, this is 10% ^ */
        return IN_TYPE_POSTSCRIPT;

    /*
    ** Test if this is the tail of a fragmented PostScript job.
    ** Microsoft WorkGroups for DOS has been known to break
    ** a job into fragments.
    **
    ** The qualification is if there are lots of parenthesis which
    ** are mostly matched and there are few curly brackets.
    ** To this we will add that at least 50% of the characters
    ** must be enclosed in PostScript strings.
    */
    if( (ccount.left_parenthesis + ccount.right_parenthesis)
    			> (in_left/20)		/* more than 5% */
    		&& (ccount.left_brace + ccount.right_brace)
    			<= (in_left/1000)	/* less than 0.1% */
    		&& fabs( (double)(ccount.left_parenthesis-ccount.right_parenthesis)
    				/
    		    (double)(ccount.left_parenthesis+ccount.right_parenthesis) )
    			< 0.05	/* less than 10% mismatch */
		&& ccount.p_quoted > (in_left/2)
    	)			/* at least 50% of characters quoted */
    	return IN_TYPE_BAD_POSTSCRIPT;

    /* If it has dot commands, it is Troff. */
    if( ccount.troff_dot_commands > 5 )
    	return IN_TYPE_TROFF;
    	
    /* If it has backslash commands, it is TeX. */
    if( ccount.tex_commands > 2 )
    	return IN_TYPE_TEX;

    /*
    ** If this has carriage returns but no line feeds and
    ** we didn't identify it as PostScript in the step above,
    ** assume it is Macintosh style ASCII text.
    */
    if( (ccount.cr>0) && (ccount.lf==0) )      
        return IN_TYPE_LP_AUTOLF;              

    /*
    ** Anything unclaimed so far we will assume
    ** to be ASCII text or line printer format.
    */
    return IN_TYPE_LP;                         
    } /* end of analyze_input() */                                          

/*
** Rewind in_handle, even if this means copying it to a 
** temporary file.  This is only used when the first buffer
** of data is still in memory.
**
** The variable "skip" specifies the number of bytes at the 
** start of the buffer which should be skipt.  If "skip" is 
** non-zero, then we _must_ copy it to a file since the
** filters are allowed to require that the file be fully seekable.
*/
static void stubborn_rewind(void)
    {
    off_t skip;
    struct stat statbuf;

    /* Compute the size of header to skip. */
    skip = in_ptr - in_buffer;

    /* Get information about the file. */
    if( fstat(in_handle, &statbuf) == -1 )
    	fatal(PPREXIT_OTHERERR, "ppr_infile.c: stubborn_rewind(): fstat() failed, errno=%d (%s)", errno, strerror(errno) );

    /*
    ** If input is a file and we don't have to skip a header,
    ** then we can simply rewind it.
    */
    if( input_is_file && skip == 0 )
    /* if( statbuf.st_mode & S_IFREG && skip==0 ) */
        {
        lseek(in_handle, 0, SEEK_SET);
        qentry.attr.input_bytes = statbuf.st_size;
        }

    /*
    ** Either it was not a file or we must skip a header.
    ** We must copy the input data to a temporary file.
    */
    else                                      
        {
        char fname[MAX_PATH];                   /* Name of temporary file. */
        int t_handle;				/* Temporary file handle. */
	int retval;

	/*
	** Reset the size of the unfiltered input file skip.
 	** We set it to skip instead of 0 because the code below
 	** adds the length of the part it copies which does not
 	** include the header.
	*/
	qentry.attr.input_bytes = skip;

	/* Build a temporary file name and open the file. */
        sprintf(fname, TEMPDIR"/ppr%ld", (long)getpid() );
        if( (t_handle=open(fname, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR)) == -1 )
            fatal(PPREXIT_OTHERERR, "ppr_infile.c: stubborn_rewind(): can't open temporary file \"%s\", errno=%d (%s)", fname, errno, strerror(errno) );

	/* On the off chance that we have emptied the buffer. */
	if(in_left == 0)
	    load_in_buffer();

	/* Copy the input to the temporary file. */
        while( in_left > 0 )
            {
	    qentry.attr.input_bytes += in_left;
            if( (retval=write(t_handle, in_ptr, in_left)) == -1 )
            	fatal(PPREXIT_OTHERERR, "ppr_infile.c: stubborn_rewind(): write() failed, errno=%d (%s)", errno, strerror(errno) );
	    else if(retval != in_left)
	    	fatal(PPREXIT_OTHERERR, "ppr_infile.c: stubborn_rewind(): disk full while writing temporary file");
            load_in_buffer();
            }

	/* Close the temporary file */
        close(t_handle);

        /* close file we just read from */
        close(in_handle);

	/* re-open the temporary file */
        if( (in_handle=open(fname, O_RDONLY)) == -1 )
            fatal(PPREXIT_OTHERERR, "can't re-open temporary file \"%s\", errno=%d (%s)", fname, errno, strerror(errno) );

	/* Delete the temporary file's name so it will
	   magically disapear when it is closed. */
        unlink(fname);
        }
    } /* end of stubborn_rewind() */

/*
** Attach the current in_handle to the stdin of the filter 
** and attach the stdout of the filter to in_handle.
**
** The arguments to this function are the name of the executable file, 
** the name to be passed in argv[0], and the name to be passed as the
** job title.  This routine gets the other filter arguments for itself.
*/
static void exec_filter(const char *filter_path, const char *filter_name, const char *title)
    {
    int pipefds[2];				/* pipe from filter */
    pid_t pid;					/* process id of filter */

    /* All these are for parsing filter options. */
    #ifdef GNUC_HAPPY
    char *clean_options=(char*)NULL;
    #else
    char *clean_options;
    #endif
    char *si, *di;
    int stage = 1;

    /*
    ** If we do not find the special title which indicates that the
    ** filter we are running is Uncompress or Gzip, do this filter
    ** option processing:
    **
    ** Merge the default filter options with those the user has
    ** provided.  Remove filter options which don't apply to this
    ** filter.  For those which contain this filter's prefix, chop
    ** off the prefix.  Convert keywords to lower case but not
    ** their arguments.
    */ 
    if( title[0] != '-' )
      {
      int item_len, key_len, prefix_len, filter_name_len;

      /* We will use this length in future comparisons: */
      filter_name_len=strlen(&filter_name[7]);

      /* Get the default options from the printer or group config file: */
      si = extract_deffiltopts();

      /*
      ** Allocate enough room for the default options and the -o ones.
      ** There may not be any -o options.  Often, we will not need as 
      ** much room as we allocate.
      */
      if(filter_options != (char*)NULL)
	di = clean_options = (char*)myalloc( strlen(si) + 1 + strlen(filter_options) + 1, sizeof(char) );
      else
      	di = clean_options = (char*)myalloc( strlen(si) + 1, sizeof(char) );

      while(TRUE)
	{
	if( *si==(char)NULL && stage==1 ) /* If end of defaults, */
	    {				/* start -o switch options */
	    stage = 2;
	    if(filter_options == (char*)NULL)	/* if no -o switches */
		break;
	    si = filter_options;
	    if(di != clean_options)	/* If there were any default options */
		*(di++)=' ';		/* leaving one space after them. */
	    }

	if( *si == (char)NULL )		/* If entirely out of options, */
	    break;			/* stop. */

	item_len = strcspn(si," \t");	/* Length of next non-space segment */
	key_len = strcspn(si,"=");	/* length of keyword */
	prefix_len = strcspn(si,"-");	/* Find distance to next hyphen */
	if( prefix_len < key_len )	/* If this item contains a prefix, */
	    {
    	    if( prefix_len==filter_name_len 
    	    		&& icmpn(si,&filter_name[7],filter_name_len) == 0)
    	    	{			/* If the correct filter prefix, */
		si += filter_name_len;	/* skip the prefix, */
		si += 1;		/* skip the hyphen. */
		}    	    	
	    else			/* If not for us, */
	    	{			/* eat it up. */
		while( *si && ! isspace(*si) )
		    si++;
		if(isspace(*si))	/* eat up space which follows */
		    si++;	    	
		continue;
	    	}
	    }

	while( *si && *si != '=' )	/* Copy the keyword, */
	    *(di++)=tolower(*(si++));	/* converting it to lower case. */
	    
	while( *si && ! isspace(*si) )	/* Then, copy the equals sign and */
	    *(di++)=(*si++);		/* the value. */

	if( isspace(*si) )		/* Finally, copy the space */
	    *(di++)=*(si++);		/* which follows. */

	while( isspace(*si) )		/* Eat extra spaces. */
	    si++;
	}
      *di=(char)NULL;			/* Terminate the resulting string. */

      /*
      ** The line is for testing.  It is normally commented out.
      */
      /* printf("clean_options: \"%s\"\n", clean_options); */
      } /* end of code that is not for uncompress/gunzip */
    
    /* Possibly gab about what we are doing. */
    if( gab_mask & GAB_INFILE_FILTER )
    	{
	if( strcmp(title,"-c")==0)
	    {
	    printf("Filter: %s -c\n",filter_name);
	    }
	else
	    {
            printf("Filter: %s \"%s\" \"%s\" \"%s\" \"%s\"\n",
            	filter_name, clean_options, qentry.destname, title, starting_directory);
    	    }
    	}

    /*
    ** Rewind file or, if we can't, copy it to a temporary file
    ** and open that instead.
    */
    stubborn_rewind();

    /*
    ** Open a pipe which will be used to convey the filter's
    ** output back to this process.
    */
    if( pipe(pipefds) )
        fatal(PPREXIT_OTHERERR, "ppr_infile.c: exec_filter(): can't make pipe, errno=%d (%s)", errno, strerror(errno) );

    if( (pid=fork()) == -1 )
        fatal(PPREXIT_OTHERERR, "ppr_infile.c: exec_filter(): fork() failed, errno=%d (%s)", errno, strerror(errno) );

    if(pid)				/* parent */
        {
        close(pipefds[1]);		/* we won't use write end */
        close(in_handle);		/* we won't read input file directly */
        in_handle = pipefds[0];		/* henceforth we will read from pipe */
        }
    else                            /* child */
        {
        close(pipefds[0]);          /* don't need read end */
        dup2(in_handle,0);          /* attach input file to stdin */
        close(in_handle);           /* we may now close input file handle */
        dup2(pipefds[1],1);         /* copy write end of pipe to stdout */
        close(pipefds[1]);          /* we don't need origional handle */
                                    /* stderr goes to parent's */
	/* 
	** If the title is a switch, assume we are executing
	** "uncompress -c" or "gunzip -c", otherwise, execute
	** a standard filter. 
	*/
	if( strcmp(title,"-c")==0 )	/* uncompress or gunzip */
            execl(filter_path, filter_name, title,(char*)NULL);
	else				/* standard filter */
            execl(filter_path, filter_name, clean_options,qentry.destname,title,starting_directory,(char*)NULL);

        _exit(242);			/* give reapchild() a hint */
        }

    in_eof = FALSE;			/* Reset buffering code. */
    logical_eof = FALSE;		/* <-- paranoid? */
    bintoken_grace = 0;
    qentry.attr.postscript_bytes = 0;	/* We were wrong about them being PostScript bytes. */
    load_in_buffer();			/* We must do load buffer again. */

    input_is_file = FALSE;		/* Even if it was, it is no longer. */
    } /* end of exec_filter() */

/* 
** This routine is used by the function below when the required 
** filter is not available.  If the "-e hexdump" switch was not used
** try the responder.  If the "-e hexdump" switch WAS used or the
** responder fails, run the file thru the hexdump filter, put a message
** in the job's log file, and try to force a banner page.
*/
static void no_filter(char *file_type_str)
    {
    char lfname[MAX_PATH];	/* log file name */
    FILE *lfile;		/* log file */

    /* If the -e hexdump switch has not been used, try to use responder first. */
    if(!nofilter_hexdump && respond(RESP_NOFILTER,file_type_str)==0)
	{			/* If respond works */
	exit(PPREXIT_NOFILTER);	/* we can exit. */
	}
    else			/* If no responder */
    	{
	/* Select the hex dump filter. */
    	exec_filter("lib/filter_hexdump","filter_hexdump",file_type_str);

	/* Try to force banner page on. */
    	qentry.do_banner=BANNER_YESPLEASE;

	/* Try to add a message to the log file. */
        sprintf(lfname,"%s/%s-%d.0-log",DATADIR,qentry.destname,qentry.id);
        if( (lfile=fopen(lfname,"a")) != (FILE*)NULL )
	    {
	    fprintf(lfile, "Can't print %s.\n", file_type_str);
	    fclose(lfile);
	    }	
	}
    } /* end of no_filter() */

/*
** If the block in the buffer is the first block of a
** compressed or gziped file, execute a filter to uncompress
** it and return TRUE.  This is called from the function
** get_input_file(), below.
*/
static int compressed(void)
    {
    if(in_left >= 3)
    	{
	/* Check for files compressed with gzip. */
    	if( in_ptr[0]==(unsigned char)'\37' && in_ptr[1]==(unsigned char)'\213' )
    	    {
    	    #ifdef GUNZIP
    	    exec_filter(GUNZIP,"gunzip","-c");
    	    #else
    	    no_filter("gzipped files");
    	    #endif
	    return TRUE;
    	    }
	/* Check for files compressed with Unix compress. */
	if( in_ptr[0]==(unsigned char)0x1f && in_ptr[1]==(unsigned char)0x9d
			&& in_ptr[2]==(unsigned char)0x90 )
	    {
	    #ifdef GUNZIP
	    exec_filter(GUNZIP,"gunzip","-c");
	    #else
	    #ifdef UNCOMPRESS	
	    exec_filter(UNCOMPRESS,"uncompress","-c");
	    #else
	    no_filter("compressed files");
	    #endif
	    #endif
	    return TRUE;
	    }
    	}
    
    return FALSE;
    } /* end of compressed() */

/*
** This is the code to implement the hack turned 
** on by the "-H keepinfile" switch.
*/
static void keepinfile(void)
    {
    char fname[MAX_PATH];
    int out_handle;
    int bytes_read;
    int bytes_written;

    /*
    ** Unless someone changes main(), this if will always be
    ** true.  We must assign the id now because we will be
    ** using it in the sprintf() below.  The code in main()
    ** will see that we have assingned the id and will
    ** not do it again.
    */
    if(qentry.id == 0)
    	qentry.id = get_next_id(NEXTIDFILE);

    /*
    ** By setting this flag, we will indicate that if we
    ** detect a fatal error we must unlink the jobs files
    ** before exiting.
    */
    files_created = TRUE;

    /*
    ** The input file will be copied into a file in the jobs directory.
    */
    sprintf(fname, DATADIR"/%s:%s-%d.%d(%s)-infile" ,qentry.destnode, qentry.destname,qentry.id, qentry.subid, qentry.homenode);

    if( (out_handle=open(fname, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR)) == -1 )
    	fatal(PPREXIT_OTHERERR, "ppr_infile.c: keepinfile(): can't create \"%s\", errno=%d (%s)", fname, errno, strerror(errno));

    while( (bytes_read=read(in_handle, in_buffer, in_bsize)) > 0 )
    	{
    	if( (bytes_written=write(out_handle, in_buffer, bytes_read)) == -1 )
    	    fatal(PPREXIT_OTHERERR, "ppr_infile.c: keepinfile(): write() failed, errno=%d (%s)", errno, strerror(errno));
    	
	if( bytes_written != bytes_read )
	    fatal(PPREXIT_OTHERERR, "ppr_infile.c: keepinfile(): disk full");
    	}
    
    if( bytes_read == -1 )
        fatal(PPREXIT_OTHERERR, "input file read error, errno=%d (%s)", errno, strerror(errno));

    close(out_handle);
    
    close(in_handle);
    
    if( (in_handle=open(fname, O_RDONLY)) == -1 )
    	fatal(PPREXIT_OTHERERR, "ppr_infile.c: keepinfile(): can't re-open \"%s\", errno=%d (%s)", fname, errno, strerror(errno));

    input_is_file = TRUE;		/* now it is! */
    } /* end of keepinfile() */

/*
** This is the principal routine which is called from outside this module.
**
** Open the input file.  If it is not PostScript, attach a filter to it 
** and set in_handle to a pipe from that filter.
**
** Return non-zero if the input file is of zero length.
*/
int get_input_file(char *name)
    {
    const char *filter_title;	/* Title string for filter */
    int x;

    in_eof = FALSE;		/* certainly not end of file yet! */

    in_bsize = IN_BSIZE;	/* size of input buffer */
    in_buffer_rock_bottom = (unsigned char*)myalloc(sizeof(unsigned char), in_bsize+10 );
    in_buffer = in_buffer_rock_bottom + 10;
                                                                     
    /* If no file name, use stdin. */
    if( name == (char*)NULL || strcmp(name, "-") == 0 )
        {
        in_handle = dup(0);	/* in_handle = 0 can cause confusion */

	input_is_file = FALSE;

	/* If we could tell that stdin was a file,
	   we would set input_is_file = TRUE. */
        }

    /* Specific file. */
    else
        {
	if(seteuid(uid))		/* open only files the user may open */
	    fatal(PPREXIT_OTHERERR, "get_input_file(): can't seteuid(%ld)", (long)uid);

	if(setegid(gid))
	    fatal(PPREXIT_OTHERERR, "get_input_file(): can't setegid(%ld)", (long)gid);

        if( (in_handle=open(name, O_RDONLY)) == -1 )
	    fatal(PPREXIT_OTHERERR, "can't open input file \"%s\", %s", name, strerror(errno));

        if(seteuid(setuid_uid))          /* become the ppr owner again */
            fatal(PPREXIT_OTHERERR, "get_input_file(): can't seteuid(%ld)", (long)setuid_uid);

	if(setegid(setgid_gid))
	    fatal(PPREXIT_OTHERERR, "get_input_file(): can't setegid(%ld)", (long)setgid_gid);

        input_is_file = TRUE;		/* !!! kludge! */
        }

    chdir(HOMEDIR);			/* I hope this is now ok. */

    /* Invoke the keepinfile hack if it was requested.
       Notice that we do this _before_ calling load_in_buffer(). */
    if( qentry.opts.hacks & (HACK_KEEPINFILE | HACK_TRANSPARENT) )
    	keepinfile();

    load_in_buffer();		/* load the first part into memory */

    again:
    if( in_left == 0 )		/* If input file is of zero length, there is not */
    	return -1;		/* much to do, just tell main() so it can abort. */

    /* If file is compressed, uncompress it and try again. */
    if( compressed() )
    	goto again;

    /*
    ** Read any PPR user identification header.
    ** These headers are used when submitting non-PostScript
    ** files for printing.  The information in such a header
    ** is similiar to that in PostScript header comments.
    ** Such a header is generated by the PRNAUTH program.
    ** The whole thing is a hack.
    */
    read_header();

    /*
    ** Look for and process PJL.  This routine will set in_type if the 
    ** PJL header contains a "enter language =" command.
    */
    check_for_PJL();

    /*
    ** Check for TBCP.  Since TBCP is supposed to _only_
    ** work when a PostScript interpreter is active, we 
    ** take the presence of TBCP as an indication that
    ** the file is a PostScript file.
    */
    {
    unsigned char *p = in_ptr; 
    int temp_left = in_left;   

    while( temp_left && ( *p == '\n' || *p == '\n' ) )
	{
    	p++;
    	temp_left--;
    	}

    if( temp_left >= 2 && p[0] == 1 && p[1] == 'M' )
    	{
	in_type = IN_TYPE_TBCP_POSTSCRIPT;    	
	in_ptr = p;
	in_left = temp_left;
    	}
    }

    /*
    ** If the input file type is still unknown, examine 1st buffer
    ** to learn file type.
    */
    if( in_type == -1 )
        in_type = analyze_input();
                              
    /* Possibly gab about what we have decided. */
    if( gab_mask & GAB_INFILE_AUTOTYPE )
	printf("in_type = %d\n",in_type);

    /* Settle on a title string for the filter. */
    if(qentry.Title != (char*)NULL)	/* Try Title from user */
    	filter_title = qentry.Title;
    else if(name != (char*)NULL)	/* Try file name */
    	filter_title = name;
    else				/* use nothing */
    	filter_title = "";

    /* Run a filter if necessary. */
    switch(in_type)
        {
        case IN_TYPE_POSTSCRIPT:        /* PostScript is already ok */
        case IN_TYPE_PJL_POSTSCRIPT:	/* PostScript with stript PJL header */
            break;
	case IN_TYPE_TBCP_POSTSCRIPT:
	    exec_filter("lib/tbcp2bin","tbcp2bin","");
	    break;
        case IN_TYPE_D_POSTSCRIPT:	/* Control-D is messy PostScript */
            warning(WARNING_PEEVE, "spurious leading ^D");
            in_getc();                  /* eat control-D */
            break;
	case IN_TYPE_BAD_POSTSCRIPT:
	    no_filter("fragmentary PostScript files");
	    break;	
	case IN_TYPE_UNKNOWN:
	    no_filter("files of unknown format");
	    break;
	case IN_TYPE_BINARY:
	    no_filter("binary files");
	    break;
        default:			/* Run a filter from the filter table. */
	    for(x=0; x < filters_count; x++)
	    	{
		if( in_type==filters[x].in_type )
		    {
		    char fpath[MAX_PATH];		    
		    sprintf(fpath, "lib/filter_%s", filters[x].name);
		    
		    /* 
		    ** Some filters may not be present.  Those filters
		    ** have an `excuse' string which explains that the filter
		    ** is not available.  If this filter has such a string,
		    ** check if it is present and offer the excuse if it is
		    ** not.  If there is no excuse or the filter is present,
		    ** just execute it.
		    */
	    	    if( filters[x].excuse != (char*)NULL && access(fpath,X_OK)==-1)
			{
			no_filter(filters[x].excuse);			
			}
		    else
	    		{
	    		exec_filter(fpath, fpath+4, filter_title);
	    		}
		    break;
		    }	    	
	    	}

	    /* If we ran off the end of the filters list, */
            if( x == filters_count )
            	fatal(PPREXIT_OTHERERR,"get_input_file(): invalid in_type: %d",in_type);
	    break;
        }

    return 0;
    } /* end of get_input_file() */

/*
** Called from main(), this function forces to input file type to
** a certain type, disabling auto detect.
**
** Return -1 if the type is unrecognized.
*/
int force_in_type(const char *type)
    {
    int x;
    
    /* Search the filters table for the type string specified.
       If we find it, set the in_type to that type and return. */
    for(x=0; x < filters_count; x++)
    	{
    	if( filters[x].name != (char*)NULL && icmp(type,filters[x].name) == 0 )
    	    {
    	    in_type=filters[x].in_type;
    	    return 0;
    	    }
    	}

    /* failure */
    in_type = IN_TYPE_UNKNOWN;
    return -1;
    }

/*
** Print the part of the text for ppr -? which relates to filters.
*/
void minus_tee_help(FILE *outfile)
    {
    int x;
    
    /* For each of the filters, construct the -T help line
       containing the name of the filter and the brief explaination. */
    for(x=0; x < filters_count; x++)
    	{
	fprintf(outfile,"\t-T %-23s %s\n", filters[x].name, filters[x].help);    	
    	}
    
    } /* end of minus_tee_help() */

/* end of file */
