/*
** ~ppr/src/ppr/ppr_dscdoc.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 12 March 1997.
*/

/*
** This module contains Adobe Document Structuring Convention (DSC)
** comment interpretation routines.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "ppr.h"

/*
** Flags which indicate that we may look for trailer comments for 
** the indicated items.
*/
static int atend_pages = FALSE;
static int atend_pageorder = FALSE;

/*
** Flags which indicate that certain comments have already been found
** in the header section.
*/
int found_For = FALSE;
int found_Routing = FALSE;
static int found_Creator = FALSE;
int found_Title = FALSE;
int found_ProofMode = FALSE;

/* globals in ppr_main.c */
extern char *AuthCode;
extern struct Media guess_media;

/*
** Routine to copy an unrecognized comment in line[] to the output file.
** If the comment line is an expanded continuation line, compress it again.
*/
void copy_comment(FILE *out)
    {
    if(cont)                                /* if was continuation */
        {                                   /* comment */
        char *ptr;                          /* make it one again */

        ptr=line+strcspn(line," \t");       /* skip `til space */
        ptr+=strspn(ptr," \t");             /* skip space */
         
        fprintf(out,"%%+ %s\n",ptr);
        }
    else                                    /* if not continuation comment, */
        {                                   /* just copy it to the */
        fprintf(out,"%s\n",line);           /* output file */
	}        
    } /* end of copy_comment() */

/*
** Handle header and trailer comments.
**
** Return non zero if line should NOT be copied to the
** comments file.
**
** We will call old_to_new_headertrailer() to convert pre DSC 3.0
** comments to DSC 3.0.
**
** tokenize() should be called before this routine is called.
**
** The argument "trailer" should be true when this routine is 
** called to handle the document trailer.
*/
int header_trailer(int trailer)
    {
    switch(line[2])
        {
        case 'P':
            if(strncmp(line,"%%Pages:",8)==0)
                {
		if(tokens[1]==(char*)NULL)
		    {
		    warning(WARNING_SEVERE,"Malformed \"%%%%Pages\" comment");
		    return -1;
		    }
                if(strcmp(tokens[1],"atend")==0)    /* if being defered, */
                    {                               /* then make a note */
                    atend_pages=TRUE;               /* of that fact */
                    return -1;
                    }
                if(trailer && !atend_pages)
                    {
                    warning(WARNING_SEVERE,"\"%%%%Pages:\" in trailer though not defered");
                    return -1;
                    }
                if(trailer || qentry.attr.pages==-1)    /* if in trailer */ 
                    sscanf(line,"%%%%Pages: %d %d",     /* or 1st Pages: */
                                    &qentry.attr.pages, /* comment */
                                    &qentry.attr.pageorder);
                return -1;      /* use only first */
                }
            if( strncmp(line,"%%PageOrder:",12)==0 )
                { 
                if(trailer && !atend_pageorder)
                    warning(WARNING_SEVERE, "\"%%%%PageOrder:\" in trailer but not defered in header");
                    
		if(tokens[1]==(char*)NULL)
		    {
		    warning(WARNING_SEVERE, "Malformed \"%%%%PageOrder:\" line");
		    return -1;
		    }

                if(strcmp(tokens[1],"Ascend")==0)
                    qentry.attr.pageorder=PAGEORDER_ASCEND;
                else if(strcmp(tokens[1],"Descend")==0)
                    qentry.attr.pageorder=PAGEORDER_DESCEND;
                else if(strcmp(tokens[1],"Special")==0)
                    qentry.attr.pageorder=PAGEORDER_SPECIAL;
                else if(strcmp(tokens[1],"atend")==0)
                    atend_pageorder=TRUE;
                else
                    warning(WARNING_SEVERE, "invalid \"%%%%PageOrder:\" comment");
                return -1;
                }
            if( strncmp(line,"%%Page",6)==0)    /* detect things like */
                {                               /* %%PageBoundingBox: */
                if(trailer)                     
                    warning(WARNING_PEEVE, "suspected page level comment in trailer section");
                else
                    warning(WARNING_PEEVE, "suspected page level comment in header comments section");
                return 0;                       /* but leave it in */
                }
	    if( strncmp(line,"%%ProofMode:",12)==0 )
	    	{
		if(trailer)
		    warning(WARNING_SEVERE,"\"%%%%ProofMode:\" not allowed in trailer");
		    
		if(found_ProofMode==FALSE)				/* If ProofMode not already set */
		    {
		    if(tokens[1]==(char*)NULL)				/* parameter missing, */
		    	qentry.attr.proofmode=PROOFMODE_SUBSTITUTE;	/* (RBIIp 775) */
		    else if(strcmp(tokens[1],"TrustMe")==0)
		        qentry.attr.proofmode=PROOFMODE_TRUSTME;
		    else if(strcmp(tokens[1],"Substitute")==0)
		        qentry.attr.proofmode=PROOFMODE_SUBSTITUTE;
		    else if(strcmp(tokens[1],"NotifyMe")==0)
		        qentry.attr.proofmode=PROOFMODE_NOTIFYME;
		    else
		        warning(WARNING_SEVERE,"\"%%%%ProofMode:\" had invalid argument \"%s\"",tokens[1]);		    

		    found_ProofMode=TRUE;
		    }		      
		return -1;
		}
            break;

        case 'F':                             
            if(strncmp(line, "%%For:", 6) == 0)
                {
                if(trailer)
                    {
                    warning(WARNING_SEVERE, "\"%%%%For:\" not allowed in trailer");
                    return -1;
                    }
                if(read_for && !found_For)	/* use only 1st */
                    {
		    int x;
                    char *ptr = &line[6+strspn(&line[6], " \t")];

                    if(*ptr == '(')
                        ptr = mystrdup(tokens[1]);
                    else		/* ^ NULL ptr safe */
                        ptr = mystrdup(ptr);

		    /*
		    ** Turn unprintable characters into periods.  This
		    ** code is here because some people somehow manage
		    ** to get deletes into their Macintosh computer names!
		    */
		    for(x=0; ptr[x]; x++)
			{
		    	if(!isprint(ptr[x]))
		    	    ptr[x] = '.';
		    	}
		    	    
		    qentry.ForLine = ptr;

                    found_For = TRUE;		/* A %%For has been found, */
                    }				/* accept no more. */
                return -1;			/* Don't keep the line. */
                }
            break;
        case 'L':
            if(strncmp(line,"%%LanguageLevel:",16)==0)
                {
                if(trailer)
                    {
                    warning(WARNING_SEVERE, "\"%%%%Languagelevel:\" not allowed in trailer");
                    return -1;      /* delete it */
                    }
                qentry.attr.langlevel = atoi(&line[16]);
                fprintf(comments, "%s\n", line);	/* this is quicker than */
                return -1;				/* returning 0 */ 
                }
            break;
        case 'T':
            if( strncmp(line,"%%Title:",8) == 0 )
                {
                if(trailer)
                    {
                    warning(WARNING_SEVERE, "\"%%%%Title:\" not allowed in trailer");
                    return -1;
                    }
                if( ! found_Title )
                    {
                    char *ptr = &line[8+strspn(&line[8], " \t")];
                    if(*ptr=='(')
                        qentry.Title = mystrdup(tokens[1]);
                    else		/* ^ NULL ptr safe */
                        qentry.Title = mystrdup(ptr);
                    found_Title = TRUE;
                    }
                return -1; 
                }
            break;
        case 'C':
            if( strncmp(line,"%%Creator:", 10) == 0 )
                {                   
                if(trailer)
                    {
                    warning(WARNING_SEVERE, "%%%%Creator: not allowed in trailer");
                    return -1;
                    }
                if(found_Creator==FALSE)
                    {
                    char *ptr=&line[10+strspn(&line[10]," \t")];
                    if(*ptr=='(')           /* if PostScript string, */
                        qentry.Creator=mystrdup(tokens[1]); /* use 1st token */
                    else                    /* otherwise, */
                        qentry.Creator=mystrdup(ptr); /* use rest of line */
                    found_Creator=TRUE;
                    }
                return -1;
                }
            if( strncmp(line, "%TCHCTAuthCode:", 15) == 0 )
                {
		/* If this is the first AuthCode line, then save the code. */
                if(AuthCode==(char*)NULL && tokens[1]!=(char*)NULL)
                    AuthCode = mystrdup(tokens[1]);
                return -1;                          /* swallow line */
                }
            break;
        case 'E':
            if(strncmp(line,"%%Extensions:",13)==0)
                {
                int x;

                if(trailer)
                    {
                    warning(WARNING_SEVERE,
                        "\"%%%%Extensions:\" not allowed in trailer");
                    return -1;      /* delete it */ 
                    }

                for(x=1;tokens[x]!=(char*)NULL;x++)
                    {
                    if(strcmp(tokens[x],"DPS")==0)
                        qentry.attr.extensions|=EXTENSION_DPS;
                    else if(strcmp(tokens[x],"CMYK")==0)
                        qentry.attr.extensions|=EXTENSION_CMYK;
                    else if(strcmp(tokens[x],"Composite")==0)
                        qentry.attr.extensions|=EXTENSION_Composite;
                    else if(strcmp(tokens[x],"FileSystem")==0)
                        qentry.attr.extensions|=EXTENSION_FileSystem;
                    else
                        warning(WARNING_SEVERE,
                            "Language extension \"%s\" is unrecognized.",tokens[x]);
                    }
                fprintf(comments,"%s\n",line);  /* this is faster than */
                return -1;                      /* returning zero */
                }
            break;
        case 'R':
            if(strncmp(line,"%%Requirements:",15)==0)
                {
                int x;

                if(trailer)
                    {
                    warning(WARNING_SEVERE,
                        "%%%%Requirments: not allowed in trailer");           
                    return -1;
                    }
                for(x=1;tokens[x]!=(char*)NULL;x++)
                    requirement(REQ_DOC,tokens[x]);
                return -1;
                }            
            if(strncmp(line,"%%Routing:",10)==0)
                {
                if(trailer)
                    {
                    warning(WARNING_SEVERE,
                        "%%%%Routing: not allowed in trailer");
                    return -1;
                    }
                if(found_Routing==FALSE)            /* use only 1st */
                    {
                    char *ptr=&line[10+strspn(&line[10]," \t")];
                    if(*ptr=='(')                   /* if PostScript string, */
                        qentry.Routing=mystrdup(tokens[1]); /* use 1st token */
                    else                            /* otherwise, */
                        qentry.Routing=mystrdup(ptr); /* use rest of line */
                    found_Routing=TRUE;
                    }
                return -1;
                }
            break;

        case 'D':
            if(strncmp(line,"%%DocumentMedia:",16)==0)
                {
                if(trailer)
                    {
                    warning(WARNING_SEVERE,
                        "%%%%DocumentMedia: not allowed in trailer");
                    return -1;
                    }
                if( (tokens[1]!=(char*)NULL) && (strcmp(tokens[1],"atend")==0) )
                    {
                    warning(WARNING_SEVERE,"\"%%%%DocumentMedia: (atend)\" not allowed");
                    return -1;
                    }
                media(MREF_DOC,1);
                return -1;
                }
            if(strncmp(line,"%%DocumentFonts:",16)==0)
                {
                int x;

		/* If it is an atend comment, just ignore it. */
                if( (tokens[1]!=(char*)NULL) && (strcmp(tokens[1],"atend")==0) )
                    return -1;

		/* Feed each of the font names to the resource handler, */
		/* telling the resource handler that it is unclear      */
		/* whether the font is required or supplied.            */
                for(x=1;tokens[x]!=(char*)NULL;
                    x+=resource(REREF_UNCLEAR,"font",x));

                return -1;	/* <-- pprdrv will reconstruct font comments */
                }
            if(strncmp(line,"%%DocumentProcSets:",19)==0)
                {
                int x;

                if( (tokens[1]!=(char*)NULL) && (strcmp(tokens[1],"atend")==0) )
                    return -1;

                for(x=1;tokens[x]!=(char*)NULL;
                    x+=resource(REREF_UNCLEAR,"procset",x));

                return -1;
                }
	    break;
	    
	case 'O':
	    if(strncmp(line,"%%Orientation:",14)==0)
	    	{
		if(tokens[1] == (char*)NULL)
		    {
		    warning(WARNING_SEVERE,"\"%%%%Orientation:\" comment with missing argument");
		    return -1;
		    }

		/* Ignore header "%%Orientation:" comments after the 1st. */
		if( ! trailer && qentry.attr.orientation != ORIENTATION_UNKNOWN )
		    return -1;

		/* Interpret arguments "Portrait" and "Landscape" but object */
		/* to anything else except "atend" in the header.            */
		if(strcmp("Portrait",tokens[1])==0)
		    qentry.attr.orientation = ORIENTATION_PORTRAIT;
		else if(strcmp("Landscape",tokens[1])==0)
		    qentry.attr.orientation = ORIENTATION_LANDSCAPE;
		else if( trailer || strcmp(tokens[1],"atend") )
                    warning(WARNING_SEVERE,"Bad argument to \"%%%%Orientation:\" comment");

		return -1;	/* <-- pprdrv will generate new comment */
	    	}
	    break;
	    	
        } /* end of switch */

    if( (tokens[1]!=(char*)NULL) && (strcmp(tokens[1],"atend")==0) )    /* no further use for */
        return -1;                      /* "(atend)" comments */
        
    old_to_new_headertrailer();     /* convert to DSC 3.0 */

    switch(line[2])
        {
        case 'D':
            if( strncmp(line,"%%DocumentNeededResources:",26)==0 )
                {
                int x;
                for(x=2;
                    tokens[x]!=(char*)NULL;
                        x+=resource(REREF_NEEDED,tokens[1],x));
                return -1;
                }
            if( strncmp(line,"%%DocumentSuppliedResources:",28)==0 )
                {
                int x;
                for(x=2;
                    tokens[x]!=(char*)NULL;
                        x+=resource(REREF_SUPPLIED,tokens[1],x));
                return -1;
                }
            break;
        } /* end of switch */
        
    return 0;       /* just copy it */
    }

/*
** Handle page level comments.  Return zero if we want it passed 
** thru to the -pages or -text file.                       
** If we return -1 it is understood that we made our own use of it 
** and don't want it copied to the output file.
**
** We call old_to_new_page() to convert old DSC comments to the
** new style. 
*/
int pagelevel(void)
    {
    old_to_new_page();          /* convert to DSC 3.0 */

    if(strncmp(line,"%%Page",6)==0)
        {
        switch(line[6])
            {
            case 'R':
                if(strncmp(line,"%%PageResources:",16)==0)
                    {
                    int x;
                    for(x=2; tokens[x]!=(char*)NULL;
                            x+=resource(REREF_PAGE,tokens[1],x));
                    return -1;  /* don't copy to output */
                    }
                if(strncmp(line,"%%PageRequirements:",19)==0)
                    {
                    int x;
                    for(x=1; tokens[x]!=(char*)NULL; x++)
                        requirement(REQ_PAGE,tokens[x]);
                    return -1;
                    }
                return 0;
            case 'M':
                if(strncmp(line,"%%PageMedia:",12)==0)
                    {
                    media(MREF_PAGE,1);
                    return -1;
                    }
                return 0;
            default:
                return 0;       /* just copy it into ddddd-iiiii.s-text */
            }
        }
        return 0;
    } /* end of pagelevel() */

/*
** Call this routine whenever a "%%BeginFeature:"
** or "%%IncludeFeature:" line is seen.
*/
void feature(void)
    {
    if(tokens[1]!=(char*)NULL && tokens[2]!=(char*)NULL)
    	{
	switch(tokens[1][1])
	    {
	    case 'B':
	    	/* If "Booklet", set booklet mode true. */
		if( read_signature && (strcmp(tokens[1],"*Booklet")==0) )
		    {
		    if(strcmp(tokens[2],"True")==0)
		    	qentry.N_Up.sigsheets=(-1);	/* emulate -s booklet */		    
		    }
		break;

	    case 'D':
	    	/* If it was a duplex feature and we are looking for them, */
	    	/* remember what it was. */
	    	if( read_duplex && (strcmp(tokens[1],"*Duplex")==0) )
		    {
		    if(strcmp(tokens[2],"None")==0)
		    	current_duplex=DUPLEX_SIMPLEX;
		    else if(strcmp(tokens[2],"DuplexNoTumble")==0)
		    	current_duplex=DUPLEX_DUPLEX;
		    else if(strcmp(tokens[2],"DuplexTumble")==0)
		    	current_duplex=DUPLEX_TUMBLE;
		    }
		break;

	    case 'M':
		/* Remeber various media characteristics in case */
		/* no %%Media: comment was provided. */
	    	if(strcmp(tokens[1],"*MediaColor")==0)
	    	    {
	    	    ASCIIZ_to_padded(guess_media.colour,tokens[2],MAX_COLOURNAME);
	    	    }
		else if(strcmp(tokens[1],"*MediaType")==0)
		    {
		    if(strcmp(tokens[2],"Plain")==0)	/* PPD file might use "Plain" */
			ASCIIZ_to_padded(guess_media.type,"",MAX_TYPENAME);
		    else
		    	ASCIIZ_to_padded(guess_media.type,tokens[2],MAX_TYPENAME);
		    }
		else if(strcmp(tokens[1],"*MediaWeight")==0)
		    {
		    guess_media.weight=getdouble(tokens[2]);
		    }
		break;	    	    

	    case 'P':
		/* For the media size, we will remember it for use if there is */
		/* no %%Media: comment. */
		if((strcmp(tokens[1],"*PageSize")==0) || (strcmp(tokens[1],"*PageRegion")==0))
		    {
		    int width, height, envelope;
		    if(pagesize(tokens[2],&width,&height,&envelope)==0)
			{
			#ifdef DEBUG_MEDIA
			printf("%dx%d, %s\n",width,height,envelope?"True":"False");
			#endif
			guess_media.width=width;
			guess_media.height=height;
			if(envelope)
			    ASCIIZ_to_padded(guess_media.type,"Envelope",MAX_TYPENAME);
			}		    
		    }
	    	break;

	    case 'S':
	    	/* If sigature, set the signature mode */
		if( read_signature && (strcmp(tokens[1],"*Signature")==0) )
		    {
		    if(strcmp(tokens[2],"True")==0)	/* If the option is */
		    	{				/* simply "True", */
		    	qentry.N_Up.sigsheets=8;	/* enable 8 sheet (32 page) */
		    	}				/* signatures. */
		    else				/* otherwise */
		    	{				/* who knows? */
			if(strspn(tokens[2],"0123456789")==strlen(tokens[2]))
			    {
			    qentry.N_Up.sigsheets=((atoi(tokens[2])+3) / 4);
			    }
			else
			    {

			    }
		    	}
		    }
		break;

	    default:
	    	break;
	    }
    	}
    } /* end of feature() */

/*=========================================================================
** A routine for each section of a DSC conforming document.
=========================================================================*/

/*
** Read the header comments.
** The comments will be put into the comments file and into the Thing
** structures.  We must return with a line in the line buffer.
** 
** Most of the comment work will be done by header_trailer().
*/
void read_comments(void)
    {
    outermost_start(OUTERMOST_HEADER_COMMENTS);

    clean_getline();                /* get 1st line, hopefully %!PS-... */
        
    if( strncmp(line, "%!PS-Adobe-", 11) == 0 )
        {                           /* if Adobe string, expect version # */
        sscanf(line, "%%!PS-Adobe-%f", &qentry.attr.DSClevel);

        if( qentry.attr.DSClevel <= 3.0 )		/* if below or at */
	    {
            fprintf(comments,"%%!PS-Adobe-3.0\n");	/* our version say ours */
	    }
	else						/* if higher, */
	    {
            fprintf(comments,"%s\n",line);		/* say that version */
	    warning(WARNING_SEVERE, "document DSC comment version number is %.1f", qentry.attr.DSClevel);
	    }

	clean_getline();				/* and refill buffer */
	}
    else                            /* If not flagged with */
        {                           /* "%!PS-Adobe-x.xx", */
        fprintf(comments,"%%!\n");  /* make no version claims. */
        }

    for( ; ! logical_eof; clean_getline() )           
        {
        if( line[0] != '%' )    /* If not a comment, then */
            break;              /* comments section is done. */

        if( line[1]==(char)NULL || line[1]==' ' || line[1]=='\t' )
            break;              /* %{space} terminates section */

        if( line_len > MAX_LINE )
            break;              /* non-conforming line length terminates */

        if( strcmp(line,"%%EndComments") == 0 )
            {                   /* swallow "%%EndComments" */
            clean_getline();    /* but, we must leave a line in the buffer */
            break; 
            }
                     
        if( strncmp(line,"%%Begin",7)==0 || /* MS-Windows 3.1 makes */
                strncmp(line,"%%Include",9)==0 )
            {                               /* this mistake */
            warning(WARNING_PEEVE,"header comments unterminated");
            break;  
            }

        if( header_trailer(0) )     /* see what this can make of it */
            continue;               /* if it handled line, go for next */

        if( line[1] == '!' )        /* we don't extra flag lines */
            continue;

        copy_comment(comments);     /* last resort, just pass it thru */
        } /* end of for loop */

    outermost_end(OUTERMOST_HEADER_COMMENTS);
    } /* end of read_comments() */

/*
** Read the document defaults into the -pages file.
** This is called by read_prolog() if a defaults section is detected.
*/
void read_defaults(void)
    {
    outermost_start(OUTERMOST_DOCDEFAULTS);

    /* Write the "%%BeginDefaults" line to the -pages file. */
    fprintf(page_comments,"%s\n",line);

    /* Copy the rest of the document defaults section. */
    while( ! logical_eof )
        {
        clean_getline();

        fprintf(page_comments,"%s\n",line);

        if(strncmp(line,"%%EndDefaults",13)==0)
            {
            clean_getline();      /* leave something for next guy */
            break;
            }

	/* Save this information for the benefit of ppr_split.c. */
	ppr_sscanf(line, "%%%%PageMedia: %#s", sizeof(default_pagemedia), default_pagemedia);
        }

    outermost_end(OUTERMOST_DOCDEFAULTS);
    } /* end of read_defaults() */

/*
** Read the prolog and document setup section into the -text file.
** At the proper place, digress to read the document defaults
** into the "-pages" file.  When this funtion is called, there
** is already a line in the input buffer.
**
** We will stop when we hit the begining of the
** first page, the begining of the trailer, or end of the file.
**
** If there will be no need to read any pages, then return non-zero.
**                                         
** If we find a document setup section or can create an empty one
** we set qentry.attr.docsetup to TRUE, otherwise we set it to FALSE.
**
** If the document has an %%EndSetup comment but no %%BeginSetup comment
** this code will not correct it.
*/
int read_prolog(void)
    {
    int end_setup_seen = FALSE;		/* made true when "%%EndSetup" seen */
    qentry.attr.prolog = FALSE;		/* we may set true later */
    qentry.attr.docsetup = FALSE;	/* we may set true later */

    /*
       Because we can't be sure that we are in the prolog until and
       unless we see a "%%BeginProlog", we will not call
       outermost_start(OUTERMOST_PROLOG) here.  It might be that
       we will have to read the document defaults section before
       we get to the prolog.
    */

    while( ! logical_eof )
        {
        if( (line[0]=='%') && (line[1]=='%') )	
            {			/* if a DSC comment line, */
            /*
            ** If start of 1st page, start document setup
            ** section if one does not exist.  Close document
	    ** setup section if it is not closed already.
	    */
            if( nest_level() == 0 && strncmp(line, "%%Page:", 7) == 0 )
                {
		if(!qentry.attr.prolog)		/* XV will catch it here */
		    {
		    warning(WARNING_PEEVE, "No \"%%%%EndProlog\" before first \"%%%%Page:\", inserting one");
		    fputs("%%EndProlog\n", text);
		    qentry.attr.prolog = TRUE;
		    }

                if(!qentry.attr.docsetup)	/* If no document setup, make one */
                    {
                    fputs("%%BeginSetup\n", text);
		    qentry.attr.docsetup = TRUE;
		    }

                if(!end_setup_seen)
                    fputs("%%EndSetup\n", text);

                return 0;           /* pages should be read */
                }

            /*
            ** If end of file is here or is coming soon,
            ** get out and say there is no need to read pages.
            */
            if( nest_level() == 0 && (strcmp(line,"%%Trailer") == 0 || strcmp(line,"%%EOF") == 0) )
                return 1;           /* 1 means no need to read pages */

            /*
            ** If we see a document defaults section,
            ** turn aside for a moment to copy it.
            */
            if( nest_level() == 0 && strncmp(line, "%%BeginDefaults", 15) == 0 )
                {
                read_defaults();
                continue;
                }

            /* Take note if we seen %%BeginSetup or %%EndSetup. */
	    if( nest_level() == 0 && strcmp(line, "%%BeginSetup") == 0 )
                {
                qentry.attr.docsetup = TRUE;	/* just set flag */

		if(!qentry.attr.prolog)		/* if no "%%EndProlog" seen, */
		    {
		    warning(WARNING_SEVERE, "No \"%%%%EndProlog\" before \"%%%%BeginSetup\", inserting one");
		    outermost_end(OUTERMOST_PROLOG);
		    fputs("%%EndProlog\n", text);
		    qentry.attr.prolog = TRUE;
		    }

		outermost_start(OUTERMOST_DOCSETUP);
                }
	    else if( nest_level() == 0 && strcmp(line, "%%EndSetup") == 0 )
		{
		qentry.attr.docsetup = TRUE;
		end_setup_seen = TRUE;			/* just set flag */
		outermost_end(OUTERMOST_DOCSETUP);
		}

            /* Look for "%%EndProlog", make note if we see it: */
            if( nest_level() == 0 && strcmp(line, "%%EndProlog") == 0 )
                {
		if(qentry.attr.prolog)	/* if already seen, */
		    {
		    warning(WARNING_SEVERE, "Extra \"%%%%EndProlog\"");
		    }
		else
		    {
		    qentry.attr.prolog = TRUE;
		    outermost_end(OUTERMOST_PROLOG);
		    }
                }            

	    /*
	    ** If we see an explicit "%%BeginProlog", make a note of it:
	    ** Notice that we don't set qentry.attr.prolog = TRUE.
	    */
	    if( nest_level() == 0 && strcmp(line, "%%BeginProlog") == 0 )
		outermost_start(OUTERMOST_PROLOG);

            /* Unrecognized comment, just copy it. */
            copy_comment(text);

	    /*
	    ** If we are in the document setup section, digress to read
	    ** media selection features and duplex selection features
	    ** and booklet and signature selection features.
	    **
	    ** By doing this we may be able to get a better idea of the
	    ** desired media or we may be able to emulate features which
	    ** are missing in the printer.
	    */
	    if( qentry.attr.docsetup && ((strncmp(line,"%%BeginFeature:",15)==0) || (strncmp(line,"%%IncludeFeature:",17)==0)) )
		feature();
            }                                   
        else                /* if not comment, just copy it */
            {
            fwrite(line, sizeof(unsigned char), line_len, text);
            if(! line_overflow)
            	fputc('\n', text);
            }
        clean_getline();    /* get the next line */
        } /* while( !logical_eof ) */
	strcpy(line, "%%Trailer");	/* make shure trailer comment added */
	line_len = 9;			/* <-- may not be strictly necessary */
	return -1;			/* no pages will follow, say so */
    } /* end of read_prolog() */

/*
** Start and end of page processing routines.
*/
void start_of_page_processing(void)
    {
    prepare_thing_bitmap();
    } /* end of start_of_page_processing() */

void end_of_page_processing(void)
    {
    dump_page_resources();
    dump_page_requirements();
    dump_page_media();
    } /* end of end_of_page_processing() */

/*
** Read in all the pages.
** Return when we hit %%Trailer or end of file, whichever comes first.
*/
void read_pages(void)
    {
    int pagetrailer = FALSE;
    int pageheader = TRUE;
    
    for( ; ! logical_eof ; clean_getline() )
        {
        if( line[0]=='%' && line[1]=='%' && nest_level()==0 && line_len<=MAX_LINE )
            {
            if( strncmp(line,"%%Page:",7)==0 )
                {
		outermost_start(OUTERMOST_SCRIPT);
                if( pagenumber++ > 0 )         /* if not start of 1st page, */
                    end_of_page_processing();  /* close old page */
                start_of_page_processing();    /* now, start new page */
                pageheader=TRUE;               /* we in header now */
                pagetrailer=FALSE;             /* certainly not in trailer */
                fprintf(page_comments,"%s\n",line);
                fprintf(page_comments,"Offset: %ld\n",ftell(text));
                fprintf(text,"%s\n",line);
                continue;
                }
            if( strcmp(line,"%%BeginPageSetup")==0 )
                {               /* this terminates page header */
                pageheader = FALSE;
                fprintf(text,"%s\n",line);
                continue;
                } 
            if( nest_level() == 0 && ( strcmp(line,"%%Trailer") == 0
                    || strcmp(line,"%%EOF") == 0 ) )
                {
                if( pagenumber > 0 )        
                    end_of_page_processing();
                strcpy(line,"%%Trailer");           /* change %%EOF to */
                fprintf(page_comments,"%s\n",line); /* "%%Trailer" */
                fprintf(page_comments,"Offset: %ld\n",ftell(text));
                return;
                }
            if( strcmp(line,"%%PageTrailer")==0 )
                {
                pagetrailer=-1;
                fprintf(text,"%s\n",line);
                continue;
                }     

            if( (tokens[1]!=(char*)NULL) && (strcmp(tokens[1],"atend")==0) )
                continue;       /* no use for (atend) comments */

	    /* Call page comment processor. */
            if( pagelevel()==0 )
                {
                if( pageheader || pagetrailer )
                    fprintf(page_comments, "%s\n", line);
                else
                    fprintf(text, "%s\n", line);  
                }
            }
        else                                    /* not DSC comment */
            {
            pageheader = 0;
            fwrite(line, sizeof(unsigned char), line_len, text);
            if(!line_overflow)
            	fputc('\n',text);
            }
        } /* end of for which loops `til broken or `til end of file */

        /* If we hit end of file, add the missing %%Trailer comment. */
        if( pagenumber > 0 )        
            end_of_page_processing();
        fprintf(page_comments,"%%%%Trailer\n");
        fprintf(page_comments,"Offset: %ld\n",ftell(text));
        strcpy(line,"%%Trailer");

    } /* end of read_pages() */

/*
** Read the document trailer.
** We will write the text to -text and process the comments,
** writing those we don't recognize to -comments.
** Those we do recognize are handled by header_trailer().
** Return when the %%EOF line is encountered.
*/
void read_trailer(void)
    {
    if(strcmp(line,"%%EOF")==0)         /* If no trailer, */
        return;                         /* exit now. */

    outermost_start(OUTERMOST_TRAILER);

    fprintf(text,"%s\n",line);          /* print the line "%%Trailer" */

    while( ! logical_eof )
        {
        clean_getline();

        if( line[0]=='%' && line[1]=='%' && line_len<=MAX_LINE )            
            {                           /* if DSC comment */
            if( nest_level() == 0 && strcmp(line, "%%EOF") == 0 )
		break;                 /* stop on unnested %%EOF */

            if( header_trailer(0) )     /* call trailer comment handler */   
                continue;               /* if it returns true, it did it */

            copy_comment(comments);     /* if unrecognized, save it */
            }
        else                            /* not DSC comment */
            {                           /* just print the line */
            fwrite(line, sizeof(unsigned char), line_len, text);
            if(!line_overflow)
            	fputc('\n',text);
            }
        }

    outermost_end(OUTERMOST_TRAILER);
    } /* end of read_trailer() */

/* end of file */
