/*
** ~ppr/src/ppop/ppop_cmds_other.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This file was last modified 21 January 1997.
*/

/*
** PPOP queue listing commands.
**
** Routines which implement other user commands.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>

#include "pprd.h"
#include "ppop.h"
#include "util_exits.h"

/*===================================================================
** Routine to display the number of jobs canceled.
===================================================================*/
void say_canceled(int count)
    {
    switch(count)
	{
	case 0:
	    printf("There were no jobs to cancel.\n");
	    break;
	case 1:
	    printf("1 job was canceled.\n");
	    break;
	default:
	    printf("%d jobs were canceled.\n",count);
	}
    } /* end of say_canceled() */
    
/*===================================================================
** ppop status {printer}
**
** Display the status of printers.  We will use the "s" command.
===================================================================*/
int ppop_status(char *argv[])
    {
    struct Destname destname;
    int len;
    char line[1024];
    FILE *FIFO, *reply_file;
    char *printer_name;
    char *printer_nodename;
    char *job_destname;
    int job_id,job_subid;
    char *job_homenode;
    int status;
    int next_retry;
    int countdown;
    char message[MAX_STATUS_MESSAGE+2];

    if(argv[0] == (char*)NULL)
        {
        fprintf(errors, "Usage: ppop status {<printer>, <group>, all}\n");
        return EXIT_SYNTAX;
        }

    if( parse_dest_name(&destname, argv[0]) )
    	return EXIT_SYNTAX;

    FIFO = get_ready(destname.destnode);

    fprintf(FIFO, "s %ld %s %s\n", (long)pid, destname.destnode, destname.destname);
    fflush(FIFO);

    if( ! machine_readable )
	{
	puts("Printer          Status");
	puts("------------------------------------------------------------");
	}
	
    if( (reply_file = wait_for_pprd(TRUE)) == (FILE*)NULL )
    	return print_reply();

    while( fgets(line, sizeof(line), reply_file) != (char*)NULL )
	{
	fgets(message, sizeof(message), reply_file);
	message[strcspn(message, "\n")] = (char)NULL;

	printer_nodename = (char*)NULL;
	printer_name = (char*)NULL;
	job_destname = (char*)NULL;
	job_homenode = (char*)NULL;

	if(ppr_sscanf(line,"%S %S %d %d %d %S %d %d %S",
		&printer_nodename, &printer_name, &status,
		&next_retry, &countdown, &job_destname, &job_id, &job_subid,
		&job_homenode) != 9 )
	    {
	    printf("Malformed response: %s",line);
	    if(printer_nodename != (char*)NULL) myfree(printer_nodename);
	    if(printer_name != (char*)NULL) myfree(printer_name);
	    if(job_destname != (char*)NULL) myfree(job_destname);
	    if(job_homenode != (char*)NULL) myfree(job_homenode);
	    continue;	    
	    }

	fputs(printer_name, stdout);		/* print printer name */

	if(! machine_readable)			/* possibly pad out */
	    {
	    len = strlen(printer_name);
	    while(len++ <= MAX_DESTNAME)
            	fputc(' ', stdout);
            }
        else					/* in machine readable mode, */
            {					/* we use a single tab */
            fputc('\t', stdout);
            }

	switch(status)
	    {
	    case PRNSTATUS_IDLE:
                printf("idle\n");
                break;
            case PRNSTATUS_PRINTING:
		if( ! machine_readable )
		    {
		    if(next_retry)
			printf("printing %s (retry %d)\n", remote_jobid(printer_nodename,job_destname,job_id,job_subid,job_homenode), next_retry);
		    else
			printf("printing %s\n", remote_jobid(printer_nodename,job_destname,job_id,job_subid,job_homenode));
		    }
		else
		    {
		    printf("printing %s %d\n", remote_jobid(printer_nodename,job_destname,job_id,job_subid,job_homenode), next_retry);
		    }
		break;
            case PRNSTATUS_CANCELING:
                printf("canceling %s\n", remote_jobid(printer_nodename,job_destname,job_id,job_subid,job_homenode));
		break;
	    case PRNSTATUS_SEIZING:
	    	printf("seizing %s\n", remote_jobid(printer_nodename,job_destname,job_id,job_subid,job_homenode));
		break;
            case PRNSTATUS_STOPPING:
                printf("stopping (printing %s)\n", remote_jobid(printer_nodename,job_destname,job_id,job_subid,job_homenode));
		break;
            case PRNSTATUS_STOPT:
                printf("stopt\n");
                break;
            case PRNSTATUS_HALTING:
                printf("printing %s (halting)\n", remote_jobid(printer_nodename,job_destname,job_id,job_subid,job_homenode));
                break;
            case PRNSTATUS_FAULT:
		if( ! machine_readable )
		    {
		    if(next_retry)
			printf("fault, retry %d in %d seconds\n", next_retry, countdown);
		    else
			printf("fault, no auto retry\n");
		    }
		else
		    {
		    printf("fault %d %d\n", next_retry, countdown);
		    }
		break;
	    case PRNSTATUS_ENGAGED:
		if( ! machine_readable )
		    printf("otherwise engaged or off-line\n");
		else
		    printf("engaged %d %d\n", next_retry, countdown);
		break;
	    case PRNSTATUS_STARVED:
		if(! machine_readable)
		    fputs("waiting for resource ration\n", stdout);
		else
		    fputs("starved\n", stdout);
		break;
            default:                  
		printf("unknown status\n");
	    }

	/* 
	** If it is now printing, otherwise engaged, or in fault print 
	** any message which may have been written into the printer 
	** status file.
	*/
	if( ! machine_readable )
	    {
	    if(status==PRNSTATUS_PRINTING || status==PRNSTATUS_ENGAGED 
	    		|| status==PRNSTATUS_STOPPING || status==PRNSTATUS_FAULT)
		{
		if(message[0])
		    printf("                 (%s)\n", message);
		}
	    }

	myfree(printer_nodename);
	myfree(printer_name);
	myfree(job_destname);
	myfree(job_homenode);
	} /* end of loop for each printer */

    fclose(reply_file);
    return EXIT_OK;
    } /* end of ppop_status() */

/*=====================================================================
** ppop message {printer}
** Retrieve the last message from a certain printer.
=====================================================================*/
int ppop_message(char *argv[])
    {
    char fname[MAX_PATH];
    FILE *statfile;
    char message[MAX_STATUS_MESSAGE + 2];
    
    if( argv[0] == (char*)NULL || argv[1] != (char*)NULL )
    	{
    	fputs("Usage: ppop message _printer_\n", errors);
    	return EXIT_SYNTAX;
    	}

    sprintf(fname, ALERTDIR"/%s.status", argv[0]);
    if( (statfile = fopen(fname, "r")) != (FILE*)NULL )
	{
	if(fgets(message, sizeof(message), statfile) != (char*)NULL)
	    {
	    fputs(message, stdout);
	    fputc('\n', stdout);
	    }
	fclose(statfile);	    
	}
    
    return EXIT_OK;
    } /* end of ppop_message() */

/*=====================================================================
** ppop media {printer}
**
** Show the media which are mounted on a specific printer 
** or all printers in a group.
** Use the "f" command. 
=====================================================================*/
int ppop_media(char *argv[])
    {
    struct Destname destname;
    FILE *FIFO, *reply_file;
    struct fcommand1 f1;
    struct fcommand2 f2;
    int len;

    if(argv[0]==(char*)NULL)
        {
        fprintf(errors,"Usage: ppop media {printer,group,all}\n");
        return EXIT_SYNTAX;
        }

    if( parse_dest_name(&destname,argv[0]) )
    	return EXIT_SYNTAX;

    FIFO = get_ready(destname.destnode);

    fprintf(FIFO,"f %ld %s %s\n",(long)pid,ppr_get_nodename(),argv[0]);
    fflush(FIFO);

    if( ! machine_readable )
	{
	printf("Printer                  Bin             Media\n");
	printf("---------------------------------------------------------------\n");
	}

    if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
	return print_reply();

    /* Read each fcommand return structure, one for each printer. */
    if( ! machine_readable )
    while( fread(&f1,sizeof(struct fcommand1),1,reply_file) )
        {
        len=printf(f1.prnname);			/* display the printer name */

	if(f1.nbins==0 && ! machine_readable)	/* If no bins, */
	    {					/* display a message to that effect. */
	    while(len++<25)
	    	fputc(' ',stdout);
	    printf("(no bins defined)\n");
	    }

        while(f1.nbins--)			/* for each bin */
            {
	    while(len++<25)			/* "tab" out */
                fputc(' ',stdout);

            fread(&f2,sizeof(struct fcommand2),1,reply_file);
            len=printf(f2.bin);			/* print the bin name */

            while(len++<16)			/* "tab" out */
                fputc(' ',stdout);

            printf("%s\n",f2.media);		/* print the media name */

	    len = 0;
            }
        fputc('\n',stdout);
        }

    else
    while( fread(&f1,sizeof(struct fcommand1),1,reply_file) )
        {
        while(f1.nbins--)			/* for each bin */
            {
            fread(&f2,sizeof(struct fcommand2),1,reply_file);

            printf("%s\t%s\t%s\n",f1.prnname,f2.bin,f2.media);
            }
        fputc('\n',stdout);
        }

    return EXIT_OK;
    } /* end of ppop_media() */

/*==========================================================================
** ppop mount {printer} {tray} {media}
**
** Mount a specific media type on a specific location.
** We do this with the "M" (Mount) command. 
==========================================================================*/
int ppop_mount(char *argv[])
    {
    struct Destname destination;
    char *binname,*medianame;      /* the three parameters */
    char padded_medianame[MAX_MEDIANAME];
    FILE *mfile;                            /* media file */
    struct Media ms;
    FILE *FIFO;

    if( argv[0]==(char*)NULL || argv[1]==(char*)NULL || argv[2]==(char*)NULL )
        {
        fprintf(errors,"Usage: ppop mount {printer} {location} {media}\n");
        return EXIT_SYNTAX;
        }         

    if( assert_am_operator() )
        return EXIT_DENIED;

    if( parse_dest_name(&destination,argv[0]) )
    	return EXIT_SYNTAX;

    binname = argv[1];			/* to variables with */
    medianame = argv[2];		/* memorable names */

    ASCIIZ_to_padded(padded_medianame,medianame,sizeof(padded_medianame));

    if( (mfile=fopen(MEDIAFILE,"r"))==(FILE*)NULL )
        {
        fprintf(errors,"Can't open \"%s\".",MEDIAFILE);
        return EXIT_INTERNAL;
        }

    while(TRUE)
        {
        if( fread(&ms,sizeof(struct Media),1,mfile) == 0 )
            {
            fclose(mfile);
            fprintf(errors,"Media \"%s\" is unknown.\n",medianame);
            return EXIT_ERROR;
            }
        if(memcmp(padded_medianame,ms.medianame,sizeof(ms.medianame))==0)
            break;
        }
    fclose(mfile);

    /*
    ** get ready for the response from pprd
    ** and send the mount command
    */
    FIFO = get_ready(destination.destnode);
    fprintf(FIFO,"M %ld %s %s %s %s\n",(long)pid,ppr_get_nodename(),argv[0],binname,medianame);
    fflush(FIFO);

    wait_for_pprd(TRUE);
        
    return print_reply();
    } /* end of ppop_mount() */

/*==========================================================================
** ppop start {printer}
**
** start a printer which was stopt
** We do this with the "t" command.
**
** ppop [w]stop {printer}
**
** Stop a printer gently, let current job finish.
** We do this with the "p" command.
** If the second parameter is true, we will wait for 
** the printer to stop.
**
** ppop halt {printer}
**
** Halt, stop a printer now, terminating printing of the current job.
** We do this with the "b" (stop with a bang) command.
** "h" is used by hold.
==========================================================================*/
int ppop_start_stop_wstop_halt(char *argv[], int variation)
    {
    int x;
    int result = 0;
    struct Destname destination;
    FILE *FIFO;

    if(argv[0] == (char*)NULL)
        {
	switch(variation)
	    {
	    case 0:
		fputs("Usage: ppop start <printer> ...\n\n"
			"This command starts a previously stopt printer.\n", errors);
		break;
	    case 1:
	    case 2:
		fputs("Usage: ppop stop <printer> ...\n"
			"Usage: ppop wstop {printer}\n\n"
			"This command stops a printer from printing.  If a job is being\n"
			"printed when this command is issued, the printer does not\n"
			"actually stop until the job is done.\n\n"
			"The second form, \"ppop wstop\" does not return until the printer\n"
			"has actually stopt.\n", errors);
		break;
	    case 3:
		fputs("Usage: ppop halt <printer> ...\n\n"
			"This command stops the printer immediately.  If a job is printing,\n"
			"the job is returned to the queue for later printing.\n", errors);
		break;
	    }

        return EXIT_SYNTAX;
        }

    if( assert_am_operator() )		/* only allow operator to do this */
        return EXIT_DENIED;

    for(x=0; argv[x] != (char*)NULL; x++)
	{
	if( parse_dest_name(&destination, argv[x]) )
	    {
	    result = EXIT_SYNTAX;
	    break;
	    }

	FIFO = get_ready(destination.destnode);

	switch(variation)
	    {
	    case 0:		/* start */
		fprintf(FIFO, "t %ld %s %s\n",(long)pid,ppr_get_nodename(), argv[x]);
		break;
	    case 1:		/* stop */
		fprintf(FIFO, "p %ld %s %s\n",(long)pid,ppr_get_nodename(), argv[x]);
		break;
	    case 2:		/* wstop */
		fprintf(FIFO, "P %ld %s %s\n",(long)pid,ppr_get_nodename(), argv[x]);
		break;
	    case 3:		/* halt */
		fprintf(FIFO, "b %ld %s %s\n", (long)pid, ppr_get_nodename(), argv[x]);
		break;
	    }

	fflush(FIFO);			/* send the command */

	wait_for_pprd(variation != 2);	/* no timeout for wstop */

	if( (result = print_reply()) ) break;
	}

    return result;
    } /* end of ppop_start() */

/*=======================================================================
** This implements ppop cancel in cases where the argument is only
** a destination name.  It is used to cancel all of a user's own
** jobs.  It uses custom_list() from ppop_cmds_listq.c to determine
** which jobs belong to this user.
=======================================================================*/
static int ppop_cancel_byuser_total;
static int ppop_cancel_byuser_inform;

static void ppop_cancel_byuser_help(void)
    { fputs("Syntax error.\n", errors); }

static void ppop_cancel_byuser_banner(void)
    { }
    
static int ppop_cancel_byuser_item(struct QEntry *qentry,        
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    FILE *FIFO, *reply_file;
    int count;
    
    if( ! is_my_job(qentry, qfileentry) )
    	return FALSE;	/* don't stop */

    FIFO = get_ready(qfileentry->destnode);
    fprintf(FIFO,"c %ld %s %s %d %d %s %d\n", (long)pid, qfileentry->destnode, qfileentry->destname, qentry->id, qentry->subid, qfileentry->homenode, ppop_cancel_byuser_inform);
    fflush(FIFO);

    if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
	{
	print_reply();
	return TRUE;	/* stop */
	}

    fscanf(reply_file, "0 %d", &count);
    ppop_cancel_byuser_total += count;

    fclose(reply_file);
    
    return FALSE;	/* don't stop */
    } /* end of ppop_cancel_byuser_item() */

static int ppop_cancel_byuser(char *destname, int inform)
    {
    char *list[2];
    int ret;

    ppop_cancel_byuser_inform = inform;
    ppop_cancel_byuser_total = 0;

    list[0] = destname;		/* custom_list() wants an argument list, */
    list[1] = (char*)NULL;	/* construct one. */

    if( (ret=custom_list(list, ppop_cancel_byuser_help, ppop_cancel_byuser_banner, ppop_cancel_byuser_item, FALSE, -1)) )
    	return ret;

    say_canceled(ppop_cancel_byuser_total);
    	
    return EXIT_OK;    	
    } /* end of ppop_cancel_byuser() */

/*========================================================================
** ppop cancel {<job>, <destination>, all}
**
** Cancel a job or all jobs on a destination.
** We do these things with the "c"ancel command.
========================================================================*/
int ppop_cancel(char *argv[], int inform)
    {
    int x;
    struct Jobname job;
    FILE *FIFO, *reply_file;
    int count;

    if(argv[0]==(char*)NULL)
        {
        fputs("Usage: ppop cancel {<job>, <destination>}\n\n"
              "This command cancels a job or all of your jobs queued for the\n"
              "specified destination.\n",errors);

        return EXIT_SYNTAX;
        }

    for(x=0; argv[x]!=(char*)NULL; x++)
	{
	if( parse_job_name(&job, argv[0]) == -1 )
	    return EXIT_SYNTAX;

	if(job.id == -1)			/* if all jobs, */
	    {
	    int ret;
	    if( (ret=ppop_cancel_byuser(argv[x], inform)) )
	    	return ret;
	    continue;
	    }

	if( job_permission_check(&job) )
	    return EXIT_DENIED;
		
	FIFO = get_ready(job.destnode);
	fprintf(FIFO,"c %ld %s %s %d %d %s %d\n", (long)pid, job.destnode, job.destname, job.id, job.subid, job.homenode, inform);
	fflush(FIFO);

	if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
	    return print_reply();

	fscanf(reply_file, "0 %d", &count);

	say_canceled(count);

	fclose(reply_file);
	}

    return EXIT_OK;
    } /* end of ppop_cancel() */

/*========================================================================
** ppop purge _destination_
**
** Cancel all jobs on a destination.  Only an operator may do this.
** We do it with the "c"ancel command and -1 for the job id and subid.
========================================================================*/
int ppop_purge(char *argv[], int inform)
    {
    int x;
    struct Jobname job;
    FILE *FIFO, *reply_file;
    int count;

    if(argv[0] == (char*)NULL)
    	{
    	fputs("Usage: ppop purge <destination> ...\n\n"
    		"This command cancels all jobs queued for a particular destination.\n"
    		"Only an operator may use this command.  Extra arguments are\n"
    		"interpreted as the names of extra destinations to purge.\n", errors);
    	
	return EXIT_SYNTAX;
    	}

    if( assert_am_operator() )
	return EXIT_DENIED;
    
    for(x=0; argv[x] != (char*)NULL; x++)
	{
	if( parse_job_name(&job, argv[x]) == -1 )
	    return EXIT_SYNTAX;
	
	if( job.id != -1 || job.subid != -1 )
	    {
	    fputs("Please use \"ppop cancel\" to cancel individual jobs.\n", errors);
	    return EXIT_SYNTAX;
	    }
	    
	FIFO = get_ready(job.destnode);
	fprintf(FIFO,"c %ld %s %s -1 -1 %s %d\n",(long)pid,job.destnode,job.destname,job.homenode, inform);
	fflush(FIFO);

	if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
	    return print_reply();

	fscanf(reply_file,"0 %d", &count);

	say_canceled(count);		/* say how many were canceled */

	fclose(reply_file);
	}

    return EXIT_OK;
    } /* end of ppop_purge() */

/*=======================================================================
** ppop clean _destination_
**
** Delete all the arrested jobs on a destination.
=======================================================================*/
static int ppop_clean_total;

static void ppop_clean_help(void)
    { fputs("Syntax error.\n", errors); }

static void ppop_clean_banner(void)
    { }
    
static int ppop_clean_item(struct QEntry *qentry,        
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    FILE *FIFO, *reply_file;
    int count;
    
    if(qentry->status != STATUS_ARRESTED)
    	return FALSE;	/* don't stop */

    /*
    ** Send a cancel command to pprd or rpprd.  The value for the 7th
    ** field (inform) does not matter since the user is never informed
    ** of the deletion of arrested jobs.
    */
    FIFO = get_ready(qfileentry->destnode);
    fprintf(FIFO,"c %ld %s %s %d %d %s 1\n", (long)pid, qfileentry->destnode, qfileentry->destname, qentry->id, qentry->subid, qfileentry->homenode);
    fflush(FIFO);

    if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
	{
	print_reply();
	return TRUE;	/* stop */
	}

    fscanf(reply_file, "0 %d", &count);
    ppop_clean_total += count;

    fclose(reply_file);

    return FALSE;	/* don't stop */
    } /* end of ppop_clean_item() */

int ppop_clean(char *argv[])
    {
    int ret;

    if(argv[0] == (char*)NULL)
    	{
    	fputs("Usage: ppop clean <destination> ...\n\n"
    		"This command will delete all of the arrested jobs\n"
    		"queued for the indicated destination or destinations\n", errors);
    	exit(EXIT_SYNTAX);
    	}

    if( assert_am_operator() )
	return EXIT_DENIED;
    
    ppop_clean_total = 0;

    if( (ret=custom_list(argv, ppop_clean_help, ppop_clean_banner, ppop_clean_item, FALSE, -1)) )
	return ret;

    say_canceled(ppop_clean_total);
	
    return EXIT_OK;    	
    } /* end of ppop_clean() */

/*=======================================================================
** ppop cancel-active <destination>
** ppop cancel-my-active <destination>
**
** Delete the active job for the destination.
=======================================================================*/
static ppop_cancel_active_total;
static int ppop_cancel_active_my;
static int ppop_cancel_active_inform;

static void ppop_cancel_active_help(void)
    { fputs("Syntax error.\n", errors); }

static void ppop_cancel_active_banner(void)
    { }
    
static int ppop_cancel_active_item(struct QEntry *qentry,        
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    FILE *FIFO, *reply_file;
    int count;
    
    if(qentry->status < 0)
    	return FALSE;

    if( ppop_cancel_active_my && ! is_my_job(qentry, qfileentry) )
    	return TRUE;

    FIFO = get_ready(qfileentry->destnode);
    fprintf(FIFO, "c %ld %s %s %d %d %s %d\n", (long)pid, qfileentry->destnode, qfileentry->destname, qentry->id, qentry->subid, qfileentry->homenode, ppop_cancel_active_inform);
    fflush(FIFO);

    if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
	{
	print_reply();
	return TRUE;
	}

    fscanf(reply_file, "0 %d", &count);

    fclose(reply_file);
    
    ppop_cancel_active_total += count;

    return FALSE;
    } /* end of ppop_cancel_active_item() */

int ppop_cancel_active(char *argv[], int my, int inform)
    {
    int ret;
    struct Jobname job;

    /* If parameter missing or it is a job id rather than a destination id, */
    if(argv[0] == (char*)NULL || parse_job_name(&job, argv[0]) || job.id != -1)
    	{
    	fprintf(errors, "Usage: ppop cancel-%sactive <destination> ...\n\n"
    		"This command will delete all of the arrested jobs\n"
    		"queued for the indicated destination or destinations\n", my ? "my-" : "");
    	exit(EXIT_SYNTAX);
    	}

    if( ! my && assert_am_operator() )
	return EXIT_DENIED;
    
    ppop_cancel_active_my = my;
    ppop_cancel_active_inform = inform;
    
    ppop_cancel_active_total = 0;

    if( (ret=custom_list(argv, ppop_cancel_active_help, ppop_cancel_active_banner, ppop_cancel_active_item, FALSE, -1)) )
	return ret;
	
    switch(ppop_cancel_active_total)
    	{
	case 0:
	    if(my)
	    	fputs("You have no active jobs to cancel.\n", stdout);
	    else
	    	fputs("There are no active jobs to cancel.\n", stdout);
	    break;
	case 1:
	    fputs("1 active job was canceled.\n", stdout);
	    break;
	default:
	    printf("%d active jobs were canceled.\n", ppop_cancel_active_total);
	    break;
    	}

    return EXIT_OK;    	
    } /* end of ppop_cancel_active() */

/*========================================================================
** ppop move {destination|job} {destination}
**
** Move a job to a new destination or
** move all jobs from one destination to another.
** We perform these functions with the "m" command.
========================================================================*/
int ppop_move(char *argv[])
    {
    struct Jobname job;
    struct Destname destination;
    FILE *FIFO;

    if( (argv[0]==(char*)NULL) || (argv[1]==(char*)NULL) )
        {
        fputs("Usage: ppop move _JOBID_ _DESTID_\n"
              "       ppop move _DESTID_ _DESTID_\n\n"
              "This command moves a job or jobs to a different queue.\n",errors);
        return EXIT_SYNTAX;
        }

    if( parse_job_name(&job, argv[0]) )
	return EXIT_SYNTAX;

    if( job.id == -1 )				/* all jobs */
	if( assert_am_operator() )
	    return EXIT_DENIED;
    else if( job_permission_check(&job) )	/* one job */
    	return EXIT_DENIED;

    if( parse_dest_name(&destination, argv[1]) )
    	return EXIT_SYNTAX;

    FIFO = get_ready(job.destnode);	/* !!! */

    fprintf(FIFO,"m %ld %s %s %d %d %s %s %s\n",
	    (long)pid,
            job.destnode,job.destname,job.id,job.subid,job.homenode,
            destination.destnode,destination.destname);
    
    fflush(FIFO);			    /* force the buffer contents out */

    wait_for_pprd(TRUE);

    return print_reply();
    } /* end of ppop_move() */

/*===========================================================================
** ppop rush {job}
**
** Move a job to the head of the queue.
===========================================================================*/
int ppop_rush(char *argv[])
    {
    int x;
    struct Jobname job;
    FILE *FIFO;
    int result = 0;

    if( argv[0]==(char*)NULL )
        {
        fputs("Usage: ppop rush _JOBID_ ...\n\n"
        	"This command moves the specified jobs to the head of the queue.\n",errors);
        return EXIT_SYNTAX;
        }

    if( assert_am_operator() )		/* only operator may rush jobs */
        return EXIT_DENIED;

    for(x=0; argv[x] != (char*)NULL; x++)
	{
	if( parse_job_name(&job, argv[x]) )
	    return EXIT_SYNTAX;

	FIFO = get_ready(job.destnode);
	fprintf(FIFO,"U %ld %s %s %d %d %s\n",(long)pid,job.destnode,job.destname,job.id,job.subid,job.homenode);
	fflush(FIFO);

	wait_for_pprd(TRUE);
    
	if( (result = print_reply()) ) break;
	}

    return result;
    } /* end of ppop_rush() */

/*=========================================================================
** ppop hold {job}
**
** Place a specific job on hold.
** We do this by sending pprd the "h" command.
**
** ppop release {job}
**
** Release a previously held or arrested job.
** We do this by sending pprd the "r" command.
=========================================================================*/
int ppop_hold_release(char *argv[], int release)
    {
    int x;
    struct Jobname job;
    FILE *FIFO;
    int result = 0;

    if(argv[0]==(char*)NULL)
        {
	if(! release)
	    {
            fputs("Usage: ppop hold _JOB_ID_ ...\n\n"
        	"This causes jobs to be placed in the held state.  A job\n"
        	"which is held will not be printed until it is released.\n", errors);
	    }
	else
	    {
            fputs("Usage: ppop release _JOB_ID_ ...\n\n"
        	"This command releases previously held or arrested jobs.\n", errors);
	    }
        return EXIT_SYNTAX;
        }

    for(x=0; argv[x] != (char*)NULL; x++)
	{
	if( parse_job_name(&job, argv[x]) )
	    return EXIT_SYNTAX;

	if( job.id == -1 )
	    {
	    fputs("You must indicate a specific job.\n", errors);
	    return EXIT_SYNTAX;
	    }

	if( job_permission_check(&job) )
	    return EXIT_DENIED;

	FIFO = get_ready(job.destnode);
	fprintf(FIFO,"%c %ld %s %s %d %d %s\n", release ? 'r' : 'h',
		(long)pid, job.destnode, job.destname, job.id, job.subid, job.homenode);
	fflush(FIFO);
    
	wait_for_pprd(TRUE);

	if( (result = print_reply()) ) break;
	}
	
    return result;
    } /* end of ppop_hold_release() */

/*==========================================================================
** ppop accept {destination}
** ppop reject {destination}
**
** Set a printer or group to accept or reject print jobs.
==========================================================================*/
int ppop_accept_reject(char *argv[], int reject)
    {
    struct Destname destination;
    FILE *FIFO;

    if(argv[0]==(char*)NULL)
        {
	if(! reject)
	    fputs("Usage: ppop accept _DESTINATION_\n", errors);
	else
	    fputs("Usage: ppop reject _DESTINATION_\n", errors);

    fputs("\n\tThis command sets the status of a destination.\n"
	"\tThe status of a destination may be displayed with\n"
	"\tthe \"ppop destination\" command.\n", errors);

    return EXIT_SYNTAX;
    }

    if( assert_am_operator() )
        return EXIT_DENIED;

    if( parse_dest_name(&destination,argv[0]) )
    	return EXIT_SYNTAX;

    FIFO = get_ready(destination.destnode);

    if(! reject)
        fprintf(FIFO,"A %ld %s %s\n",(long)pid,destination.destnode,destination.destname);
    else
        fprintf(FIFO,"R %ld %s %s\n",(long)pid,destination.destnode,destination.destname);

    fflush(FIFO);

    wait_for_pprd(TRUE);

    return print_reply();
    } /* end of ppop_accept_reject() */

/*===========================================================================
** ppop dest {destination}
**
** Show the type and status of one or more destinations.
===========================================================================*/
int ppop_destination(char *argv[], int comment_too)
    {
    struct Destname destination;
    FILE *FIFO, *reply_file;
    int isgroup,accepting,isprotected;
    char line[1024];
    char *destname;
    int len;

    if(argv[0]==(char*)NULL)
        {
        fputs("Usage: ppop dest[ination] {_DESTINATION_,all}\n"
              "       ppop ldset {_DESTINATION_,all}\n\n"
        	"\tThis command displays the status of a print\n"
        	"\tdestination.  A print destination is a printer\n"
        	"\tor a group of printers.  A destination may be\n"
        	"\tbe set to either accept or reject print jobs\n"
        	"\tsent to it.  If a print job is rejected, it is\n"
        	"\tcanceled and the user is so informed.\n\n"
        	"\tThe ppop ldest form of this command displays the\n"
        	"\tcomment attached to the printer or group.\n",errors);

        return EXIT_ERROR;
        }

    if( parse_dest_name(&destination,argv[0]) )
    	return EXIT_SYNTAX;

    FIFO = get_ready(destination.destnode);
    fprintf(FIFO,"D %ld %s %s\n",(long)pid,destination.destnode,destination.destname);
    fflush(FIFO);

    /* If a human is reading our output, give column names. */
    if( ! machine_readable )
	{
	printf("Destination      Type      Status      Protected  %s\n",comment_too ? "Comment" : "");
	puts(  "---------------------------------------------------------------------------");
	/*     "1234567890123456 printer   accepting   yes"       */
	/*     "                 group     rejecting   no"        */
	}

    if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
	return print_reply();		/* if error, print it */

    while( fgets(line,sizeof(line),reply_file) != (char*)NULL )
        {
	destname = (char*)NULL;
	if( ppr_sscanf(line,"%S %d %d %d",&destname,&isgroup,&accepting,&isprotected) != 4 )
	    {
	    printf("Malformed response: %s",line);
	    if(destname != (char*)NULL)
	    	myfree(destname);
	    continue;
	    }

	fputs(destname,stdout);			/* print destination name */
	len=strlen(destname);			/* <-- fputs() does not return length in all C libraries */

	if( machine_readable )			/* move to next column */
	    {
	    fputc('\t',stdout);
	    }
	else
	    {
	    while( len++ < 17 )
                fputc(' ',stdout);
            }

        if(isgroup)				/* say if it is a group */
	    { fputs("group",stdout); len=5; }
        else					/* or a printer */
	    { fputs("printer",stdout); len=7; }

	if( machine_readable )
	    {
	    fputc('\t',stdout);
	    }
	else
	    {
            while( len++ < 10 )			/* to to next column */
        	fputc(' ',stdout);
            }

	if(accepting)				/* say if it */
            fputs("accepting",stdout);		/*  is accepting */
	else					/*  or rejecting */
	    fputs("rejecting",stdout);

	if( machine_readable )
	    fputc('\t',stdout);
	else
	    fputs("   ",stdout);

	if(isprotected)				/* Is use restricted? */
	    {fputs("yes",stdout); len=3;}
	else
	    {fputs("no",stdout); len=2;}

	if(comment_too)				/* If we should display the */
	    {					/* destination comment */
	    char fname[MAX_PATH];
	    char line[MAX_CONFLINE+2];
	    FILE *f;

	    if( isgroup )
		strcpy(fname,GRCONF);
	    else
	    	strcpy(fname,PRCONF);
	    	
	    strcat(fname,"/");
	    strcat(fname,destname);
	    
	    if( (f=fopen(fname,"r")) != (FILE*)NULL )
	    	{
		while( fgets(line,sizeof(line),f) != (char*)NULL )	    	
	    	    {
		    if( strncmp(line,"Comment: ",9) == 0 )
		    	{
			if(machine_readable)
			    {
			    fputc('\t',stdout);
			    }
			else
			    {
			    while(len++ < 11)
			    	fputc(' ',stdout);
			    }		    	

			fwrite( &line[9], sizeof(char), strcspn(&line[9],"\n"), stdout);
		    	break;
		    	}
	    	    }

		fclose(f);
	    	}
	    }

	fputc('\n',stdout);		/* end of this entry in the list */

	myfree(destname);
        }

    fclose(reply_file);

    return EXIT_OK;
    } /* end of ppop_destination() */

/*===========================================================================
** ppop alerts {printer}
**
** Show the alert messages for a specific printer.
===========================================================================*/
int ppop_alerts(char *argv[])
    {
    char *prn=argv[0];
    char fname[MAX_PATH];
    struct stat statbuf;
    FILE *f;
    int c;

    if( (prn==(char*)NULL) || ((int)strlen(prn) > (int)MAX_DESTNAME) )
        {
        fprintf(errors,"Invalid printer name.\n");
        return EXIT_SYNTAX;
        }

    sprintf(fname,"%s/%s",PRCONF,prn);
    if(stat(fname,&statbuf))
        {
        fprintf(errors,"Printer \"%s\" does not exist.\n",prn);
        return EXIT_ERROR;
        }

    sprintf(fname,"%s/%s",ALERTDIR,prn);
    if((f=fopen(fname,"r"))==(FILE*)NULL)
        {
        fprintf(errors,"No alerts for printer \"%s\".\n",prn);
        return EXIT_OK;
        }

    while( (c=fgetc(f))!=EOF )
        fputc(c,stdout);

    fclose(f);    
    return EXIT_OK;
    } /* end of ppop_alerts() */

/*==========================================================================
** ppop log {job}
**
** Show the log for a specific print job.
==========================================================================*/
int ppop_log(char *argv[])
    {
    struct Jobname job;
    char fname[MAX_PATH];
    struct stat statbuf;
    FILE *f;
    int c;

    if(parse_job_name(&job,argv[0]))
        {
        fprintf(errors,"Invalid job id.\n");
        return EXIT_SYNTAX;
        }

    if(job.id == -1)
    	{
    	fprintf(errors,"You must indicate a specific job.\n");
    	return EXIT_SYNTAX;
    	}

    /* construct the name of the queue file */
    sprintf(fname,"%s/%s:%s-%d.%d(%s)",
	QUEUEDIR,
	job.destnode,job.destname,job.id,job.subid,job.homenode);

    /* make sure the queue file is there */
    if(stat(fname,&statbuf))
        {
        fprintf(errors,"Job \"%s\" does not exist.\n",local_jobid(job.destname,job.id,job.subid,job.homenode));
        return EXIT_ERROR;
        }

    /* construct the name of the log file */
    sprintf(fname,"%s/%s:%s-%d.%d(%s)-log",DATADIR,
	job.destnode,job.destname,job.id,job.subid,job.homenode);

    /* open the log file */
    if((f=fopen(fname,"r"))==(FILE*)NULL)
        {
        fprintf(errors,"No log for job \"%s\".\n",local_jobid(job.destname,job.id,job.subid,job.homenode));
        return EXIT_OK;
        }

    while( (c=fgetc(f))!=EOF )
	fputc(c,stdout);
	
    fclose(f);    
    return EXIT_OK;
    } /* end of ppop_log() */

/* end of file */
