/*
** ~ppr/src/ppop/ppop_cmds_listq.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** Last modified 20 February 1997.
*/

/*
** This module contains the code for the ppop queue
** listing sub-commands such as "ppop list", "ppop short",
** "ppop lpq" and "ppop qquery".
*/

#include "global_defines.h"
#include "global_structs.h"
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <stdlib.h>
#include "pprd.h"
#include "ppop.h"
#include "util_exits.h"

/*
** Those subcommands which allow PPRDEST to be used as the default argument
** should call this.
*/
char **allow_PPRDEST(char *argv[])
    {
    static char *PPRDEST_argv[2];
    
    if(argv[0] == (char*)NULL && (PPRDEST_argv[0] = getenv("PPRDEST")) != (char*)NULL)
	{
	PPRDEST_argv[1] = (char*)NULL;
	argv = PPRDEST_argv;
	}
    return argv;
    }

/*
** Routine for printing the job status.
**
** It is passed the status code and the pointer to the file structure
** for the communications file since if the job is printing we will have
** to read the name of the printer as the next record in the comfile.
**
** It is also passed the ``handle'' of the queue file so it can read
** the required media names.  Lastly, it is passed the indent to use when
** printing continuation lines.
*/
static void job_status(struct QEntry *qentry, struct QFileEntry *qfileentry,
         char *onprinter, FILE *vfile, int indent, int reason_indent)
    {
    char tempstr[MAX_QFLINE+2];		/* for reading queue file */
    int x;

    /* Print a message appropriate to the status. */
    switch(qentry->status)
        {
        case STATUS_WAITING:			/* <--- waiting for printer */
            printf("waiting for printer\n");    
            if(qentry->never)			/* If one or more counted out, */
                {				/* assume it is a group */
                for(x=0;x<indent;x++)		/* and explain. */
                    fputc(' ',stdout);
                printf("(Not all \"%s\"\n",qfileentry->destname);
                for(x=0;x<indent;x++)
                    fputc(' ',stdout);
                printf("members are suitable.)\n");
                }
            break;
        case STATUS_WAITING4MEDIA:		/* <--- waiting for media */
	    if(vfile != (FILE*)NULL)		/* If we have an open queue file to work with. */
		{
                printf("waiting for media:\n");
                while( fgets(tempstr, sizeof(tempstr), vfile) != (char*)NULL && strcmp(tempstr, QF_ENDTAG1) )
                    {
                    if(strncmp(tempstr,"Media: ",7)==0)
                        {
                        for(x=0;x<indent;x++)	/* indent */
                            fputc(' ',stdout);
			for(x=7; tempstr[x]!=(char)NULL && tempstr[x]!=' '; x++)
			    fputc(tempstr[x],stdout);
			fputc('\n',stdout);
                        }
                    }
		}
	    else			/* Caller wants to do it itself. */
		{
		printf("waiting for media\n");
		}
            break;
        case STATUS_HELD:		/* <--- job is held */
            printf("held\n");
            break;
        case STATUS_ARRESTED:		/* <--- job is arrested */
            printf("arrested\n");  
	    if(vfile != (FILE*)NULL)
	      {
	      /* Scan the rest of the queue file. */
              while( fgets(tempstr, sizeof(tempstr), vfile) != (char*)NULL && strcmp(tempstr, QF_ENDTAG1) )
                {
		/* Look for "Reason:" lines */
                if(strncmp(tempstr,"Reason: ",8)==0)
                    {
		    char *ptr;

		    tempstr[strcspn(tempstr,"\n")]=(char)NULL;	/* remove trailing \n */

		    /* print indented, wrapping at commas */
		    for(ptr=&tempstr[8]; *ptr; )
		    	{
                    	for(x=0;x<reason_indent;x++)	/* indent */
                            fputc(' ',stdout);
			
			do  {
			    if( *ptr == '|' )		/* break at | but */
			    	{			/* don't print the | */
			    	ptr++;
			    	break;
			    	}
			    fputc(*ptr,stdout);
			    } while( *(ptr++) != ',' && *ptr );

			fputc('\n',stdout);
			}
                    break;	/* print only the first reason */
                    } /* end of if this is a "Reason:" line */
                } /* end of queue file line loop */
	      } /* end if if there is an open queue file */
            break;
        case STATUS_CANCEL:		/* <--- job is being canceled */
            printf("being canceled\n");
            break;
	case STATUS_SEIZING:
	    printf("being seized\n");
	    break;
        default:			/* <--- job is being printed */ 
            if(qentry->status >= 0)	/* if greater than -1, is prnid */
                {                       /* read the name of the printer */
		int found = FALSE;
		long bytes_sent;
		int pages_started;
		int pages_printed;

                printf("printing on %s\n",onprinter);

		/* 
		** If we are handed the open queue file, get the values
		** and print from the last "Progress:" line. 
		*/
		if(vfile != (FILE*)NULL)
		    {
		    while( fgets(tempstr, sizeof(tempstr), vfile) != (char*)NULL && strcmp(tempstr, QF_ENDTAG1) )
			{
			if (sscanf(tempstr, "Progress: %ld %d %d", &bytes_sent, &pages_started, &pages_printed) == 3 )
			    found = TRUE;
			}
                    }

		if(found)					/* If "Progress: line was found, */
		    {
		    for(x=0; x<reason_indent; x++)		/* indent */
			fputc(' ', stdout);

		    /* Print percent of bytes sent. */
		    printf( "%d%%", (int)((bytes_sent*(long)100) / qfileentry->attr.postscript_bytes) );

		    if( qfileentry->attr.pages > 0 || pages_started > 0 || pages_printed )
			printf(", page %d", pages_started);

		    if( pages_printed > 0 )
			printf(" (%d)", pages_printed);

		    fputc('\n', stdout);
		    }

                }
            else	/* if hit default case but not a positive status */
		{
                fputs("unknown status\n", stdout);
                }
        } /* end of switch */

    } /* end of job_status() */

/* 
** There are two possible flag pages, so we
** need to execute this code twice.  Theirfor
** it is a subroutine.
*/
static char *describe_flag_page_setting(int setting)
    {
    switch(setting)
	{
	case BANNER_DONTCARE:
	    return "Unspecified";
	case BANNER_YESPLEASE:
	    return "Yes";
	case BANNER_NOTHANKYOU:
	    return "No";
	default:
	    return "INVALID";
	}
    } /* end of describe_flag_page_setting() */
    
static char *describe_proofmode(int proofmode)
    {
    switch(proofmode)
    	{
    	case PROOFMODE_NOTIFYME:
    	    return "NotifyMe";
    	case PROOFMODE_SUBSTITUTE:
    	    return "Substitute";
    	case PROOFMODE_TRUSTME:
    	    return "TrustMe";
    	default:
    	    return "INVALID";
    	}
    } /* end of describe_proofmode() */

static char *describe_pageorder(int pageorder)
    {
    switch(pageorder)
    	{
    	case PAGEORDER_ASCEND:
    	    return "Ascend";
    	case PAGEORDER_DESCEND:
    	    return "Descend";
    	case PAGEORDER_SPECIAL:
    	    return "Special";
    	default:
    	    return "INVALID";
    	}
    } /* end of describe_pageorder() */
    
static char *describe_sigpart(int sigpart)
    {
    switch(sigpart)
    	{
    	case SIG_FRONTS:
    	    return "Fronts";
    	case SIG_BACKS:
    	    return "Backs";
    	case SIG_BOTH:
    	    return "Both";
    	default:
    	    return "INVALID";
    	}
    } /* end of describe_sigpart() */
    
static char *describe_orientation(int orientation)
    {
    switch(orientation)
    	{
    	case ORIENTATION_PORTRAIT:
    	    return "Portrait";
    	case ORIENTATION_LANDSCAPE:
    	    return "Landscape";
    	case ORIENTATION_UNKNOWN:
    	    return "Unknown";
    	default:
    	    return "INVALID";
    	}
    } /* end of describe_orienation() */
    
/*==================================================================== 
** This routine is passed pointers to functions with which 
** to print a queue listing.
**
** This routine all by itself does not implement a command
**
** This uses the "l" command too.
**
** One, (*help)() is called if the wrong number of arguments 
** is passed in.  
**
** The second, (*banner) is called to print the column headings.
**
** The third, (*item) is called once for each queue entry.
**      The first argument is the ASCIIZ name of the destination.
**      The second argument is the pprd queue structure.
**      The Third argument is a structure containing the information
**      from the queue file.
**      The third argument is the handle of the response file.  This
**      is provided so that (*item)() may pass it to job_status().
**      The fourth argument is the handle of the queue file which 
**      should be passed to job_status().
**	If the fifth argument is non-zero, then the header will be
**	suppressed when there are no files in the queue.
====================================================================*/
int custom_list(char *argv[],
		void(*help)(void),      /* function for syntax error */
		void(*banner)(void),    /* function to print column labels */
		int(*item)(struct QEntry *qentry,
		    struct QFileEntry*,
		    char *onprinter,
                    FILE *qstream),	/* file to read Media: from */
		int suppress,		/* True if should suppress header on empty queue */
		int arrested_drop_time)
    {
    int arg_index;
    FILE *FIFO, *reply_file;
    struct Jobname job;			/* Split up job name */
    struct QFileEntry qfileentry;       /* full queue entry */
    struct QEntry qentry;
    int header_printed = FALSE;
    char line[1024];
    time_t time_now;
    int stop = FALSE;

    char *destnode;			/* the data from one line */
    char *destname;
    int id;
    int subid;
    char *homenode;
    int priority;
    int status;
    int never;
    int notnow;
    int pass;
    long int arrest_time;
    char *onprinter;

    if( (argv[0] == (char*)NULL) )
	{			/* If no destinations specified, */
	(*help)();		/* invoke the help function */
	return EXIT_SYNTAX;	/* and exit */
	}

    if(arrested_drop_time >= 0)		/* maybe get current time to compare to arrest time */
	time(&time_now);		/* of arrested jobs */

    if(!suppress)			/* If we should always print header, */
	{				/* print it now */
	if( ! machine_readable )	/* unless a program is reading our output. */
	    (*banner)();
	header_printed = TRUE;
	}

    /* Loop to process each argument: */
    for(arg_index=0; argv[arg_index] != (char*)NULL; arg_index++)
	{
	/* Separate the components of the job name. */
	if( parse_job_name(&job, argv[arg_index]) == -1 )
	    return EXIT_SYNTAX;

	/* Prepare to communicate with pprd or rpprd. */
	FIFO = get_ready(job.destnode);

	fprintf(FIFO,"l %ld %s %s %d %d %s\n", (long)pid, job.destnode, job.destname, job.id, job.subid, job.homenode);

	fflush(FIFO);					/* send command */

	if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )	/* wait for pprd */
	    return print_reply();
	
	/*
	** Loop to process each queue entry.
	*/
	while( ! stop && fgets(line,sizeof(line),reply_file) != (char*)NULL )
	    {
	    destnode = destname = homenode = onprinter = (char*)NULL;
	    never = notnow = pass = 0;

	    if( ppr_sscanf(line,"%S %S %d %d %S %d %d %S %d %d %d %ld",
			&destnode, &destname, &id, &subid, &homenode,
			&priority, &status, &onprinter, &never, &notnow, &pass, &arrest_time) < 8 )
		{
		printf("Invalid response line: %s", line);
		if(destnode != (char*)NULL) myfree(destnode);
		if(destname != (char*)NULL) myfree(destname);
		if(homenode != (char*)NULL) myfree(homenode);
		if(onprinter != (char*)NULL) myfree(onprinter);
		continue;
		}

	    /*
	    ** Read additional information from the queue file included
	    ** in the reply file.
	    */
	    if(read_struct_QFileEntry(reply_file, &qfileentry) == -1)
		printf("Invalid queue entry:\n");
        
	    /*
	    ** Fill in some missing fields
	    ** in the qfileentry structure.
	    */
	    qfileentry.destnode = destnode;
	    qfileentry.destname = destname;
	    qfileentry.id = qentry.id = id;
	    qfileentry.subid = qentry.subid = subid;
	    qfileentry.homenode = homenode;
	    qentry.priority = priority;
	    qentry.status = status;
	    qentry.never = never;
	    qentry.notnow = notnow;
	    qentry.pass = pass;

	    /*
	    ** Normally we print the column headings if they have not 
	    ** been printed already and the -M switch was not used. 
	    ** Then, we call the routine which the caller has specified
	    ** as the one that should be called for each entry.
	    **
	    ** Notice that this whole block of code is skipt if 
	    ** arrest_dropout_time is non-negative and the job is arrested and
	    ** the time in seconds since it was arrested is greater than
	    ** arrest_dropout_time.
	    */
	    if(status != STATUS_ARRESTED || arrested_drop_time < 0 || (time_now - arrest_time) <= arrested_drop_time)
		{
		if(!header_printed)
		    {
		    if( ! machine_readable )
			(*banner)();
		    header_printed = TRUE;
		    }

		stop = (*item)(&qentry, &qfileentry, onprinter, reply_file);
		}
		
	    /*
	    ** Eat up any remaining part of the queue file.
	    */
	    while( fgets(line, sizeof(line), reply_file ) != (char*)NULL && strcmp(line,QF_ENDTAG2) ) ;

	    /*
	    ** Free memory in the qfileentry.
	    ** This will free destnode, destname, and homenode as well.
	    */
            destroy_struct_QFileEntry(&qfileentry);
	    myfree(onprinter);
            } /* end of while, loop back for next job */

        fclose(reply_file);
	} /* end of loop for each argument */

    return EXIT_OK;			/* no errors */
    } /* end of custom_list() */

/*=========================================================================
** ppop short {destination}
** List jobs in the print queue,
=========================================================================*/
static void ppop_short_help(void)
    {
    fputs("Usage: ppop short {_PRINTER_NAME_, all}\n\n"
	"This command shows all the print jobs queue for\n"
	"a certain destination or for all destinations.\n",errors);
    }

static void ppop_short_banner(void)
    {
    printf("Job ID                  For                  Status\n");
    printf("--------------------------------------------------------------\n");
    }
    
static int ppop_short_item(struct QEntry *qentry,        
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    int len;

    len = printf("%s",remote_jobid(qfileentry->destnode,qfileentry->destname,qfileentry->id,qfileentry->subid,qfileentry->homenode));

    while(len++<24)			/* print padded job id */
 	fputc(' ',stdout);             

    len = printf("%s",qfileentry->ForLine);	/* print it */
    while(len++<21)				/* pad to 21 characters */ 
	fputc(' ',stdout);

    /* now print the status */
    job_status(qentry, qfileentry, onprinter, (FILE*)NULL, 45, 45);
					/* media indent, reason indent */
    return FALSE;
    } /* end of ppop_short_item() */

int ppop_short(char *argv[])
    {
    argv = allow_PPRDEST(argv);
    return custom_list(argv, ppop_short_help, ppop_short_banner, ppop_short_item, FALSE, A_switch_value);
    }

/*================================================================
** ppop list {destination}
**
** Normal queue listing.
** This uses custom_list().
================================================================*/
static void ppop_list_help(void)
    {
    fputs("Usage: ppop list {all, _group_, _printer_ ,_jobid_} ...\n\n"
    	"This command will print information about the specified jobs.\n"
    	"Jobs may be specified by job id, by queue, or \"all\" may\n"
    	"be specified.  Multiple specifications may be used with this\n"
    	"command.\n", errors);
    }

void ppop_list_banner(void)
    {
/*      123456789012345678901234567890123456789012345678901234567890 */
printf("Queue ID        For                      Time      Pgs Status\n");    
printf("----------------------------------------------------------------------------\n");
/*      12345678-xxxx.y David Chappell           21 May 79 999 waiting for printer
        glunkish-1004   Joseph Smith             11:31pm   ??? printing on glunkish
        melab_deskjet-1004
                        Joseph Andrews           11:35pm   001 waiting for media
                                                               letterhead
*/
    }

static int ppop_list_item(struct QEntry *qentry,        
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    struct tm *tm_time;
    time_t time_now;
    char timestr[10];
    char pagesstr[4];
    char *jobname;

    /* Decide on a representation of the date
       it was submitted.  This depends on how much that
       date differs from the current time. */
    tm_time = localtime((time_t*)&qfileentry->time);
    time_now = time((time_t*)NULL);
    if( difftime(time_now, (time_t)qfileentry->time) >= (24*60*60) )
        strftime(timestr,sizeof(timestr),"%d-%b-%y",tm_time);  
    else
        strftime(timestr,sizeof(timestr),"%I:%M%p",tm_time);  

    /* Convert the number of pages to ASCII. */
    if( (qfileentry->attr.pages < 0) || (qfileentry->attr.pages > 999) )
        strcpy(pagesstr,"???");
    else
        sprintf(pagesstr,"%3.3d",qfileentry->attr.pages);

    jobname = remote_jobid(qfileentry->destnode,qfileentry->destname,qentry->id,qentry->subid,qfileentry->homenode);
    if( strlen(jobname) > 15 )
    	printf("%s\n               ", jobname);
    else
    	printf("%-15.15s", jobname);
    
    printf(" %-24.24s %-9.9s %s ",
        qfileentry->ForLine != (char*)NULL ? qfileentry->ForLine : "(unknown)",
        timestr, pagesstr);

    job_status(qentry, qfileentry, onprinter, qstream, 55, 55);
    						/* media indent, reason indent */
    return FALSE;
    } /* end of ppop_list_item() */
 
int ppop_list(char *argv[], int suppress)
    {
    argv = allow_PPRDEST(argv);
    return custom_list(argv, ppop_list_help, ppop_list_banner, ppop_list_item, suppress, A_switch_value);
    }

/*================================================================
** ppop lpq {destination}
**
** LPQ style listing.  This command exists to support lprsrv(8).
** It uses custom_list().  Please notice that the -M (machine
** readable format) switch should have no effect on this command.
** This is because this command exists primarily to provide output
** which will be read by other programs.  A separate format for
** reading by programs would be absurd, especially as ppop has
** other subcommands such as "ppop qquery" which are much better
** suited to that sort of thing.  
================================================================*/

static int lpqlist_rank;
static int lpqlist_banner_called;
static char *lpqlist_destnode;
static char *lpqlist_destname;
static char **lpqlist_argv;

static const char *count_suffix(int count)
    {
    switch(count % 10)
	{
	case 1:
	    return "st";
	case 2:
	    return "nd";
	case 3:
	    return "rd";
	default:
	    return "th";
	}
    }
    
/*
** custom_list() calls this if it can't parse the destination name
*/
static void ppop_lpq_help(void)
    {
    fputs("Usage: ppop lpq {all, _group_, _printer_} [user...] [id...]\n\n"
	"The \"ppop lpq\" subcommand prints a queue listing in a format\n"
	"similiar to that of the Berkeley Unix \"lpq\" command.\n", errors);
    }

/*
** ppop_lpq_banner() calls this to print the progress of 
** the job a printer is working on.  This routine writes
** to stdout.  It prints a line fragment of either the form:
**
** " (0% sent)"
**  or
** " (101% sent, 2 of 7 pages completed)"
**
** !!! This routine must be kept up to date with the queue file format. !!!
** !!! This routine is not ready for distributed printing. !!!
*/
static void ppop_lpq_banner_progress(const char *printer_job_destname, int printer_job_id, int printer_job_subid, const char *printer_job_homenode)
    {
    char fname[MAX_PATH];
    FILE *f;

    sprintf(fname, QUEUEDIR"/%s:%s-%d.%d(%s)", ppr_get_nodename(), printer_job_destname, printer_job_id, printer_job_subid, printer_job_homenode);
    if( (f=fopen(fname, "r")) != (FILE*)NULL )
	{
	char line[MAX_QFLINE+2];
	long int postscript_bytes, bytes_sent;
	int pages, pages_printed, N, copies;

	bytes_sent = pages_printed = pages = postscript_bytes = 0;
	copies = N = 1;

	while(fgets(line, sizeof(line), f) != (char*)NULL)
	    {
	    if( sscanf(line, "Progress: %ld %*d %d", &bytes_sent, &pages_printed) )
	    	continue;

	    if( sscanf(line, "Attr: %*d %*f %*d %*d %*d %*d %*d %*d %d %*d %*d %*d %ld", &pages, &postscript_bytes) )
	        continue;
	                                                                                                                                                                                                                                    
	    if( sscanf(line, "Opts: %*d %d", &copies) )
		continue;

	    if( sscanf(line, "N_Up: %d", &N) )
	    	continue;
	    }    

	if(copies == -1) copies = 1;
	pages = (pages + N - 1) / N * copies;

	printf(" (%d%% sent", (int)((bytes_sent*(long)100) / postscript_bytes));

	if(pages_printed > 0 && pages > 0)
	    printf(", %d of %d pages completed", pages_printed, pages);

	fputc(')', stdout);

	fclose(f);
	}
    } /* end of ppop_lpq_banner_progress() */

/*
** This is called once, just before the first queue entry is printed.
*/
static void ppop_lpq_banner(void)
    {
    char line[1024];
    FILE *FIFO, *reply_file;
    char *printer_name;
    char *printer_nodename;
    char *printer_job_destname;
    int printer_job_id, printer_job_subid;
    char *printer_job_homenode;
    int printer_status;
    int printer_next_retry;
    int printer_countdown;
    char printer_message[MAX_STATUS_MESSAGE+2];

    /* Ask for the status of all the printers: */
    FIFO = get_ready(lpqlist_destnode);
    fprintf(FIFO, "s %ld %s %s\n", (long)pid, lpqlist_destnode, lpqlist_destname);
    fflush(FIFO);

    if( (reply_file=wait_for_pprd(TRUE)) == (FILE*)NULL )
    	{
    	print_reply();
    	return;
    	}

    /*
    ** The printer status comes back as a line of space
    ** separated data and a line with the last status message
    ** on it.
    */
    while( fgets(line, sizeof(line), reply_file) != (char*)NULL )
	{
	fgets(printer_message, sizeof(printer_message), reply_file);
	printer_message[strcspn(printer_message, "\n")] = (char)NULL;

	printer_nodename = (char*)NULL;
	printer_name = (char*)NULL;
	printer_job_destname = (char*)NULL;
	printer_job_homenode = (char*)NULL;

	if( ppr_sscanf(line, "%S %S %d %d %d %S %d %d %S",
		&printer_nodename, &printer_name, &printer_status,
		&printer_next_retry, &printer_countdown,
		&printer_job_destname, &printer_job_id, &printer_job_subid,
		&printer_job_homenode) != 9 )
	    {
	    printf("Malformed response: \"%s\"", line);
	    if(printer_nodename != (char*)NULL) myfree(printer_nodename);
	    if(printer_name != (char*)NULL) myfree(printer_name);
	    if(printer_job_destname != (char*)NULL) myfree(printer_job_destname);
	    if(printer_job_homenode != (char*)NULL) myfree(printer_job_homenode);
	    continue;	    
	    }

	/*
	** If there is more than one printer we will prefix the message
	** about each with its name.
	*/
	if(strcmp(lpqlist_destname, printer_name))
	    printf("%s: ", printer_name);

	/*
	** Print the status of this printer.  These messages should 
	** be compatible with the Samba parsing code.
	**
	** Samba expects:
	** OK: "enabled", "online", "idle", "no entries", "free", "ready"
	** STOPPED: "offline", "disabled", "down", "off", "waiting", "no daemon"
	** ERROR: "jam", "paper", "error", "responding", "not accepting", "not running", "turned off"
	*/
	switch(printer_status)
	    {
	    case PRNSTATUS_IDLE:
                printf("idle");
                break;
            case PRNSTATUS_PRINTING:
		if(printer_next_retry)
		    printf("printing %s (%d%s retry)", remote_jobid(printer_nodename,printer_job_destname,printer_job_id,printer_job_subid,printer_job_homenode), printer_next_retry, count_suffix(printer_next_retry));
		else
                    printf("printing %s", remote_jobid(printer_nodename,printer_job_destname,printer_job_id,printer_job_subid,printer_job_homenode));

		ppop_lpq_banner_progress(printer_job_destname, printer_job_id, printer_job_subid, printer_job_homenode);

		break;
            case PRNSTATUS_CANCELING:
		fputs("canceling active job", stdout);
		break;
	    case PRNSTATUS_SEIZING:
	    	fputs("seizing active job", stdout);
	    	break;
            case PRNSTATUS_STOPPING:
		printf("stopping (still printing %s)", remote_jobid(printer_nodename,printer_job_destname,printer_job_id,printer_job_subid,printer_job_homenode));
		break;
            case PRNSTATUS_STOPT:
                printf("printing disabled");
                break;
            case PRNSTATUS_HALTING:
                printf("halting (still printing %s)", remote_jobid(printer_nodename,printer_job_destname,printer_job_id,printer_job_subid,printer_job_homenode));
                break;
            case PRNSTATUS_FAULT:
		if(printer_next_retry)
		    printf("error, %d%s retry in %d seconds", printer_next_retry, count_suffix(printer_next_retry), printer_countdown);
		else
		    printf("error, no auto retry");
		break;
	    case PRNSTATUS_ENGAGED:
		printf("otherwise engaged or offline");
		break;
	    case PRNSTATUS_STARVED:
		printf("waiting for resource ration");
		break;
            default:                  
		printf("unknown status\n");
	    }

	if(printer_message[0] && (printer_status==PRNSTATUS_PRINTING
				|| printer_status==PRNSTATUS_ENGAGED
				|| printer_status==PRNSTATUS_STOPPING
				|| printer_status==PRNSTATUS_FAULT))
	    fprintf(stdout, "; %s\n", printer_message);
	else
	    fputs("\n", stdout);

	myfree(printer_nodename);
	myfree(printer_name);
	myfree(printer_job_destname);
	myfree(printer_job_homenode);
	} /* end of loop for each printer */

    fclose(reply_file);

    /*
    ** Finally, print the banner:
    **                           1234567890123456789012345678901234567890
    **    1st    chappell   8021 entropy.tex                           100801 bytes
    */
    puts("Rank   Owner      Job  Files                                 Total Size");

    /* Suppress "no entries": */
    lpqlist_banner_called = TRUE;
    } /* end of ppop_lpq_banner() */

/*
** This is called for each job in the queue.
*/
static int ppop_lpq_item(struct QEntry *qentry,        
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    char sizestr[16];
    char rankstr[8];
    char *format;		/* printf() format string to use */
    #define fixed_for_LENGTH 10
    char fixed_for[fixed_for_LENGTH + 1];
    #define fixed_name_LENGTH 37
    char fixed_name[fixed_name_LENGTH + 1];
    
    /* Advance the rank number to the next value: */
    lpqlist_rank++;

    /*
    ** See if this file should be excluded because of not matching
    ** the pattern set by the extra arguments.
    */
    if(lpqlist_argv[0] != (char*)NULL)
    	{
	int x;
	int ok = FALSE;
	const char *user;
	#ifdef GNUC_HAPPY
	int user_len = 0;
	#else
	int user_len;
	#endif
	
	if(qfileentry->proxy_for != (char*)NULL)
	    {
	    user = qfileentry->proxy_for;
	    user_len = strcspn(user, "@");
	    }
	else if(qfileentry->ForLine != (char*)NULL)	/* probably never false */
	    {
	    user = qfileentry->ForLine;
	    user_len = strlen(qfileentry->ForLine);
	    }
	else						/* probably never invoked */
	    {
	    user = qfileentry->username;
	    user_len = strlen(user);
	    }
	
	for(x=0; lpqlist_argv[x] != (char*)NULL; x++)
	    {
	    if( (strlen(lpqlist_argv[x]) == user_len && strncmp(lpqlist_argv[x], user, user_len) == 0)
			|| qentry->id == atoi(lpqlist_argv[x]) )
		{
		ok = TRUE;
		break;
		}
	    }
	if(!ok) return FALSE;	/* skip but don't stop */
    	}

    /* Build a gramaticaly correct size string. */
    if(qfileentry->attr.postscript_bytes==1)
	strcpy(sizestr,"1 byte");
    else
        sprintf(sizestr,"%ld bytes", qfileentry->attr.postscript_bytes);

    /*
    ** If the job is printing, its "Rank" is "active",
    ** otherwise it is "1st", "2nd", etc.
    */
    if(qentry->status >= 0)		/* if printing */
	{
    	strcpy(rankstr, "active");
    	}
    else				/* if not printing */
    	{
    	sprintf(rankstr, "%d%s", lpqlist_rank, count_suffix(lpqlist_rank));
    	}

    /*
    ** Change spaces in the user name to underscores because some programs
    ** which parse LPQ output, such as Samba get confused by them.
    */
    if( qfileentry->ForLine != (char*)NULL )
    	{
    	const char *ptr = qfileentry->ForLine;
    	int x;
    	
    	for(x=0; ptr[x] && x < fixed_for_LENGTH; x++)
    	    {
    	    if(isspace(ptr[x]))
		fixed_for[x] = '_';
	    else
	    	fixed_for[x] = ptr[x];
    	    }

	fixed_for[x] = (char)NULL;
    	}
    else
    	{
    	strcpy(fixed_for, "(unknown)");
    	}

    /*
    ** Select a string for the LPQ filename filed.  We would prefer the 
    ** actual file name but will accept the title.
    **
    ** Change spaces in the file name to underscores because some programs
    ** which parse LPQ output, such as Samba get confused by them.
    **
    ** We also change all slashes to hyphens if the name doesn't 
    ** begin with a slash because Samba tries to extract the basename
    ** and gets confused by dates.
    */
    {
    const char *ptr;
    int x;
    if( (ptr=qfileentry->lpqFileName) == (char*)NULL )
	if( (ptr=qfileentry->Title) == (char*)NULL )
	    ptr = "stdin";	    
	
    for(x=0; ptr[x] && x < fixed_name_LENGTH; x++)
	{
	if(isspace(ptr[x]))
		fixed_name[x] = '_';
	else if(ptr[x] == '/' && ptr[0] != '/')
	    	fixed_name[x] = '-';
	else
	    	fixed_name[x] = ptr[x];
	}

    fixed_name[x] = (char)NULL;
    }

    /*
    ** If the job is arrested or held we use a format string which
    ** preempts part of the job name field to indicate that fact.
    ** Otherwise, we use a standard lpq format.
    */
    switch(qentry->status)
	{
	case STATUS_ARRESTED:
	    format = "%-6.6s %-10.10s %-4d %-26.26s (arrested) %s\n";
	    break;
	
	case STATUS_HELD:
	    format = "%-6.6s %-10.10s %-4d %-30.30s (held) %s\n";
	    break;

	default:
	    format = "%-6.6s %-10.10s %-4d %-37.37s %s\n";
	    break;
	}

    /* Print the whole line. */
    printf(format,
 	rankstr,
	fixed_for,
 	qentry->id,       
	fixed_name,
        sizestr);
        
    return FALSE;
    } /* end of ppop_lpq_item() */
 
int ppop_lpq(char *argv[])
    {
    int retval;
    struct Jobname job;
    #define new_argv_SIZE 21
    char *new_argv[new_argv_SIZE];
    int x;

    /* If no argument, use PPRDEST: */
    argv = allow_PPRDEST(argv);

    /* If first parameter is empty or it is a job id rather than a destination id, */
    if(argv[0] == (char*)NULL || parse_job_name(&job, argv[0]) || job.id != -1)
    	{
	ppop_lpq_help();
	exit(EXIT_SYNTAX);
    	}

    lpqlist_rank = 0;			/* reset number for "Rank" column */
    lpqlist_banner_called = FALSE;	/* not called yet! */
    lpqlist_destnode = job.destnode;
    lpqlist_destname = job.destname;
    lpqlist_argv = new_argv;
    
    for(x=0; x < (new_argv_SIZE-1) && argv[x+1] != (char*)NULL; x++)
	{
    	new_argv[x] = argv[x+1];
	argv[x+1] = (char*)NULL;
    	}
    new_argv[x] = (char*)NULL;

    retval = custom_list(argv, ppop_lpq_help, ppop_lpq_banner, ppop_lpq_item, TRUE, A_switch_value);

    if( !lpqlist_banner_called && retval != EXIT_SYNTAX )
    	puts("no entries");

    return retval;
    } /* end of ppop_lpq() */

/*================================================================
** ppop details {destination}
**
** Full debuging listing of the queue.
** This uses custom_list().
================================================================*/
static void ppop_details_help(void)
    {
    fputs("Usage: ppop details {all, <group>, <printer>, <job>}\n\n"
	"This command displays a detailed description of all jobs,\n"
	"the jobs queued for the indicated group, the jobs queued\n"
	"for the indicated printer, or the indicated job.\n",errors);
    }

/* This kind of queue listing does not have column headings. */
static void ppop_details_banner(void)
    { /* empty */ }

static int ppop_details_item(struct QEntry *qentry,        
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    char line[MAX_QFLINE+2];

    /* print job name */
    printf("Job ID: %s\n", remote_jobid(qfileentry->destnode,qfileentry->destname,qentry->id,qentry->subid,qfileentry->homenode));

    /* Say which part of the whole this is. */
    if( qfileentry->attr.parts == 1 )
    	printf("Part: 1 of 1\n");
    else
    	printf("Part: %d of %d\n", qentry->subid,qfileentry->attr.parts);

    /* Print submitting user id. */
    printf("User: %s (%ld)\n", qfileentry->username,(long)qfileentry->user);

    /* Proxy information */
    printf("Proxy For: %s\n", qfileentry->proxy_for!=(char*)NULL ? qfileentry->proxy_for : "");

    /* Who is it for? */
    printf("For: %s\n",qfileentry->ForLine!=(char*)NULL ? qfileentry->ForLine : "(unknown)" );

    /* print the date it was submitted */
    printf("Submission Time: %s", ctime((time_t*)&qfileentry->time) );

    /* print priority */
    printf("Current Priority: %d\n", qentry->priority);
    printf("Origional Priority: %d\n", qfileentry->priority);

    /* What was the file name? */
    printf("lpq filename: %s\n", qfileentry->lpqFileName != (char*)NULL ?qfileentry->lpqFileName : "");

    /* What is the title? */
    printf("Title: %s\n", qfileentry->Title!=(char*)NULL ? qfileentry->Title : "");

    /* Who or what is the creator? */
    printf("Creator: %s\n", qfileentry->Creator!=(char*)NULL ? qfileentry->Creator : "");

    /* what are the routing instructions */
    printf("Routing: %s\n", qfileentry->Routing!=(char*)NULL ? qfileentry->Routing : "");

    /* Flag pages */
    printf("Banner: %s\n",describe_flag_page_setting(qfileentry->do_banner));
    printf("Trailer: %s\n",describe_flag_page_setting(qfileentry->do_trailer));

    /* response methode and address */
    printf("Respond by: %s %s\n", qfileentry->response_method, qfileentry->response_address);
    if(qfileentry->responder_options != (char*)NULL)
    	printf("Responder options: %s\n", qfileentry->responder_options);

    /* attributes */
    printf("Required Language Level: %d\n", qfileentry->attr.langlevel);
    printf("Required Extensions: %d\n", qfileentry->attr.extensions);
    printf("DSC Level: %f\n", qfileentry->attr.DSClevel);
    if(qfileentry->attr.pages >= 0)
	printf("Pages: %d\n", qfileentry->attr.pages);
    else
	printf("Pages: ?\n");
    printf("Page Order: %s\n", describe_pageorder(qfileentry->attr.pageorder));
    printf("Prolog Present: %s\n", qfileentry->attr.prolog ? "True" : "False");
    printf("DocSetup Present: %s\n", qfileentry->attr.docsetup ? "True" : "False");
    printf("Script Present: %s\n", qfileentry->attr.script ? "True" : "False");
    printf("ProofMode: %s\n", describe_proofmode(qfileentry->attr.proofmode));
    printf("Orientation: %s\n", describe_orientation(qfileentry->attr.orientation));
    printf("Pages Per Sheet: %d\n", qfileentry->attr.pagefactor);
    printf("Unfiltered Size: %ld\n", qfileentry->attr.input_bytes);
    printf("PostScript Size: %ld\n", qfileentry->attr.postscript_bytes);

    /* N-Up */
    printf("N-Up N: %d\n", qfileentry->N_Up.N);
    printf("N-Up Borders: %s\n", qfileentry->N_Up.borders?"True":"False");
    printf("Signiture Sheets: %d\n", qfileentry->N_Up.sigsheets);
    printf("Signiture Part: %s\n", describe_sigpart(qfileentry->N_Up.sigpart));

    /* options */
    if(qfileentry->opts.copies < 0)
    	fputs("Copies: ?\n", stdout);
    else
        printf("Copies: %d\n", qfileentry->opts.copies);
    printf("Collate: %s\n", qfileentry->opts.collate ? "True" : "False");
    printf("Auto Bin Select: %s\n", qfileentry->opts.binselect ? "True" : "False");
    printf("Keep Bad Features: %s\n", qfileentry->opts.keep_badfeatures ? "True" : "False");

    /* "Draft" notice */
    printf("Draft Notice: %s\n", qfileentry->Draft ? qfileentry->Draft : "");

    /* Print the job status */
    printf("Status: ");
    job_status(qentry, qfileentry, onprinter, (FILE*)NULL, 8, 8);
	
    /* show the never and notnow masks */          
    printf("Never mask: %d\n", qentry->never);
    printf("NotNow mask: %d\n", qentry->notnow);

    /* Copy the tail end of the queue file to stdout. */
    while( fgets(line, sizeof(line), qstream) != (char*)NULL && strcmp(line, QF_ENDTAG1) ) 
	{
	if( strncmp(line, "End", 3) == 0 )	/* skip delimiters */
	    continue;
    	fputs(line, stdout);
    	}

    /* Put a blank line after this entry. */
    printf("\n");

    return FALSE;		/* don't stop */
    } /* end of ppop_details_item() */
 
int ppop_details(char *argv[])
    {
    return custom_list(argv, ppop_details_help, ppop_details_banner, ppop_details_item, FALSE, A_switch_value);
    } /* end of ppop_details() */

/*================================================================
** ppop qquery {destination}
**
** This command is for use by queue display programs which 
** run ppop to get their data.
================================================================*/

#define MAX_QQUERY_ITEMS 20
int qquery_query[MAX_QQUERY_ITEMS];
int qquery_query_count;

static void ppop_qquery_help(void)
    {
    fputs("Wrong syntax!\n",errors);
    } /* end of ppop_qquery_help() */

static void ppop_qquery_banner(void)
    { /* empty */ }

static void fputs_untabbed(const char *string, FILE *file)
    {
    for( ; *string; string++)
    	{
    	if( *string != '\t' )
    	    fputc(*string, file);
    	else
    	    fputc(' ', file);
    	}    
    }

static int ppop_qquery_item(struct QEntry *qentry,
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    char *status;			/* status string */
    char status_scratch[MAX_DESTNAME+20];
    char explain[256];
    char media[256];			/* list of required media */
    int mi=0;				/* media index */
    char tempstr[MAX_QFLINE+2];		/* for reading queue file lines */
    int x;				/* loop counter */
    char *ptr;				/* general use pointer */

    explain[0] = (char)NULL;		/* start empty */

    while( fgets(tempstr, sizeof(tempstr), qstream) != (char*)NULL && strcmp(tempstr, QF_ENDTAG1) )
	{
	if(strncmp(tempstr, "Media: ", 7) == 0)
	    {
	    ptr = &tempstr[7];
	    if(mi)
		media[mi++] = ' ';
	    while( *ptr != (char)NULL && *ptr != ' ' )
		media[mi++] = *(ptr++);
            }
        else if(strncmp(tempstr, "Reason: ", 8) == 0)
            {
	    char *ptr = &tempstr[8];
	    int di;

	    for(di=0; *ptr!='\n'; ptr++)
		{
		switch(*ptr)
		    {
		    case ',':
			explain[di++] = ',';
		    case '|':
			explain[di++] = ' ';
			break;
		    default:
			explain[di++] = *ptr;
			break;
		    }
		}
	    explain[di] = (char)NULL;
	    }			
	}
    media[mi] = (char)NULL;

    /* Determine the status. */
    switch(qentry->status)
        {
        case STATUS_WAITING:		/* <--- waiting for printer */
            status = "waiting for printer";
            if(qentry->never)		/* If one or more counted out, */
                sprintf(explain, "Not all \"%s\" members are suitable", qfileentry->destname);
	    else
		explain[0] = (char)NULL;
            break;
        case STATUS_WAITING4MEDIA:	/* <--- waiting for media */
	    status = "waiting for media";
	    strcpy(explain, media);
            break;
        case STATUS_HELD:		/* <--- job is held */
            status = "held";
            break;
        case STATUS_ARRESTED:		/* <--- job is arrested */
            status = "arrested";
            break;
        case STATUS_CANCEL:		/* <--- job is being canceled */
            status = "being canceled";
            break;
	case STATUS_SEIZING:
	    status = "being seized";
	    break;
        default:			/* <--- job is being printed */ 
            if(qentry->status>=0)       /* if greater than -1, is prnid */
                {                       /* read the name of the printer */
		sprintf(status_scratch, "printing on %s", onprinter);
		status = status_scratch;
                }
            else	/* if hit default case but not a positive status */
		{
                status = "unknown status";
                }
        } /* end of switch(qentry->status) */

    /* Print the requested fields. */
    for(x=0; x < qquery_query_count; x++)
    	{
	if(x)
	    fputc('\t', stdout);
	    
  	switch(qquery_query[x])
  	    {
  	    case 0:			/* jobname */
  		fputs(remote_jobid(qfileentry->destnode,qfileentry->destname,qfileentry->id,qfileentry->subid,qfileentry->homenode),stdout);
  		break;
  	    case 1:			/* for */	    
		fputs_untabbed(qfileentry->ForLine != (char*)NULL ? qfileentry->ForLine : "(unknown)", stdout);
		break;
  	    case 2:			/* title */
		fputs_untabbed(qfileentry->Title != (char*)NULL ? qfileentry->Title : "", stdout);
		break;
  	    case 3:			/* status */
		fputs_untabbed(status, stdout);
		break;
  	    case 4:			/* explain */
		fputs(explain, stdout);
		break;
  	    case 5:			/* copies */
		if(qfileentry->opts.copies < 0)
  	    	    fputs("1?", stdout);
		else
  	    	    printf("%d", qfileentry->opts.copies);
		break;
  	    case 6:			/* copiescollate */
		if(qfileentry->opts.collate)
  	            fputs("true", stdout);
		else
  	            fputs("false", stdout);
		break;
  	    case 7:			/* pagefactor */
  	    	printf("%d", qfileentry->attr.pagefactor);
  	    	break;
  	    case 8:			/* routing */
  	    	if( qfileentry->Routing != (char*)NULL )
  	    	    fputs_untabbed(qfileentry->Routing, stdout);
  	    	break;
  	    case 9:			/* creator */
  	        if( qfileentry->Creator != (char*)NULL )
  	            fputs_untabbed(qfileentry->Creator, stdout);
  	        break;
  	    case 10:			/* nupn */
  	    	printf("%d", qfileentry->N_Up.N);
  	    	break;
  	    case 11:			/* nupborders */
  	    	fputs( qfileentry->N_Up.borders ? "true" : "false", stdout);
  	    	break;
  	    case 12:			/* sigsheets */
  		printf("%d", qfileentry->N_Up.sigsheets);
  		break;
  	    case 13:			/* sigpart */
		fputs(describe_sigpart(qfileentry->N_Up.sigpart), stdout);
  	    	break;
  	    case 14:			/* pageorder */
  		fputs(describe_pageorder(qfileentry->attr.pageorder), stdout);
  		break;
  	    case 15:			/* proofmode */
                fputs(describe_proofmode(qfileentry->attr.proofmode), stdout);
		break; 
	    case 16:			/* priority */
	        printf("%d", qentry->priority);
	        break;
	    case 17:			/* opriority */
	        printf("%d", qfileentry->priority);
	        break;
	    case 18:			/* banner */
		fputs(describe_flag_page_setting(qfileentry->do_banner), stdout);
		break;
	    case 19:			/* trailer */
		fputs(describe_flag_page_setting(qfileentry->do_trailer), stdout);
		break;
	    case 20:			/* inputbytes */
		printf("%ld", qfileentry->attr.input_bytes);
		break;
	    case 21:			/* postscriptbytes */
		printf("%ld", qfileentry->attr.postscript_bytes);
		break;
	    case 22:			/* prolog */
		fputs( (qfileentry->attr.prolog ? "yes" : "no"), stdout);
		break;
	    case 23:			/* docsetup */
		fputs( (qfileentry->attr.docsetup ? "yes" : "no"), stdout);
		break;
	    case 24:			/* script */
		fputs( (qfileentry->attr.script ? "yes" : "no"), stdout);
		break;
	    case 25:			/* orientation */
		fputs( describe_orientation(qfileentry->attr.orientation), stdout);
		break;
	    case 26:			/* draft */
		if( qfileentry->Draft != (char*)NULL )
		    fputs_untabbed(qfileentry->Draft, stdout);
		break;
	    case 27:			/* username */
	        fputs(qfileentry->username, stdout);
	        break;
	    case 28:			/* userid */
	    	printf("%ld", (long)qfileentry->user);
	    	break;
	    case 29:			/* proxy */
		if(qfileentry->proxy_for != (char*)NULL)
		    fputs_untabbed(qfileentry->proxy_for, stdout);
		break;
	    case 30:			/* longsubtime */
		fputs(ctime((time_t*)&qfileentry->time), stdout);
		break;
	    case 31:			/* subtime */
		{
		struct tm *tm_time;
		time_t time_now;
		char timestr[10];
		tm_time = localtime((time_t*)&qfileentry->time);	/* break up queue time */
		time_now = time((time_t*)NULL);				/* get current time */
		if( difftime(time_now, (time_t)qfileentry->time) >= (24*60*60) )
		    strftime(timestr, sizeof(timestr), "%d-%b-%y", tm_time);  
		else
		    strftime(timestr, sizeof(timestr), "%I:%M%p", tm_time);  
		fputs(timestr,stdout);
		}
		break;
	    case 32:			/* pages */
		if(qfileentry->attr.pages >= 0)
		    printf("%d", qfileentry->attr.pages);
		else
		    fputs("???", stdout);
	    	break;	    
	    case 33:			/* lpqfilename */
	    	if(qfileentry->lpqFileName != (char*)NULL)
	    	    fputs_untabbed(qfileentry->lpqFileName, stdout);
		else if(qfileentry->Title != (char*)NULL)
		    fputs_untabbed(qfileentry->Title, stdout);
		break;
	    case 34:			/* totalpages */
		{
		int total = qfileentry->attr.pages;
		if(qfileentry->opts.copies > 1) total *= qfileentry->opts.copies;
		if(total >= 0)
		    printf("%d", total);
		else
		    fputs("???", stdout);
		}
		break;
	    case 35:			/* totalsides */
		{
		int total = qfileentry->attr.pages;
		if(qfileentry->opts.copies > 1) total *= qfileentry->opts.copies;
		total = (total + qfileentry->N_Up.N - 1) / qfileentry->N_Up.N;
		if(total >= 0)
		    printf("%d", total);
		else
		    fputs("???", stdout);
		}
		break;
	    case 36:			/* totalsheets */
		{
		int total = qfileentry->attr.pages;
		if(qfileentry->opts.copies > 1) total *= qfileentry->opts.copies;
		total = (total + qfileentry->attr.pagefactor - 1) / qfileentry->attr.pagefactor;
		if(total >= 0)
		    printf("%d", total);
		else
		    fputs("???", stdout);
		}
		break;
  	    case 37:			/* fulljobname */
  		printf("%s:%s-%d.%d(%s)", qfileentry->destnode, qfileentry->destname, qfileentry->id, qfileentry->subid, qfileentry->homenode);
  		break;

	    } /* end of switch */
    	} /* end of for loop */

    fputc('\n',stdout);

    return FALSE;	/* don't stop */
    } /* end of ppop_qquery_item() */
 
int ppop_qquery(char *argv[])
    {
    int x;
    char *ptr;
    int retval;

    for( x=0; x<MAX_QQUERY_ITEMS && (ptr=argv[x+1])!=(char*)NULL; x++ )
    	{
	if(strcmp(ptr,"jobname")==0)
	    qquery_query[x] = 0;
	else if(strcmp(ptr,"for")==0)
	    qquery_query[x] = 1;	    			    
	else if(strcmp(ptr,"title")==0)
	    qquery_query[x] = 2;
	else if(strcmp(ptr,"status")==0)
	    qquery_query[x] = 3;	    
	else if(strcmp(ptr,"explain")==0)
	    qquery_query[x] = 4;
	else if(strcmp(ptr,"copies")==0)
	    qquery_query[x] = 5;
	else if(strcmp(ptr,"copiescollate")==0)
	    qquery_query[x] = 6;
	else if(strcmp(ptr,"pagefactor")==0)
	    qquery_query[x] = 7;
	else if(strcmp(ptr,"routing")==0)
	    qquery_query[x] = 8;
	else if(strcmp(ptr,"creator")==0)
	    qquery_query[x] = 8;
	else if(strcmp(ptr,"nupn")==0)
	    qquery_query[x] = 10;
	else if(strcmp(ptr,"nupborders")==0)
	    qquery_query[x] = 11;
	else if(strcmp(ptr,"sigsheets")==0)
	    qquery_query[x] = 12;
	else if(strcmp(ptr,"sigpart")==0)
	    qquery_query[x] = 13;
	else if(strcmp(ptr,"pageorder")==0)
	    qquery_query[x] = 14;
	else if(strcmp(ptr,"proofmode")==0)
	    qquery_query[x] = 15;
	else if(strcmp(ptr,"priority")==0)
	    qquery_query[x] = 16;
	else if(strcmp(ptr,"opriority")==0)
	    qquery_query[x] = 17;
	else if(strcmp(ptr,"banner")==0)
	    qquery_query[x] = 18;
	else if(strcmp(ptr,"trailer")==0)
	    qquery_query[x] = 19;
	else if(strcmp(ptr,"inputbytes")==0)
	    qquery_query[x] = 20;
	else if(strcmp(ptr,"postscriptbytes")==0)
	    qquery_query[x] = 21;
	else if(strcmp(ptr,"prolog")==0)
	    qquery_query[x] = 22;
	else if(strcmp(ptr,"docsetup")==0)
	    qquery_query[x] = 23;
	else if(strcmp(ptr,"script")==0)
	    qquery_query[x] = 24;
	else if(strcmp(ptr,"orientation")==0)
	    qquery_query[x] = 25;
	else if(strcmp(ptr,"draft")==0)
	    qquery_query[x] = 26;
	else if(strcmp(ptr,"username")==0)
	    qquery_query[x] = 27;
	else if(strcmp(ptr,"userid")==0)
	    qquery_query[x] = 28;
	else if(strcmp(ptr,"proxy")==0)
	    qquery_query[x] = 29;
	else if(strcmp(ptr, "longsubtime") == 0)
	    qquery_query[x] = 30;
	else if(strcmp(ptr, "subtime") == 0)
	    qquery_query[x] = 31;
	else if(strcmp(ptr, "pages") == 0)
	    qquery_query[x] = 32;
	else if(strcmp(ptr, "lpqfilename") == 0)
	    qquery_query[x] = 33;
	else if(strcmp(ptr, "totalpages") == 0)
	    qquery_query[x] = 34;
	else if(strcmp(ptr, "totalsides") == 0)
	    qquery_query[x] = 35;
	else if(strcmp(ptr, "totalsheets") == 0)
	    qquery_query[x] = 36;
	else if(strcmp(ptr, "fulljobname") == 0)
	    qquery_query[x] = 37;


	else
	    {
	    fprintf(errors, "Unrecognized field: %s\n", ptr);
	    return EXIT_SYNTAX;
	    }
	    
	argv[x+1] = (char*)NULL;	/* remove it so as not to disturb custom_list() */
    	}

    qquery_query_count = x;

    retval = custom_list(argv, ppop_qquery_help, ppop_qquery_banner, ppop_qquery_item, FALSE, A_switch_value);

    return retval;
    } /* end of ppop_qquery() */

/*===================================================================
** ppop progress {job}
**
** Display the progress of a job.  This command is intended for use
** by frontends.  It emmits three integers separated by tabs.
** They are the percentage of the job's bytes transmitted, the number
** of "%%Page:" comments sent, and the number of pages which the
** printer claims to have printed.  (Many printers make no such claims.
** Currently this is only supported for HP PJL printers.)
===================================================================*/
static void ppop_progress_help(void) { }

static void ppop_progress_banner(void) { }

static int ppop_progress_item(struct QEntry *qentry,
	struct QFileEntry *qfileentry, 
	char *onprinter,
	FILE *qstream)
    {
    char line[MAX_QFLINE+2];
    int found = FALSE;
    long bytes_sent;
    int pages_started;
    int pages_printed;

    while( fgets(line, sizeof(line), qstream) != (char*)NULL && strcmp(line, QF_ENDTAG1) )
	{
	if (ppr_sscanf(line, "Progress: %ld %d %d", &bytes_sent, &pages_started, &pages_printed) == 3 )
	    found = TRUE;
	}    

    if(found)		/* If "Progress: line was found, */
	{
	printf("%d%%\t%d\t%d\n",
	    (int)((bytes_sent*(long)100) / qfileentry->attr.postscript_bytes),
	    pages_started, pages_printed);
	}

    return FALSE;	/* don't stop */
    } /* end of ppop_progress() */

int ppop_progress(char *argv[])
    {
    return custom_list(argv, ppop_progress_help, ppop_progress_banner, ppop_progress_item, FALSE, -1);
    } /* end of ppop_progress() */
    
/* end of file */
