/*
** ~ppr/src/ppop/ppop.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 20 February 1997.
*/

/*
** Operators utility for PostScript page printers.  It allows the queue
** to be viewed, jobs to be canceled, jobs to be held, printers to
** be started and stopped, and much more.
*/

#include "global_defines.h"
#include "global_structs.h"
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include "pprd.h"
#include "ppop.h"
#include "util_exits.h"
#include "version.h"

/*
** Things for communicating with the spooler.
*/
pid_t pid;                      /* this process id */

/*
** User identification information.
*/
static uid_t uid;				/* user id of invoking user */
static uid_t euid;				/* effective uid (should always be ppr) */
static uid_t as_uid;				/* uid whose jobs we can manipulate */
static const char *as_uid_name;			/* -u switch value */
static const char *proxy_for = (char*)NULL;	/* -X switch value */
int A_switch_value = -1;

/*
** Output formating information.
*/
int machine_readable = FALSE;	/* machine readable mode */
FILE *errors = stderr;		/* file we should send stderr type messages to */

/*
** Exit codes verious library routines should use
** if there are fatal errors.
*/
const int lib_memory_fatal = EXIT_INTERNAL;
const int lib_misc_fatal = EXIT_INTERNAL;

/*
** Function for breaking up a queue file name into its constituent parts.
** If it is not a valid job name return -1, else return 0.
*/
int parse_job_name(struct Jobname *job, const char *jobname)
    {
    int len;
    const char *ptr;
    int c;

    /* Set pointer to start of job name. */
    ptr = jobname;

    /* If a destination node name is specified, then claim it. */
    if( (size_t)(len=strcspn(ptr,":")) != strlen(ptr) )
	{
	if( len > MAX_NODENAME )
	    {
	    fputs("Destionation node name is too long.\n", errors);
	    return -1;
	    }
	    
	strncpy(job->destnode,jobname,len);
	job->destnode[len] = (char)NULL;
	ptr += (len+1);
	}
    else		/* otherwise, assume the destination node is this node */
    	{
    	strcpy(job->destnode, ppr_get_nodename());
    	}

    /*
    ** Find the full extent of the destination name.  Keep in mind that
    ** embedded hyphens are now allowed.
    */
    len = 0;
    while(TRUE)
	{
	len += strcspn(&ptr[len], "-");

	if( ptr[len] == (char)NULL )
	    break;

	if( (c=ptr[len+1+strspn(&ptr[len+1], "0123456789")]) == '.' || c == '(' || c==(char)NULL )
	    break;

	len++;	
	}

    /*
    ** If the job name we found above is too long
    ** or it does not have a hyphen after it, this is
    ** a syntax error.
    */
    if( len > MAX_DESTNAME )
	{
	fputs("Destination name is too long.\n", errors);
	return -1;
	}

    /*
    ** Copy the printer or group name into the structure.
    */
    strncpy(job->destname, ptr, len);
    job->destname[len] = (char)NULL;
    ptr += len;

    /*
    ** If there is a hyphen, read the jobid after the hyphen.
    */
    if( *ptr == '-' )
	{
	ptr++;
	job->id = atoi(ptr);
	ptr += strspn(ptr, "0123456789");

	/*
	** If a dot comes next, read the number after it;
	** otherwise, use the subid zero.
	*/
	if( *ptr == '.' )
	    {
	    ptr++;
	    job->subid = atoi(ptr);
	    ptr += strspn(ptr, "0123456789");
	    }
	else
	    {
	    job->subid = 0;
	    }
	}

    /*
    ** Since there was no id number specified, 
    ** use the wildcard numbers for id number
    ** and subid number.
    */
    else
    	{
    	job->id = -1;
    	job->subid = -1;
    	}

    /*
    ** If a left parenthesis comes next, read the sending node name.
    */
    if( *ptr == '(' )
    	{
    	ptr++;
    	if( (len=strcspn(ptr, ")")) > MAX_NODENAME || ptr[len] != ')' )
    	    return -1;
	strncpy(job->homenode, ptr, len);
	job->homenode[len] = (char)NULL;
	ptr += len;
	ptr++;
	}
    else
	{
	strcpy(job->homenode,ppr_get_nodename());
	}

    /*
    ** If there is anything left, it is an error.
    */
    if( *ptr )
    	return -1;

    return 0;                           /* indicate read ok */
    } /* end of parse_job_name() */

/*
** Parse a printer or group name.
*/
int parse_dest_name(struct Destname *dest, const char *destname)
    {
    int len;
    const char *ptr;

    ptr = destname;
    
    if( (size_t)(len=strcspn(ptr,":")) < strlen(ptr) )
	{
	if( len > MAX_NODENAME )
	    {
	    fputs("Node name is too long.\n", errors);
	    return -1;
	    }
	strncpy(dest->destnode,ptr,len);
	dest->destnode[len] = (char)NULL;
	ptr += len;
	ptr++;
	}
    else
    	{
    	strcpy(dest->destnode,ppr_get_nodename());
    	}
    
    if( strlen(ptr) > MAX_DESTNAME )
    	{
    	fputs("Destination name is too long.\n", errors);
    	return -1;
    	}
    	
    if(strpbrk(ptr, DEST_DISALLOWED) != (char*)NULL)
    	{
    	fputs("Destination name contains a disallowed character.\n", errors);
    	return -1;
    	}

    if(strchr(DEST_DISALLOWED_LEADING, (int)ptr[0]) != (char*)NULL)
    	{
    	fputs("Destination name begins with a disallowed character.\n", errors);
    	return -1;
    	}

    strcpy(dest->destname,ptr);
    
    return 0;
    } /* end of parse_dest_name() */

/*
** Handle fatal errors.
** Print a message and exit.
*/
void fatal(int exitval, const char message[], ... )
    {
    va_list va;

    va_start(va, message);
    fputs("Fatal: ", errors);
    vfprintf(errors, message, va);
    fputc('\n', errors);
    va_end(va);

    exit(exitval);
    } /* end of fatal() */

/*
** Non-fatal errors.
** (This function is called by the library
** function read_queue_file().)
*/
void error(const char message[], ... )
    {
    va_list va;

    va_start(va, message);
    fprintf(errors, "Error: ");
    vfprintf(errors, message,va);
    fprintf(errors, "\n");
    va_end(va);
    } /* end of error() */

/*======================================================================
** IPC functions
======================================================================*/

static sigset_t oset;			/* old signal set */
static FILE *LFIFO = (FILE*)NULL;	/* channel to pprd */
static FILE *RFIFO = (FILE*)NULL;	/* channel to rpprd */
static FILE *FIFO;			/* channel to pprd or rpprd */
static int sigcaught = FALSE;		/* flag which is set when SIGUSR1 received */
static char temp_file_name[MAX_PATH];	/* name of temporary file we get answer in */
static FILE *reply_file;		/* streams library thing for open reply file */

/*
** Handler for signal from pprd (user signal 1).
** We receive this signal when pprd wants us to know
** that the reply file is ready.
*/
void user_sighandler(int sig)
    {
    sigcaught = TRUE;
    }

/*
** This SIGALRM handler is used to limit the time we will wait
** for SIGUSR1 from pprd or rpprd.
*/
void alarm_sighandler(int sig)
    {
    fatal(EXIT_INTERNAL, "Timeout waiting for response");
    }
    
/*
** Get ready to communicate with a spooler daemon.
** If the nodename is our nodename, we will communicate
** with pprd, otherwise we will be communicating with
** rpprd.
**
** Get ready for a SIGUSR1 from pprd.
**
** To do this, we clear the signal received flag
** and block the signal.  (We will unblock it with
** sigsuspend().
*/
FILE *get_ready(const char *nodename)
    {
    int fifo;
    sigset_t set;		/* storage for set containing SIGUSR1 */

    if( strcmp(nodename, ppr_get_nodename() ) == 0 )
    	{
    	if( LFIFO == (FILE*)NULL )
    	    {
	    if( (fifo=open(FIFO_NAME, O_WRONLY | O_NDELAY)) == -1 )
		fatal(EXIT_NOSPOOLER, "can't open FIFO, pprd is probably not running.");
	    LFIFO = fdopen(fifo, "w");
	    }
	FIFO = LFIFO;
	}
    else
	{
	if( RFIFO == (FILE*)NULL )
	    {
	    if( (fifo=open(FIFO_NAME_RPPRD, O_WRONLY | O_NDELAY)) == -1 )
	    	fatal(EXIT_NOSPOOLER, "can't open FIFO, rpprd is probably not running.");
	     RFIFO = fdopen(fifo, "w");
	     }
	FIFO = RFIFO;
	}
	
    sigcaught = FALSE;

    sigemptyset(&set);
    sigaddset(&set, SIGUSR1);
    sigprocmask(SIG_BLOCK, &set, &oset);

    return FIFO;
    } /* end of get_ready() */

/*
** Wait to receive SIGUSR1 from pprd.
** (This function uses oset which is defined above.)
** If pprd reports an error, we return a NULL file
** pointer and the caller must call print_reply();
** otherwise we will return a pointer to the reply file.
*/
FILE *wait_for_pprd(int do_timeout)
    {
    struct stat buf;

    /* Start a timeout period */
    if(do_timeout)
	alarm(60);

    /* wait for pprd to fill the file */
    while(sigcaught == 0)
        sigsuspend(&oset);

    /* Cancel the timeout */
    if(do_timeout)
	alarm(0);

    /* restore the old signal mask */
    sigprocmask(SIG_SETMASK, &oset, (sigset_t*)NULL);

    /* Stat the file to get the setting of the other execute bit: */
    if(stat(temp_file_name, &buf))
	{
	fprintf(errors, "Can't stat reply file \"%s\", errno=%d %s\n", temp_file_name, errno, strerror(errno));
	reply_file = (FILE*)NULL;
    	return (FILE*)NULL;		/* things are bad */
	}
    
    if( (reply_file=fopen(temp_file_name, "r")) == (FILE*)NULL )
        {
        fprintf(errors, "Couldn't open reply file \"%s\", errno=%d (%s)\n", temp_file_name, errno, strerror(errno) );
	reply_file = (FILE*)NULL;
        return (FILE*)NULL;
        }

    unlink(temp_file_name);		/* now that file is open, we can dispense with the name */

    /* Other execute indicates data: */
    return (buf.st_mode & S_IXOTH) ? reply_file : (FILE*)NULL ;
    } /* end of wait_for_pprd() */

/*
** Print a plain text reply from pprd.
** Return the value at the start of the plain text reply.
** If an internal error occurs, return -1.
*/
int print_reply(void)
    {
    int retcode = 0;				/* the number we will read */
    int c;					/* one character from reply file */

    if(reply_file == (FILE*)NULL)
	return EXIT_INTERNAL;

    while( isdigit((c=fgetc(reply_file))) )	/* convert ASCII decimal */
	{					/* number to an int */
	retcode *= 10;
	retcode += (c-'0');
	}

    while( (c=fgetc(reply_file)) != EOF )	/* print the rest */
	fputc(c,stdout);			/* of the text */

    fclose(reply_file);

    return retcode;				/* return the code from the reply */
    } /* end of print_reply() */

/*====================================================================
** Security functions
**==================================================================*/

/*
** Function to return true if the current user is a privledged
** user.  A privledged user is defined as "root", "ppr", a user
** or a member of the group called "ppop".  The first time this
** routine is called, it will cache the answer.
**
** A privledged user may start and stop printers, cancel other
** people's jobs, and mount media.
*/
int privledged(void)
    {
    static int answer = -1;	/* -1 means undetermined */
    static uid_t last_as_uid = -1;
    
    if(answer == -1 || as_uid != last_as_uid)
    	{			/* if undetermined, */
	last_as_uid = as_uid;
	answer = FALSE;

	/* Of course, "ppr" is privledged, as is uid 0 (root). */
	if( as_uid == 0 || as_uid == euid )
	    {
	    answer = TRUE;
	    }

	/* As are all users who are members of the group "ppop". */
	else		
	    {
	    struct passwd *pw;
    	    struct group *gr;
    	    int x;
	    char *ptr;

	    if( (pw=getpwuid(as_uid)) == (struct passwd *)NULL )
	    	{
	    	fprintf(errors,"privledeged(): getpwuid(%ld) failed, errno=%d (%s)\n", (long)as_uid, errno, strerror(errno));
	    	exit(EXIT_INTERNAL);
	    	}

	    if( (gr=getgrnam(GROUP_PPOP)) != (struct group *)NULL )
	        {
	        x=0;
    	        while( (ptr=gr->gr_mem[x++]) != (char*)NULL )
    	    	    {
    	    	    if( strcmp(ptr,pw->pw_name)==0 )
    	    	        {
    	    	        answer = TRUE;
    	    	        break;
    	    	        }
    	            }
    	        }
    	    }
    	} /* end of if answer not determined yet */

    return answer;
    } /* end of privledged() */

/*
** Return zero if the user has the operator privledge,
** if not, print an error message and return non-zero.
**
** Perhaps this is this reverse logic?
*/
int assert_am_operator(void)
    {
    if( privledged() )
	{
	return 0;
	}
    else
	{
	fputs("You are not allowed to perform the requested\n"
            "operation because you are not an operator.\n",errors);
	return -1;
	}
    } /* end of assert_am_operator() */

/*
** Return non-zero if the indicated job does not belong to the user 
** and the user is not an operator.  If non-zero is returned, also 
** prints an error message.
**
** If the user is a proxy (the -X switch has been used) then the proxy
** id's must match or the proxy id must be "*" on the same machine
** as the job's proxy id.  If the -X switch is used then even operators
** cannot delete jobs that were not submitted under their Unix uid.
*/
int job_permission_check(struct Jobname *job)
    {
    char fname[MAX_PATH];
    FILE *f;
    char temp[256];
    long int job_uid=-1;
    char job_user_name[9];
    char job_proxy_for[127];

    /*
    ** Operators are privledged and "own" all jobs unless they use the
    ** -X switch to say they are acting as proxies or the -u switch
    ** to get the job permissions of another user.
    **
    ** So, unless an operator is trying to do something fancy we
    ** can stop right here.
    */
    if(privledged() && proxy_for == (char*)NULL)
        return 0;

    /*
    ** Open the queue file.  If we wan't open it,
    ** just say it is ok to manipulate the job.
    ** Somebody will eventually figure it out.
    */
    sprintf(fname, "%s/%s:%s-%d.%d(%s)", QUEUEDIR, job->destnode, job->destname, job->id, job->subid, job->homenode);
    if( (f=fopen(fname,"r")) == (FILE*)NULL )
	return 0;
  
    /* Read the queue file and find the "User:" line. */
    job_proxy_for[0] = (char)NULL;
    while( fgets(temp,sizeof(temp),f) != (char*)NULL )
        {
        if( ppr_sscanf(temp, "User: %ld %#s %#s",
        	&job_uid,
        	sizeof(job_user_name), job_user_name,
        	sizeof(job_proxy_for), job_proxy_for) >= 2 )
            {
	    break;
            }
        }
    fclose(f);

    /* Check to see that we got a "User:" line: */
    if(job_uid == -1)
    	{
    	fprintf(errors, "Queue file error, no \"User:\" line.\n");
	return -1;    
	}

    /*
    ** If you (a Unix user) didn't submit it, you can't modify it.
    ** You can't do it even if you are an operator if you have used
    ** the -u or -X switch.
    **
    ** If this is a proxy job we will mention the fact since
    ** some people will not notice it otherwise and be confused.
    */
    if( as_uid != job_uid )
	{
	if( as_uid != uid )		/* did you forget you used the -u switch? */
	    {
	    fprintf(errors, "You may not perform this operation because the job \"%s\"\n"
		"does not belong to \"%s\".\n",
			remote_jobid(job->destnode, job->destname, job->id, job->subid, job->homenode),
	    as_uid_name);
	    }
	else
	    {
	    fprintf(errors, "You may not perform this operation because \"%s\"\n"
		"is not your job and you are not an operator.\n",
			remote_jobid(job->destnode, job->destname, job->id, job->subid, job->homenode));
	    }

	/* This might help explain things: */
	if(job_proxy_for[0] != (char)NULL)
	    {
	    fprintf(errors,"\nThe user \"%s\" submitted %s as proxy\n"
	    	"for \"%s\".\n",
			job_user_name,
			remote_jobid(job->destnode, job->destname, job->id, job->subid, job->homenode),
			job_proxy_for);
	    }

        return -1;
        }

    /*
    ** If the user who invoked ppop provided an -X switch
    ** and it does not match the one which was provided when
    ** submitting the job, then refuse to do it.
    */
    if(proxy_for != (char*)NULL)		/* if -X switch used, */
	{
	char *host;
	int len;

	if(job_proxy_for[0] == (char)NULL)	/* not a proxy job */
	    {
	    fprintf(errors,"You (\"%s\") may not perform this operation on\n"
		"job \"%s\" because it was not submited from your network node.\n",
		proxy_for, remote_jobid(job->destnode, job->destname, job->id, job->subid, job->homenode) );
	    
	    return -1;	    
	    }
	if( strcmp(job_proxy_for, proxy_for)		/* don't match exactly */
		&& ( strcmp(proxy_for, "*@*")
			|| strchr(proxy_for, '@') == (char*)NULL )
		&& ( strncmp(proxy_for, "*@", 2)
			|| (host=strchr(job_proxy_for, '@')) == (char*)NULL
			|| strcmp(&proxy_for[1], host) )
		&& ( (len=strcspn(proxy_for, "@")) != strcspn(job_proxy_for, "@")
			|| strncmp(proxy_for, job_proxy_for, len)
			|| strcmp(&proxy_for[len+1], "*") ) )
	    {
	    fprintf(errors,"You (\"%s\") may not perform this operation on\n"
		"job \"%s\" which was submitted by \"%s\".\n",
		proxy_for, remote_jobid(job->destnode, job->destname, job->id, job->subid, job->homenode), job_proxy_for);

	    return -1;
	    }
    	}

    /*
    ** All tests passed.  You may do what you want with this job.
    */
    return 0;
    } /* end of job_permission_check() */

/*
** Return 0 if the job matches the current user id and proxy_for.
** Return non-zero if it does not.  Notice that operator
** privledge plays no part in this.  Other than that, the
** criteria should be the same as for job_permission_check().
**
** The global variables as_uid and proxy_for are used by
** this function.
*/
int is_my_job(struct QEntry *qentry, struct QFileEntry *qfileentry)
    {
    if( as_uid != qfileentry->user )			/* user must match */
        return FALSE;

    if( proxy_for != (char*)NULL )			/* if user is now a proxy, */
	{
	char *host;
	int len;

	if( qfileentry->proxy_for == (char*)NULL )	/* if not a proxy job, */
	    return FALSE;	    

	if( strcmp(qfileentry->proxy_for, proxy_for)		/* don't match exactly */
		&& ( strcmp(proxy_for, "*@*")
			|| strchr(proxy_for, '@') == (char*)NULL )
		&& ( strncmp(proxy_for, "*@", 2)
			|| (host=strchr(qfileentry->proxy_for, '@')) == (char*)NULL
			|| strcmp(&proxy_for[1], host) )
		&& ( (len=strcspn(proxy_for, "@")) != strcspn(qfileentry->proxy_for, "@")
			|| strncmp(proxy_for, qfileentry->proxy_for, len)
			|| strcmp(&proxy_for[len+1], "*") ) )
	    return FALSE;
    	}

    return TRUE;
    } /* end of is_my_job() */

/*=========================================================================
** Main function and associated functions, dispatch command
** to proper command handler.
=========================================================================*/
int main_help(void)
	{
	fputs("\nDestination commands:\n"
        	"\tppop dest[ination] {<destination>, all}\n"
        	"\tppop ldest {<destination>, all}\n"
        	"\tppop accept <destination>\n"
        	"\tppop reject <destination>\n",stdout);

        fputs("\nPrint job commands:\n"
		"\tppop list {<destination>, <job>, all} ...\n"
		"\tppop short {<destination>, <job>, all} ...\n"
		"\tppop details {<destination>, <job>, all} ...\n"
		"\tppop lpq {<destination>, <job>, all} [<user>] [<id>] ...\n"
		"\tppop qquery {<destination>, <job>, all} <field name 1> ...\n"
		"\tppop move {<job>, <old_destination>} <new_destination>\n"
		"\tppop hold <job> ...\n"
		"\tppop release <job> ...\n"
		"\tppop [s]cancel {<job>, <destination>, all} ...\n"
		"\tppop [s]purge {<destination>, all} ...\n"
		"\tppop [s]cancel-active {<destination>, all} ...\n"
		"\tppop [s]cancel-my-active {<destination>, all} ...\n"
		"\tppop clean <destination> ...\n"
		"\tppop rush <job> ...\n"
		"\tppop log <job>\n"
		"\tppop progress <job>\n", stdout);

	fputs("\nPrinter commands:\n"
		"\tppop status {<destination>, all} ...\n"
		"\tppop start <printer> ...\n"
		"\tppop halt <printer> ...\n"
		"\tppop stop <printer> ...\n"
		"\tppop wstop <printer>\n"
		"\tppop message <printer>\n"
		"\tppop alerts <printer>\n", stdout);

	fputs("\nMedia commands:\n"
		"\tppop media {<destination>, all}\n"
		"\tppop mount <printer> <bin> <media>\n", stdout);

	return EXIT_OK;
	} /* end of main_help() */
        
/*
** Examine the command and run the correct proceedure.
** Return the value returned by the proceedure function.
** If the command is unknown, return -1.
*/
int dispatch(char *argv[])
    {
    /*
    ** Since these are used by other programs we
    ** put them first so they will be the fastest.
    */
    if(strcmp(argv[0],"qquery")==0)
	return ppop_qquery(&argv[1]);
    if(strcmp(argv[0],"message")==0)
    	return ppop_message(&argv[1]);
    if(strcmp(argv[0],"progress")==0)
    	return ppop_progress(&argv[1]);

    if(strcmp(argv[0],"short")==0)
        return ppop_short(&argv[1]);
    else if(strcmp(argv[0],"list")==0)
        return ppop_list(&argv[1], 0);
    else if(strcmp(argv[0],"nhlist")==0)	/* For lprsrv */
        return ppop_list(&argv[1], 1);
    else if(icmp(argv[0],"lpq")==0)
    	return ppop_lpq(&argv[1]);
    else if(strcmp(argv[0],"mount")==0)
        return ppop_mount(&argv[1]);
    else if(strcmp(argv[0],"media")==0)
        return ppop_media(&argv[1]);
    else if(strcmp(argv[0],"start")==0)
        return ppop_start_stop_wstop_halt(&argv[1], 0);
    else if(strcmp(argv[0],"stop")==0)
        return ppop_start_stop_wstop_halt(&argv[1], 1);
    else if(strcmp(argv[0],"wstop")==0)
        return ppop_start_stop_wstop_halt(&argv[1], 2);
    else if(strcmp(argv[0], "halt") == 0)
        return ppop_start_stop_wstop_halt(&argv[1], 3);
    else if(strcmp(argv[0], "cancel") == 0)
        return ppop_cancel(&argv[1], 1);
    else if(strcmp(argv[0], "scancel") == 0)
        return ppop_cancel(&argv[1], 0);
    else if(strcmp(argv[0], "purge") == 0)
        return ppop_purge(&argv[1], 1);
    else if(strcmp(argv[0], "spurge") == 0)
        return ppop_purge(&argv[1], 0);
    else if(strcmp(argv[0], "clean") == 0)
        return ppop_clean(&argv[1]);
    else if(strcmp(argv[0], "cancel-active") == 0)
    	return ppop_cancel_active(&argv[1], FALSE, 1);
    else if(strcmp(argv[0], "scancel-active") == 0)
    	return ppop_cancel_active(&argv[1], FALSE, 0);
    else if(strcmp(argv[0], "cancel-my-active") == 0)
    	return ppop_cancel_active(&argv[1], TRUE, 1);
    else if(strcmp(argv[0], "scancel-my-active") == 0)
    	return ppop_cancel_active(&argv[1], TRUE, 0);
    else if(strcmp(argv[0], "move") == 0)
        return ppop_move(&argv[1]);
    else if(strcmp(argv[0],"rush")==0)
        return ppop_rush(&argv[1]);
    else if(strcmp(argv[0],"hold")==0)
        return ppop_hold_release(&argv[1], FALSE);
    else if(strcmp(argv[0],"release")==0)
        return ppop_hold_release(&argv[1], TRUE);
    else if(strcmp(argv[0],"status")==0)
        return ppop_status(&argv[1]);
    else if(strcmp(argv[0],"accept")==0)
        return ppop_accept_reject(&argv[1], FALSE);
    else if(strcmp(argv[0],"reject")==0)
        return ppop_accept_reject(&argv[1], TRUE);
    else if((strcmp(argv[0],"destination")==0) || (strcmp(argv[0],"dest")==0))
        return ppop_destination(&argv[1],FALSE);
    else if((strcmp(argv[0],"ldest")==0))
        return ppop_destination(&argv[1],TRUE);
    else if(strcmp(argv[0],"details")==0)
        return ppop_details(&argv[1]);
    else if(strcmp(argv[0],"alerts")==0)
        return ppop_alerts(&argv[1]);
    else if(strcmp(argv[0],"log")==0)
        return ppop_log(&argv[1]);
    else if(strcmp(argv[0],"help")==0)
	return main_help();
    else
        return -1;			/* return `dispatcher failed' code */
    } /* end of dispatch() */

/*
** interactive mode function
** Return the result code of the last command executed.
**
** In interactive mode, we present a prompt, read command
** lines, and execute them.
*/
int interactive_mode(void)
    {
    char *ar[16];		/* argument vector constructed from line[] */
    char *ptr;			/* used to parse arguments */
    unsigned int x;		/* used to parse arguments */
    int errorlevel=0;		/* return value from last command */

    if( ! machine_readable )	/* If a human will be reading our output, */
	{
	puts("PPOP, Page Printer Operator's utility");
	puts(VERSION);
	puts(COPYRIGHT);
	puts(AUTHOR);
	puts("\n\"help\" for command list,");
	puts("\"exit\" to quit.\n");
	}
    else			/* If a machine will be reading our output, */
	{
    	puts("*READY\t"SHORT_VERSION);
    	fflush(stdout);
    	}

    /* 
    ** Read input lines until end of file.
    ** 
    ** Notice that the prompt printed when in machine readable
    ** mode is blank.  Alos notice that we do not explicitly 
    ** flush stdout and yet the prompt is printed even though
    ** it does not end with a line feed.  This is mysterious.
    */
    while( (ptr=ppr_get_command("ppop>", machine_readable)) != (char*)NULL )
	{
	for(x=0; x<sizeof(ar); x++)	/* parse into ar[] */
	    {
	    ar[x]=strtok(ptr," \t\n");	/* get next token */
	    if(ar[x]==(char*)NULL)	/* stop if no new token */
	    	break;
	    ptr = (char*)NULL;		/* clear ptr so as not to reset strtok() */
	    }
	
	/*
	** The variable x will be an index into ar[] which will
	** indicate the first element that has any significance.
	** If the line begins with the word "ppop" will will
	** increment x.
	*/
	x = 0;
	if( ar[0]!=(char*)NULL && strcmp(ar[0],"ppop")==0 )
	    x = 1;
	
	/*
	** If no tokens remain in this command line, 
	** go on to the next command line.
	*/
	if(ar[x]==(char*)NULL)
	    continue;

	/*
	** If the command is "exit", break out of
	** the line reading loop.
	*/
	if(strcmp(ar[x],"exit")==0 || strcmp(ar[x],"quit")==0)
	    break;

	/*
	** Call the dispatch() function to execute the command.  If the
	** command is not recognized, dispatch() will return -1.  In that
	** case we print a helpful message and change the errorlevel to
	** zero since -1 is not a valid exit code for a program.
	*/
	if( (errorlevel=dispatch(&ar[x])) == -1 )
	    {
	    if( ! machine_readable )			/* A human gets english */
		puts("Try \"help\" or \"exit\".");
	    else					/* A program gets a code */
	    	puts("*UNKNOWN");

	    errorlevel=0;
	    }
	else if(machine_readable)		/* If a program is reading our output, */
	    {					/* say the command is done */
	    printf("*DONE\t%d\n", errorlevel);	/* and tell the exit code. */
	    }

	if(machine_readable)			/* If stdout is a pipe as seems likely */
	    fflush(stdout);			/* when -M is used, we must flush it. */
    	} /* While not end of file */

    return errorlevel;			/* return result of last command (not counting exit) */
    } /* end of interactive_mode() */
    
/*
** Handler for sigpipe.
*/
void pipe_sighandler(int sig)
    {
    fputs("Spooler has shut down.\n", errors);

    if(machine_readable)
    	fprintf(errors,"*DONE %d\n", EXIT_NOSPOOLER);

    exit(EXIT_NOSPOOLER);
    } /* end of pipe_sighandler() */

/*
** Print help.
*/
void help(FILE *outfile)
    {
    fputs("Valid switches:\n"
		"\t-X <principal string>\n"
		"\t--proxy-for=<principal string>\n"
		"\t-M\n"
		"\t--machine-readable\n"
		"\t--version\n"
		"\t-u <user>\n"
		"\t--as-user=<user>\n"
		"\t-A <seconds>\n"
		"\t--arrest-interest-time=<seconds>\n"
		"\t--help\n"
		"\nTry \"ppop help\" for help with subcommands.\n", outfile);
    } /* end of help() */			

/*
** Command line options:
*/
static const char *option_chars = "X:Mu:A:";
static const struct ppr_getopt_opt option_words[] =
	{
	{"proxy-for", 'X', TRUE},
	{"machine-readable", 'M', FALSE},
	{"as-user", 'u', TRUE},
	{"arrest-interest-interval", 'A', TRUE},
	{"help", 1000, FALSE},
	{"version", 1001, FALSE},
	{(char*)NULL, 0, FALSE}
	} ;

/*
** main function
** If there is a command on the invokation line, execute it,
** otherwise, go into interactive mode
*/
int main(int argc, char *argv[])
    {
    int result;         /* value to return */
    int optchar;	/* for getopt() */
    struct ppr_getopt_state getopt_state;

    /*
    ** Get the uid and euid and stash them away.
    ** They are later used to determine priveledges.
    */
    as_uid = uid = getuid();
    euid = geteuid();

    /* Parse the options. */
    ppr_getopt_init(&getopt_state, argc, argv, option_chars, option_words);
    while( (optchar=ppr_getopt(&getopt_state)) != -1 )
    	{
    	switch(optchar)
    	    {
    	    case 'X':			/* -X or --proxy-for */
		proxy_for = getopt_state.optarg;
    	    	break;

	    case 'M':			/* -M or --machine-readable */
	    	machine_readable = TRUE;
		errors = stdout;
		break;

	    case 'u':			/* -u or --as-user */
		if( ! privledged() )
		    {
		    fprintf(errors, "You are not allowed to use %s.\n", getopt_state.name);
		    exit(EXIT_DENIED);
		    }
		else
		    {
		    struct passwd *pw;
		    if( (pw=getpwnam(getopt_state.optarg)) == (struct passwd *)NULL )
			{
			fprintf(errors, "The user \"%s\" does not exist.\n", getopt_state.optarg);
			exit(EXIT_NOTFOUND);
			}
		    as_uid = pw->pw_uid;
		    as_uid_name = getopt_state.optarg;
		    }
		break;

	    case 'A':			/* -A or --arrest-interest-interval */
		A_switch_value = atoi(getopt_state.optarg);
		break;

	    case 1000:			/* --help */
	    	help(stdout);
	    	exit(EXIT_OK);

	    case 1001:			/* --version */
		if(machine_readable)
		    {
		    puts(SHORT_VERSION);
		    }
		else
		    {
		    puts(VERSION);
		    puts(COPYRIGHT);
		    puts(AUTHOR);
		    }
	    	exit(EXIT_OK);

	    case '?':			/* help or unrecognized switch */
		fprintf(errors, "Unrecognized switch: %s\n\n", getopt_state.name);
		help(errors);
		exit(EXIT_SYNTAX);

	    case ':':			/* argument required */
	    	fprintf(errors, "The %s option requires an argument.\n", getopt_state.name);
		exit(EXIT_SYNTAX);

	    case '!':			/* bad aggreation */
	    	fprintf(errors, "Switches, such as %s, which take an argument must stand alone.\n", getopt_state.name);
	    	exit(EXIT_SYNTAX);
		
	    case '-':			/* spurious argument */
	    	fprintf(errors, "The %s switch does not take an argument.\n", getopt_state.name);
	    	exit(EXIT_SYNTAX);

	    default:			/* missing case */
	    	fprintf(errors, "Missing case %d in switch dispatch switch()\n", optchar);
	    	exit(EXIT_INTERNAL);
		break;
    	    }
    	}

    /* Change to the home directory of PPR. */
    chdir(HOMEDIR);

    /*
    ** Determine and store our process id.
    ** It is part of the temporary file name.
    */
    pid = getpid();

    /*
    ** Install a SIGPIPE handler so we can produce an
    ** intelligible message when we try to run a command
    ** on a spooler which has shut down.
    */
    signal(SIGPIPE, pipe_sighandler);

    /*
    ** Install handler for SIGUSR1.  PPRD will send
    ** us SIGUSR1 when the reply is ready.
    */
    signal(SIGUSR1, user_sighandler);

    /*
    ** This handles timeouts waiting for SIGUSR1.
    */
    signal(SIGALRM, alarm_sighandler);

    /*
    ** Construct the name of the temporary file.
    */
    sprintf(temp_file_name, TEMPDIR"/ppop%ld", (long)pid);

    /*
    ** If no subcommand, go interactive, otherwise, execute commmand.
    */
    if( argc == getopt_state.optind )                    
	{
        result = interactive_mode();
        }
    else
	{
	if( (result=dispatch(&argv[getopt_state.optind])) == -1 )
	    fprintf(errors,"Unknown command, try \"ppop help\".\n");
	}

    /* Clean up by closing the FIFO. */
    fclose(FIFO);

    /* Exit with the result of the last command. */
    return result;
    } /* end of main() */

/* end of file */
