/*
** ~ppr/src/ppad/ppad_printer.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 27 February 1997.
*/

/*
** This module is part of the printer administrator's utility.
** It contains the code to implement those sub-commands which
** manipulate printers.
*/

#include "global_defines.h"
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <dirent.h>
#include "util_exits.h"
#include "interface.h"
#include "interfaces.h"
#include "ppad.h"

/*
** Send the spooler a command to re-read a printer definition.
*/
void reread_printer(char *printer)
    {
    write_fifo("NP %s\n", printer);
    } /* end of rearead_printer() */

/*
** Update the "DefFiltOpts:" lines of
** any groups which have this printer as a member.
**
** We must call this if we change the PPD file or change any
** of the "PPDOpts:" lines.
*/
int update_groups_deffiltopts(char *printer)
    {
    DIR *dir;
    struct dirent *direntp;
    int len;
    int is_a_member;

    if(debug_level > 0)
	printf("Updating \"DefFiltOpts:\" for groups of which \"%s\" is a member.\n", printer);

    if( (dir=opendir(GRCONF))==(DIR*)NULL )
    	{
    	fputs("ppad_printer.c: update_groups_deffiltopts(): opendir() failed\n", errors);
    	return EXIT_INTERNAL;
    	}
    	
    while( (direntp=readdir(dir)) != (struct dirent*)NULL )
    	{
    	if( direntp->d_name[0] == '.' )		/* skip . and .. */
	    continue;				/* and temporary files */    	

	len=strlen(direntp->d_name);		/* Emacs style backup files */
	if( len > 0 && direntp->d_name[len-1]=='~' )
	    continue;				/* should be skipt. */

	is_a_member = FALSE;			/* start by assuming it is not */

	if( grpopen(direntp->d_name, FALSE) )	/* open the group file for read */
	    {
	    fprintf(errors, "ppad_printer.c: update_groups_deffiltopts(): grpopen(\"%s\", FALSE) failed", direntp->d_name);
	    closedir(dir);
	    return EXIT_INTERNAL;
	    }	    

	while(confread())			/* read until end of file */
	    {
	    if(strncmp(confline, "Printer: ",9) == 0)
	    	{
	    	if(strcmp(&confline[9], printer) == 0)
		    {
	    	    is_a_member = TRUE;
		    break;
	    	    }
	    	}	    
	    }

	confclose();				/* close the group file */

	/*
	** If membership was detected, then call the routine in 
	** ppad_group.c which group_deffiltopts() calls.
	*/
	if(is_a_member)
	    {
	    if(debug_level > 0)
	    	printf("  Updating \"DefFiltOpts:\" for group \"%s\".\n", direntp->d_name);
	    _group_deffiltopts(direntp->d_name);
	    }
	}

    closedir(dir);

    return EXIT_OK;
    } /* end of update_groups_deffiltopts() */

/*
** Read the word provided and return BANNER_* 
*/
int flag_code(char *option)
    {
    if(icmp(option,"never")==0)
    	return BANNER_FORBIDDEN;
    if(icmp(option,"no")==0)
    	return BANNER_DISCOURAGED;
    if(icmp(option,"yes")==0)
    	return BANNER_ENCOURAGED;
    if(icmp(option,"always")==0)
    	return BANNER_REQUIRED;
    return BANNER_INVALID;    		/* no match */
    } /* end of flag_code() */

/*
** Take a BANNER_* value and return a description
*/
char *flag_description(int code)
    {
    switch(code)
    	{
    	case BANNER_FORBIDDEN:
    	    return "never";
    	case BANNER_DISCOURAGED:
    	    return "no";
    	case BANNER_ENCOURAGED:
    	    return "yes";
    	case BANNER_REQUIRED:
    	    return "always";
    	default:
    	    return "<invalid>";
    	}
    } /* end of flag_description() */

char *long_flag_description(int code)
    {
    switch(code)
    	{
    	case BANNER_FORBIDDEN:
    	    return "forbidden";
    	case BANNER_DISCOURAGED:
    	    return "discouraged";
    	case BANNER_ENCOURAGED:
    	    return "encouraged";
    	case BANNER_REQUIRED:
    	    return "required";
    	default:
    	    return "<invalid>";
    	}
    } /* end of flag_description() */

char *jobbreak_description(int jobbreak)
    {
    switch(jobbreak)
	{
	case JOBBREAK_NONE:
	    return "none";
	case JOBBREAK_SIGNAL:
	    return "signal";
	case JOBBREAK_CONTROL_D:
	    return "control-d";
    	case JOBBREAK_PJL:
    	    return "pjl";
    	case JOBBREAK_SIGNAL_PJL:
	    return "signal/pjl";
	case JOBBREAK_SAVE_RESTORE:
	    return "save/restore";
	case JOBBREAK_NEWINTERFACE:
	    return "newinterface";
	case JOBBREAK_TBCP:
	    return "tbcp";
    	default:
    	    return "<invalid>";
	}
    } /* end of jobbreak_description() */

char *outputorder_description(int outputorder)
    {
    switch(outputorder)
    	{
    	case 0:
    	    return "ppd";
    	case -1:
    	    return "Reverse";
    	case 1:
    	    return "Normal";
    	default:
    	    return "<invalid>";
    	}
    } /* end of outputorder_description() */    

/*
** Read a printer's configuration file and
** print a report.
** It is not necessary to be an operator in order to 
** execute this command.
*/
int printer_show(char *argv[])
    {
    char *printer=argv[0];		/* Argument is printer to show. */
    int x;
    char *ptr;

    char *interface = (char*)NULL;
    char *address = (char*)NULL;
    char *options = (char*)NULL;
    int jobbreak = JOBBREAK_CONTROL_D;
    int feedback = FALSE;
    char *PPDFile = (char*)NULL;
    char *comment = (char*)NULL;
    int bincount = 0;
    char *bins[MAX_BINS];
    int banner = BANNER_DISCOURAGED;
    int trailer = BANNER_DISCOURAGED;
    int alert_frequency = 0;
    char *alert_method = (char*)NULL;
    char *alert_address = (char*)NULL;
    int charge_duplex_sheet = -1;	/* charge in cents */
    int charge_simplex_sheet = -1;
    int outputorder = 0;		/* unknown outputorder */
    char *deffiltopts = (char*)NULL;
    char *switchset = (char*)NULL;
    char PPDOpts[1024] = { (char)NULL };
    
    if(printer==(char*)NULL)
    	{
    	fputs("You must specify a printer\n",errors);
    	return EXIT_SYNTAX;
    	}
	
    if(prnopen(printer,FALSE))	/* if printer does not exist, */
	{
	fprintf(errors,"The printer \"%s\" does not exist.\n",printer);
	return EXIT_BADDEST;
	}
	    
    while(confread())		/* read each configuration file line */
	{
	if( ppr_sscanf(confline,"Interface: %Z",&ptr) == 1 )
	    {
	    int index;

	    if(interface!=(char*)NULL) myfree(interface);
	    interface = ptr;

	    /* Set global default jobbreak and feedback. */
	    jobbreak = JOBBREAK_CONTROL_D;
	    feedback = FALSE;

	    /* Look in the table to see if there is a default for this interface. */
	    for(index=0;interfaces[index].name!=(char*)NULL;index++)
		{
		if(strcmp(interface,interfaces[index].name)==0)
		    {
		    jobbreak = interfaces[index].jobbreak;
		    feedback = interfaces[index].feedback;
		    break;
		    }
		}
	    }
	else if(strncmp(confline,"Address: ",9)==0)
	    {
	    if(address != (char*)NULL) myfree(address);	/* discard any preceeding address lines */

	    ptr = &confline[9+strspn(&confline[9]," \t")];	/* eat up whitespace */

	    if(*ptr == '"')				/* if quoted */
	    	{
		ptr++;
		ptr[strcspn(ptr,"\"")] = (char)NULL;	/* use until quote */
	    	}

	    address = mystrdup(ptr);
	    }
	else if( ppr_sscanf(confline,"Options: %Z",&ptr) == 1 )
	    {
	    if(options!=(char*)NULL) myfree(options);
	    options = ptr;
	    }
	else if( ppr_sscanf(confline,"JobBreak: %d",&jobbreak) == 1 )
	    {
	    /* no code */
	    }
	else if( strncmp(confline,"Feedback: ",10) == 0 )
	    {
	    feedback = torf(&confline[10]);
	    }
	else if( ppr_sscanf(confline,"PPDFile: %Z",&ptr) == 1 )
	    {
	    if(PPDFile!=(char*)NULL) myfree(PPDFile);
	    PPDFile = ptr;
	    }
	else if( ppr_sscanf(confline,"Bin: %Z",&ptr) == 1 )
	    {
	    if(bincount<MAX_BINS)
	    	bins[bincount++] = ptr;
	    }
	else if(ppr_sscanf(confline, "Comment: %Z", &ptr) == 1)
	    {
	    if(comment != (char*)NULL) myfree(comment);
	    comment = ptr;
	    }
	else if(strncmp(confline, "FlagPages: ", 7) == 0)
	    {
	    sscanf(confline, "FlagPages: %d %d", &banner, &trailer);	    
	    }
	else if(strncmp(confline, "Alert: ", 7) == 0)
	    {
	    int x=7;
	    int len;

	    if(alert_method!=(char*)NULL) myfree(alert_method);
	    if(alert_address!=(char*)NULL) myfree(alert_address);

	    x+=strspn(&confline[x]," \t");		/* eat up space */
	    alert_frequency=atoi(&confline[x]);

            x+=strspn(&confline[x]," \t-0123456789");   /* skip spaces and */
                                                        /* digits */
            len=strcspn(&confline[x]," \t");            /* get word length */
            alert_method = (char*)myalloc(len+1,sizeof(char));
            strncpy(alert_method,&confline[x],len);     /* copy */
            alert_method[len]=(char)NULL;		/* terminate */
            x+=len;                                     /* move past word */
            x+=strspn(&confline[x]," \t");              /* skip spaces */

            len=strcspn(&confline[x]," \t\n");          /* get length */
            alert_address = (char*)myalloc(len+1,sizeof(char));
            strncpy(alert_address,&confline[x],len);	/* copy */
            alert_address[len]=(char)NULL;		/* terminate */
	    }
	else if( strncmp(confline,"OutputOrder: ",13) == 0 )
	    {
	    ptr = &confline[13+strspn(&confline[13]," \t")];

	    if(strcmp(ptr,"Normal")==0)
	    	outputorder = 1;
	    else if(strcmp(ptr,"Reverse")==0)
	        outputorder = -1;
	    else
		outputorder = -100;
	    }
	else if(strncmp(confline, "Charge: ", 8) == 0)
	    {
	    float x1, x2;
	    int count;
	    
	    count = sscanf(confline, "Charge: %f %f", &x1, &x2);
	    charge_duplex_sheet = (int)(x1 * 100.0);
    	    if(count == 2)
    	    	charge_simplex_sheet = (int)(x2 * 100.0);
	    }
	else if(ppr_sscanf(confline, "Switchset: %Z", &ptr) == 1)
	    {
	    if(switchset!=(char*)NULL) myfree(switchset);
	    switchset = ptr;
	    }
	else if(ppr_sscanf(confline, "DefFiltOpts: %Z", &ptr) == 1)
	    {
	    if(deffiltopts!=(char*)NULL) myfree(deffiltopts);
	    deffiltopts = ptr;
	    }
	else if(strncmp(confline, "PPDOpt:", 7) == 0)
	    {
	    if( (strlen(PPDOpts) + strlen(confline) + 2) > (sizeof(PPDOpts)/sizeof(char)) )
		{
		fputs("ppad_printer.c: printer_show(): PPDOpts overflow.\n",errors);
		return EXIT_INTERNAL;
		}

	    strcat(PPDOpts, confline);
	    strcat(PPDOpts, "\n");
	    }

	} /* end of loop for each configuration file line */
	    
    confclose();
		
    /*
    ** Now that we have retrieved the information, we will print it.
    */
    if( ! machine_readable )
    {
    printf("Printer name: %s\n",printer);
    printf("Comment: %s\n",comment!=(char*)NULL?comment:"<none>");
    printf("Interface: %s\n", interface != (char*)NULL ? interface : "<undefined>");
    printf("Address: \"%s\"\n", address != (char*)NULL ? address : "<undefined>");
    printf("Options: %s\n", options != (char*)NULL ? options : "");
    printf("JobBreak: %s\n", jobbreak_description(jobbreak));
    printf("Feedback: %s\n", feedback ? "True" : "False");
    printf("PPDFile: %s\n", PPDFile != (char*)NULL ? PPDFile : "<undefined>");
    printf("Flags: %s %s (banners %s, trailers %s)\n",
    	flag_description(banner),
    	flag_description(trailer),
    	long_flag_description(banner),
    	long_flag_description(trailer));

    printf("Bins: ");			/* print a list of all */
    for(x=0; x < bincount; x++)		/* the bins we found */
    	{
    	if(x==0)
    	    printf("%s",bins[x]);
    	else
    	    printf(", %s",bins[x]);

	myfree(bins[x]);
    	}
    putc('\n',stdout);
    
    printf("OutputOrder: %s\n", outputorder_description(outputorder));

    if(charge_duplex_sheet == -1)
    	puts("Charge: none");
    else if(charge_simplex_sheet == -1)
    	printf("Charge: %s per sheet\n", money(charge_duplex_sheet));
    else
    	{   	/* money() uses static array for result */
    	printf("Charge: %s per duplex sheet, ", money(charge_duplex_sheet));
    	printf("%s per simplex sheet\n", money(charge_simplex_sheet));
    	}

    switch(alert_frequency)
    	{
	case 0:
    	    puts("Alert frequency: 0 (never send alerts)");
    	    break;
	case 1:
	    puts("Alert frequency: 1 (send alert on every error)");
	    break;
	case -1:
	    printf("Alert frequency: -1 (send alert on first error, send notice on recovery)\n");
	    break;
    	default:
	    if(alert_frequency > 0)
    	    	printf("Alert frequency: %d (send alert every %d errors)\n",alert_frequency,alert_frequency);
	    else
    	    	printf("Alert frequency: %d (send alert after %d errors, send notice on recovery)\n",alert_frequency,(alert_frequency * -1));
    	    break;
	}
	    	
    printf("Alert method: %s\n",alert_method!=(char*)NULL?alert_method:"none");
    printf("Alert address: %s\n",alert_address!=(char*)NULL?alert_address:"none");    	

    print_deffiltopts(deffiltopts);

    print_switchset(switchset);

    fputs(PPDOpts,stdout);
    }

    /* Machine readable output. */
    else
    {
    printf("name\t%s\n",printer);
    printf("comment\t%s\n", comment!=(char*)NULL ? comment : "");
    printf("interface\t%s\n", interface!=(char*)NULL ? interface : "<undefined>");
    printf("address\t%s\n", address!=(char*)NULL ? address : "<undefined>");
    printf("options\t%s\n", options!=(char*)NULL ? options : "");
    printf("jobbreak\t%s\n",jobbreak_description(jobbreak));
    printf("feedback\t%s\n", feedback ? "true" : "false");
    printf("ppdfile\t%s\n", PPDFile!=(char*)NULL ? PPDFile : "<undefined>");
    printf("banner\t%s\n",flag_description(banner));
    printf("trailer\t%s\n",flag_description(trailer));

    fputs("bins\t",stdout);		/* print a list of all */
    for(x=0; x < bincount; x++)		/* the bins we found */
    	{
    	if(x==0)
    	    printf("%s",bins[x]);
    	else
    	    printf(" %s",bins[x]);

	myfree(bins[x]);
    	}
    putc('\n',stdout);
    
    printf("outputorder\t%s\n",outputorder_description(outputorder));
    if(charge_duplex_sheet == -1)
    	{
    	fputs("charge\tnone\n", stdout);
    	}
    else
    	{
    	printf("change\t%s", money(charge_duplex_sheet));
    	if(charge_simplex_sheet != -1)
    	    printf("\t%s", money(charge_simplex_sheet));
    	fputc('\n', stdout);
    	}
    printf("alert_frequency\t%d\n",alert_frequency);	    	
    printf("alert_method\t%s\n",alert_method!=(char*)NULL?alert_method:"none");
    printf("alert_address\t%s\n",alert_address!=(char*)NULL?alert_address:"none");    	
    printf("deffiltopts\t%s\n",deffiltopts!=(char*)NULL ? deffiltopts : "");
    printf("switchset\t%s\n",switchset!=(char*)NULL ? switchset : "");
    } /* end of machine_readable */

    /* Free some memory which we allocated: */
    if(interface!=(char*)NULL) myfree(interface);
    if(address!=(char*)NULL) myfree(address);
    if(options!=(char*)NULL) myfree(options);
    if(comment!=(char*)NULL) myfree(comment);
    if(PPDFile!=(char*)NULL) myfree(PPDFile);
    if(alert_method!=(char*)NULL) myfree(alert_method);
    if(alert_address!=(char*)NULL) myfree(alert_address);

    return EXIT_OK;
    } /* end of printer_show() */

/*
** Set a printer's comment.
*/
int printer_comment(char *argv[])
    {
    char *printer=argv[0];
    char *comment=argv[1];

    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (comment==(char*)NULL) )
    	{
	fputs("You must supply the name of an existing printer and\n"
		"a comment to attach to it.\n",errors);
    	return EXIT_SYNTAX;
    	}

    /* make sure the printer exists */
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Comment: ",9)==0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}
    confwrite("Comment: %s\n",comment);
    while(confread())				/* copy rest of file, */
	{					/* deleting further */
	if(strncmp(confline,"Comment: ",9)==0)	/* "Comment:" lines. */
	    continue;
	else
	    confwrite("%s\n",confline);
    	}
    confclose();

    return EXIT_OK;    
    } /* end of printer_comment() */

/*
** Set a printer's interface and the interface's address.
** We will delete "JobBreak:" and "Feedback:" lines if the 
** interface is changed or if there was previously no "Interface:"
** line.  We will also delete all "JobBreak:" and "Feedback:" lines
** which appear before the "Interface:" line.
**
** We will only ask the spooler to re-read the configuration file
** if this command creates the printer.
*/
int printer_interface(char *argv[])
    {
    char *printer=argv[0];
    char *interface=argv[1];
    char *address=argv[2];
    char fname[MAX_PATH];
    int int_changed=FALSE;
    int c;
    struct stat statbuf;
    
    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (interface==(char*)NULL) || (address==(char*)NULL) )
	{
	fputs("You must specify a printer, either new or existing,\n"
		"an interface, and an address for the interface\n"
		"to send the job to.\n",errors);
	return EXIT_SYNTAX;
	}

    if(strlen(printer) > MAX_DESTNAME)
    	{
    	fputs("The printer name is too long.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(strpbrk(printer, DEST_DISALLOWED) != (char*)NULL)
    	{
    	fputs("Printer name contains a disallowed character.\n", errors);
    	return EXIT_SYNTAX;
    	}

    if(strchr(DEST_DISALLOWED_LEADING, (int)printer[0]) != (char*)NULL)
    	{
    	fputs("Printer name begins with a disallowed character.\n", errors);
    	return EXIT_SYNTAX;
    	}

    /* Make sure the interface exists. */
    sprintf(fname,"%s/%s",INTDIR,interface);
    if( stat(fname,&statbuf) || ! (statbuf.st_mode & S_IXUSR) )
    	{
    	fprintf(errors,"The interface \"%s\" does not exist or is not executable.\n",interface);
    	return EXIT_NOTFOUND;
    	}

    /* if new printer */
    if( prnopen(printer,TRUE) )		/* try to open printer config for write */
    	{				/* if we fail, create a new printer */
	FILE *newconf;
	FILE *defaults;
	char fname[MAX_PATH];
	
	sprintf(fname,"%s/%s",PRCONF,printer);		/* create a config file */
	if( (newconf=fopen(fname,"w")) == (FILE*)NULL )
	    {
	    fprintf(errors,"Failed to create printer config file \"%s\".\n", fname);
	    return EXIT_INTERNAL;
	    }
	    
	fprintf(newconf, "Interface: %s\n", interface);		/* write the interface */
	fprintf(newconf, "Address: \"%s\"\n", address);
	fprintf(newconf, "PPDFile: Apple LaserWriter Plus\n");	/* specify a good generic PPD file */
	fprintf(newconf, "DefFiltOpts: level=1 colour=False resolution=300 freevm=172872 mfmode=CanonCX\n");

	if( (defaults=fopen(NEWPRN_CONFIG,"r")) != (FILE*)NULL )
	    {
	    while( (c=fgetc(defaults)) != -1 )
		fputc(c,newconf);	    
	    fclose(defaults);	    
	    }
	    
    	fclose(newconf);
	reread_printer(printer);	/* tell pprd to re-read the configuration */
    	}				/* (really for the 1st time) */
    
    /* if old printer */
    else
        {
	while(confread())	/* copy up to 1st "Interface:" or */
	    {			/* "Address:" line */
	    if( (strncmp(confline,"Interface: ",11)==0)
			|| (strncmp(confline,"Address: ",9)==0) )
		{
		break;
		}
	    else		/* deleting "JobBreak:" and */
		{		/* "Feedback:" lines as we go */
		if( strncmp(confline,"JobBreak: ",10)
			&& strncmp(confline,"Feedback: ",10)
			&& strncmp(confline,"Options: ",9) )
		    {
	            confwrite("%s\n",confline);
	            }
	        }
	    }

	/* If the new interface name is different, set int_changed to */
	/* non-zero, zero if it has not changed. */
	if( ppr_sscanf(confline,"Interface: %#s",sizeof(fname),fname) )
	    {
	    int_changed=strcmp(interface,fname);	    
	    }

	confwrite("Interface: %s\n",interface);	/* place new lines */
	confwrite("Address: \"%s\"\n",address);

	while(confread())		/* and copy rest of old file, */
	    {				/* removing more old lines */
	    if( strncmp(confline,"Interface: ",11)
	    		&& strncmp(confline,"Address: ",9)
	    		&& (strncmp(confline,"JobBreak: ",10) || !int_changed)
	    		&& (strncmp(confline,"Feedback: ",10) || !int_changed)
	    		&& (strncmp(confline,"Options: ",9) || !int_changed) )
	    	confwrite("%s\n",confline);
	    }
	confclose();
        }

    return EXIT_OK;
    } /* end of printer_interface() */
    
/*
** Set a printer's interface options string.
*/
int printer_options(char *argv[])
    {
    char *printer = argv[0];
    char *options = argv[1];

    if( am_operator() )
    	return EXIT_DENIED;

    if(printer == (char*)NULL || options == (char*)NULL)
	{
	fputs("Insufficient parameters.\n", errors);
	return EXIT_SYNTAX;
	}

    /* make sure the printer exists */
    if(prnopen(printer, TRUE))
    	{
    	fprintf(errors, "Printer \"%s\" does not exist.\n", printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline, "Options:", 8) == 0)	/* Delete "Options: " lines before */
	    continue;					/* the "Interface: " line. */
	if(strncmp(confline,"Interface:", 10) == 0)	/* Break loop after copying */
	    {						/* the "Interface: " line */
	    confwrite("%s\n", confline);
	    break;
	    }
	confwrite("%s\n", confline);
    	}

    /* If we have a meaningful new line to write, write it now. */
    if(strcmp(options, "none") && options[0])
	confwrite("Options: %s\n", options);

    while(confread())				/* copy rest of file, */
	{
	if(strncmp(confline, "Options:", 8) == 0)
	    continue;
	else
	    confwrite("%s\n", confline);
    	}
    confclose();

    return EXIT_OK;
    } /* end of printer_options() */

/*
** Set a printer's jobbreak flag.
*/
static void printer_jobbreak_help(void)
    {
    fputs("You must supply the name of an existing printer and a jobbreak mode setting.\n"
	"Valid jobbreak modes are \"signal\", \"control-d\", \"pjl\", \"signal/pjl\",\n"
	"\"save/restore\", \"newinterface\", \"tbcp\", and \"default\".\n",errors);
    } /* end of printer_jobbreak_help() */
    
int printer_jobbreak(char *argv[])
    {
    char *printer=argv[0];
    int newstate;

    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (argv[1]==(char*)NULL) )
	{
	printer_jobbreak_help();
    	return EXIT_SYNTAX;
	}

    if(icmp(argv[1],"none")==0)
	newstate = JOBBREAK_NONE;
    else if(icmp(argv[1],"signal")==0)
    	newstate = JOBBREAK_SIGNAL;
    else if(icmp(argv[1],"control-d")==0)
    	newstate = JOBBREAK_CONTROL_D;
    else if(icmp(argv[1],"pjl")==0)
    	newstate = JOBBREAK_PJL;
    else if(icmp(argv[1],"signal/pjl")==0)
    	newstate = JOBBREAK_SIGNAL_PJL;
    else if(icmp(argv[1],"save/restore")==0)
    	newstate = JOBBREAK_SAVE_RESTORE;
    else if(icmp(argv[1],"newinterface")==0)
    	newstate = JOBBREAK_NEWINTERFACE;
    else if(icmp(argv[1],"default")==0)
    	newstate = JOBBREAK_DEFAULT;
    else if(icmp(argv[1],"tbcp")==0)
    	newstate = JOBBREAK_TBCP;
    else
    	{
	printer_jobbreak_help();
    	return EXIT_SYNTAX;
    	}
    
    /* Make sure the printer exists. */
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"JobBreak:", 9) == 0)	/* delete "JobBreak:" lines */
	    continue;
	if(strncmp(confline,"Interface:", 10) == 0)	/* stop after "Interface:" line */
	    {
	    confwrite("%s\n", confline);
	    break;
	    }    
	confwrite("%s\n", confline);
    	}

    /*
    ** If the new jobbreak setting is not "default",
    ** write a new "JobBreak:" line.
    */
    if(newstate != JOBBREAK_DEFAULT)
	confwrite("JobBreak: %d\n", newstate);

    while(confread())				/* copy rest of file, */
	{
	if(strncmp(confline,"JobBreak:", 9) == 0) /* deleting any further "JobBreak: " lines */
	    continue;
	else
	    confwrite("%s\n", confline);
    	}
    confclose();
    
    return EXIT_OK;
    } /* end of printer_jobbreak() */
    
/*
** Set a printer's feedback flag.
*/
int printer_feedback(char *argv[])
    {
    char *printer = argv[0];
    int newstate;

    if( am_operator() )
    	return EXIT_DENIED;

    if( printer==(char*)NULL || argv[1]==(char*)NULL 
	|| ( (newstate=torf(argv[1]))==ANSWER_UNKNOWN && icmp(argv[1],"default") ) )
	{
	fputs("You must supply the name of an existing printer\nand \"true\", \"false\", or \"default\".\n",errors);
	return EXIT_SYNTAX;
	}

    /* make sure the printer exists */
    if(prnopen(printer, TRUE))
    	{
    	fprintf(errors, "Printer \"%s\" does not exist.\n", printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Feedback:", 9) == 0)	/* delete "Feedback:" lines */
	    continue;
	if(strncmp(confline,"Interface:", 10) == 0)	/* stop after copying the */
	    {						/* "Interface:" line */
	    confwrite("%s\n",confline);
	    break;
	    }
	confwrite("%s\n", confline);
    	}

    /* If the new feedback setting is not "default", */
    /* write a new "Feedback:" line.                 */
    if( newstate != ANSWER_UNKNOWN )
	confwrite("Feedback: %s\n", newstate ? "True" : "False");

    while(confread())				/* copy rest of file, */
	{
	if(strncmp(confline, "Feedback:", 9) == 0)
	    continue;
	else
	    confwrite("%s\n", confline);
    	}
    confclose();

    return EXIT_OK;    
    } /* end of printer_feedback() */

/* 
** Set a printer's PPD file.
**
** We do this by modifying the configuration file's 
** "PPDFile:" line.  Since the spooler does not cache
** this information, there is no need to tell it that
** we have done so.
*/
int printer_ppd(char *argv[])
    {
    char *printer = argv[0];
    char *ppdname = argv[1];
    char ppdfname[MAX_PATH];
    FILE *testopen;
    int retval;
    
    if( am_operator() )
    	return EXIT_DENIED;

    if(printer == (char*)NULL || ppdname == (char*)NULL)
    	{
    	fputs("You must supply the name of an existing printer and a PPD file.\n", errors);
    	return EXIT_SYNTAX;
    	}

    /*
    ** Make sure the printer exists,
    ** opening its configuration file if it does.
    */
    if(prnopen(printer, TRUE))
    	{
    	fprintf(errors, "Printer \"%s\" does not exist.\n", printer);
    	return EXIT_BADDEST;
    	}

    /* make sure the PPD file exists */
    if(ppdname[0]=='/')
    	strcpy(ppdfname, ppdname);
    else
	sprintf(ppdfname, "%s/%s", PPDDIR, ppdname);

    if( (testopen=fopen(ppdfname,"r")) == (FILE*)NULL )
    	{
    	fprintf(errors, "PPD file \"%s\" does not exist.\n", ppdname);
	confabort();
    	return EXIT_NOTFOUND;
    	}
    else
    	{
    	fclose(testopen);
    	}

    /* Get ready to collect information for a "DefFiltOpts:" line. */
    deffiltopts_open();

    /* Consider the printer's PPD file.  If this fails, stop. */
    if( (retval=deffiltopts_add_ppd(ppdname, (char*)NULL)) != EXIT_OK )
    	{
    	confabort();
    	return retval;
    	}

    /*
    ** Modify the printer's configuration file. 
    **
    ** First, copy up to the first "PPDFile:" lines, deleting
    ** "DefFiltOpts:" lines as we go.
    */
    while(confread())
    	{
	if(strncmp(confline,"PPDFile: ", 9) == 0)	/* stop at */
	    break;

	if(strncmp(confline,"DefFiltOpts: ", 13) == 0)	/* delete */
	    continue;

	if(strncmp(confline,"PPDOpt: ", 8) == 0)	/* delete */
	    continue;

	confwrite("%s\n",confline);
    	}

    /* Write the new "PPDFile:" lines. */
    confwrite("PPDFile: %s\n", ppdname);

    /*
    ** Copy the rest of the file, deleting "DefFiltOpts:" lines
    ** and extra "PPDFile:" lines.
    */
    while(confread())
    	{
	if(strncmp(confline, "PPDFile:", 6) == 0)	/* delete */
	    continue;

	if(strncmp(confline, "DefFiltOpts:", 12) == 0)	/* delete */
	    continue;

	if(strncmp(confline, "PPDOpt:", 7) == 0)	/* delete */
	    continue;

	confwrite("%s\n", confline);
    	}

    /* Insert a new "DefFiltOpts:" line. */
    confwrite("DefFiltOpts: %s\n", deffiltopts_line() );

    /* Free any remaining deffiltopts data space. */
    deffiltopts_close();

    /* Close the printer configuration file. */
    confclose();

    /* Update any groups which have this printer as a member. */
    return update_groups_deffiltopts(printer);
    } /* end of printer_ppd() */
    
/*
** Change the configuration file "Alert: " lines which
** tells where to send alerts and how often.
*/
int printer_alerts(char *argv[])
    {
    char *printer=argv[0];
    int frequency;
    char *method=argv[2];
    char *address=argv[3];

    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (argv[1]==(char*)NULL)
		|| (strspn(argv[1],"-0123456789")!=strlen(argv[1]))
    		|| (method==(char*)NULL) || (address==(char*)NULL) )
	{
	fputs("You must supply the name of a printer, an alert frequency, such as \"7\",\n"
    		"an alert method, such as \"mail\", and an alert address,\n"
    		"such as \"alertreaders\".\n",errors);
	return EXIT_SYNTAX;
	}

    frequency=atoi(argv[1]);
    
    /* make sure the printer exists */
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Alert: ",7)==0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}

    confwrite("Alert: %d %s %s\n",frequency,method,address);

    while(confread())				/* copy rest of file, */
	{					/* deleting further */
	if(strncmp(confline,"Alert: ",7)==0)	/* "Alert:" lines. */
	    continue;
	else
	    confwrite("%s\n",confline);
    	}
    confclose();

    reread_printer(printer);	/* tell pprd to re-read the configuration */

    return EXIT_OK;
    } /* end of printer_alert() */
   
/*
** Change the configuration file "Alert:" line to change the
** alert frequency.
** The absence of an  "Alert:" line is an error.
*/
int printer_frequency(char *argv[])
    {
    char *printer;
    int frequency;			/* dispatch every frequency alerts */
    char *method=(char*)NULL;		/* by method */
    char *address=(char*)NULL;		/* to address */
    
    if( am_operator() )
    	return EXIT_DENIED;

    if( (argv[0]==(char*)NULL) || (argv[1]==(char*)NULL) )
	{
	fputs("You must specify a printer and a new alert frequency.\n",errors);
	return EXIT_SYNTAX;
	}    
    
    printer = argv[0];
    frequency = atoi(argv[1]);
    
    /* make sure the printer exists */
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors, "Printer \"%s\" does not exist.\n", printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Alert: ",7)==0)
	    {
	    int x=7;	/* 7 is length of "Alert: " */
	    int len;

            x+=strspn(&confline[x]," \t0123456789");    /* skip spaces and */
                                                        /* digits */
            len=strcspn(&confline[x]," \t");            /* get word length */
	    method = mystrndup(&confline[x], len);
            x += len;					/* move past word */
            x += strspn(&confline[x]," \t");		/* skip spaces */

            len = strcspn(&confline[x]," \t\n");	/* get length */
	    address = mystrndup(&confline[x],len);

	    break;
	    }
	else
	    {
	    confwrite("%s\n",confline);
	    }
    	}

    if( (method==(char*)NULL) || (address==(char*)NULL) )
    	{
    	confabort();
	fputs("No alert method and address defined, use \"ppad alerts\".\n",errors);
    	return EXIT_NOTFOUND;
	}

    confwrite("Alert: %d %s %s\n",frequency,method,address);

    while(confread())				/* copy rest of file, */
	{					/* deleting further */
	if(strncmp(confline,"Alert: ",7)==0)	/* "Alert:" lines. */
	    continue;
	else
	    confwrite("%s\n",confline);
    	}

    confclose();		/* put new file into place */

    reread_printer(printer);	/* tell pprd to re-read the configuration */

    return EXIT_OK;
    } /* end of printer_frequency() */

/*
** Change the configuration file "FlagPages:" line which
** tells whether or not to print banner and trailer pages.
*/
int printer_flags(char *argv[])
    {
    char *printer=argv[0];
    int banner;
    #ifdef GNUC_HAPPY
    int trailer=0;
    #else
    int trailer;
    #endif

    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (argv[1]==(char*)NULL) || (argv[2]==(char*)NULL) )
	{
	fputs("You must supply the name of an existing printer, a new banner\n"
    		"option, and a new trailer option.  Valid banner and trailer\n"
    		"options are \"never\", \"no\", \"yes\", and \"always\".\n",errors);
	return EXIT_SYNTAX;
	}

    if( ((banner=flag_code(argv[1]))==BANNER_INVALID)
    		|| ((trailer=flag_code(argv[2]))==BANNER_INVALID) )
    	{
    	fputs("Banner and trailer must be set to \"never\", \"no\",\n"
		"\"yes\", or \"always\".\n",errors);
	return EXIT_SYNTAX;
	}    	

    /* make sure the printer exists */
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"FlagPages: ",11)==0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}

    confwrite("FlagPages: %d %d\n",banner,trailer);

    while(confread())				/* copy rest of file, */
	{
	if(strncmp(confline,"FlagPages: ",11)==0)
	    continue;
	else
	    confwrite("%s\n",confline);
    	}

    confclose();

    /* no need to reread_printer(printer) */

    return EXIT_OK;
    } /* end of printer_flags() */
    
/*
** Change the configuration "OutputOrder:" line.
** The direction may be "Normal", "Reverse" or "ppd".
** If the direction is "ppd", the "OutputOrder:" line is deleted 
** from the configuration file.
*/
int printer_outputorder(char *argv[])
    {
    char *printer=argv[0];
    int newstate;

    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (argv[1]==(char*)NULL) )
    	{
    	fputs("You must supply the name of an existing printer and\n"
		"a new outputorder.\n",errors);
    	return EXIT_SYNTAX;
    	}

    if(icmp(argv[1],"Normal")==0)
    	newstate=1;
    else if(icmp(argv[1],"Reverse")==0)
        newstate=-1;
    else if(icmp(argv[1],"PPD")==0)
    	newstate=0;
    else
        {
	fputs("Set outputorder to \"Normal\", \"Reverse\", or \"PPD\".\n",errors);
        return EXIT_SYNTAX;
        }
    
    /* make sure the printer exists */
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"OutputOrder: ",13)==0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}

    switch(newstate)		/* if newstate is zero, line will */
    	{			/* not be replaced */
    	case -1:
    	   confwrite("OutputOrder: Reverse\n");
    	   break;
        case 1:
           confwrite("OutputOrder: Normal\n");
           break;
        }

    while(confread())				/* copy rest of file, */
	{
	if(strncmp(confline,"OutputOrder: ",13)==0)
	    continue;
	else
	    confwrite("%s\n", confline);
    	}
    confclose();

    /* no need to reread_printer(printer) */

    return EXIT_OK;
    } /* end of printer_direction() */
    
/*
** Change the charge made for printing.
**
** !!! BUG BUG BUG !!!
** We get the printer configuration re-read, but
** we really ought to get the group configurations
** re-read too.
*/
int printer_charge(char *argv[])
    {
    char *printer = argv[0];
    int newcharge_duplex_sheet;
    int newcharge_simplex_sheet;

    if( am_operator() )
    	return EXIT_DENIED;

    if(printer == (char*)NULL || argv[1] == (char*)NULL)
    	{
    	fputs("Insufficient parameters.  You must supply the name of a printer\n"
    		"and an amount to change per sheet.  If you wish you may specify\n"
    		"an amount to charge per duplex sheet and then an amount to charge\n"
    		"per simplex sheet.  If you specify only one amount, it will apply\n"
    		"to both.  To remove a printer charge, set the charge to \"none\".\n", errors);
    	return EXIT_SYNTAX;
    	}

    if(icmp(argv[1], "none") == 0)
    	{
        newcharge_duplex_sheet = newcharge_simplex_sheet = -1;
        }
    else
        {
        newcharge_duplex_sheet = (int)(getdouble(argv[1]) * 100.0);
    		/* truncate to int */

    	if(argv[2] != (char*)NULL)
    	    newcharge_simplex_sheet = (int)(getdouble(argv[2]) * 100.0);
    	else
    	    newcharge_simplex_sheet = -1;
    	}

    /* Make sure the printer exists. */
    if(prnopen(printer, TRUE))
    	{
    	fprintf(errors, "Printer \"%s\" does not exist.\n", printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline, "Charge:", 7) == 0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}

    if(newcharge_duplex_sheet != -1)
    	{
    	if(newcharge_simplex_sheet != -1)
            confwrite("Charge: %.2f %.2f\n", (double)newcharge_duplex_sheet/100.0, (double)newcharge_simplex_sheet/100.0);
    	else
            confwrite("Charge: %.2f\n", (double)newcharge_duplex_sheet/100.0);
        }

    while(confread())				/* copy rest of file, */
	{
	if(strncmp(confline, "Charge:", 7) == 0)
	    continue;
	else
	    confwrite("%s\n", confline);
    	}

    confclose();

    reread_printer(printer);

    return EXIT_OK;    
    } /* end of printer_charge() */
    
/*
** Extract the printer bins list from the PPD file 
** and put it in the configuration file.
**
** This will overwrite any bin list in the configuration file.
*/
int printer_bins_ppd(char *argv[])
    {
    char *printer=argv[0];		/* name of printer whose configuration should be changed */
    char *ppdname=(char*)NULL;
    char *ppdline;			/* a line read from the PPD file */
    int x;
    int ret;

    if( am_operator() )
    	return EXIT_DENIED;

    if( printer==(char*)NULL )
    	{
    	fputs("You must supply the name of an existing printer.\n",errors);
    	return EXIT_SYNTAX;
    	}

    /* make sure the printer exists */
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())				/* copy all lines */
    	{
	if(strncmp(confline,"PPDFile:",8)==0)
	    {
	    ppdname = &confline[8+strspn(&confline[8]," \t")];
	    ppdname = mystrndup(ppdname, strcspn(ppdname," \t\n"));
	    }

	if(strncmp(confline,"Bin:",4)==0)	/* don't copy those */
	    continue;				/* that begin with "Bin: " */
	else
	    confwrite("%s\n",confline);
    	}

    if(ppdname == (char*)NULL)
    	{
    	fputs("Printer configuration file does not have a \"PPDFile:\" line.\n", errors);
    	confabort();
    	return EXIT_NOTFOUND;
    	}
    
    if( (ret=ppd_open(ppdname, errors)) )
        {
        confabort();
        return ret;
        }

    while( (ppdline=ppd_readline()) != (char*)NULL )
	{
	if(strncmp(ppdline,"*InputSlot",10)==0)	/* Use only "*InputSlot" */
	    {
	    x = 10;
	    x += strspn(&ppdline[x]," \t");	/* Skip to start of options keyword */
	
	    ppdline[x+strcspn(&ppdline[x],":/")] = (char)NULL;
						/* terminate at start of translation or code */
	    confwrite("Bin: %s\n", &ppdline[x]);
	    }
	}

    confclose();

    /* Instruct pprd to reread the printer configuration file. */
    reread_printer(printer);

    return EXIT_OK;
    } /* end of printer_bins_ppd() */
    
/*
** Add a bin to a printer's bin list.
*/
int printer_bins_add(char *argv[])
    {
    char *printer=argv[0];
    char *bin=argv[1];
    int count=0;
    
    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (bin==(char*)NULL) )
    	{
	fputs("You must specify an existing printer and a new bin name.\n",errors);
    	return EXIT_SYNTAX;
    	}

    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"The printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}
    	
    while(confread())				/* copy up to the 1st */
    	{					/* "Bin:" line, */
	if(strncmp(confline,"Bin: ",5)==0)
	    break;
	confwrite("%s\n",confline);
    	}

    do	{					/* copy all the "Bin:" lines */
	if(strncmp(confline,"Bin: ",5))
	    break;
	if(strcmp(&confline[5+strspn(&confline[5]," \t")],bin)==0)
	    {
	    fprintf(errors,"Printer \"%s\" already has a bin called \"%s\".\n",printer,bin);
	    confabort();
	    return EXIT_ALREADY;
	    }
    	confwrite("%s\n",confline);
    	count++;
    	} while(confread());

    if(count >= MAX_BINS)
    	{
    	fprintf(errors,"Printer \"%s\" already has %d bins, %d are allowed.\n",printer,count,MAX_BINS);
	confabort();
	return EXIT_OVERFLOW;
	}    	

    confwrite("Bin: %s\n",bin);			/* write our "Bin:" line */

    while(confread())				/* copy the */
    	{					/* rest of the file */
	confwrite("%s\n",confline);
    	}
    	
    confclose();

    reread_printer(printer);			/* necessary because pprd keeps track of mounted media */
    return EXIT_OK;
    } /* end of printer_bins_add() */
    
/*
** Remove a bin from a printer's bin list.
*/
int printer_bins_delete(char *argv[])
    {
    char *printer=argv[0];
    char *bin=argv[1];
    char *ptr;
    int found=FALSE;
    
    if( am_operator() )
    	return EXIT_DENIED;

    if( (printer==(char*)NULL) || (bin==(char*)NULL) )
    	{
	fputs("You must specify a printer and a bin to remove from it.\n",errors);
    	return EXIT_SYNTAX;
    	}

    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"The printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}
    	
    while(confread())				/* copy up to the "Bin:" line */
    	{					/* we are looking for */
	if(strncmp(confline,"Bin: ",5)==0)
	    {
	    ptr=&confline[5+strspn(&confline[5]," ")];
	    if(strcmp(ptr,bin)==0)
	        {
	        found=TRUE;
	        break;
	        }
	    }
	confwrite("%s\n",confline);
    	}

    while(confread())				/* copy the */
    	{					/* rest of the file */
	confwrite("%s\n",confline);
    	}
    	
    confclose();

    if(found)
	{
    	reread_printer(printer);
	return EXIT_OK;
	}
    else
    	{
	fprintf(errors,"The printer \"%s\" does not have a bin called \"%s\".\n",printer,bin);
    	return EXIT_BADBIN;
    	}
    } /* end of printer_bins_delete() */

/*
** Delete a printer configuration file and inform the spooler
** that we have deleted it so it can do so too.
*/
int printer_delete(char *argv[])
    {
    char *printer=argv[0];
    char fname[MAX_PATH];
    DIR *dir;
    struct dirent *direntp;
    int len;
    int is_a_member;
    
    if( am_operator() )
    	return EXIT_DENIED;

    if(printer==(char*)NULL)
    	{
    	fputs("You must specify a printer to delete.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    ppop2("halt",printer);		/* halt printer */
    ppop2("reject",printer);		/* accept no more jobs */
    ppop2("cancel",printer);		/* cancel all existing jobs */

    /* Remove the printer from membership in each */
    /* and every group.   Rather laborious, don't */
    /* you think? */
    if( (dir=opendir(GRCONF))==(DIR*)NULL )
    	{
    	fputs("ppad_printer.c: printer_delete(): opendir() failed\n",errors);
    	return EXIT_INTERNAL;
    	}
    	
    while( (direntp=readdir(dir)) != (struct dirent*)NULL )
    	{
    	if( direntp->d_name[0] == '.' )		/* skip . and .. */
	    continue;				/* and temporary files */    	

	len=strlen(direntp->d_name);		/* Emacs style backup files */
	if( len > 0 && direntp->d_name[len-1]=='~' )
	    continue;				/* should be skipt. */

	is_a_member=FALSE;

	if( grpopen(direntp->d_name,FALSE) )
	    {
	    fprintf(errors,"ppad_printer.c: printer_ppd(): grpopen(\"%s\",FALSE) failed",direntp->d_name);
	    closedir(dir);
	    return EXIT_INTERNAL;
	    }	    

	while(confread())
	    {
	    if(strncmp(confline,"Printer: ",9)==0)
	    	{
	    	if(strcmp(&confline[9],printer)==0)
		    {
	    	    is_a_member=TRUE;
		    break;
	    	    }
	    	}	    
	    }

	confclose();

	if(is_a_member)
	    _group_remove(direntp->d_name,printer);
	}

    closedir(dir);

    /* Remove the printer configuration file. */
    sprintf(fname,"%s/%s",PRCONF,printer);
    if(unlink(fname))
    	{
	if(errno==ENOENT)
    	    {
    	    fprintf(errors,"The printer \"%s\" does not exist.\n",printer);
    	    return EXIT_BADDEST;
    	    }
    	else
    	    {
	    fprintf(errors,"unlink(\"%s\") failed, errno=%d\n",fname,errno);
    	    return EXIT_INTERNAL;
    	    }
    	}
    else		/* It worked, now remove the mounted file, */
    	{		/* alert log, and status file. */
	sprintf(fname,"%s/%s",MOUNTEDDIR,printer);
	unlink(fname);

        sprintf(fname,"%s/%s",ALERTDIR,printer);
        unlink(fname);

        sprintf(fname,"%s/%s.status",ALERTDIR,printer);
        unlink(fname);

    	reread_printer(printer);
        return EXIT_OK;
    	}
    } /* end of printer_delete() */
    
/*
** Set the default Alert: line.
*/
int printer_new_alerts(char *argv[])
    {
    int frequency;
    char *method;
    char *address;
    FILE *newprn;

    if( am_operator() )
    	return EXIT_DENIED;

    if( (argv[0]==(char*)NULL) 
		|| (strspn(argv[0],"-0123456789")!=strlen(argv[0]))
    		|| (argv[1]==(char*)NULL) || (argv[2]==(char*)NULL) )
	{
	fputs("You must supply an alert frequency, such as \"7\", an alert method,\n"
		"such as \"mail\", and an alert address, such as \"alertreaders\".\n",errors);
    	return EXIT_SYNTAX;
	}

    frequency=atoi(argv[0]);
    method=argv[1];
    address=argv[2];
    
    if( (newprn=fopen(NEWPRN_CONFIG,"w")) == (FILE*)NULL )
    	{
    	fprintf(errors,"Unable to create \"%s/%s\"\n",HOMEDIR,NEWPRN_CONFIG);
    	return EXIT_INTERNAL;
    	}

    fprintf(newprn,"Alert: %d %s %s\n",frequency,method,address);

    fclose(newprn);
    return EXIT_OK;
    } /* end of printer_new_alerts() */

/*
** Just tell the spooler to re-read the configuration file.
*/
int printer_touch(char *argv[])
    {
    char *printer=argv[0];

    if( am_operator() )
    	return EXIT_DENIED;

    if(printer==(char*)NULL)
    	{
    	fputs("You must supply the name of a printer.\n",errors);
    	return EXIT_SYNTAX;
    	}

    /* make sure the printer exists */
    if(prnopen(printer,FALSE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    confclose();

    reread_printer(printer);	/* tell pprd to re-read the configuration */

    return EXIT_OK;
    } /* end of printer_touch() */

/*
** Change the switchset line in the configuration file.
*/
int printer_switchset(char *argv[])
    {
    char *printer=argv[0];	/* name of printer */
    char newset[256];		/* new set of switches */
    
    if( am_operator() )
    	return EXIT_DENIED;
    	
    if( printer==(char*)NULL )
    	{
	fputs("You must supply the name of a printer and a set of switches.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(prnopen(printer,TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n",printer);
    	return EXIT_BADDEST;
    	}

    /* convert "none" to nothing */
    if((argv[1]!=(char*)NULL) && (icmp(argv[1],"none")==0))
	argv[1]=(char*)NULL;

    /* convert the switch set to a line */
    if(argv[1]!=(char*)NULL && make_switchset_line(newset, &argv[1]))
    	{
    	fputs("Bad set of switches.\n",errors);
    	return EXIT_SYNTAX;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline, "Switchset: ", 9) == 0)
	    break;
	else
	    confwrite("%s\n", confline);
    	}

    if(argv[1] != (char*)NULL)			/* if new set, */
    	confwrite("Switchset: %s\n", newset);	/* write it */

    while(confread())				/* copy rest of file, */
	{					/* deleting further */
	if(strncmp(confline, "Switchset: ", 9) == 0) /* "Switchset:" lines. */
	    continue;
	else
	    confwrite("%s\n", confline);
    	}

    confclose();
    return EXIT_OK;    
    } /* end of printer_switchset() */

/*
** This command is called to re-read the printer's "PPDFile:"
** line and construct an appropriate "DefFiltOpts:" line.
**
** The "ppad ppd" and "ppad ppdopts" commands do this too.
*/
int printer_deffiltopts(char *argv[])
    {
    char *printer = argv[0];
    char *PPDFile = (char*)NULL;
    char *InstalledMemory = (char*)NULL;
    
    if( am_operator() )
    	return EXIT_DENIED;
    	
    if( printer==(char*)NULL )
    	{
    	fputs("You must supply the name of a printer.\n", errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(prnopen(printer, TRUE))
    	{
    	fprintf(errors,"Printer \"%s\" does not exist.\n", printer);
    	return EXIT_BADDEST;
    	}

    /* Get ready to collect information for the line. */
    deffiltopts_open();

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline, "DefFiltOpts:", 12) == 0)	/* delete */
	    continue;
	    
	if(strncmp(confline, "PPDFile:", 8) == 0)
	    {
	    int len;

	    if(PPDFile != (char*)NULL)
	    	{
	    	fputs("Warning: All but last \"PPDFile:\" line ignored.\n", errors);
	    	myfree(PPDFile);
	    	}

	    PPDFile = &confline[8];
	    PPDFile += strspn(PPDFile, " \t");
	    len = strlen(PPDFile);
	    while( --len >= 0 )
	    	if( isspace( PPDFile[len] ) )
	    	    PPDFile[len] = (char)NULL;
	    	else
	    	    break;

	    PPDFile = mystrdup(PPDFile);	    
	    }

	else if(strncmp(confline, "PPDOpt: *InstalledMemory ", 25) == 0)
	    {
	    if(InstalledMemory != (char*)NULL) myfree(InstalledMemory);
	    InstalledMemory = mystrndup(&confline[25], strcspn(&confline[25], " "));
	    }

	confwrite("%s\n",confline);
    	}

    if(PPDFile != (char*)NULL)
	{
	deffiltopts_add_ppd(PPDFile, InstalledMemory);
	myfree(PPDFile);
	}

    if(InstalledMemory != (char*)NULL)
    	myfree(InstalledMemory);

    confwrite("DefFiltOpts: %s\n", deffiltopts_line() );

    deffiltopts_close();

    confclose();

    return EXIT_OK;    
    } /* end of printer_deffiltopts() */

/*
** This command opens the PPD file and asks the user for an answer for each
** option found therein.  For each answer it generates a "PPDOpts:" line
** for the printer configuration file.
*/
int printer_ppdopts(char *argv[])
    {
    char *printer = argv[0];		/* printer whose configuration should be edited */
    char *PPDFile = (char*)NULL;	/* name of PPD file to open */
    char *InstalledMemory = (char*)NULL;
    char *ppdline;			/* a line read from the PPD file */
    char *ui_open;			/* the UI section we are int, NULL otherwise */
    char *ptr;
    unsigned int next_value;
    int c;
    char *values[100];			/* the list of possible values */
    int ret;
    
    /* Make sure we have the necessary authority. */
    if( am_operator() )
    	return EXIT_DENIED;
    	
    /* Make sure the required parameter is supplied. */
    if( printer == (char*)NULL )
    	{
    	fputs("You must supply the name of a printer.\n", errors);
    	return EXIT_SYNTAX;
    	}
    	
    /* Open the printer's configuration file for modification. */
    if(prnopen(printer, TRUE))
    	{
    	fprintf(errors, "Printer \"%s\" does not exist.\n", printer);
    	return EXIT_BADDEST;
    	}

    /*
    ** Set whole values array to NULL so we will
    ** later know what we must myfree().
    */
    for(next_value=0; next_value < (sizeof(values)/sizeof(char*)); next_value++)
    	values[next_value] = (char*)NULL;

    /*
    ** Copy the existing printer configuration file, discarding 
    ** "PPDOpt:" and "DefFiltOpts:" lines and noting the 
    */
    while(confread())
    	{
	if(strncmp(confline, "PPDOpt:", 7) == 0)	/* delete */
	    continue;

	if(strncmp(confline, "DefFiltOpts:", 12) == 0)	/* delete */
	    continue;
	
	confwrite("%s\n", confline);			/* copy to output file */

	if(strncmp(confline, "PPDFile: ", 9) == 0)	/* note and stop at */
	    {
	    int len;
	    PPDFile = &confline[9];
	    PPDFile += strspn(PPDFile, " \t");		/* remove leading white space */
	    for(len=strlen(PPDFile); len > 0; len--)	/* remove trailing space */
	    	if( ! isspace(PPDFile[len-1]) )
		    break;
	    PPDFile = mystrndup(PPDFile, len);
	    break;
	    }
    	}

    /* If there was no "PPDFile:" line, we can't continue. */
    if( PPDFile == (char*)NULL )
    	{
    	fprintf(errors, "Printer \"%s\" has no \"PPDFile:\" line in its configuration file.\n", printer);
	confabort();
	return EXIT_BADDEST;
    	}

    /* Open the PPD file. */
    if( (ret=ppd_open(PPDFile, errors)) )
    	return ret;

    /*
    ** Read the PPD file, ask the questions, and
    ** generate "PPDOpts:" lines.
    */ 
    ui_open = (char*)NULL;
    next_value = 0;
    fputc('\n',stdout);
    while( (ppdline=ppd_readline()) != (char*)NULL )
    	{
	if( strncmp(ppdline, "*OpenUI", 7) == 0 ) 
	    {
	    /*
	    ** If we already have a pointer to the name of the current
	    ** UI block, since we have just seen the start of a new one,
	    ** we know that there was an unclosed one since if it had been
	    ** closed the pointer would have been set to NULL.
	    **
	    ** After printing the warning, we need only free the string.
	    ** In order to discard the values from the unclosed block,
	    ** next_value is set to 1 a little later.  We must set ui_open
	    ** to NULL here since if we aren't interested in the new section,
	    ** we will not save its name in ui_open.
	    */
	    if( ui_open != (char*)NULL )
		{
    		fprintf(errors, "WARNING: Unclosed UI block \"%s\" in PPD file\n", ui_open);
		myfree(ui_open);
		ui_open = (char*)NULL;
    		}

	    /* Move pointer to second word on line */
	    ptr = &ppdline[7];
	    ptr += strspn(ptr," \t");
	    
	    /* If this is not an option UI block, we are not interested. */
	    if( strncmp(ptr, "*Option", 7) && strncmp(ptr, "*InstalledMemory", 16) )
	    	continue;

	    /* Truncate after the end of the translation string. */
	    ptr[strcspn(ptr, ":\n")] = (char)NULL;		

	    /* Print the option name and its translation string. */
	    printf("%s\n", ptr);
	    
	    /* Save the option name and translation string. */
	    ui_open = mystrdup(ptr);

	    /* Set to indicated no accumulated values. */
	    next_value = 1;
	    
	    continue;
	    }

	/* If between "*OpenUI" and "*CloseUI" and this is an "*Option" line, */
	if( ui_open != (char*)NULL && (strncmp(ppdline, "*Option", 7) == 0 || strncmp(ppdline, "*InstalledMemory", 16) == 0) )
	    {
	    if( strncmp(ppdline, ui_open, strcspn(ui_open, "/")) )
	    	{
	    	fprintf(errors, "WARNING: spurious option found in UI section \"%s\".\n", ui_open);
	    	continue;
	    	}

	    /* set ptr to start of second word */
	    ptr = &ppdline[7];
	    ptr += strcspn(ptr," \t");
	    ptr += strspn(ptr," \t");
		
	    /* Truncate at end of translation string. */
	    ptr[strcspn(ptr, ":\n")] = (char)NULL;

	    printf("%d) %s\n", next_value, ptr);

	    if( next_value >= (sizeof(values)/sizeof(char*)) )
	    	{
		fputs("ppad_printer.c: printer_ppdopts(): values[] overflow\n", errors);
	    	return EXIT_INTERNAL;
	    	}

	    if(values[next_value] != (char*)NULL) myfree(values[next_value]);
	    values[next_value++] = mystrdup(ptr);

	    continue;
	    }

	/* If this is the end of an option, ask for a choice. */
	if( ui_open && strncmp(ppdline, "*CloseUI:", 9) == 0 )
	    {
	    /* Move pointer to second word on line */
	    ptr = &ppdline[9];
	    ptr += strspn(ptr," \t");
	    
	    if( strncmp(ptr, ui_open, strcspn(ui_open, "/")) )
		{
		fputs("WARNING: mismatched \"*OpenUI\", \"*CloseUI\" in PPD file.\n", errors);
		fprintf(errors, "(\"%s\" closed by \"%.*s\".)\n", ui_open, (int)strcspn(ptr,"/\n"), ptr);
		}

	    /* Present the user with the list of options we have gathered. */
	    do  {
		printf("Choose 1 thru %d or 0 if unknown> ", next_value-1);
		fflush(stdout);
		if( fgets(ppdline, sizeof(ppdline), stdin) == (char*)NULL )
		    {
		    c = 0;
		    break;
		    }
		c = atoi(ppdline);
		} while( ! isdigit( ppdline[strspn(ppdline," \t")] ) || c >= next_value );

	    fputc('\n',stdout);
	    	
	    /*
	    ** If the user did not choose zero (unknown), write 
	    ** a line containing the option and the chosen value
	    ** into the printer configuration file.
	    */
	    if( c != 0 )
		{
		confwrite("PPDOpt: %.*s %.*s", strcspn(ui_open, "/"), ui_open, strcspn(values[c],"/"), values[c]);

		if( strchr(ui_open, '/') != (char*)NULL && strchr(values[c],'/') != (char*)NULL )
		    confwrite(" (%s %s)", (strchr(ui_open, '/') + 1), (strchr(values[c], '/') + 1));

		confwrite("\n");

		/* If this is the amount of installed memory, feed the 
		   value to the code which is generating the "DefFiltOpts:" line. */
		if( strncmp(ui_open, "*InstalledMemory", 16) == 0 )
		     InstalledMemory = mystrndup(values[c], strcspn(values[c],"/"));
		}

	    myfree(ui_open);
	    ui_open = (char*)NULL;
	    }
    	} /* end of PPD reading loop */

    /* Sanity check. */
    if( ui_open != (char*)NULL )
	{
    	fprintf(errors, "WARNING: Unclosed UI block in PPD file: \"%s\".\n", ui_open);
    	myfree(ui_open);
	}

    /*
    ** Copy the rest of the configuration file,
    ** deleting old "PPDOpt:" and "DefFiltOpts:" lines as we go.
    */
    while(confread())
    	{
	if(strncmp(confline, "PPDOpt:", 7) == 0)
	    continue;
	
	if(strncmp(confline, "DefFiltOpts:", 12) == 0)
	    continue;

	confwrite("%s\n", confline);
    	}

    /* Emmit a new "DefFiltOpts:" line. */
    deffiltopts_open();
    if( (ret=deffiltopts_add_ppd(PPDFile, InstalledMemory)) )
	return ret;
    confwrite("DefFiltOpts: %s\n", deffiltopts_line() );
    deffiltopts_close();

    confclose();

    /* Free any lingering memory blocks */
    if(PPDFile != (char*)NULL) myfree(PPDFile);
    if(InstalledMemory != (char*)NULL) myfree(InstalledMemory);
    for(next_value=0; next_value < (sizeof(values)/sizeof(char*)); next_value++)
    	if( values[next_value] != (char*)NULL )
    	    myfree(values[next_value]);

    /*
    ** Update the default filter options of any
    ** groups which have this printer as a member.
    */
    return update_groups_deffiltopts(printer);
    } /* end of printer_ppdopts() */

/* end of file */
