/*
** ~ppr/src/ppad/ppad_group.c
** Copyright 1995, Trinity College Computing Center
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This file was last modified 24 October 1995.
*/

/*
** This module is part of the administrators utility.  It contains the code
** for those sub-commands which manipulate groups of printers.
*/

#include "global_defines.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include "util_exits.h"
#include "ppad.h"

/*
** Send the spooler a command to re-read a group definition
*/
void reread_group(char *group)
    {
    write_fifo("NG %s\n",group);
    }

/*
** Show a groups members, its rotate setting, and its comment.
*/
int group_show(char *argv[])
    {
    char *group=argv[0];		/* The name of the group to show is the argument. */
    int rotate=TRUE;			/* Is rotate set for this group? */
    char *comment=(char*)NULL;		/* Group comment. */
    int member_count=0;			/* Keep count of members. */
    char *members[MAX_GROUPSIZE];	/* The names of the members. */
    char *deffiltopts=(char*)NULL;	/* The default filter options string. */
    char *switchset=(char*)NULL;	/* The compressed switchset string. */
    int x;
    char *ptr;
    
    if(group==(char*)NULL)
    	{
    	fputs("You must supply the name of a group to show.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(grpopen(group,FALSE))	/* Open the configuration file. */
    	{
    	fprintf(errors,"The group \"%s\" does not exist.\n",group);
    	return EXIT_BADDEST;
    	}
    	
    while(confread())		/* Read all the lines in the config file. */
    	{
    	if(strncmp(confline,"Rotate: ",8)==0)
    	    {
    	    rotate=torf(&confline[8]);
    	    continue;
    	    }
	if(strncmp(confline,"Comment: ",9)==0)
	    {
	    if(comment!=(char*)NULL) myfree(comment);
	    comment=mystrdup(&confline[9]);
	    continue;
	    }
	if(strncmp(confline,"Printer: ",9)==0)
	    {
	    if(member_count < MAX_GROUPSIZE)
	    	{
	    	members[member_count++]=mystrdup(&confline[9]);
	    	}
	    }
	if(strncmp(confline,"Switchset: ",11)==0)
	    {
	    if(switchset!=(char*)NULL) myfree(switchset);
	    switchset=mystrdup(&confline[11]);
	    continue;
	    }
	if( ppr_sscanf(confline,"DefFiltOpts: %Z",&ptr) == 1 )
	    {
	    if(deffiltopts!=(char*)NULL) myfree(deffiltopts);
	    deffiltopts=ptr;
	    }

    	}
    
    confclose();	/* We are done with the configuration file. */

    printf("Group name: %s\n",group);
    printf("Comment: %s\n",comment!=(char*)NULL?comment:"<none>");
    printf("Members:");
    for(x=0;x<member_count;x++)		/* Show what printers are members. */
	{
    	if(x==0)
    	    printf(" %s",members[x]);
    	else
    	    printf(", %s",members[x]);

	myfree(members[x]);		/* Free the member */
	}
    printf("\n");			/* End group members line. */
    printf("Rotate: %s\n",rotate?"True":"False");
    print_deffiltopts(deffiltopts);
    print_switchset(switchset);

    if(deffiltopts!=(char*)NULL) myfree(deffiltopts);
    if(switchset!=(char*)NULL) myfree(switchset);

    return EXIT_OK;
    } /* end of group_show() */
    
/*
** Set a group's comment.    
*/
int group_comment(char *argv[])
    {
    char *group=argv[0];
    char *comment=argv[1];

    if( am_operator() )
    	return EXIT_DENIED;

    if( (group==(char*)NULL) || (comment==(char*)NULL) )
	{
	fputs("You must supply the name of a group and\n"
		"a comment to attach to it.\n",errors);
	return EXIT_SYNTAX;
	}

    /* make sure the group exists */
    if(grpopen(group,TRUE))
    	{
    	fprintf(errors,"Group \"%s\" does not exist.\n",group);
    	return EXIT_BADDEST;
    	}

    /* Modify the printer's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Comment: ",9)==0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}

    confwrite("Comment: %s\n",comment);

    while(confread())				/* copy rest of file, */
	{					/* deleting further */
	if(strncmp(confline,"Comment: ",9)==0)	/* "Comment:" lines. */
	    continue;
	else
	    confwrite("%s\n",confline);
    	}
    confclose();
    /* no need to reread_group(group) */

    return EXIT_OK;
    } /* end of group_comment() */
    
/*
** Set a group's rotate setting.
*/
int group_rotate(char *argv[])
    {
    char *group=argv[0];
    int newstate;

    if( am_operator() )
    	return EXIT_DENIED;

    if( (group==(char*)NULL) || (argv[1]==(char*)NULL) 
    		|| ((newstate=torf(argv[1]))==ANSWER_UNKNOWN) )
	{
	fputs("You must supply the name of a group and \"true\" or \"false\".\n",errors);
	return EXIT_SYNTAX;
	}

    /* make sure the group exists */
    if(grpopen(group,TRUE))
    	{
    	fprintf(errors,"Group \"%s\" does not exist.\n",group);
    	return EXIT_BADDEST;
    	}

    /* Modify the group's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Rotate: ",8)==0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}

    confwrite("Rotate: %s\n",newstate?"True":"False");

    while(confread())				/* copy rest of file, */
	{
	if(strncmp(confline,"Rotate: ",8)==0)
	    continue;
	else
	    confwrite("%s\n",confline);
    	}
    confclose();
    
    reread_group(group);			/* pprd must know the rotate setting */
    return EXIT_OK;
    } /* end of group_rotate() */

/*
** Add a member to a group, creating the group if
** it does not already exist.
*/
int group_add(char *argv[])
    {
    char *group=argv[0];
    char *newmember=argv[1];
    int count=0;
    int retval;
    
    if( am_operator() )
    	return EXIT_DENIED;

    if( (group==(char*)NULL) || (newmember==(char*)NULL) )
    	{
	fputs("You must specify a new or existing group and a printer to add.\n",errors);
    	return EXIT_SYNTAX;
    	}

    if(argv[2]!=(char*)NULL)
	{
	fputs("Too many parameters.  You may only add one\n"
		"member at a time.\n",errors);
	return EXIT_SYNTAX;
	}

    if(strlen(group) > MAX_DESTNAME)
    	{
    	fputs("The group name is too long.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(strpbrk(group,DEST_DISALLOWED) != (char*)NULL)
	{
	fputs("Group name contains a disallowed character.\n", errors);
	return EXIT_SYNTAX;
	}

    if(strchr(DEST_DISALLOWED_LEADING, (int)group[0]) != (char*)NULL)
    	{
    	fputs("Group name begins witha disallowed character.\n", errors);
	return EXIT_SYNTAX;
	}
	
    /*
    ** Make sure the preposed new member really exists.
    ** We do this by trying to open its configuration file
    ** with prnopen().
    */
    if(prnopen(newmember,FALSE))
    	{
    	fprintf(errors,"The printer \"%s\" does not exist.\n",newmember);
    	return EXIT_BADDEST;
    	}
    else
    	{
    	confclose();
    	}

    /*
    ** Get ready to gather information for the "DefFiltOpts:" line.
    */
    deffiltopts_open();

    /*
    ** If the group to which we are adding a printer
    ** does not exist, create it.
    */
    if(grpopen(group,TRUE))
    	{
	char fname[MAX_PATH];
	FILE *newgroup;
	
	sprintf(fname,"%s/%s",GRCONF,group);
	if( (newgroup=fopen(fname,"w"))==(FILE*)NULL )
	    {
	    fprintf(errors,"Failed to create group config file \"%s\".\n",fname);
	    deffiltopts_close();
	    return EXIT_INTERNAL;
	    }
	    
	/* The printer we are adding will be the only member. */
	fprintf(newgroup,"Printer: %s\n",newmember);

	/* It's PPD file will be the only one considered. */
	deffiltopts_add_printer(newmember);

	/* We will add the newly created "DefFiltOpts:" line. */
	fprintf(newgroup,"DefFiltOpts: %s\n",deffiltopts_line());
	
	fclose(newgroup);
    	} /* end of if a new group */

    /* The group already exist, add a member. */
    else
    	{    	
	char *ptr;

	while(confread())				/* Copy up to but not including the 1st */
    	    {						/* "Printer:" line, */
	    if(strncmp(confline,"DefFiltOpts: ",13)==0)
	        continue;
	    if(strncmp(confline,"Printer: ",9)==0)
	    	break;
	    confwrite("%s\n",confline);
    	    }

    	/*
    	** Copy all the remaining lines.
    	*/
    	do  {			
	    /*
	    ** If it is a printer line, make sure it is not the 
	    ** printer we think we are adding.  If it is not, 
	    ** call deffiltopts_add_printer() so it can consider
	    ** its PPD file.
	    */
	    if(strncmp(confline,"Printer: ",9)==0)
		{
		ptr=&confline[9+strspn(&confline[9]," \t")];
		if(strcmp(ptr,newmember)==0)
	    	    {
		    fprintf(errors,"Printer \"%s\" is already a member of \"%s\".\n",newmember,group);
	    	    confabort();
		    deffiltopts_close();
	    	    return EXIT_ALREADY;
	    	    }
		deffiltopts_add_printer(ptr);	/* Consider its PPD file. */
		count++;
		}
		
	    else if(strncmp(confline,"DefFiltOpts: ",13)==0)	/* Delete old "DefFiltOpts:" lines. */
	        continue;

	    confwrite("%s\n",confline);
    	    } while(confread());

	/*
	** See if adding our printer will make the group too big.
	*/
	if(count >= MAX_GROUPSIZE)
	    {
	    fprintf(errors,"Group \"%s\" already has %d members, %d are allowed.\n",group,count,MAX_GROUPSIZE);
	    confabort();
	    deffiltopts_close();
	    return EXIT_OVERFLOW;
	    }

    	/* Add a "Printer:" line for the new member. */
    	confwrite("Printer: %s\n",newmember);

	/* Consider the information in the new printer's PPD file. */
	if( (retval=deffiltopts_add_printer(newmember)) != EXIT_OK )
	    {
	    confabort();
	    return retval;
	    }

	/* Emmit the new "DefFiltOpts:" line. */
	confwrite("DefFiltOpts: %s\n",deffiltopts_line());

        confclose();
	} /* end of if not a new group */
	
    deffiltopts_close();	/* Dealloacate PPD file study structures. */

    reread_group(group);	/* necessary because pprd keeps track of mounted media */
    return EXIT_OK;
    } /* end of group_add() */
    
/*
** Remove a member from a group.
**
** sucess:	EXIT_OK
** bad group:	EXIT_BADDEST
** bad member:  EXIT_NOTFOUND
*/
int _group_remove(char *group, char *member)
    {
    char *ptr;
    int found=FALSE;

    if(grpopen(group,TRUE))
    	return EXIT_BADDEST;
    	
    deffiltopts_open();

    /*
    ** Copy the configuration file.
    */
    while(confread())
    	{
	if(strncmp(confline,"DefFiltOpts: ",13)==0)	/* delete old lines */
	    continue;

	if(strncmp(confline,"Printer: ",9)==0)		/* if member name, */
	    {
	    ptr=&confline[9+strspn(&confline[9]," \t")];

	    if(strcmp(ptr,member)==0)			/* If it is the one we */
	        {					/* are deleting, */
	        found=TRUE;				/* set a flag */
	        continue;				/* and don't copy. */
	        }

	    deffiltopts_add_printer(ptr);		/* Otherwise, add its PPD file, */
	    }

	confwrite("%s\n",confline);
    	}

    /* Write a new "DefFiltOpts:" line. */
    confwrite("DefFiltOpts: %s\n",deffiltopts_line());

    deffiltopts_close();

    confclose();

    if(found)
    	{
    	reread_group(group);
	return EXIT_OK;
    	}
    else
    	{
	fprintf(errors,"The printer \"%s\" was not a member of the group \"%s\".\n",member,group);
    	return EXIT_NOTFOUND;
    	}
    	
    } /* end of _group_remove() */

int group_remove(char *argv[])
    {
    char *group=argv[0];
    char *member=argv[1];
    
    if( am_operator() )
    	return EXIT_DENIED;

    if( (group==(char*)NULL) || (member==(char*)NULL) )
    	{
    	fputs("You must specify a group and a printer to remove from it.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(argv[2]!=(char*)NULL)
	{
	fputs("Too many parameters.\n",errors);
	return EXIT_SYNTAX;
	}

    switch(_group_remove(group,member))
    	{
    	case EXIT_OK:
    	    return EXIT_OK;
    	case EXIT_BADDEST:
    	    fprintf(errors,"The group \"%s\" does not exist.\n",group);
    	    return EXIT_BADDEST;
    	case EXIT_NOTFOUND:
	    fprintf(errors,"The group \"%s\" does not have a member called \"%s\".\n",group,member);
    	    return EXIT_NOTFOUND;
	default:
	    fputs("ppad_group.c: group_remove(): internal error\n",errors);
	    return EXIT_INTERNAL;
    	}
    } /* end of group_remove() */
    
/*
** Delete a group.
*/
int group_delete(char *argv[])
    {
    char *group=argv[0];
    char fname[MAX_PATH];
    
    if( am_operator() )
    	return EXIT_DENIED;

    if(group==(char*)NULL)
    	{
	fputs("You must specify a group to delete.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(argv[1]!=(char*)NULL)
    	{
    	fputs("Too many parameters.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    /* Run ppop to */
    /* accept no more jobs */
    /* and cancel existing jobs. */
    ppop2("reject",group);
    ppop2("cancel",group);

    sprintf(fname,"%s/%s",GRCONF,group);
    if(unlink(fname))
    	{
	if(errno==ENOENT)
    	    {
    	    fprintf(errors,"The group \"%s\" does not exist.\n",group);
    	    return EXIT_BADDEST;
    	    }
    	else
    	    {
	    fprintf(errors,"unlink(\"%s\") failed, errno=%d\n",fname,errno);
    	    return EXIT_INTERNAL;
    	    }
    	}
    else
    	{
    	reread_group(group);
    	return EXIT_OK;
    	}
    } /* end of group_remove() */

/*
** Tell the spooler to re-read a group configuration.
*/
int group_touch(char *argv[])
    {
    char *group=argv[0];

    if( am_operator() )
    	return EXIT_DENIED;

    if( (group==(char*)NULL) )
    	{
    	fputs("You must supply the name of a group.\n",errors);
    	return EXIT_SYNTAX;
    	}

    /* make sure the printer exists */
    if(grpopen(group,FALSE))
    	{
    	fprintf(errors,"Group \"%s\" does not exist.\n",group);
    	return EXIT_BADDEST;
    	}

    confclose();

    reread_group(group);	/* tell pprd to re-read the configuration */

    return EXIT_OK;
    } /* end of group_touch() */

/*
** Change the "Switchset:" line.
*/
int group_switchset(char *argv[])
    {
    char *group=argv[0];	/* name of group */
    char newset[256];		/* new set of switches */
    
    if( am_operator() )
    	return EXIT_DENIED;
    	
    if( group==(char*)NULL )
    	{
	fputs("You must supply the name of a group and a set of switches.\n",errors);
    	return EXIT_SYNTAX;
    	}
    	
    if(grpopen(group,TRUE))
    	{
    	fprintf(errors,"Group \"%s\" does not exist.\n",group);
    	return EXIT_BADDEST;
    	}

    /* convert "none" to nothing */
    if((argv[1]!=(char*)NULL) && (icmp(argv[1],"none")==0))
    	argv[1]=(char*)NULL;

    /* convert the switch set to a line */
    if( argv[1]!=(char*)NULL && make_switchset_line(newset,&argv[1]) )
    	{
    	fputs("Bad set of switches.\n",errors);
    	return EXIT_SYNTAX;
    	}

    /* Modify the group's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Switchset: ",9)==0)
	    break;
	else
	    confwrite("%s\n",confline);
    	}

    if( argv[1]!=(char*)NULL )			/* if new switch set */
        confwrite("Switchset: %s\n",newset);	/* install it */

    while(confread())				/* copy rest of file, */
	{					/* deleting further */
	if(strncmp(confline,"Switchset: ",9)==0)/* "Switchset:" lines. */
	    continue;
	else
	    confwrite("%s\n",confline);
    	}

    confclose();
    return EXIT_OK;    
    } /* end of group_switchset() */

/*
** Set the DefFiltOpts: line for a group.
*/
int _group_deffiltopts(char *group)
    {
    if(grpopen(group,TRUE))
    	{
    	fprintf(errors,"Group \"%s\" does not exist.\n",group);
    	return EXIT_BADDEST;
    	}

    deffiltopts_open();

    /* Modify the group's configuration file. */
    while(confread())
    	{
	if(strncmp(confline,"Printer: ",9)==0)
	    deffiltopts_add_printer(&confline[9]);

	if(strncmp(confline,"DefFiltOpts: ",13)!=0)
	    confwrite("%s\n",confline);
    	}

    confwrite("DefFiltOpts: %s\n",deffiltopts_line());

    deffiltopts_close();

    confclose();
    return EXIT_OK;    
    } /* end of _group_deffiltopts() */

int group_deffiltopts(char *argv[])
    {
    char *group=argv[0];	/* name of group */
    
    if( am_operator() )
    	return EXIT_DENIED;
    	
    if( group==(char*)NULL )
    	{
    	fputs("You must supply the name of a group.\n",errors);
    	return EXIT_SYNTAX;
    	}

    return _group_deffiltopts(group);    	
    } /* end of group_deffiltopts() */

/* end of file */
