/*
** ~ppr/src/ppad/ppad_filt.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 20 March 1997.
*/

/*
** This file contains the code which generates a "DefFiltOpts:" line.
*/

#include "global_defines.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include "ppad.h"
#include "util_exits.h"

/* #define DEBUG_FILT 1 */

/*
** The variables we use the compile a "DefFiltOpts:" line.
*/
static int open_called = FALSE;		/* has deffiltopts_open() been called? */
static int level;			/* -1 if not seen yet, otherwise lowest */
static char *resolution = (char*)NULL;
static int resolution_conflict;		/* TRUE if not all are the same */
static int colour;			/* TRUE if any have colour yet */
static int freevm;			/* zero means not found yet, otherwise, smallest yet */
static char product[64];
static char modelname[64];
static char nickname[64];
static char *mfmode = (char*)NULL;
static int mfmode_conflict;

/*
** Given a Product, ModelName, Nickname, and resolution return
** a MetaFont mode name.
**
** The name will be returned in myalloc()ed memory.
*/
static char *get_mfmode(const char *product, const char *modelname, const char *nickname, const char *resolution)
    {
    FILE *modefile;
    char line[256];
    int linenum;
    const char *p, *m, *n, *r;
    int plen, mlen, nlen, rlen;
    char *f1, *f2, *f3, *f4, *f5;
    int f1len, f2len, f3len, f4len, f5len;
    char *ptr = (char*)NULL;
    
    #ifdef DEBUG_FILT
    printf("get_mfmode(product=\"%s\", modelname=\"%s\", nickname=\"%s\"\n",
	product != (char*)NULL ? product : "<NULL>",
	modelname != (char*)NULL ? modelname : "<NULL>",
	nickname != (char*)NULL ? nickname : "<NULL>");
    #endif

    p = product != (char*)NULL    ? product : "";
    m = modelname != (char*)NULL  ? modelname : "";
    n = nickname != (char*)NULL   ? nickname : "";
    r = resolution != (char*)NULL ? resolution : "";

    if(debug_level >= 2)
    	printf("Looking up mfmode for product \"%s\", modelname \"%s\", nickname \"%s\".\n", p, m, n);

    plen = strlen(p);
    mlen = strlen(m);
    nlen = strlen(n);
    rlen = strlen(r);
    
    if( (modefile=fopen(MFMODES, "r")) == (FILE*)NULL )
	{
	fprintf(errors, "ppad: can't open \""MFMODES"\", errno=%d (%s)\n", errno, strerror(errno));
	return (char*)NULL;
	}

    for(linenum=1; fgets(line,sizeof(line), modefile) != (char*)NULL; linenum++)
    	{
	if(line[0]==';' || line[0]=='#' || line[strspn(line, " \t\r\n")] == (char)NULL)
	    continue;
	    
	f1 = line;			/* first parameter */
	f1len = strcspn(f1, ":");
	
	f2 = f1 + f1len;		/* second */
	if(*f2 == ':') f2++;
	f2len = strcspn(f2, ":");
	
	f3 = f2 + f2len;		/* third */
	if(*f3 == ':') f3++;
	f3len = strcspn(f3, ":");
	
	f4 = f3 + f3len;		/* fourth */
	if(*f4 == ':') f4++;
	f4len = strcspn(f4, ":");
	
	f5 = f4 + f4len;		/* fifth */
	if(*f5 == ':') f5++;
	f5len = strcspn(f5, ": \t\n\r");
	
	if(f2[-1] != ':' || f3[-1] != ':' || f4[-1] != ':' || f5[-1] != ':')
	    {
	    fprintf(errors, "Warning: syntax error in "MFMODES" lines %d.\n", linenum);
	    continue;
	    }

	if(debug_level >= 3)
	    printf("line %d: product \"%.*s\", modelname \"%.*s\", nickname \"%.*s\", resolution \"%.*s\"\n", linenum, f1len, f1, f2len, f2, f3len, f3, f4len, f4);

	if(f1[0] != '*' && (plen != f1len || strncmp(p, f1, plen)) )
	    continue;
	
	if(f2[0] != '*' && (mlen != f2len || strncmp(m, f2, mlen)) )
	    continue;

	if(f3[0] != '*' && (nlen != f3len || strncmp(n, f3, nlen)) )
	    continue;
	        	
	if(f4[0] != '*' && (rlen != f4len || strncmp(r, f4, rlen)) )
	    continue;

	ptr = (char*)myalloc( (f5len+1), sizeof(char) );
	strncpy(ptr, f5, f5len);
	ptr[f5len] = (char)NULL;

	if(debug_level >= 2)
	    printf("Match at "MFMODES" line %d, mfmode=%s\n", linenum, ptr);

	break;
    	}

    fclose(modefile);

    if(ptr == (char*)NULL)
    	fputs("Warning: "MFMODES" has no matching clause!\n", errors);

    return ptr;
    } /* end of get_mfmode() */

/*
** This routine initializes the variables in order to
** get ready to read PPD files.
*/
void deffiltopts_open(void)
    {
    #ifdef DEBUG_FILT
    fputs("deffiltopts_open()\n", stdout);
    #endif

    if(debug_level >= 2)
    	printf("Gathering information for \"DefFiltOpts:\" line.\n");

    open_called = TRUE;			/* this routine has been called */
    level = -1;				/* unknown */
    resolution = (char*)NULL;		/* unknown */
    resolution_conflict = FALSE;	/* not yet */
    colour = FALSE;			/* assume false */
    freevm = 0;				/* unknown */

    mfmode = (char*)NULL;		/* unknown */
    mfmode_conflict = FALSE;		/* not yet */
    } /* end of deffiltopts_open() */
    
/*
** Read a PPD file and contribute its contents to 
** the information we will use to supply a "DefFiltOpts:" line.
**
** Various PPAD sub-commands call this.  They call it when they
** copy the "PPDFile:" line.
*/
int deffiltopts_add_ppd(const char *name, const char *InstalledMemory)
    {
    int ret;
    char *line;				/* the line we are working on */
    int tempint;			/* integer for ppr_sscanf() */
    char *tempptr;			/* char pointer for ppr_sscanf() */
    int thisppd_resolution = FALSE;	/* <-- We use these variables in */
    int thisppd_level = FALSE;		/* order to honour only the */
    int thisppd_colour = FALSE;		/* first instance of certain */
    int thisppd_freevm = 0;		/* keywords. */
    int thisppd_product = FALSE;
    int thisppd_modelname = FALSE;
    int thisppd_nickname = FALSE;
    int vmoptions_count = 0;
    struct { char *opt; int val; } vmoptions[MAX_VMOPTIONS];

    #ifdef DEBUG_FILT
    printf("deffiltopts_add_ppd(\"%s\", \"%s\")\n", name, InstalledMemory != (char*)NULL ? InstalledMemory : "NULL");
    #endif

    if(debug_level > 1)
    	printf("Extracting deffiltopts information from PPD file \"%s\".\n", name);

    if( ! open_called )
    	fatal(EXIT_INTERNAL, "deffiltopts_add_ppd(): not open");

    product[0] = (char)NULL;		/* clear metafont information */
    modelname[0] = (char)NULL;
    nickname[0] = (char)NULL;

    /* Open the PPD file. */
    if( (ret=ppd_open(name, errors)) )
    	return ret;

    /* Examine each line in the PPD file. */
    while( (line=ppd_readline()) != (char*)NULL )
    	{
	/* Short circuit comments, blank lines, and such: */
	if(line[0] != '*' || strncmp(line, "*%", 2) == 0)
	    continue;

	if(debug_level > 5)
	    printf("PPD: %s", line);

	/*
	** If this is a "*LanguageLevel:" line and either there has
	** been no previous such line or this is a lower level,
	** then accept this as the final level.
	*/
	if( !thisppd_level && ppr_sscanf(line,"*LanguageLevel: \"%d\"", &tempint)==1 )
	    {
	    if( level == -1 || tempint < level )
	    	level = tempint;
	    thisppd_level = TRUE;
	    continue;
	    }
	    
	/*
	** If this is the first "*DefaultResolution:" line in this file
	** and there has been no resolution conflict detected previously,
	*/
	if( ! thisppd_resolution && ! resolution_conflict && ppr_sscanf(line, "*DefaultResolution: %S", &tempptr) == 1 )
	    {
	    /* If no resolution statement in prior file, save this one. */
	    if(resolution == (char*)NULL)
	    	{
		resolution = tempptr;
	    	}
	    /* If resolution statement in prior file, */
	    else
	    	{
		/* If they disagree, declare a conflict. */
		if( strcmp(resolution, tempptr) != 0)
		    {
		    myfree(resolution);
		    resolution = (char*)NULL;
		    resolution_conflict = TRUE;
		    }
		myfree(tempptr);
	    	}
	    thisppd_resolution = TRUE;
	    continue;
	    }

    	/* 
    	** If this is a "*FreeVM:" line and it is the first one
    	** in this file, accept this new value.
    	*/
    	if( thisppd_freevm == 0 && ppr_sscanf(line, "*FreeVM: \"%d\"", &tempint) == 1 )
    	    {
	    thisppd_freevm = tempint;
    	    continue;
    	    }
    	
	/*
	** If this is a "*ColorDevice:" line and it says true, set 
	** colour to TRUE.
	*/
	if( !thisppd_colour && strncmp(line, "*ColorDevice:", 13) == 0 )
	    {
	    tempptr = &line[13];
	    tempptr += strspn(tempptr," \t");
	    if( tempptr[0] == 'T' )
	    	colour = TRUE;
	    thisppd_colour = TRUE;
	    continue;
	    }

	/*
	** Collect *Product, *ModelName, and *NickName information
	** for determining the MetaFont mode.
	*/
	if( !thisppd_product && strncmp(line, "*Product:", 9) == 0 )
	    {
	    tempptr = &line[9];
	    tempptr += strspn(tempptr, " \t\"(");
	    tempptr[strcspn(tempptr,")")] = (char)NULL;
	    strncpy(product, tempptr, sizeof(product));	/* do a */
	    product[sizeof(product)-1] = (char)NULL;	/* truncating copy */

	    thisppd_product = TRUE;
	    continue;
	    }
	if( !thisppd_modelname && strncmp(line,"*ModelName:", 11) == 0 )
	    {
	    tempptr = &line[11];
	    tempptr += strspn(tempptr, " \t\"");
	    tempptr[strcspn(tempptr,"\"")] = (char)NULL;
	    strncpy(modelname, tempptr, sizeof(modelname));	/* do a */
	    modelname[sizeof(modelname)-1] = (char)NULL;	/* truncating copy */

	    thisppd_modelname = TRUE;
	    continue;
	    }
	if( !thisppd_nickname && strncmp(line, "*NickName:", 10) == 0 )
	    {
	    tempptr = &line[10];
	    tempptr += strspn(tempptr," \t\"");
	    tempptr[strcspn(tempptr,"\"")] = (char)NULL;
	    strncpy(nickname, tempptr, sizeof(nickname));	/* do a */
	    nickname[sizeof(nickname)-1] = (char)NULL;		/* truncating copy */

	    thisppd_nickname = TRUE;
	    continue;
	    }
	    
	/*
	** VMOption 
	*/
	if( strncmp(line, "*VMOption", 9) == 0 )
    	    {
	    char *ptr;
	    if(vmoptions_count >= MAX_VMOPTIONS)
	    	{
	    	fprintf(errors, "deffiltopts_add_ppd(): VMOptions overflow\n");
	    	}
	    else
	    	{
		ptr = &line[9];
		ptr += strspn(ptr, " \t");
		vmoptions[vmoptions_count].opt = mystrndup(ptr, strcspn(ptr, "/:"));
		
		ptr += strcspn(ptr, ":");
		ptr += strspn(ptr, ": \t\"");
		ppr_sscanf(ptr, "%d", &vmoptions[vmoptions_count].val);
		
		vmoptions_count++;
	    	}

    	    continue;
    	    }

    	} /* end of line reading loop */

    /*
    ** See if we can use VMOption information gathered above.
    */
    if(InstalledMemory != (char*)NULL)
    	{
	int x;
	
	for(x=0; x < vmoptions_count; x++)
	    {
	    #ifdef DEBUG_FILT
	    printf("vmoptions %d: %s %d\n", x, vmoptions[x].opt, vmoptions[x].val);
	    #endif

	    if( strcmp(InstalledMemory, vmoptions[x].opt) == 0 )
	    	{
		#ifdef DEBUG_FILT
		printf("match!\n");
		#endif
		thisppd_freevm = vmoptions[x].val;
		break;
	    	}
	    }
    	}

    /*
    ** If this PPD file has yielded the lowest freevm
    ** value yet, then let it prevail.
    */
    if(freevm == 0 || thisppd_freevm < freevm)
    	freevm = thisppd_freevm;

    /*
    ** If we have not yet determined that not all the devices use the 
    ** same mfmode then see if this one uses the same mfmode
    ** as the others.
    */
    if(! mfmode_conflict)
    	{
    	tempptr = get_mfmode(product, modelname, nickname, resolution);

	if(mfmode==(char*)NULL)		/* If 1st printer, */
    	    {				/* keep it. */
    	    mfmode = tempptr;
     	    }
	else				/* otherwise, */
    	    {
    	    if(strcmp(mfmode, tempptr) != 0)	/* if they don't match, */
	    	{
	    	myfree(mfmode);			/* throw away both. */
	    	mfmode = (char*)NULL;
	    	mfmode_conflict = TRUE;
	    	}
	    myfree(tempptr);	
	    }
	}

    #ifdef DEBUG_FILT
    printf("deffiltopts_add_ppd(): done\n");
    #endif

    return EXIT_OK;
    } /* end of deffiltopts_add_ppd() */

/*
** Determine the PPD file name for a printer and call
** deffiltopts_add_ppd() with that name.
**
** Various PPAD sub-commands that edit gropu files call this.  They
** call it each time they copy a "Printer:" line.
*/
int deffiltopts_add_printer(const char *name)
    {
    char fname[MAX_PATH];
    FILE *f;
    char line[MAX_CONFLINE+2];
    char *PPDFile = (char*)NULL;
    char *InstalledMemory = (char*)NULL;
    int ret;
    
    #ifdef DEBUG_FILT
    printf("deffiltopts_add_printer(\"%s\")\n",name);
    #endif

    sprintf(fname, "%s/%s", PRCONF,name);
    if( (f=fopen(fname, "r"))==(FILE*)NULL )
    	{
    	fprintf(errors,"Member printer \"%s\" does not exist.\n",name);
    	return EXIT_BADDEST;
    	}
    	
    while( fgets(line,sizeof(line),f) != (char*)NULL )
    	{
	if( strncmp(line, "PPDFile:", 8) == 0 )
	    {
	    char *ptr = &line[8];
	    int len = strlen(ptr);			/* remove */

	    while(len>0 && isspace(ptr[--len]))		/* trailing space */
	    	ptr[len] = (char)NULL;
	    while( *ptr && isspace(*ptr) ) ptr++;	/* skip leading space */

	    if(PPDFile != (char*)NULL)
	    	{
		fprintf(errors, "Warning: Ignoring all but last \"PPDFile:\" line for %s.\n", name);
	    	myfree(PPDFile);
	    	}

	    PPDFile = mystrdup(ptr);
	    }
	else if(strncmp(line, "PPDOpt: *InstalledMemory ", 25) == 0)
	    {
	    if(InstalledMemory != (char*)NULL) myfree(InstalledMemory);
	    InstalledMemory = mystrndup(&line[25], strcspn(&line[25], " "));
	    }
    	}

    fclose(f);

    if(PPDFile != (char*)NULL)
	{
    	ret = deffiltopts_add_ppd(PPDFile, InstalledMemory);

	if(ret==EXIT_BADDEST)
    	    fprintf(errors, "Printer \"%s\" has an invalid PPD file \"%s\".\n", name, PPDFile);

	myfree(PPDFile);
	}
    else	
	{
	fprintf(errors, "Printer \"%s\" has no PPD file assigned.\n", name);
	ret = EXIT_BADDEST;
	}

    if(InstalledMemory != (char*)NULL) myfree(InstalledMemory);

    #ifdef DEBUG_FILT
    printf("deffiltopts_add_printer(): done\n");
    #endif

    return ret;
    } /* end of deffiltopts_add_printer() */

/*
** Construct the key/value pairs for the "DefFiltOpts:" line.
*/
char *deffiltopts_line(void)
    {
    static char result_line[256];

    #ifdef DEBUG_FILT
    fputs("deffiltopts_line()\n", stdout);
    #endif

    if( ! open_called )
    	fatal(EXIT_INTERNAL, "deffiltopts_line(): not open");

    sprintf(result_line, "level=%d colour=%s", level == -1 ? 1 : level, colour ? "True" : "False");

    if(resolution != (char*)NULL)
	sprintf(&result_line[strlen(result_line)]," resolution=%.*s", (int)strspn(resolution, "0123456789"), resolution);
    
    if(freevm != 0)
	sprintf(&result_line[strlen(result_line)]," freevm=%d",freevm);

    if(mfmode != (char*)NULL)
    	sprintf(&result_line[strlen(result_line)]," mfmode=%s",mfmode);
    
    #ifdef DEBUG_FILT
    printf("deffiltopts_line(): returning \"%s\"\n", result_line);
    #endif

    if(debug_level >= 2)
    	printf("Line is: %s\n", result_line);

    return result_line;
    } /* end of deffiltopts_line() */

/*
** This routine is here so we can free any allocated memory.
*/
void deffiltopts_close(void)
    {
    #ifdef DEBUG_FILT
    fputs("deffiltopts_close()\n",stdout);
    #endif

    if( ! open_called )
    	fatal(EXIT_INTERNAL, "deffiltopts_close(): not open");

    if( resolution != (char*)NULL ) myfree(resolution);
    if( mfmode != (char*)NULL ) myfree(mfmode);

    open_called = FALSE;
    } /* end of deffiltopts_close() */

/* end of file */
