/*
** ~ppr/src/ppad/ppad_conf.c
** Copyright 1995, 1996, 1997, Trinity College Computing Center
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Last modified 27 February 1997.
*/

/*
** Part of the administrator's utility, this module contains code to
** edit printer and group configuration files.
*/

#include "global_defines.h"
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef USE_SPAWN
#include <spawn.h>
#endif
#include "util_exits.h"
#include "ppad.h"

#define STATE_CLOSED 0		/* no file open */
#define STATE_NOMODIFY 1	/* file open for read */
#define STATE_MODIFY 2		/* file open for write */

static state = STATE_CLOSED;
static FILE *confin;
static char confin_name[MAX_PATH];
static FILE *confout;
static char confout_name[MAX_PATH];
char confline[MAX_CONFLINE+2];

/*
** Open a printer configuration file.
**
** prnname is the name of the printer we want to open, 
** modify should be non-zero if we want to edit the file.
** If there is any old error, return -1, if the printer
** does not exist, return 1.
*/
int prnopen(char *prnname, int modify)
    {
    sprintf(confin_name, "%s/%s", PRCONF, prnname);

    if(debug_level >= 1)
	printf("Opening \"%s\" %s.\n", confin_name, modify ? "for edit" : "read-only");

    again:		/* <-- for locking retries */

    /*
    ** If modifying the file, we must open it for read
    ** and open a temporary file for write.
    */
    if(modify)
	{
	if( (confin=fopen(confin_name,"r+")) == (FILE*)NULL )
            return 1;		/* must open for update if we want */
				/* an exclusive lock */
	sprintf(confout_name,"%s/.ppad%ld",PRCONF,(long)getpid());
				/* this must be in configuration dir */
				/* the leading dot tells pprd to ignore */
	if( (confout=fopen(confout_name,"w")) == (FILE*)NULL )
	    fatal(EXIT_INTERNAL,"error opening temporary file");
	    
	if(lock_exclusive(fileno(confin),FALSE))
    	    {
    	    fclose(confin);
	    printf("Waiting for lock to clear.\n");
    	    sleep(1);
    	    goto again;
    	    }

	state = STATE_MODIFY;
	}

    /*
    ** If we will not be modifying the file, we need
    ** only open it for read.
    */
    else
    	{
	if( (confin = fopen(confin_name, "r")) == (FILE*)NULL )
            return 1;

	state = STATE_NOMODIFY;
    	}

    return 0;
    } /* end of prnopen() */
    
/*
** Open a group configuration file.
**
** Just like the function above, except here we work on 
** a group file.
*/
int grpopen(char *grpname, int modify)
    {
    sprintf(confin_name, "%s/%s", GRCONF, grpname);

    if(debug_level >= 1)
	printf("Opening \"%s\" %s.\n", confin_name, modify ? "for edit" : "read-only");

    again:		/* <-- for locking retries */

    /* 
    ** If modifying the file, we must open it for read
    ** and open a temporary file for write. 
    */
    if(modify)
	{
	if( (confin=fopen(confin_name,"r+")) == (FILE*)NULL )
            return 1;		/* open for update required for */
				/* exclusive lock */
	sprintf(confout_name,"%s/.ppad%ld",GRCONF,(long)getpid());
				/* this must be in configuration dir */
				/* the leading dot tells pprd to ignore */
	if( (confout=fopen(confout_name,"w")) == (FILE*)NULL )
	    fatal(EXIT_INTERNAL,"error opening temporary file");
	    
	/* Lock the file against modifications by others. */
	if(lock_exclusive(fileno(confin),FALSE))
    	    {
    	    fclose(confin);
	    printf("Waiting for lock to clear.\n");
    	    sleep(1);
    	    goto again;
    	    }

	state = STATE_MODIFY;
	}

    /* 
    ** If we will not be modifying the file, we need
    ** only open it for read. 
    */
    else
    	{
	if( (confin=fopen(confin_name,"r")) == (FILE*)NULL )
            return 1;

	state = STATE_NOMODIFY;
    	}

    return 0;
    } /* end of grpopen() */
    
/*
** Read a line from the configuration file. 
** If we suceed, return TRUE.
** The line is returned in the global variable "confline[]".
*/     
int confread(void)
    {
    int x;
    
    if(state == STATE_CLOSED)
        fatal(EXIT_INTERNAL, "confread(): attempt to read without open file");
        
    if( fgets(confline, sizeof(confline), confin) == (char*)NULL )
        return FALSE;
        
    /* Remove trailing spaces. */
    x = strlen(confline);
    while( (--x>=0) && isspace(confline[x]) )
    	confline[x] = (char)NULL;

    if(debug_level >= 3)
    	printf("<%s\n", confline);

    return TRUE;
    } /* end of confread() */
    
/*
** Write a line to the temporary file.
*/
int confwrite(char *format_str, ...)
    {
    va_list va;

    if(state != STATE_MODIFY)
    	fatal(EXIT_INTERNAL, "confwrite(): internal error, file not open for modify");

    va_start(va,format_str);

    vfprintf(confout, format_str, va);

    if(debug_level >= 3)
    	{
    	fputc('>', stdout);
    	vfprintf(stdout, format_str, va);
    	}

    va_end(va);

    return 0;
    } /* end of confwrite() */

/*
** Close the configuration file, replacing it with the temporary 
** file if necessary.  (It is necessary if the configuration file
** was opened for modification.)
*/
int confclose(void)
    {
    switch(state)
    	{
    	case STATE_CLOSED:
	    fatal(EXIT_INTERNAL, "confclose(): no file open");
    	case STATE_NOMODIFY:
	    if(debug_level >= 1)
	    	printf("Closing \"%s\".\n", confin_name);
	    fclose(confin);
	    state = STATE_CLOSED;
	    break;
    	case STATE_MODIFY:
	    if(debug_level >= 1)
	    	printf("Saving new \"%s\".\n", confin_name);
	    fclose(confin);
	    fclose(confout);
	    unlink(confin_name);
	    rename(confout_name, confin_name);
	    state=STATE_CLOSED;    	
    	    break;
    	}

    return 0;
    } /* end of confclose() */

/*
** Close the configuration file and delete the new copy.
**
** This is used to abort the modification of the 
** configuration file.
*/
int confabort(void)
    {
    switch(state)
        {
        case STATE_CLOSED:
            fatal(EXIT_INTERNAL,"confabort(): internal error, no file open");
        case STATE_NOMODIFY:
	    fatal(EXIT_INTERNAL,"confabort(): internal error, not open in write mode");            
	case STATE_MODIFY:
	    if(debug_level >= 1)
	    	printf("Discarding changes to \"%s\".\n", confin_name);
	    fclose(confin);
	    fclose(confout);
	    unlink(confout_name);
	    state=STATE_CLOSED;
	    break;
	}
    
    return 0;    
    } /* end of confabort() */
    
/*
** Run ppop to disable a printer before deleting or something like that.
** Return -1 if we fail, otherwise return ppop exit status.
*/
int ppop2(char *parm1, char *parm2)
    {
    int fd;			/* file descriptor for opening /dev/null */

    #ifdef USE_SPAWN
    int saved_stdin;
    int saved_stdout;
    int saved_stderr;
    
    saved_stdin=dup(0);
    saved_stdout=dup(1);
    saved_stderr=dup(2);

    if( (fd=open("/dev/null",O_RDWR)) != -1 )
	fatal(EXIT_INTERNAL,"failed to open /dev/null");

    dup2(fd,0);
    dup2(fd,1);
    dup2(fd,2);
    if(fd > 2)
    	close(fd);    	

    spawnl(PPOP_PATH,parm1,parm2,(char*)NULL);

    dup2(saved_stdin,0);
    dup2(saved_stdout,1);
    dup2(saved_stderr,2);

    return 0;

    #else			/* use fork() and exec() */
    int wstat;

    switch(fork())
    	{
    	case -1:		/* fork() failed */
    	    return -1;
    	case 0:    		/* child */
	    if( (fd=open("/dev/null",O_RDWR)) == -1 )
	    	fatal(EXIT_INTERNAL,"failed to open /dev/null");
	    	
	    dup2(fd,0);
	    dup2(fd,1);
	    dup2(fd,2);
	    if(fd > 2)
	        close(fd);
	    execl(PPOP_PATH,"ppop",parm1,parm2,(char*)NULL);
    	    _exit(255);
    	default:		/* parent */
    	    if(wait(&wstat)==-1)	/* if wait failed */
    	    	return -1;
    	    if(!WIFEXITED(wstat))	/* if ppop didn't exit normally */
    	    	return -1;
	    if(WEXITSTATUS(wstat)==255)	/* if execl() probably failed */
	        return -1;
    	    return WEXITSTATUS(wstat);
    	}	
    #endif			/* use spawn() or fork() and exec() */
    } /* end of ppop2() */

/*
** Write a string to the fifo.
*/
void write_fifo(char *string, ... )
    {
    #ifdef USE_SOCKET_FIFO
    
    #error Code missing
    
    #else			/* use Posix FIFO */
    int fifo;
    FILE *FIFO;
    va_list va;
    
    /* open the fifo */
    if( (fifo=open(FIFO_NAME,O_WRONLY | O_NDELAY)) == -1 )
        fatal(EXIT_NOSPOOLER,"can't open FIFO, pprd probably not running.");
    FIFO = fdopen(fifo, "w");

    va_start(va, string);
    vfprintf(FIFO, string, va);
    va_end(va);
    
    fclose(FIFO);
    #endif
    } /* end of write_fifo() */
    
/*
** Make the text of a "Switchset:" line.
**
** This converts the argument vector passed to it into a compressed
** format for placing in the configuration file "Switchset:" line.
*/
int make_switchset_line(char *line, char *argv[])
    {
    int x, dptr, z;
    int c;
    
    for(x=dptr=0; argv[x] != (char*)NULL; )
    	{
    	if(argv[x][0] != '-')		/* switch must begin with minus */
   	     return -1;

	if(dptr)			/* if not first one, */
	     line[dptr++] = '|';	/* insert a separator */

	if(argv[x][1] == '-')		/* long option? */
	    {
	    int y;
	    for(y=1; argv[x][y]; y++)
	    	line[dptr++] = argv[x][y];
	    }
	else				/* short option */
	    {
	    line[dptr++] = argv[x][1];		/* copy switch char */

	    if(argv[x][2] != (char)NULL)	/* look for lack of separation after switch */
		return -1;
	    }

	x++;

	/* If the switch has an argument, copy it too. */
	if(argv[x] != (char*)NULL && argv[x][0] != '-')
	    {
	    if(argv[x-1][1] == '-')		/* if was long, */
	    	line[dptr++] = '=';		/* add separator */

	    for(z=0; (c=argv[x][z]); z++)
	    	line[dptr++] = c;
	    x++;
   	    }		
    	}

    line[dptr] = (char)NULL;
    return 0;
    } /* end of make_switchset_line() */

/*
** Print a switchset line in an inteligable form.
** In other words, it uncompresses a "Switchset:" line.
**
** This function is in this module becuase it is called
** by both ppad_printer.c and ppad_group.c.
*/
void print_switchset(char *switchset)
    {
    int x;			/* line index */
    #ifdef GNUC_HAPPY
    char optchar=0;		/* switch character we are processing (0 is for GNU-C) */
    #else
    char optchar;
    #endif
    char *argument;		/* character's argument */
    int len;

    printf("Switchset macro: ");

    if(switchset == (char*)NULL)	/* If it is a NULL pointer, that means */
        {				/* that no "Switchset:" line was found. */
        printf("none\n");
        return;
        }

    len = strlen(switchset);
    for(x=0; x < len; x++)	/* move thru the line */
    	{
	optchar = switchset[x++];	/* get an option character */

	argument = &switchset[x];				/* The stuff after the switch */
	argument[strcspn(argument, "|\n")] = (char)NULL;	/* NULL terminate */

	if(optchar == '-')			/* if a long option, */
	    {
	    char *p = strchr(argument, '=');
	    if(p != (char*)NULL && strchr(++p, ' ') != (char*)NULL)
		printf("--%.*s='%s' ", (int)strcspn(argument, "="), argument, p);
	    else
		printf("--%s ", argument);
	    }
	else if(argument[0] == (char)NULL)	/* short option with no argument */
	    {
	    printf("-%c ", optchar);
	    }
	else					/* short option with argument, */
	    {
	    if(strchr(argument, ' ') != (char*)NULL)	/* if contains spaces, */
	    	printf("-%c '%s' ", optchar, argument);	/* embed optarg in quotes */
	    else					/* not spaces, */
	    	printf("-%c %s ", optchar, argument);	/* no quotes */
	    }

    	x += strlen(argument);		/* move past this one */
    	}				/* (x++ in for() will move past the NULL) */
    
    putchar('\n');
    } /* end of print_switchset() */

/*
** Print the default filter options, line wrapping them if necessary.
*/
void print_deffiltopts(char *deffiltopts)
    {
    int x;
    int opts_len;
    int out_len=23;		/* length of "Default filter options:" */
    int word_len;
    
    fputs("Default filter options:",stdout);	/* print the heading */

    if(deffiltopts == (char*)NULL)
    	{
    	puts(" <missing>");
    	return;
    	}

    opts_len=strlen(deffiltopts);		/* determine total length of options */

    for(x=0; x < opts_len; x++)			/* while options remain */
    	{
    	word_len=strcspn(&deffiltopts[x]," \t"); /* how long is this element? */
    	
    	if( (out_len+word_len+1) >= 80)		/* If leading space and element */
	    {					/* will not fit, */
	    fputs("\n    ",stdout);		/* start a new line. */
	    out_len=4+word_len;    	
	    }
	else					/* Otherwise, */
	    {					/* add to this one */
	    putchar(' ');
	    out_len++;				/* a space */
	    out_len+=word_len;			/* and the element. */
	    }

	while(word_len--)			/* Print the element. */
	    {
	    fputc(deffiltopts[x++], stdout);
	    }
    	}    
    
    putchar('\n');
    } /* end of print_deffiltopts() */

/* end of file */
