/*
** ~ppr/src/ppad/ppad.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file last modified 4 December 1996.
*/

/*
** Administration program for PostScript page printers.  This program
** edits the media database and edits printer and group configuration
** files.
*/

#include "global_defines.h"
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include "util_exits.h"
#include "ppad.h"
#include "version.h"

/* Values the library routines should pass to exit() after fatal errors: */
const int lib_memory_fatal = EXIT_INTERNAL;
const int lib_misc_fatal = EXIT_INTERNAL;

/* misc globals */
static uid_t uid;
static uid_t euid;
int machine_readable = FALSE;
FILE *errors = stderr;
int debug_level = 0;

/*
** Handle fatal errors.
** Print a message and exit.
*/
void fatal(int exitval, const char *message, ... )
    {
    va_list va;

    va_start(va,message);
    fputs("Fatal:  ",errors);
    vfprintf(errors,message,va);
    fputc('\n',errors);
    va_end(va);

    exit(exitval);
    } /* end of fatal() */

/*
** Function to return true if the current user is a privledged
** user.  A privledged user is defined as "root", "ppr", or a 
** member of the group called "ppad".  The first time this
** routine is called, it will cache the answer.
**
** This is a higher level of privledge than that defined for
** ppr(1) and ppop(8).
*/
int privledged(void)
    {
    static int answer = -1;	/* -1 means undetermined */
    
    if(answer == -1)		/* if undetermined */
    	{
	answer=0;		/* Start with "false". */

	/* Of course, "ppr" is privledged, as is uid 0 (root). */
	if( uid==0 || uid == euid )
	    {
	    answer=1;		/* Change answer to "true". */
	    }

	/* As are all users who are members of the group "ppr". */
	else		
	    {
	    struct passwd *pw;
    	    struct group *gr;
    	    int x;
	    char *ptr;

	    if( (pw=getpwuid(uid)) == (struct passwd *)NULL )
	    	{
	    	fprintf(errors,"privledeged(): getpwuid() failed, errno=%d (%s)\n",errno,strerror(errno));
	    	exit(EXIT_INTERNAL);
	    	}

	    if( (gr=getgrnam(GROUP_PPAD)) != (struct group *)NULL )
	        {
	        x=0;
    	        while( (ptr=gr->gr_mem[x++]) != (char*)NULL )
    	    	    {
    	    	    if( strcmp(ptr,pw->pw_name)==0 )
    	    	        {
    	    	        answer=1;	/* Change to "true". */
    	    	        break;
    	    	        }
    	            }
    	        }
    	    }
    	} /* end of if answer not determined yet */

    return answer;
    } /* end of privledged() */

/*
** Return non-zero if the user is an operator.
** Also, print an error message.
*/
int am_operator(void)
    {
    if( privledged() )
	{
	return 0;
	}
    else
	{
	fputs("You are not allowed to perform the requested operation\n"
		"because you are not a PPR administrator.\n",errors);
	return -1;
	}
    } /* end of am_operator() */

/*
** Print the help screen.
*/
void help_media(FILE *out)
    {
    fputs("Media Management:\n"
	"\tppad media show <name>\n"
	"\tppad media put <name> <width> <length> <weight> <type> <banner?>\n"
	"\tppad media delete <name>\n"
	"\tppad media export <filename>\n", out);
    } /* end of help_media() */

void help_printer(FILE *out)
    {
    fputs("Printer Managment:\n"
	"\tppad show <printer>\n"
	"\tppad comment <printer> <comment>\n"
	"\tppad interface <printer> <interface> <address>\n"
	"\tppad options <printer> <options>\n"
	"\tppad jobbreak <printer> {none, signal,control-d, pjl, signal/pjl}\n"
	"\tppad feedback <printer> {True, False}\n"
	"\tppad ppd <printer> <filename>\n"
	"\tppad alerts <printer> <frequency> <method> <address>\n"
	"\tppad frequency <printer> <frequency>\n"
	"\tppad flags <printer> {never, no, yes, always} {never, no, yes, always}\n"
	"\tppad outputorder <printer> {Normal, Reverse, PPD}\n"
	"\tppad bins ppd <printer>\n"
	"\tppad bins add <printer> <bin>\n"
	"\tppad bins delete <printer> <bin>\n"
	"\tppad delete <printer>\n"
	"\tppad touch <printer>\n"
	"\tppad switchset <printer> [<switches> ...]\n"
	"\tppad deffiltopts <printer>\n"
	"\tppad ppdopts <printer>\n", out);
    } /* end of help_printer() */
	
void help_group(FILE *out)
    {
    fputs("Group Management:\n"
	"\tppad group show <group>\n"
	"\tppad group comment <group> <comment>\n"
	"\tppad group rotate <group> {true, false}\n"
	"\tppad group add <group> <printer>\n"
	"\tppad group remove <group> <printer>\n"
	"\tppad group delete <groupname>\n"
	"\tppad group touch <groupname>\n"
	"\tppad group switchset <printer> [<switches> ...]\n"
	"\tppad group deffiltopts <group>\n", out);
    } /* end of help_group() */

void help_other(FILE *out)
    {
    fputs("New printer default:\n"
	"\tppad new alerts <frequency> <methode> <address>\n", out);

    fputs("\nDispatch reminder mail:\n"
	"\tppad remind\n", out);
    } /* end of help_other() */

/*
** Print help information on switches.
*/
void help(FILE *out)
    {
    fputs("Valid switches:\n"
	"\t--version\tprint version\n"
	"\t--help\tprint help\n"
	"\t-M\tmachine readable output\n"
	"\t--machine-readable\n"
	"\t-d <n>\tset debug level to <n>\n"
	"\t--debug=<n>\n", out);

    fputs("\nAdditional help:\n"
    	"\tppad help printer\n"
    	"\tppad help group\n"
    	"\tppad help media\n"
    	"\tppad help other\n", out);
    } /* end of help() */

/*
** Command dispatcher.
*/
int dispatch(char *argv[])
    {
    /* media commands */
    if( icmp(argv[0], "media") == 0 && argv[1] != (char*)NULL )
	{
	if(icmp(argv[1], "show") == 0)
	    return media_show(&argv[2]);
	if(icmp(argv[1], "put") == 0)
	    return media_put(&argv[2]);
	if(icmp(argv[1], "delete") == 0)
	    return media_delete(&argv[2]);
	if(icmp(argv[1], "export") == 0)
	    return media_export();
        }

    /* printer commands */
    if(icmp(argv[0], "show") == 0)
	return printer_show(&argv[1]);
    if(icmp(argv[0], "comment") == 0)
        return printer_comment(&argv[1]);
    if(icmp(argv[0], "interface") == 0)
    	return printer_interface(&argv[1]);
    if(icmp(argv[0], "options") == 0)
        return printer_options(&argv[1]);
    if(icmp(argv[0], "jobbreak") == 0)
    	return printer_jobbreak(&argv[1]);
    if(icmp(argv[0], "feedback") == 0)
    	return printer_feedback(&argv[1]);
    if(icmp(argv[0], "ppd") == 0)
    	return printer_ppd(&argv[1]);
    if(icmp(argv[0], "alerts") == 0)
    	return printer_alerts(&argv[1]);
    if(icmp(argv[0], "frequency") == 0)
    	return printer_frequency(&argv[1]);
    if(icmp(argv[0], "flags") == 0)
    	return printer_flags(&argv[1]);
    if(icmp(argv[0], "outputorder") == 0)
    	return printer_outputorder(&argv[1]);
    if(icmp(argv[0], "charge") == 0)
    	return printer_charge(&argv[1]);
    if( (icmp(argv[0], "bins") == 0) && (argv[1]!=(char*)NULL) )
    	{
	if(icmp(argv[1], "ppd") == 0)
	    return printer_bins_ppd(&argv[2]);
	if(icmp(argv[1], "add") == 0)
	    return printer_bins_add(&argv[2]);
	if(icmp(argv[1], "delete") == 0)
	    return printer_bins_delete(&argv[2]);
	}
    if(icmp(argv[0], "delete") == 0)
        return printer_delete(&argv[1]);    
    if(icmp(argv[0], "touch") == 0)
    	return printer_touch(&argv[1]);
    if(icmp(argv[0], "switchset") == 0)
    	return printer_switchset(&argv[1]);
    if(icmp(argv[0], "deffiltopts") == 0)
	return printer_deffiltopts(&argv[1]);
    if(icmp(argv[0], "ppdopts") == 0)
    	return printer_ppdopts(&argv[1]);

    /* group commands */
    if( (icmp(argv[0], "group") == 0) && (argv[1]!=(char*)NULL) )
    	{
	if(icmp(argv[1], "show") == 0)
	    return group_show(&argv[2]);
	if(icmp(argv[1], "comment") == 0)
	    return group_comment(&argv[2]);
	if(icmp(argv[1], "rotate") == 0)
	    return group_rotate(&argv[2]);
	if(icmp(argv[1], "add") == 0)
	    return group_add(&argv[2]);
	if(icmp(argv[1], "remove") == 0)
	    return group_remove(&argv[2]);
	if(icmp(argv[1], "delete") == 0)
	    return group_delete(&argv[2]);
	if(icmp(argv[1], "touch") == 0)
	    return group_touch(&argv[2]);
	if(icmp(argv[1], "switchset") == 0)
	    return group_switchset(&argv[2]);
	if(icmp(argv[1], "deffiltopts") == 0)
	    return group_deffiltopts(&argv[2]);
    	}

    /* new printer default */
    if( icmp(argv[0], "new") == 0 && argv[1] != (char*)NULL )
    	{
    	if(icmp(argv[1], "alerts") == 0)
    	    return printer_new_alerts(&argv[2]);
	}
	
    /* Remind (nag) command. */
    if(icmp(argv[0], "remind") == 0)
    	{
	write_fifo("n\n");
	return 0;
    	}

    /* Help commands */
    if(icmp(argv[0], "help") == 0)
    	{
	if(argv[1] == (char*)NULL)
	    help(stdout);
	else if(icmp(argv[1], "printer") == 0)
	    help_printer(stdout);    	
	else if(icmp(argv[1], "group") == 0)
	    help_group(stdout);    	
	else if(icmp(argv[1], "media") == 0)
	    help_media(stdout);    	
	else if(icmp(argv[1], "other") == 0)
	    help_other(stdout);    	
	else
	    {
	    help(errors);
	    return EXIT_SYNTAX;
	    }
	return EXIT_OK;
    	}

    return -1;
    } /* end of dispatch() */

/*
** interactive mode function
** Return the result code of the last command executed.
**
** In interactive mode, we present a prompt, read command
** lines, and execute them.
*/
int interactive_mode(void)
    {
    char *ar[16];		/* argument vector constructed from line[] */
    char *ptr;			/* used to parse arguments */
    unsigned int x;		/* used to parse arguments */
    int errorlevel = 0;		/* return value from last command */

    if( ! machine_readable )
	{
	puts("PPAD, Page Printer Administrator's utility");
	puts(VERSION);
	puts(COPYRIGHT);
	puts(AUTHOR);
	puts("\n\"help\" for command list, ");
	puts("\"exit\" to quit.\n");
	}
    else		/* terse, machine readable banner */
	{
    	puts("*READY\t"VERSION);
    	fflush(stdout);
    	}

    /*
    ** Read input lines until end of file.
    */
    while( (ptr=ppr_get_command("ppad>",machine_readable)) != (char*)NULL )
	{
	for(x=0; x<sizeof(ar); x++)	/* parse into ar[] */
	    {
	    ar[x]=strtok(ptr, " \t\n");	/* get next token */
	    if(ar[x]==(char*)NULL)	/* stop if no new token */
	    	break;
	    ptr = (char*)NULL;		/* clear ptr so as not to reset strtok() */
	    }
	
	/*
	** The variable x will be an index into ar[] which will
	** indicate the first element that has any significance.
	** If the line begins with the word "ppad" will will
	** increment x.
	*/
	x=0;
	if( ar[0] != (char*)NULL && strcmp(ar[0], "ppad") == 0 )
	    x=1;
	
	/*
	** If no tokens remain in this command line, 
	** go on to the next command line.
	*/
	if(ar[x] == (char*)NULL)
	    continue;

	/*
	** If the command is "exit", break out of
	** the line reading loop.
	*/
	if(strcmp(ar[x], "exit") == 0 || strcmp(ar[x], "quit") == 0)
	    break;

	/*
	** Call the dispatch() function to execute the command.  If the
	** command is not recognized, dispatch() will return -1.  In that
	** case we print a helpful message and change the errorlevel to
	** zero since -1 is not a valid exit code for a program.
	*/
	if( (errorlevel = dispatch(&ar[x])) == -1 )
	    {
	    if( ! machine_readable )			/* A human gets english */
		puts("Try \"help\" or \"exit\".");
	    else					/* A program gets a code */
	    	puts("*UNKNOWN");

	    errorlevel=0;
	    }
	else if(machine_readable)		/* If a program is reading our output, */
	    {					/* say the command is done */
	    printf("*DONE\t%d\n ", errorlevel);	/* and tell the exit code. */
	    }

	if(machine_readable)			/* In machine readable mode output */
	    fflush(stdout);			/* is probably a pipe which must be flushed. */
	} /* While not end of file */

    return errorlevel;			/* return result of last command (not counting exit) */
    } /* end of interactive_mode() */
    
static const char *option_chars = "Md:";
static const struct ppr_getopt_opt option_words[] =
	{
	{"machine-readable", 'M', FALSE},
	{"debug", 'd', TRUE},
	{"help", 1000, FALSE},
	{"version", 1001, FALSE},
	{(char*)NULL, 0, FALSE}
	} ;

/*
** Main function.
**
** This depends on the argv[] array being terminated by
** a NULL character pointer.
*/
int main(int argc, char *argv[])
    {
    int optchar;
    int retval;
    struct ppr_getopt_state getopt_state;

    chdir(HOMEDIR);

    uid = getuid();		/* save for authority checking */
    euid = geteuid();

    /*
    **
    */
    ppr_getopt_init(&getopt_state, argc, argv, option_chars, option_words);
    while( (optchar=ppr_getopt(&getopt_state)) != -1 )
    	{
    	switch(optchar)
    	    {
	    case 'M':				/* machine readable */
	    	machine_readable = TRUE;
	    	errors = stdout;		/* send error messages to stdout */
	    	break;

	    case 'd':				/* debug */
	        debug_level = atoi(getopt_state.optarg);
	        break;

	    case 1000:				/* --help */
	        help(stdout);
	        exit(EXIT_OK);
	        
	    case 1001:				/* --version */
		if(machine_readable)
		    {
		    puts(SHORT_VERSION);
		    }
		else
		    {
	    	    puts(VERSION);
	    	    puts(COPYRIGHT);
	    	    puts(AUTHOR);
	    	    }
	    	exit(EXIT_OK);

	    case '?':				/* unrecognized switch */
		fprintf(errors, "Unrecognized switch: %s\n\n", getopt_state.name);
		help(errors);
		exit(EXIT_SYNTAX);

	    case ':':			/* argument required */
	    	fprintf(errors, "The %s option requires an argument.\n", getopt_state.name);
		exit(EXIT_SYNTAX);

	    case '!':			/* bad aggreation */
	    	fprintf(errors, "Switches, such as %s, which take an argument must stand alone.\n", getopt_state.name);
	    	exit(EXIT_SYNTAX);
		
	    case '-':			/* spurious argument */
	    	fprintf(errors, "The %s switch does not take an argument.\n", getopt_state.name);
	    	exit(EXIT_SYNTAX);

	    default:				/* unrecognized */
	    	fprintf(errors, "Missing case %d in switch switch().\n", optchar);
	    	exit(EXIT_INTERNAL);
	    	break;
    	    }
    	}

    /*
    ** If there is a command, dispatch it, otherwise
    ** invoke interactive mode.
    */
    if( getopt_state.optind < argc )
	retval = dispatch(&argv[getopt_state.optind]);
    else
    	retval = interactive_mode();

    /* command was not recognized, give help */
    if( retval == -1 )
    	{
	fputs("ppad: unknown sub-command, try \"ppad help\"\n",errors);
	exit(EXIT_SYNTAX);
	}

    return retval;
    } /* end of main() */

/* end of file */
