/*
** ~ppr/src/papsrv/papsrv_conf.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** This file was last modified 22 November 1996.
*/

/*
** Routines to read the PAPSRV configuration file.  The PAPSRV 
** configuration file is described in the man page papsrv.conf(5).
*/

#include "global_defines.h"
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include <string.h>
#include <memory.h>
#include <errno.h>
#include "papsrv.h"

int yylex(void);
extern FILE *yyin;
struct ADV *printer;            /* for communicating with yylex() */

char *ppd_fname[MAX_PPD_NEST];	/* names of all nested PPD files */
FILE *ppd_f[MAX_PPD_NEST];	/* stdio structure pointers for same */
int ppd_nest_level;		/* number of PPD files now open */

/*======================================================================
** Font id routines
======================================================================*/
static char **fonts=(char **)NULL;	/* pointer to an array of character pointers */
static int fonts_count=0;		/* number of fonts in the array */
static int fonts_space=0;		/* number of array slots allocated */

/*
** Given a font name, return an id number
** which will represent it in future.
*/
SHORT_INT make_font_id(char *fontname)
    {
    int x;
    
    for(x=0;x<fonts_count;x++)
	{
	if( strcmp(fonts[x],fontname) == 0 )
    	    return x;
	}    

    if( fonts_count == fonts_space )	/* if more space needed */
        {
	fonts_space+=50;		/* make space for 50 more */
        fonts = (char**)myrealloc(fonts, fonts_space, sizeof(char*) );
        }
        
    fonts[fonts_count++] = mystrdup(fontname);
    return x;
    } /* end of make_font_id() */
    
/*
** If the font name supplied is known, return the font id
** otherwise, return -1.
*/
SHORT_INT get_font_id(char *fontname)
    {
    int x;

    for(x=0;x<fonts_count;x++)
    	{
    	if(strcmp(fonts[x],fontname)==0)
    	    return x;
    	}    
    
    return -1;    
    } /* end of get_font_id() */

/*
** Given a font id, return the font name.
** If the font id is not valid, return a NULL string pointer.
*/
char *get_font_name(SHORT_INT fontid)
    {
    if( (fontid>=fonts_count) || (fontid<0) )
    	return (char*)NULL;
    
    return fonts[fontid];    
    } /* end of get_font_name() */

/*====================================================================
** Routines for identifying and reading the PPD file
====================================================================*/

/*
** Given a printer name, return the PPD file name for the printer.
** The returned string is in a static buffer.
**
** The variable entry_index is the adv[] array position at which 
** we should store PPD option settings we happen to read.  If it 
** is -1, will will ignore them.
*/
char *get_printer_ppd(char *name, int entry_index)
    {                   
    FILE *pf;
    char line[MAX_CONFLINE+1];
    static char tempstr[MAX_PATH+15];

    DODEBUG_STARTUP(("get_printer_ppd(name=\"%s\", entry_index=%d)", name, entry_index));

    tempstr[0] = (char)NULL;

    /* Open the printer's configuration file. */
    sprintf(tempstr,"%s/%s",PRCONF,name);
    if( (pf=fopen(tempstr,"r")) == (FILE*)NULL )
        fatal(0,"get_printer_ppd(): fopen(\"%s\",\"r\") failed, errno=%d (%s)",tempstr,errno,strerror(errno));

    /* Read the printer's configuration file. */
    while( fgets(line,sizeof(tempstr),pf) != (char*)NULL )
        {
        if(strncmp(line, "PPDFile:", 8) == 0)
            {
	    char *ptr;
	    int len;

            ptr = &line[8+strspn(&line[8], " \t")];
            len = strlen(ptr);				/* remove */
            while( len && isspace(ptr[--len]) )		/* trailing spaces */
                ptr[len]=(char)NULL;
	    strcpy(tempstr,ptr);
            }
        else if(entry_index != -1 && strncmp(line, "PPDOpt:", 7) == 0)
            {
	    int len;
	    char *name, *value;
	    
	    name = &line[7+strspn(&line[7]," \t")];	/* skip to start of first word */
	    len = strcspn(name," \t");			/* determine length of first word */
	    name[len] = (char)NULL;			/* terminate  first word */
	    
	    value = name + len + 1;			/* move past first word and terminator */
	    value += strspn(value, "\t ");		/* move past spaces */
	    len = strcspn(value, " \t\r\n");		/* determine length of second word */
	    value[len] = (char)NULL;			/* terminate second word */

	    if(strcmp(name, "*InstalledMemory") == 0)
	    	{
		adv[entry_index].InstalledMemory = mystrdup(value);	    	
	    	}
	    else
	    	{
		struct OPTION *opt = (struct OPTION*)myalloc(1, sizeof(struct OPTION));
		opt->name = mystrdup(name);
		opt->value = mystrdup(value);
	        opt->next = adv[entry_index].options;	/* insert new option in chain */
                adv[entry_index].options = opt;
		}
            }
        }

    fclose(pf);

    if(tempstr[0] == (char)NULL)
	fatal(0,"Printer \"%s\" has no \"PPDFile:\" line in its config file", name);
	
    return tempstr;					/* return the result */
    } /* end of get_printer_ppd() */

/*
** Return the PPD file name for a group.
** If its printers do not all have the same one,
** return a null character pointer.
**
** Unlike the above routine, this one returns the 
** string in a newly allocated heap block.
*/
char *get_group_ppd(char *name)
    {
    char fname[MAX_PATH];
    char fname2[MAX_PATH];                        
    FILE *gf;                   /* group file */
    char tempstr[MAX_PATH+15];
    char *ptr;
    int len;
    char *ppdname=(char*)NULL;
    struct stat statbuf;
    int grline=0;

    DODEBUG_STARTUP(("get_group_ppd(name=\"%s\")", name));

    /* open the group configuration file */
    sprintf(fname,"%s/%s",GRCONF,name);
    if( (gf=fopen(fname,"r")) == (FILE*)NULL )
        fatal(0,"get_group_ppd(): open failed");
    
    /* read it line by line */
    while( fgets(tempstr,sizeof(tempstr),gf) != (char*)NULL )
        {
        grline++;

        if(strncmp(tempstr,"Printer:",8)==0)
            {
            ptr=&tempstr[8+strspn(&tempstr[8]," \t")];
            len=strlen(ptr);                    /* remove */
            while( len && isspace(ptr[--len]) ) /* trailing spaces */
                ptr[len] = (char)NULL;		/* from printer name */

            sprintf(fname2, "%s/%s", PRCONF, ptr); /* check if printer exists */
            if( stat(fname2, &statbuf) )
		fatal(0, "printer \"%s\" does not exist (%s line %d)", ptr, fname2, grline);

            if(ppdname==(char*)NULL)            /* if none found yet, */
                {                               /* then use this one */
                ppdname = mystrdup( get_printer_ppd(ptr, -1) );
                }
            else
                {
		if( strcmp( ppdname, get_printer_ppd(ptr, -1) ) )
                    {                           /* if not same as last, */
                    fclose(gf);                 /* then close group file */
                    return (char*)NULL;         /* and tell caller we failed */
                    }
                }
            }        
        }

    fclose(gf);			/* close group configuration file */
    return ppdname;		/* return the name we settled on */
    } /* end of get_group_ppd() */ 

/*
** Read a PPD file for the benefit of a specific listen name.
** This function is passed a pointer to the ADV structure for the
** printer, the name of the PPD file, and the current line of the 
** PAPSRV configuration file.
*/
static int printer_fontspace;
void read_ppd(struct ADV *adv, char *ppdname, int line_count)
    {
    char *fname;

    DODEBUG_STARTUP(("read_ppd(adv=?, ppdname=\"%s\", line_count=%d)", ppdname, line_count));

    /*
    ** Make sure the PPD file name hasn't been specified twice,
    ** once automatically and once manually or twice manually.
    */
    if( adv->PPDfile != (char*)NULL )
        fatal(1, "Duplicate or unnecessary \"PPDfile:\" line (line %d)", line_count);

    /* save PPD file name */
    adv->PPDfile = ppdname;

    if(ppdname[0] == '/')
	{
	fname = mystrdup(ppdname);
	}
    else
	{
	fname = (char*)myalloc(sizeof(PPDDIR) + 1 + strlen(ppdname) + 1, sizeof(char));
	sprintf(fname, "%s/%s", PPDDIR, ppdname);
	}

    if( (yyin=fopen(fname, "r")) == (FILE *)NULL )
        fatal(1, "couldn't open \"%s\", errno=%d (%s)", fname, errno, strerror(errno));

    printer = adv;		/* set global for benefit of yylex() */
    printer_fontspace = 0;

    ppd_nest_level = 0;
    ppd_fname[ppd_nest_level] = fname;
    ppd_f[ppd_nest_level] = yyin;

    yylex();

    fclose(yyin);
    myfree(fname);

    DODEBUG_STARTUP(("read_ppd(): done"));
    } /* end of read_ppd() */

/*
** This helper function is called by the lexer that reads the PPD file.
*/
void add_font(char *fontname)
    {
    DODEBUG_STARTUP(("add_font(fontname=\"%s\")", fontname));

    if( printer->fontcount == printer_fontspace )	/* if more space needed */
    	{
    	printer_fontspace += 50;
    	printer->fontlist = (SHORT_INT*)myrealloc((void*)printer->fontlist, printer_fontspace, sizeof(SHORT_INT) );
    	}

    printer->fontlist[printer->fontcount++] = make_font_id(fontname);
    } /* end of add_font() */

/*===============================================================
** Read the PAPSRV configuration file.
** The format of this file is described in papsrv(5).
===============================================================*/
void read_conf(char *conf_fname)
    {
    FILE *conf;                 /* the configuration file */
    char line[256];             /* the current line */
    char PPRparms[256];         /* temp storate for PPRparms: line */
    int x;                      /* used for removing trailing space */
    char *newargv[MAX_ARGV];    /* temporary parameter array */
    int line_count = 0;
    char fname[MAX_PATH];       /* use for printer/group determination */
    struct stat statbuf;
    struct ADV *thisadv;
    int eof = FALSE;		/* end of file flag */

    DODEBUG_STARTUP(("read_conf(conf_fname=\"%s\")", conf_fname));

    PPRparms[0]=(char)NULL;	/* start with empty string */

    if( (conf=fopen(conf_fname,"r")) == (FILE*)NULL )
        fatal(0,"can't open configuration file \"%s\".",conf_fname);

    for(x=0; x<PAPSRV_MAX_NAMES; x++)		/* set the whole */
        {               			/* array to default */
        adv[x].PPRname = (char*)NULL;		/* values */
        adv[x].PPDfile = (char*)NULL;
        adv[x].PAPname = (char*)NULL;
	adv[x].fontcount = 0;
	adv[x].fontlist = (SHORT_INT *)NULL;
        adv[x].LanguageLevel = 0;
        adv[x].PSVersion = (char*)NULL;
        adv[x].Resolution = (char*)NULL;
        adv[x].BinaryOK = ANSWER_UNKNOWN;
        adv[x].FreeVM = 0;
        adv[x].InstalledMemory = (char*)NULL;
	adv[x].VMOptionFreeVM = 0;
        adv[x].Product = (char*)NULL;
        adv[x].ColorDevice = ANSWER_UNKNOWN;
        adv[x].RamSize = 0;
        adv[x].FaxSupport = (char*)NULL;
        adv[x].TTRasterizer = (char*)NULL;
        adv[x].ForceAUFSSecurity = FALSE;
        adv[x].options = (struct OPTION *)NULL;
        }

    while( ! eof )		/* Read until the end of the papsrv conf file */
        {
	if( fgets(line,sizeof(line),conf) == (char*)NULL )
	    {
	    eof = TRUE;
	    line[0] = (char)NULL;
	    }

        line_count++;			    /* count lines for error messages */

        if(line[0]=='#' || line[0]==';')    /* ignore comments */
            continue;

        x = strlen(line);			/* remove trailing spaces, */
        while( (--x >= 0) && isspace(line[x]) )	/* tabs, and line feeds */
            line[x] = (char)NULL; 

        if(name_count==PAPSRV_MAX_NAMES)
            fatal(0,"too many advertised names in config file");

        thisadv = &adv[name_count];           /* shortcut for later */

        /* "PPRname:" lines */
        if(strncmp(line,"PPRname:",8)==0)
            {
	    thisadv->PPRname = mystrdup(&line[8+strspn(&line[8]," \t")]);

	    sprintf(fname,"%s/%s",PRCONF,thisadv->PPRname);
	    if( stat(fname,&statbuf) == 0 ) /* try a printer */
		{                           /* if it exists, */
		char *ptr = mystrdup( get_printer_ppd(thisadv->PPRname, name_count) );
				   /* ^ function returns pointer to its own block */
		read_ppd(thisadv,ptr,line_count);
		}

            else                /* otherwise, try a group */
                {
                sprintf(fname,"%s/%s",GRCONF,thisadv->PPRname);
                if( stat(fname,&statbuf) == 0 )
                    {
                    char *ptr;
                    if((ptr=get_group_ppd(thisadv->PPRname))!=(char *)NULL)
                        {    /* ^ function returns pointer to heap block */
                        read_ppd(thisadv,ptr,line_count);
                        }
                    }
                else            /* neither printer nor group */
                    {           /* this is very bad */
                    fatal(0,"\"%s\" is not a printer or group (line %d)",
                        thisadv->PPRname, line_count);
                    }
                }
            continue;
            }

        /* 
        ** Make sure "PPRname:" has been found before we
        ** allow any other lines. 
        */
        if(adv[name_count].PPRname==(char*)NULL)
            {
	    if(line[0]==(char)NULL)		/* Blank lines have no */
	        continue;			/* meaning yet. */
            fatal(0,"\"PPRname:\" must be first line in record (line %d)", line_count);
            }

        if(icmpn(line,"PAPname:",8)==0)
            {
	    char *ptr;
	    int len;
	    char name[MAX_ATALK_NAME_COMPONENT_LEN+1];
	    char type[MAX_ATALK_NAME_COMPONENT_LEN+1];
	    char zone[MAX_ATALK_NAME_COMPONENT_LEN+1];
	    
	    ptr = &line[8];
	    ptr += strspn(ptr," \t");		/* eat leading whitespace */
	    
	    if( (len=strcspn(ptr,":@")) > MAX_ATALK_NAME_COMPONENT_LEN )
	    	fatal(0,"AppleTalk name too long (line %d)", line_count);

	    strncpy(name,ptr,len);		/* copy name */
	    name[len] = (char)NULL;
	    ptr += len;

	    if( *ptr == ':' )		/* if type follows, */
	    	{
		ptr++;			/* skip over colon */

		if( (len=strcspn(ptr,"@")) > MAX_ATALK_NAME_COMPONENT_LEN )
		    fatal(0,"AppleTalk name \"type\" element too long (line %d)", line_count);
		    
		strncpy(type, ptr, len);
		type[len] = (char)NULL;
		ptr += len;
	    	}
	    else
	    	{
	    	strcpy(type, "LaserWriter");
	    	}
	    	
	    if( *ptr == '@' )		/* if zone follows, */
	    	{
		ptr++;			/* skip over at sign */

		if( (len=strlen(ptr)) > MAX_ATALK_NAME_COMPONENT_LEN )
		    fatal(0,"AppleTalk name \"zone\" element too long (line %d)", line_count);
		    
		strncpy(zone,ptr,len);	/* copy zone */
		zone[len] = (char)NULL;
		ptr += len;
	    	}
	    else			/* use default zone */
	    	{
	    	strcpy(zone, default_zone);
	    	}
	    	
	    /* Assemble the whole name. */
	    sprintf(line, "%s:%s@%s", name, type, zone);

	    /* Make a copy to keep. */
            adv[name_count].PAPname = mystrdup(line);

	    continue;
            }

        if(icmpn(line,"PPRparms:",9)==0)
            {
            strcpy(PPRparms,&line[9+strspn(&line[9]," \t")]);
            continue;
            }

        if(icmpn(line,"BinaryOK:",9)==0)
            {
            if((thisadv->BinaryOK=torf(&line[9])) == ANSWER_UNKNOWN)
                fatal(0,"Invalid \"BinaryOK:\" option");
            continue;
            }

        if(icmpn(line,"LanguageLevel:",14)==0)
            {
            adv[name_count].LanguageLevel=atoi(&line[14]);
            continue;
            }

        if(icmpn(line,"Product:",8)==0)
            {
            adv[name_count].Product=mystrdup(&line[8+strspn(&line[8]," \t")]);
            continue;
            }

        if(icmpn(line,"PSVersion:",10)==0)
            {
            adv[name_count].PSVersion=mystrdup(&line[10+strspn(&line[10]," \t")]);
            continue;
            }

        if(icmpn(line,"FaxSupport:",11)==0)
            {
            adv[name_count].FaxSupport=mystrdup(&line[11+strspn(&line[11]," \t")]);
            continue;
            }

        if(icmpn(line,"Resolution:",11)==0)
            {
            adv[name_count].Resolution=mystrdup(&line[11+strspn(&line[11]," \t")]);
            continue;
            }

        if( (icmpn(line,"PPDfile:",8)==0) 	/* explicitly set PPD file */
			|| (strncmp(line,"PPDFile:",8)==0) )
            {
            read_ppd(&adv[name_count],&line[8+strspn(&line[8]," \t")],line_count);
            continue;
            }

	if(icmpn(line,"TTRasterizer:",13)==0)
	    {
	    adv[name_count].TTRasterizer=mystrdup(&line[13+strspn(&line[13]," \t")]);
	    continue;
	    }

	if(icmpn(line,"ForceAUFSSecurity:",18)==0)
	    {
	    if( (adv[name_count].ForceAUFSSecurity=torf(&line[18+strspn(&line[18]," \t")])) == ANSWER_UNKNOWN )
	    	fatal(0,"Invalid \"ForceAUFSSecurity:\" option");
	    if( adv[name_count].ForceAUFSSecurity && aufs_security_dir == (char*)NULL )
	    	fatal(0,"\"ForceAUFSSecurity:\" option TRUE when no -X switch used");
	    continue;
	    }

	if(icmpn(line,"RamSize:",8)==0)
	    {
	    if( (adv[name_count].RamSize = atoi(&line[8])) <= 0 )
	    	fatal(0,"Bad value for \"RamSize:\" option");
	    continue;
	    }

	/*
	** Blank line, end of record.
	*/
        if(line[0] == (char)NULL)
            {				/* Here we use the lines we read along */
            int n;          		/* with the PPD file to put together */
            char *ptr;			/* a complete picture of the printer. */
            char tempstr[256];  	/* <-- used in excerpting */
            int len;

	    if( adv[name_count].PAPname==(char*)NULL		/* must have PAPname */
		    || adv[name_count].PPRname==(char*)NULL )	/* & PPRname */
		{
                fatal(0,"record %d is incomplete",name_count+1);
                }

	    /* 
	    ** Parse the paramaters line and put the parameters into
	    ** the argv[] array for this queue. 
	    */
            ptr = PPRparms;				/* start at line start */
            for(n=0; (n<(MAX_ARGV-1)) && *ptr; n++)	/* loop for args */
                {
                switch(*ptr)
                    {
		    case 34:				/* double quote */
			len=strcspn(ptr+1,"\"");	/* find next " */
			if(ptr[len+1]!='"')
			    fatal(0,"read_conf(): unmatched quote"); 
			strncpy(tempstr,ptr+1,len);	/* copy intervening */
			tempstr[len]=(char)NULL;	/* terminate */ 
			ptr+=(len+2);			/* eat up */
			break;
		    case 39:				/* single quote */ 
			len=strcspn(ptr+1,"'");
			if(ptr[len+1]!=39)
                            fatal(0,"read_conf(): unmatched quote");
                        strncpy(tempstr,ptr+1,len);
                        tempstr[len]=(char)NULL;
                        ptr+=(len+2);
                        break;
                    default:				/* not quoted */
                        len=strcspn(ptr," \t");
                        strncpy(tempstr,ptr,len);
                        tempstr[len]=(char)NULL;
                        ptr+=len;
                    }

                newargv[n] = mystrdup(tempstr);		/* put in ptr in tmp array */

                ptr += strspn(ptr,"\t ");		/* eat up space */
                } /* end of for loop */

            newargv[n++] = (char*)NULL;					/* terminate array */
            adv[name_count].argv = (char**)myalloc(n,sizeof(char*));	/* duplicate */
            memcpy(adv[name_count].argv, newargv, n*sizeof(char*));

	    /* determine if the destination is protected */
	    adv[name_count].isprotected = destination_protected(ppr_get_nodename(), adv[name_count].PPRname);

            /* advertise the name */
            add_name(name_count);

            name_count++;		/* increment count of records read */
            PPRparms[0] = (char)NULL;	/* delete the temp storeage */
            continue;
            } /* end of if blank line */

        fatal(0,"unrecognized keyword in configuration file: \"%s\"",line);
        } /* end of while() loop */

    fclose(conf);
    } /* end of read_conf() */

/* end of file */
