/*
** ~ppr/src/misc_filters/fortran.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Fortran carriage control filter for the PPR spooling system.
** Usage: filter_fortran _options_ _printer_name_ _title_
**
** Last modified 15 February 1996.
*/

#include "global_defines.h"
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

int noisy = FALSE;

#define INCH 72.0
#define MM 2.835

#ifndef A4
int unrotated_page_width  = (int) ((8.5 * INCH)+0.5);
int unrotated_page_length = (int) ((11.0 * INCH)+0.5);
#else
int unrotated_page_width  = (int) ((210.0 * MM)+0.5);
int unrotated_page_length = (int) ((297.0 * MM)+0.5);
#endif

char *title;					/* Title string (argv[1]) */
int rotated_page_width;
int rotated_page_length;
int width=132;
int length=66;
int lm = (int)(0.5 * INCH);
int rm = (int)(0.5 * INCH);
int tm = (int)(0.5 * INCH);
int bm = (int)(0.5 * INCH);
int landscape;			/* TRUE or FALSE */

/*
** Decide on landscape or portrait.
*/
void decide(void)
    {
    if( ((double)width/(double)length) > 1.0 )
    	{
	landscape=TRUE;
	rotated_page_width=unrotated_page_length;
	rotated_page_length=unrotated_page_width;
	}
    else
    	{
	landscape=FALSE;    	
    	rotated_page_width=unrotated_page_width;
    	rotated_page_length=unrotated_page_length;
    	}    
    } /* end of decide() */

/*
** Emmit the PostScript code which must go at the start
** of the document.
*/
void emmit_header(void)
	{
	puts("%!PS-Adobe-3.0");
	puts("%%DocumentData: Clean7Bit");
	printf("%%Title: %s\n",title);
	puts("%%Creator: PPR Fortran carriage control");
	puts("%%Pages: (atend)");
	puts("%%DocumentNeededResources: font Courier");
	puts("%%ProofMode: TrustMe");	
	puts("%%EndComments\n");

	puts("%%BeginProlog");
	puts("/bd{bind def}bind def");
	puts("/selectfont where{pop}");
	puts(" {/selectfont{exch findfont exch dup type /arraytype eq");
	puts("  {makefont}{scalefont}ifelse setfont}bd");
	puts(" }ifelse");
	puts("/bp{save landscape{90 rotate 0 plength  neg translate}if"); 
	puts(" lm plength tm sub moveto}def");
	puts("/nl{currentpoint exch pop yspace sub lm exch moveto}bd");
	puts("/p{show nl}bd");
	puts("/ep{showpage restore}def");
	puts("%%EndProlog\n");
	
	puts("%%BeginSetup");
	puts("%%IncludeResource: font Courier");
	printf("/Courier %2.2f selectfont\n",
		( ( (double)(rotated_page_width - lm - rm) / (double)width ) / 0.6 ) );
	printf("/lm %d def\n",lm);
	printf("/tm %d def\n",rm);
	printf("/plength %d def\n",rotated_page_length);
	printf("/landscape %s def\n",landscape ? "true" : "false");
	printf("/yspace %d def\n",( (rotated_page_length - tm - bm) / length ) );
	puts("%%EndSetup\n");
	} /* end of emmit_header() */
	
/*
** Emmit the PostScript code which ends the document.
** This function requires to know the total number
** of pages in the document.
*/
void emmit_trailer(int page_count)
	{
	puts("%%Trailer");
	printf("%%%%Pages: %d\n",page_count);
	puts("%%EOF");
	}

/*
** Emmit PostScript code for start of page.
** This code requires to know the number of the page it is staring.
*/
void begin_page(int page)
	{
	printf("%%%%Page: %d %d\n",page,page);
	puts("bp");
	}
	
/*
** Emmit PostScript code for end of page.
*/
void end_page(void)
	{
	puts("ep\n");
	}

/*
** Get an input line.  Remove line termination.
** Valid line termination is CR, LF, CRLF, or LFCR.
** Lines which are too long will be truncated.
*/
int getline(char *buffer, int maxlen)
    {
    int c,nextc;
    int len;
    static int eof=FALSE;
	
    if(eof)		/* If end of file previously found */
	return -1;
	
    len=0;
    while( (c=fgetc(stdin)) != EOF )
    	{
	if(c==13 || c==10)
	    {
	    if( (nextc=fgetc(stdin)) == c || (nextc != 13 && nextc != 10) )
		ungetc(nextc,stdin);
	    break;
	    }

    	if(len < maxlen)	/* If there is room left, store the */
    	    buffer[len++]=c;	/* character */
    	}
	
    if(c==EOF)			/* If we hit end of file, */
	eof=TRUE;		/* note it for future. */

    return len;
    } /* end of getline() */

/*
** Read the lines and emmit them.
*/
int process_input(void)
    {
    int page=0;		/* current page */
    char *buffer;	/* buffer to hold line */
    int in_page=FALSE;
    int len,x,c;
	
    if( (buffer=(char*)malloc(sizeof(char)*width)) == (char*)NULL )
	{
	fprintf(stderr,"Failed to allocate memory for line buffer\n");
	exit(1);
	}
		
    /* Read each input line */
    while( (len=getline(buffer,width)) != -1 ) /* -1 is EOF */
	{
	if(len==0)
	    {
	    puts("nl");
 	    continue;
 	    }

	switch(buffer[0])		/* The key field */
	    {
	    case '1':			/* New page */
	    	if(in_page)		/* End current page if it exists. */
	    	    end_page();
	    	in_page=FALSE;
	    	break;
	    }

	/* Start a new page if necessary */
	if(! in_page )
		{
		begin_page(++page);
		in_page=TRUE;
		}
			
	/* Start the string for this line. */
	fputc('(',stdout);

	/* From one end of the line to the other */
	for(x=1; x < len; x++)
    	    {
	    /* Send the character, possibly as an octal number */
	    if(isprint((c=buffer[x])))
	        {
		switch(c)
		    {
		    case '(':			/* Escape these */
		    case ')':			/* characters which have */
		    case 0x5C:			/* special significance within */
		        printf("\\%c",c);	/* PostScript strings. */
		    	break;
		    default:			/* Just write most characters. */
			if(c<' ' || c>'~')      /* possibly in octal */
			    printf("\\%.3o",c);
			else
			    fputc(c,stdout);
	    	    	break;
	    	    }
	    	}
	    else				/* If not printable, */
	    	{				/* express it in octal. */
	    	printf("\\%o",c);
	    	}
	    }	

	puts(")p");			/* End of line */		
	} /* end of line reading while loop */
		
    if(in_page)				/* If we are still in a page, */
    	end_page();			/* close it. */

    return page;
    } /* end of process_input() */

int main(int argc, char *argv[])
    {
    int page_count;

    /* Usage is "filter_fortran "options" "printer" "title"". */
    if(argc < 4)
	{
	fprintf(stderr,"filter_fortran: invokation error\n");
	exit(1);
	}

    /* Set the title to the first argument. */
    title=argv[3];
	
    /* Process the parameters in the second argument. */
    {
    int rval;
    char name[32], value[64];
    	
    /* Handle all the name=value pairs. */
    options_start(argv[1]);
    while( (rval=options_get_one(name,sizeof(name),value,sizeof(value))) == 1 )
	{
	if(strcmp(name,"noisy")==0)
	    {
	    if( (noisy=torf(value)) != ANSWER_UNKNOWN )
	    	filter_option_error(1,"Invalid noisy option.");
	    }
	else if(strcmp(name,"width")==0)
	    {	
	    if( (width=atoi(value)) < 1 )
	    	filter_option_error(1,"Unreasonable value for width.");
	    }
	if(strcmp(name,"length")==0)
	    {
	    if( (length=atoi(value)) < 1 )
	    	filter_option_error(1,"Unreasonable value for length.");
	    }
	else if(noisy)
	    {
	    fprintf(stderr,"Ignoring option: \"%s=%s\"\n",name,value);
	    }
	} /* end of options loop */

    /* 
    ** If options_get_one() detected an error, print it
    ** and exit.
    */
    if( rval == -1 )
        {
	filter_option_error(1,options_error);
        }

    } /* end of option parsing block */

    /* Decide on portrait or landscape. */
    decide();

    /* Emmit the PostScript header. */
    emmit_header();

    /* Print all of the input. */
    page_count=process_input();

    /* Close the PostScript output. */
    emmit_trailer(page_count);

    /* Exit, indicating to PPR that we were sucessfull. */
    return 0;
    } /* end of main() */

/* end of file */
