/*
** ~ppr/src/misc/xpprgrant.c
** Copyright 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software and documentation are provided "as is" without
** express or implied warranty.
**
** Last modified 19 November 1996.
*/

/*
** This program should get setuid "ppr".  It reads the invoking user's .Xauthority
** file and the DISPLAY environment variable and grants PPR permission to connect
** X clients to that display.
*/

#include "global_defines.h"
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/wait.h>
#include <pwd.h>
#include "util_exits.h"

int main(int argc, char *argv[])
    {
    char *display;
    int fds[2];
    pid_t childpid1, childpid2;
    struct passwd *pass;
    int ret, wstat;
    
    /*
    ** Make sure the environment variable "DISPLAY" is defined.
    ** If it is not, xauth will not know what to export.
    */
    if( (display=getenv("DISPLAY")) == (char*)NULL )
    	{
    	fputs("DISPLAY not set.\n", stderr);
    	return EXIT_SYNTAX;
    	}

   if( geteuid() != 0 )
   	{
   	fprintf(stderr, "This program must be setuid root!\n");
   	return EXIT_INTERNAL;
   	}

    printf("Granting PPR permission to connect X clients to display \"%s\".\n", display);

    /*
    ** We need to know ppr's uid so that we can run the second xauth
    ** program as ppr.
    */
    if( (pass=getpwnam("ppr")) == (struct passwd*)NULL )
    	{
    	fprintf(stderr, "Can't find user \"ppr\" in password database.\n");
    	return EXIT_INTERNAL;
    	}

    if( pipe(fds) == -1 )
    	{
    	fprintf(stderr, "pipe() failed, errno=%d (%s)\n", errno, strerror(errno) );
    	return EXIT_INTERNAL;
    	}

    /* Launch a file for the reading end of the pipe. */
    if( (childpid2 = fork()) == -1 )
    	{
    	fprintf(stderr, "fork() failed, errno=%d (%s)\n", errno, strerror(errno) ) ;
    	return EXIT_INTERNAL;
    	}
    
    /*
    ** This is the first child process.  In this one we will run xauth to add to 
    ** ppr's .Xauthority file.  We must specify the path to the .Xauthority
    ** file since the default (which is determined using the HOME environment
    ** variable) is not suitable.
    */
    if( childpid2 == 0 )
    	{
	dup2(fds[0], 0);
	close(fds[1]);

	setuid(pass->pw_uid);	/* become ppr */

    	execl(XWINBINDIR"/xauth", "xauth", "-f", HOMEDIR"/.Xauthority", "nmerge", "-", (char*)NULL);

    	fprintf(stderr, "child: exec() failed, errno=%d (%s)\n", errno, strerror(errno) );
	exit(242);
    	}

    /* Launch a file for the writing end of the pipe. */
    if( (childpid1 = fork()) == -1 )
    	{
    	fprintf(stderr, "fork() failed, errno=%d (%s)\n", errno, strerror(errno) ) ;
    	return EXIT_INTERNAL;
    	}
    
    /*
    ** The second child process must run xauth too, but must
    ** have its stdout connected to the write end of
    ** the pipe.  It must also become the user so that 
    ** the user's .Xauthority file may be read.
    */
    if( childpid1 == 0 )
	{
	dup2(fds[1], 1);
	close(fds[0]);
	
	setuid(getuid());	/* become the user */

	execl(XWINBINDIR"/xauth", "xauth", "nextract", "-", display, (char*)NULL);

	fprintf(stderr, "exec() of xauth failed, errno=%d, (%s)\n", errno, strerror(errno) );
	exit(242);
	}

    /* If we don't close these, it will confuse the children. */
    close(fds[0]);
    close(fds[1]);

    /*
    ** Wait for the children to exit.
    */
    while( (ret=wait(&wstat)) != -1 || errno != ECHILD )
    	{
	if( ret == -1 )
	    {
	    fprintf(stderr, "wait() failed, errno=%d (%s)\n", errno, strerror(errno) );
	    return EXIT_INTERNAL;
	    }	

	if( WIFEXITED(wstat) )
	    {
	    if( WEXITSTATUS(wstat) != 0 )
	    	{
	    	if(ret == childpid1) fprintf(stderr, "The first xauth exited with code %d.\n", WEXITSTATUS(wstat) );
		else if(ret == childpid2) fprintf(stderr, "The second xauth exit with code %d.\n", WEXITSTATUS(wstat) );
		else fprintf(stderr, "Internal error, unknown process %ld exited.\n", (long)ret);
	    	}
	    }    	
	else
	    {
	    if(WIFSIGNALED(wstat))
		fprintf(stderr, "Child %ld was killed by %s.\n", (long)ret, strsignal(WTERMSIG(wstat)) );
	    if(WCOREDUMP(wstat))
	    	fprintf(stderr, "Child %ld dumped core.\n", (long)ret);
	    return EXIT_INTERNAL;
	    }	
    	}

    return EXIT_OK;
    } /* end of main() */

/* end of file */

