/*
** ~ppr/src/misc/ppr2samba.c
** Copyright 1995, 1996, Trinity College Computing Center.
** Written by David Chappell.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This file was last modified 9 December 1996.
*/

/*
** This program scans PPR's printer and group configuration files
** and generates a file for Samba to include from smb.conf.  It
** also generates a pseude printcap file for backward compatibility
** with pre-1.30 versions of ppr2samba.
*/

#include "global_defines.h"
#include <errno.h>
#include <dirent.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include "util_exits.h"
#include "version.h"

/* Should we support the old printcap method
   of sharing PPR printers with Samba? */
#define PRINTCAP_SUPPORT 1

/*
** The name of the "printcap" file and
** the name of the smb.conf include file.
*/
#define PRINTCAP_NAME CONFDIR"/printcap"
#define SMB_CONF_INCLUDE_NAME CONFDIR"/smb-include.conf"

/*
** The sizes of certain data structures.
** These sizes will be based uppon the limits in pprd.
*/
#define MAX_SHAREABLE_PRINTERS MAX_PRINTERS
#define MAX_GROUP_MEMBERS MAX_GROUPSIZE

/*
** A structure to hold the driver name of each printer
** for later use while generating records for groups.
*/
static struct
    {
    char *printer;
    char *drivername;
    } printers[MAX_SHAREABLE_PRINTERS];

static int printer_count = 0;

/* The files we will be sending our output to: */
#ifdef PRINTCAP_SUPPORT
FILE *PRINTCAP_FILE;
#endif
FILE *SMB_CONF_FILE;

/* Should be be verbose? */
int debug = FALSE;
FILE *errors = stderr;

/*
** Stuff that is required because we are calling
** library routines.
*/
const int lib_memory_fatal = EXIT_INTERNAL;
const int lib_misc_fatal = EXIT_INTERNAL;

void fatal(int exitval, const char *message, ... )
    {
    va_list va;
    va_start(va,message);
    fputs("Fatal:  ", errors); vfprintf(errors, message, va); fputc('\n', errors);
    va_end(va);
    exit(exitval);
    } /* end of fatal() */

/*
** Emmit one printcap and one smb.conf record.
*/
static void emmit_record(const char *name, const char *comment, const char *drivername, const char *proto)
    {
    #ifdef PRINTCAP_SUPPORT
    /* Print a printcap line.  If there was a comment, we will emmit
       it as an alias for the printer since this is a well recognized
       way to include a comment in a BSD lpr printer name. */
    if( comment != (char*)NULL )
	fprintf(PRINTCAP_FILE, "%s|%s:\n", name, comment);
    else
	fprintf(PRINTCAP_FILE, "%s:\n", name);
    #endif

    /* Emmit the smb.conf section. */
    fprintf(SMB_CONF_FILE, "[%s]\n", name);
    fprintf(SMB_CONF_FILE, "  comment = %s\n", comment != (char*)NULL ? comment : "");
    fprintf(SMB_CONF_FILE, "  printer = %s\n", name);
    if(drivername != (char*)NULL) fprintf(SMB_CONF_FILE, "  printer driver = %s\n", drivername);
    fprintf(SMB_CONF_FILE, "  copy = %s\n", proto != (char*)NULL ? proto : "pprproto");
    fprintf(SMB_CONF_FILE, "  browseable = yes\n\n");
    } /* end of emmit_record() */

/*
** Search the printer configuration file directory and make an 
** entry in the printcap and smb.conf file for each printer found there.
**
** Rememer the drivername of each printer because we will need
** it to determine the drivername of any group which has it as 
** a member.
*/
static void do_printers(void)
    {
    DIR *DIRECTORY;
    struct dirent *d;
    char fname[MAX_PATH];
    FILE *CONFFILE;
    char line[MAX_CONFLINE+2];
    int include;
    char *printer, *comment, *ppd, *drivername, *proto;
    
    if( (DIRECTORY = opendir(PRCONF)) == (DIR*)NULL )
    	{
    	fprintf(errors, "opendir(\"%s\" failed, errno=%d (%s)\n", PRCONF, errno, strerror(errno) );
	exit(EXIT_NOTFOUND);
	}
	
    while( (d = readdir(DIRECTORY)) != (struct dirent *)NULL )
	{
	/* Skip directories and hidden files.  These
	   hidden files are likely to include temporary
	   files created by ppad which were not deleted
	   due to a crash. */
	if( d->d_name[0] == '.' )
	    continue;
	    
	/* Skip Emacs style backup files. */
	if( strlen(d->d_name) > 0 && d->d_name[strlen(d->d_name) - 1] == '~' )
	    continue;

	/* Make a copy of the printer name that we can keep and
	  clear the comment, ppd and drivername pointers. */
	printer = mystrdup(d->d_name);
	comment = ppd = drivername = proto = (char*)NULL;
	include = 1;

	if(debug) printf("Sharing printer \"%s\"\n", printer);

	/* Make sure we have room in the array. */
	if( printer_count >= MAX_SHAREABLE_PRINTERS )
	    {
	    fprintf(errors, "Array overflow, printer \"%s\" not shared.\n", printer);
	    continue;
	    }

	/* Open the printer configuration file. */
	sprintf(fname, "%s/%s", PRCONF, d->d_name);
	if( (CONFFILE = fopen(fname,"r")) == (FILE*)NULL )
	    {
	    fprintf(errors, "Failed to open \"%s\", errno=%d (%s)\n", fname, errno, strerror(errno) );
	    exit(EXIT_NOTFOUND);
	    }

	/* Scan it for a "Comment:" line and a "PPDFile:" line,
	    possibly an "ms-driver-name:" line and maybe a "ppr2samba:" line. */
	while( fgets(line, sizeof(line), CONFFILE) != (char*)NULL )
	    {
	    if(strncmp(line, "Comment:", 8) == 0)
	    	{
	    	comment = &line[8+strspn(&line[8]," \t")];
		comment = mystrndup(comment, strcspn(comment, "\n"));
		if(debug) printf("  comment = %s\n", comment);
	    	}
	    else if(strncmp(line, "PPDFile:", 8) == 0)
		{
		ppd = &line[8+strspn(&line[8], " \t")];
		ppd = mystrndup(ppd, strcspn(ppd, "\n"));
		if(debug) printf("  ppd = %s\n", ppd);
		}
	    else if(strncmp(line, "ms-driver-name:", 15) == 0)
	    	{
		drivername = &line[15+strspn(&line[15], " \t")];
		drivername = mystrndup(drivername, strcspn(drivername, "\n"));	    	
		if(debug) printf("  Drivername from config file: %s\n", drivername);
	    	}
	    else if( strncmp(line, "ppr2samba:", 10) == 0)
	    	{
		if(proto != (char*)NULL)
		    {
		    myfree(proto);
		    proto = (char*)NULL;
		    }
	    	if(ppr_sscanf(line, "ppr2samba: %d %S", &include, &proto) < 1)
	    	    fprintf(stderr, "Warning: invalid \"ppr2samba:\" line.\n");
	    	}
	    }
	    
	/* Close the printer configuration file. */
	fclose(CONFFILE);

	/* If the driver name was not specified with a "ms-driver-name:" line
	   we must open the PPD file and look for the "*ShortNickName:"
	   line which MS-Windows 95 uses for the driver name.  If there 
	   is none, we will fall back to the "*NickName:" line.
	   (The Adobe PPD documentation says that the "*ShortNickName:"
	   line must come before the "*NickName:" line if it is to 
	   be considered valid.)
	   */
	if( drivername == (char*)NULL && ppd != (char*)NULL )
	    {
	    char *ppdline;

	    if( ppd_open(ppd, errors) == EXIT_OK )
	    	{
		while( (ppdline=ppd_readline()) != (char*)NULL )
		    {
		    if( drivername == (char*)NULL 
		    	&& (strncmp(ppdline, "*ShortNickName:", 15) == 0
		    	   || strncmp(ppdline, "*NickName:", 9) == 0) )
		    	{
			drivername = &ppdline[strcspn(ppdline, ":")];
			drivername = &drivername[strspn(drivername, ": \t\"")];
			drivername = mystrndup(drivername, strcspn(drivername, "\"\n"));
			if(debug) printf("  ms-driver-name from PPD file: %s\n", drivername);
		    	}
		    }
	    	}
	    }

	/* Emmit the printcap and smb.conf records. */
	if(include)
	    emmit_record(printer, comment, drivername, proto);

	/* Stash information away for later use by do_groups(). */
	printers[printer_count].printer = printer;
	printers[printer_count].drivername = drivername;
	printer_count++;

	/* We don't need to save these: */
	if(ppd != (char*)NULL) myfree(ppd);
	if(proto != (char*)NULL) myfree(proto);

	if(debug) printf("\n");
	} /* end of directory reading loop */

    closedir(DIRECTORY);
    } /* end of do_printers() */

static void do_groups(void)
    {
    DIR *DIRECTORY;
    struct dirent *d;
    char fname[MAX_PATH];
    FILE *CONFFILE;
    char line[MAX_CONFLINE+2];
    char *name, *comment, *drivername, *proto;
    int include;
    int deallocate_drivername;
    int members[MAX_GROUP_MEMBERS];		/* printer array indexes */
    int member_count;
    
    if( (DIRECTORY = opendir(GRCONF)) == (DIR*)NULL )
    	{
    	fprintf(errors, "opendir(\"%s\" failed, errno=%d (%s)\n", GRCONF, errno, strerror(errno) );
	exit(EXIT_NOTFOUND);
	}
	
    while( (d = readdir(DIRECTORY)) != (struct dirent *)NULL )
	{
	/* Skip directories and hidden files.  These
	   hidden files are likely to include temporary
	   files created by ppad which were not deleted
	   due to a crash. */
	if( d->d_name[0] == '.' )
	    continue;
	    
	/* Skip Emacs style backup files. */
	if( strlen(d->d_name) > 0 && d->d_name[strlen(d->d_name) - 1] == '~' )
	    continue;

	/* Clear the comment, ppd and drivername pointers. */
	name = d->d_name;
	comment = drivername = proto = (char*)NULL;
	include = 1;
	deallocate_drivername = FALSE;
	member_count = 0;

	if(debug) printf("Sharing group \"%s\"\n", name);

	/* Open the file. */
	sprintf(fname, "%s/%s", GRCONF, d->d_name);
	if( (CONFFILE = fopen(fname,"r")) == (FILE*)NULL )
	    {
	    fprintf(errors,"Failed to open \"%s\", errno=%d (%s)\n",fname,errno,strerror(errno));
	    exit(EXIT_NOTFOUND);
	    }

	/* Scan it for a "Comment:" line and a "PPDFile:" line. */
	while( fgets(line,sizeof(line),CONFFILE) != (char*)NULL )
	    {
	    if( strncmp(line, "Comment:", 8) == 0 )
	    	{
	    	comment = &line[8+strspn(&line[8]," \t")];
		comment = mystrndup(comment, strcspn(comment, "\n"));
		if(debug) printf("  comment = %s\n", comment);
	    	}
	    else if( strncmp(line, "ms-driver-name:", 15) == 0 )
	    	{
		drivername = &line[15+strspn(&line[15], " \t")];
		drivername = mystrndup(drivername, strcspn(drivername, "\n"));	    	
		deallocate_drivername = TRUE;
		if(debug) printf("  drivername specified in config file: %s\n", drivername);
	    	}
	    else if( strncmp(line, "Printer:", 8) == 0 )
	    	{
		char *ptr = &line[8+strspn(&line[8], " \t")];
		ptr[strcspn(ptr, " \t\n")] = (char)NULL;

		if(debug) printf("  Member: \"%s\"\n", ptr);

		if(member_count >= MAX_GROUP_MEMBERS)
		    {
		    fprintf(errors, "WARNING: Group \"%s\" has too many members.\n", name);
		    }
		else
		    {
		    int x;
		    for(x=0; x < printer_count; x++)
			{
			if( strcmp(printers[x].printer, ptr) == 0 )
			    {
			    members[member_count++] = x;
			    break;
			    }
			}
		    if(x == printer_count)
			fprintf(errors, "WARNING: Group \"%s\" member \"%s\" does not exist.\n", name, ptr);
		    }
	    	}
	    else if( strncmp(line, "ppr2samba:", 10) == 0)
	    	{
		if(proto != (char*)NULL)
		    {
		    myfree(proto);
		    proto = (char*)NULL;
		    }
	    	if(ppr_sscanf(line, "ppr2samba: %d %S", &include, &proto) < 1)
	    	    fprintf(stderr, "Warning: invalid \"ppr2samba:\" line.\n");
	    	}
	    } /* end of while(the configuration file lines last) */
	    
	fclose(CONFFILE);

	/* If the driver name was not specified with a "ms-driver-name:" line
	   we must check each of the printers and hope they all have
	   the same drivername.  Of course, if the first member doesn't 
	   have a drivername set, then there is no sense even trying.
	   Furthur, if any subsequent member does not have a driver name
	   set or has a different driver name set, then we have failed.
	   */
	if( drivername == (char*)NULL )
	    {
	    int x, p;
	    char *ptr;

	    if(debug) printf("  Searching for a common drivername:\n");

	    for(x=0; x < member_count; x++)
	    	{
		p = members[x];
		if( (ptr = printers[p].drivername) == (char*)NULL
			    || (drivername != (char*)NULL && strcmp(ptr, drivername)) )
		    {
		    if(debug)
		    	{
			if(ptr == (char*)NULL)
			    printf("    Member \"%s\" has no driver name, oh well!\n", printers[p].printer);
			else
			    printf("    Member \"%s\" has \"%s\", oh well!\n", printers[p].printer, ptr);
		    	}
		    drivername = (char*)NULL;
		    break;
		    }
		else
		    {
		    if(debug) printf("    Member \"%s\" has \"%s\", ok so far.\n", printers[p].printer, ptr);
		    drivername = ptr;
		    }
		}
	    if(debug && drivername != (char*)NULL) printf("    drivername = %s\n", drivername);
	    }

	if(debug)
	    {
	    if(comment == (char*)NULL) printf("  No comment found.\n");
	    if(drivername == (char*)NULL) printf("  ms-driver-name couldn't be determined.\n");
	    }

	/* Emmit one printcap record and one smb.conf record. */
	if(include)
	    emmit_record(name, comment, drivername, proto);

	if(comment != (char*)NULL) myfree(comment);
	if(deallocate_drivername) myfree(drivername);
	if(proto != (char*)NULL) myfree(proto);

	if(debug) printf("\n");
	} /* end of directory reading loop */

    closedir(DIRECTORY);
    } /* end of do_groups() */

static void help(FILE *outfile)
    {
    fputs("Usage: ppr2samba [-d] [--debug] [-v] [--version] [-?] [--help]\n", outfile);
    }

int main(int argc, char *argv[])
    {
    int x;

    for(x=1; x < argc; x++)
    	{
	if( strcmp(argv[x], "-d") == 0 || strcmp(argv[x], "--debug") == 0 )
	    {
	    debug = TRUE;
	    errors = stdout;	/* <-- so we can use more on the output */
	    continue;
	    }
	if( strcmp(argv[x], "-v") == 0 || strcmp(argv[x], "--version") == 0 )
	    {
	    puts(COPYRIGHT);
	    puts(VERSION);
	    return EXIT_OK;
	    }
	if( strcmp(argv[x], "-?") == 0 || strcmp(argv[x], "--help") == 0 )
	    {
	    help(stdout);
	    return EXIT_OK;
	    }

	help(stderr);
	return EXIT_SYNTAX;
	}

    if(debug) printf("ppr2samba\n" VERSION "\n\n");

    #ifdef PRINTCAP_SUPPORT
    /* Open the printcap file for write. */
    if( (PRINTCAP_FILE = fopen(PRINTCAP_NAME, "w")) == (FILE*)NULL )
    	{
    	fprintf(errors, "Failed to open \"" PRINTCAP_NAME "\", errno=%d (%s)\n", errno, strerror(errno) );
    	exit(EXIT_DENIED);
    	}
    #endif

    /* Open the smb.conf include file for write. */
    if( (SMB_CONF_FILE = fopen(SMB_CONF_INCLUDE_NAME, "w")) == (FILE*)NULL )
    	{
    	fprintf(errors, "Failed to open \"" SMB_CONF_INCLUDE_NAME "\", errno=%d (%s)\n", errno, strerror(errno) );
	exit(EXIT_DENIED);
	}

    /* Add an entry for each printer. */
    do_printers();

    /* Add an entry for each group. */
    do_groups();

    /* Close the printcap file and smb.conf include files. */
    #ifdef PRINTCAP_SUPPORT
    fclose(PRINTCAP_FILE);
    #endif
    fclose(SMB_CONF_FILE);

    /* We are done. */
    if(debug) printf("Done.\n");
    return EXIT_OK;
    } /* end of main() */

/* end of file */

